#ifndef BICOVIEWER_HH
#define BICOVIEWER_HH

// Debug Flag -- Define BV_DEBUG to display debug messages

//   Test for uniqueness then define/undefine as needed
// #ifdef BV_DEBUG
// cout << "WARNING:  BicoViewer Debug Flag previously defined. Not unique."<<endl;
// #undef BV_DEBUG
// #endif

// #define BV_DEBUG


// C++ headers
#include <sys/stat.h>
#include <pthread.h>
#include <unistd.h>
#include <cstdlib>
#include <cstdio>
#include <cmath>
#include <string>
#include <complex>

// DMT headers
#include <FSeries.hh>
#include <FSpectrum.hh>
#include <Complex.hh>   
#include "DatEnv.hh"
#include "TSeries.hh"
#include "FixedLenTS.hh" 
#include "MonServer.hh"    // Monitor Data API 
#include "DecimateBy2.hh"
#include "OperStateCondList.hh"
#include <Time.hh>
#include <Hanning.hh>
#include "DVector.hh"
#include "Dacc.hh"

// FrameCPP headers for ChanList function
#include "framecpp/FrameCPP.hh"
#include "framecpp/FrameH.hh"
#include "framecpp/FrRawData.hh"
#include "framecpp/FrAdcData.hh"
#include "framecpp/Types.hh"

#include "BicoDir.hh"

// ROOT headers
#include <TTimer.h>
#include <TROOT.h>
#include <TEnv.h>
#include <TApplication.h>
#include <TGraph.h>
#include <TStyle.h>
#include <TGFrame.h>
#include <TGButton.h>
#include <TGLabel.h>
#include <TRootEmbeddedCanvas.h>
#include <TCanvas.h>
#include <TLatex.h>
#include <TH1.h>
#include <TH2.h>

// GUI headers
#include "gmutex.hh"
#include "ligogui/TLGChannelBox.hh"
#include "ligogui/TLGEntry.hh"

class BicoViewer ;

// @name Bicoherence Viewer
//    
// Interactive GUI-based application to view Bicoherence and other higher order statistics.
// 
// 	\begin{verbatim}
// 	usage: bicoviewer
// 	\end{verbatim}
// 
// 	@author Steve Penn

//@{

// Output Configuration record.
// ----------------------------
//     @memo Output Configuration info
//     @author Steve Penn

class BVOutConf : public thread::mutex  {
 
   public:
     
     BVOutConf ();       // Initialized the output configuration
     ~BVOutConf ();      // Destroy the output configuration
  
  
     bool   fWriteBico;    // Write Bicoherence GIF Files flag
//      float  fLimitGIF;    // Limit on Number of GIF files
	  bool   fUseGIFformat; // Output plot format (1 = GIF, 0 - ASCII)
	  bool   fWriteNow;        // Write Current Plot flag
};




/** Plot Configuration record.

    @memo Plot Configuration info
    @author Steve Penn (based on a GUI template by Daniel Sigg)
  ***********************************************************************/

class BVPlotConf : public thread::mutex  {
 
   public:
     
     BVPlotConf ();       // Initialized the plot configuration
     ~BVPlotConf ();      // Destroy the plot configuration
  

     bool   fZLogScale;     // Z Log Scale
     float  fXLimit[2];     // X limits
     float  fYLimit[2];     // Y limits
      
};
 


/** Plot Data

    @memo Data info
    @author Steve Penn (Based on template from Daniel Sigg)
  ***********************************************************************/
class BVPlotData : public thread::mutex {
   public:
     
     BVPlotData ();  // Initialized a data object     
     ~BVPlotData();  // Destruct data object
     
     void CreateArrays(const int&);	// Allocate the Data Arrays
     void DeleteArrays(void);			// Delete the Data Arrays
		
   
//      bool fNew;             	// New data flag    
     Time fFrameStart;  		// Frame Start in GPS sec, Active Plot
          
     
     Float_t*  fBic ;  	// Bicoherence Array Data

     Float_t*  fPSD1 ;  	// PSD Chan 1 Data
     Float_t*  fPSD2 ;  	// PSD Chan 2 Data
     Float_t*  fPSD3 ;  	// PSD Chan 3 Data
};

   


/** Main window of Bicoherence monitor.

    @memo BicoViewer main window
    @author Steve Penn (based on template from Daniel Sigg)
  ***********************************************************************/
class BVWindow : public TGMainFrame {

//    friend class BicoViewer;  // The data monitor is a friend
    
   public:
  
     // Create the main window
     BVWindow (const TGWindow *p, UInt_t w, UInt_t h, BicoViewer& monitor,
                    const char* chnnames);
     
     virtual ~BVWindow();       // Destroy the main window
                                   
     virtual void CloseWindow();     // Close the main window
                                 
     virtual Bool_t ReadCalcConf();  // Refresh the Calculation Configuration  
                                 
     virtual Bool_t ReadOutConf();   // Refresh the Output Configuration 
                                 
     virtual Bool_t ReadPlotConf();  // Refresh the Plot Configuration 
                                 
     virtual Bool_t DrawPlots();     // Refresh the Plot
  
     // Process GUI messages
     virtual Bool_t ProcessMessage(Long_t msg, Long_t parm1, Long_t);

     // Handle timer message, ie, display new data if ready
     virtual Bool_t HandleTimer(TTimer* timer);
   
     
   protected:
   
     BicoViewer*	fMonitor;  			// Monitor
     BicoDirConf*	fCalcConf;     	// Calculation Configuration
     BVPlotData*	fPlotDataActive;	// Plot Data Active
//      BVPlotData*	fPlotDataQueue;	// Plot Data Queue
     BVOutConf 	fOutConf;      	// Output Configuration
     BVPlotConf	fPlotConf; 			// Plot Configuration

     
     // Main GUI Layout
     // ---------------  
     TGCompositeFrame*     fFrame[5]; // Frames
     TGLayoutHints*        fLayout[7]; // Layout hints
     TRootEmbeddedCanvas*  fEmbedCvsBico; // Embedded Canvas
     TCanvas*              fCvsBico;      // Plot Canvas, adopted by embedded canvas
     TRootEmbeddedCanvas*  fEmbedCvsPSD1; // Embedded Canvas
     TCanvas*              fCvsPSD1;      // Plot Canvas, adopted by embedded canvas
     TRootEmbeddedCanvas*  fEmbedCvsPSD2; // Embedded Canvas
     TCanvas*              fCvsPSD2;      // Plot Canvas, adopted by embedded canvas
     TRootEmbeddedCanvas*  fEmbedCvsPSD3; // Embedded Canvas
     TCanvas*              fCvsPSD3;      // Plot Canvas, adopted by embedded canvas


     // Name Panel
     // ----------
     TGCompositeFrame*  fFrameName;     // Frames
     TGLayoutHints*     fLayoutName[3]; // Layout hints
     TGLabel*           fLabelName[2];  // Labels
     
     
     // Calculation Parameters Panel
     // ----------------------------
     TGCompositeFrame*  fFrameCalc[33];  // Frames
     TGLayoutHints*     fLayoutCalc[10];  // Layout hints
     TGLabel*           fLabelCalc[21];  // Labels
     ligogui::TLGChannelCombobox*  fChannelSel[3];  // Channel Selection Box
     ligogui::ChannelEntry*  chanNames;   // Channel & Rate List for Channel Widget
     Int_t      chanTotal;                // # of channels listed in widget
     TGComboBox*  fFreqMaxBox;  // F_maximum ComboBox
     TGComboBox*  fFreqResBox;  // F_resolution ComboBox
     TGComboBox*  fTimeSpanBox;     // Span ComboBox
     TGComboBox*  fOverlapBox;  // Overlap ComboBox
     TGComboBox*  fWindowTypeBox;   // Windowing CheckBox  
//      TGCheckButton*  fWindow2DCheck;   // Windowing CheckBox 

          
     // Save Results Panel
     // ------------------
     TGCompositeFrame*  fFrameSave[5];  // Frames
     TGLayoutHints*     fLayoutSave[6]; // Layout hints
     TGLabel*           fLabelSave;     // Labels
     TGCheckButton*  fWriteBicoCheck;   // Write Bicoherence plots CheckBox
     TGComboBox*     fFileFormatBox;    // File format for saved plot file
     TGButton*       fButtonWriteNow;      // Write Current Button
     
     
	  // Plot Limts Panel
	  // ----------------
	  TGCompositeFrame*  fFramePlot[12];  // Frames
	  TGLayoutHints*     fLayoutPlot[10]; // Layout hints
	  TGLabel*           fLabelPlot[6];     // Labels
     TGComboBox*  fZScaleBox;   // Z Scale ComboBox
     ligogui::TLGNumericControlBox*  fXLimitBox[2];  // X Limits TextBox
     ligogui::TLGNumericControlBox*  fYLimitBox[2];  // Y Limits TextBox
	  
	  
	  // Control Buttons Panel
	  // ---------------------
	  TGCompositeFrame*  fFrameCtrl[4];  // Frames
	  TGLayoutHints*     fLayoutCtrl[7]; // Layout hints
	  TGLabel*           fLabelCtrl[3];  // Labels
	  TGTextButton*  fControlButton[4];      // Control Buttons (Start/Pause/Resume/Quit)


     // Timing Variables
     // ----------------
     TTimer*  fHeartbeat;       // Heartbeat timer
     Int_t    fSkipHeartbeats;  // Skip Heartbeats
     Int_t    fRefreshGUItime;  // GPS second of GUI Refresh
     Int_t    fPlotTimer;       // Time Until Next Plot
     Int_t    fEstPlotTime;     // Estimated Time between Plots
     bool     fPaused;          // True if paused

//      Time fFrameStartTime;      // Frame Start in GPS sec
     char frameTimeString[32];
   
     
     // GUI Style variables
     // -------------------
     static GContext_t    fgButtonGC;    // Button foreground context
     static FontStruct_t  fgButtonFont;  // Button font
     static GContext_t    fgMonNameGC;   // Monitor Name foreground context
     static FontStruct_t  fgMonNameFont; // Monitor Name font
                    
     
     // Plotting Variables
     // ------------------
     bool firstPlot;
     bool newPlot;
     bool log_scale;
     bool isOnline;
     

     TH2F* h_bicoh ;
     TH1F* h_psd1 ;
     TH1F* h_psd2 ;
     TH1F* h_psd3 ;
        
     std::string sChanNames[4];
     std::string sBicTitle;
     std::string sPSD1Title;
     std::string sPSD2Title;
     std::string sPSD3Title;
     std::string sXAxisLabel;
     std::string sYAxisLabel;
     std::string sXAxisPSD;
     std::string sYAxisPSD;
   
     std::string sOutDirName;
     std::string sOutFileName;
     Int_t fPlotNumber;
     
     
     float  fmin;
     float  fmax;
     int Nbs ;
     int NChans ;
     
};



/** BicoViewer is the object implementing the main program

    @memo BicoViewer main program
    @author Steve Penn (from a template by Daniel Sigg)
  ***********************************************************************/
class BicoViewer : public DatEnv {
      
   friend class BVWindow;  // The main window is a friend
   
   public:
     
     BicoViewer (int argc, const char* argv[]);  // Construct monitor
     virtual ~BicoViewer();                      // Destruct monitor
     virtual void ProcessData();              // Process data
   
     
   protected:
      
      BicoDirConf fCalcConf;  	// Configuration from window
      BicoDirConf fBicoConf;  	// Configuration for monitor
      
      BicoDir		fBicoDir; 		// BicoDir class
      BVPlotData 	fPlotData1;  	// Data
//       BVPlotData 	fPlotData2;  	// Data
//       BVPlotData* fCalcData;		// Points to Data Arrays for Calc output
      BVWindow* 	fMainWindow;   // Pointer to main window

      
//       float* fBic_Data1 ;      // Fixed Pointers to 3 Bicoherence Arrays
//       float* fBic_Data2 ;  
// 
//       float* fPSD1_Data1 ;      // Fixed Pointers to 3 arrays for PSD 1
//       float* fPSD1_Data2 ;  
// 
//       float* fPSD2_Data1 ;      // Fixed Pointers to 3 arrays for PSD 2
//       float* fPSD2_Data2 ;  
// 
//       float* fPSD3_Data1 ;      // Fixed Pointers to 3 arrays for PSD 3
//       float* fPSD3_Data2 ;  
 
            
      bool log_scale;
//       bool fActivePlotIsData1;
//       bool fCalcDataIsData1;
      bool fNewPlot;
      bool fOnline;
      
      


    private:
 
       void SetParameters(void);
      
      // Configuration Record
      // --------------------
      // BicoDirConf conf;
     
      // Data Channel variables
      // ----------------------
      bool fFirstTime;             // First Pass
      int iFrame ;
//      Interval dataInterval;
//       float windowNorm ;  // volume of window

      // DMT Data Containers
      // -------------------
      TSeries*   tData1 ;
      TSeries*   tData2 ;
      TSeries*   tData3 ;

      
} ;

//@}
#endif

