#ifndef IMPULSE_HH
#define IMPULSE_HH
#include "Chirp.hh"
#include "Interval.hh"

/**  Impulse template generator.
  *  Impulse is a template generator based on the Chirp class. It generates 
  *  an impulse.
  */
class Impulse : public Chirp {
public:

  /**  Impulse Constructor. 
    *  Generate a inpulse of duration dt, amplitude A and delay d.
    *  The template extends from T0 for a time dT.
    */
  explicit Impulse(const Interval& dur, double ampl = 1.0,
                const Interval& delay = Interval(0), 
                const Interval& dT=Interval(60.0), 
		const Time& t0=Time(0));

  /**  Impulse destructor.
    */
  ~Impulse(void) {}

  /**  Frequency domain template bin.
    *  Complex Frequency domain representation of the template at frequency
    *  Freq.
    */
  fComplex Fspace(double Freq, double dF=0) const;

  /**  Time domain template bin.
    *  Tspace returns the waveform value at time 't0'.
    */ 
  double Tspace(const Time& t0) const;

  /**  Get the t0.
    *  Return the template start time. 
    */
  Time getT0(void)   const;

  /**  Get the End time.
    *  Return the template end time. The end-time is defined to be the 
    *  time at which the template frequency reaches a maximum.
    */
  Time getTEnd(void) const;

  /**  Get the Critical time.
    *  Return the template critical time. The critical time is defined 
    *  explicitly or implicitly when the object is constructed.
    */
  Time getTc(void) const;

  /**  Time versus Frequency.
    *  TvsF(f) returns the absolute (GPS) time at which the in-spiral 
    *  frequency is exactly 'f'.
    */
  Time   TvsF(double f) const;

  /**  Amplitude.
    *  Ampl(t) returns the wave-form amplitude.
    */
  double Ampl(const Time& t) const;

  /**  Set the fnunction amplitude.
    *  @memo set the amplitude.
    */
  void setAmp(double amp);
  
  /**  Duration.
    *  Duration returns the impulse duration.
    */
  Interval Duration(void) const;

  /**  Set the impulse duration
    *  @memo set the duration.
    */
  void setDuration(const Interval& dur);

  /**  Delay.
    *  Delay returns the impulse delay.
    */
  Interval Delay(void) const;

  /**  Set the impulse delay
    *  @memo set the delay.
    */
  void setDelay(const Interval& delay);

private:
  //---------------------------  External Parameters
  /**  Template start time.
    *  Absolute time of the start of template generation.
    */
  Time   mT0;

  /**  Template limit time.
    *  The template limit time is defined to be the absolute (GPS) time at 
    *  which the template ends.
    */
  Time   mTl;

  /**  Critical time.
    *  The critical time is the absolute (GPS) time at which the Phi_c is 
    *   defined.
    */
  Time   mTc;

  /**  Impulse amplitude.
    */
  double mAmpl;
  
  /**  Impulse duration.
    */
  Interval mDuration;

  /**  Impulse delay.
    */
  Interval mDelay;
};

#endif  //  IMPULSE_HH
