#ifndef CHIRP_HH
#define CHIRP_HH
//
//    Class to define astrophysics templates
//
#include "Complex.hh"
#include "Time.hh"

/**  Base class of astrophysical template functions.
  *  The chirp class defines a waveform function API that may be used to 
  *  generate templates.
  */
class Chirp {
public:
  typedef fComplex Complex;

  /**  Default (null) constructor.
    *  Nothing to initialize.
    */
  Chirp(void) {}

  /**  Chirp destructor.
    *  The destructor is defined to be virtual in case the derived 
    *  template classes need to perform cleanup or ststistical processing.
    */
  virtual ~Chirp(void) {}

  /**  Template representation in the frequency domain.
    *  Interface to a function which will return the frequency domain 
    *  representation of the astrophysical waveform.
    */
  virtual Complex Fspace(double Freq, double dF=0) const = 0;

  /**  Template representation in the time domain.
    *  Interface to a function which will return the time domain 
    *  representation of the astrophysical waveform.
    */
  virtual double Tspace(const Time& t0) const = 0;

  /**  Template representation in the time domain.
    *  Return a time domain representation of the astrophysical waveform
    *  heterodyned by the specified frequency.
    */
  Complex Tspace(const Time& t0, double f0) const;

  //-----------------------------------  Accessors
  /**  Get the starting time.
    *  GetT0 returns the earliest time for which the waveform is defined.
    */
  virtual Time getT0(void) const = 0;

  /**  Get the critical time.
    *  Returns the time at which the phase is defined.
    */
  virtual Time getTc(void) const = 0;

  /**  Get the end time.
    *  GetTEnd returns the latest time for which the waveform is defined.
    */
  virtual Time getTEnd(void) const = 0;

private:
};
#endif  //   CHIRP_HH
