/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef XSIL_TABLE_HH
#define XSIL_TABLE_HH

#include "xsil/xobj.hh"
#include "xsil/stream.hh"
#include "xsil/column.hh"
#include <list>
#include <vector>
#include <string>

namespace xsil {
    class Xwriter;

    /**  This class defines tables to be output in %XSIL.
      *  @memo %XSIL table class.
      *  @author J. Zweizig
      *  @version 1.2; Modified July 16, 2010.
      *  @ingroup IO_xsil
      */
    class table : public xobj {
    public:
	/// Data type for a vector of STL strings.
	typedef std::vector<std::string> string_vect;

    public:
	/**  XSil table default constructor.
	  *  @memo default constructor.
	  */
	table(void);

	/**  Construct a table xsil object and set the name and type attributes.
	  *  @memo Named table constructor.
	  *  @param Name Table name attribute.
	  *  @param Type Table type attribute.
	  */
	table(const char* Name, const char* Type=0);

	/**  XSil table destructor.
	  *  @memo Deconstructor.
	  */
	virtual ~table(void);

	/**  Write an XSil table definition to an XML output file.
	  *  @memo Write table to an XML file.
	  *  @param xout XML output file to receive the table definition.
	  */
	void Spew(Xwriter& xout) const;

	/**  Clone an %XSIL table definition.
	  *  @memo Clone a table object.
	  *  @return Pointer to the cloned object.
	  */
	virtual table* Clone(void) const;

	/**  Test for data in the table.
	  *  @memo Test whether %table is empty.
	  *  @return true if the %table (data stream) is empty.
	  */
	bool empty(void) const;

	/**  Return the index of a specified column.
	  *  @memo Get named column index.
	  *  @param name Column name
	  *  @return Column number or -1 if column not found.
	  */
	int findColumn(const std::string& name) const;

	/**  Get the XSIL object type.
	  *  @memo Get object type.
	  *  @return Pointer to the object type string.
	  */
	const char* getObjType(void) const;

	/**  Get table units
	  *  @memo get Table units.
	  *  @return Pointer to string containing the Table units.
	  */
	const char* getUnit(void) const;

	/**  Add a column to the table.
	  *  @memo Add a column.
	  *  @param Name Pointer to column name string.
	  *  @param Type Pointer to column type string.
	  *  @param Unit Pointer to column unit string.
	  *  @return Pointer to added column.
	  */
	column* addColumn(const char* Name, const char* Type=0, 
			  const char* Unit=0);

	/**  Add a column to the table.
	  *  @memo Add a column.
	  *  @param col  Column definition to be added to table.
	  *  @return Pointer to added column.
	  */
	column* addColumn(const column& col);

	/**  Copy one row of data cells from the stream to the specified vector.
	  *  If the row doesn't exist, the size of the vector will be zero.
	  *  @memo Get a row of data from the stream.
	  *  @param indata String vector of data from one row.
	  */
	void readRow(string_vect& indata);

	/**  Get reference to the specified column.
	  *  @memo Reference a column.
	  *  @param name Name of column to be referenced.
	  *  @return reference to the name column.
	  */
	column& refColumn(const char* name);

	/**  Get reference to the specified column.
	  *  @memo Reference a column.
	  *  @param name Pointer to string with requested name.
	  *  @return reference to the name column.
	  */
	const column& refColumn(const char* name) const;

	/**  Get reference to an initialization stream.
	  *  @memo Reference initialization stream.
	  *  @return reference to the initialization stream.
	  */
	Stream& refStream(void);

	/**  Get reference to an initialization stream.
	  *  @memo Reference initialization stream.
	  *  @return reference to the initialization stream.
	  */
	const Stream& refStream(void) const;

	/**  Add a stream to the array.
	  *  @memo  Add a stream to the array.
	  *  @param istr Stream object to be added to the array.
	  */
	void setStream(const Stream& istr);

	/**  Set the table units.
	  *  @memo Set the table units.
	  *  @param Unit Pointer to table unit string.
	  */
	void setUnit(const char* Unit);

    private:
	typedef std::list<column> ColumnList;
	typedef ColumnList::const_iterator const_column_iterator;
	typedef ColumnList::iterator       column_iterator;

    private:
	std::string mUnit;
	ColumnList mColumn;
	Stream mData;
    };

    //======================================  Inline method definitions
    inline bool 
    xsil::table::empty(void) const {
	return mData.empty();
    }

    inline const char*
    xsil::table::getUnit(void) const {
	return mUnit.c_str();
    }

    //======================================  Stream reference functions.
    inline xsil::Stream& 
    xsil::table::refStream(void) {
	return mData;
    }

    inline const xsil::Stream& 
    xsil::table::refStream(void) const {
	return mData;
    }

}

#endif  //  XSIL_TABLE_HH
