/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef XSIL_ARRAY_HH
#define XSIL_ARRAY_HH

#include "xsil/xobj.hh"
#include "xsil/stream.hh"
#include "xsil/dim.hh"
#include <string>
#include <vector>

namespace xsil {
    class Xwriter;

/**  The array class contains data elements organized as an N-dimensional 
  *  array of a single data type. The last dimension varies the most rapidly.
  *  The array class contains an xsil stream to initialize the array data.
  *  @memo XSIL Array class. 
  *  @author J. Zweizig
  *  @version 1.1; Modified December 13, 1999
  *  @ingroup IO_xsil
  */
class array : public xobj {
public:
    /**  Default constructor.
      *  @memo Default constructor.
      */
    array(void);

    /**  Data constructor.
      *  @memo Default constructor.
      *  @param Name Pointer to the array name.
      *  @param Type Pointer to the array type.
      *  @param Unit Pointer to the array units.
      */
    array(const char *Name, const char* Type=0, const char* Unit=0);

    /**  Destructor.
      *  @memo Destructor.
      */
    ~array(void);
    /**  Write the array definition to an XML file.
      *  @memo Write an XML array definition.
      *  @param xout XML writer object to receive the array definition.
      */
    void Spew(Xwriter& xout) const;

    /**  Clone the array object definition.
      *  @memo Clone an array definition object.
      *  @return Pointer to te cloned object.
      */
    array* Clone(void) const;

    /**  Get the object type.
      *  @memo Get object type.
      *  @return Pointer to the object type string.
      */
    const char* getObjType(void) const;

    /**  Get the array units.
      *  @memo Get the units.
      *  @return poitner to the units string.
      */
    const char* getUnit(void) const;

    /**  Get the number of dimensions.
      *  @memo Get the number of array dimensions.
      *  @return Number of dimenstion defined for this array.
      */
    int getNDim(void) const;

    /**  Get the number of dimensions.
      *  @memo Get the number of array dimensions.
      *  @return Number of dimenstion defined for this array.
      */
    int getTotLen(void) const;

    /**  Get the specified dimension.
      *  @memo get the specified dimension.
      *  @param iDim Dmension number to be interrogated.
      *  @return the iDimth dimension.
      */
    int getDim(int iDim) const;
  
    /**  Read data from the array stream into a double STL vector.
      *  @brief Get data from the array
      *  @param vf STL vector object to receive the data.
      */
    void getData(std::vector<double>& vf);
  
    /**  Read data from the array stream into a float STL vector.
      *  @brief Get data from the array
      *  @param vf STL vector object to receive the data.
      */
    void getData(std::vector<float>&  vf);
  
    /**  Read data from the array stream into an int STL vector.
      *  @brief Get data from the array
      *  @param vf STL vector object to receive the data.
      */
    void getData(std::vector<int>&    vf);
  
    /**  Read data from the array stream into a string STL vector.
      *  @brief Get data from the array
      *  @param vf STL vector object to receive the data.
      */
    void getData(std::vector<std::string>& vf);

    /**  Add a dimension to the array.
      *  @memo Add a dimension.
      *  @return Dimension number.
      *  @param Name Dimension name
      *  @param nDim Size of added dimension.
      */
    int addDim(const char* Name, int nDim);

    /**  Add a dimension to the array.
      *  @memo Add a dimension.
      *  @return Dimension number.
      *  @param Dim Dimension block.
      */
    int addDim(const dim& Dim);

    /**  Set the array units.
      *  @memo set the array units.
      *  @param Unit Pointer to the units string.
      */
    void setUnit(const char* Unit);

    /**  Add a stream to the array.
      *  @memo  Add a stream to the array.
      *  @param istr Stream object to be added to the array.
      */
    void setStream(const Stream& istr);

    /**  Set remote stream.
      *  @memo Set a remote stream.
      *  @param File Name of file containing remote data.
      */
    void setRemote(const char* File);

    /**  Add a local initialization stream with specified integer data.
      *  @memo  Add a data stream.
      *  @param N Number of data words.
      *  @param Data Integer array containing data for the array.
      */
    void FillData(int N, int* Data);

    /**  Add a local initialization stream with specified float data.
      *  @memo  Add a data stream.
      *  @param N Number of data words.
      *  @param Data Float array containing data for the array.
      */
    void FillData(int N, float* Data);

    /**  Add a local initialization stream with specified double data.
      *  @memo  Add a data stream.
      *  @param N Number of data words.
      *  @param Data Double array containing data for the array.
      */
    void FillData(int N, double* Data);

    /**  Add a local initialization stream with specified string data.
      *  @memo  Add a data stream.
      *  @param N Number of data words.
      *  @param Data String array containing data for the array.
      */
    void FillData(int N, std::string* Data);

    /**  Get reference to an initialization stream.
      *  @memo Reference initialization strem.
      *  @return reference to the initialization stream.
      */
    Stream& refStream(void);

    /**  Get reference to an initialization stream.
      *  @memo Reference initialization strem.
      *  @return reference to the initialization stream.
      */
    const Stream& refStream(void) const;

private:
    std::string mUnit;
    std::vector<dim> mDim;
    Stream mData;
};

}

//--------------------------------------  Stream reference functions.
inline int 
xsil::array::getNDim(void) const {
    return mDim.size();
}

inline int 
xsil::array::getDim(int N) const {
    return mDim[N].getDim();
}

inline const char* 
xsil::array::getUnit(void) const {
    return mUnit.c_str();
}

inline xsil::Stream& 
xsil::array::refStream(void) {
    return mData;
}

inline const xsil::Stream& 
xsil::array::refStream(void) const {
    return mData;
}

#endif  //  XSIL_ARRAY_HH
