VCSID("@(#)$Id: Xsil.cc 7582 2016-03-01 22:25:48Z john.zweizig@LIGO.ORG $");
#define __NOMEMMAPFILE
#include "PConfig.h"
#include "xml/Xsil.hh"
#include "expat.h"
#include "tconv.h"
#include <stdio.h>
#include <inttypes.h>
#include <sstream>
#include <fstream>
#include <iostream>
#include <cstring>
#include <cstdlib>
#ifndef __NOMEMMAPFILE
#include <sys/types.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/mman.h>
#endif

namespace xml {
   using namespace std;


   const char* const xmlContainerLdas[] = 
   {"IGWDFrame", "AdcData", "AdcInterval", 0};

   bool ldas_container (const char* s) 
   {
      for (const char* const* p = xmlContainerLdas; *p; ++p) {
         if (strcmp (*p, s) == 0) {
            return true;
         }
      }
      return false;
   }


   const char table_uuencode[65] = 
   "`!\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_";
   const char table_base64[65] =
   "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";

   const char itable_uuencode[257] =
   "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
   "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
   "\377\001\002\003\004\005\006\007\010\011\012\013\014\015\016\017"
   "\020\021\022\023\024\025\026\027\030\031\032\033\034\035\036\037"
   "\040\041\042\043\044\045\046\047\050\051\052\053\054\055\056\057"
   "\060\061\062\063\064\065\066\067\070\071\072\073\074\075\076\077"
   "\000\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
   "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
   "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
   "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
   "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
   "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
   "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
   "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
   "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
   "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377";
   const char itable_base64[257] =
   "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
   "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
   "\377\377\377\377\377\377\377\377\377\377\377\076\377\377\377\077"
   "\064\065\066\067\070\071\072\073\074\075\377\377\377\377\377\377"
   "\377\000\001\002\003\004\005\006\007\010\011\012\013\014\015\016"
   "\017\020\021\022\023\024\025\026\027\030\031\377\377\377\377\377"
   "\377\032\033\034\035\036\037\040\041\042\043\044\045\046\047\050"
   "\051\052\053\054\055\056\057\060\061\062\063\377\377\377\377\377"
   "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
   "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
   "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
   "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
   "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
   "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
   "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
   "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377";


//______________________________________________________________________________
   bool littleEndian ()
   {
      int test = 0;
      *(char*) &test = 1;
      return (test == 1);
   }

//______________________________________________________________________________
   inline void swap64 (uint64_t* ww)
   {
      uint32_t temp;
      temp = 
         (((((uint32_t*)ww)[0])&0xff)<<24) |
         (((((uint32_t*)ww)[0])&0xff00)<<8) |
         (((((uint32_t*)ww)[0])&0xff0000)>>8) |
         (((((uint32_t*)ww)[0])&0xff000000)>>24); 
      ((uint32_t*)ww)[0] = 
         (((((uint32_t*)ww)[1])&0xff)<<24) |
         (((((uint32_t*)ww)[1])&0xff00)<<8) |
         (((((uint32_t*)ww)[1])&0xff0000)>>8) |
         (((((uint32_t*)ww)[1])&0xff000000)>>24); 
      ((uint32_t*)ww)[1] = temp;
   }

//______________________________________________________________________________
   inline void swap32 (uint32_t* w)
   {
      *((uint32_t*)w) = 
         ((((*(uint32_t*)w))&0xff)<<24) |
         (((*((uint32_t*)w))&0xff00)<<8) |
         (((*((uint32_t*)w))&0xff0000)>>8) |
         (((*((uint32_t*)w))&0xff000000)>>24);   
   }

//______________________________________________________________________________
   inline void swap16 (uint16_t* w)
   {
      *((uint16_t*)w) = 
         ((((*(uint16_t*)w))&0xff)<<8) |
         (((*((uint16_t*)w))&0xff00)>>8);   
   }

//______________________________________________________________________________
   void swapByteOrder (char* p, int num, int elsize) 
   {
      switch (elsize) {
         default:
         case 1:
            {
               break;
            }
         case 2:
            {
               uint16_t* x = (uint16_t*) p;
               for (int i = 0; i < num; i++, x++) {
                  swap16 (x);
               }
               break;
            }
         case 4:
            {
               uint32_t* x = (uint32_t*) p;
               for (int i = 0; i < num; i++, x++) {
                  swap32 (x);
               }
               break;
            }
         case 8:
            {
               uint64_t* x = (uint64_t*) p;
               for (int i = 0; i < num; i++, x++) {
                  swap64 (x);
               }
               break;
            }
      }
   }

//______________________________________________________________________________
   ostream& base64encode (ostream& os, const char* pp, int num, int elsize)
   {
      const char* table = table_base64;
      int i = 0;
      int len = num * elsize;
   
      if (pp == 0 || !len) {
         return os;
      }
      const char* p = pp;
      // char* temp = 0;
      // if (littleEndian()) {
         // temp = new (nothrow) char [len];
         // if (temp == 0) {
            // return os;
         // }
         // memcpy (temp, pp, len);
         // swapByteOrder (temp, num, elsize);
         // p = temp;
      // }
   
      // convert
      while (i < len) {
         os.put(table[(p[i++] >> 2) & 0x3F]);
         if (i >= len) {
            break;
         }
         os.put(table[((p[i-1] << 4) | ((p[i] >> 4) & 0x0F)) & 0x3F]);
         i++;
         if (i >= len) {
            break;
         }
         os.put(table[((p[i-1] << 2) | ((p[i] >> 6) & 0x03)) & 0x3F]);
         os.put(table[p[i] & 0x3F]);
         i++;
         if (i % 48 == 0) {
            os << endl;
         }
      }
   
      // patch end
      switch (i % 3) {
         case 0: 
            {
               break;
            }
         case 1: 
            {
               os.put(table[(p[i-1] & 0x03) << 4]);
               os << "==";
               break;
            }
         case 2: 
            {
               os.put(table[(p[i-1] & 0x0F) << 2]);
               os << "=";
               break;
            }
      }
      if (len % 48 != 0) {
         os << endl; 
      }
   
      // if (temp) delete [] temp;
      return os;
   }


//______________________________________________________________________________
   bool base64decode (const char* code, int codelen,
                     char* p, int len, int ctype)
   {
      const char* 	table =
         ((ctype == 1) ? itable_uuencode : itable_base64);
      int		i = 0;
      int		bits = 0;
      int		tmp = 0;
      int		pos = 0;
      int 		decod;
   
      while (i < len) {
         // beyond end of string
         if (pos >= codelen) {
            cerr << "base64: too many" << endl;
            return false;
         }
         // skip blanks
         int c = code[pos++];
         if ((c == ' ') || (c == '\t') || (c == '\n')) {
            continue;
         }
         // illegal character
         if ((decod = table[(c & 0xff)]) == '\377') {
            cerr << "base64: illegal @ " << pos << " " << (int) c << endl;
            return false;
         }
         // deoced
         tmp = (tmp << 6) | (decod & 0x3F);
         bits += 6;
         if (bits >= 8) {
            p[i++] = (tmp >> (bits - 8)) & 0xFF;
            bits -= 8;
         }
      }
      return true;
   }

//______________________________________________________________________________
   std::string xsilStringEscape (const char* s)
   {
      string p;
      for (; *s; ++s) {
         if (*s == '<') {
            p += "&lt;";
         }
         else if (*s == '>') {
            p += "&gt;";
         }
         else if (*s == '&') {
            p += "&amp;";
         }
         else if (*s == '"') {
            p += "&quot;";
         }
         else if (*s == '\'') {
            p += "&apos;";
         }
         else {
            p += *s;
         }
      }
      return p;
   }

//______________________________________________________________________________
   std::string xsilEscape (const char* p, int len)
   {
      std::string s;
      if (!p) {
         return s;
      }
      if (len < 0) len = strlen (p);
   
      // non printable characters?
      bool nonprint = false;
      for (int i = 0; i < len; ++i) {
         if ((p[i] < 32) || (p[i] >= 127)) {
            nonprint = true;
            break;
         }
      }
   
      // octal representation
      if (nonprint) {
         char buf[10];
         for (int i = 0; i < len; ++i) {
            sprintf (buf, "\\%03o", (unsigned char)p[i]);
            s += buf;
         }
      }
      
      // ASCII representation
      else {
         s = p;
         string::size_type pos;
         while ((pos = s.find ('<')) != string::npos) {
            s.erase (pos, 1); s.insert (pos, "&lt;");
         }
         while ((pos = s.find ('>')) != string::npos) {
            s.erase (pos, 1); s.insert (pos, "&gt;");
         }
         while ((pos = s.find ('&')) != string::npos) {
            s.erase (pos, 1); s.insert (pos, "&amp;");
         }
         pos = s.size();
         while (pos > 0) {
            --pos;
            if (s[pos] == '\\') {
               s.insert (pos, "\\\\"); 
            }
            else if (s[pos] == ',') {
               s.insert (pos, "\\,"); 
            }
         }
      }
      return s;
   }

//______________________________________________________________________________
   std::string xsilUnescape (const char* p)
   {
      std::string s;
      s = p;
      // :TODO: check for mixed representation
      string::size_type pos = 0;
      while ((pos = s.find (" ", pos)) != string::npos) {
         if (!pos || s[pos - 1] != '\\') s.erase (pos, 1);
         else ++pos;
      }
      // special characters
      while ((pos = s.find ("\\ ")) != string::npos) {
         s.erase (pos, 2); s.insert (pos, " ");
      }
      while ((pos = s.find ("\\,")) != string::npos) {
         s.erase (pos, 2); s.insert (pos, ",");
      }
      while ((pos = s.find ("\\\\")) != string::npos) {
         s.erase (pos, 2); s.insert (pos, "\\");
      }
      while ((pos = s.find ("&lt;")) != string::npos) {
         s.erase (pos, 4); s.insert (pos, "<");
      }
      while ((pos = s.find ("&gt;")) != string::npos) {
         s.erase (pos, 4); s.insert (pos, ">");
      }
      while ((pos = s.find ("&amp;")) != string::npos) {
         s.erase (pos, 5); s.insert (pos, "&");
      }
      // remove octal numbers
      pos = s.size();
      int digits = 0;
      while (pos > 0) {
         --pos; 
         if ((s[pos] == '\\') && (digits >= 3)) {
            char c = 64 * (s[pos+1] - '0') + 
               8 *  (s[pos+2] - '0') +  (s[pos+3] - '0');
            s.erase (pos, 4);
            s.insert (s.begin() + pos, c);
            digits = 0;
         }
         else {
            digits = (isdigit(s[pos])) ? digits + 1 : 0;
         }
      }
      return s;
   }

//______________________________________________________________________________
   int gdsNameDataType (const string& name) 
   {
      if ((strcasecmp (name.c_str(), "byte") == 0) ||
         (strcasecmp (name.c_str(), "char") == 0) || // ldas rubbish
         (strcasecmp (name.c_str(), "char_u") == 0)) {
         return gds_int8;
      } 
      else if ((strcasecmp (name.c_str(), "short") == 0) ||
              (strcasecmp (name.c_str(), "int_2s") == 0) || // ldas rubbish
              (strcasecmp (name.c_str(), "int_2u") == 0)) {
         return gds_int16;
      } 
      else if ((strcasecmp (name.c_str(), "int") == 0) ||
              (strcasecmp (name.c_str(), "int_4s") == 0) || // ldas rubbish
              (strcasecmp (name.c_str(), "int_4u") == 0)) {
         return gds_int32;
      } 
      else if ((strcasecmp (name.c_str(), "long") == 0) ||
              (strcasecmp (name.c_str(), "int_8s") == 0) || // ldas rubbish
              (strcasecmp (name.c_str(), "int_8u") == 0)) {
         return gds_int64;
      } 
      else if ((strcasecmp (name.c_str(), "float") == 0) ||
              (strcasecmp (name.c_str(), "real_4") == 0)) { // ldas rubbish
         return gds_float32;
      } 
      else if ((strcasecmp (name.c_str(), "double") == 0) ||
              (strcasecmp (name.c_str(), "real_8") == 0)) { // ldas rubbish
         return gds_float64;
      } 
      else if ((strcasecmp (name.c_str(), "floatComplex") == 0) ||
              (strcasecmp (name.c_str(), "complex_8") == 0)) { // ldas 
         return gds_complex32;
      } 
      else if ((strcasecmp (name.c_str(), "doubleComplex") == 0) ||
              (strcasecmp (name.c_str(), "complex_16") == 0)) { // ldas 
         return gds_complex64;
      } 
      else if ((strcasecmp (name.c_str(), "string") == 0) ||
              (strcasecmp (name.c_str(), "lstring") == 0)) { // ldas rubbish
         return gds_string;
      } 
      else if (strcasecmp (name.c_str(), "channel") == 0) {
         return gds_channel;
      } 
      else if (strcasecmp (name.c_str(), "boolean") == 0) {
         return gds_bool;
      } 
      else if (strcasecmp (name.c_str(), "time") == 0) {
         return gds_time;
      } 
      else if (strcasecmp (name.c_str(), "table") == 0) {
         return gds_table;
      } 
      else {
         return gds_void;
      }
   }

//______________________________________________________________________________
   int ldasNameDataType (const string& name) 
   {
      if ((strcasecmp (name.c_str(), "int_1s") == 0) ||
         (strcasecmp (name.c_str(), "int_1u") == 0)) {
         return gds_int8;
      } 
      else if ((strcasecmp (name.c_str(), "int_2s") == 0) ||
              (strcasecmp (name.c_str(), "int_2u") == 0)) {
         return gds_int16;
      } 
      else if ((strcasecmp (name.c_str(), "int_4s") == 0) ||
              (strcasecmp (name.c_str(), "int_4u") == 0)) {
         return gds_int32;
      } 
      else if ((strcasecmp (name.c_str(), "int_8s") == 0) ||
              (strcasecmp (name.c_str(), "int_8u") == 0)) {
         return gds_int64;
      } 
      else if (strcasecmp (name.c_str(), "real_4") == 0) {
         return gds_float32;
      } 
      else if (strcasecmp (name.c_str(), "real_8") == 0) {
         return gds_float64;
      } 
      else if (strcasecmp (name.c_str(), "complex_8") == 0) {
         return gds_complex32;
      } 
      else if (strcasecmp (name.c_str(), "complex_16") == 0) {
         return gds_complex64;
      } 
      else if (strcasecmp (name.c_str(), "lstring") == 0) {
         return gds_string;
      } 
      else if (strcasecmp (name.c_str(), "ilwd:char") == 0) {
         return gds_string;
      } 
      else if (strcasecmp (name.c_str(), "ilwd:char_u") == 0) {
         return gds_string;
      } 
      else {
         return gds_void;
      }
   }

//______________________________________________________________________________
   int gdsDatumSize (int datatype) 
   {
      switch (datatype) {
         case gds_int8:
            return 1;
         case gds_int16:
            return 2;
         case gds_int32:
         case gds_float32:
            return 4;
         case gds_int64:
         case gds_float64:
         case gds_complex32:
            return 8;
         case gds_complex64:
            return 16;
         case gds_string:
         case gds_channel:
            return 1;
         case gds_bool:
            return sizeof (bool);
         case gds_time:
            return sizeof (Time);
         default: 
            return 0;
      }
   }

//______________________________________________________________________________
   bool gdsDatumComplex (int datatype)
   {
      return ((datatype == gds_complex32) || (datatype == gds_complex64));
   }

//______________________________________________________________________________
   // bool gdsValueDataType (void* value, int datatype, 
                     // const string& datum)
   // {
      // istringstream	is (datum.c_str());
      // if (value == 0) {
         // return false;
      // }
      // switch (datatype) {
         // case gds_int8:
            // is >> *((char*) value);
            // if (*((char*) value) == 0) {
               // *((char*) value) = ' ';
            // }
            // break;
         // case gds_int16:
            // is >> *((short*) value);
            // break;
         // case gds_int32:
            // is >> *((int*) value);
            // break;
         // case gds_float32:
            // is >> *((float*) value);
            // break;
         // case gds_int64:
            // is >> *((long long*) value);
            // break;
         // case gds_float64:
            // is >> *((double*) value);
            // break;
         // case gds_complex32:
            // float	re, im;
            // is >> re >> im;
            // *(complex<float>*) value = complex<float> (re, im);
            // break;
         // case gds_complex64:
            // double	dre, dim;
            // is >> dre >> dim;
            // *(complex<double>*) value = complex<double> (dre, dim);
            // break;
         // case gds_string:
         // case gds_channel:
            // is >> *((char*) value);
            // break;
         // case gds_bool:
            // *((bool*) value) = (datum.size() > 0) && 
               // ((datum[0] == 't') || (datum[0] == 'y') || 
               // (datum[0] == 'T') || (datum[0] == 'Y') || 
               // (datum[0] == '1'));
            // break;
         // default: 
            // break;
      // }
      // return !!is;
   // }

//______________________________________________________________________________
   char* readValues (int datatype, const string& txt, int dim)
   {
      char* value = 0;
   
      // valid?
      if (datatype == gds_void) {
         return value;
      }
      // read string
      else if ((datatype == gds_string) || (datatype == gds_channel)) {
         if (!txt.empty()) {
            value = new (nothrow) char [txt.size() + 1];
            if (value != 0) {
               value[txt.size()] = 0;
               strncpy (value, txt.c_str(), txt.size());
            }
         }
         else {
            value = 0;
         }
         return value;
      }
      // read numbers
      else {
         // allocate memory
         int elSize = gdsDatumSize (datatype);
         int size = dim * elSize;
         value = new (nothrow) char [size];
         if (value == 0) {
            return 0;
         }
         memset (value, 0, size);
         // set temporary vars
         istringstream 	is (txt.c_str());
         char* 		val = value;
         int		max = dim;
         string 	datum;
         // read in values
         for (int num = 0; num < max; num++, val += elSize) {
            switch (datatype) {
               case gds_int8:
                  is >> *((char*) val);
                  if (*((char*) val) == 0) {
                     *((char*) val) = ' ';
                  }
                  break;
               case gds_int16:
                  is >> *((short*) val);
                  break;
               case gds_int32:
                  is >> *((int*) val);
                  break;
               case gds_float32:
                  is >> *((float*) val);
                  break;
               case gds_int64:
                  is >> *((long long*) val);
                  break;
               case gds_float64:
                  is >> *((double*) val);
                  break;
               case gds_complex32:
                  float	re, im;
                  is >> re >> im;
                  *(complex<float>*) val = complex<float> (re, im);
                  break;
               case gds_complex64:
                  double	dre, dim;
                  is >> dre >> dim;
                  *(complex<double>*) val = complex<double> (dre, dim);
                  break;
               case gds_bool:
                  is >> datum;
                  *((bool*) val) = (datum.size() > 0) && 
                     ((datum[0] == 't') || (datum[0] == 'y') || 
                     (datum[0] == 'T') || (datum[0] == 'Y') || 
                     (datum[0] == '1'));
                  break;
               case gds_time:
                  {
                     is >> datum;
                     // make a 64 bit number out of the time
                     string::size_type pos = datum.find ('.');
                     if (pos != string::npos) {
                        if (datum.size() - pos < 10) {
                           datum.insert (datum.size(), 
                                        pos + 10 - datum.size(), '0');
                        }
                        else if (datum.size() - pos > 10) {
                           datum.erase (pos + 10);
                        }
                        datum.erase (pos, 1);
                     }
                     else if (datum.size() < 12) {
                        datum += "000000000";
                     }
                     // now read it
                     char* p = readValues (gds_int64, datum, 1);
                     if (p != 0) {
                        unsigned long sec = *((long long*) p) / 1000000000;
                        unsigned long nsec = *((long long*) p) % 1000000000;
                        *(Time*)val = Time (sec, nsec);
                        delete [] p;
                     }
                     else {
                        *(Time*)val = Time (0, 0);
                     }
                     break;
                  }
               default: 
                  break;
            }
            if (!is) {
               return value;
            }
         }
         return value;
      }
   }

//______________________________________________________________________________
   void* readTableValue (int datatype, string& text)
   {
      // get item
      string::size_type pos = 0, prev_pos = 0;
      do { // skip escaped comma \, 
         pos = text.find (',', prev_pos);
         prev_pos = pos + 1;
      } while (pos && text[pos - 1] == '\\' && pos != string::npos);
      // string::size_type pos = text.find (',', pos);
      string item;
      if (pos == string::npos) {
         item = text;
         text = "";
      }
      else {
         item.append (text, 0, pos);
         text.erase (0, pos + 1);
         while (!text.empty() && isspace (text[0])) {
            text.erase (0, 1);
         }
      }
      // read string
      if ((datatype == gds_string) || (datatype == gds_channel)) {
      	// remove extra spaces 
         while (!item.empty() && isspace (item[0])) {
            item.erase (0, 1);
         }
         while (!item.empty() && isspace (item[item.size()-1])) {
            item.erase (item.size()-1, 1);
         }
         // remove quotes
         if ((item.size() > 1) && (item[0] == '"') && 
            (item[item.size() - 1] == '"')) {
            item.erase (0, 1);
            item.erase (item.size() - 1, 1);
         
         }
      	// remove extra spaces inside of quotes
         while (!item.empty() && isspace (item[0])) {
            item.erase (0, 1);
         }
         while (!item.empty() && isspace (item[item.size()-1])) {
            item.erase (item.size()-1, 1);
         }
      
         // unescape
         string* s = new string;   // make sure we use a string
         *s = xsilUnescape (item.c_str()); // to avoid truncating a zeros
         return s;
      }
      // read numbers
      else {
         return readValues (datatype, item, 1);
      }
   }

//______________________________________________________________________________
   string GetTimeString (unsigned long sec, unsigned long nsec, 
                     bool utc = false) 
   {
      char buf[1024];
      if (utc) {
         utc_t mtime;
         TAItoUTC (sec, &mtime);
         strftime (buf, 100, "%Y-%m-%d %H:%M:%S", &mtime);
         int usec = nsec / 1000;
         if (usec != 0) {
            sprintf (buf + strlen (buf), ".%06i", usec);
            int i = strlen (buf) - 1;
            while ((i >= 0) && (buf[i] == '0')) {
               buf[i] = 0;
               i--;
            }
         }
      }
      else if ((nsec <= 0) || (nsec >= 1000000000)) {
         sprintf (buf, "%lu.0", sec);
      }
      else {
         //int usec = nsec / 1000;
         sprintf (buf, "%lu.%09lu", sec, nsec);
         int i = strlen (buf) - 1;
         while ((i >= 0) && (buf[i] == '0')) {
            buf[i] = 0;
            i--;
         }
      }
      return string (buf);
   }


//////////////////////////////////////////////////////////////////////////
//                                                                      //
// xsilIndent , xsilTagBegin, xsilTagEnd, xsilDimAttr, xsilEncodingAttr //
// xsilBase64                                                           //
//                                                                      //
// XML base object                                                      //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   std::ostream& xsilIndent::write (std::ostream& os) const 
   {
      return (os << std::setw (2 * fLevel) << " "); 
   }

//______________________________________________________________________________
   std::ostream& xsilTagBegin::write (std::ostream& os) const
   {
      return (os << "<" << fTag); 
   }

//______________________________________________________________________________
   std::ostream& xsilTagEnd::write (std::ostream& os) const 
   {
      return (os << "</" << fTag << ">"); 
   }

//______________________________________________________________________________
   std::ostream& xsilDimAttr::write (std::ostream& os) const 
   {
      if (fDim > 1) os << xmlAttrDim << fDim << xmlAttrClosing; 
      return os; 
   }

//______________________________________________________________________________
   std::ostream& xsilEncodingAttr::write (std::ostream& os) const 
   {
      return (os << xmlAttrEncoding << 
             (littleEndian() ? xmlEncodeLE : xmlEncodeBE) << 
             xmlAttrClosing); 
   }


//////////////////////////////////////////////////////////////////////////
//                                                                      //
// xsil_base, xsilHeader, xsilTrailer                                   //
//                                                                      //
// XML base object                                                      //
//                                                                      //
//////////////////////////////////////////////////////////////////////////

   std::ostream& xsil_base::write (std::ostream& os) const 
   {
      if (fName) os << xmlAttrName << fName << xmlAttrClosing;
      if (fUnit) os << xmlAttrUnit << fUnit << xmlAttrClosing;
      if (fComment) os << xmlAttrComment << fComment << xmlAttrClosing;
      return os; 
   }

//______________________________________________________________________________
   std::ostream& xsilHeader::write (std::ostream& os) const {
      return (os << xmlVersion << std::endl << 
             xmlDocType << std::endl << "<" << xmlLigoLW << ">"); }

//______________________________________________________________________________
   std::ostream& xsilTrailer::write (std::ostream& os) const 
   {
      return (os << "</" << xmlLigoLW << ">"); 
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// xsilComment                                                          //
//                                                                      //
// XML comment object                                                   //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   std::ostream& xsilComment::write (std::ostream& os) const 
   {
      os << xsilIndent (fLevel) << xsilTagBegin (xmlComment) << 
         xmlTagClosing;
      if (fName) os << xsilStringEscape (fName);
      os << xsilTagEnd (xmlComment);
      return os; 
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// xsilTime                                                             //
//                                                                      //
// XML time object                                                      //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   std::ostream& xsilTime::write (std::ostream& os, int format) const 
   {
      os << xsilIndent (fLevel) << xsilTagBegin (xmlTime);
      if (fName && (strlen (fName) > 0)) 
         os << xmlAttrName << fName << xmlAttrClosing;
      os << xmlAttrType << "GPS" << xmlAttrClosing << xmlTagClosing;
      os << GetTimeString (fSec, fNSec) << xsilTagEnd (xmlTime);
      if (format > 0) {
         string tmpname = string (fName ? fName : "") + "UTC";
         os << endl << xsilIndent (fLevel) << xsilTagBegin (xmlTime);
         os << xmlAttrName << tmpname << xmlAttrClosing;
         os << xmlAttrType << "ISO-8601" << xmlAttrClosing << xmlTagClosing;
         os << GetTimeString (fSec, fNSec, true) << xsilTagEnd (xmlTime);
      }
      return os; 
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// xsilDataBegin                                                        //
//                                                                      //
// XML data begin object                                                //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   std::ostream& xsilDataBegin::write (std::ostream& os) const 
   {
      os << xsilIndent (fLevel) << xsilTagBegin (xmlContainer);
      this->xsil_base::write (os);
      if (fType) os << xmlAttrType << fType << xmlAttrClosing; 
      os << xmlTagClosing << std::endl;
      if (fFlag) {
         os << xsilIndent (fLevel+1) << xsilTagBegin (xmlParam) <<
            xmlAttrName << "Flag" << xmlAttrClosing <<
            xmlAttrType << xsilDataTypename<const char*>() << 
            xmlAttrClosing << xmlTagClosing << fFlag << 
            xsilTagEnd (xmlParam);
      }
      return os; 
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// xsilTable                                                            //
//                                                                      //
// XML table objects                                                    //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   std::ostream& xsilTableBegin::write (std::ostream& os) const
   {
      os << xsilIndent (fLevel) << xsilTagBegin (xmlTable);
      this->xsil_base::write (os);
      if (fType) os << xmlAttrType << fType << xmlAttrClosing; 
      os << xmlTagClosing;
      return os; 
   }

//______________________________________________________________________________
   std::ostream& xsilTableEnd::write (std::ostream& os) const
   {
      os << xsilIndent (fLevel) << xsilTagEnd (xmlTable);
      return os;
   }

//______________________________________________________________________________
   std::ostream& xsilTableDataBegin::write (std::ostream& os) const
   {
      os << xsilIndent (fLevel) << xsilTagBegin (xmlStream);
      this->xsil_base::write (os);
      os << xmlTagClosing;
      return os; 
   }

//______________________________________________________________________________
   std::ostream& xsilTableDataEnd::write (std::ostream& os) const
   {
      os << xsilIndent (fLevel) << xsilTagEnd (xmlStream);
      return os;
   }

//______________________________________________________________________________
   std::ostream& xsilTableEntryDelimiter::write (std::ostream& os) const
   {
      os << ",";
      return os;
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// xsilParameter                                                        //
//                                                                      //
// XML parameter object                                                 //
//                                                                      //
//////////////////////////////////////////////////////////////////////////

template class xsilParameter<bool>;
template class xsilParameter<char>;
template class xsilParameter<short>;
template class xsilParameter<int>;
template class xsilParameter<long long>;
template class xsilParameter<float>;
template class xsilParameter<double>;
template class xsilParameter<complex<float> >;
template class xsilParameter<complex<double> >;
template class xsilParameter<const char*>;
template class xsilParameter<string>;


//////////////////////////////////////////////////////////////////////////
//                                                                      //
// xsilArray                                                            //
//                                                                      //
// XML array object                                                     //
//                                                                      //
//////////////////////////////////////////////////////////////////////////

template class xsilArray<float>;
template class xsilArray<double>;
template class xsilArray<complex<float> >;
template class xsilArray<complex<double> >;


//////////////////////////////////////////////////////////////////////////
//                                                                      //
// xsilDataEnd                                                          //
//                                                                      //
// XML data end object                                                  //
//                                                                      //
//////////////////////////////////////////////////////////////////////////

template class xsilDataEnd<float>;
template class xsilDataEnd<double>;
template class xsilDataEnd<complex<float> >;
template class xsilDataEnd<complex<double> >;


//////////////////////////////////////////////////////////////////////////
//                                                                      //
// xsilHandler                                                          //
//                                                                      //
// XML handler object                                                   //
//                                                                      //
//////////////////////////////////////////////////////////////////////////

   bool xsilHandler::CommentHandler (const std::string& comment)
   {
      fComment = comment;
      return true;
   }

//______________________________________________________________________________
template <class X, class Y>
   bool convert (Y*& y, const X* x, int n = 1) 
   {
      if (n <= 0) 
         return false;
      y = new (nothrow) Y [n];
      if (y == 0) 
         return false;
      for (int i = 0; i < n; i++) {
         y[i] = x[i];
      }
      return true;
   }

//______________________________________________________________________________
#ifdef __GNU_STDC_OLD
template <>
   bool convert (complex<float>*& y, const complex<double>* x, int n)
   {
      if (n <= 0) 
         return false;
      y = new (nothrow) complex<float> [n];
      if (y == 0) 
         return false;
      for (int i = 0; i < n; i++) {
         y[i] = complex<float> (x[i].real(), x[i].imag());
      }
      return true;
   }
#endif

//______________________________________________________________________________
   bool xsilHandler::ParameterHandler (int type, void* x, int size,
                     const attrlist& attr)
   {
      //cout << "  Parameter<" << type << "> size = " << size << endl;
      if (x == 0) {
         return false;
      }
      // set name comment and unit
      attrlist::const_iterator ni = attr.find (xmlName);
      //attrlist::const_iterator ci = attr.find (xmlComment);
      //attrlist::const_iterator ui = attr.find (xmlUnit);
      if (ni == attr.end()) {
         return false;
      }
      //fComment = (ci == attr.end()) ? string("") : ci->second;
      //fUnit = (ui == attr.end()) ? string ("") : ui->second;
   
      // pass data to handler
      short* 		i16 = 0;
      int* 		i32 = 0;
      long long*	i64 = 0;
      float* 		f32 = 0;
      double* 		f64 = 0;
      complex<float>* 	c32 = 0;
      complex<double>* 	c64 = 0;
      switch (type) {
         default:
         case gds_void:
            {
               return false;
            }
         case gds_string:
         case gds_channel: 
            {
               string val ((char*) x);
               return HandleParameter (ni->second, attr, val);
            }
         case gds_int8:
            { 
               bool ret = HandleParameter (ni->second, attr, 
                                    *(char*)x, size);
               // convert to short
               if (!ret && convert (i16, (char*)x, size)) {
                  ret = HandleParameter (ni->second, attr, *i16, size);
                  delete [] i16;
               }
               // convert to int
               if (!ret && convert (i32, (char*)x, size)) {
                  ret = HandleParameter (ni->second, attr, *i32, size);
                  delete [] i32;
               }
               // convert to long long
               if (!ret && convert (i64, (char*)x, size)) {
                  ret = HandleParameter (ni->second, attr, *i64, size);
                  delete [] i64;
               }
               return ret;
            }
         case gds_int16:
            {
               bool ret = HandleParameter (ni->second, attr, 
                                    *(short*)x, size);
               // convert to int
               if (!ret && convert (i32, (short*)x, size)) {
                  ret = HandleParameter (ni->second, attr, *i32, size);
                  delete [] i32;
               }
               // convert to long long
               if (!ret && convert (i64, (short*)x, size)) {
                  ret = HandleParameter (ni->second, attr, *i64, size);
                  delete [] i64;
               }
               return ret;
            }
         case gds_int32:
            {
               bool ret = HandleParameter (ni->second, attr, 
                                    *(int*)x, size);
               // convert to long long
               if (!ret && convert (i64, (int*)x, size)) {
                  ret = HandleParameter (ni->second, attr, *i64, size);
                  delete [] i64;
               }
               return ret;
            }
         case gds_int64:
            {
               bool ret = HandleParameter (ni->second, attr, 
                                    *(long long*)x, size);
               // convert to int
               if (!ret && convert (i32, (long long*)x, size)) {
                  ret = HandleParameter (ni->second, attr, *i32, size);
                  delete [] i32;
               }
               return ret;
            }
         case gds_float32:
            {
               bool ret = HandleParameter (ni->second, attr, 
                                    *(float*)x, size);
               // convert to double
               if (!ret && convert (f64, (float*)x, size)) {
                  ret = HandleParameter (ni->second, attr, *f64, size);
                  delete [] f64;
               }
               return ret;
            }
         case gds_float64:
            {
               bool ret = HandleParameter (ni->second, attr, 
                                    *(double*)x, size);
               // convert to float
               if (!ret && convert (f32, (double*)x, size)) {
                  ret = HandleParameter (ni->second, attr, *f32, size);
                  delete [] f32;
               }
               return ret;
            }
         case gds_complex32:
            {
               bool ret = HandleParameter (ni->second, attr, 
                                    *(complex<float>*)x, size);
               // convert to double complex
               if (!ret && convert (c64, (complex<float>*)x, size)) {
                  ret = HandleParameter (ni->second, attr, *c64, size);
                  delete [] c64;
               }
               return ret;
            }
         case gds_complex64:
            {
               bool ret = HandleParameter (ni->second, attr, 
                                    *(complex<double>*)x, size);
               // convert to float complex
               if (!ret && convert (c32, (complex<double>*)x, size)) {
                  ret = HandleParameter (ni->second, attr, *c32, size);
                  delete [] c32;
               }
               return ret;
            }
         case gds_bool:
            {
               return HandleParameter (ni->second, attr, *(bool*)x, size);
            }
      };
   
      return false;
   }

//______________________________________________________________________________
   bool xsilHandler::DataHandler (const std::string& name,
                     int type, void* x, int size, 
                     int dim1, int dim2, int dim3, int dim4)
   {
      // cout << "Data " << name << "<" << type << "> size = " << size << 
         // " dim1 = " << dim1 << " dim2 = " << dim2 << " dim3 = " << 
         // dim3 << " dim4 = " << dim4 << endl;
      if (x == 0) {
         return false;
      }
      if (type == gds_float32) {
         return HandleData (name, (float*)x, dim1, dim2, dim3, dim4);
      }
      else if (type == gds_complex32) {
         return HandleData (name, (complex<float>*)x, 
                           dim1, dim2, dim3, dim4);
      }
      else if (type == gds_float64) {
         return HandleData (name, (double*)x, dim1, dim2, dim3, dim4);
      }
      else if (type == gds_complex64) {
         return HandleData (name, (complex<double>*)x, 
                           dim1, dim2, dim3, dim4);
      }
      else {
         return false;
      }
   }

//______________________________________________________________________________
   bool xsilHandler::TableEntryHandler (int row, int col, int type, void* x)
   {
      //cout << "  Parameter<" << type << "> size = " << size << endl;
      if (x == 0) {
         return false;
      }
      // pass data to handler
      short* 		i16 = 0;
      int* 		i32 = 0;
      long long*	i64 = 0;
      float* 		f32 = 0;
      double* 		f64 = 0;
      complex<float>* 	c32 = 0;
      complex<double>* 	c64 = 0;
      switch (type) {
         default:
         case gds_void:
            {
               return false;
            }
         case gds_string:
         case gds_channel: 
            {
               return HandleTableEntry (row, col, *(string*)x);
            }
         case gds_int8:
            { 
               bool ret = HandleTableEntry (row, col, *(char*)x);
               // convert to short
               if (!ret && convert (i16, (char*)x)) {
                  ret = HandleTableEntry (row, col, *i16);
                  delete [] i16;
               }
               // convert to int
               if (!ret && convert (i32, (char*)x)) {
                  ret = HandleTableEntry (row, col, *i32);
                  delete [] i32;
               }
               // convert to long long
               if (!ret && convert (i64, (char*)x)) {
                  ret = HandleTableEntry (row, col, *i64);
                  delete [] i64;
               }
               return ret;
            }
         case gds_int16:
            {
               bool ret = HandleTableEntry (row, col, *(short*)x);
               // convert to int
               if (!ret && convert (i32, (short*)x)) {
                  ret = HandleTableEntry (row, col, *i32);
                  delete [] i32;
               }
               // convert to long long
               if (!ret && convert (i64, (short*)x)) {
                  ret = HandleTableEntry (row, col, *i64);
                  delete [] i64;
               }
               return ret;
            }
         case gds_int32:
            {
               bool ret = HandleTableEntry (row, col, *(int*)x);
               // convert to long long
               if (!ret && convert (i64, (int*)x)) {
                  ret = HandleTableEntry (row, col, *i64);
                  delete [] i64;
               }
               return ret;
            }
         case gds_int64:
            {
               bool ret = HandleTableEntry (row, col, *(long long*)x);
               // convert to int
               if (!ret && convert (i32, (long long*)x)) {
                  ret = HandleTableEntry (row, col, *i32);
                  delete [] i32;
               }
               return ret;
            }
         case gds_float32:
            {
               bool ret = HandleTableEntry (row, col, *(float*)x);
               // convert to double
               if (!ret && convert (f64, (float*)x)) {
                  ret = HandleTableEntry (row, col, *f64);
                  delete [] f64;
               }
               return ret;
            }
         case gds_float64:
            {
               bool ret = HandleTableEntry (row, col, *(double*)x);
               // convert to float
               if (!ret && convert (f32, (double*)x)) {
                  ret = HandleTableEntry (row, col, *f32);
                  delete [] f32;
               }
               return ret;
            }
         case gds_complex32:
            {
               bool ret = HandleTableEntry (row, col, 
                                    *(complex<float>*)x);
               // convert to double complex
               if (!ret && convert (c64, (complex<float>*)x)) {
                  ret = HandleTableEntry (row, col, *c64);
                  delete [] c64;
               }
               return ret;
            }
         case gds_complex64:
            {
               bool ret = HandleTableEntry (row, col, 
                                    *(complex<double>*)x);
               // convert to float complex
               if (!ret && convert (c32, (complex<double>*)x, 1)) {
                  ret = HandleTableEntry (row, col, *c32);
                  delete [] c32;
               }
               return ret;
            }
         case gds_bool:
            {
               return HandleTableEntry (row, col, *(bool*)x);
            }
         case gds_time:
            {
               return HandleTableEntry (row, col, *(Time*)x);
            }
      };
      return false;
   }


//////////////////////////////////////////////////////////////////////////
//                                                                      //
// xsilHandlerIgnore                                                    //
//                                                                      //
// XML handler object (ignore it)                                       //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   class xsilHanderIgnore : public xsilHandler {
   public:
      xsilHanderIgnore (bool ignore = false) : xsilHandler (ignore) {
      }
      virtual bool CommentHandler (const std::string& comment) {
         return false; }
      virtual bool ParameterHandler (int type, void* x, int size,
                        const attrlist& attr) {
         return false; }
      virtual bool DataHandler (const std::string& name,
                        int type, void* x, int size, 
                        int dim1, int dim2 = 0, int dim3 = 0,
                        int dim4 = 0) {
         return false; }
      virtual bool TableEntryHandler (int row, int col, int type, 
                        void* x) {
         return false; }
   };



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// << attrlist		                                                //
//                                                                      //
// Output operator for attribute list                                   //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   ostream& operator<< (ostream& os, const xsilHandler::attrlist& attr) 
   {
      for (xsilHandler::attrlist::const_iterator i = attr.begin();
          i != attr.end(); i++) {
         //if (i != attr.begin()) os << " ";
         os << " " << i->first << "=\"" << i->second << "\"";
      }
      return os;
   }


//////////////////////////////////////////////////////////////////////////
//                                                                      //
// xsilHandlerUnknown                                                   //
//                                                                      //
// Handles unknown records                                              //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   xsilHandlerUnknown::xsilHandlerUnknown (std::ostream& os, 
                     const attrlist* attr, bool ignore)
   : xsilHandler (ignore), fOs (&os), fTrailer (attr != 0),
   fComplex (false), fDouble (false), fData (0)
   {
      fDim[0] = 0; fDim[1] = 0; fDim[2] = 0; fDim[3] = 0;
      if (fTrailer) {
         *fOs << xsilIndent (1) << xsilTagBegin (xmlContainer);
         *fOs << (*attr);
         *fOs << xmlTagClosing << endl;
      }
   }

//______________________________________________________________________________
   xsilHandlerUnknown::~xsilHandlerUnknown()
   {
      if (fData == 0) {
         xsilDataEnd<float> ().write (*fOs, fTrailer);
      }
      else if (fComplex) {
         if (fDouble) {
            xsilDataEnd<complex<double> > d
               (fDim[0], fDim[1], fDim[2], fDim[3], 
               (complex<double>*)fData);
            d.write (*fOs, fTrailer);
         }
         else {
            xsilDataEnd<complex<float> > d
               (fDim[0], fDim[1], fDim[2], fDim[3], 
               (complex<float>*)fData);
            d.write (*fOs, fTrailer);
         }
      }
      else {
         if (fDouble) {
            xsilDataEnd<double> d (fDim[0], fDim[1], fDim[2], fDim[3], 
                                 (double*)fData);
            d.write (*fOs, fTrailer);
         }
         else {
            xsilDataEnd<float> d (fDim[0], fDim[1], fDim[2], fDim[3], 
                                 (float*)fData);
            d.write (*fOs, fTrailer);
         }
      }
      *fOs << endl;
      if (fData) delete [] (char*) fData;
   }

//______________________________________________________________________________
   bool xsilHandlerUnknown::CommentHandler (
                     const std::string& comment)
   {
      *fOs << xsilComment (comment) << endl;
      return true;
   }

//______________________________________________________________________________
   bool xsilHandlerUnknown::HandleParameter (const std::string& name,
                     const attrlist& attr, const bool& p, int N)
   {
      attrlist::const_iterator ui = attr.find (xmlUnit);
      const char* u = (ui == attr.end()) ? 0 : ui->second.c_str();
      *fOs << xsilParameter<bool> (name.c_str(), u, p, N) << endl;
      return true;
   }

//______________________________________________________________________________
   bool xsilHandlerUnknown::HandleParameter (const std::string& name,
                     const attrlist& attr, const char& p, int N)
   {
      attrlist::const_iterator ui = attr.find (xmlUnit);
      const char* u = (ui == attr.end()) ? 0 : ui->second.c_str();
      *fOs << xsilParameter<char> (name.c_str(), u, p, N) << endl;
      return true;
   }

//______________________________________________________________________________
   bool xsilHandlerUnknown::HandleParameter (const std::string& name,
                     const attrlist& attr, const short& p, int N)
   {
      attrlist::const_iterator ui = attr.find (xmlUnit);
      const char* u = (ui == attr.end()) ? 0 : ui->second.c_str();
      *fOs << xsilParameter<short> (name.c_str(), u, p, N) << endl;
      return true;
   }

//______________________________________________________________________________
   bool xsilHandlerUnknown::HandleParameter (const std::string& name,
                     const attrlist& attr, const int& p, int N)
   {
      attrlist::const_iterator ui = attr.find (xmlUnit);
      const char* u = (ui == attr.end()) ? 0 : ui->second.c_str();
      *fOs << xsilParameter<int> (name.c_str(), u, p, N) << endl;
      return true;
   }

//______________________________________________________________________________
   bool xsilHandlerUnknown::HandleParameter (const std::string& name,
                     const attrlist& attr, const long long& p, int N)
   {
      attrlist::const_iterator ui = attr.find (xmlUnit);
      const char* u = (ui == attr.end()) ? 0 : ui->second.c_str();
      *fOs << xsilParameter<long long> (name.c_str(), u, p, N) << endl;
      return true;
   }

//______________________________________________________________________________
   bool xsilHandlerUnknown::HandleParameter (const std::string& name,
                     const attrlist& attr, const float& p, int N)
   {
      attrlist::const_iterator ui = attr.find (xmlUnit);
      const char* u = (ui == attr.end()) ? 0 : ui->second.c_str();
      *fOs << xsilParameter<float> (name.c_str(), u, p, N) << endl;
      return true;
   }

//______________________________________________________________________________
   bool xsilHandlerUnknown::HandleParameter (const std::string& name,
                     const attrlist& attr, const double& p, int N)
   {
      attrlist::const_iterator ui = attr.find (xmlUnit);
      const char* u = (ui == attr.end()) ? 0 : ui->second.c_str();
      *fOs << xsilParameter<double> (name.c_str(), u, p, N) << endl;
      return true;
   }

//______________________________________________________________________________
   bool xsilHandlerUnknown::HandleParameter (const std::string& name,
                     const attrlist& attr, const complex<float>& p, 
                     int N)
   {
      attrlist::const_iterator ui = attr.find (xmlUnit);
      const char* u = (ui == attr.end()) ? 0 : ui->second.c_str();
      *fOs << xsilParameter<complex<float> > (name.c_str(), u, p, N) << endl;
      return true;
   }

//______________________________________________________________________________
   bool xsilHandlerUnknown::HandleParameter (const std::string& name,
                     const attrlist& attr, const complex<double>& p, 
                     int N)
   {
      attrlist::const_iterator ui = attr.find (xmlUnit);
      const char* u = (ui == attr.end()) ? 0 : ui->second.c_str();
      *fOs << xsilParameter<complex<double> > (name.c_str(), u, p, N) << endl;
      return true;
   }

//______________________________________________________________________________
   bool xsilHandlerUnknown::HandleParameter (const std::string& name,
                     const attrlist& attr, const string& p)
   {
      attrlist::const_iterator ui = attr.find (xmlUnit);
      const char* u = (ui == attr.end()) ? 0 : ui->second.c_str();
      *fOs << xsilParameter<const char*> (name.c_str(), u, p.c_str()) << endl;
      return true;
   }

//______________________________________________________________________________
   bool xsilHandlerUnknown::HandleTime (const std::string& name,
                     const attrlist& attr,
                     unsigned long sec, unsigned long nsec)
   {
      *fOs << xsilTime (name.c_str(), sec, nsec) << endl;
      return true;
   }

//______________________________________________________________________________
   bool xsilHandlerUnknown::HandleData (const std::string& name,
                     float* x, int dim1, int dim2, int dim3, int dim4) 
   {
      if (!name.empty()) {
         xsilArray<float> (name.c_str(), dim1, dim2, dim3, 
                          dim4, x).write (*fOs, fTrailer);
         *fOs << endl;
         return false;
      }
      else {
         if (fData) delete [] (char*) fData;
         fDim[0] = dim1;
         fDim[1] = dim2;
         fDim[2] = dim3;
         fDim[3] = dim4;
         fData = x;
         fDouble = false;
         fComplex = false;
         return true; 
      }
   }

//______________________________________________________________________________
   bool xsilHandlerUnknown::HandleData (const std::string& name,
                     std::complex<float>* x, 
                     int dim1, int dim2, int dim3, int dim4) 
   {
      if (!name.empty()) {
         xsilArray<complex<float> > (name.c_str(), dim1, dim2, dim3, 
                              dim4, x).write (*fOs, fTrailer);
         *fOs << endl;
         return false;
      }
      else {
         if (fData) delete [] (char*) fData;
         fDim[0] = dim1;
         fDim[1] = dim2;
         fDim[2] = dim3;
         fDim[3] = dim4;
         fData = (float*)x;
         fDouble = false;
         fComplex = true;
         return true; 
      }
   }

//______________________________________________________________________________
   bool xsilHandlerUnknown::HandleData (const std::string& name,
                     double* x, int dim1, int dim2, int dim3, int dim4) 
   {
      if (!name.empty()) {
         xsilArray<double> (name.c_str(), dim1, dim2, dim3, 
                           dim4, x).write (*fOs, fTrailer);
         *fOs << endl;
         return false;
      }
      else {
         if (fData) delete [] (char*) fData;
         fDim[0] = dim1;
         fDim[1] = dim2;
         fDim[2] = dim3;
         fDim[3] = dim4;
         fData = x;
         fDouble = true;
         fComplex = false;
         return true;
      }
   }

//______________________________________________________________________________
   bool xsilHandlerUnknown::HandleData (const std::string& name,
                     std::complex<double>* x, 
                     int dim1, int dim2, int dim3, int dim4) 
   {
      if (!name.empty()) {
         xsilArray<complex<double> > (name.c_str(), dim1, dim2, dim3, 
                              dim4, x).write (*fOs, fTrailer);
         *fOs << endl;
         return false;
      }
      else {
         if (fData) delete [] (char*) fData;
         fDim[0] = dim1;
         fDim[1] = dim2;
         fDim[2] = dim3;
         fDim[3] = dim4;
         fData = (float*)x;
         fDouble = true;
         fComplex = true;
         return true; 
      }
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// xsilHandlerQueryUnknown                                              //
//                                                                      //
// Handler query for unknown data objects                               //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   xsilHandler* xsilHandlerQueryUnknown::GetHandler (const attrlist& attr) 
   {
      attrlist::const_iterator ni = attr.find (xmlName);
      if (fOs && (ni != attr.end()) &&
         (strncasecmp (ni->second.c_str(), "Index", 5) != 0)) {
         return new xsilHandlerUnknown (*fOs, &attr, false);
      }
      else {
         return 0;
      }
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// xsilHandlerTemp                                                      //
//                                                                      //
// XML temporary handler (used by parser)                               //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   bool xsilHandlerTemp::FlushTableEntries (bool comma)
   {
      if ((fHandler == 0) || fColumns.empty()) {
         return false;
      }
      // ignore text until first comma
      if (fIgnoreNextComma) {
         string::size_type pos = fText.find (',');
         if (pos == string::npos) {
            if (!comma) {
               fText = "";
            }
            return true; // nothing to do
         }
         fText.erase (0, pos + 1);
         fIgnoreNextComma = false;
      }
      // Check for comma
      string left;
      if (comma) {
         string::size_type pos = fText.rfind (',');
         if (pos == string::npos) {
            return true; // nothing to do
         }
         left.assign (fText, pos + 1, string::npos);
         //fText.erase (pos);
      }
      // Call column handlder with index -1 if first entry
      if ((fColumn == 0) && (fRow == 0)) {
         fHandler->HandleTableColumn (-1, "", gds_void, attrlist());
      }
      // Read table entries from comma delimited stream
      while (!fText.empty() && (fHandler != 0)) {
         int type = fColumns[fColumn];
         void* p = readTableValue (type, fText);
         if (p) {
            fHandler->TableEntryHandler (fRow, fColumn, type, p);
            if ((type == gds_string) || (type == gds_channel)) {
               delete (string*) p;
            }
            else {
               delete [] (char*)p;
            }
            ++fColumn;
            if (fColumn >= (int)fColumns.size()) {
               fColumn = 0;
               ++fRow;
            }
         }
      }
      fText = left;
      return true;
   }

//______________________________________________________________________________
   bool xsilHandlerTemp::SetTableEntry()
   {
      if ((fHandler == 0) || fColumns.empty()) {
         return false;
      }
      // Call column handlder with index -1 if first entry
      if ((fColumn == 0) && (fRow == 0)) {
         fHandler->HandleTableColumn (-1, "", gds_void, attrlist());
      }
      if (!fHandler->TableEntryHandler (fRow, fColumn, gds_table, 0)) {
         return false;
      }
      ++fColumn;
      if (fColumn >= (int)fColumns.size()) {
         fColumn = 0;
         ++fRow;
      }
      fIgnoreNextComma = true;
      return true;
   }


//////////////////////////////////////////////////////////////////////////
//                                                                      //
// xsilParser                                                           //
//                                                                      //
// XML parser object                                                    //
//                                                                      //
//////////////////////////////////////////////////////////////////////////

   class xsilHandlerQueryDef : public xsilHandlerQuery {
   protected:
      bool		fIgnore;
   public:
      explicit xsilHandlerQueryDef (bool ignore) : fIgnore (ignore) {
      }
      virtual xsilHandler* GetHandler (const attrlist& attr) {
         return new xsilHanderIgnore (fIgnore); }
      virtual xsilHandler* GetTableHandler (const attrlist& attr) {
         return new xsilHanderIgnore (fIgnore); }
   };

   static xsilHandlerQueryDef DefHandler (false);
   static xsilHandlerQueryDef IgnAllHandler (true);

//______________________________________________________________________________
   xsilHandlerQuery& xsilParser::DefaultHandler() 
   {
      return DefHandler;
   }

//______________________________________________________________________________
   xsilHandlerQuery& xsilParser::IgnoreAllHandler()
   {
      return IgnAllHandler;
   }

//______________________________________________________________________________
extern "C" 
   void xsilParserStartelement (xsilParser* p, const char* name, 
                     const char** attributes)
   {
      // reformat attributes
      xsilParser::attrlist atts;
      const char* const* a = attributes;
      while ((*a != 0) && (*(a+1) != 0)) {
         atts.insert (xsilParser::attrlist::value_type 
                     (string (*a), string (*(a+1))));
         a += 2;
      }
      p->Startelement (name, atts);
   }

//______________________________________________________________________________
extern "C"
   void xsilParserTexthandler (xsilParser* p, const char* text,
                     int len)
   {
      p->Texthandler (text, len);
   }

//______________________________________________________________________________
extern "C"
   void xsilParserEndelement (xsilParser* p, const char* name)
   {
      p->Endelement (name);
   }

//______________________________________________________________________________
   xsilParser::xsilParser () : fXML (0)
   {
      Init ();
   }

//______________________________________________________________________________
   xsilParser::~xsilParser ()
   {
      Done();
   }

//______________________________________________________________________________
   void xsilParser::Init()
   {
      fIgnore = 0;
      fEndXML = false;
      fWork.clear();
      if (fXML) {
         return;
      }
      fXML = (void*) XML_ParserCreate (NULL);
      if (fXML) {
         XML_SetUserData ((XML_Parser)fXML, this);
         XML_SetElementHandler 
            ((XML_Parser)fXML, 
            (XML_StartElementHandler) xsilParserStartelement, 
            (XML_EndElementHandler) xsilParserEndelement);
         XML_SetCharacterDataHandler 
            ((XML_Parser)fXML, 
            (XML_CharacterDataHandler) xsilParserTexthandler);
      }
   }

//______________________________________________________________________________
   void xsilParser::Done()
   { 
      char	   	buf[1];
      if (fXML) {
         XML_Parse ((XML_Parser)fXML, buf, 0, 1);
         XML_ParserFree ((XML_Parser)fXML);
         fXML= 0;
      }
   }

//______________________________________________________________________________
   void xsilParser::AddHandler (xsilHandlerQuery& handler)
   {
      fHandler.push_back (&handler);
   }

//______________________________________________________________________________
   bool xsilParser::Parse (const char* p, int len)
   {
      if (!fXML) {
         Init();
         if (!fXML) {
            return false;
         }
      }
      int err = XML_Parse ((XML_Parser)fXML, p, len, 0);
      return (err != 0);
   }

//______________________________________________________________________________
   bool xsilParser::Parse (std::istream& is)
   { 
      const int		max_line = 1024;
      char		line[max_line];
      string 		line2;
   
      while (!is.eof()) {
         //is.getline (line, max_line);
         getline (is, line2);
         fNewLine = true;
         if (!Parse (line2.c_str(), line2.size())) {
            return false;
         }
         if (fEndXML) {
            return true;
         }
      
         // fast optimization for data objects which are
         // base64 or uu encoded (bypass parser)
         if (!fWork.empty() &&
            (fWork.back().fStatus == xsilHandlerTemp::arraystream) &&
            (fWork.back().fEncoding >= 0) &&
            (fWork.back().fText.empty()) &&
            (fWork.back().fDim.size() > 0) &&
            (fWork.back().fDim.size() <= 2)) {
            fWork.back().fSize = fWork.back().fDim[0];
            if (fWork.back().fDim.size() > 1) {
               fWork.back().fSize *= fWork.back().fDim[1];
            }
            if (fWork.back().fSize <= 0) {
               continue;
            }
            int elSize = gdsDatumSize (fWork.back().fDataType);
            bool cmplx = gdsDatumComplex (fWork.back().fDataType);
            int size = fWork.back().fSize * elSize;
            int maxsize = 4 * size / 3 + 100 + 2*max_line;
            char* code = new (nothrow) char [maxsize]; // stores coded data
            int cur = 0; 		// current position
            int len;			// line length
            const char* table = (fWork.back().fEncoding % 2== 0) ?
               itable_base64 : itable_uuencode;
            // read encoded data
            do {
               is.getline (line, max_line);
               len = strlen (line);
               if ((len > 0) && (line[len-1] == '\r')) { // DOS
                  line[len-1] = 0;
                  --len;
               }
               if (cur + len < maxsize) {
                  strcpy (code + cur, line);
                  cur += len;
               }
            } while (is && (table[int(line[0]) & 0xff] != '\377') && 
                    (cur + max_line < maxsize));
            // now check if this is </stream>
            if (!is || (strstr (line, xmlStream) == 0)) {
               // fast read failed: read like normal
               if (!Parse (code, cur)) {
                  delete [] code;
                  return false;
               }
               delete [] code;
               continue;
            }
            // looks ok: decode
            char* p = new (nothrow) char [size];
            if ((p != 0) &&
               base64decode (code, cur, p, size, fWork.back().fEncoding % 2)) {
               if ((!littleEndian() && (fWork.back().fEncoding >= 1000)) ||
                  (littleEndian() && (fWork.back().fEncoding < 1000))) {
                  if (cmplx) {
                     swapByteOrder (p, 2 * fWork.back().fSize, elSize / 2);
                  }
                  else {
                     swapByteOrder (p, fWork.back().fSize, elSize);
                  }
               }
               fWork.back().SetData (p);
               delete [] code;
               // finally parse </stream> line here
               if (!Parse (line, len)) {
                  return false;
               }   
            }
            // failed try slow
            else {
               delete [] p;
               // fast read failed: read like normal
               if (!Parse (code, cur)) {
                  delete [] code;
                  return false;
               }
               delete [] code;
            }
         }
      }
      return fEndXML;
   }

//______________________________________________________________________________
   bool xsilParser::ParseFile (const char* filename)
   {
   #ifdef __NOMEMMAPFILE
      // open file
      ifstream inp (filename);
      if (!inp) {
         return false;
      }
      // parse it
      else {
         bool ret = Parse (inp);
         Done();
         return ret;
      }
   #else
      // open file
      int fd = open (filename, O_RDONLY);
      if (fd == -1) {
         return false;
      }
      // get length of file
      int len = lseek (fd, 0, SEEK_END);
      if (len == -1) {
         close (fd);
         return false;
      }
      // map file into memory
      char* addr = (char*)mmap (0, len, PROT_READ, MAP_PRIVATE, fd, 0);
      close (fd);
      if (addr == MAP_FAILED) {
         return false;
      }
      // parse it
      bool ret = Parse (addr, len);
      Done();
      munmap (addr, len);
      return ret;
   #endif
   }

//______________________________________________________________________________
   void xsilParser::Startelement (const char* name, const attrlist& attr) 
   { 
      // ignore invalid xsil syntax 
      if (fIgnore > 0) {
         fIgnore++;
      }
      
      // data object
      else if ((strcmp (name, xmlContainer) == 0) ||
              ldas_container (name)) {
         xsilHandler* handler = 0;
         // look for handler unless this is the global container
         if (!fWork.empty()) {
            // first check if nested
            if (fWork.back().fHandler != 0) {
               handler = fWork.back().fHandler->GetHandler (attr);
            }
            // second loop through list
            handlerquerylist::iterator i = fHandler.begin();
            while ((handler == 0) && (i != fHandler.end())) {
               handler = (*i)->GetHandler (attr);
               i++;
            }
         }
         // If none found use default handler
         if (handler == 0) {
            handler = DefaultHandler().GetHandler (attr);
            if (handler == 0) {
               // error
               return;
            }
         }
         // push handler on stack
         fWork.push_back (xsilHandlerTemp (handler));
      }
      
      // ignore invalid xsil syntax 
      else if (fWork.empty()) {
         fIgnore++;
      }
      
      // table object
      else if (strcmp (name, xmlTable) == 0) {
         xsilHandler* handler = 0;
         // look for handler unless this is the global container
         // first check if nested
         if (fWork.back().fHandler != 0) {
            handler = fWork.back().fHandler->GetTableHandler (attr);
            // Check if table-in-table
            if (fWork.back().fIsTable &&
               (fWork.back().fStatus == xsilHandlerTemp::stream)) {
               // flush table values first
               fWork.back().FlushTableEntries (true);
               fWork.back().fText = "";
               // call table handler
               if (!fWork.back().SetTableEntry()) {
                  handler = 0; // table-in-table not supported
               }
            }
         }
         // second loop through list
         handlerquerylist::iterator i = fHandler.begin();
         while ((handler == 0) && (i != fHandler.end())) {
            handler = (*i)->GetTableHandler (attr);
            i++;
         }
         // If none found use default table handler
         if (handler == 0) {
            handler = DefaultHandler().GetTableHandler (attr);
            if (handler == 0) {
               // error
               return;
            }
         }
         // push table handler on stack
         fWork.push_back (xsilHandlerTemp (handler, true));
      }
      
      // parameter object 
      else if (strcmp (name, xmlParam) == 0) {
         // ignore invalid xsil syntax 
         if (fWork.back().fIsTable ||
            (fWork.back().fStatus != xsilHandlerTemp::normal) ||
            (attr.find (xmlName) == attr.end())) {
            fIgnore++;
         }
         // new parameter
         else {
            fWork.back().fStatus = xsilHandlerTemp::param;
            fWork.back().fPrmAttr = attr;
            fWork.back().fPrmName = attr.find (xmlName)->second;
            attrlist::const_iterator ti = attr.find (xmlType);
            attrlist::const_iterator li = attr.find (xmlDim);
            fWork.back().fPrmType = (ti != attr.end()) ?
               gdsNameDataType (ti->second.c_str()) : gds_void;
            int dim = 1;
            if ((fWork.back().fPrmType != gds_string) && 
               (fWork.back().fPrmType != gds_void) &&
               (li != attr.end())) {
               dim = atoi (li->second.c_str());
               if (dim < 1) {
                  dim = 1;
               }
            }
            fWork.back().fPrmSize = dim;
            fWork.back().fText = "";
         }
      }
      
      // time object
      else if (strcmp (name, xmlTime) == 0) {
         // ignore invalid xsil syntax 
         if (fWork.back().fIsTable ||
            (fWork.back().fStatus != xsilHandlerTemp::normal)) {
            fIgnore++;
         }
         // new time parameter
         else {
            fWork.back().fStatus = xsilHandlerTemp::time;
            fWork.back().fPrmAttr = attr;
            fWork.back().fPrmName = (attr.find (xmlName) == attr.end()) ?
               xmlTime : attr.find (xmlName)->second.c_str();
            attrlist::const_iterator ti = attr.find (xmlType);
            //attrlist::const_iterator li = attr.find (xmlDim);
            fWork.back().fPrmType = 
               ((ti != attr.end()) && (ti->second == "GPS")) ? 
               gds_int64 : gds_string;
            fWork.back().fPrmSize = 1;
            fWork.back().fText = "";
         }
      }
      
      // column object
      else if (strcmp (name, xmlColumn) == 0) {
         // ignore invalid xsil syntax 
         if (!fWork.back().fIsTable ||
            (fWork.back().fStatus != xsilHandlerTemp::normal)) {
            fIgnore++;
         }
         // new column
         else {
            // name & attr
            fWork.back().fStatus = xsilHandlerTemp::column;
            fWork.back().fPrmAttr = attr;
            fWork.back().fPrmName = "";
            attrlist::const_iterator ni = attr.find (xmlName);
            if (ni != attr.end()) {
               fWork.back().fPrmName = ni->second.c_str();
            }
            // type
            fWork.back().fPrmType = gds_void;
            attrlist::const_iterator ti = attr.find (xmlType);
            if (ti != attr.end()) {
               fWork.back().fPrmType = gdsNameDataType (ti->second.c_str());
               if (fWork.back().fPrmType == gds_void) {
                  fWork.back().fPrmType = ldasNameDataType (ti->second.c_str());
               }
            }
            // size & text
            fWork.back().fPrmSize = 1;
            fWork.back().fText = "";
         }
      }
      
      // comment
      else if (strcmp (name, xmlComment) == 0) {
          // ignore invalid xsil syntax 
         if (fWork.back().fStatus != xsilHandlerTemp::normal) {
            fIgnore++;
         }
         else {
            fWork.back().fStatus = xsilHandlerTemp::comment;
         }
      }
      
      // array object
      else if (strcmp (name, xmlArray) == 0) {
         // ignore invalid xsil syntax 
         if (fWork.back().fIsTable ||
            (fWork.back().fStatus != xsilHandlerTemp::normal) ||
            (attr.find (xmlType) == attr.end())) {
            fIgnore++;
         }
         // new data array
         else {
            fWork.back().fStatus = xsilHandlerTemp::array;
            //attrlist::const_iterator ti = attr.find (xmlType);
            fWork.back().fDataType = 
               gdsNameDataType (attr.find (xmlType)->second.c_str());
            fWork.back().fDataName = "";
            if (attr.find (xmlName) != attr.end()) {
               fWork.back().fDataName = attr.find (xmlName)->second;
            }
            fWork.back().fDim.clear();
         }
      }
      
      // dimension object
      else if (strcmp (name, xmlDim) == 0) {
         // ignore invalid xsil syntax 
         if (fWork.back().fIsTable ||
            fWork.back().fStatus != xsilHandlerTemp::array) {
            fIgnore++;
         }
         else {
            fWork.back().fStatus = xsilHandlerTemp::arraydim;
            fWork.back().fText = "";
         }
      }
      
      // stream object
      else if (strcmp (name, xmlStream) == 0) {
         // distinguish table from array
         if (fWork.back().fIsTable) {
            fWork.back().fEncoding = -1;
            attrlist::const_iterator ti = attr.find (xmlType);
            attrlist::const_iterator di = attr.find (xmlDelimiter);
           // ignore invalid xsil syntax
            if ((fWork.back().fStatus != xsilHandlerTemp::normal) ||
               ((ti != attr.end()) && (ti->second != "Local")) ||
               ((di != attr.end()) && (di->second != ","))) {
               fIgnore++;
            }
            else {
               fWork.back().fStatus = xsilHandlerTemp::stream;
            }
         }
         else {
            // ignore invalid xsil syntax
            if ((fWork.back().fStatus != xsilHandlerTemp::array) ||
               (attr.find (xmlEncode) == attr.end())) {
               fIgnore++;
            }
            else {
               attrlist::const_iterator ti = attr.find (xmlType);
               attrlist::const_iterator ei = attr.find (xmlEncode);
               if (ei->second.find ("uuencode") != string::npos) {
                  fWork.back().fEncoding = 1;
               }
               else if (ei->second.find ("base64") != string::npos) {
                  fWork.back().fEncoding = 0;
               }
               else {
                  fWork.back().fEncoding = -1;
               }
               if (ei->second.find ("LittleEndian") != string::npos) {
                  fWork.back().fEncoding += 1000;
               }
               else if (ei->second.find ("BigEndian") == string::npos) {
                  fWork.back().fEncoding = -1;
               }
               if ((fWork.back().fEncoding == -1) ||
                  ((ti != attr.end()) && (ti->second != "Local"))) {
                  fIgnore++;
               }
               else {
                  fWork.back().fStatus = xsilHandlerTemp::arraystream;
                  fWork.back().SetData (0);
                  fWork.back().fText = "";
               }
            }
         }
      }
      
      // others: ignore
      else {
         fIgnore++;
      }
   }

//______________________________________________________________________________
   void xsilParser::Texthandler (const char* p, int size)
   { 
      if (fWork.empty() || (fIgnore > 0)) {
         return;
      }
      // no text allowed for normal and array
      if ((fWork.back().fStatus == xsilHandlerTemp::normal) ||
         (fWork.back().fStatus == xsilHandlerTemp::array)) {
         return;
      }
      // append text
      if (!fWork.back().fText.empty() && fNewLine) {
         fWork.back().fText += "\n";
      }
      fWork.back().fText.append (p, size);
      fNewLine = false;
      // flush table entries if text larger than 100kB
      if (fWork.back().fIsTable &&
         (fWork.back().fStatus == xsilHandlerTemp::stream) &&
         (fWork.back().fText.size() > 100000)) {
         fWork.back().FlushTableEntries (true);
      }
   }

//______________________________________________________________________________
   void xsilParser::Endelement (const char* name)
   {
      if (fWork.empty()) {
         return;
      }
      if (fIgnore > 0) {
         fIgnore--; 
         return;
      }
   
      // data object
      if ((strcmp (name, xmlContainer) == 0) ||
         ldas_container (name)) {
         fWork.pop_back();
         if (fWork.empty()) {
            fEndXML = true;
         }
      }
      
      // table object
      else if (strcmp (name, xmlTable) == 0) {
         fWork.pop_back();
      }
      
      // parameter object 
      else if (strcmp (name, xmlParam) == 0) {
         // read parameter value
         if (fWork.back().fHandler != 0) {
            char* p = readValues (fWork.back().fPrmType, 
                                 fWork.back().fText, fWork.back().fPrmSize);
            if (p != 0) {
               fWork.back().fHandler->ParameterHandler
                  (fWork.back().fPrmType, p, fWork.back().fPrmSize,
                  fWork.back().fPrmAttr);
            }
            delete [] p;
         }
         fWork.back().fStatus = xsilHandlerTemp::normal;
         fWork.back().fText = "";
      }
      
      // time object
      else if (strcmp (name, xmlTime) == 0) {
         // make sure we have a 64 bit number
         if (fWork.back().fPrmType == gds_int64) {
            string::size_type pos = fWork.back().fText.find ('.');
            if (pos != string::npos) {
               if (fWork.back().fText.size() - pos < 10) {
                  fWork.back().fText.insert 
                     (fWork.back().fText.size(),
                     pos + 10 - fWork.back().fText.size(), '0');
               }
               else if (fWork.back().fText.size() - pos > 10) {
                  fWork.back().fText.erase (pos + 10);
               }
               fWork.back().fText.erase (pos, 1);
            }
            else if (fWork.back().fText.size() < 12) {
               fWork.back().fText += "000000000";
            }
         }
         // now read time value
         if (fWork.back().fHandler != 0) {
            if (fWork.back().fPrmType == gds_int64) {
               char* p = readValues (gds_int64, fWork.back().fText, 1);
               if (p != 0) {
                  unsigned long sec = *((long long*) p) / 1000000000;
                  unsigned long nsec = *((long long*) p) % 1000000000;
                  fWork.back().fHandler->HandleTime 
                     (fWork.back().fPrmName, fWork.back().fPrmAttr, 
                     sec, nsec);
                  delete [] p;
               }
            }
            else {
               fWork.back().fHandler->ParameterHandler 
                  (gds_string, (void*) fWork.back().fText.c_str(), 
                  fWork.back().fText.size(), fWork.back().fPrmAttr);
            }
         }
         fWork.back().fStatus = xsilHandlerTemp::normal;
         fWork.back().fText = "";
      }
      
      // column object 
      else if (strcmp (name, xmlColumn) == 0) {
          // make sure we have a non-empty column name
         if (fWork.back().fPrmName.empty()) {
            char buf[64];
            sprintf (buf, "Column%zi", fWork.back().fColumns.size());
            fWork.back().fPrmName = buf;
         }
         // call column handler
         fWork.back().fHandler->HandleTableColumn
            (fWork.back().fColumns.size(), fWork.back().fPrmName,
            fWork.back().fPrmType, 
            fWork.back().fPrmAttr);
         fWork.back().fColumns.push_back (fWork.back().fPrmType);
         fWork.back().fStatus = xsilHandlerTemp::normal;
         fWork.back().fText = "";
      }
      
      // comment
      else if (strcmp (name, xmlComment) == 0) {
         if (fWork.back().fHandler != 0) {
            fWork.back().fHandler->CommentHandler (fWork.back().fText);
         }
         fWork.back().fComment = fWork.back().fText;
         fWork.back().fStatus = xsilHandlerTemp::normal;
         fWork.back().fText = "";
      }
      
      // array object
      else if (strcmp (name, xmlArray) == 0) {
         //cout << "end of array" << endl;
         if ((fWork.back().fHandler != 0) && 
            (fWork.back().fDim.size() <= 2)) {
            int dim1 = fWork.back().fDim.size() > 0 ? 
               fWork.back().fDim[0] : fWork.back().fSize;
            int dim2 = fWork.back().fDim.size() > 1 ? 
               fWork.back().fDim[1] : 0;
            int dim3 = fWork.back().fDim.size() > 2 ? 
               fWork.back().fDim[2] : 0;
            int dim4 = fWork.back().fDim.size() > 3 ? 
               fWork.back().fDim[3] : 0;
            if (fWork.back().fHandler->DataHandler
               (fWork.back().fDataName, fWork.back().fDataType, 
               fWork.back().fData, fWork.back().fSize, 
               dim1, dim2, dim3, dim4)) {
               //cout << "Don't delete data object!" << endl;
               fWork.back().fData = 0; 
            }
            else {
               fWork.back().SetData (0);
            }
         }
         fWork.back().fStatus = xsilHandlerTemp::normal;
      }
      
      // dimension object
      else if ((strcmp (name, xmlDim) == 0) && 
              (fWork.back().fStatus  == xsilHandlerTemp::arraydim)) {
         int dim = atoi (fWork.back().fText.c_str());
         if (dim > 0) {
            fWork.back().fDim.push_back (dim);
         }
         fWork.back().fStatus = xsilHandlerTemp::array;
         fWork.back().fText = "";
      }
      
      // stream object (table)
      else if ((strcmp (name, xmlStream) == 0) && 
              (fWork.back().fIsTable)) {
         fWork.back().fText += ",";
         fWork.back().FlushTableEntries ();
         fWork.back().fStatus = xsilHandlerTemp::normal;
         //cout << "end of table stream 2" << endl;
      }
      
      // stream object (array)
      else if (strcmp (name, xmlStream) == 0) {
         //cout << "end of stream" << endl;
         // convert data stream from ASCII into binary
         if ((fWork.back().fData == 0) &&
            (fWork.back().fEncoding >= 0) &&
            (fWork.back().fDim.size() > 0) &&
            (fWork.back().fDim.size() <= 2)) {
            fWork.back().fSize = fWork.back().fDim[0];
            if (fWork.back().fDim.size() > 1) {
               fWork.back().fSize *= fWork.back().fDim[1];
            }
            if (fWork.back().fSize > 0) {
               int elSize = gdsDatumSize (fWork.back().fDataType);
               bool cmplx = gdsDatumComplex (fWork.back().fDataType);
               int size = fWork.back().fSize * elSize;
               char* p = new char [size];
               if ((p != 0) &&
                  base64decode (fWork.back().fText.c_str(), 
                               fWork.back().fText.size(), p, 
                               size, fWork.back().fEncoding % 2)) {
                  if ((!littleEndian() && (fWork.back().fEncoding >= 1000)) ||
                     (littleEndian() && (fWork.back().fEncoding < 1000))) {
                     if (cmplx) {
                        swapByteOrder (p, 2 * fWork.back().fSize, elSize / 2);
                     }
                     else {
                        swapByteOrder (p, fWork.back().fSize, elSize);
                     }
                  }
                  fWork.back().SetData (p);
               }
               else {
                  delete [] p;
               }
            }
         }
         fWork.back().fStatus = xsilHandlerTemp::array;
         fWork.back().fText = "";
         //cout << "end of stream 2" << endl;
      }
   }


}
