/* -*- mode: c++; c-basic-offset: 3; -*- */
#include "web/webclient.hh"
#include <curl/curl.h>
#include <cstring>
#include <iostream>
#include <sstream>

//////////////////////////////////////////////////////////////////////////
//                                                                      //
//     Callback functions                                               //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
extern "C" {
   static void 
   gbl_init(void) {
      static bool tbd(true);
      if (tbd) {
	 tbd = false;
	 curl_global_init(CURL_GLOBAL_ALL);
      }
   }

   //===================================  Write callback function
   static size_t
   write_callback(const char* ptr, size_t size, size_t nmemb, void* thimjig) {
      size_t N = size*nmemb;
      return reinterpret_cast<web::http_request*>(thimjig)->wbuf(ptr, N);
   }
}

namespace web {
   using namespace std;

//////////////////////////////////////////////////////////////////////////
//                                                                      //
// Constants                                                            //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   const int _CONNECT_TIMEOUT = 20; // timeout for connecting to web server
   const int _TIMEOUT = 30; // timeout for web access

//////////////////////////////////////////////////////////////////////////
//                                                                      //
// http_request                                                         //
//                                                                      //
//////////////////////////////////////////////////////////////////////////

//======================================  Default constructor.
   http_request::http_request(void)
      : fData (0), fLen (0), fCurl(0), fDebug(0)
   {
      gbl_init();
   }

//______________________________________________________________________________
   http_request::http_request (const char* url, const char* addr, int port)
      : fData (0), fLen (0), fCurl(0), fDebug(0)
   {
      gbl_init();
      request (url, addr, port);
   }

   //===================================  Destructor
   http_request::~http_request() {
      close();
   }

   //===================================  Close the libcurl connection.
   void
   http_request::clear(void) {
      fFill = 0;
   }

   //===================================  Close the libcurl connection.
   void
   http_request::close(void) {
      if (fData) {
	 delete [] fData;
	 fLen = 0;
	 fData = 0;
      }
      if (fCurl) {
	 curl_easy_cleanup(fCurl);
	 fCurl = 0;
      }
   }


   //===================================  Define a set-option script
#define WEBC_SET_OPT(opt,val,msg) \
   if (!rc && (rc=curl_easy_setopt(fCurl, opt, val))) setmsg=msg

   //===================================  Open the server
   bool
   http_request::open(const char* addr, int port) {

      if (!fCurl) {
         CURLcode rc = CURLE_OK;
         string   setmsg;
	 fCurl = curl_easy_init();
	 if (!fCurl) return false;
	 WEBC_SET_OPT(CURLOPT_NOSIGNAL, 1, "no-signal");

	 //-----------------------------  Verbose error handling
	 if (fDebug > 1) {
	    WEBC_SET_OPT(CURLOPT_VERBOSE, 1, "verbose");
	 } else {
	    WEBC_SET_OPT(CURLOPT_VERBOSE, 0, "noverbose");
	 }
	 WEBC_SET_OPT(CURLOPT_WRITEFUNCTION, &write_callback, "write function");
	 WEBC_SET_OPT(CURLOPT_WRITEDATA, this, "write function argument");
	 WEBC_SET_OPT(CURLOPT_USERPWD, ":", "user/password");
	 const char* cookiefile = getenv("DMTWEBCLIENT_COOKIE");
	 if (!cookiefile) cookiefile = "/dev/null";
	 WEBC_SET_OPT(CURLOPT_COOKIEFILE, cookiefile, "cookie file");
	 if (rc != CURLE_OK) {
	    cerr << "http_request::open couldn't set " << setmsg 
		 << " error: " << curl_easy_strerror(rc) << endl;
	    return false;
	 }
      }

      //---------------------------------  Build server and protocol string.
      ostringstream my_url;
      if (port == 443) my_url << "https://";
      else             my_url << "http://";
      string ip_addr = addr;
      string::size_type pos = ip_addr.find("/");
      //---------------------------------  addr only if port is default.
      if (!port || port == 80 || port == 443) {
	 my_url << addr;
      }
      //---------------------------------  <addr>:<port> if no '/'
      else if (pos == string::npos) {
	 my_url << addr << ":" << port;
      }
      //---------------------------------  insert port number before first '/'
      else {
	 my_url << ip_addr.substr(0, pos-1) << ":" << port 
		<< ip_addr.substr(pos);
      }
      fServer = my_url.str();
      //cout << "open addr: " << addr << " port: " << port << " fServer: "
      //     << fServer << endl;
      return true;
   }

   //===================================  Request a URL
   bool 
   http_request::request(const char* url) {

      //--------------------------------  Get and check the url.
      if (!url || !*url || fServer.empty()) {
	 cerr <<  "http_request::request: URL or server not specified" << endl;
	 return false;
      }

      string my_url = fServer + "/" + url;
      if (fDebug) cout << "request URL: " << url << endl;

      //--------------------------------  Set up gss negotiation.
      CURLcode rc = CURLE_OK;
      string setmsg;

      WEBC_SET_OPT(CURLOPT_HTTPAUTH, CURLAUTH_GSSNEGOTIATE, "gss negotiation");
      WEBC_SET_OPT(CURLOPT_FOLLOWLOCATION, 1L, "follow location");
      WEBC_SET_OPT(CURLOPT_UNRESTRICTED_AUTH, 1L, "trusted location");
      WEBC_SET_OPT(CURLOPT_SSL_VERIFYPEER, 0L, "noverify peer cert");
      WEBC_SET_OPT(CURLOPT_URL, my_url.c_str(), "server URL");
      if (rc != CURLE_OK) {
	 cerr << "http_request::request couldn't set " << setmsg 
	      << " error: " << curl_easy_strerror(rc) << endl;
         return false;
      }

      //--------------------------------  Clear the input buffer and perform 
      //                                  the action.
      clear();
      rc = curl_easy_perform(fCurl);
      if (rc != CURLE_OK) {
	 cerr << "libcurl action failed on URL: " << my_url 
	      << " error: " << curl_easy_strerror(rc) << endl;
         return false;
      }
      return true;
   }

   //===================================  Full request call.
   bool 
   http_request::request(const char* url, const char* addr, int port)
   {
      //--------------------------------  Make sure that the socket is open
      if (!open(addr, port)) return false;

      //--------------------------------  Perform the request.
      return request(url);
   }

   //===================================  Set the debug print level
   void 
   http_request::set_debug(int lvl) {
      fDebug = lvl;
   }

   //===================================  Write the data to the internal buffer.
   size_t
   http_request::wbuf(const char* ptr, size_t N) {
      if (!fData) fFill = fLen = 0; // j.i.c.

      size_t nTotal = fFill + N;
      if (fLen < nTotal) {
	 fLen = nTotal + 16;
	 char* tmpbuf = new char[fLen];
	 if (fFill) memcpy(tmpbuf, fData, fFill);
	 if (fData) delete[] fData;
	 fData = tmpbuf;
      }
      memcpy(fData+fFill, ptr, N);
      fFill = nTotal;
      if (fLen > fFill) fData[fFill] = 0;

      if (fDebug) {
	 long response = 0;
	 curl_easy_getinfo(fCurl, CURLINFO_RESPONSE_CODE, &response);
	 cout << "Received data, response= " << response 
	      << " length= " << N << " total-length=" << nTotal << endl;
	 if (fDebug > 2) {
	    cout << "Data: " << endl << string(ptr, N) << endl;
	 }
      }
      return N;
   }

   //===================================  Inline hex decription
   inline int
   getxdig(int x) {
      if (x >= '0' && x <= '9') return x - '0';
      if (x >= 'a' && x <= 'f') return x - 'a' + 10;
      if (x >= 'A' && x <= 'F') return x - 'A' + 10;
      return -1;
  }

   //===================================  Static sp-char replacement function.
   string 
   http_request::mangle (const string& s)
   {
      const char* mark = "-_.!~*'()";
      const char* hex = "0123456789ABCDEF";
      string::size_type pos = 0;
      string r = s;
      while (pos < r.size()) {
         char c = r[pos];
         if (!isalnum (c) && (strchr (mark, c) == 0)) {
            string encode = 
               "%" + string (1, hex[c >> 4]) + string (1, hex[c&0x0F]);
            r.erase (pos, 1);
            r.insert (pos, encode);
         }
         ++pos;
      }
      return r;
   }

   //===================================  Static sp-char replacement function.
   string 
   http_request::demangle (const string& s)
   {
     string r = s;
     string::size_type j = 0;
     string::size_type n = r.size();
     for (string::size_type i=0; i < n; i++) {
       char decode = r[i];
       if (decode == '%' && i+2 < n) {
         int x1 = getxdig(r[i+1]);
         if (x1 >= 0) {
            int x2 = getxdig(r[i+2]);
            if (x2 >= 0) {
                decode = char(x1*16 + x2);
                i += 2;
            }
         }
       }
       r[j++] = decode;
     }
     if (j<n) r.erase(j, n-j);
     return r;
   }
}
