/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef PARSELINE_HH
#define PARSELINE_HH

#include <time.h>
#include <fstream>
#include <string>

//======================================  Line Parser class
/**  Class ParseLine reads a file and splits up each line into words 
  *  delimited by one or more blank space characters or by quotes. Lines 
  *  may be continued in the file using a backslash resulting in up to 1024 
  *  character command lines. Words from the command line can be accessed 
  *  using the getArg() or operator[] methods. The words may also be 
  *  converted to numeric values with getDouble(), getHex() or getInt().
  *
  *  A line parser is typically used in the following manner:
  *
  *  \begin{verbatim}
     ParseLine pl("command.file");
     while (pl.getLine() >= 0) {
        std::string arg1 = pl[0];
	unsigned int hex_arg = pl.getHex(1);
        ...
     }
\end{verbatim}
  *
  *  @memo Parse lines from a command file.
  *  @author J. Zweizig
  *  @version 1.3; modified June 30, 2004
  *  @ingroup IO_parsel
  */
class ParseLine {
public: 
  /**  Construct a line parser on a file.
    *  @memo Construct a parser
    *  @param file Input file name.
    */
  ParseLine(const char* file);

  /**  Construct a line parser on a stream.
    *  @memo Construct a parser
    *  @param stream Input file name.
    */
  ParseLine(std::istream& stream);

  /**  Destroy a command line parser.
    *  @memo ParseLine destructor.
    */
  ~ParseLine(void);

  /**  Read a line from the file into the internal buffer and parse it 
    *  into words. getLine() returns the number of words found on the 
    *  line. A negative number is returned if an error occurred while
    *  trying to read the line.
    *  @memo Read and parse a line.
    *  @return Number of words on line.
    */
  int getLine(void);

  /**  Get a pointer to the ith word on a line. The argument index is tested
    *  to verify that it is in the valid rage \c (0\<=i\<count). If the 
    *  argument is out of range, a null pointer is returned.
    *  @memo Get pointer to an argument.
    *  @param i Argument number.
    *  @return Constant pointer to the specified argument string or NULL.
    */
  const char* getArg(int i) const;

  /**  Get the number of words on the last line read in.
    *  @memo Get word count.
    *  @return The number of words read in.
    */
  int getCount(void) const;

  /**  Get a double float from the ith word on a line. 
    *  \brief Get the ith word converted to a float number.
    *  \param i Index of argument to be returned.
    *  \return The value of the ith word.
    */
  double getDouble(int i) const;

  /**  Get a hex number from the ith word on a line. 
    *  @memo Get a hex argument.
    *  @param i Number of argument to fetch.
    *  @return Integer argument value
    */
  unsigned long getHex(int i) const;

  /**  Get a long integer from the ith word on a line. 
    *  @memo Get an integer argument.
    *  @param i Number of argument to fetch.
    *  @return Integer argument value
    */
  long getInt(int i) const;

  /**  Get the current line number.
    */
  long getLineNumber(void) const;

  /**  Get a range of values specified as \c lo-high or \c lo:hi. The data
    *  field is two double precision values separaterd by a single character
    *  asspecified by the sep column. By default the separator character is
    *  '-'. Some characters e.g. space, numbers will cause problems. The 
    *  \c lo and \c hi variables are left unchanged if the argument number is 
    *  not valid. The \c hi variable is left unchanged if the specified 
    *  separator was not found.
    *  @memo Get a value range
    *  @param i  Number of argument to fetch.
    *  @param lo Low value of range
    *  @param hi High value of range
    *  @param sep Separation character.
    */
  void getRange(int i, double& lo, double& hi, char sep='-') const;

  /**  Test that the file is open.
    *  @memo Test if file is open.
    *  @return true if file is open and ready to be read.
    */
  bool isOpen(void) const;

  /**  Get a pointer to the ith argument string from the current line. No
    *  test for a valid input index is made.
    *  @memo Get an argument pointer.
    *  @return A pointer to the specified argument string.
    *  @param i Index (in the range 0 -> N-1) of the requested argument.
    */
  const char* operator[](int i) const;

  /**  Set comment characters. The line parsing stops at the first unquoted 
    *  comment character. By default the comment character is '#'.
    *  \brief Set comment character(s).
    *  \param st Comment character string
    */
  void setComment(const std::string& st);

  /**  Set the character translation table to its default values.
    */
  void setDefault(void);

  /**  Set the delimiter charaters. By default, space and tab are delimeters.
    *  \brief set delimiter characters
    *  \param st Delimiter character string.
    */
  void setDelim(const std::string& st);

  /**  Specify a log file to receive a transcription of the lines read 
    *  from the command file.
    *  @memo Specify a log file
    *  @param log stream to receive a transcription of the input file.
    */
  void setLog(std::ostream& log);

  /**  Set parenthesis charaters. Parentheses are netstable quote characters.
    *  Parentheses are specified in open / close parentheses pairs.
    *  By default, the "()" characters are parentheses.
    *  \brief Set parenthesis string.
    *  \param st Parenthesis string.
    */
  void setParen(const std::string& st);

  /**  Set quote characters. All charaters inside a quoted string are ignored,
    *  including delimiters, parentheses and other kinds of quotes.
    *  \brief set the Quote character.
    *  \param st Quote character string.
    */
  void setQuote(const std::string& st);

  /**  Set the escape character. By default the escape character is '\'.
    *  @brief set the escape character.
    *  \param st Escape character
    */
  void setEscape(const std::string& st);

private:
  int  appendLine(int offset);
  enum chtype {
    kDefault,
    kDelim,
    kParen,
    kQuote,
    kComment,
    kEscape
  };

private:
  static const int table_size=256;          // table_size must be a power of 2
  static const int table_mask=table_size-1;
  std::ifstream mFile;
  std::istream& mStream;
  std::ostream* mLog;
  int         mCount;
  char        mTable[table_size];
  const char* mArg[128];
  char        mLine[1024];
  long        mLineNumber;
};

//======================================  Inline functions
inline bool
ParseLine::isOpen(void) const {
    return mStream.good();
}

inline int
ParseLine::getCount(void) const {
    return mCount;
}

inline long
ParseLine::getLineNumber(void) const {
    return mLineNumber;
}

inline const char* 
ParseLine::operator[](int i) const {
    return mArg[i];
}

#endif  //   PARSELINE_HH
