/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "ParseLine.hh"
#include <stdlib.h>

using namespace std;

//======================================  Construct a line parser
ParseLine::ParseLine(const char* file) 
  : mFile(file), mStream(mFile), mLog(0), mCount(0), mLineNumber(0)
{
    setDefault();
}

//======================================  Construct a line parser
ParseLine::ParseLine(istream& stream) 
  : mStream(stream), mLog(0), mCount(0), mLineNumber(0)
{
    setDefault();
}

//======================================  Destroy a line parser
ParseLine::~ParseLine(void) {
    if (mFile.is_open()) mFile.close();
}

//======================================  Read a line
int
ParseLine::appendLine(int off) {
    int segL = mStream.getline(mLine+off, sizeof(mLine)-off).gcount();
    mLineNumber++;
    return segL;
}

//======================================  Read a line
int 
ParseLine::getLine(void) {
    mCount = -1;
    while (mStream.good() && mCount < 0) {
        int len = appendLine(0);
	while (len && (mLine[len-1] == ' ' || mLine[len-1] == '\t')) {
	    mLine[--len] = 0;
	}
	if (len > 0 && mLog) *mLog << mLine << endl;
	if (!len || *mLine == '#') continue;
	if (!mLine[len-1]) len--;
	while (len && mLine[len-1] == '\\') {
	    len--;
	    int segL = appendLine(len);
	    if (!mStream.good()) {
	        if (mLog) *mLog << "**** Missing continuation line!" << endl;
		return -1;
	    }
	    if (segL > 0 && mLog) *mLog << ">_ " << mLine+len << endl;
	    segL += len;
	    if (!mLine[segL-1]) segL--;
	    int i = len;
	    for ( ; i < segL && (mLine[i] == ' ' || mLine[i] == '\t') ; i++);
	    if (len == i) len = segL;
	    else          while (i < segL) mLine[len++] = mLine[i++];
	}
	mLine[len] = 0;

	mCount = 0;
	bool inword(false), escape(false);
	char qtype(0);
	char* insert(0);
	for (char* p=mLine ; *p ; ++p) {
	    if (qtype) {
	        if (*p == qtype && !escape) {
		    *p    = 0;
		    qtype = 0;
		} else {
		    *insert++ = *p;
		}
		continue;
	    }
	    switch (mTable[int(*p) & table_mask]) {
	    case kDelim:
	        *p=0;
		inword = false;
		break;
	    case kComment:
	        *p-- = 0;  //  slightly tricky termination
		break;
	    case kQuote:
	        qtype = *p;
		insert = p;
		*insert++ = 0;
		mArg[mCount++] = insert;
		break;
	    case kEscape:
	        escape = true;
		break;
	    default:
	        if(!inword) {
		    mArg[mCount++] = p;
		    insert = p;
		}
		inword = true;
		*insert++ = *p;
	    }
	}
    }
    return mCount;
}

//======================================  Set logging
void
ParseLine::setLog(ostream& log) {
    mLog = &log;
}

//======================================  Read a character argument
const char* 
ParseLine::getArg(int i) const {
    if (i >= 0 && i < mCount) return mArg[i];
    return 0;
}

//======================================  Read a float argument
double
ParseLine::getDouble(int i) const {
    if (i >= 0 && i < mCount) return strtod(mArg[i], 0);
    return 0.0;
}

//======================================  Read a hex argument
unsigned long
ParseLine::getHex(int i) const {
    if (i >= 0 && i < mCount) return strtol(mArg[i], 0, 16);
    return 0;
}

//======================================  Read an integer argument
long
ParseLine::getInt(int i) const {
    if (i >= 0 && i < mCount) return strtol(mArg[i], 0, 0);
    return 0;
}

//======================================  Read an integer argument
void
ParseLine::getRange(int i, double& lo, double& hi, char sep) const {
    char* p = const_cast<char*>(mArg[i]);
    if (i >= 0 && i < mCount) {
        lo = strtod(p, &p);
        if (*p++ == sep) hi = strtod(p, &p);
    }
}

//======================================  Set the default table values
void 
ParseLine::setComment(const string& st) {
    for (int i=0; i<table_size; ++i) if(mTable[i]==kComment)mTable[i]=kComment;
    int N = st.size();
    for (int i=0; i<N; ++i) {
        int j = int(st[i]) & table_mask;
	mTable[j] = kComment;
    }
}

//======================================  Set the default table values
void 
ParseLine::setDefault(void) {
    for (int i=0; i<table_size; ++i) mTable[i] = kDefault;
    setDelim(" \t\r");
    setParen("()");
    setQuote("\"'");
    setComment("#");
    setEscape("\\");
}

//======================================  Specify delimiters
void 
ParseLine::setDelim(const string& st) {
    for (int i=0; i<table_size; ++i) if (mTable[i]==kDelim) mTable[i]=kDefault;
    int N=st.size();
    for (int i=0; i<N; ++i) {
        int j = int(st[i]) & table_mask;
	mTable[j] = kDelim;
    }
}

//======================================  Specify delimiters
void 
ParseLine::setEscape(const string& st) {
    for (int i=0; i<table_size; ++i) if (mTable[i]==kEscape) mTable[i]=kDefault;
    int N=st.size();
    for (int i=0; i<N; ++i) {
        int j = int(st[i]) & table_mask;
	mTable[j] = kEscape;
    }
}

//======================================  Set parentheses
void 
ParseLine::setParen(const string& st) {
    for (int i=0; i<table_size; ++i) if (mTable[i]==kParen) mTable[i]=kDefault;
    int N=st.size();
    for (int i=0; i<N; ++i) {
        int j = int(st[i]) & table_mask;
	mTable[j] = kParen;
    }
}

//======================================  Specify quote characters
void 
ParseLine::setQuote(const string& st) {
    for (int i=0; i<table_size; ++i) if (mTable[i]==kQuote) mTable[i]=kDefault;
    int N=st.size();
    for (int i=0; i<N; ++i) {
        int j = int(st[i]) & table_mask;
	mTable[j] = kQuote;
    }
}
