/* -*- mode: c++; c-basic-offset: 4; -*- */
//
//    Shared memory partition consumer class.
//
//    Public Methods:
//        LSMP_CON()
//            Construct a null consumer (no data access)
//
//        LSMP_CON(LSMP *part, int nbuf, int mask)
//            Construct a consumer from partition *part. The consumer 
//            can allocate up to nbuf segment from *psrt and will
//            read triggeres as defined by 'mask'.
//
//        ~LSMP_CON()
//            Clean up and detach consumer.
//
//        char *get_buffer();
//            Get a pointer to a new buffer.
//
//        void free_buffer();
//            Free up an allocated buffer.
//
#include "lsmp_con.hh"
#include "lsmp_int.hh"
#include <iostream>
#include <sys/ipc.h>
#include <sys/sem.h>
#include "SysError.hh"
#include <unistd.h>

//======================================  OSX alarm handler
#ifdef P__DARWIN
#include <signal.h>
volatile int alarm_triggered=0;
void alarm_handler(int sig) {
    alarm_triggered = 1;
}
#endif

//======================================  Constructors
LSMP_CON::LSMP_CON(void) 
    : icon(-1), ibuf(-1), maxtime(-1.0)
{}

LSMP_CON::LSMP_CON(const char *name, int nbuf, int mask) 
    : LSMP(name), icon(-1), ibuf(-1), maxtime(-1.0)
{
    if (valid()) get_consumer(nbuf, mask);
}

//======================================  Destructor
LSMP_CON::~LSMP_CON() {
    if (icon >= 0) {
        if (ibuf >= 0) free_buffer();
	free_consumer(icon);
	icon = -1;
    }
}

//======================================  Accessors
LSMP::eventid_type
LSMP_CON::getEvtID(void) const {
    if (ibuf < 0) return 0;
    return bufptr[ibuf].data_ID;
}

int 
LSMP_CON::getMask(void) const {
    if (icon < 0) return -1;
    return conptr[icon].trig_mask;
}

int 
LSMP_CON::getNBuffer(void) const {
    if (icon < 0) return 0;
    return conptr[icon].mxbuf;
}

int 
LSMP_CON::getNSkip(void) const {
    if (icon < 0) return 0;
    return conptr[icon].min_sep;
}

//======================================  Set reseve buffer count.
void
LSMP_CON::setNBuffer(int nbuf) {
    if (icon < 0) return;
    if (nbuf < 0) {
        conptr[icon].mxbuf  = pointer->nbuf;
	conptr[icon].flags |= READALL;
    } else {
        conptr[icon].mxbuf  = nbuf;
	conptr[icon].flags &= ~READALL;
    }
}

//======================================  Set buffer skip count.
void
LSMP_CON::setNSkip(int nskip) {
    if (icon < 0) return;
    if (nskip <= 0) conptr[icon].min_sep = 0;
    else            conptr[icon].min_sep = nskip;
    conptr[icon].skip_ctr = 0;
}

//======================================  Set maximum data wait time.
void
LSMP_CON::setTimeout(double timeout) {
    maxtime = timeout;
}

//======================================  Get a pointer to a buffer
const char*
LSMP_CON::find_dataID(eventid_type ID) {
    if (!valid()) return (const char *) 0;
    if (ibuf >= 0) {
        std::cout << "LSMP_CON::find_dataID: Consumer already has buffer." 
		  << std::endl;
	return (const char *) 0;
    }
    ibuf = get_by_ID(ID);
    return getBuffAddr();
}

//---------------------------------------------------------------------------
//
//    int get_consumer()
//
//    Allocate a consumer slot
//
//---------------------------------------------------------------------------
void
LSMP_CON::get_consumer(int nbuf, int mask) {
    union semun setv;
    if (icon >= 0) return;

    //---------------------------------  Get access to the global region. 
#ifndef USE_BUILTIN_ATOMICS
    while(!gate(true));
#endif

    //---------------------------------  Look for a free consumer slot
    for (int i=0 ; i < LSMP_MAXCONS ; i++) {
	if (!pointer->conresrv.tset(i)) {
	    icon = i;
	    LSMP_consbk *pcon = conptr + i;
	    if (nbuf < 0) {
	        pcon->flags = READALL;
		pcon->mxbuf = pointer->nbuf;
	    } else {
	        pcon->flags = 0;
	        pcon->mxbuf = nbuf;
	    }
	    pcon->trig_mask = mask;
	    pcon->min_time  = 0;
	    pcon->min_sep   = 0;
	    pcon->time_ctr  = 0;
	    pcon->skip_ctr  = 0;
	    pcon->seg_ctr   = 0;
	    pcon->seg_tot   = 0;
	    pcon->pid       = my_procid();
	    pointer->ncons++;
#ifndef COLDREAD
#ifdef USE_BUILTIN_ATOMICS
	    while(!gate(true));
#endif
	    for (int j=pointer->full.head ; j>=0 ; j=bufptr[j].link) {
	        bufptr[j].seen_mask.set(icon);
	    }
	    setv.val = 0;
#ifdef USE_BUILTIN_ATOMICS
	    gate(false);
#endif
#else
	    setv.val = getBufferCount();
#endif
	    semctl(pointer->con_semid[CON_WORD(i)], CON_BIT(i), SETVAL, setv);

	    pointer->conmask.tset(i);
	    break;
	}
    }

    //-----------------------------------  Open the partition gate
#ifndef USE_BUILTIN_ATOMICS
    gate(false);
#endif
    return;
}

//---------------------------------------------------------------------------
//
//    int get_buffer(int flags);
//
//    Get an unread buffer for the specified consumer.
//
//    Parameter:
//      flags  Flags indicating allocation mode Or'ed from the following
//             NOWAIT return immediately if no buffers are available.
//
//    Returns:
//      -1     No flags buffer are available and NOWAIT specified or sleep 
//             interrupted by signal.
//      >= 0   Identifier of allocated buffer.
//
//---------------------------------------------------------------------------
const char*
LSMP_CON::get_buffer(int flags) {
    struct sembuf sbuf;

    //---------------------------------  Get a few useful pointers
    if (!valid()) return (const char *) 0;
    if (ibuf >= 0) {
        std::cout << "LSMP_CON::get_buffer: Consumer already has buffer." 
		  << std::endl;
	return (const char *) 0;
    }

    //---------------------------------  Get a few useful pointers
    int mywd   = CON_WORD(icon);
    int conbt  = CON_BIT(icon);
    LSMP_consbk* pcon = conptr + icon;

    //---------------------------------  Set the wait flags and timeout pointer
    int wflag = (flags & NOWAIT) ? IPC_NOWAIT : 0;
    timespec timeout, *tmax = 0;
    if (maxtime >= 0) {
	timeout.tv_sec  = long(maxtime);
	timeout.tv_nsec = long((maxtime - double(timeout.tv_sec))*1e9 + 0.5); 
	tmax = &timeout;
    }

    //----------------------------------  Tell producers that a consumer is
    //                                    waiting if RQSYNCH is in effect
    if (testFlag(RQSYNCH)) {
        sbuf.sem_num = conbt;
	sbuf.sem_flg = IPC_NOWAIT;
	sbuf.sem_op  = 0;
	if (semop(pointer->con_semid[mywd], &sbuf, 1) != -1) {
	    pcon->setWait();
	    sbuf.sem_num = gbl_synch;
	    sbuf.sem_flg = 0;
	    sbuf.sem_op  = 1;
	    semop(pointer->gbl_semid, &sbuf, 1);
	}
    }

    //----------------------------------  Find buffer for on-demand consumers
    int i=-1;
    if (!pcon->mxbuf && !pcon->isReadAll() && !testFlag(EXPOSE)) {
	if (!gate(true)) return (const char *) 0;
	for (i=pointer->full.head; i>=0; i=bufptr[i].link) {
	    LSMP_buffer* pbuf = bufptr + i;
	    if ((pbuf->trig & pcon->trig_mask) && !(pbuf->seen_mask[icon])) {
		pbuf->seen_mask.set(icon);
		if (pcon->skip_ctr <= 0) {
		    pcon->seg_ctr++;
		    pcon->seg_tot++;
		    pcon->skip_ctr = pcon->min_sep;
		    pbuf->use_count++;
		    break;
		} else {
		    pcon->skip_ctr--;
		}
	    }
	}
	gate(false);
    }

    //----------------------------------  Wait for a segment.
    while (i<0) {
        bool semdec = true;
        pcon->setWait();
	sbuf.sem_num = conbt;
	sbuf.sem_flg = wflag;
	sbuf.sem_op  = -1;
#ifdef P__DARWIN
	int rc;
	signal(SIGALRM,alarm_handler);
	if (maxtime > 1) {
	    alarm(maxtime);
	} else {
	    alarm(1);
	}
	rc=semop(pointer->con_semid[mywd], &sbuf, 1);
	if(rc == -1) {
	    if (errno == EINTR)       return (char *) 0;
	    else if (alarm_triggered) semdec = false;
	    else throw SysError("LSMP_CON::get_buffer consumer wait failed");
	}
	alarm(0);
#else
	if (semtimedop(pointer->con_semid[mywd], &sbuf, 1, tmax) == -1) {
	    if (errno == EINTR)       return (char *) 0;
	    else if (errno == EAGAIN) semdec = false;
	    else throw SysError("LSMP_CON::get_buffer consumer wait failed");
	}
#endif
	//-------------------------------  Access the control area. Reset the 
	//                                 semaphore if gate() fails.
	if (!gate(true)) {
	    if (semdec) {
		int sverr = errno;
	        sbuf.sem_num = conbt;
		sbuf.sem_flg = 0;
		sbuf.sem_op  = 1;
		semop(pointer->con_semid[mywd], &sbuf, 1);
		errno = sverr;
	    }
	    return (const char *) 0;
	}

	//-------------------------------  Look for a reserved segment
	//    Note: pcon->seg_ctr is incremented when the segment is reserved.
	for (i = pointer->full.head ; i>=0 ; i=bufptr[i].link) {
	    if (bufptr[i].reserve_mask[icon]) break;
	}

	//-------------------------------  Look for any available segment.
	if ((i < 0) && !pcon->mxbuf && !testFlag(EXPOSE)) {
	    for (i=pointer->full.head ; i>=0 ; i=bufptr[i].link) {
	        LSMP_buffer* pbuf = bufptr + i;
		if ((pbuf->trig & pcon->trig_mask) && !pbuf->seen_mask[icon]){
		    if (pcon->skip_ctr <= 0) {
		        pcon->seg_ctr++;
		        pcon->skip_ctr = pcon->min_sep;
			break;
		    } else {
		        pbuf->seen_mask.set(icon);
		        pcon->skip_ctr--;
		    }
		}
	    }
	}

	if (i >= 0) {
	    pcon->seg_tot++;
	    bufptr[i].use_count++;
	    bufptr[i].seen_mask.set(icon);
	}
 
	//--------------------------------  Release exclusive partition access
	gate(false);
	if (i<0 && ((flags & NOWAIT) != 0 || maxtime >= 0)) {
	    errno = EAGAIN;
	    return 0;
	}
    }

    ibuf = i;
    return getBuffAddr();
}

//---------------------------------------------------------------------------
//
//    int get_by_ID(int flags);
//
//    Get a full buffer for the specified consumer.
//
//    Parameter:
//      dataID Identifier of data record to be selected.
//
//    Returns:
//      -1     No flags buffer are available and NOWAIT specified or sleep 
//             interrupted by signal.
//      >= 0   Identifier of allocated buffer.
//
//---------------------------------------------------------------------------
int 
LSMP_CON::get_by_ID(eventid_type dataID) {
    int i;
    LSMP_consbk *pcon;

    //---------------------------------  Get a few useful pointers
    if (!valid()) return -1;
    pcon   = conptr + icon;

    //-------------------------------  Access the control area.
    while (!gate(true));

    //-------------------------------  Look for segment with the right data ID
    for (i=pointer->full.head ; i>=0 ; i=bufptr[i].link) {
	if (bufptr[i].data_ID == dataID) {
	    pcon->seg_tot++;
	    bufptr[i].use_count++;
	    bufptr[i].seen_mask.set(icon);
	    break;
	}
    }

    //--------------------------------  Release exclusive partition access
    gate(false);
    return i;
}

//---------------------------------------------------------------------------
//
//    bool latest_buffer();
//
bool
LSMP_CON::latest_buffer(void) const {
    if (ibuf < 0) return false;
    if (bufptr[ibuf].link < 0) return true;
    return false;
}

//---------------------------------------------------------------------------
//
//    eventid_type latest_evtID(void) const;
//
LSMP::eventid_type 
LSMP_CON::latest_evtID(void) const {
    int lbuf = pointer->full.tail;
    if (lbuf < 0) return 0;
    return bufptr[lbuf].data_ID;
}

//---------------------------------------------------------------------------
//
//    bool free_buffer();
//
//    Free a buffer from a consumer.
//
//    Parameters:
//      ibuf   Identifier of the buffer to be released
//
//    Return values:
//      false  Buffer was freed successfully
//      true   An error occurred while freeing buffer.
//
//---------------------------------------------------------------------------
bool 
LSMP_CON::free_buffer() {
    struct sembuf sbuf;
    LSMP_consbk *pcon;
    LSMP_buffer *pbuf;

    //------------------------------------  Get a few pointers, validate args.
    if (!valid()) return true;
    if (ibuf < 0)  return true;
    pcon = conptr + icon;
    if (!pcon->seg_ctr && (ibuf < 0)) {
        std::cout << "free_buffer: Consumer has no buffers" << std::endl;
	return true;}

    pbuf = bufptr + ibuf;
    if (!pbuf->use_count && !pbuf->seen_mask[icon]) {
        std::cout << "free_buffer: Buffer is not in use by this consumer." 
		  << std::endl;
	return true;}

    //------------------------------------  Access the control area.
    bool rc = false;
    while(!gate(true));

    //------------------------------------  Decrement buffer use count & clear
    //                                      the reservation mask bit.
    pbuf->use_count--;
    pbuf->reserve_mask.clear(icon);
    pcon->seg_ctr--;

    //------------------------------------  Free buffer if no longer in use
    if (!pbuf->inUse()) {
        if (!testFlag(SCAVAGE)) {
	    if (pointer->full.remove(bufptr,ibuf) >= 0) {
	        pointer->free.link(bufptr,ibuf);
		rc = false;
	    } else {
	        rc = true;
	    }
	}

	//--------------------------------  Bump the free semaphore.
	sbuf.sem_num = gbl_empty;
	sbuf.sem_flg = 0;
	sbuf.sem_op  = 1;
	semop(pointer->gbl_semid, &sbuf, 1);
    }

    //-----------------------------------  Release the global area
    gate(false);
    ibuf   = -1;
    return rc;
}
