/* -*- mode: c++; c-basic-offset: 4; -*- */
//
//    Shared memory partition accessor class.
//
//    Public Methods:
//        LSMP_ACCESS()
//            Construct a null consumer (no data access)
//
//        LSMP_ACCESS(const char *part)
//            Construct an accessor from partition *part.
//
//        ~LSMP_ACCESS()
//            Clean up and detach partition.
//
//        void Unhang();
//            Free up an dispossessed consumer.
//
#include "lsmp.hh"
#include "lsmp_int.hh"
#include <signal.h>
#include <errno.h>
#include <sys/sem.h>
#include <iostream>

using namespace std;

//----------------------------------  PID test function: 
//                                    returns true if process exists
static bool TestPID(pid_t pid) {
    if (kill(pid, 0) != -1) return true;
    if (errno != ESRCH)     return true;
    return false;
}


//----------------------------------  Constructors
LSMP_ACCESS::LSMP_ACCESS() {
}

LSMP_ACCESS::LSMP_ACCESS(const char *name) 
  : LSMP(name)
{
}

//-----------------------------------  Destructor
LSMP_ACCESS::~LSMP_ACCESS() {
}

//---------------------------------------------------------------------------
//
//    void Repair(int icon)
//
//    Remove orphaned consumers 
//
//---------------------------------------------------------------------------
void
LSMP_ACCESS::Repair(int jcon) {
    struct sembuf sbuf;

    //---------------------------------  Test for reparable  partition
    if (!valid()) return;
    if (my_procid() != getpid()) {
      cerr << "Suspected PID namespace - Repair() can not be run" << endl;
      return;
    }

    //---------------------------------  Get the necessary pointers
    int jmin = jcon;
    int jmax = jcon+1;
    if (jcon == -1) {
        jmin = 0;
	jmax = LSMP_MAXCONS;
    } else if (jmin < 0) {
        return;
    } else if (jmax > LSMP_MAXCONS) {
        return;
    }

    //---------------------------------  Look for derelict consumer slots
    for (int icon=jmin ; icon < jmax ; icon++) {
	if (pointer->conmask[icon]) {
	    if (!TestPID(conptr[icon].pid)) {
	        free_consumer(icon);
		while(!gate(true));
		pointer->gbl_count--;
		gate(false);
	    }
	} else {
	    bool inactive = false;
	    while(!gate(true));
	    if (!(pointer->conmask[icon])) {
	        for (int i = pointer->full.head ; i>=0 ; i = bufptr[i].link) {
		    inactive |= bufptr[i].reserve_mask.test(icon);
		}
	    }
	    gate(false);
	    if (inactive) free_consumer(icon);
	}
    }

    //----------------------------------  Release buffer when no consumers
    while(!gate(true));

    //----------------------------------  Mask of consumers with allocated data
    ConFlags active_cons;
    for (int i=0 ; i<LSMP_MAXCONS ; i++) {
        if (pointer->conmask[i] && conptr[i].seg_ctr != 0) active_cons.set(i);
    }

    //----------------------------------  Delete buffers whose data isn't held
    //                                    by consumer.
    int n = -1;
    for (int i = pointer->full.head ; i>=0 ; i=n) {
        LSMP_buffer* pbuf = bufptr + i;
        n = pbuf->link;
	if (pbuf->inUse()) {
	    pbuf->reserve_mask &= active_cons;
	    int Nb = (pbuf->seen_mask & active_cons).count();
	    if (pbuf->use_count > Nb) pbuf->use_count = Nb;
	    if (!pbuf->inUse()) {
	        sbuf.sem_flg = 0;
		sbuf.sem_op  = 1;
		sbuf.sem_num = gbl_empty;
		semop(pointer->gbl_semid, &sbuf, 1);
	    }
	}
    }
    gate(false);

    //----------------------------------  Find buffers held by dead producers
    for (int i=0 ; i<pointer->nbuf ; i++) {
        LSMP_buffer* pbuf = bufptr + i;
        if (!pbuf->queued() && !TestPID(pbuf->owner_pid)){

	    //--------------------------  Set the buffer parameters
	    pbuf->link      = -1;
	    pbuf->trig      =  0;
	    pbuf->ldata     =  0;
	    pbuf->use_count =  0;
	    pbuf->reserve_mask.zero();
	    pbuf->seen_mask.zero();

	    //--------------------------  Add buffer to the free queue
	    while (!gate(true));
	    pointer->free.link(bufptr,i);
	    gate(false);

	    //---------------------------  Bump the free semaphore
	    sbuf.sem_flg = 0;
	    sbuf.sem_op  = 1;
	    sbuf.sem_num = gbl_empty;
	    semop(pointer->gbl_semid, &sbuf, 1);
	}
    }
    return;
}
//---------------------------------------------------------------------------
//
//    void Spew(int jbuf, std::ostream& out)
//
//    Dump a bufffer to a file, no questions asked.
//
//---------------------------------------------------------------------------
void
LSMP_ACCESS::Spew(int jBuf, std::ostream& Out, int len) {
    //----------------------------------  Get access to the global region.
    while (!gate(true));

    //----------------------------------  Write out the data
    if (len <= 0) len  = buffer_length(jBuf);
    const char*   addr = buffer_addr(jBuf);
    if (addr) Out.write(addr, len);

    //----------------------------------  Release the global region.
    gate(false);
}
