/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef LMSG_SOCKET_HH
#define LMSG_SOCKET_HH
#include "lmsg/MsgTypes.hh"
#include "lmsg/MsgAddr.hh"

namespace lmsg {
  class Buffer;
  class BufferPool;
  class SocketPool;

  /**  Class to manipulate sockets. The socket class is a generic socket API.
    *  @memo Abstract socket class.
    *  @author  John Zweizig
    *  @version 1.2; Modified October 16, 2000
    *  @ingroup IO_lmsg
    */
  class Socket {
  public:

    /**  Import the IP socket address type.
     */
    typedef struct sockaddr_in   ipsock_t;

    /**  Construct an empty socket.
      *  @memo Construct a socket.
      */
    Socket(void);

    /**  Release the socket and delete all asociated storage.
      *  @memo Destructor
      */
    virtual ~Socket(void);

    /**  Get the address to which the socket is bound. If the IP-Node portion
      *  of the address is zero, it indicates that the same port is allocated
      *  on ALL networks known by the node.
      *  @memo Get socket IP address.
      *  @return The IP address to which the socket is bound socket.
      */
    MsgAddr getAddr(void) const;

    /**  Get the debug printout level.
      *  @memo Get debug level.
      *  @return The debug printout level;
      */
    int getDebug(void) const;

    /**  Get the address of the latest communication partner of this socket.
      *  If a connection has been made, the partner address will be the 
      *  address of socket to which it is connected. If no connection has 
      *  been made, the address specifies the port to which a message was 
      *  most recently sent or from which a message was most recently 
      *  received. The IP node portion of the address may be zero if the
      *  latest message was sent to or received from the current node.
      *  @memo Get socket IP address.
      *  @return The IP address to which the socket is bound socket.
      */
    MsgAddr getPartner(void) const;

    /**  Get the system receive buffer length.
      *  @memo Get the receive buffer length..
      *  @return The number of bytes allocated for the system receive buffer.
      */
    int getRecvBuf(void) const;

    /**  Get the system send buffer length.
      *  @memo Get the send buffer length..
      *  @return The number of bytes allocated for the system send buffer.
      */
    int getSendBuf(void) const;

    /**  Test if the socket is connected to a partner.
      *  @memo Test if socket is connected.
      *  @return true if the socket is connected.
      */
    bool isConnected(void) const;

    /**  Test whether the socket is open.
      *  @memo Test if socket is open.
      *  @return true if the socket is open.
      */
    bool isOpen(void) const;

    /**  Enable or disable asynchronous operation of the socket. If 
      *  asynchronous operation is enabled, the socket will generate a 
      *  SIGIO signal each time a message is received for the socket.
      *  Note that a SIGIO signal may be generated at any time after 
      *  asynchronous operation has been enabled, so that SIGIO handling
      *  should be specified to the operating system before asynchronous
      *  operation is enabled.
      *  @memo Allow asynchronous message handling.
      *  @return Standard lmsg error codes (OK, NotOpen or SystemError).
      */
    virtual error_type async(bool enable=true);

    /**  Bind this socket to a specified address.
      *  @memo Bind socket to an address.
      *  @param addr Address of the socket.
      *  @return Standard lmsg error codes (OK, NotOpen or SystemError).
      */
    virtual error_type bind(const MsgAddr* addr);

    /**  Connect this socket to a specified partner socket. If the connection 
      *  is successful, the socket patner address is updated.
      *  @memo connect to a partner address.
      *  @param addr Address of partner to which a connection is to be made.
      *  @return Standard lmsg error codes (OK, NotOpen or SystemError).
      */
    virtual error_type connect(const MsgAddr& addr);

    /**  Disconnect the socket from a ciurrent parner. If no connection has 
      *  been made, diconnect returns successfully.
      *  @memo Disconnect from partner.
      *  @return Standard lmsg error codes (OK, NotOpen or SystemError).
      */
    virtual error_type disconnect(void);

    /**  Read data from a socket.
      *  @memo Receive a message.
      *  @return Standard lmsg error codes (OK, NotOpen or SystemError).
      *  @param data Address of data buffer.
      *  @param len  Number of bytes to read.
      */
    virtual error_type read(char* data, size_type len);

    /**  Receive a message from a partner socket. If the socket is connected,
      *  the message will be received from the connected partner. If no 
      *  connection has been made, the message may be received from any port.
      *  The socket partner address is updated with the IP address of the 
      *  port from which the message was received.
      *  @memo Receive a message.
      *  @return Standard lmsg error codes (OK, NotOpen or SystemError).
      *  @param Data Address of pointer in which the address of the buffer 
      *              containing the received message will be stored.
      */
    virtual error_type receive(Buffer** Data);

    /**  Receive a message from a partner socket. If the socket is connected,
      *  the message will be received from the connected partner. If no 
      *  connection has been made, the message may be received from any port.
      *  The socket partner address is updated with the IP address of the 
      *  port from which the message was received.
      *  @memo Receive a message.
      *  @return Standard lmsg error codes (OK, NotOpen or SystemError).
      *  @param Data Address of the buffer to receive the message.
      */
    virtual error_type receive(Buffer* Data);

    /**  Send a message to a specified partner.
      *  @memo Send a message.
      *  @return Standard lmsg error codes (OK, NotOpen or SystemError).
      *  @param b  Buffer containing the message to be sent.
      */
    virtual error_type send(Buffer* b);

    /**  Send a message to a specified partner.
      *  @memo Send a message.
      *  @return Standard lmsg error codes (OK, NotOpen or SystemError).
      *  @param to Address of port to which the message will be sent.
      *  @param b  Buffer containing the message to be sent.
      */
    virtual error_type send(const MsgAddr& to, Buffer* b);

    /**  Wait a specified maximum time or until a message is received. Wait
      *  can be used to test for an available message by specifying a zero
      *  wait time  or to wait forever until a message is received by 
      *  specifying a negative wait time;
      *  @memo Wait for a message.
      *  @return lmsg error codes (OK, NotOpen, SystemError or TimeOut).
      *  @param maxtime Maximum time to wait in seconds.
      */
    error_type wait(wtime_type maxtime) const;

    /**  Wait a specified maximum time or until a message is received by 
      *  one or more Sockets. Wait can be used to test for an available 
      *  message by specifying a zero wait time  or to wait forever until 
      *  a message is received by specifying a negative wait time;
      *  @memo Wait for a message.
      *  @return lmsg error code (OK, NotOpen, Continue, SystemError or 
      *          TimeOut).
      *  @param maxtime Maximum wait time in seconds.
      *  @param N       Number of socket in the list.
      *  @param list    List of sockets to be waited on.
      */
    static error_type waitAny(wtime_type maxtime, int N, const Socket* list[]);

    /**  Specify the debug message level.
      *  @memo Specify a debug level.
      *  @param level of debug messages to be printed.
      */
    void setDebug(int level);

    /**  Specify a buffer pool to be used for received messages.
      *  @memo Specify a buffer pool.
      *  @param pool Pointer to the buffer pool
      */
    void setPool(BufferPool* pool);

    /**  Specify a pool of which this socket is to be a member.
      *  @memo Specify asocket pool.
      *  @param pool Pointer to the socket pool
      */
    void setPool(SocketPool* pool);

    /**  Specify a system receive buffer size.
      *  @memo Specify receive buffer length.
      *  @param nBytes Receive buffer length
      */
    error_type setRecvBuf(int nBytes);

    /**  Specify a system send buffer size.
      *  @memo Specify send buffer length.
      *  @param nBytes Send buffer length
      */
    error_type setSendBuf(int nBytes);

    /**  Return the socket to its pool.
      *  @memo Return Socket to its pool.
      */
    void Return(void);

  protected:
    ///  Address of this socket
    MsgAddr     myAddr;
    /// Address of partner socket
    MsgAddr     mPartner;
    /// Pool containing buffers to be used by read()
    BufferPool* mBPool;
    /// Status bit indicating socket is connected.
    bool        mConnected;
    ///  Socket file descriptor
    int         mSocket;

  private:
    SocketPool* mSPool;
    int         mDebug;
  };
} // namespace lmsg

inline int
lmsg::Socket::getDebug(void) const {
    return mDebug;
}

inline bool
lmsg::Socket::isConnected(void) const {
    return mConnected;
}

inline bool
lmsg::Socket::isOpen(void) const {
    return mSocket>=0;
}

#endif // LMSG_SOCKET_HH
