/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef LMSG_MSGHEADER_HH
#define LMSG_MSGHEADER_HH

#include "lmsg/MsgAddr.hh"
#include "lmsg/MsgTypes.hh"

namespace lmsg {

  /**  The message header contains the message type and routing information.
    *  @memo Message header class.
    *  @author J. Zweizig
    *  @version 1.1; Modified 2/1/2000
    *  @ingroup IO_lmsg
    */
  class MsgHeader {

    //----------------------------------  4-byte integer (alpha long is 8B)
    typedef int count_t;
  public:

      /** Data type used to hold message type code.
        */
      typedef count_t MsgType;

    /**  Message header destructor.
      *  @memo Destructor.
      */
    ~MsgHeader(void);

    /**  Construct an empty message header.
      *  @memo Empty header constructor.
      */
    MsgHeader(void);

    /**  Construct a message header with the specified type, 
      *  length and destination.
      *  @memo Full constructor.
      */
    MsgHeader(count_t type, count_t length, const MsgAddr& dest);

    /**  The destination field of the message header is returned.
      *  @memo get the destination address.
      *  @return The destination address.
      */
    const MsgAddr& getDest(void) const;

    /**  The source field of the message header is returned.
      *  @memo   Get the source address.
      *  @return The source address.
      */
    const MsgAddr& getSource(void) const;

    /**  A mutable reference to the destination address of the message 
      *  header is returned.
      *  @memo   Get a reference to the destination address.
      *  @return A mutable reference to the destination address.
      */
    MsgAddr& getDest(void);

    /**  A mutable reference to the source address of the message 
      *  header is returned.
      *  @memo   Get a reference to the source address.
      *  @return A mutable reference to the source address.
      */
    MsgAddr& getSource(void);

    /**  The type field is returned from the message header.
      *  @memo   Get the message type.
      *  @return The message type.
      */
    MsgType getMsgType(void) const;

    /**  The message data length field is returned from the header.
      *  @memo   Get the message data length.
      *  @return The message data length.
      */
    count_t getMsgLength(void) const;

    /**  The block ID field is returned from the message header.
      *  @memo   Get the block ID.
      *  @return The message Block ID.
      */
    count_t getBlockID(void) const;

    /**  The transaction identifier field is returned from the message 
      *  header.
      *  @memo   Get the message transaction ID.
      *  @return The transaction identifier.
      */
    count_t getTransID(void) const;

    /**  Dump the contents of the message header to an output stream.
      *  @memo Dump the header.
      *  @param out Stream to which the header is dumped.
      */
    void Dump(std::ostream& out) const;

    /**  Translate the header contents from external (big endian) format 
      *  to the local format.
      *  @memo Convert to internal format.
      */
    void Import(void);

    /**  Translate the header contents from the local format to the 
      *  external (big endian) format.
      *  @memo Convert to external format.
      */
    void Export(void);

    /**  Set the message header destination address.
      *  @memo Set the destination address.
      *  @param addr Address to which the destination field will be set.
      */
    void setDest(const MsgAddr& addr);

    /**  Set the message header source address.
      *  @memo Set the source address.
      *  @param addr Address to which the message source field will be set.
      */
    void setSource(const MsgAddr& addr);

    /**  Set the message type field.
      *  @memo Set the type.
      */
    void setMsgType(MsgType type);

    /**  Set the message data length field.
      *  @memo Set the data length.
      */
    void setMsgLength(count_t length);

    /**  Set the message block identifier field.
      *  @memo Set the block ID.
      */
    void setBlockID(count_t id);

    /**  Set the transaction identifier field.
      *  @memo Set the transaction ID.
      */
    void setTransID(count_t id);

  private:
    count_t mLength;
    MsgType mType;
    MsgAddr mDest;
    MsgAddr mSource;
    count_t mTransID;
    count_t mBlockID;
  };
} // namespace lmsg

//======================================  Message header accessors
inline lmsg::MsgHeader::count_t 
lmsg::MsgHeader::getBlockID(void) const {
    return mBlockID;
}

inline const lmsg::MsgAddr& 
lmsg::MsgHeader::getDest(void) const {
    return mDest;
}

inline lmsg::MsgAddr& 
lmsg::MsgHeader::getDest(void) {
    return mDest;
}

inline lmsg::MsgHeader::count_t 
lmsg::MsgHeader::getMsgLength(void) const {
    return mLength;
}

inline lmsg::MsgHeader::MsgType 
lmsg::MsgHeader::getMsgType(void) const {
    return mType;
}

inline const lmsg::MsgAddr& 
lmsg::MsgHeader::getSource(void) const {
    return mSource;
}

inline lmsg::MsgAddr& 
lmsg::MsgHeader::getSource(void) {
    return mSource;
}

inline lmsg::MsgHeader::count_t 
lmsg::MsgHeader::getTransID(void) const {
    return mTransID;
}


#endif // LMSG_MSGHEADER_HH
