/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef LMSG_MSGADDR_HH
#define LMSG_MSGADDR_HH

#include <string>
#include <iosfwd>

#ifndef __CINT__
struct sockaddr_in;
#else
struct sockaddr_in {
        short        sin_family;
        short        sin_port;
        unsigned int sin_addr;
        char         sin_zero[8];
};
#endif

namespace lmsg {

  /**  The message address class holds a message source or destination 
    *  address consisting of an IP node address, an IP port number and
    *  a process number. The process number may be used for a level of
    *  routing beyond the usual IP (node, port) addresses if desired.
    *  The IP node and port numbers are always stored in network byte
    *  order. This is the same convention as that used in the sockaddr_in
    *  structure. The process number is stored in the local byte ordering.
    *  @memo Message Address Class.
    *  @version 1.1; Last modified June 6, 2000
    *  @author J. Zweizig
    *  @ingroup IO_lmsg
    */
  class MsgAddr {
  public:
    /** IP node address data type.
      */
    typedef unsigned int   ipaddr_t;

    /**  IP port number data type.
      */
    typedef unsigned short ipport_t;

    /**  Socket address data type.
      */
    typedef ::sockaddr_in ipsock_t;
  public:

    /**  Address destructor.
      */
    ~MsgAddr(void);

    /**  An address is constructed and the ip address and all address fields
      *  are set to zero.
      *  @memo Default address constructor. 
      */
    MsgAddr(void);

    /**  A message address is constructed and initialized to the IP address
      *  and port IDs specified by the argumetn socket address.
      *  @memo Construct a message address from a socket address.
      *  @param sock IP socket to which the MsgAddr will be initialized.
      */
    MsgAddr(const ipsock_t& sock);

    /**  A message address is constructed and initialized from another
      *  message address.
      *  @memo  copy constructor.
      *  @param addr Address to which the new MsgAddr will be initialized.
      */
    MsgAddr(const MsgAddr& addr);

    /**  A message address is constructed and initialized from the component
      *  quantities. Only the IP node address need be specified. Note that 
      *  the port address is in host byte ordering!
      *  @memo  Data constructor.
      *  @param node IP address to which the new MsgAddr will be initialized.
      *  @param port IP port to which the new MsgAddr will be initialized.
      *  @param subp Process ID to which the new MsgAddr will be initialized.
      */
    MsgAddr(ipaddr_t node, ipport_t port=0, ipport_t subp=0);

    /**  The message address is replaced with a new value.
      *  @memo Assignment operator.
      *  @param addr New message address.
      *  @return Mutable reference to the assigned object.
      */
    MsgAddr& operator=(const MsgAddr& addr);

    /**  The object is compared with the right hand side value for 
      *  equality.
      *  @memo compare for equality
      *  @param s Value to which the address is compared.
      *  @return true if the addresses are equal.
      */
    bool operator==(const MsgAddr& s) const;

    /**  The object is compared with the right hand side value for 
      *  inequality.
      *  @memo Compare for inequality
      *  @param s Value to which the address is compared.
      *  @return true if the addresses are unequal.
      */
    bool operator!=(const MsgAddr& s) const;

    /**  Test the address of the operator and return true if the address and
      *  port numbers are zero.
      *  @memo Test for invalidity
      *  @return true if the addresses is not valid.
      */
    bool operator!(void) const;

    /**  Fetch the IP node ID.
      *  @memo Fetch the IP node ID.
      *  @return The IP node number of the message address.
      */
    ipaddr_t getIPAddr(void) const;

    /**  Fetch the IP port ID.
      *  @memo Fetch the IP port ID.
      *  @return The IP port number of the message address.
      */
    ipport_t getIPPort(void) const;

    /**  Fetch the sub-process number.
      *  @memo Fetch the subprocess ID.
      *  @return The sun-process number of the message address.
      */
    ipport_t getSubProcess(void) const;

    /**  Get the ip socet name from the message address.
      *  @memo get the IP address.
      *  @param sock IP socket name structure to receive the socket address.
      */
    void getAddr(ipsock_t& sock) const;

    /**  Set the IP node number in the message address.
      *  @memo Set the IP node number.
      *  @param sock IP node number.
      */
    void setIPAddr(ipaddr_t sock);

    /**  Set the IP port number in the message address.
      *  @memo Set the IP port number.
      *  @param port IP port number.
      */
    void setIPPort(ipport_t port);

    /**  Set the subprocess number in the message address.
      *  @memo Set the subprocess number.
      *  @param proc subprocess number.
      */
    void setSubProcess(ipport_t proc);

    /**  Set the IP fields (node and port) in the message address.
      *  @memo Set the IP address.
      *  @param sock IP name structure.
      */
    void setAddr(const ipsock_t& sock);

    /**  Convert a symbolic symbolic IP address in the form 
      *  "<node-name>:<port>" to a message address. If either the node-name
      *  or the port number is not specified, the field is set to zero.
      *  @memo Conver a symbolic IP address to a MsgAddr.
      *  @param hostName IP host name and port number.
      *  @return Message address corresponding to the specified host name.
      */
    static MsgAddr findIPHost(const std::string& hostName);

    /**  Convert the message address node number to a symbolic value.
      *  @memo Get the node name.
      *  @return Symbolic IP node name.
      */
    std::string getIPName(void) const;

  private:
    ipaddr_t ip_addr;
    ipport_t ip_port;
    ipport_t subprocess;
  };

} // namespace lmsg

//--------------------------------------  Non Member method
/**  Convert the message address to a text string and write it to an
  *  output string.
  *  @memo Write message address as a text string.
  */
std::ostream& 
operator<<(std::ostream& out, const lmsg::MsgAddr& addr);

//--------------------------------------  Inline methods
#ifndef __CINT__
inline bool 
lmsg::MsgAddr::operator==(const MsgAddr& s) const {
    return (ip_addr == s.ip_addr) && 
           (ip_port == s.ip_port) && 
           (subprocess == s.subprocess);
}

inline bool 
lmsg::MsgAddr::operator!=(const MsgAddr& s) const {
    return !operator==(s);
}

inline bool 
lmsg::MsgAddr::operator!(void) const {
    return (!ip_addr && !ip_port);
}
#endif // !def(__CINT__)

#endif // LMSG_MSGADDR_HH

