/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef DATAMSG_HH
#define DATAMSG_HH

#include "lmsg/Message.hh"

namespace lmsg {
    class TransInput;
    class TransOutput;

  /**  Template for messages with a single data item. The data item may 
    *  be any defined class with the following properties.
    *  <ul>
    *    <li> A copy constructor</li>
    *    <li> An \c operator>>(TransInput&, T&) defined either intrinsically 
    *  in the \c TransInput class or as an external function.</li>
    *    <li> An \c operator<<(TransOutput&,const T&) defined either 
    *  intrinsically in the TransOutput class or as an external function.</li>
    *  </ul>
    *  @memo Data message template.
    *  @author John Zweizig
    *  @version $Revision: 7980 $; Modified $Date: 2018-01-04 17:26:54 -0800 (Thu, 04 Jan 2018) $
    *  @ingroup IO_lmsg
    */
  template<class T, unsigned int MsgID>
  class DataMsg : public Message {
  public:

    /**  Construct a message.
      *  @memo Default constructor.
      */
    DataMsg(void) {}

    /**  Construct a message identical to an existing message.
      *  @memo Copy constructor.
      */
    DataMsg(const DataMsg<T,MsgID>& x) : mData(x.mData) {}

    /**  Construct a message and initialize data to a specified value.
      *  @memo Data constructor.
      */
    explicit DataMsg(const T& x) : mData(x) {}

    /**  Destroy a message.
      *  @memo Default destructor.
      */
    virtual ~DataMsg(void) {}

    /**  Get the message text length.
      *  @memo Text length.
      *  @return Always zero for dataless messages.
      */
    size_type size(void) const {return 0;}

    /**  Write the message data to translation stream.
      *  @memo Get message data.
      */
    void getData(TransOutput& out) const;

    /**  Get the message type.
      *  @memo Message type.
      *  @return As specified by template argument.
      */
    MsgHeader::MsgType getType(void) const {return MsgID;}

    /**  Read data from a transltion stream into the message.
      *  @memo Set message data.
      */
    void setData(TransInput& in);

    /**  Get a reference to the data portion of the message.
      *  @brief Reference message data
      *  @return Reference to data portion of message.
      */
    T& refData(void) {return mData;}

    /**  Get a reference to the data portion of the message.
      *  @brief Reference constant message data
      *  @return Constant reference to data portion of message.
      */
    const T& refData(void) const {return mData;}

  private:
    T mData;
  };

}  // namespace lmsg

#endif   // DATAMSG_HH
