#ifndef HTML_COLOR_HH
#define HTML_COLOR_HH

#include "html/Attrib.hh"
#include <string>

namespace html {
  class writer;


  /**  Text color attribute class.
    *  @brief Text color attribute class.
    *  @author John Zweizig
    *  @version 1.1; Last modified March 4, 2008
    *  @ingroup IO_html
    */
  class color : public attrib {
  public:
    /**  Construct a color attribute object that specifies a default 
      *  color.
      *  @brief Default constructor.
      */
    color(void);

    /**  Construct a color attribute object and initialize it with red,
      *  green and blue intensities.
      *  @brief RGB color constructor.
      *  @param red   Red color intensity.
      *  @param green Green color intensity.
      *  @param blue  Blue color intensity.
      */
    color(int red, int green, int blue);

    /**  Construct a color attribute object and initialize it with named
      *  color value.
      *  @brief Name color constructor.
      *  @param clr Color name.
      */
    color(const std::string& clr);

    /**  Destroy a color object.
      *  @brief Destructor.
      */
     ~color(void);

    /**  Make an identical copy of the color object.
      *  @brief Clone alignment.
      *  @return Pointer to the cloned color object.
      */
    color* clone(void) const;

    /**  Test whether the color is the default value.
      *  @return true if default color.
      */
    bool isDefault(void) const;

    /**  Write the color attribute to the specified html writer.
      *  @brief Write the color attribure.
      *  @param w Html writer reference.
      */
    void putAttr(writer& w) const;

    /**  Set the color to the default.
      *  @brief Set default color.
      */
    void setDefault(void);

    /**  Set the color by specifying the red, green and blue intensities.
      *  @brief Set the color with RGB.
      *  @param red   Red color intensity.
      *  @param green Green color intensity.
      *  @param blue  Blue color intensity.
      */
    void setRGB(int red, int green, int blue);

    /**  Set the color by specifying a known color name.
      *  @brief Set the color by name.
      *  @param name Color name.
      */
    void setColorName(const std::string& name);

    /**  Test whether this color is equal to the specified.
      *  @brief Compare operator.
      *  @param x color object to be compared
      *  @return True if this color is equal to the argument.
      */
     bool operator==(const color& x) const;

  private:
    enum ColorMode {
      k_default,
      k_rgb,
      k_named
    };

  private:
    ColorMode mMode;
    int mRed;
    int mGreen;
    int mBlue;
    std::string mName;
  };

}

#endif //  HTML_COLOR_HH
