/*----------------------------------------------------------------------*/
/*                                                                      */
/* Module Name: frameio					                */
/*                                                                      */
/* Module Description:  fast frame input/output			        */
/*                                                                      */
/*----------------------------------------------------------------------*/

#include "framefast/frameio.hh"
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <errno.h>
#include <unistd.h>
#include <sys/uio.h>
#include <limits.h>
#include <sys/types.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <iostream>
#include <algorithm>

#ifndef IOV_MAX
#define IOV_MAX UIO_MAXIOV
#endif

namespace framefast {

   using namespace std;


//////////////////////////////////////////////////////////////////////////
//                                                                      //
// memory frame storage                                                 //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   void memory_frame_storage::reset() 
   {
      if (fOwn && fData) {
         //cerr << "delete memory frame of size " << fLength << endl;
         delete [] fData;
      }
      fData = 0; 
      fLength = 0; 
      fOwn = false;
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// file frame storage                                                   //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   bool file_frame_storage::load (const char* filename)
   {
      int		fd;		// file id
   
      // unload old frame first
      reset();
      if (!filename) {
         return true;
      }
   
      // open file
      fd = open (filename, O_RDONLY);
      if (fd == -1) {
         //if (fDebug) cerr << "file open failed errno = " << errno << endl;
         return false;
      }
      // determine file length
      fLength = lseek (fd, 0, SEEK_END);
      if (fLength == -1) {
         //if (fDebug) cerr << "seek failed errno = " << errno << endl;
         close (fd);
         return false;
      }
   
      // read in file
      fData = new (nothrow) char [fLength];
      if (fData == 0) {
         //if (fDebug) cerr << "memory allocation error " << endl;
         fLength = 0;
         close (fd);
         return false;
      }
      lseek (fd, 0, SEEK_SET);
      int size = read (fd, (char*)fData, fLength);
      if (size != fLength) {
         //if (fDebug) cerr << "file read failed = " << errno << endl;
         delete [] fData; 
         fData = 0;
         fLength = 0;
         close (fd);
         return false;
      }
      close (fd);
      fOwn = true;
      setname (filename);
      return true;
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// memory mapped file frame storage                                     //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   bool mmap_frame_storage::map (const char* filename)
   {
      int		fd;		// file id
   
      // unload old frame first
      reset();
      if (!filename) {
         return true;
      }
   
      // open file
      fd = open (filename, O_RDONLY);
      if (fd == -1) {
         //if (fDebug) cerr << "file open failed errno = " << errno << endl;
         return false;
      }
      // determine file length
      fLength = lseek (fd, 0, SEEK_END);
      if (fLength == -1) {
         //if (fDebug) cerr << "seek failed errno = " << errno << endl;
         close (fd);
         return false;
      }
   
      // map file into memory
      fAddr = (char*) mmap (0, fLength, PROT_READ, MAP_PRIVATE, fd, 0);
      // QFS requires the exec flag to be set, so try again
      if (fAddr == MAP_FAILED) {
         fAddr = (char*) mmap (0, fLength, PROT_READ | PROT_EXEC, MAP_PRIVATE, fd, 0);
      }
      close (fd);
      if (fAddr == MAP_FAILED) {
         fAddr = 0;
         fLength = 0;
         //if (fDebug) cerr << "memory map failed errno = " << errno << endl;
         return false;
      }
      //if (fDebug > 1) cerr << "mmap          @ " << (int) fAddr << endl;
      setname (filename);
      return true;
   }

//______________________________________________________________________________
   void mmap_frame_storage::reset() 
   {
      //if (fDebug > 1) cerr << "munmap        @ " << (int) fAddr << endl;
      if (fAddr) {
         //cerr << "unmap frame of size " << fLength << endl;
         munmap ((char*)fAddr, fLength);
      }
      fAddr = 0;
      fLength = 0;
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// memory_out				                                //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   bool memory_out::open (int len)
   {
      // if data ptr is NULL, allocate memory
      if ((fData == 0) && (len > 0)) {
         fData = new (nothrow) char [len + 10];
         if (fData) {
            fLength = len;
            fOwn = true;
         }
         else {
            fLength = 0;
            fOwn = false;
         }
      }
      if (!fData || (len > fLength)) {
         return false;
      }
      else {
         return true;
      }
   }

//______________________________________________________________________________
   bool memory_out::write (const char* p, int size)
   {
       // check length
      if (fSofar + size > fLength) {
         size = fLength - fSofar;
      }
      // copy data
      memcpy ((char*)fData + fSofar, p, size);
      fSofar += size;
      return true;
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// desc_out				                                //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   bool desc_out::write (const char* p, int len) 
   {
      if (fd < 0) {
         return false;
      }
      fSofar += len;
      return (::write (fd, p, len) == len);
   }

//______________________________________________________________________________
   bool desc_out::write (const src_dest_t* s, int slen)
   {
      if (fd < 0) {
         return false;
      }
      iovec iov[IOV_MAX];
      for (int j = 0; j < (slen + IOV_MAX - 1) / IOV_MAX; j++) {
         int num = min (slen - j * IOV_MAX, IOV_MAX);
         for (int i = 0; i < num; i++) {
            iov[i].iov_base = (char*)s[j * IOV_MAX + i].fAddr;
            iov[i].iov_len  = s[j * IOV_MAX + i].fLen;
            fSofar += iov[i].iov_len;
         }
         if (writev (fd, iov, num) < 0) {
            return false;
         }
      }
      return true;
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// file_out				                                //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   bool file_out::open (int len)
   {
      fd = ::creat (fFilename.c_str(), 00666);
      if (fd < 0) {
         return false;
      }
      return true;
   }

//______________________________________________________________________________
   void file_out::close()
   {
      if (fd >= 0) ::close (fd);
      fd = -1;
   }


}
