/* -*-  mode: c++; c-basic-offset: 4; -*- */
#include <cstdio>
#include <time.h>
#include <unistd.h>
#include "Trend.hh"
#include "DVector.hh"
#include "FrWriter.hh"
#include "Dacc.hh"
#include "frame_name.hh"
#include <stdexcept>
#include <iostream>
#include <iomanip>
#include <fstream>
#include <cerrno>

using namespace std;

//=======================================  Constructors
Trend::Trend(void)
  : mType(kNonStandard), mSample(0.0), mMaxPoints(1024), mNAccess(0), 
    mStartFrame(0), mEndFrame(0), mLastData(0), mFWriter(0), 
    mAutoUpdate(true), mFrameMax(1), mFrameCount(0), mWriteTOC(true)
{
    setIFO(0);
}

Trend::Trend(const Trend& x) {
    *this = x;
}

Trend::Trend(const char* Name, TrendType type, count_t FrameL)
  : mName(Name), mType(kNonStandard), mSample(0.0), mMaxPoints(0), 
    mNAccess(0), mStartFrame(0), mEndFrame(0), mLastData(0), mFWriter(0), 
    mAutoUpdate(true), mFrameMax(1), mFrameCount(0), mWriteTOC(true)
{
    //-----------------------------------  Fill in the IFO name
    setIFO(0);

    //----------------------------------  Set the interval as appropriate
    setType(type);
    if (FrameL) {
	mMaxPoints = FrameL;
    } 
    else if (!mMaxPoints) {
	mMaxPoints = 1024;
    }
}

//======================================  Trend destructor 
Trend::~Trend() {
    if (!isEmpty()) {
         try {
	     writeFrame();
	 } catch (exception& e) {
	     cerr << "Exception writing final trend: " << e.what() << endl;
	 }
    }
    close();
    mDict.clear();
}

//======================================  Clear any accumulated data
void
Trend::clear(void) {
    mStartFrame = Time(0);
    mEndFrame   = Time(0);
    mLastData   = Time(0);
    for (chan_iter i=mDict.begin(); i != mDict.end(); ++i) {
        i->second.clear();
    }
    close();
}

//======================================  Close the frame writer
void
Trend::close(void) {
    if (mFWriter) {
        mFWriter->close();
        delete mFWriter;
	mFWriter = 0;
    }
}

//======================================  Dump the trend contents
ostream&
Trend::dump(ostream& os) const {
    os << "Trend " << mName << " Status" << endl;
    os << "Type: ";
    switch (mType) {
    case kSecond:
        os << "second";
	break;
    case kMinute:
        os << "minute";
	break;
    default:
        os << "non-standard";
    }
    os << " tSample: " << mSample 
       << " max points: " << mMaxPoints << endl;
    os << setprecision(12) << "Frame start: " << mStartFrame.totalS() 
       << " Frame End: " << mEndFrame.totalS() 
       << " Last data: " << mLastData.totalS() << endl;

    os << "Channel start-acc tSample" << endl;
    for (const_chan_iter i=mDict.begin(); i != mDict.end(); ++i) {
        i->second.dump(os);
    }
    return os;
}

//======================================  Open a frame writer
void
Trend::open(void) {
    if (mFWriter) return;
    mFWriter = new FrWriter(mName.c_str(), 1);

    //----------------------------------  Build a default frame name
    string File;
    if (mFileName.empty()) {
	//------------------------------  Directory name
	string dirName;
	const char* dmtout = getenv("DMTRENDOUT");
	if (dmtout) dirName = dmtout;

	//-----------------------------  Build a prefix
	string prefix = mIFO.substr(0,1) + "-" + mName;
	if (mType == kMinute) prefix += "_M";
	else                  prefix += "_T";

        int dt = int(mSample * double(mMaxPoints));
	frame_name frName(dirName, prefix, dt);
	dirName = frName.dir_name(mStartFrame);
	frName.make_dir(dirName);
	File = frName.file_path(mStartFrame);
    }

    //----------------------------------  Use a hard-coded path. Note that 
    //                                    this is for backward compatibility 
    //                                    and does not support directory
    //                                    construction.
    else {
	File = TimeString(mStartFrame, mFileName.c_str());
    }
    const char* pFile = File.c_str();

    //----------------------------------  Check whether the file exists
    if (!access(pFile, F_OK)) {
        string renFile(File);
	renFile += ".~1~";
	rename(pFile, renFile.c_str());
    }

    //----------------------------------  Open the output file
    errno = 0;
    if (mFWriter->open(pFile, mWriteTOC) || !mFWriter->is_open()) {
        cerr << "Trend: Unable to open frame file: " << File << endl;
	perror("Last error");
	delete mFWriter;
	mFWriter = 0;
    }
    mFrameCount = 0;
}

//======================================  Assignment operator
Trend&
Trend::operator=(const Trend& x) {
    mName       = x.mName; 
    mFileName   = x.mFileName; 
    mIFO        = x.mIFO; 
    mType       = x.mType;
    mSample     = x.mSample; 
    mMaxPoints  = x.mMaxPoints;
    mNAccess    = 0; 
    mStartFrame = x.mStartFrame;
    mEndFrame   = x.mEndFrame;
    mLastData   = x.mLastData;
    mDict       = x.mDict;
    mFWriter    = 0;
    mAutoUpdate = x.mAutoUpdate;
    mFrameMax   = x.mFrameMax;
    mFrameCount = 0;
    mWriteTOC   = x.mWriteTOC;
    return *this;
}

//======================================  Addition operator
Trend&
Trend::operator+=(const Trend& x) {
    for (const_chan_iter i=x.mDict.begin(); i != x.mDict.end(); ++i) {
        string name(i->first);
        ChanMap::iterator p = mDict.find(name);
	if (p != mDict.end()) p->second += i->second;
    }
    return *this;
}

//======================================  Look for specified channel
bool 
Trend::exists(const char* ChName) const {
    return mDict.find(ChName) != mDict.end();
}

//======================================  Find specified channel in the list
TrendChan& 
Trend::find(const char* ChName) {
    ChanMap::iterator p = mDict.find(ChName);
    if (p == mDict.end()) 
      throw runtime_error(string("Trend channel not defined: ")+ChName);
    return p->second;
}

const TrendChan& 
Trend::find(const char* ChName) const {
    ChanMap::const_iterator p = mDict.find(ChName);
    if (p == mDict.end()) 
      throw runtime_error(string("Trend channel not defined: ")+ChName);
    return p->second;
}

//======================================  Test if empty
bool
Trend::isEmpty(void) const {
  if (!mStartFrame) return true;
    for (const_chan_iter i=mDict.begin(); i != mDict.end(); i++) {
        if (!(i->second.isEmpty())) return false;
    }
    return true;
}

bool
Trend::isEmpty(const Time& start, const Time& stop) const {
    if (!mStartFrame) return true;
    for (const_chan_iter i=mDict.begin() ; i != mDict.end() ; i++) {
        if (i->second.getNSample(start, stop) != 0) return false;
    }
    return true;
}

//======================================  Add a trend channel
void 
Trend::addChannel(const char* Name) {
    if (mSample < Interval(1.0)) {
        cerr << "Invalid sample time in Trend: " << mName << "." << endl;
    } if (exists(Name)) {
        cerr << "Trend channel " << Name << " already exists." << endl;
    } else {
        TrendChan::validName(Name, mMonID);
        mDict[Name] = TrendChan(Name, mSample);
    }
}

//======================================  Read in the specified trend.
void
Trend::read(const string& file) {
    Dacc In(file.c_str());
    if (!In.isOpen()) throw runtime_error("Unable to open trend frame");
    In.setIgnoreMissingChannel(true);
    for (chan_iter i=mDict.begin(); i != mDict.end(); ++i) {
        i->second.setReadout(In);
    }
    In.fillData(double(mMaxPoints)*mSample);
    for (chan_iter i=mDict.begin(); i != mDict.end(); ++i) {
        i->second.setReadData(In);
	if (i == mDict.begin()) {
	    mStartFrame = i->second.refAvgSeries().getStartTime();
	    mEndFrame   = mStartFrame + double(mMaxPoints)*mSample;
	    mLastData   = i->second.refAvgSeries().getEndTime();
	} else if (mStartFrame != i->second.refAvgSeries().getStartTime() ||
		   mLastData   != i->second.refAvgSeries().getEndTime()) {
	    cerr << "Inconsistent data in channel:" << i->first 
		 << " start frame: " << mStartFrame << " start data: "
		 << i->second.refAvgSeries().getStartTime()
		 << " end frame: " << mLastData << " end data: "
		 << i->second.refAvgSeries().getEndTime() << endl;
	}
    }
}

//======================================  Set the update mode
void 
Trend::setAutoUpdate(bool enable) {
    mAutoUpdate = enable;
}

//======================================  Set the output file name
void 
Trend::setFile(const char* file) {
    const char* dmtout = getenv("DMTRENDOUT");
    if (dmtout && *dmtout && file[0] != '/' && file[0] != '.') {
	mFileName  = dmtout;
	if (*(--mFileName.end()) != '/') mFileName += "/";
	mFileName += file;
    } else {
	mFileName = file;
    }
}

//======================================  Set the frame length (in time)
void 
Trend::setFrameLen(int nSample) {
    mMaxPoints = nSample;
}

//======================================  Set the frame count
void 
Trend::setFrameCount(int nCount) {
    mFrameMax = nCount;
}

//======================================  set the IFO ID
void 
Trend::setIFO(const char* ifo) {
    if (ifo) {
        mIFO = ifo;
    } else {
        const char* ifoName = getenv("DMTIFOS");
	if (ifoName) {
	    mIFO = ifoName;
	    count_t len = mIFO.size();
	    if (mIFO.find(" ") < len || mIFO.find(",") < len || 
		mIFO.find(":") < len) mIFO = mIFO.substr(0, 1);
	} else if (getenv("LIGOSITE")) {
	    string Site(getenv("LIGOSITE"));
	    if      (Site == "LHO") mIFO = "H0";
	    else if (Site == "LLO") mIFO = "L0";
	}
	if (mIFO.empty()) mIFO = "Xn";
    }
}

//======================================  Set the trend name
void 
Trend::setMonID(const std::string& monid) {
    mMonID = monid;
}

//======================================  Set the trend name
void 
Trend::setName(const char* Name) {
    mName = Name;
}

//======================================  Set the sample time (min/sec/etc)
void 
Trend::setSample(Interval time) {
    mSample = time;
}

//=======================================  Set the trend type (min/sec/etc)
void 
Trend::setType(TrendType type) {
    mType = type;
    switch (mType) {
    case kSecond:
        mSample = Interval(1.0);
	mMaxPoints = 60;
	break;
    case kMinute:
        mSample = Interval(60.0);
	mMaxPoints = 60;
	break;
    default:
        mSample = Interval(0.0);
	break;
    }
}

//======================================  Start at a specified frame
void
Trend::setFrame(const Time& t0) {
    if (!mStartFrame) {
        startFrame(t0);
    } else if (t0 >= mEndFrame || Almost(t0, mEndFrame)) {
        if (mAutoUpdate) Update(t0);
    } else if (t0 < mStartFrame && !Almost(t0,mStartFrame)) {
        cout << "Sample Time (" << t0 << ") before frame start (" 
	     << mStartFrame << ")." << endl;
        throw runtime_error("Attempt to write an earlier frame");
    }
}

//======================================  Write a TOC with the trend frames
void 
Trend::setWriteTOC(bool enabletoc) {
    mWriteTOC = enabletoc;
}

//======================================  Set up all channels in a frame
void 
Trend::startFrame(const Time& t) {

    //----------------------------------  Calculate the frame start time
    Time::ulong_t tS = t.getS();
    switch (mType) {
    case kSecond:
        tS -= tS%60;
	break;
    case kMinute:
        tS -= tS%3600;
	break;
    default:
        break;
    }
    mStartFrame = Time(tS, 0);
    mEndFrame = mStartFrame + double(mMaxPoints)*mSample;

    //-----------------------------------  Synchronize all the channels
    for (chan_iter i=mDict.begin() ; i != mDict.end() ; i++) {
        i->second.startFrame(mStartFrame);
    }
}

//======================================  Assignment
void 
Trend::synch(const Time& t) {
    for (chan_iter i=mDict.begin() ; i != mDict.end() ; i++) {
        try {
	    i->second.synch(t);
	} catch (exception& e) {
	    cerr << "Error synchronizing channel " << i->first 
		 << ": " << e.what() << endl;
	}
    }
}

//======================================  Add TSeries data to trend point
void 
Trend::trendData(const char* Name, const TSeries& ts) {
    Time t = ts.getStartTime();
    if (!t) throw runtime_error("Attempt to trend data with t=0.");
    setFrame(t);
    find(Name).addData(ts);
    Time endSeries = t + double(ts.getNSample() - 1) * ts.getTStep();
    if (endSeries > mLastData) mLastData = endSeries;
}

//======================================  Add single datum to a trend point
void 
Trend::trendData(const char* Name, const Time& t, math_t point) {
    if (!t) throw runtime_error("Attempt to trend data with t=0.");
    setFrame(t);
    find(Name).addData(t, point);
    if (t > mLastData) mLastData = t;
}

//======================================  Conditionally write a frame
void 
Trend::Update(const Time& tLast) {
    Time t0 = !tLast ? mLastData : tLast;
    if (t0 < mEndFrame && !Almost(t0, mEndFrame)) return;
    for (Time t=mStartFrame ; !isEmpty() && t0 >= mEndFrame ; t = mEndFrame) {
	writeFrame();
	if (isEmpty()) mStartFrame = Time(0);
	else           startFrame(mEndFrame);
    }
}

//======================================  Write out a trend
void 
Trend::writeFrame(void) {

    //----------------------------------  Pad to the end, be sure there's data
    synch(mEndFrame);
    if (isEmpty(mStartFrame, mEndFrame)) return;

    //----------------------------------  Open frame writer, construct frame
    if (!mFWriter) open();
    if (!mFWriter) return;
    mFWriter->buildFrame(mStartFrame, mEndFrame-mStartFrame);
    mFWriter->addHistory(mName, Now(), "Written by Trend version $Name$");
    mFWriter->addWriterHistory();

    //----------------------------------  Fill in data for all channels.
    for (chan_iter i=mDict.begin() ; i != mDict.end() ; i++) {
        i->second.write(*mFWriter);
        i->second.discard(mEndFrame);
    }

    //----------------------------------  Write out the frame
    mFWriter->writeFrame();
    mFrameCount++;
    if (mFrameMax && mFrameCount >= mFrameMax) close();
}

//======================================  Write an NDS index file
void 
Trend::writeIndex(const char* file) const {

    //----------------------------------  Figure out full file name
    string fName;
    const char* tDir = getenv("DMTRENDOUT");
    if (tDir) {
        fName  = tDir;
	if (fName.find_last_of("/")+1 != fName.length()) fName += "/";
    }
    if (!file || !*file)           fName += "channel.cfg";
    else if (*file != '/' && tDir) fName += file;
    else                           fName  = file;

    //----------------------------------  Open index file. Bail on failure.
    ofstream out(fName.c_str());
    if (!out.good()) return;

    //----------------------------------  Specify data type,etc.
    out << "[datatype]" << endl;
    switch (mType) {
    case kSecond:
        out << "secondtrend" << endl;
	break;
    case kMinute:
        out << "minutetrend" << endl;
	break;
    case kNonStandard:
        out << "secondtrend" << endl;
	break;
    }
    out << "[signals]" << endl;

    //----------------------------------  Write out all channel names
    for (const_chan_iter i=mDict.begin() ; i != mDict.end() ; i++) {
        out << i->first << " 32bit_float" << endl;
    }
    out.close();
}

