/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef FILELIST_HH
#define FILELIST_HH

#include <list>
#include <string>
#include <iosfwd>

/**  List of frame files to be processed by a data accessor (Dacc).
  *  \brief Input frame file list.
  *  \author john.zweizig@ligo.org
  *  \version $Id: FileList.hh 7688 2016-08-15 17:18:20Z ed.maros@LIGO.ORG $
  */
class FileList {
public:
    /// Frame file list container type
    typedef std::list< std::string > file_list;

    /// Frame file list iterator type
    typedef file_list::iterator file_iter;

    /// Constant frame file list iterator type
    typedef file_list::const_iterator const_file_iter;

public:
    /**  Construct an empty frame file list.
      *  \brief Default constructor.
      */
    FileList(void);

    /**  Clear and delete a frame file list.
      *  \brief Destructor.
      */
    ~FileList(void);

    /**  Add a file path to the frame file list.
      *  \brief Add a file path.
      *  \param name Frame file path.
      */
    void addFile(const std::string& name);

    /**  Add the contents of a file list.
      *  \brief Add a list of files.
      *  \param listnm Path to file list.
      */
    void addList(const std::string& listnm);

    /**  Test whether the file list is empty.
      *  \brief Test for empty.
      *  \return true if list is empty.
      */
    bool empty(void) const;

    /**  Return a reference to the first frame file path in the list.
      *  \brief First file path.
      *  \return Reference to path string for first file in the list.
      */
    const std::string& first(void) const;

    /**  Return the first frame file path in the list.
      *  \brief First file path.
      *  \return Pointer to path string for first file in the list.
      */
    const char* getFront(void) const;

    /**  Return the number of frame file paths currently in the list.
      *  \brief List size.
      *  \return Number of file paths in the list.
      */
    unsigned long size(void) const;

    /**  Print the contents of the file list to the specified STL output 
      *  stream.
      *  \brief Print the file list.
      *  \param out Output stream to which the frame list is to be written.
      *  \return Reference to the output stream (out).
      */
    std::ostream& print(std::ostream& out) const;

    /**  Remove the first entry from the frame filepath list.
      *  \brief Remove first entry
      */
    void pop_front(void);

private:
    /**  mFile contains the input data file name(s)
      *  @memo Input data file name.
      */
    file_list mFile;
};

//======================================  Inline methods.
inline bool
FileList::empty(void) const {
    return mFile.empty();
}

inline const char*
FileList::getFront(void) const {
    if (empty()) return 0;
    return mFile.front().c_str();
}

inline void 
FileList::pop_front(void) {
    mFile.pop_front();
}

inline unsigned long
FileList::size(void) const {
    return mFile.size();
}

#endif // FILELIST_HH
