/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef DACC_HH
#define DACC_HH

#include "DaccAPI.hh"
#include "DaccIn.hh"
#include "Time.hh"
#include "Interval.hh"
#include "Channel.hh"
#include <string>
#include <iosfwd>

class ChanList;   // Hide the channel list from root.
class TSeries;
class FSeries;

/**  Dacc methods are used to access individual data channels from frames.
  *  By decoupling the data access from the frame structure, Dacc relieves
  *  the DMT user from writing frame catination code, etc. The input 
  *  data are unpacked by Dacc into the TSeries objects.
  *  @memo LIGO DMT data access class.
  *  @author  John G. Zweizig
  *  @version 1.6; Last modified June 15, 2001
  */
class Dacc : public DaccAPI, public DaccIn
{
public:
    /// Frequency series data type
    typedef FSeries fseries_type;

public:
    //--------------------------------------  Constructors and destructors.
    /**  Default (Null) constructor.
      *  \brief Default constructor
      */
    Dacc(void);

    /**  Construct and connect to source.
      *  \brief Construct and open
      *  \param Source Input path name.
      *  \param start GPS time for start of data.
      */
    explicit Dacc(const std::string& Source, const Time& start=Time(0));

#ifndef __CINT__
    /**  Construct using an already allocated frame reader.
      *  \brief Construct and connect to FrameCPP input stream.
      *  \param reader Pointer to FrameCPP input stream.
      */
    explicit Dacc(FrameCPP::IFrameStream* reader);
#endif

    /**  Destroy Dacc and release data.
      *  \brief Destructor.
      */
    ~Dacc(void);

    //---------------------------------------  Status manipulation
    /**  Set the DaccIn buffer count.
      *  @memo Set the DaccIn Buffer count.
      *  \param N Number of buffers to reserve.
      */
    void setBuffer(int N) {DaccIn::setBuffer(N);}

    /**  Set a processing flag for a specified channel. The processing flags
      *  the allow the user to ignore exceptional conditions (e.g. invalid data)
      *  are defined in the Channel::ChanFlags enumerator.
      *  @memo Set the channel processing flag.
      *  \param chan Channel name
      *  \param flag Number of flag to be set.
      *  \param state New value for the specified flag.
      */
    void setChannelFlag(const std::string& chan, Channel::ChanFlags flag, 
			bool state=true);

    /**  If the debug flag is set, explanatory messages will be printed 
      *  when an error is encountered.
      *  \brief Set the debug flag.
      *  \param N Debug print level.
      */
    void setDebug(int N) {DaccIn::setDebug(N);}

    /**  Set or clear the IgnoreMissingChannel flag. If this flag is set, a 
      *  call to fillData will succede (\e i.e the return code will be 0)
      *  even if one of more channels are not found.
      *  \brief Set the "Ignore missing channel" flag.
      *  \param yn  New setting for the "Ignore missing channel" flag.
      */
    void setIgnoreMissingChannel(bool yn) {mIgnoreMissChan=yn;}

    /**  Set the noWait flag as specified. If noWait is set, the Dacc 
      *  methods that would otherwise wait for online data (e.g. synch, 
      *  fillData) will instead return with error code -8.
      *  @memo Set the noWait flag.
      *  \param now New setting for the nowait flag.
      */
    void setNoWait(bool now=true);

    /**  The default stride is used if a time interval is not specified
      *  when fillData() is invoked.
      *  @memo Define the default Stride.
      *  \param Dt Default data stride length.
      */
    void setStride(Interval Dt);

    //--------------------------------------  Channel list manipulation
    /**  The specified channel is added to the list of channels to be
      *  processed by fillData(). fillData() looks for the channel name 
      *  first in the FrAdcData list folowed by the FrProcData list. A 
      *  channel may be included in the list 
      *  only once. Subsequent calls to addChannel() will replace the 
      *  original request. \a TSptr is used by the calling program to access 
      *  the data. If the pointer address is \c NULL or not specified, an 
      *  internal pointer is allocated, and the channel is accessed 
      *  exclusively with the refData() method. The TSeries is then created 
      *  and destroyed under Dacc control. If the pointer address is 
      *  specified a non-zero pointer indicates a pre-allocated
      *  TSeries. If the pointer is \c NULL no TSeries is pre-allocated
      *  and the TSeries will be created when fillData() is invoked. Note 
      *  that if the TSeries is not preallocated, the TSeries data vector 
      *  will have the same data type as the channel data (usually short). 
      *  \a Decimate gives the number of samples to be averaged together
      *  before they are entered into the TSeries. If \a Decimate \<= 0, it
      *  is set to 1. If no TSeries is passed to fillData() and the 
      *  Decimation \> 1, a \c float data vector is allocated.
      *  @memo Add a channel to the request list.
      *  @param Name     Name of channel to collect data from.
      *  @param Decimate Decimation factor.
      *  @param TSptr    Address of TSeries pointer.
      */
    void addChannel(const std::string& Name, int Decimate=0, TSeries **TSptr=0);

    /**  Add an process frequency series to the list. The purpose of the 
      *  method is similar to that of addChannel() except that the 
      *  channel being searched for is a processed data frequency series.
      *  @memo Add a frequency series data channel to the request list.
      *  @param Name  Name of channel to collect data from.
      *  @param FSptr Address of an frequency series pointer.
      */
    void addFSeries(const std::string& Name, fseries_type **FSptr=0);

    /**  Add a Processed data channel to the list. The purpose of the method 
      *  and its arguments are identical to those for addChannel() except 
      *  that any search for the channel is limited to processed data (\e i.e.
      *  data in FrProcData structures).
      *  @memo Add a processed data channel to the request list.
      *  @param Name     Name of channel to collect data from.
      *  @param Decimate Decimation factor.
      *  @param TSptr    Address of TSeries pointer.
      */
    void addProcessed(const std::string& Name, int Decimate=0, TSeries **TSptr=0);

    /**  Add an Adc channel to the list. The purpose of the method and its
      *  arguments are identical to those for addChannel except that any 
      *  search for the channel is limited to the raw ADC data (\e i.e.
      *  data in FrAdcData structures).
      *  @memo Add a raw data channel to the request list.
      *  @param Name     Name of channel to collect data from.
      *  @param Decimate Decimation factor.
      *  @param TSptr    Address of TSeries pointer.
      */
    void addRaw(const std::string& Name, int Decimate=0, TSeries **TSptr=0);

    /**  Add a Simulated data channel to the list. The purpose of the method 
      *  and its arguments are identical to those for addChannel() except 
      *  that any search for the channel is limited to simulated data (\e i.e.
      *  data in FrSimData structures).
      *  @memo Add a simulated data channel to the request list.
      *  @param Name     Name of channel to collect data from.
      *  @param Decimate Decimation factor.
      *  @param TSptr    Address of TSeries pointer.
      */
    void addSimulated(const std::string& Name, int Decimate=0, TSeries **TSptr=0);

    /**  The named channel is removed from the request list.
      *  @memo Remove a channel from the request list.
      *  \param Name Name of channel to be removed.
      */
    void rmChannel(const std::string& Name);

    /**  The data pointer for the frame currently in memory is incremented
      *  past the specified time. If the interval is not specified, 
      *  negative or zero, the rest of the data in the current frame are 
      *  skipped. If a discontinuity in the frame times is found, #flush#
      *  stops skipping at the start of the new data segment.
      *  @return Non-zero if an end of file is found.
      *  @memo Skip data.
      *  @param Stride Time interval to be skipped.
      */
    int flush(Interval Stride=Interval(0.0));

    /**  Data from channels specified by addChannel() are copied to TSeries
      *  objects.
      *  \brief  Fill the requested TSeries.
      *  \param Stride Time interval to be read in.
      *  \param start  Start a stride.
      *  \return 
      *  <table>
      *    <tr><td>0</td><td>Successful completion.</td></tr>
      *    <tr><td>-1</td><td>Frame start not contiguous to previous data.</td>
      *        </tr>
      *    <tr><td>-2</td><td>Sample rate incompatible with previous data.</td>
      *        </tr>
      *    <tr><td>-3</td><td>Requested data not found in current frame.</td>
      *        </tr>
      *    <tr><td>-4</td><td>Error reading frame.</td></tr>
      *    <tr><td>-5</td><td>Frame data are not self-consistent.</td></tr>
      *    <tr><td>-6</td><td>TSeries is not allocated.</td></tr>
      *    <tr><td>-7</td><td>Unsupported data type.</td></tr>
      *    <tr><td>-8</td><td>Signal received while reading.</td></tr>
      *    <tr><td>-9</td><td>Invalid data in structure.</td></tr>
      *  </table>
      */
    int fillData(Interval Stride=Interval(0.0), bool start=true);

    /**  Zero all TSeries in the channel list. Allocate sufficient
      *  memory to hold the given sample time.
      *  @memo   Reset channel TSeries.
      *  @param  Dt     Time interval to be used in allocating memory.
      */
    void zeroChans(Interval Dt);

    /**  Copy data to channel TSeries from a single frame.
      *  @memo   Fill the requested TSeries from a single frame.
      *  @param  Offset Offset of start of data from start of fram.
      *  @param  Dt     Time interval to be read in.
      *  \return 
      *  <table>
      *    <tr><td>0</td><td>Successful completion.</td></tr>
      *    <tr><td>-1</td><td>Frame start not contiguous to previous data.</td>
      *        </tr>
      *    <tr><td>-2</td><td>Sample rate incompatible with previous data.</td>
      *        </tr>
      *    <tr><td>-3</td><td>Requested data not found in current frame.</td>
      *        </tr>
      *    <tr><td>-4</td><td>Error reading frame.</td></tr>
      *    <tr><td>-5</td><td>Frame data are not self-consistent.</td></tr>
      *    <tr><td>-6</td><td>TSeries is not allocated.</td></tr>
      *    <tr><td>-7</td><td>Unsupported data type.</td></tr>
      *  </table>
      */
    int fillChans(Interval Offset, Interval Dt);

    /**  Frames are read from the specified input file(s) until one 
      *  containing the specified time is found or a time later than 
      *  than \a STime is encountered. If \a STime is not specified or 
      *  specified as %Time(0), the result is equivalent to synch().
      *  The time of the first sample to be read can be obtained with 
      *  getCurrentTime().
      *  \brief Skip to the specified time.
      *  \param STime Requested next GPS to be read.
      *  \return Return value indicates the status as follows:
      *    * 0:  Successful completion
      *    * -4: No more data frames.
      *    * -8: No data currently available and NoWait was specified.
      */
    int seek(Time STime=Time(0));

    /**  The frame that will provide that the data for the next read operation 
      *  is read into memory. If unread input data are already in the buffer,
      *  no further data are read or discarded. The current time is set to the 
      *  start of the newly read frame.
      *  \brief Synchronize the current frame buffer.
      *  \return Return value indicates the status as follows:
      *    * 0:  Successful completion
      *    * -4: No more data frames.
      *    * -8: No data currently available and NoWait was specified.
      */
    int synch(void);

    //---------------------------------  Accessors
    /**  Returns the current data collection time i.e. the time of the 
      *  first sample not used by fillData. getCurrentTime() returns 
      *  Time(0) if no frame has been read. Note that data filling may 
      *  not start at getCurrentTime if there are gaps in the data stream
      *  or if filData errors occur. Use synch to position the current 
      *  time to the start of the next frame if current frame is not 
      *  available.
      *  \brief  Get the Current time.
      *  \return Start time for next data fill.
      */
    Time getCurrentTime(void) const {return getTime()+mOffset;}

    /**  Returns the level of debug information that was requested.
      *  @memo Get the debug flag.
      *  @return Debug print level.
      */
    int getDebug(void) {return DaccIn::getDebug();}

    /**  Returns the data diration collected with the last invocation of
     *   fillData.
      *  @memo Get the duration of the last fill.
      *  @return Duration of last fillData() stride.
      */
    Interval getFillStride(void) const {return mFillStride;}

    /**  Returns the time at which data collection started with the
      *  last invocation of fillData.
      *  @memo Get the start time of the last fill.
      *  @return Start time of last fillData() stride.
      */
    Time getFillTime(void) const {return mFillTime;}

    /**  Return a pointer to message text for the specified error.
      *  @memo Get error message text.
      *  \param err Error code.
      *  @return Constant pointer to text.
      */
    static const char* getMsgText(int err);

    /**  Returns an offset of the current point in the frame.
      *  \brief Get the current offset into the frame.
      *  \return Current offset in frame data. 
      */
    Interval getOffset(void) const {return mOffset;}

    /**  Returns the time interval over which data are collected.
      *  \brief Get the stride interval.
      *  \return Default stride interval.
      */
    Interval getStride(void) const {return mStride;}

    /**  Test to see is the names channel has been requested..
      *  @memo Test if channel was requested.
      *  @return True if the specified channel has been requested.
      *  @param Name Channel name to be tested.
      */
    bool isChannelRead(const std::string& Name) const;

    /**  A formatted list of requested channels, their decimation factors, 
      *  and latest time copied is printed to the specified output stream. 
      *  \brief List requested channels.
      *  \param out STL output stream to which channels are listed.
      *  \return Output stream reference.
      */
    std::ostream& list(std::ostream& out) const;

    /**  Returns a pointer to the last %TSeries filled for the named channel.
      *  \brief Get a pointer to the data.
      *  \param name Channel name string pointer.
      *  \return Constant pointer to current data %TSeries.
      */
    const TSeries* refData(const std::string& name) const;

    /**  Returns a pointer to the last %TSeries filled for the named channel.
      *  \brief Get a pointer to the data.
      *  \param name Channel name string pointer.
      *  \return Pointer to current data %TSeries.
      */
    TSeries* refData(const std::string& name);

    /**  Returns a pointer to the last frequency series filled for the named 
      *  channel.
      *  \brief Get a pointer to frequency series data.
      *  \param name Channel name string pointer.
      *  \return Pointer to current data frequency series.
      */
    const fseries_type* refFData(const std::string& name) const;

    /**  Returns a pointer to the last frequency series filled for the named 
      *  channel.
      *  \brief Get a pointer to %fseries_type data.
      *  \param name Channel name string pointer.
      *  \return Pointer to current data %fseries_type.
      */
    fseries_type* refFData(const std::string& name);

    //---------------------------------- Make some DaccIn methods available
    const char* getFile(void) const {
	return DaccIn::getFile();}
    std::string getFrameID(void) const {
	return DaccIn::getFrameID();}
    std::ostream& listFiles(std::ostream& out) const {
    	return DaccIn::listFiles(out);}
    void addPath(const std::string& path) {
	DaccIn::addFile(path);
    }
    bool isOnline(void) const {
	return DaccIn::isOnline();
    }

    void addPathList(const std::string& path) {
	DaccIn::addFileList(path);
    }

    bool testData(bool wait) const {
	return DaccIn::waitData();
    }

    int open(void) {return DaccIn::open();}

    void close(void) {DaccIn::close();}

    //----------------------------------------------------------------------

private:
    typedef std::list<Channel> ChannelList;
    typedef ChannelList::iterator chan_iter;
    typedef ChannelList::const_iterator const_chan_iter;

    const_chan_iter findChannel(const std::string& Name, 
				Channel::ChanType t=Channel::kUnknown) const;
    chan_iter       findChannel(const std::string& Name, 
				Channel::ChanType t=Channel::kUnknown);

  private:
    /**  Start time of last fill.
      *  @memo Time of last fill.
      */
    Time mFillTime;

    /**  Time offset of the first unused sample within the current frame.
      *  @memo Time offset of the first unused sample in the current frame.
      */
    Interval mOffset;

    /**  Default data collection time interval (stride).
      *  @memo Default data collection time interval (stride).
      */
    Interval mStride;

    /**  List of requested channels.
      *  @memo List of requested channels.
      */
    ChannelList mChanList;

    /**  Flag indicating whether a fill is in progress. mFillRun is set
      *  to true by fillData() at the start of a fill. It remains true
      *  if the fill is interrupted by a signal while waiting for data.
      *  It is reset to false when the requested time series are filled
      *  with data, or when an unrecoverable error occurs.
      */
    bool     mFillRun;

    /**  Number of seconds of data already copied into the TSeries by the 
      *  current (last) fillData run.
      */
    Interval mFillOffset;

    /** Stride used by the current (last fill).
     */
    Interval mFillStride;

    /** Don't wait for data if none is currently available.
      */
    bool     mNoWait;

    /** If set, missing channels are ignored while filling data.
      */
    bool     mIgnoreMissChan;
};
#endif // DACC_HH
