#ifndef TESTPAK_HH
#define TESTPAK_HH

#include <string>
#include <iosfwd>
#include <cmath>

class Time;

/**  The testPak class is used to manage a package test function. The Package 
  *  verification procedure is assumed to be subdivided into "Sections" which
  *  are in turn subdivided into tests. The TestPak manager is used to compare
  *  numeric values or vectors with correct or incorrect values. It keeps 
  *  counts of the numbers and names of the tests being performed and counts
  *  errors detected in each step.
  *  @memo Package test manager.
  *  @author John Zweizig
  *  @version 1.1; Last modified: April 27, 2004
  */
class testPak {
public:
  typedef unsigned long count_type;
  typedef unsigned long size_type;


  /** testPak constructor.
    */
  testPak(const char* ttl, std::ostream& out);

  /** testPak Destructor.
    */
  ~testPak(void);

  /**  Start a new test section.
    */
  void startSection(const char* sec, const char* title);

  /**  Start a new subtest.
   */
  void startTest(const char* sec, const char* title);

  /**  End a test section.
    */
  void endSection(void);

  /**  End a subtest.
    */
  void endTest(void);

  /**  Print an error text.
    */
  void printError(const std::string& txt);

  /**  Print an error text with a good and bad value.
    */
  void printErrorVal(const char* txt, size_type inx, double val, double expct, 
		     double err=0);
  void printErrorVal(const char* txt, double val, double expct, double err=0);
  void printErrorVal(const char* txt, const Time& val, const Time& expct);
  bool testDiffer(double val, double expct, double err, const char* txt);
  bool testError(double val, double expct, double err, const char* txt);
  bool testError(size_type inx, double val, double expct, double err,
		 const char* txt);
  bool testError(const Time& val, const Time& expct, const char* txt);

  bool testFailed(void) const;

  //template<class T>
  //bool testVector(const T* val, const T* expct, size_type N, const char* txt);
  bool testVector(const float* val, const float* expct, size_type N, 
		  const char* txt);

private:
  std::ostream& mOut;
  std::string   mPackage;
  std::string   mSection;
  std::string   mSecTitle;
  std::string   mTest;
  std::string   mTestTitle;
  count_type    mSecCount;
  count_type    mTestCount;
  count_type    mTestSec;
  count_type    mErrorCount;
  count_type    mErrorTest;
  count_type    mErrorSec;  
};

//template<class T>
//inline bool
//testPak::testVector(const T* v, const T* x, testPak::size_type N, 
//		    const char* txt) {
//    bool rc = false;
//    for (size_type i=0; i<N; i++) rc |= testError(v[i], x[i], 0.0, txt);
//    return rc;
//}

inline bool 
testPak::testError(double val, double expct, double err, const char* txt) {
    if (val == expct) return false;
    if (err != 0 && fabs(val - expct) < err) return false;
    printErrorVal(txt, val, expct, err);
    return true;
}

inline bool 
testPak::testDiffer(double val, double expct, double err, const char* txt) {
    if (val != expct) return false;
    if (err != 0 && fabs(val - expct) > err) return false;
    printErrorVal(txt, val, val, err);
    return true;
}

inline bool
testPak::testFailed(void) const {
    return mErrorTest != 0;
}

#endif
