/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef GENERATOR_SINC_HH
#define GENERATOR_SINC_HH

#include "DataSource.hh"

namespace generator {

  /**  %Sinc is an analytical function data source that generates a sinc
    *  function of arbitrary frequency and amplitude. 
    *  The generated waveform is: 
    *
    *  \f[ x(t)  = A { \sin{\omega (t-t_0)} \over {\omega (t-t_0)} } \f]
    *
    *  \f$ \omega = 2 \pi F \f$ 
    *
    *  The wave-form parameters that may be specified by the generation
    *  control function are:
    *
    *  <table>
    *  <tr><td>Parameter </td>
    *      <td> Default </td>
    *      <td> Meaning </td>
    *  </tr>
    *  <tr><td>\c A </td>
    *      <td> -  </td>
    *      <td> Amplitude at \c t=to </td>
    *  </tr>
    *  <tr><td>\c F </td>
    *      <td> -  </td>
    *      <td> Sinc frequency (in Hz) </td>
    *  </tr>
    *  <tr><td>\c Width </td>
    *      <td> 6 </td>
    *      <td> Half-width of the waveform to be generated (in 2/F units) </td>
    *  </tr>
    *  </table> 
    *
    *  At the default width value of 6.0, the function is zero and the next
    *  peak would be \c ~A/20.
    *  @memo %Sinc function generator.
    *  @version 1.0; last modified June 15, 2004
    *  @author  John Zweizig
    */
  class Sinc : public DataSource {
  public:
      /**  Default constructor
        *  @memo Default constructor.
        */
      Sinc(void);

      /**  Construct a %Sinc data source and initialize the waveform 
        *  parameters.
	*  @memo Parameter constructor
	*  @param Amp  the amplitude at \c t=to.
	*  @param Freq the sinusoid frequency (in Hz).
	*  @param wid  the half width of the generated data in units of 2/F.
        */
      Sinc(double Amp, double Freq, double wid=6.0);

      /**  Destroy a %Sinc data source.
        *  @memo Destructor
        */
      ~Sinc(void);

      /**  Create an identical copy of a %Sinc data source.
        *  @memo Clone a %Sinc object
	*  @return Pointer to newly created %Sinc data source.
        */
      Sinc* clone(void) const;

      /**  Return a constant string containing the data source type, "Sinc"
        *  @memo Get the source data type name string.
	*  @return Constant name string pointer
        */
      const char* getDataType(void) const;

      /**  Return the duration of the data to be generated. For %Sinc this
        *  is evaluated on an event by event basis and is 2W/F.
	*  @memo Get event duration.
	*  @return Signal duration in seconds.
        */
      Interval getDuration(void) const;

  private:
      /**  Calculate the event data series.
        *  @memo Get the event signal time series
	*  @param t0 Event time
	*  @param dT Sample time
	*  @param N  Number of data samples
	*  @param data Preallocated vector into which the data will be stored.
        */
      void getSeries(const Time& t0, Interval dT, int N, gen_sample_type* data);
  };

  //====================================  Inline methods.
  inline const char*
  Sinc::getDataType(void) const {
      return "Sinc";
  }

} // namespace generator

#endif  // GENERATOR_SINC_HH
