#ifndef GENERATOR_SINGAUSS_HH
#define GENERATOR_SINGAUSS_HH

#include "DataSource.hh"

namespace generator {

  /**  %SinGauss is an analytical function data source that generates a sinusoid
    *  of arbitrary phase multiplied by a gaussian envelope (i.e. a 
    *  sine-gaussian). The generated waveform is: 
    *
    *  \f[x(t) = A  \sin(\omega(t-t_0) + \phi_0) e^{-(t-t_0)^2 / 2 \sigma^2}\f]
    *
    *  \f[ \omega = 2 \pi F ; \;  \sigma = Q/\omega ; \; \phi_0 = Phi \f]
    *
    *  The wave-form parameters that may be specified by the generation
    *  control function are:
    *
    *  <table>
    *  <tr><td> Parameter </td>
    *      <td> Default </td>
    *      <td> Meaning </td>
    *  </tr>
    *  <tr><td>\c A </td>
    *      <td> - </td>
    *      <td> Gaussian envelope amplitude at \c t=to </td>
    *  </tr>
    *  <tr><td>\c F </td>
    *      <td> - </td>
    *      <td> Sinusoid frequency (in Hz)</td>
    *  </tr>
    *  <tr><td>\c Q </td>
    *      <td> - </td>
    *      <td> Quality factor</td>
    *  </tr>
    *  <tr><td>\c Phi </td>
    *      <td>  0 </td>
    *      <td> Phase (\f$ \Phi \f$) at \c t=to (in degrees) </td>
    *  </tr>
    *  <tr><td>\c Width </td>
    *      <td> 6 </td>
    *      <td> Half-width of the waveform to be generated (in \f$ \sigma \f$
    *                units)</td>
    *  </tr>
    *  </table> 
    *
    *  At the default width value of 6.0, the gaussian envelope has decreased 
    *  to \f$1.5 \times 10^{-8}\f$ of its central amplitude.
    *  @memo Sine-Gaussian function generator.
    *  @version 1.1; last modified April 21, 2004
    *  @author  John Zweizig
    */
  class SinGauss : public DataSource {
  public:
      /**  Default constructor
        */
      SinGauss(void);

      /**  Construct a %SinGauss data source and initialize the waveform 
        *  parameters.
	*  @param Amp  the amplitude at \c t=to.
	*  @param Freq the sinusoid frequency (in Hz).
	*  @param Q    the quality factor.
	*  @param Phi  the sinusoid phase at \c t=to (in degrees).
	*  @param wid  the half width of the generated data in units of
	*              Q/(2*pi*F).
        */
      SinGauss(double Amp, double Freq, double Q, double Phi=0.0, 
	       double wid=6.0);
 
      /**  Destroy a %SinGauss data source.
        *  \brief Destructor.
        */
      ~SinGauss(void);

      /**  Create an identical copy of a %SinGauss object.
        *  \brief Clone a %SunGauss object.
	*  \return Pointer to the newly created clone.
        */
      SinGauss* clone(void) const;

      /**  Return a constant string containing the data source type "SinGauss"
        *  \brief Function name string. 
	*  \return constant character pointer to function name.
        */
      const char* getDataType(void) const;

      /**  Return the duration of the data to be generated. For %SinGauss this
        *  is evaluated on an event by event basis and is 
	*  \f$2 Width \times {Q \over F} \f$.
	*  \brief Get data duration.
	*  \return Data duration.
        */
      Interval getDuration(void) const;

  private:
      /**  Calculate the event data series.
        */
      void getSeries(const Time& t0, Interval dT, int N, gen_sample_type* data);
  };

  //====================================  Inline methods.
  inline const char*
  SinGauss::getDataType(void) const {
      return "SinGauss";
  }

} // namespace generator

#endif  // GENERATOR_SINGAUSS_HH
