/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "DataSource.hh"
#include "DVecType.hh"
#include "rndm.hh"
#include <sstream>
#include <iostream>
#include <iomanip>
#include <cmath>
#include <stdexcept>

using namespace std;

namespace generator {

  //====================================  Source constructor
  DataSource::DataSource(RepMode mode) 
    : mRptMode(mode), mRate(0.0),mSample(1./16384.), mDebug(0), mFlags(0), 
      mDataType(-1), mNextTrig(0), mCurrent(0), mLatest(0), mEventNo(0), 
      mSumTime(0)
  {
  }
  
  //====================================  Source destructor
  DataSource::~DataSource(void) {
  }

  //====================================  Default sample rate
  double
  DataSource::defaultRate(const Time& t0, Interval dt) {
      return 16384.0;
  }

  //====================================  Default sample rate
  long
  DataSource::defaultType(const Time& t0, Interval dt) {
      return DVector::t_float;
  }

  //====================================  Dump the source status
  ostream&
  DataSource::dump(ostream& out) const {
      out << "Dump of data source: " << getName() << " type: " << getDataType()
	  << " units: " << units() << endl;
      out << "Timing: ";
      if (mRptMode == kContinuous) {
	  out << "Continuous";
      } else if (mRptMode == kSingle) {
	  out << "Single trigger at " << mNextTrig;
      } else if (mRptMode == kRate) {
	  out << "Average rate = " << mRate;
      } else if (mRptMode == kFixed) {
	  out << "Fixed step of " << mRate;
      }
      out << endl;
      out << "Parameters: " << endl;
      for (const_param_iter i=mParam.begin() ; i != mParam.end(); ++i) {
	  out << "   " << i->first << ":  ";
	  i->second.dump(out);
      }
      return out;
  }

  //====================================  Dump the source defintion string
  ostream&
  DataSource::dumpDefinition(ostream& out) const {
      out << getDataType() << "(";
      for (const_param_iter i=mParam.begin() ; i != mParam.end(); ++i) {
	  if (i != mParam.begin()) out << ",";
	  out << i->first << "=";
	  i->second.dumpDefinition(out);
      }
      out << ")";
      return out;
  }

  //====================================  Set a data type
  void 
  DataSource::setDataType(const string& dtype) {
      if (dtype == "short") {
	  mDataType = DVecType<short>::getDataType();
      }
      else if (dtype == "int") {
	  mDataType = DVecType<int>::getDataType();
      }
      else if (dtype == "float") {
	  mDataType = DVecType<float>::getDataType();
      }
      else if (dtype == "double") {
	  mDataType = DVecType<double>::getDataType();
      }
      else if (dtype == "uint") {
	  mDataType = DVecType<uint32_t>::getDataType();
      }
      else {
	  throw runtime_error("Invalid data type name");
      }
  }

  //====================================  Set poisson repeat mode
  void 
  DataSource::setTriggerRate(double rate) {
      mRate    = rate;
      mRptMode = kRate;
  }
  
    //====================================  Set poisson repeat mode
    void 
    DataSource::setTriggerRepeat(Interval rpt) {
	mRate    = 1.0/double(rpt);
	mRptMode = kFixed;
    }

  //====================================  Set single event mode
  void 
  DataSource::setTriggerTime(const Time& t0) {
      mCurrent = t0;
      mRptMode = kSingle;
  }

  void 
  DataSource::setUnits(const std::string& unit) {
      mUnits = unit;
      mSeries.setUnits(unit);
  }

  //====================================  Get the data series
  TSeries 
  DataSource::getTimeSeries(const Time& t0, Interval dT) const {
      return mSeries.extract(t0, dT);
  }

  //====================================  Get a string parameter value
  const char* 
  DataSource::getParameter(const string& name) const {
      const_param_iter p = findParam(name);
      if (p == mParam.end()) return reinterpret_cast<const char*>(0);
      return p->second.getString();
  }

  //====================================  Get a numeric parameter value
  double      
  DataSource::getNumeric(const string& name) const {
      const_param_iter p = findParam(name);
      if (p == mParam.end()) return 0.0;
      return p->second.getNumeric();
  }

  //====================================  Set a parameter
  void 
  DataSource::setParameter(const string& name, const string& value) {
      if (value.find("(") == value.npos) { 
	  mParam[name].setParameter(value);
      } else { 
	  mParam[name].setDistribution(value);
      }
  }

  //====================================  Set a parameter
  void 
  DataSource::setParameter(const string& name, double value) {
      mParam[name].setParameter(value);
  }

  //====================================  Generate the appropriate TSeries
  void 
  DataSource::generate(const Time& t0, Interval dt) {
      if (mDebug) {
	  cout << "Starting generation of " << getSourceName() << " for GPS "
	       << t0.getS() << " (length " << dt << ")." << endl;
      }

      Interval tStep = getSample();
      if (!tStep) {
	  double rate = defaultRate(t0, dt);
	  if (!rate) {
	      string msg = "DataSource::generate: Unspecified sample rate in ";
	      msg += getName();
	      throw runtime_error(msg);
	  }
	  mSample = Interval(1.0/rate);
	  tStep = mSample;
      }

      //--------------------------------  Construct a time series if mSeries 
      //                                  is empty or not up to date.
      if (mSeries.isEmpty() || mSeries.getEndTime() <= t0) {
	  DVector* dv = 0;
	  long N = long(dt/tStep + 0.5);
	  if (mDataType < 0) mDataType = defaultType(t0, dt);
	  switch (mDataType) {
	  case DVector::t_short:
	      dv = new DVecType<short>(N);
	      break;
	  case DVector::t_int:
	      dv = new DVecType<int>(N);
	      break;
	  case DVector::t_float:
	      dv = new DVecType<float>(N);
	      break;
	  case DVector::t_double:
	      dv = new DVecType<double>(N);
	      break;
	  case DVector::t_uint:
	      dv = new DVecType<uint32_t>(N);
	      break;
	  default:
	      dv = new DVecType<float>(N);
	      break;
	  }
	  dv->replace_with_zeros(0, N, N);
	  mSeries = TSeries(t0, tStep, dv);
	  mSeries.setUnits(mUnits);
      }
      Time tEnd = t0 + dt;
      mSeries.extend(tEnd);

      //--------------------------------  Get first event parameters.
      if (!mCurrent) {
	  switch (mRptMode) {
	  case kRate:
	  case kFixed:
	      mCurrent = t0;
	      break;
	  case kRetrigger:
	      mTrigEnd = t0;
	      break;
	  default:
	      break;
	  }
      }
      randomize();
      if (isContinuous()) {
	  mTrigStart = t0;
	  mTrigEnd   = tEnd;
      }

      //--------------------------------  Loop while there's an event
      while (mTrigStart < tEnd && (mNextTrig != Time(0) || isContinuous())) {
	  if (mDebug > 1) {
	      cout << "   Generating ";
	      if (!isContinuous()) cout << getSourceName() << " ";
	      cout << "for GPS " << mTrigStart << " - " << mTrigEnd << endl;
	  }

	  //-----------------------------  Add in data for one trigger.
	  mSeries.extend(mTrigEnd);
	  int inx0 = mSeries.getBin(mTrigStart);
	  int len  = mSeries.getBin(mTrigEnd) - inx0;
	  TSeries source_ts;
	  getSourceTSeries(mSeries.getBinT(inx0), tStep, len, source_ts);
	  mSeries.add_overlap(source_ts);

	  //----------------------------  Save the source data.
	  if (getSaveFlag()) {
	      saveEvent(t0, dt);
	  }

	  //------------------------------  Reset the pointers.
	  mNextTrig = Time(0);
	  if (mRptMode == kSingle) mCurrent = Time(0);
	  mSumTime += mTrigEnd - mTrigStart;
	  mTrigStart = mTrigEnd;
	  randomize();
      }
      if (mDebug) cout << getSourceName() << " generation complete for GPS "
		       << t0.getS() << " - " << tEnd.getS() << ", " 
		       << mSeries.getNSample() << " samples." << endl;
      mLatest = tEnd;
  }

    //==================================  Get data from source
    void 
    DataSource::getSeries(const Time& t0, Interval dT, int N, 
			  gen_sample_type* data) {
	string msg = "Data source: ";
	msg += getName();
	msg += " not implemented";
	throw runtime_error("Data source not implemented");
    }

    //==================================  Get data from source
    void 
    DataSource::getSourceTSeries(const Time& t0, Interval dT, int N, 
				 TSeries& data)
    {
	DVecType<gen_sample_type> dvf;
	dvf.replace_with_zeros(0, dvf.size(), N);
	getSeries(t0, dT, N, dvf.refTData());
	data = TSeries(t0, dT, dvf);
	data.setUnits(mUnits);
    }

  //==================================== Select event time & set parameters
  ostream& 
  DataSource::print_stats(ostream& out) const {
      out << "Data Source: " << getName() 
	  << " Type: " << getDataType() << endl;
      out << "Source definition:          "; dumpDefinition(out) << endl;
      out << "Number of events generated: " << mEventNo << endl;
      out << "Total time:                 " << mSumTime << endl;
      return out;
  }

  //==================================== Select event time & set parameters
  void 
  DataSource::randomize(void) {
      if (!mNextTrig) {

	  //----------------------------  Generate a time
	  switch (mRptMode) {
	  case kRate: {
	      mNextTrig = mCurrent;
	      double z  = Rndm();
	      if (z <= 0) mNextTrig += Interval(999./mRate);
	      else        mNextTrig += -log(z)/mRate;
	      mCurrent = mNextTrig;
	      break;
	  }

	  case kFixed:
	      mNextTrig = mCurrent + Interval(1.0/mRate);
	      mCurrent = mNextTrig;
	      break;

	  case kSingle:
	      mNextTrig = mCurrent;
	      break;

	  case kRetrigger:
	      mNextTrig = mTrigEnd;
	      mCurrent  = mNextTrig;
	      break;

	  default:
	      mCurrent = mNextTrig;
	  }

	  //----------------------------  Generate parameters
	  if (mNextTrig != Time(0)) {
 	      bool step = true;
	      for (param_iter i=mParam.begin() ; i != mParam.end() ; ++i) {
		  if (i->second.isConst()) {
		      continue;
		  } else if (! i->second.isStep()) {
		      i->second.sampleParameter();
		  } else if (step) {
		      i->second.sampleParameter();
		      step = i->second.atStart();
		  }
	      }
	      mTrigStart = mNextTrig  + getStartOffset();
	      mTrigEnd   = mTrigStart + getDuration();
	      mEventNo++;
	  }
      }
  }

  //====================================  Release data up to the specified time
  void 
  DataSource::release(const Time& t0) {
      Time tBegin = mSeries.getStartTime();
      if (tBegin < t0) mSeries.eraseStart(t0 - tBegin);
  }

  //====================================  Save the current event.
  void 
  DataSource::saveEvent(const Time& t0, Interval dT) {
      ostringstream name;
      name << mName << setw(8) << setfill('0') << mEventNo;
      SrcEvent ev(name.str(), getSourceName(), mCurrent, mCurrent-mTrigStart, 
		  mTrigEnd-mCurrent, mParam);
      saveEvent(ev);
  }

  //====================================  Save the specified event.
  void 
  DataSource::saveEvent(const SrcEvent& ev) {
      mRecord.push_back(ev);
  }

  //====================================  Set the sample rate
  void 
  DataSource::setSample(Interval dT) {
      mSample = dT;
  }

  //====================================  Set the debuggin printout level
  void 
  DataSource::setDebug(int lvl) {
      mDebug = lvl;
  }

  //====================================  Set the source name
  void 
  DataSource::setName(const string& name) {
      mName = name;
  }

  //====================================  Set the save flag
  void 
  DataSource::setSaveFlag(bool yorn) {
      if (yorn) mFlags |=  flSvEvents;
      else      mFlags &= ~flSvEvents;
  }

  //====================================  Set the write flag
  void 
  DataSource::setWriteFlag(bool yorn) {
      if (yorn) mFlags |=  flWrite;
      else      mFlags &= ~flWrite;
  }

  //====================================  Get the source decription
  string
  DataSource::getSourceName(void) const {
      ostringstream ss;
      ss << getDataType();
      if (!mName.empty()) ss << " " << mName;
      ss << "(";
      for (const_param_iter i=mParam.begin() ; i != mParam.end() ; ++i) {
	  if (i != mParam.begin()) ss << ", ";
	  ss << i->first << "=" << i->second.getString();
      }
      ss << ")";
      return ss.str();
  }

  DataSource::SrcEvent::SrcEvent(void) {
  }

  DataSource::SrcEvent::SrcEvent(const std::string& name, 
				 const std::string& comment, 
				 const Time& t0, Interval before, 
				 Interval after, const ParamMap& map) 
    : mName(name), mComment(comment), mGPSMax(t0), 
      mTimeBefore(before), mTimeAfter(after), mAmplitude(0.0)
  {
      for (DataSource::const_param_iter i=map.begin(); i != map.end() ; i++) {
	  if (i->first == "A") mAmplitude = i->second.getNumeric();
	  mParamList.push_back(ParamEntry(i->first, i->second.getNumeric()));
      }
  }

}  // namespace generator
