/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "DMTGen.hh"
#include "DecimateBy2.hh"
#include "ParseLine.hh"
#include "FilterDesign.hh"
#include "MultiPipe.hh"
#include "Delay.hh"
#include "DaqErrFilter.hh"
#include "IfoResponse.hh"
#include "FDPipe.hh"
#include "MathOp.hh"
#include "recolor.hh"
#include "FDResample.hh"
#include "DVecType.hh"
#include "fSeries/PSD.hh"
#include "rndm.hh"
#include <signal.h>
#include <unistd.h>
#include <sys/stat.h>
#include <iostream>

using namespace std;
using namespace generator;
using namespace containers;

//======================================  Main function
int
main(int argc, const char* argv[]) {
    DMTGen myPgm(argc, argv);
    if (!myPgm.configured()) return 1;
    int rc = myPgm.generate();
    return rc;
}

//======================================  Test file readable.
inline bool 
testRead(const std::string& file) {
    return (access(file.c_str(), R_OK) == 0);
}

//======================================  Constructor
DMTGen::DMTGen(int argc, const char* argv[]) 
    : mConfigured(false), mWriteMode(kFile), mRunID(-666),
      mWriter("DMTGen", mRunID), mVerbStat(0), mDebug(0),
      mNFramesGen(0), mNFramesOut(0), mRealTime(false)
{
    //----------------------------------  Initialize 
    DataSource::init_dict(mSourceDict);

    //----------------------------------  Parse command list
    bool syntax(false);
    string config;
    for (int i=1 ; i < argc ; ++i) {
        string argi = argv[i];
        if (argi == "-conf" ) {
	    config = argv[++i];
	} else if (argi == "-debug" ) {
	    if (i+1<argc && *argv[i+1] != '-') mDebug = strtol(argv[++i], 0,0);
	    else                               mDebug = 1;
        } else if (argi == "-realtime" ) {
	    mRealTime = 1;
	} else {
	    syntax = true;
	}
    }

    //----------------------------------  Set up parameter list
    mDict.addPar("Identifier", "DMT generation run");
    mDict.addPar("RunID",         -666);
    mDict.addPar("FilePrefix", "DMTGen");
    mDict.addPar("StartGPS",         0);
    mDict.addPar("EndGPS",           0);
    mDict.addPar("GenerStride",    4.0);
    mDict.addPar("FrameLength",   16.0);
    mDict.addPar("Compression",     "");
    mDict.addPar("RealTime", mRealTime);
    mDict.addPar("RandomSeed",       0);
    mDict.addPar("SaveRndmState",   "");
    mDict.addPar("OutputDirectory", "");
    mDict.addPar("VerboseLog",      "");
    mGener.setDebug(mDebug);

    //----------------------------------  Get the configuration
    if (config.empty()) {
        cerr << "Configuration file not specified." << endl;
	syntax = true;
    }
    if (syntax) {
        cerr << "Command line syntax:" << endl;
	cerr << "  DMTGen -conf <config-file> [-debug <dbg-level>]" 
	     << " [-realtime]" << endl;
	return;
    }
    if (readConfig(config)) {
        cerr << "Error reading configuration file: " << config << endl;
	return;
    }
    Interval Stride(mDict.getDouble("GenerStride"));
    mGener.setGenerStride(Stride);

    //----------------------------------  Get the real-time flag
    mRealTime = mDict.getInt("RealTime");

    //----------------------------------  Check that the start time is valid
    mStartTime = Time(mDict.getInt("StartGPS"));
    if (mRealTime) {	
	Time now(Now() + Stride);
	unsigned long nSec = now.getS();
	nSec -= nSec % long(Stride);
	if (nSec > mStartTime.getS()) {
	    cerr << "Start time advanced to GPS " << nSec 
		 << " for real-time generation" << endl;
	    mStartTime = Time(nSec);
	}
    }
    else if (!mStartTime) {
        cerr << "Start time not specified" << endl;
	return;
    }

    mDt = mDict.getDouble("FrameLength");
    if (mDt <= Interval(0.0)) {
        cerr << "Frame length not positive" << endl;
	return;
    }

    mEndTime = Time(mDict.getInt("EndGPS"));
    if (!mEndTime && !mRealTime) {
        mEndTime = mStartTime + mDt;
    } 
    else if (!mRealTime && mEndTime < mStartTime) {
        cerr << "Run ends before start" << endl;
	return;
    }

    mRunID   = mDict.getInt("RunID");
    mIdent   = mDict.getString("Identifier");

    //----------------------------------  Get output frame path specifiers
    mFrameName.set_prefix(mDict.getString("FilePrefix"));
    string OutDir = mDict.getString("OutputDirectory");
    mFrameName.set_directory(OutDir);
    string::size_type l = strlen(DACC_ONLDEV);
    if (OutDir.size() > l && OutDir.substr(0,l) == DACC_ONLDEV) {
	mWriteMode = kSMP;
    }
    mFrameName.set_dt(mDt);

    mWriter.setRunID(mRunID);
    mWriter.setName(mIdent);
    string compression = mDict.getString("Compression");
    if (compression == "gzip") {
        mWriter.setCompress(FrVectRef::kGZip);
    } else  if (compression == "zero-suppress-2") {
        mWriter.setCompress(FrVectRef::kZeroSuppress2);
    } else  if (compression == "zero-suppress-4") {
        mWriter.setCompress(FrVectRef::kZeroSuppress4);
    } else  if (compression == "zero-suppress-or-gzip") {
        mWriter.setCompress(FrVectRef::kZeroSuppOrGZip);
    } else if (!compression.empty()) {
        cout << "Unrecognized compression mode: " << compression 
	     << " Ignored!" << endl;
    }

    //----------------------------------  Check for a random seed
    std::string stateFile  = mDict.getString("SaveRndmState");
    unsigned long rndmSeed = mDict.getInt("RandomSeed");

    if (!stateFile.empty() && testRead(stateFile)) {
	RestoreRndmState(stateFile);
    }
    else if (rndmSeed != 0.0) {
	RndmSeed(rndmSeed);
    }

    //----------------------------------  Verbose stream
    string verbFile = mDict.getString("VerboseLog");
    if (!verbFile.empty()) {
	mVerbStream.open(verbFile.c_str(), ofstream::out | ofstream::app);
	if (mVerbStream.is_open()) {
	    mVerbStat = 1;
	} else {
	    cerr << "Unable to open verbose log file: " << verbFile << endl;
	    mVerbStat = 0;
	}
    }


    //----------------------------------  Catch termination signals
    mTerm.add(SIGTERM);
    mTerm.add(SIGINT);

    //----------------------------------  All done;
    mConfigured = true;
}

//======================================  Destructor
DMTGen::~DMTGen(void) {
    //----------------------------------  Save random state
    std::string stateFile = mDict.getString("SaveRndmState");   
    if (!stateFile.empty()) {
	cout << "Saving random number generator state to: " 
	     << stateFile << endl;
	SaveRndmState(stateFile);
    }

    //----------------------------------  Delete the source dictionary.
    DataSource::kill_dict(mSourceDict);

    //-----------------------------------  Delete all the defined filters
    for (filt_iter i=mFilterMap.begin(); i != mFilterMap.end(); ++i) {
	Pipe* p = const_cast<Pipe*>(i->second);
	delete p;
	i->second = 0;
    }
    mFilterMap.clear();

    //-----------------------------------  Print statistics
    mGener.print_stats(cout);
    cout << "Generation end time =        " << mGener.getLatest().getS() 
	 << endl;
    cout << "Number of frames generated = " << mNFramesGen << endl;
    cout << "Number of frames written =   " << mNFramesOut << endl;
    cout << "DMTGen Completed." << endl;
}

//======================================  Read fSeries
fSeries
readFSeries(const std::string& file) {
    fSeries fs;

    ParseLine rf(file.c_str());
    if (!rf.isOpen()) {
	cout << "readFSeries: Unable to open filter file: " << file << endl;
	throw runtime_error("readConfig: can not open fSeries file");
    }

    std::vector<double> response;
    double f0 = -1;
    double dF = 0;
    while (rf.getLine() >= 0) {
	double f = rf.getDouble(0);
	double r = rf.getDouble(1);
	if (response.empty()) {
	    f0 = f;
	}
	else if (dF == 0) {
	    dF = f - f0;
	}
	else if (size_t((f-f0)/dF + 0.5) != response.size()) {
	    cout << "readFSeries: Unequal frequency spacing: f = " << f
		 << " expected: " << f0 + size_t() * dF << endl;
	    throw runtime_error("readConfig: can not open fSeries file");
	}
	response.push_back(r);
    }
    DVectD dv(response.size(), &response.front());
    return fSeries(f0, dF, Time(0), 1.0/dF, dv);
}

//======================================  Unfiltered decimation class
class decimate : public NullPipe {
public:
    using NullPipe::apply;
    decimate(int factor) : _factor(factor) {}
    ~decimate(void) {}
    decimate* clone(void) const {return new decimate(*this);}
    TSeries apply(const TSeries& ts);
private:
    long _factor;
};

TSeries
decimate::apply(const TSeries& ts) {
    prep(ts);
    long off = long((ts.getStartTime() - _start)/_step + 0.5) % _factor;
    if (off) off = _factor - off;
    long nSample = (ts.getNSample() + _factor - off - 1) / _factor;
    Time t0 = ts.getStartTime() + _step * double(off);
    Interval dT = _step * double(nSample * _factor);
    return ts.extract(t0, dT).decimate(_factor);
}

//======================================  Read a configuration file.
bool
DMTGen::readConfig(const std::string& file) {
    ParseLine pl(file.c_str());
    if (!pl.isOpen()) return true;
    pl.setLog(cout);

    //----------------------------------  Loop over commands.
    bool syntax = false;
    while (pl.getLine() >= 0) {
        int nArgs = pl.getCount();
	if (!nArgs) continue;
	string cmd = pl[0];

	//------------------------------  Specify a channel to be generated.
	if (cmd == "Channel") {
	    int cid = mGener.addChannel(pl[1]);
	    string units;

	    //--------------------------  Loop over global modifiers
	    int i=2;
	    for ( ; i<nArgs && *(pl[i]) == '-' ; ++i) {
	        string argi = pl[i];
		if (argi == "-daqchain") {
		    const_filt_iter pF=mFilterMap.find(pl[++i]);
		    if (pF == mFilterMap.end()) {
		        cout << "Unrecognized filter name" << endl;
			syntax = true;
		    } else {
		        mGener.refChannel(cid).setDaqChain(pF->second);
		    }
		} else if (argi == "-sample") {
		    mGener.refChannel(cid).setSample(pl.getDouble(++i));
		} else if (argi == "-units") {
		    units = pl[++i];
		} else {
		    cout << "Unrecognized argument: " << argi << endl;
		    syntax = true;
		}
	    }
	    
	    //--------------------------  Loop over source specifiers
	    for ( ; i<nArgs ; ++i) {
	        const char* p  = pl[i];
		const char* p0 = p;
		string name;
		const Pipe* filter=0;
		MultiPipe*  mpfilt=0;
		for ( ; *p ; p++) {
		    if (*p == '|') {
		        if (p0 == pl[i]) {
			    name = string(p0, p-p0);
			} else {
			    string filt(p0, p-p0);
			    const_filt_iter pF=mFilterMap.find(filt);
			    if (pF == mFilterMap.end()) {
			        cout << "Unidentified filter: " << filt << endl;
				syntax = true;
				break;
			    }
			    if (!filter) {
			        filter = pF->second;
			    } else {
			        if (!mpfilt) {
				    mpfilt = new MultiPipe;
				    mpfilt->addPipe(*filter);
				}
				mpfilt->addPipe(*(pF->second));
			    }
			}
			p0 = p + 1;
		    }
		}

		//----------------------  handle the string after the last |
		if (p0 == pl[i]) {
		    name = string(p0, p-p0);
		} else if (p != p0) {
		    string filt(p0, p-p0);
		    const_filt_iter pF=mFilterMap.find(filt);
		    if (pF == mFilterMap.end()) {
		        cout << "Unidentified filter: " << filt << endl;
			syntax = true;
			continue;
		    }
		    if (!filter) {
		        filter = pF->second;
		    } else {
		        if (!mpfilt) {
			    mpfilt = new MultiPipe;
			    mpfilt->addPipe(*filter);
			}
			mpfilt->addPipe(*(pF->second));
		    }
		}

		//----------------------  Add the source descriptor
		int sid = mGener.getSourceID(name);
		if (sid < 0) {
		    cerr << "Unidentified source: " << name << endl;
		    syntax = true;
		    continue;
		} else if (mpfilt) {
		    mGener.refChannel(cid).addComponent(sid, mpfilt);
		} else {
		    mGener.refChannel(cid).addComponent(sid, filter);
		}
		if (units.empty()) {
		    units = mGener.refSource(sid).units();
		} else if (!mGener.refSource(sid).units().empty() &&
			   units != mGener.refSource(sid).units()) {
		    cerr << "Warning: Source " << name << " units string is "
			 << " not compatible with channel " << pl[1] << endl;
		}
		delete mpfilt;
	    }
	    mGener.refChannel(cid).setUnits(units);
	}

	//------------------------------  Define a data source
	else if (cmd == "Define") {
	}

	//------------------------------  Add a filter to the database.
	else if (cmd == "Filter") {
	    if (nArgs < 3) {
	        cout << "Syntax error in filter definition: <3 arguments"
		     << endl;
		cout << "    Filter description syntax is: "
		     << "'Filter <name> <type> <arguments>'" << endl;
		syntax = true;
		continue;
	    }

	    //--------------------------  Don't allow redefinition.
	    string name=pl[1];
	    if (mFilterMap.find(name) != mFilterMap.end()) {
	        cout << "Filter: " << name << " is already defined." <<endl;
		syntax = true;
		continue;
	    }

	    //--------------------------  Create Daq Error 'filter'.
	    string typ=pl[2];
	    Pipe*  fPipe(0);
	    if (typ == "DaqErr") {
	        if (nArgs < 4) {
		    cout << "Error in Filter specification. Syntax is:" << endl
			 << "Filter <name> DaqErr <type> [-clust <width>]" 
			 << "[-avgwid <width>] [-rate <err-rate>]" << endl;
		    syntax = true;
		    continue;
		}
		DaqErrFilter temp(pl[3]);
		for (int j=4; j<nArgs; j++) {
		    string argj = pl[j];
		    if (argj == "-clust") {
		        temp.setClusterWidth(pl.getInt(++j));
		    }
		    else if (argj == "-avgwid") {
		        temp.setAvgWidth(pl.getDouble(++j));
		    }
		    else if (argj == "-rate") {
		        temp.setErrRate(pl.getDouble(++j));
		    }
		    else {
		        cout << "Unrecognized argument: " << argj << endl;
			syntax = true;
			break;
		    }
		}
		fPipe = temp.clone();
	    }

	    //--------------------------  Create a time delay 'filter'
	    else if (typ == "Decimate") {
	        if (nArgs < 4 || nArgs > 5) {
		    cout << "Error in Filter specification. Syntax is: "
			 << "Filter <name> Decimate <decimation factor>"
			 << "[<type>]" << endl;
		    syntax = true;
		    continue;
		}
		string type = "simple";
		if (nArgs == 5) type = pl[4];
		long factor = pl.getInt(3);
		if (type == "simple") {
		    fPipe = new decimate(factor);
		}
		else if (type == "decimateby2") {
		    int nStage = 0;
		    int fact = factor;
		    while (fact > 1) {
			nStage++;
			fact /= 2;
		    }
		    if (!nStage || factor != (1 << nStage)) {
			cout << "Invalid decimation factor for DecimateBy2: "
			     << factor << ". Factor must be a power of 2." 
			     << endl;
			syntax = true;
			continue;
		    }
		    fPipe = new DecimateBy2(nStage);
		}
		else {
		    cout << "Error in decimation filter type. Valid types are:"
			 << endl;
		    cout << "  simple, decimateby2" << endl;
		    syntax = true;
		    continue;		    
		}
	    }

	    //--------------------------  Create a time delay 'filter'
	    else if (typ == "Delay") {
	        if (nArgs < 4) {
		    cout << "Error in Filter specification. Syntax is: "
			 << "Filter <name> Delay <delay-time>" << endl;
		    syntax = true;
		    continue;
		}
		fPipe = new Delay(pl.getDouble(3));
	    }

	    //--------------------------  Design a filter
	    else if (typ == "Design") {
	        if (nArgs < 5 || pl.getDouble(3) <= 0.0) {
		    cout << "Error in Filter specification. Syntax is: "
			 << "Filter <name> Design <sample-rate> <descriptor>"
			 << endl;
		    syntax = true;
		    continue;
		}
		FilterDesign fd(pl[4], pl.getDouble(3));
		fPipe = fd.release();
	    }

	    //--------------------------  f-Domain filter.
	    else if (typ == "FDfilter") {
	        if (nArgs < 4) {
		    cout << "Error in Filter specification. Syntax is: "
			 << "Filter <name> FDFilter <file-name>" << endl;
		    syntax = true;
		    continue;
		}
		DFT dft;
		static_cast<fSeries&>(dft) = readFSeries(pl[3]);
		fPipe = new FDPipe(FDFilter(dft), FDPipe::outmux);
	    }

	    //--------------------------  Arithmetic operation with a constant
	    else if (typ == "MathOp") {
		if (nArgs < 4) {
		    cout << "Error in MathOp filter specification. Syntax is: "
			 << "Filter <name> MathOp <op1> <val1> [<op2> <val2>...]"
			 << endl;
		    syntax = true;
		    continue;
		}
		MultiPipe mp;
		for (int i=3; i<nArgs; i+=2) {
		    double val = (i+1)<nArgs? pl.getDouble(i+1): 0.0;
		    fPipe = new MathOp(pl[i], val);
		    if (nArgs > 5) {
			mp.addPipe(fPipe);
			fPipe = 0;
		    }
		}
		if (!fPipe) fPipe = mp.clone();
	    }

	    //--------------------------  t-Domain recoloring filter
	    else if (typ == "Recolor") {
	        if (nArgs < 4) {
		    cout << "Error in Recolor filter specification. Syntax is: "
			 << "Filter <name> Recolor <out-psd> [-inPSD <in>]"
			 << endl;
		    syntax = true;
		    continue;
		}
		PSD inPSD, outPSD;
		static_cast<fSeries&>(outPSD) = readFSeries(pl[3]);
		outPSD.refDVect() *= outPSD.refDVect();
		double filt_len = 0.0;
		for (int j=4; j<nArgs; j++) {
		    string argj = pl[j];
		    if (argj == "-inPSD") {
			if (++j < nArgs) {
			    static_cast<fSeries&>(inPSD) = readFSeries(pl[j]);
			    inPSD.refDVect() *= inPSD.refDVect();
			} else {
			    syntax = true;
			    break;
			}
		    }
		    else if (argj == "-length") {
			if (++j < nArgs) {
			    filt_len = pl.getDouble(j);
			} else {
			    syntax = true;
			    break;
			}
		    }
		    else {
			cerr << "Invalid recolor argument: " << argj << endl;
			syntax = true;
			break;
		    }
		}
		if (syntax) break;
		if (inPSD.empty()) fPipe = new recolor(outPSD);
		else               fPipe = new recolor(inPSD, outPSD);
		if (filt_len) {
		    dynamic_cast<recolor*>(fPipe)->setFilterLength(filt_len);
		}
	    }

	    //--------------------------  f-Domain resampling filter
	    else if (typ == "Resample") {
	        if (nArgs < 4 || pl.getDouble(3) <= 0.0) {
		    cout << "Error in Filter specification. Syntax is: "
			 << "Filter <name> Resample <new-sample-rate>"
			 << endl;
		    syntax = true;
		    continue;
		}
		FDResample fdr(pl.getDouble(3));
		fPipe = new FDPipe(fdr, FDPipe::overlap);
	    }
 
 	    //--------------------------  f-Domain noise spectrum
 	    else if (typ == "IfoNoise") {

		//----------------------  Check argument validity
 	        if (nArgs < 4) {
 		    cout << "Error in Filter specification. Syntax is: "
 			 << "Filter <name> ifoNoise <ifo-name>" << endl;
 		    syntax = true;
 		    continue;
 		}

		//----------------------  Generate PSD for the IFO
		string ifoname(pl[3]);
		double sampleRate = 16384;
		double nSeconds   = 1.0;
		int    nBin = int(0.5 * sampleRate * nSeconds + 0.5) + 1;
		DVectD dvd(nBin);

		//----------------------  Aligo noise (as parameterized by PS)
		if  (ifoname == "aLigo") {
		    double f0 = 215;
		    for (int i=1; i<nBin; i++) {
			double x = double(i) / f0;
			double x2 = x*x;
			dvd[i]  = pow(x, -4.14);
			dvd[i] -= 5.0 / x2;
			dvd[i] += 111.0 * (1 - x2 + 0.5*x2*x2)/(1 + 0.5*x2);
			dvd[i] *= 1e-49;
			//dvd[i] = sqrt(dvd[i]);
		    }
		}
	    
		//----------------------  Geo noise (as parameterized by PS)
		else if  (ifoname == "Geo") {
		    double f0 = 150;
		    for (int i=1; i<nBin; i++) {
			double x = double(i) / f0;
			dvd[i]  = pow(3.4*x, -30);
			dvd[i] += 34.0 / x;
			double x2 = x*x;
			dvd[i] += 20.0 * (1 - x2 + 0.5*x2*x2)/(1 + 0.5*x2);
			dvd[i] *= 1e-46;
			//dvd[i] = sqrt(dvd[i]);
		    }
		}

 		//----------------------  Virgo noise (as parameterized by PS)
		else if  (ifoname == "Virgo") {
		    double f0 = 500;
		    dvd[0] = 1.0;
		    for (int i=1; i<nBin; i++) {
			double x = double(i) / f0;
			dvd[i]  = 1.0 + pow(6.23*x, -5) + 2.0/x + x*x;
			dvd[i] *= 3.24e-46;
			//dvd[i] = sqrt(dvd[i]);
		    }
		}
 
		//----------------------  bogus ifo name
		else {
		    cerr << "Unknown or unimplemented ifo " << ifoname << endl;
		    syntax = true;
		    continue;
		}
		dvd[0] = dvd[1];
		PSD psd;
		static_cast<fSeries&>(psd) 
		   = fSeries(0, 1/nSeconds, Time(0), Interval(nSeconds), dvd);
		fPipe = new recolor(psd);
	    }

	    //--------------------------  f-Domain calibrated response filter
	    else if (typ == "IfoResponse") {
	        if (nArgs < 4) {
		    cout << "Error in Filter specification. Syntax is: "
			 << "Filter <name> IfoResponse <xml-calib>"
			 << endl;
		    syntax = true;
		    continue;
		} else if (!testRead(pl[3])) {
		    cout << "Unable to access file: " << pl[3] << endl;
		    syntax = true;
		    continue;
		}
		IfoResponse fdr("Calibration", pl[3]);
		fPipe = new FDPipe(fdr, FDPipe::overlap);
	    }

	    //--------------------------  Unknown filter type
	    else {
	        cout << "Unrecognized filter type: " << typ << endl;
		syntax = true;
		continue;
	    }

	    //--------------------------  Add the design to the database.
	    if (fPipe) mFilterMap[name] = fPipe;
	}

	//------------------------------  Include a configuration file
	else if (cmd == "Include") {
	    if (nArgs < 2) {
	        cerr << "Missing file name. Include syntax: " << endl;
	        cerr << "   Include <file-path>" << endl;
	        syntax = true;
	    } else {
	        syntax = readConfig(pl[1]);
	    }
	}

	//------------------------------  Set specified parameters
	else if (cmd == "Parameter") {
	    if (mDict.exists(pl[1])) {
	        mDict.setPar(pl[1], pl[2]);
	    } else {
	        cerr << "Parameter " << pl[1] << " isn't defined." << endl;
		syntax = true;
	    }
	}

	//------------------------------  Define a data source
	else if (cmd == "Source") {
	    DataSource* src = DataSource::parse(pl[2], mSourceDict);
	    if (!src) {
	        cerr << "Unparseable source description - source " 
		     << pl[1] << " not defined." << endl;
		syntax = true;
		continue;
	    }
	    src->setDebug(mDebug);
	    int sid = mGener.addSource(pl[1], src);
	    for (int i=3; i<nArgs; ++i) {
	        string argi = pl[i];
		if (argi == "-rate") {
		    double rate = pl.getDouble(++i);
		    mGener.refSource(sid).setTriggerRate(rate);
		} 
		else if (argi == "-repeat") {
		    Interval rpt = pl.getDouble(++i);
		    mGener.refSource(sid).setTriggerRepeat(rpt);
		}
		else if (argi == "-sample") {
		    double rate = pl.getDouble(++i);
		    mGener.refSource(sid).setSample(Interval(1./rate));
		} 
		else if (argi == "-simdata") {
		    mGener.refSource(sid).setWriteFlag(true);
		} 
		else if (argi == "-simevent") {
		    mGener.refSource(sid).setSaveFlag(true);
		} 
		else if (argi == "-time") {
		    double t0 = pl.getDouble(++i);
		    long gps = long(t0);
		    mGener.refSource(sid).setTriggerTime(Time(gps)+Interval(t0-gps));
		} 
		else if (argi == "-type") {
		    mGener.refSource(sid).setDataType(pl[++i]);
		} 
		else if (argi == "-units") {
		    mGener.refSource(sid).setUnits(pl[++i]);
		} 
		else {
		    cerr << "Unidentified argument: " << argi << endl;
		    syntax = true;
		    break;
		}
	    }
	}

	//------------------------------  Undefined configuration command.
	else {
	    cerr << "Undefined command: " << cmd << endl;
	    syntax = true;
	}
    }
    return syntax;
}

//======================================  Generate the data.
int
DMTGen::generate(void) {
    //

    //----------------------------------  Loop over generation strides.
    Time writeTime = mStartTime;
    Interval dtGen = mDict.getDouble("GenerStride");
    for (Time now=mStartTime ; !mTerm && (!mEndTime || now < mEndTime); 
	 now += dtGen) {
        if (mDebug) cout << "Generating data for GPS " << now.getS()
			 << " (generation length=" << dtGen << ")" << endl;
	try {
	    mGener.generate(now, dtGen);
	} catch (std::exception& e) {
	    cerr << "Exception: " << e.what() 
		 << " caught while generating data for gps " << now.getS()
		 << "-" << (now+dtGen).getS() << endl;
	    return 2;
	}
	if (mDebug) cout << "Data generated to " << mGener.getLatest().getS()
			 << endl;
	Time endFrame = writeTime + mDt;
	while (mGener.getLatest() >= endFrame) {
	    if (mRealTime && Now() < endFrame) {
		long wait_us = long ((endFrame - Now()) * 1e6);
		if (mDebug) cout << "Waiting " << wait_us 
				 << "us for end-frame time (" 
				 << endFrame.totalS() << ")" << endl;
		usleep(wait_us);
	    }
	    mNFramesGen++;
	    if (mDebug) cout << "Writing data for GPS " << now.getS()
			     << " (frame length=" << mDt << ")" << endl;
	    writeFrame(writeTime, mDt);
	    mGener.releaseSourceData(endFrame);
	    mGener.releaseChannelData(endFrame);
	    writeTime = endFrame;
	    endFrame += mDt;
	}
    }

    //----------------------------------  Write out (short?) last frame
    if (writeTime < mEndTime) {
        mGener.generate(mEndTime, Interval(0));
	Interval dt = mGener.getLatest() - writeTime;
	if (double(dt) > 0) {
	    mNFramesGen++;
	    writeFrame(writeTime, dt);
	}
	writeTime += dt;
	mGener.releaseSourceData(writeTime);
	mGener.releaseChannelData(writeTime);
    }
    return 0;
}

//======================================  Write generated data
void 
DMTGen::writeFrame(const Time& t0, Interval dT) {
    Time now(Now());
    if (mVerbStat) {
	mVerbStream << now.getS() << " put " << t0.getS();
    }

    //----------------------------------  Start the frame
    mWriter.buildFrame(t0, dT);
    mWriter.addHistory("DMTGener", now, mIdent);

    //----------------------------------  Write out selected simulated data
    int  nSrc  = mGener.getNSource();
    for (int i=0 ; i < nSrc ; ++i) {
        DataSource& src(mGener.refSource(i));
	if (src.getSaveFlag()) {
	    unsigned long N=src.getNEvent();
	    unsigned long nCopied=0;
	    for (unsigned long j=0 ; j<N ; j++) {
	        const DataSource::SrcEvent& sev(src.refEvent(j));
	        if (sev.getGPSMax() >= t0 + dT) break;
		mWriter.addSimEvent(sev.getName(), 
				    sev.getComment(), 
				    sev.getInputs(), 
				    sev.getGPSMax(),
				    sev.getTimeBefore(), 
				    sev.getTimeAfter(), 
				    sev.getAmplitude(),
				    sev.getParamList()
				    );
		nCopied++;
	    }
	    src.eraseEvent(nCopied);
	}
	if (src.getWriteFlag()) {
	    mWriter.addSimSeries(src.getName(), src.getTimeSeries(t0,dT));
	}
    }

    //----------------------------------  Write data channels and detector info
    int  nChan = mGener.getNChannel();
    string dList;
    for (int i=0 ; i < nChan ; ++i) {
        GenChan& chan(mGener.refChannel(i));
	string cName = chan.getChannel();
	if (dList.find(cName.substr(0,3)) == string::npos) {
	    dList += cName.substr(0,3);
	    mWriter.addStdDetector(cName.c_str());
	    if (mDebug) cout << "Adding detector: "<< cName.substr(0,2)<< endl;
	}
	mWriter.addRawSeries(chan.getChannel(),chan.response().extract(t0,dT));
	if (mDebug) cout << "  writing channel " << chan.getChannel()
			 << " (" << chan.response().getNSample() 
			 << " samples)" << endl; 
    }

    //----------------------------------  Open a writer to a frame file
    string outdir(mFrameName.dir_name(t0));
    if (mWriteMode == kFile) {
	mFrameName.make_dir(outdir);
	string file_path = mFrameName.file_path(t0, dT);
	if (mDebug) cout << "  opening file: " << file_path << endl;
	if (mWriter.open(file_path.c_str(), true)) {
	    mWriter.erase();
	    cerr << "DMTGen: Failed to open frame " << file_path
		 << " for output." << endl;
	    if (mVerbStat) {
		mVerbStream << "0 Failed to open frame: " << file_path << endl;
		mVerbStream.flush();
	    }
	    return;
	}
    } 

    //----------------------------------  Open a writer to shared memory.
    else {
	if (mWriter.open(outdir.c_str(), true)) {
	    mWriter.erase();
	    cerr << "DMTGen: Failed to open partition " << outdir
		 << " for output." << endl;
	    if (mVerbStat) {
		mVerbStream << "0 Failed to open partition: " << outdir << endl;
		mVerbStream.flush();
	    }
	    return;
	}
    }

    //----------------------------------  Write the frame
    try {
	if (mWriter.writeFrame()) {
	    if (mVerbStat) {
		mVerbStream << "0 Write error." << endl;
		mVerbStream.flush();
	    }
	    cerr << "Error writing frame with GPS " << t0.getS()
		 << " to output Directory " << outdir << endl;
	} else {
	    if (mVerbStat) {
		mVerbStream << "0 OK." << endl;
		mVerbStream.flush();
	    }
	    mNFramesOut++;
	}
    } catch (std::exception& e) {
	if (mVerbStat) {
	    mVerbStream << "0 Caught exception in write: " << e.what() << endl;
	    mVerbStream.flush();
	}
	cerr << "Caught exception in writeFrame: " << e.what() << endl;
    }

    //----------------------------------  Clean up
    mWriter.close();
    mWriter.erase();
}
