/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef GDSPLOT_HH
#define GDSPLOT_HH

#include <string>
#include "Time.hh"
class TCanvas;
class TSeries;
class DVector;
class TH1;
class WelchPSD;

namespace containers {
    class fSeries;
    class ASD;
    class PSD;
}

class GDSPlot {
public:
    /**  Construct a GDSPlot instance. The GDSPlot constructor is passed a 
      *  optional pointer to the plot unit used by the underlying plotting 
      *  package (Root). If no plot device is specified, One is constructed 
      *  by GDS for this GDSPlot instance.
      *  \brief Construct a new GDSPlot instance.
      *  \param tc Pinter to the Root TCanvas instance to be used.
      */
    GDSPlot(TCanvas* tc=0);

    /**  Destroy this GDSPlot instance. 
      *  \brief Destructor.
      */
    ~GDSPlot(void);

    /**  Plot specified colored boxes.
      *  \brief Plot colored boxes.
      *  \param nbox Number of boxes to be plotted.
      *  \param x0   Vector of minimum x values for each box.
      *  \param x1   Vector of maximum x values for each box.
      *  \param y0   Vector of minimum y values for each box.
      *  \param y1   Vector of maximum y values for each box.
      *  \param z    Vector z values (colors) for each box.
      */ 
    void boxes(size_t nbox, const double* x0, const double* x1, 
	       const double* y0, const double* y1, const double* z);

    /**  Build a legend from existing traces. The legend is plotted at the 
      *  position indicated by \a pos.
      *  \brief Build a legend.
      *  \param pos Legend position composed of 'l' (left), 'c' (x center),
      *             'r' (right), 't' (top), 'm' (y middle) or 'b' (bottom).
      */
    void legend(const std::string& pos="br");
    
    /**  Clear a canvas and prepare for a new plot. All setting are maintained.
      *  \brief Clear canvas.
      */
    void new_plot(void);

    /**  Plot a Spectrum (PSD) instance.
      *  \brief Plot a spectrum
      *  \param psd  PSD to be plotted.
      */
    void plot(const containers::ASD& asd);

    /**  Plot a Spectrum (PSD) instance.
      *  \brief Plot a spectrum
      *  \param psd  PSD to be plotted.
      */
    void plot(const containers::fSeries& pfs);

    /**  Plot a Spectrum (PSD) instance.
      *  \brief Plot a spectrum
      *  \param psd  PSD to be plotted.
      */
    void plot(const containers::PSD& psd);
    
    /**  Plot a time-series instance.
      *  \brief plot a time series
      *  \param ts Time series (TSeries) instance to be plotted.
      */
    void plot(const TSeries& ts);
    
    /**  Make a 2D plot of a data vector.
      *  \brief Plot a data vector.
      *  \param dvec  Data vector (DVector) instance to be plotted.
      *  \param xmin  x value of first bin.
      *  \param dx    x value increment
      *  \param title Title string for this plot.
      */
    void plot(const DVector& dvec, double xmin, double dx, 
	      const std::string& title="");

    /**  Write the current Canvas cntents to the specified file. The file codex
      *  is inferred from the extension of the file name. At present the 
      *  implemented codices include pdf and png.
      *  \brief Print the plot.
      *  \param file File name to ewhich the plot is printed.
      */
    void print(const std::string& file) const;

    /**  Read a file containing a series of colors defined by their RGB values
      *  mapped against a scale value in the range (0, 1). The format of the 
      *  file is 4 columns (scale, red, green, blue) and amy contain comments
      *  started with a '#'.
      *  \brief Read a palette file.
      *  \param palette Palette file name.
      */
    void read_palette(const std::string& palette);

    /**  Set the root TCanvas address.
      *  \brief Set the root TCanvas.
      *  \param tc  Pointer to the canvas to be used.
      *  \param kill_on_exit If set, the canvas will be deleted on destruction.
      */
    void set_canvas(TCanvas* tc, bool kill_on_exit=false);

    /**  Set the color to be used in 1D (histogram, time series, etc.) plots.
      *  \brief set trace color.
      *  \param color as defined by root color map.
      */
    void set_color(int trace_col);

    /**  Set the color to be used in 1D (histogram, time series, etc.) plots.
      *  \brief set trace color by index number.
      *  \param color_index as defined by root color map.
      */
    void set_color_index(int color_index);

    /**  Set the pallet to the default color map. For now, the default color 
      *  map is "jet";
      *  \brief Use defaul color map.
      */
    void set_palette(void);

    /**  Set the pallet to a named color map. The currently defined color 
      *  maps are "jet", "hot", "copper" and "bone".
      *  \brief Select a pre-defined color map
      *  \param palette Color map name.
      */
    void set_palette(const std::string& palette);

    /**  Set x, and y ranges for following plot(s).
      *  \brief Set x and y range.
      *  \param xmin Minimum x value.
      *  \param xmax Maximum x value.
      *  \param ymin Minimum y value.
      *  \param ymax Maximum y value.
      */
    void set_range(double xmin, double xmax, double ymin, double ymax);

    /**  Set the smoothing. the \a nSmooth argument indicates the number 
      *  of smoothing passes are to be performed. Setting this argument 
      *  zero disables smoothing. The \a smoothOpt argument soecifies the 
      *  kernel to be used. 2D kernels include "k5a" (five points in each
      *  direction, default), "k5b" stronger than k5a, and "k3a".
      *  \brief Set smoothing parameters.
      *  \param nSmooth Number of smoothing passes or zero.
      *  \param smoothOpt Smoothing kernel.
      */
    void set_smooth(int nSmooth, const std::string& smoothOpt);

    /**  Set the graphics style. The default style (0) sets pads to flat white,
      *  disables statistics boxes, etc. Style (1) is used for plots with 
      *  z-axix color-bar and leaves room on the right for the z-axis title. 
      *  This function is used in the constructor.
      *  \brief Set the graphics style.
      *  \param id Style identifier.
      */
    void set_style(int id=0);

    /**  Set the plot size in.
      *  \brief set plot size.
      *  \param xsize X dimension in cm.
      *  \param ysize Y dimension in cm.
      */
    void set_size(double xsize, double ysize) const;

    /**  Set the trace name used in the legend.
      *  \brief Set trace name.
      *  \param name Trace name to be used in the legend;
      */
    void set_trace_name(const std::string& name);

    /**  Set the z range (color dimension) for x-y plots.
      *  \brief Set the z (color) dimension range.
      *  \param zmin Minimum z value.
      *  \param zmax Maximum z value.
      */
    void set_zrange(double zmin, double zmax);

    /**  Plot a spectrogram from the specified time series. If fMin or fMax are
      *  omitted or \e \<=\ 0, they are replaced with \e fMin=1/tSeg and 
      *  \e fMax=f_Nyquist. 
      *  \brief Plot a spectrogram.
      *  \param ts   %Time series
      *  \param tSeg %Time interval to be plotted.
      *  \param fMin Minimum frequesncy.
      *  \param fMax Maximum frequency
      */
    void spectrogram(const TSeries& ts, Interval tSeg,
		     double fMin=0, double fMax=0, 
		     const WelchPSD* psdest=0);
    
    /**  Plot a 3D surface using color for the z axis.
      *  \brief Generate a 3d plot.
      *  \param nx  Number of x bins
      *  \param x   Vector of x bin edges (nx+1 elements).
      *  \param ny  Number of y bins
      *  \param y   Vector of y bin edges (ny+1 elements).
      *  \param z   List of nx*ny bin contents with column number (x coordinate)
      *             varying most rapidly, (\e i.e. z[0,0], z[0,1], ... 
      *             z[0,nx-1], z[0,1], ... z[nx-1, ny-1]).
      */
    void surf(int nx, const double* x, int ny, const double* y, 
	      const double* z);

    /**  Set the plot title string.
      *  \brief Plot title string.
      *  \param titl Tile string.
      */
    void title(const std::string& titl);

    /**  Plot a welch PSD estimate of the specified time series. This uses
      *  the default settings, i.e. 1 second span with 50% overlap and Hanning 
      *  window. The fMin and fMax parameters epcify the range of frequencies
      *  to be plotted. If fMax is specified/defaulted to zero, the upper
      *  limit is the nyquist frequency.
      *  \brief Plot estimated PSD.
      *  \param ts Time series to estimate PSD.
      *  \param fMin Minimum frequency to plot
      *  \param fMax Maximum frequency to plot
      */
    void welch(const TSeries& ts, double fMin=0, double fMax=0);

    /**  Set the x-axis label string.
      *  \brief X-axis label.
      *  \param titl Text to be plotted below x axis.
      */
    void xlabel(const std::string& titl);

    /**  Set the x axis scale to logarithmic/linear
      *  \brief Set logarithmic x scale.
      *  \param logx set to true for logarithmic or false for linear scale. 
      */
    void xlog(bool logx=true);

    /**  Treat the x axis as a time. Set the unitsas appropriate to avoid 
      *  excessively large or small numbers.
      *  \brief Set a time axis.
      *  \param dt Time increment.
      *  \param xttl x title to be used.
      */
    double xTimeScale(double dt, const std::string& xttl="Time");
    
    /**  Set the y-axis label string.
      *  \brief Y-axis label.
      *  \param titl Text to be plotted to the left of the y axis.
      */
    void ylabel(const std::string& titl);

    /**  Set the y axis scale to logarithmic/linear
      *  \brief Set logarithmic y scale.
      *  \param logy set to true for logarithmic or false for linear scale. 
      */
    void ylog(bool logy=true);

    /**  Set the z-axis label string.
      *  \brief Z-axis label.
      *  \param titl Text to be plotted to the right of the Z axis color bar.
      */
    void zlabel(const std::string& titl);

private:
    /** Set the title and axis labels from the stored values.
     */
    void set_axes(TH1& histo);
    void set_palette(size_t n, const double* x, const double* r,
		     const double* g, const double* b, bool reverse=false);

private:
    TCanvas* mCanvas;
    bool     myCanvas;
    int    mTrace;
    double mXmin;
    double mXmax;
    double mYmin;
    double mYmax;
    double mZmin;
    double mZmax;
    bool   mLogX;
    bool   mLogY;
    int    mPalletID;
    int    mNextColor;
    int    mNSmooth;
    std::string mTitle;
    std::string mTraceName;
    std::string mXlabel;
    std::string mYlabel;
    std::string mZlabel;
    std::string mSmoothOpt;
};
#endif // !defined(GDSPLOT_HH)

