/* -*-  mode: c++; c-basic-offset: 4; -*- */
#ifndef STR_STACK_HH
#define STR_STACK_HH

#include "eval_stack.hh"
#include <string>
#include <vector>

/**  The %str_token is a concrete instance of the eval_token API. %str_tokens
  *  are used in evaluation formulae with the calc_engine and the str_stack.
  *  Each %str_token contains a value that may be either a string or a number.
  *  Numeric values are stored as both a string an a double precision 
  *  floating point value to prevent loss of precision and needless 
  *  conversion.
  */
class str_token : public eval_token {
public:
    /**  Data types. For now the only data types are string, numeric and 
      *  logical (boolean values). In the future this should be templated 
      *  to a more descriptive class.
      */
    enum type_enum {
	kNone,     ///< Not a data type
	kString,   ///< String token
	kNumeric,  ///< Numeric token
	kLogical   ///< Boolean token
    };
    using eval_token::datype;

public:
    //----------------------------------  Constructors
    /**  Default (empty) token constructor.
      *  \brief Default constructor.  
      */
    str_token(void);

    /**  Construct a token from a string. The token type may be specified 
      *  explicitly.
      *  \brief String constructor.
      *  \param s Token string.
      *  \param ty Token type.
      */
    str_token(const std::string& s, type_enum ty=kNone);

    /**  Destroy a token
      *  \brief Destructor.
      */
    virtual ~str_token(void);

    //----------------------------------  eval_token virtual methods
    /**  Clone a token.
      *  \brief clone a token.
      *  \return Pointer to the cloned token.
      */
    str_token* clone(void) const;

    /**  Get the enumerated token type code.
      *  \brief Get the token type
      *  \return Token type.
      */
    datype  getType(void) const;

    /**  Get the token contents cast to a boolean value.
      *  \brief Get logical token value.
      *  \return Boolean value.
      */
    bool    getLogical(void) const;

    /**  Get the token contents cast to a numeric value.
      *  \brief Get numeric token value.
      *  \return Numeric value.
      */
    double  getNumeric(void) const;

    /**  Get the token contents cast to a string value.
      *  \brief Get token string value.
      *  \return String value.
      */
    std::string  getString(void) const;

    /**  Set the token to a boolean value.
      *  \brief Set boolean token.
      *  \param yorn Boolean value.
      */
    void setLogical(bool yorn);

    /**  Set the token to a numeric value.
      *  \brief Set numeric token.
      *  \param val Numeric value.
      */
    void setNumeric(double val);

    /**  Set the token to a string value.
      *  \brief Set string token.
      *  \param s String value.
      */
    void setString(const std::string& s);

    //----------------------------------  Specialized methods.
    /**  Get a reference to the token string.
     *  \brief Get token string reference.
     *  \return Reference to the token string.
     */
    std::string& refString(void);

    /**  Get a constant reference to the token string.
     *  \brief Get constant token string reference.
     *  \return Constant reference to the token string.
     */
    const std::string& refString(void) const;

    /**  Set the token type based on the string contents.
      *  \brief Set token type.
      */
    void setType(void);

private:
    std::string mValue;
    double      mNumeric;
    type_enum   mType;
};

//====================================== inline functions
inline std::string
str_token::getString(void) const {
    return mValue;
}

inline std::string& 
str_token::refString(void) {
    return mValue;
}

inline const std::string& 
str_token::refString(void) const {
    return mValue;
}

/**  The evaluation stack holds currently active intermediate results.
  *  In the initial implementation, data are stored as strings.
  *  @memo Evaluation stack.
  *  @author John Zweizig
  *  @version 1.0; Last modified July 6, 2004
  */
class str_stack : public eval_stack {
public:
    using eval_stack::push;

    /**  Evaluation stack constructor.
      *  \brief Default constructor.
      */
    str_stack(void);

    /**  Evaluation stack destructor. 
      *  \brief Destructor.
      */
    ~str_stack(void) {}

    /**  Dump stack contents.
      *  \brief Dump stack contents.
      *  \param out Output stream.
      */
    void dump(std::ostream& out) const;

    /**  Perform an operation on the top stack entries.
      *  \brief Perform an operation.
      *  \param op Operation to be performed.
      */
    void evaluate(OpsEnum op);

    /**  Test whether stack entry is numeric.
      *  \brief Test whether token is numeric.
      *  \param lvl Stack position of token to evaluate.
      *  \return true if specified token is numeric.
      */
    bool numeric(int lvl) const;

    /**  Convert a string to a token and push it onto the top of the stack.
      *  \brief Push a string onto the stack.
      *  \param str String to be pushed onto the stack.
      */
    void push(const std::string& str);

    /**  Push a numeric value. The double precision float value is converted
      *  to a string before being pushed on to the stack.
      *  \brief Push a number onto the stack.
      *  \param val Numeric value to be pushed onto the stack.
      */
    void push(double val);

    /**  Push a boolean value onto the stack.
      *  \brief Push a boolean onto the stack.
      *  \param val Boolean value to be pushed onto the stack.
      */
    void push(bool val);
};

inline bool 
str_stack::numeric(int off) const {
    return type(off) == str_token::kNumeric;
}

#endif // !defined(STR_STACK_HH)
