/* -*-  mode: c++; c-basic-offset: 4; -*- */
#include "str_stack.hh"
#include <sstream>
#include <iomanip>
#include <stdexcept>
#include <cmath>
#include <cstdlib>

using namespace std;

//======================================  String stack class
str_token::str_token(void) 
  : mType(kNone)
{
}

str_token::str_token(const string& s, type_enum t)
  : mValue(s), mType(t)
{
    if (t == kNumeric || (!s.empty() && t == kNone)) setType();
}

str_token::~str_token(void) 
{
}

str_token* 
str_token::clone(void) const {
    return new str_token(mValue, mType);
}

eval_token::datype
str_token::getType() const {
    return mType;
}

bool
str_token::getLogical(void) const {
    bool rc = false;
    switch (mType) {
    case kNumeric:
	rc = getNumeric() != 0.0;
	break;
    case kLogical:
        rc = (mValue == "true");
	break;
    default:
        throw runtime_error("Can't convert to bool");
    }
    return rc;
}

double
str_token::getNumeric() const {
    if (mType == kNumeric) return mNumeric;
    const char* ptr = mValue.c_str();
    double x = strtod(ptr, const_cast<char**>(&ptr));
    if (*ptr) throw runtime_error("Not a valid number");
    return x;
}

void 
str_token::setLogical(bool yorn) {
    if (yorn) mValue = "true";
    else      mValue = "false";
    mType = kLogical;
}

void 
str_token::setNumeric(double val) {
    long ival = long(val);
    ostringstream str;
    str << setprecision(16);
    if (double(ival) == val) str << ival;
    else                     str << val;
    mValue = str.str();
    mNumeric = val;
    mType  = kNumeric;
}

void 
str_token::setString(const string& s) {
    mValue = s;
    setType();
}

void 
str_token::setType() {
    if (mValue.empty()) {
        mType = kNone;
    } else {
        const char* p = mValue.c_str();
	mNumeric = strtod(p, const_cast<char**>(&p));
	if (!*p) mType = kNumeric;
	else if (mValue == "true") mType = kLogical;
	else if (mValue == "false") mType = kLogical;
	else mType = kString;
    }
}

//======================================  String stack class
str_stack::str_stack(void) {
}

//======================================  Eval stack Dump
void 
str_stack::dump(std::ostream& out) const {
    out << "-----------------  Stack Dump, size=" << size() << endl;
    for (unsigned int inx=size(); inx; ) {
	out << --inx << "  ";
	unsigned int ioff = size() - inx;
        switch (type(ioff)) {
	case str_token::kNone:
	  out << "None    ";
	  break;
	case str_token::kString:
	  out << "String  ";
      	  break;
	case str_token::kNumeric:
	  out << "Numeric ";
      	  break;
	case str_token::kLogical:
	  out << "Logical ";
	}
	out << peek(ioff).getString() << endl;
    }
}

//======================================  Perform operation on the top elements
//                                        of the stack.
void 
str_stack::evaluate(OpsEnum op) {
    double toparg;
    string topstr;
    bool   topBool;
    switch (op) {
    case kAdd:
      push(pop_numeric() + pop_numeric());
      break;
    case kSub:
      toparg = pop_numeric();
      push(pop_numeric() - toparg);
      break;
    case kMpy:
      push(pop_numeric() * pop_numeric());
      break;
    case kDiv:
      toparg = pop_numeric();
      push(pop_numeric() / toparg);
      break;
    case kModulo:
      toparg = pop_numeric();
      push(fmod(pop_numeric(), toparg));
      break;
    case kLogAnd:
      //--> Note that only one false argument will be popped in 
      //    "pop_logical() && pop_logical()"
      topBool = pop_logical();
      push(pop_logical() && topBool);
      break;
    case kLogOr:
      //--> Note that only one true argument will be popped in 
      //    "pop_logical() || pop_logical()"
      topBool = pop_logical();
      push(pop_logical() || topBool);
      break;
    case kCmpEq:
      if (numeric(1) && numeric(2)) {
	  push(pop_numeric()==pop_numeric());
      } else {
	  push(pop_string()==pop_string());
      }
      break;
    case kCmpNE:
      if (numeric(1) && numeric(2)) {
	  push(pop_numeric()!=pop_numeric());
      } else {
	  push(pop_string()!=pop_string());
      }
      break;
    case kCmpLt:
      if (numeric(1) && numeric(2)) {
	toparg = pop_numeric();
	push(pop_numeric() < toparg);
      } else {
	topstr = pop_string();
	push(pop_string() < topstr);
      }
      break;
    case kCmpLE:
      if (numeric(1) && numeric(2)) {
	toparg = pop_numeric();
	push(pop_numeric() <= toparg);
      } else {
	topstr = pop_string();
	push(pop_string() <= topstr);
      }
      break;
    case kCmpGt:
      if (numeric(1) && numeric(2)) {
	toparg = pop_numeric();
	push(pop_numeric() > toparg);
      } else {
	topstr = pop_string();
	push(pop_string() > topstr);
      }
      break;
    case kCmpGE:
      if (numeric(1) && numeric(2)) {
	toparg = pop_numeric();
	push(pop_numeric() >= toparg);
      } else {
	topstr = pop_string();
	push(pop_string() >= topstr);
      }
      break;
    case kOr:
    case kAnd:
	throw runtime_error(string("Undefined operation: ")+getOpSym(op));
      break;
    default:
      throw logic_error("Invalid opcode in evaluate");
    }
}

//======================================  Push a numeric value
void 
str_stack::push(double val) {
    str_token* t = new str_token;
    t->setNumeric(val);
    push(t);
}

//======================================  push a bool value
void 
str_stack::push(bool val) {
    str_token* t = new str_token;
    t->setLogical(val);
    push(t);
}

//======================================  Push a string value.
void
str_stack::push(const string& str) {
    push(new str_token(str, str_token::kString));
}
