/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef DatTEMPLATETRIG_HH
#define DatTEMPLATETRIG_HH

#include "DatEnv.hh"
#include "MonServer.hh"
#include "FixedLenTS.hh"
#include "Trend.hh"
#include "TrigClient.hh"
#include "autopipe.hh"
#include <vector>
#include <string>

//
//    Data Monitor template class.

/**  @name DatTemplateTrig
  *  @memo Template of a data environment (DatEnv) based monitor.
  *  The DatTemplate monitor is a template for a data monitor program.
  *  DatTemplate makes a simple calculation of the standard deviation of
  *  a list of channels, serves the results for display by the DMT viewer,
  *  stores the result as a minute trend and generates segments and/or 
  *  triggers if the RMS excedes specified limits.
  *
  *  {\bf {\Large Running DatTemplate}}
  *
  *  The DatTemplate command line syntax is:
  *  \begin{verbatim}
       DatTemplateTrig [-conf <config-file>] [-frames <number>] [-stride <time>]
     \end{verbatim}
  *  The #<number># argument specifies the number of data strides to process
  *  before terminating the program. The #<time># argument specifies the data
  *  stride. If "-dvIndex" is specified, a trend index file will be written
  *  to '\$DMTOUTPUT/channel.cnf'.
  *
  *  {\bf {\Large Configuration File}}
  *
  *  DatTemplate reads a configuration file specifying one channel per line.
  *  The channel options are 
  *  \begin{verbatim}
     <channel-name> [-filter <filter>] [-settle <s-time>] [-seg <seg-thresh>] \
                    [-trig <trig-thresh>]
     \end{verbatim}
  *
  *
  *  {\bf {\Large DatTemplate Output}}
  *
  *  {\bf Trend channels}
  *  DatTemplate produces a minute trend of the sigma value called 
  *  #<channel>_sigma#
  *
  *  {\bf Monitor data channels}
  *  DatTemplate publishes a time series history of the sigma values 
  *  with the name #<channel>_sigma#
  *
  *  @author John Zweizig; Caltech/LIGO
  *  @version 1.1; Last modified October 23, 2008
  */
//{
//}

class Pipe;

class chan_data {
public:
    chan_data(const std::string& chan);
    ~chan_data(void);
    float computeSigma(const TSeries& ts);
    const char* getChannel(void) const;
    const char* getSigmaName(void) const;
    const TSeries& refHistory(void) const;
    void setFilter(const std::string& f);
    void setSettle(Interval dt);
    void setSeg(double thresh);
    void setTrig(double thresh);
    void trigger(TrigClient& tc);
private:
    std::string mChannel;       //  Channel name
    std::string mFilter;        //  Filter description
    Interval    mSettle;        //  Settling time.
    std::string mSigmaName;     //  Channel data and trend name
    auto_pipe   mPipe;          // Pipe pointer
    FixedLenTS  mHistory;       //  X sigma history.
    double      mSegThresh;     //  Optional segment threshold
    double      mTrigThresh;    //  Optional trigger threshold
    double      mLastRms;       //  Save the RMS
    Time        mLastTime;       //  Save the Last time stamp
    bool        mSegState;      //  Segment state
    Time        mSegStart;      //  Segment start time
};

class DatTemplateTrig : public DatEnv, MonServer {
public:
    DatTemplateTrig(int argc, const char *argv[]); // Constructor
    ~DatTemplateTrig(void);                        // Destructor
    void ProcessData(void);                      // Process 1 data stride
    void configure(const char* file);            // Configure channels
    void Attention(void);                        // Handle Message signal.
public:
    typedef std::vector<chan_data>    chan_list;
    typedef chan_list::iterator       chan_iter;
    typedef chan_list::const_iterator const_chan_iter;
private:
    int         maxFrame;       //  Number of frames to process
    Interval    mStep;          //  Time stride.
    Trend       mTrend;         //  Trend writer
    chan_list   mList;          //  Channel list
    TrigClient  mTrigC;         // Trigger client
};

//======================================  Inline methods
inline const char*
chan_data::getChannel(void) const {
    return mChannel.c_str();
}

inline const char*
chan_data::getSigmaName(void) const {
    return mSigmaName.c_str();
}

inline const TSeries&
chan_data::refHistory(void) const {
    return mHistory;
}


#endif     //  DatTEMPLATEMC_HH
