//
//    Skeleton Frame processing routine Template
//
#include "DatTemplateMC.hh"
#include "FilterDesign.hh"

#ifndef __CINT__

//-->  The next three lines are needed if you are going to generate triggers.
//     The text in PIDTITLE should describe the monitor function.
#define PIDCVSHDR "$Header: https://redoubt.ligo-wa.caltech.edu/svn/gds/trunk/DMT/Templates/DatTemplateMC.cc 7128 2014-07-16 22:29:21Z john.zweizig@LIGO.ORG $"
#define PIDTITLE  "Data monitor template"
#include "ProcIdent.hh"

#include "Dacc.hh"
#include "ParseLine.hh"
#include <iostream>
#include <string>
#include <cstdlib>
#include <pwd.h>
#include <unistd.h>

using namespace std;

//======================================  Generate the main routine.
EXECDAT(DatTemplateMC)
#endif               // !def(__CINT__)

//======================================  Skeleton object constructor.
DatTemplateMC::DatTemplateMC(int argc, const char *argv[]) 
  : DatEnv(argc, argv), maxFrame(999999), mStep(2.0)
{
    //----------------------------------  Look for arguments
    bool dvIndex = false;
    const char* config = "DatTemplateMC.cfg";

    bool syntax  = false;
    for (int i=1 ; i<argc ; i++) {
        string argi = argv[i];
	if (argi == "-conf") {
	    config = argv[++i];
	} else if (argi == "-frame") {
	    maxFrame = strtol(argv[++i], 0, 0);
	} else if (argi == "-stride") {
	    mStep = strtod(argv[++i], 0);
	} else if (isDatEnvArg(argv[i])) {
	    i++;
	} else {
	    syntax = true;
	    cerr << "Unrecognized argument: " << argi << endl;
	}
    }

    //----------------------------------  Bail out if command line not correct
    if (syntax) {
        cerr << "Command syntax:" << endl;
	cerr << argv[0] 
	     << " [-conf <file>] [-frames <nMax>] [-stride <time>] [-dvIndex] \\"
	     << endl
	     << "      [<DatEnv-args>]" 
	     << endl;
	finish();
	return;
    }

    //----------------------------------  Build the configuration database
    configure(config);

    //----------------------------------  set the data accessor
    getDacc().setStride(mStep);       // in before calling ProcessData

    //----------------------------------  Specify DMT Viewer channel names
    //--- replaced cuserid() with getpwuid - K. Thorne June, 2011
    struct passwd *pws;
    pws = getpwuid(geteuid());
    setServerName(pws->pw_name);

    //----------------------------------  Set up the trender
    mTrend.setName("DatTemplateMC");    //  Set trend frame name
    mTrend.setFrameCount(1);          //  Set frames per file
    mTrend.setType(Trend::kMinute);   //  Set trend type (minute trends)
    mTrend.setAutoUpdate(false);      //  Avoid automatic screwup.

    //----------------------------------  Add channels to trends and Server
    int N = mList.size();
    for (int i=0; i<N; ++i) {
        getDacc().addChannel(mList[i].getChannel());
	serveData(mList[i].getSigmaName(), &mList[i].refHistory(), 0);
	mTrend.addChannel(mList[i].getSigmaName());       
    }

    //----------------------------------  Optional dataviewer index
    if (dvIndex) {
        string filename;
	const char* dmtout = getenv("DMTOUTPUT");
	if (dmtout) {
	    filename = dmtout;
	    filename += "/";
	}
	filename += "channel.cfg";
	mTrend.writeIndex(filename.c_str());
    }
}

//======================================  COnfigure
void
DatTemplateMC::configure(const char* file) {
    ParseLine pl(file);

    bool syntax  = false;
    while (pl.getLine() >= 0) {
        int nArg = pl.getCount();
        string Chan = pl[0];
	chan_data ch(Chan);
	for (int i=1; i<nArg; ++i) {
	    string argi = pl[i];
	    if (argi == "-filter") {
	        ch.setFilter(pl[++i]);
	    } else if (argi == "-settle") {
	        ch.setSettle(pl.getDouble(++i));
	    } else {
	        syntax = true;
		cerr << "Unrecognized argument: " << argi << endl;
	    }
	}
	mList.push_back(ch);
    }

    //----------------------------------  Bail out if command line not correct
    if (syntax) {
        cerr << "Channel description syntax:" << endl;
	cerr << "<channel> [-filter <filter>] [-settle <s-time>] \\"
	     << endl;
	finish();
	return;
    }
}

//======================================  Skeleton object destructor.
DatTemplateMC::~DatTemplateMC() {
    cout << "DatTemplateMC is finished" << endl;
}

//======================================  Frame processing function.
void
DatTemplateMC::ProcessData(void) {
    int N = mList.size();
    for (int i=0; i<N; ++i) {
        //------------------------------  Get pointers to the current data.
        const TSeries* ts = getDacc().refData(mList[i].getChannel());
	if (!ts || ts->isEmpty()) {
            cout << "Channel: " << mList[i].getChannel() << " was not found." 
		 << endl;
	    continue;
	}

	//------------------------------  Calculate the sigma
	float Sig = mList[i].computeSigma(*ts);

	//------------------------------  Trend the data points
	Time t0 = ts->getStartTime();
	mTrend.trendData(mList[i].getSigmaName(), t0, Sig);
    }
    mTrend.Update();
}


//======================================  Handle Messages
void
DatTemplateMC::Attention(void) {
    MonServer::Attention();
}

//======================================  chan_data constructor
chan_data::chan_data(const std::string& chan)
  : mChannel(chan), mSettle(0.0), mHistory(43200)
{
    mSigmaName = mChannel + "_sigma";
}

//======================================  chan_data copy constructor
chan_data::chan_data(const chan_data& c)
  : mChannel(c.mChannel), mFilter(c.mFilter),  mSettle(c.mSettle), 
    mSigmaName(c.mSigmaName), mPipe(c.mPipe), mHistory(c.mHistory)
{}

//======================================  chan_data destructor
chan_data::~chan_data(void) {
}

//======================================  Compute channel sigma
float
chan_data::computeSigma(const TSeries& ts) {

    //----------------------------------  Construct the filter.
    //     Construct a filter if it is defined, but it isn't already
    //     there. The filter must be constructed here after the sample 
    //     rate is available.
    //
    if (mPipe.null() && !mFilter.empty()) {
        double samplerate(1./double(ts.getTStep()));
	FilterDesign fd(mFilter.c_str(), samplerate);
	mPipe.set(fd.release());
    }

    //----------------------------------  Filter the data
    //     The auto_pipe filters or passes the data depending on whether 
    //     a filter has been defined.
    TSeries tf(mPipe(ts));

    //----------------------------------  Calculate channel sigma.
    float Sig = sqrt( tf*tf/tf.getNSample() - pow(tf.getAverage(), 2) );

    //----------------------------------  set up data for display.
    mHistory.fixedAppend(tf.getStartTime(), tf.getInterval(), &Sig);
    return Sig;
}

//======================================  Set filter value
void 
chan_data::setFilter(const string& f) {
    mFilter = f;
    mPipe.set(0);
}

//======================================  Set settle time
void 
chan_data::setSettle(Interval dt) {
    mSettle = dt;
}
