#ifndef LSC_EMULATE_HH
#define LSC_EMULATE_HH

#include "TSeries.hh"
#include "FilterDB.hh"
#include <list>
#include <string>

class DynConfig;

/**  The LscEmul class emulates the workings of the front-end LSC software. 
  *  The computations are made in stages with the inputs and outputs of 
  *  each stage corresponding to channels that can be read out. The stages 
  *  are as follows:
  *  \begin{enumerate}
  *    \item Calculation of I and Q phases
  *    \begin{itemize}
  *      \item Inputs:     Error photodiode signals
  *      \item Outputs:    Combined error signals (AS\_I, AS\_Q, POB\_I, 
  *                        POB\_Q, RFL\_I, RFL\_Q).
  *      \item Processing: 
  *            The raw error signals are summed over all appropriate 
  *            photodiode signals giving raw I and Q phase signals. The
  *            raw signals are offset by specified constants, optionally
  *            dewhitened and rotated by a specified phase to give the 
  *            true I and Q phases.
  *    \end{itemize}
  *    \item Calculation of control signals
  *    \begin{itemize}
  *      \item Inputs:      Error Signals
  *      \item Outputs:     Control signals (MICH, PRC, DARM, CARM)
  *      \item Processing:  
  *            Linear combinations of the error signal phases are calculated
  *            by multiplying a vector of the error signals by the input
  *            matrix.
  *    \end{itemize}
  *    \item 
  *    \begin{itemize}
  *      \item Inputs:
  *      \item Outputs:
  *      \item Processing:
  *    \end{itemize}
  */
class LscEmul {
public:
    enum SeriesIn {
        kRawAs1I,
	kRawAs1Q,
        kRawAs2I,
	kRawAs2Q,
        kRawAs3I,
	kRawAs3Q,
        kRawAs4I,
	kRawAs4Q,
	kRawPoI,
	kRawPoQ,
	kRawRflI,
	kRawRflQ,
	kDataAsI,
	kDataAsQ,
	kDataPoI,
	kDataPoQ,
	kDataRflI,
	kDataRflQ,
	kDArmCtl,
	kMichCtl,
	kPrcCtl,
	kCArmCtl
    };

    enum SeriesOut {
	kPrepAsI,
	kPrepAsQ,
	kPrepPoI,
	kPrepPoQ,
	kPrepRflI,
	kPrepRflQ,
	kDArmCalc,
	kMichCalc,
	kPrcCalc,
	kCArmCalc
    };

    class LscPd {
    public:
        LscPd(const std::string& name);
        bool active(void) const;
        TSeries getPortI(const TSeries& PdI, const TSeries& PdQ);
        TSeries getPortQ(const TSeries& PdI, const TSeries& PdQ);
        void setup(DynConfig& f, const std::string& lscPfx);
        void setFilter(const FilterDB& f);
    private:
        std::string  _name;
        double       _I_offset;
        double       _Q_offset;
        double       _I_gain;
        double       _Q_gain;
        double       _I_limit;
        double       _Q_limit;
        double       _phase;
        double       _pi;
        FilterModule _I_fm;
        FilterModule _Q_fm;
    };

public:
    LscEmul(DynConfig& cfg, const std::string& ifo=std::string(""));
    ~LscEmul(void);
    bool compare(SeriesOut x) const;
    void emulate(void);
    const TSeries& getSeries(SeriesOut out) const;
    void reset(void);
    void setDebug(bool x);
    void setIfo(const std::string& ifo);
    void setSeries(SeriesIn in, const TSeries& ts);
    void setup(const Time& t0);
    void getLscConfig(const Time& t0);

//======================================  Utility functions
private:
    void addLscVbl(const std::string& name, double& vbl, double def);
    void defLscConfig(void);

//======================================  LscEmulate control variables
private:
    DynConfig&  mDyn;
    FilterDB    mFiltDB;
    std::string mLscPrefix;
    std::string mIFO;
    bool        mDebug;

    LscPd  mAs1Pd;
    LscPd  mAs2Pd;
    LscPd  mAs3Pd;
    LscPd  mAs4Pd;
    LscPd  mRflPd;
    LscPd  mPobPd;

    double mITMTRX_00;
    double mITMTRX_01;
    double mITMTRX_02;
    double mITMTRX_03;
    double mITMTRX_04;
    double mITMTRX_05;
    double mITMTRX_10;
    double mITMTRX_11;
    double mITMTRX_12;
    double mITMTRX_13;
    double mITMTRX_14;
    double mITMTRX_15;
    double mITMTRX_20;
    double mITMTRX_21;
    double mITMTRX_22;
    double mITMTRX_23;
    double mITMTRX_24;
    double mITMTRX_25;
    double mITMTRX_30;
    double mITMTRX_31;
    double mITMTRX_32;
    double mITMTRX_33;
    double mITMTRX_34;
    double mITMTRX_35;

    double mDARM_OFFSET;
    double mMICH_OFFSET;
    double mPRC_OFFSET;
    double mCARM_OFFSET;

    double mDARM_GAIN;
    double mMICH_GAIN;
    double mPRC_GAIN;
    double mCARM_GAIN;

    double mDARM_LIMIT;
    double mMICH_LIMIT;
    double mPRC_LIMIT;
    double mCARM_LIMIT;

    double mDARM_SW1;
    double mMICH_SW1;
    double mPRC_SW1;
    double mCARM_SW1;

    double mDARM_SW1A;
    double mMICH_SW1A;
    double mPRC_SW1A;
    double mCARM_SW1A;

    double mLscPi;

//======================================  Intermediate data series
private:
    //----------------------------------  IIR Filters
    FilterModule mDArmFm;
    FilterModule mMichFm;
    FilterModule mPrcFm;
    FilterModule mCArmFm;
 
    //----------------------------------  Raw demodulated PD data
    TSeries mAs1QRaw;
    TSeries mAs1IRaw;
    TSeries mAs2QRaw;
    TSeries mAs2IRaw;
    TSeries mAs3QRaw;
    TSeries mAs3IRaw;
    TSeries mAs4QRaw;
    TSeries mAs4IRaw;
    TSeries mPoQRaw;
    TSeries mPoIRaw;
    TSeries mRflQRaw;
    TSeries mRflIRaw;

    //----------------------------------  Dewhitened and phase adjusted
    TSeries mAsQPrep;
    TSeries mAsIPrep;
    TSeries mPoQPrep;
    TSeries mPoIPrep;
    TSeries mRflQPrep;
    TSeries mRflIPrep;

    //----------------------------------  Same as above? from DAQ
    TSeries mAsQData;
    TSeries mAsIData;
    TSeries mPoQData;
    TSeries mPoIData;
    TSeries mRflQData;
    TSeries mRflIData;

    //----------------------------------  Calculated control signals
    TSeries mDArmCalc;
    TSeries mMichCalc;
    TSeries mPrcCalc;
    TSeries mCArmCalc;

    //----------------------------------  Control signals from frame
    TSeries mDArmCtrl;
    TSeries mMichCtrl;
    TSeries mPrcCtrl;
    TSeries mCArmCtrl;

    //----------------------------------  Calculation Status words
public:
    typedef unsigned long stat_type;
    enum StatBits {
        kAdcOvfl=1,
	kLimiter=2
    };

private:
    stat_type mDArmStat;
    stat_type mMichStat;
    stat_type mPrcStat;
    stat_type mCArmStat;    
};

//======================================  Inline functions
inline void 
LscEmul::setDebug(bool x) {
    mDebug  = x;
}

inline bool
LscEmul::LscPd::active(void) const {
    return _I_gain != 0 && _Q_gain != 0;
}


#endif // LSC_EMULATE_HH
