//
//    DMT Data processing base class implementation.
//
//-------------------------------------  Header files.
#include "GridWare.hh"

#ifndef __CINT__
#include <iostream>
#include <fstream>
#include "Interval.hh"
#include "Time.hh"
#include <stdexcept>
#include <string>
#else
#define DACC_ONLDEV "/online/"

#endif // __CINT__

using namespace std;

//======================================  GridWare constructor.
GridWare::GridWare(int argc, const char *argv[]) 
  : mActive(false), mDebug(0), mEOF(false), mAttnMask(0), mMaxFrames(0),
    mAlignMult(0), mAlignOff(0), mStatusFile("monitor.status")
{
  //----------------------------------  Parse the arguments
  for (int i=1 ; i<argc && argv ; i++) {
    string argi = argv[i];
    if (argi == "-inlist") {
      if (++i >= argc) return;
      getDacc().addSingle(argv[i]);
    } else if (argi == "-inlists") {
      if (++i >= argc) return;
      getDacc().addMulti(argv[i]);
    } else if (argi == "-debug") {
      if (++i >= argc) mDebug = 1;
      else             mDebug = strtol(argv[i], 0, 0);
      getDacc().setDebug(mDebug);
    } else if (argi == "-maxframes") {
      mMaxFrames = strtol(argv[++i], 0, 0);
    } else if (argi == "-status") {
      mStatusFile = argv[++i];
    }
  }
  
#ifndef __CINT__
    //----------------------------------  Handle signals
    mTerm.add(SIGTERM);
    mTerm.add(SIGHUP);
    mTerm.add(SIGINT);

    // mAttn.setMode(SigFlag::kBlock);
    mAttn.add(SIGUSR1);
    mAttn.add(SIGIO);

#else
    gROOT->SetInterrupt(kTRUE);
#endif   // !def(__CINT__)

    //----------------------------------  Open a frame stream (FrameCPP)
    // if (getDacc().open()) return;
    mActive = true;
}

//======================================  Test if GridWare command line argument
bool
GridWare::isGridWareArg(const char* argc) const {
    string arg(argc);
    if (arg == "-inlist")    return true;
    else if (arg == "-inlists") return true;
    else if (arg == "-debug")     return true;
    else if (arg == "-maxframes") return true;
    else if (arg == "-status") return true;
    return false;
}

//--------------------------------------  GridWare object destructor.
GridWare::~GridWare() 
{
    //----------------------------------  DMT has terminater
    cout << "Data environment has terminated with Term/Attn/finish/EOF =" 
	 << bool(mTerm) << "/" << bool(mAttn) << "/" << !mActive << "/" 
	 << mEOF << endl;

#ifndef __CINT__
    //----------------------------------  Release signals
    mTerm.zero();
    mAttn.zero();

#endif //  __CINT__

    //----------------------------------  Close the Input file/partition.
    getDacc().close();
}

//--------------------------------------  Main processing loop function.
void
GridWare::MainLoop(void) 
{
  getDacc().open();
  
  //----------------------------------  Loop until terminated.
  bool newFill = true;
  while(!testTerm() && mActive) {
    
    //------------------------------  Check for an attention interrupt.
    if (testAttn()) {
      Attention();
      if (!mActive) continue;
    }
    
    //------------------------------  Get the data identifier
    if (getDacc().synch()) {
      mEOF = true;
      cerr << "Synch failed" << endl;
      break;
    }
    
    //--------------------------  Force the stride alignment
    if (newFill && mAlignMult != 0) {
      Time t  = getDacc().getCurrentTime();
      long ts = t.getS();
      Time Tv = Time(ts - (ts%mAlignMult)) + Interval(mAlignOff);
      if (Tv < t && !Almost(Tv, t)) Tv += Interval(mAlignMult);
      if (!Almost(Tv, t)) {
	if (Debug()) cerr << "Aligning to (" << mAlignMult << ", " 
			  << mAlignOff << ") - skip to " << Tv 
			  << endl;
	if (getDacc().seek(Tv)) continue;
	if (!Almost(getDacc().getCurrentTime(), Tv)) continue;
      }
    }
    /* 
    //--------------------------  Generate a frame name
#ifndef __CINT__
    string FrameID = getDacc().getFrameID();
    mRawID = trig::TrigRaw(getDacc().getFile(), FrameID.c_str(), 
			   getDacc().getOffset(), getDacc().getStride());
    if (Debug() > 2) cout << "Frame ID: " << FrameID << endl;
#endif
    */
    //--------------------------  Read a frame
    int rc;
    try {
      rc = getDacc().fillData(Interval(0.0), newFill);
    } catch(exception& e) {
      cerr << "Exception in fillData: " << e.what() << endl;
      finish();
      break;
    }
    if (rc == 0) {
      try {
	ProcessData();
	Status( getDacc().getCurrentTime() );
	newFill = true;
      } catch(exception& e) {
	cerr << "Exception in ProcessData: " << e.what() << endl;
	finish();
      } catch(...) {
	cerr << "Unidentified exception in ProcessData." << endl;
	finish();
      }
    } else if (rc == -8) {
      newFill = false;
    } else if (rc <= -4) {
      cerr << "MainLoop: Error while reading frame." << endl;
      mEOF = true;
      break;
    } else if (Debug()) {
      cerr << "MainLoop: Frame missing from input stream." << endl;
      newFill = true;
    }
  }

  if (mMaxFrames && mIn.getTotalFrames() >= mMaxFrames) {
    cout << "Maximum frame count (" << mMaxFrames << ") exceeded." 
	 << endl;
    mActive = false;
  }
}

//======================================  Set stride alignment
void
GridWare::setStrideAlignment(unsigned long mult, double off) {
    mAlignMult = mult;
    if (!mAlignMult) {
        mAlignOff = 0.0;
    } else {
        mAlignOff = off;
	double dMult = mAlignMult;
	mAlignOff -= double(long(mAlignOff/dMult)*dMult);
	if (mAlignOff < 0.0) mAlignOff += dMult;
    }
}

void
GridWare::Status( Time t ) {
  unsigned long int time = t.getS();
  ofstream of( mStatusFile.c_str() );
  of << time;
  of.close();
}

//--------------------------------------  Background interrupt handling.
#ifndef __CINT__  

void
GridWare::Attention(void) {
}

bool 
GridWare::testAttn(void) {
    mAttnMask = mAttn.getSigFlags();
    if (mAttnMask) mAttn.clearFlags(mAttnMask);
    return (mAttnMask != 0);
}

bool 
GridWare::testTerm(void) const {
    return mTerm;
}

//--------------------------------------  Root interrupt handling.
#else   
void
GridWare::Attention(void) {
    mActive = false;
}

bool 
GridWare::testAttn(void) {
    gSystem->ProcessEvents();
    mAttnMask = gROOT->IsInterrupted() ? 1 : 0;
    return (mAttnMask != 0);
}

bool 
GridWare::testTerm(void) {
    return false;
}

#endif
