/* -*- mode: c++; c-basic-offset: 4; -*- */
//
//  Class PSD:  Computes power spectrum (with windowing and detrending) 
//  from TSeries data.
//    version:  1.8 (Sept 24 2004)
//    authors:  Kevin C. Schlaufman (kcs149@psu.edu)
//              Patrick J. Sutton (psutton@ligo.caltech.edu)
//
///////////////////////////////////////////////////////////////////////////

#ifndef PSD_HH
#define PSD_HH

#include "Hamming.hh"
class FSpectrum;
class FDFilter;
namespace containers {
    class PSD;
}

/** Power spectrum class.
  * @memo Power spectrum class.
  * @author Kevin C. Schlaufman
  */
class PSD {
public:
    /** Constructs a PSD object with a Hamming window and the desired 
      * number of averages.
      * @memo Constructor with Hamming window.
      * @param win_choice Pointer to a Hamming window.
      * @param intervals Number of averages to be used.
      */
    PSD(const window_api* win_choice, int intervals);

    /** Constructs a PSD object with a square (no) window and the desired 
      * number of averages.
      * @memo Constructor with square (no) window.
      * @param square Dummy pointer.
      * @param intervals Number of averages to be used.
      */
    PSD(const char *square, int intervals);

    /** Constructs a PSD object with the default window type (Hamming) and 
      * the desired number of averages.
      * @memo Constructor with Hamming window.
      * @param intervals Number of averages to be used.
      */
    PSD(int intervals);

    /** Constructs a PSD object with the default window type (Hamming) and 
      * the default number of averages (10).
      * @memo Constructor with Hamming window and 10 averages.
      */
    PSD(void);

    /** Destroy a PSD object.
      * @memo Destructor.
      */
    ~PSD(void);

    /** Generates a power spectral density from the given TSeries pointer.
      * @memo Generates a power spectral density.
      * @param total FSpectrum in which power spectrum is written.
      * @param ts Pointer to TSeries holding data to be Fourier transformed.
      */
    void generate(containers::PSD& total, const TSeries* ts);

    /** Generates a power spectral density from the given TSeries pointer.
      * @memo Generates a power spectral density.
      * @param total FSpectrum in which power spectrum is written.
      * @param ts Pointer to TSeries holding data to be Fourier transformed.
      */
    void generate(FSpectrum& total, const TSeries* ts);

    /** Generates a power spectral density from the given TSeries pointer
      * after filtering it with the specified FDFilter.
      * @memo Generates a power spectral density.
      * @param total FSpectrum in which power spectrum is written.
      * @param ts Pointer to TSeries holding data to be Fourier transformed.
      * @param fd F-domain filter to apply to DFT before calculating the %PSD.
      */
    void generate(FSpectrum& total, const TSeries* ts, FDFilter& fd);

    /**  Set the overlap between averaged segments. The valid range is:
      *  \c 0\ <=\ overlap\ <\ 1 and by default is set to 0;
      *  \brief Set  segment overlap
      */
    void set_overlap(double ovlp);
    
private:
    int n;
    window_api* win;
    double overlap;
};

#endif     //  PSD_HH
