#ifndef HISTOGRAM2_HH
#define HISTOGRAM2_HH

#include "Histogram1.hh"

/** The Histogram2 class holds a 2-dimensional histogram data.
  * @brief Histogram API.
  * @author Masahiro Ito
  * @version 1.0; Last modified October 15, 2001
  */


class Histogram2
{
public:
  /// Bin Type
  enum EBinType{ kUndefinedBin,
		 kFixedBin,
		 kVariableBin };
  
  /// axis type
  enum EAxisSelect{ kXAxis = 0,
		    kYAxis = 1 };

  /**  Histogram axis data type.
    *  @brief Axis type;
    */
  typedef double xbin_t;

  /**  Histogram bin content data type.
    *  @brief Bin content type;
    */
  typedef double histdata_t;

  /**  Histogram statistics data type.
    *  @brief Statistics type;
    */
  typedef double stat_t;

private:
  /// Number of Bins in X-Axis
  int fNbinx;

  /// Number of Bins in Y-Axis
  int fNbiny;

  /// Number of data entries
  int fNEntries;

  /// Sum of weights
  stat_t fTsumw;

  /// Sum of weights^2
  stat_t fTsumw2;

  /// Sum of weight * xdata
  stat_t fTsumwx;

  /// Sum of weight * (xdata^2)
  stat_t fTsumwx2;

  /// Sum of weight * ydata
  stat_t fTsumwy;

  /// Sum of weight * (ydata^2)
  stat_t fTsumwy2;

  /// Sum of weight * xdata * ydata
  stat_t fTsumwxy;

  /// array for bin contents + Over/Underflow
  histdata_t* fArray;

  /// array for bin error^2 + Over/Underflow if fSumw2 is allocated, fBinErrorFlag = true.
  stat_t* fSumw2;

  /// array for lower bin edges + highest bin edge in X-axis
  xbin_t* fXbins;

  /// array for lower bin edges + highest bin edge in Y-axis
  xbin_t* fYbins;

  /// Histogram title
  std::string fTitle;

  /// X-axis label
  std::string fXLabel;

  /// Y-axis label
  std::string fYLabel;

  /// Bin contetnt axis label
  std::string fNLabel;

  /// bin type (0:Undefined 1: Fixed, 2: Variable)
  int fBinType;

  /// flag to check bin-error array is ON/OFF. (true: ON, flase: OFF) 
  bool fBinErrorFlag;

  /// Time stamp
  Time fTime;

  /// Look for the corresponding bin number to the data
  int SearchBin(int min, int max, xbin_t x, int axis) const;

protected:
  /// Free old memory, then allocate new memory
  void Allocate(int nbinx = 0, int nbiny = 0);

public:

  /**  A histogram object is initialized.
    *  @brief Default constructor.
    */
  Histogram2():fArray(0),fSumw2(0),fXbins(0),fYbins(0){ Reset(); }

  /**  The argument histogram is duplicated into the new histogram.
    *  @brief  Copy Constructor.
    *  @param h Histogram to be duplicated.
    */
  Histogram2(const Histogram2& h):
    fArray(0),fSumw2(0),fXbins(0),fYbins(0) { *this = h; }

  /**  A 2-d histogram is created with a specified number of bins, 
    *  lower edge and upper edge.
    *  Bin spacings are fixed.
    *  @brief  Dataless constructor.
    *  @param name Parameter name string
    *  @param nbinx number of bins.
    *  @param xmin lower edge.
    *  @param xmax upper edge.
    *  @param nbiny number of bins.
    *  @param ymin lower edge.
    *  @param ymax upper edge.
    *  @param xlabel label for the X-axis.
    *  @param ylabel label for the Y-axis.
    *  @param nlabel upper for the bin content axis.
    */
  Histogram2(const char* name, int nbinx, xbin_t xmin, xbin_t xmax,
	     int nbiny, xbin_t ymin, xbin_t ymax, const char* xlabel = "", 
	     const char* ylabel = "",const char* nlabel = "");

  /**  A 2-d histogram is created with a specified number of bins 
    *  and bin edges.
    *  Bin spacings are variable.
    *  @brief  Dataless constructor.
    *  @param name Parameter name string
    *  @param nbinx number of bins.
    *  @param xbins lower bin edges
    *  (the last element is the upper edge).
    *  @param nbiny number of bins.
    *  @param ybins lower bin edges
    *  (the last element is the upper edge).
    *  @param xlabel label for the X-axis.
    *  @param ylabel label for the Y-axis.
    *  @param nlabel upper for the bin content axis.
    */
  Histogram2(const char* name, int nbinx, xbin_t* xbins,
	     int nbiny, xbin_t* ybins, const char* xlabel = "", 
	     const char* ylabel = "",const char* nlabel = "");

  /**  Histogram2 destructor.
    * @brief Histogram2 Destructor.
    */
  virtual ~Histogram2();

  /**  Bin contents, number of data entries and time are zeroed.
    *  @brief   Clear bin contents, number of data entries and time.
    */
  virtual void Clear();

  /**  Dump the histogram contents to the specified output stream
    *  @brief Dump the histogram contents
    *  @param out Output stream
    *  @return Reference to theoutput stream.
    */
  virtual std::ostream& Dump(std::ostream& out) const;

  /**  The corresponding bin to the data is incremented by the given weight.
    *  The default value of the weight is 1.
    *  @brief  Increment the corresponding bin to the data by the weight.
    *  @param x Data for X-axis.
    *  @param y Data for Y-axis
    *  @param w Weight.
    */
  virtual void Fill(xbin_t x, xbin_t y, double w = 1);

  /**  Corresponding bin to each input data array element is 
    *  incremented by given weights.
    *  @brief  Increment corresponding bin to each data array element 
    *  by given weights.
    *  @param ntimes Number of entries in the data array.
    *  @param x Data array for X-axis.
    *  @param y Data array for Y-axis.
    */
  virtual void FillN(int ntimes, const xbin_t* x, const xbin_t* y);

  /**  Corresponding bin to each input data array element is 
    *  incremented by given weights.
    *  @brief  Increment corresponding bin to each data array element 
    *  by given weights.
    *  @param ntimes Number of entries in the data array.
    *  @param x Data array for X-axis.
    *  @param y Data array for Y-axis.
    *  @param w Array of weights.
    */
  virtual void FillN(int ntimes, const xbin_t* x, 
		     const xbin_t* y, const double* w);

  /**  All bin contents of the histogram is copied to an array.
    *  (Nbinx+2)*(Nbiny+2) arrays required. 
    *  (Nbinx,Nbiny: Number of bins)
    *  @brief  Copy bin contens to a double array.
    *  @param data Double array into which the bin contents to be copied.
    */
  virtual void GetBinContents(histdata_t* data) const;

  /**  The content of the specified bin is returned.
    *  @brief  Return bin content.
    *  @param xbin Bin number in X-direction.
    *  (0: Underflow, N+1: Overflow where N is Number of bins)
    *  @param ybin Bin number in Y-direction.
    *  (0: Underflow, N+1: Overflow where N is Number of bins)
    *  @return Nth bin content.
    */
  virtual histdata_t GetBinContent(int xbin, int ybin) const;

  /**  Bin edges to the specified axis are copied to an array.
    *  N+1 arrays required.
    *  bin[0] to bin[N-1] hold low-edges.
    *  bin[N] contains the upper edge.
    *  (N: Number of bins)
    *  @brief  Copy bin edges to a double array.
    *  @param bin Double array into which bin edges to be copied.
    *  @param axis Select Axis. (0: X-axis, 1: Y-axis)
    */
  virtual void GetBinLowEdges(xbin_t* bin, int axis) const;

  /**  Bin edge to the specified axis and bin number is returned.
    *  bin[0] to bin[N-1] hold low-edges.
    *  bin[N] contains the upper edge.
    *  (N: Number of bins)
    *  @brief  Copy bin edges to a double array.
    *  @param n Bin number
    *  @param axis Select Axis. (0: X-axis, 1: Y-axis)
    *  @return Nth bin edge.
    */
  virtual xbin_t GetBinLowEdge(int n, int axis) const;

  /**  The center of the n-th bin in the specified axis is returned.
    *  @brief  Return the center of n-th bin content in the specified axis.
    *  @param n Bin number. ( n = 1 to N where N is Number of bins)
    *  @param axis Select Axis. (0: X-axis, 1: Y-axis)
    *  @return n-th bin center.
    */
  virtual xbin_t GetBinCenter(int n, int axis) const;

  /**  All bin errors of the histogram is copied to an array.
    *  (Nbinx+2)*(Nbiny+2) arrays required. 
    *  (Nbinx,Nbiny: Number of bins)
    *  @brief  Copy bin errors to a double array.
    *  @param err Double array into which the bin errors to be copied.
    */
  virtual bool GetBinErrors(stat_t* err) const;

  /**  The error of the specified bin is returned.
    *  @brief  Return bin error.
    *  @param xbin Bin number in X-direction.
    *  (0: Underflow, N+1: Overflow where N is Number of bins)
    *  @param ybin Bin number in Y-direction.
    *  (0: Underflow, N+1: Overflow where N is Number of bins)
    *  @return bin error.
    */
  virtual stat_t GetBinError(int xbin, int ybin) const;

  /**  In the specified axis, bin number corresponding to the argument value is returned.
    *  @brief  Return the bin number in the specified axis.
    *  @param val Data.
    *  @param axis Select Axis. (0: X-axis, 1: Y-axis)
    *  @return bin number.
    */
  virtual int GetBinNumber(xbin_t val, int axis) const;

  /**  The bin spacing for a fixed bin spacing histogram is returned.
    *  @brief   Return bin spacing for fixed bin spacing histogram.
    *  @param  axis Select Axis. (0: X-axis, 1: Y-axis)
    *  @return bin spacing. 0 if bin spacing is variable or undefined.
    */
  virtual xbin_t GetBinSpacing(int axis) const;

  /**  The number of data entries is returned. 
    *  @brief   Return the number of data entries.
    *  @return Number of data entries.
    */
  virtual int GetNEntries(void) const { return fNEntries; }

  /**  The number of bins in the specified axis is returned. 
    *  @brief   Return number of bins.
    *  @return Number of bins.
    */
  virtual int GetNBins(int axis) const;

  /**  The title of the histogram is returned. 
    *  @brief   Return the title of the histogram.
    *  @return Title of the histogram.
    */
  virtual const char* GetTitle(void) const;

  /**  The label of the X-axis is returned. 
    *  @brief   Return the label of the X-axis.
    *  @return X-axis Label.
    */
  virtual const char* GetXLabel(void) const;

  /**  The label of the Y-axis is returned. 
    *  @brief   Return the label of the Y-axis.
    *  @return Y-axis Label.
    */
  virtual const char* GetYLabel(void) const;

  /**  The label of the bin-count axis is returned. 
    *  @brief   Return the label of the bin-count axis.
    *  @return bin-count axis Label.
    */
  virtual const char* GetNLabel(void) const;

  /**  The minimum of bin contents is returned. 
    *  @brief   Return the minimum of bin contents.
    *  @return Minimum of bin contents.
    */
  virtual histdata_t GetMinContent(void) const;

  /**  The maximum of bin contents is returned. 
    *  @brief   Return the maximum of bin contents.
    *  @return Maximum of bin contents.
    */
  virtual histdata_t GetMaxContent(void) const;

  /**  The bin number for the minimum bin content is returned. 
    *  @brief   Return the bin number for the minimum bin content.
    *  @return Bin number for the minimum bin contents.
    */
  virtual histdata_t GetMinContentBin(int& xbin, int& ybin) const;

  /**  The bin number for the maximum bin content is returned. 
    *  @brief   Return the bin number for the maximum bin content.
    *  @return Bin number for the maximum bin contents.
    */
  virtual histdata_t GetMaxContentBin(int& xbin, int& ybin) const;

  /**  The mean value in the specified axis is returned. 
    *  @brief   Return mean value.
    *  @param  axis Select Axis. (0: X-axis, 1: Y-axis)
    *  @return Mean.
    */
  virtual stat_t GetMean(int axis) const;

  /**  The standard deviation value in the specified axis is returned. 
    *  @brief   Return standard deviation.
    *  @param  axis Select Axis. (0: X-axis, 1: Y-axis)
    *  @return standard deviation.
    */
  virtual stat_t GetSdev(int axis) const;

  /**  The statistics of the histogram are copied to the specified array.
    *  <table>
    *  <tr><td>stats[0] </td><td> Sum of weight </td></tr>
    *  <tr><td>stats[1] </td><td> Sum of weight^2 </td></tr>
    *  <tr><td>stats[2] </td><td> Sum of weight * xdata </td></tr>
    *  <tr><td>stats[3] </td><td> Sum of weight * xdata^2 </td></tr>
    *  <tr><td>stats[4] </td><td> Sum of weight * ydata </td></tr>
    *  <tr><td>stats[5] </td><td> Sum of weight * ydata^2 </td></tr>
    *  <tr><td>stats[6] </td><td> Sum of weight * xdata  * ydata </td></tr>
    *  </table>
    *  @brief   Copy statistics of the histogram to an array.
    *  @param  stats Array to store the statistics.
    */
  virtual void GetStats(stat_t *stats) const;

  /**  The statistics of the histogram are replaced by new values.
    *  <table>
    *  <tr><td>stats[0] </td><td> Sum of weight </td></tr>
    *  <tr><td>stats[1] </td><td> Sum of weight^2 </td></tr>
    *  <tr><td>stats[2] </td><td> Sum of weight * xdata </td></tr>
    *  <tr><td>stats[3] </td><td> Sum of weight * xdata^2 </td></tr>
    *  <tr><td>stats[4] </td><td> Sum of weight * ydata </td></tr>
    *  <tr><td>stats[5] </td><td> Sum of weight * ydata^2 </td></tr>
    *  <tr><td>stats[6] </td><td> Sum of weight * xdata  * ydata </td></tr>
    *  </table>
    *  @brief   Copy statistics of the histogram to an array.
    *  @param  stats Array contains new statistics.
    */
  virtual void PutStats(const stat_t *stats);

  /**  A 2-D histogram is projected into a 1-D histogram along the specified axis.
    *  @brief Project a 2-D histogram into a 1-D histogram.
    *  @param title Title of the 1-d histogram.
    *  @param axis Select Axis. (0: X-axis, 1: Y-axis)
    *  @param firstbin First bin to be projected.
    *  @param lastbin Last bin to be projected.
    *  @return Projected 1-D histogram.
    */
  virtual Histogram1* Projection(const char* title, int axis,
				 int firstbin = 0, int lastbin = 99999);

  /**  The type of the histogram bin is returned.
    *  @brief   Return the bin type.
    *  @return Bin type.
    *  <table>
    *  <tr><td>0 </td><td> Undefined </td></tr>
    *  <tr><td>1 </td><td> Fixed Bin </td></tr>
    *  <tr><td>2 </td><td> Variable Bin </td></tr>
    *  </table>
    */
  virtual int GetBinType(void) const { return fBinType; }

  /**  GPS time is returned.
    *  @brief  Return GPS time.
    *  @return GPS time.
    */
  virtual Time GetTime(void) const { return fTime; }

  /**  If error of bins are defined, return true.
    *  @brief  Return true, if bin errors are defined.
    *  @return true if bin errors are defined, otherwise false.
    */
  virtual bool IsErrorFlagON(void) const { return fBinErrorFlag; }

  /**  New title is copied to the title string.
    *  @brief  Set the histogram title.
    *  @param name Pointer to the character array.
    */
  virtual void SetTitle(const char* name) { fTitle = name; }

  /**  New label is copied to the X-axis label string.
    *  @brief  Set the X-axis label.
    *  @param xlabel Pointer to the character array.
    */
  virtual void SetXLabel(const char* xlabel) { fXLabel = xlabel; }

  /**  New label is copied to the Y-axis label string.
    *  @brief  Set the Y-axis label.
    *  @param ylabel Pointer to the character array.
    */
  virtual void SetYLabel(const char* ylabel) { fYLabel = ylabel; }

  /**  New label is copied to the bin-count axis label string.
    *  @brief  Set the bin-count axis label.
    *  @param nlabel Pointer to the character array.
    */
  virtual void SetNLabel(const char* nlabel) { fNLabel = nlabel; }

  /**  The histogram bins are redefined.
    *  @brief  Set the histogram bins.
    *  @param nbinx Number of bins in X-axis.
    *  @param xmin Lower edge in X-axis.
    *  @param xmax Upper edge in X-axis.
    *  @param nbiny Number of bins in Y-axis.
    *  @param ymin Lower edge in Y-axis.
    *  @param ymax Upper edge in Y-axis.
    */
  virtual void SetBinLowEdges(int nbinx, xbin_t xmin, xbin_t xmax,
			      int nbiny, xbin_t ymin, xbin_t ymax);

  /**  The histogram bins are redefined.
    *  @brief  Set the histogram bins.
    *  @param nbinx Number of bins in X-axis.
    *  @param xbins Array of low-edges in X-axis (the last element is the upper edge).
    *  @param nbiny Number of bins in Y-axis.
    *  @param ybins Array of low-edges in Y-axis (the last element is the upper edge).
    */
  virtual void SetBinLowEdges(int nbinx, const xbin_t* xbins, 
			      int nbiny, const xbin_t* ybins);

  /**  The histogram bin contents are replaced by new contents.
    *  Require (Nbinx+2)*(Nbiny+2) data array.
    *  @brief  Replace the contents of bins.
    *  @param data Array of new contents.
    */
  virtual void SetBinContents(const histdata_t* data);

  /**  The specified bin content is replaced by the new content.
    *  @brief  Replace the specified bin content by the new content.
    *  @param xbin Bin number in X-Axis.
    *  @param ybin Bin number in Y-Axis.
    *  @param content New content.
    */
  virtual bool SetBinContent(int xbin, int ybin, histdata_t content);

  /**  The histogram bin errors are replaced by new errors.
    *  Require (Nbinx+2)*(Nbiny+2) data array.
    *  @brief  Replace the error of bins.
    *  @param err Array of new errors.
    */
  virtual void SetBinErrors(const stat_t* err);

  /**  The specified bin error is replaced by the new error.
    *  @brief  Replace the specified bin error by the new error.
    *  @param x Bin number in X-Axis.
    *  @param y Bin number in Y-Axis.
    *  @param err New error.
    */
  virtual bool SetBinError(int x, int y, stat_t err);

  /**  The number of entries to the histogram is replaced by the new number.
    *  @brief  Replace the number of entries with new number.
    *  @param n Number of entries to be replaced.
    */
  virtual void SetNEntries(int n) { fNEntries = n; }

  /**  The number of bins in the specified axis is changed to new number.
    *  @brief  Set number of bins.
    *  @param nbin Number of bins.
    *  @param axis Select Axis. (0: X-axis, 1: Y-axis)
    */
  virtual void SetNBins(int nbin, int axis);

  /**  The bin type of the histogram is replaced to the specifed type.
    *  @brief  Set the bin type.
    *  @param type Bin type. (0: Undefined, 1: Fixed, 2: Variable)
    */
  virtual void SetBinType(int type);

  /**  A time stamp in GPS seconds is set.
    *  @brief  Set a time stamp (GPS).
    *  @param t GPS time.
    */
  virtual void SetTime(const Time& t) { fTime = t; }

  /**  The errors of bins are cleared or 
    *  updated based on the current bin content.
    *  @brief  Clear or Update errors of bins based on the current bin content.
    *  @param reset (true: clear bin error with zero, 
    *  false: update bin error)
    */
  virtual void Sumw2(bool reset = true);

  /**  All the histogram data and statistics are cleared.
    *  @brief  Clear all the histogram data.
    */
  virtual void Reset(void);

  /**  A constant reference to the Nth bin of (Nbinx+2)*(Nbiny+2) 
    *  data array is returned.
    *  (Nbinx = # of bin in X-axis, Nbiny = # of bin in Y-axis)
    *  @brief   Get the reference to the Nth bin.
    *  @param  n Bin number.
    *  @return Reference to the Nth bin.
    */
  virtual const histdata_t& operator [] (int n) const { return fArray[n]; }

  /**  Reference to the Nth bin of (Nbinx+2)*(Nbiny+2) 
    *  data array is returned.
    *  (Nbinx = # of bin in X-axis, Nbiny = # of bin in Y-axis)
    *  @brief   Get the reference to the Nth bin.
    *  @param  n Bin number.
    *  @return Reference to the Nth bin.
    */
  virtual       histdata_t& operator [] (int n)       { return fArray[n]; }

  /**  The Histogram2 data are copied to the current Histogram2.
    *  @brief  Copy histogram.
    *  @param h Constant Histogram2 to be copied.
    *  @return Reference to the updated Histogram2.
    */
  Histogram2& operator = (const Histogram2& h);

  /**  The Histogram2 data are added to the current Histogram2.
    *  @brief  Add a histogram to the current histogram.
    *  @param h Constant Histogram2 to be added.
    *  @return Reference to the updated Histogram1.
    */
  Histogram2& operator += (const Histogram2& h);

  /**  A bias is added to each bin of the histogram.
    *  @brief  Add a bias to each bin.
    *  @param bias Number to be added.
    *  @return Reference to the updated Histogram2.
    */
  Histogram2& operator += (histdata_t bias);

  /**  The current Histogram2 is subtracted by another Histogram2 data.
    *  @brief  Subtract the histogram from the current histogram.
    *  @param h Constant Histogram2 to subtract.
    *  @return Reference to the updated Histogram2.
    */
  Histogram2& operator -= (const Histogram2& h);

  /**  Bin contents are multiplied by a scalar.
    *  @brief  Multiply bin contents by a scalar.
    *  @param scale Number to be multiplied.
    *  @return Reference to the updated Histogram2.
    */
  Histogram2& operator *= (double scale);

  /**  Current Histogram2 is multiplied by the argument histogram.
    *  @brief  Multiply current Histogram2 by the argument histogram.
    *  @param h Constant Histogram2 to multiply.
    *  @return Reference to the updated Histogram2.
    */
  Histogram2& operator *= (const Histogram2& h);

  /**  Current Histogram2 is divied by the argument histogram bin by bin.
    *  @brief  Divide bin contents by bin contents of the argument histogram.
    *  @param h Constant Histogram2 to divide.
    *  @return Reference to the updated Histogram2.
    */
  Histogram2& operator /= (const Histogram2& h);
};

/** @name Histogram2 functions
  * This section contains functions of Histogram2 that are not members of the 
  * Histogram2 class. 
  * @brief Non-member functions of Histogram2.
  * @author Masahiro Ito
  * @version 1.0; Last modified October 15, 2001
  */
//@{

/**  A bias is added to bins of the histogram.
  *  @brief  Add a bias to each bin.
  *  @param bias Number to be added.
  *  @return Biased Histogram2.
  */
Histogram2 operator + (Histogram2::histdata_t bias, const Histogram2& h);
Histogram2 operator + (const Histogram2& h, Histogram2::histdata_t bias);

/**  Two Histogram2 data are added together.
  *  @brief  Add two Histogram2 data together.
  *  @param h Constant Histogram2 to be added.
  *  @return Summed Histogram2.
  */
Histogram2 operator + (const Histogram2& h1, const Histogram2& h2);

/**  A histogram is subtracted by the other.
  *  @brief  Subtract a Histogram2 from the other.
  *  @param h Constant Histogram2 to subtract.
  *  @return Subtracted Histogram2.
  */
Histogram2 operator - (const Histogram2& h1, const Histogram2& h2);

/**  The histogram is multiplied by a scalar.
  *  @brief  Multiply the histogram by a scalar.
  *  @param scale Number to be multiplied.
  *  @return Scaled Histogram2.
  */
Histogram2 operator * (double scale, const Histogram2& h);
Histogram2 operator * (const Histogram2& h, double scale);

/**  Two histograms are multiplied together.
  *  @brief  Multiply two histograms together.
  *  @param h1 Histogram1 to be multiplied.
  *  @param h2 Histogram1 to be multiplied.
  *  @return Multiplied Histogram1.
  */
Histogram2 operator * (const Histogram2& h1, const Histogram2& h2);

/**  The first histogram is devided by the second.
  *  @brief  Divide the first histogram by the second.
  *  @param h1 Histogram1 to be divided.
  *  @param h2 Histogram1 to divide.
  *  @return Divided Histogram1.
  */
Histogram2 operator / (const Histogram2& h1, const Histogram2& h2);

//@}

#endif // HISTOGRAM2_HH
