(* Copyright (c) INRIA and Microsoft Corporation. All rights reserved. *)
(* Licensed under the Apache 2.0 License. *)

(** Decorate each file with a little bit of boilerplate, then print it *)

open Utils
open PPrint

let mk_includes =
  separate_map hardline (fun x -> string "#include " ^^ string x)

let filter_includes file includes =
  KList.filter_some (List.rev_map (function
    | Some file', h when file = file' -> Some h
    | None, h -> Some h
    | _ -> None
  ) includes)

let kremlib_include () =
  if !Options.minimal then
    empty
  else
    mk_includes [ "\"kremlib.h\"" ]

(* A Pprint document with:
 * - #include X for X in the dependencies of the file, followed by
 * - #include Y for each -add-include Y passed on the command-line
 *)
let includes_for file files =
  let extra_includes = mk_includes (filter_includes file !Options.add_include) in
  let includes = mk_includes (List.rev_map (Printf.sprintf "\"%s.h\"") files) in
  separate hardline [ includes; extra_includes ]

let invocation (): string =
  KPrint.bsprintf
{|
  This file was generated by KreMLin <https://github.com/FStarLang/kremlin>
  KreMLin invocation: %s
  F* version: %s
  KreMLin version: %s
|}
    (String.concat " " (Array.to_list Sys.argv))
    !Driver.fstar_rev
    !Driver.krml_rev

let header (): string =
  let default = invocation () in
  if !Options.header = "" then
    "/* " ^ default ^ " */"
  else
    let fmt = Utils.file_get_contents !Options.header in
    try
      let fmt = Scanf.format_from_string fmt "%s" in
      KPrint.bsprintf fmt default
    with Scanf.Scan_failure _ ->
      fmt

(* A pair of a header, containing:
 * - the boilerplate specified on the command-line by -header
 * - #include Y for each -add-early-include Y passed on the command-line
 * - #include "kremlib.h"
 * - the #ifndef #define guard,
 * and a footer, containing:
 * - the #endif
 *)
let prefix_suffix name =
  Driver.detect_fstar_if ();
  Driver.detect_kremlin_if ();
  let prefix =
    string (header ()) ^^ hardline ^^
    mk_includes (filter_includes name !Options.add_early_include) ^^ hardline ^^
    kremlib_include () ^^ hardline ^^ hardline ^^
    string (Printf.sprintf "#ifndef __%s_H" name) ^^ hardline ^^
    string (Printf.sprintf "#define __%s_H" name) ^^ hardline
  in
  let suffix =
    hardline ^^
    string (Printf.sprintf "#define __%s_H_DEFINED" name) ^^ hardline ^^
    string "#endif"
  in
  prefix, suffix

let in_tmp_dir name =
  Driver.mk_tmpdir_if ();
  let open Driver in
  if !Options.tmpdir <> "" then
    !Options.tmpdir ^^ name
  else
    name

let write_one name extern_c prefix program suffix =
  let if_cpp doc =
    string "#if defined(__cplusplus)" ^^ hardline ^^
    doc ^^ hardline ^^
    string "#endif"
  in
  with_open_out_bin (in_tmp_dir name) (fun oc ->
    let doc =
      prefix ^^
      (if extern_c then if_cpp (string "extern \"C\" {") ^^ hardline else empty) ^^
      hardline ^^
      separate_map (hardline ^^ hardline) PrintC.p_decl_or_function program ^^
      hardline ^^
      (if extern_c then hardline ^^ if_cpp (string "}") ^^ hardline else empty) ^^
      suffix ^^ hardline
    in
    PPrint.ToChannel.pretty 0.95 100 oc doc
  )

let write_c files =
  Driver.detect_fstar_if ();
  Driver.detect_kremlin_if ();
  List.map (fun file ->
    let name, _, program = file in
    let header = header () in
    let prefix = string (Printf.sprintf "%s\n\n#include \"%s.h\"" header name) ^^ hardline in
    let prefix =
      if !Options.add_include_tmh then
        string "#ifdef WPP_CONTROL_GUIDS" ^^ hardline ^^
        string (Printf.sprintf "#include <%s.tmh>" name) ^^ hardline ^^
        string "#endif" ^^ hardline ^^ prefix
      else
        prefix
    in
    write_one (name ^ ".c") false prefix program empty;
    name
  ) files

let write_h files =
  List.map (fun file ->
    let name, deps, program = file in
    let prefix, suffix = prefix_suffix name in
    let prefix = prefix ^^ hardline ^^ includes_for name deps in
    write_one (name ^ ".h") !Options.extern_c prefix program suffix;
    name
  ) files

let write_makefile user_ccopts custom_c_files c_files h_files =
  let concat_map ext files =
    String.concat " " (List.map (fun f -> f ^ ext) files)
  in
  Utils.with_open_out_bin (in_tmp_dir "Makefile.include") (fun oc ->
    KPrint.bfprintf oc "USER_TARGET=%s\n" !Options.exe_name;
    KPrint.bfprintf oc "USER_CFLAGS=%s\n" (concat_map "" (List.rev user_ccopts));
    KPrint.bfprintf oc "USER_C_FILES=%s\n" (concat_map "" custom_c_files);
    KPrint.bfprintf oc "ALL_C_FILES=%s\n" (concat_map ".c" c_files);
    KPrint.bfprintf oc "ALL_H_FILES=%s\n" (concat_map ".h" h_files)
  );
  Utils.cp Driver.(!Options.tmpdir ^^ "Makefile.basic") Driver.(!misc_dir ^^ "Makefile.basic")

let write_def m c_files =
  let dst = in_tmp_dir (Filename.chop_extension !Options.exe_name ^ ".def") in
  with_open_out_bin dst (fun oc ->
    KPrint.bfprintf oc "LIBRARY %s\n\nEXPORTS\n"
      (Filename.basename (Filename.chop_extension !Options.exe_name));
    List.iter (fun (_, decls) ->
      List.iter (function
        | Ast.DFunction (_, flags, _, _, name, _, _)
          when not (List.mem Common.Private flags) ->
            let name = GlobalNames.to_c_name m name in
            KPrint.bfprintf oc "  %s\n" name
        | _ -> ()
      ) decls
    ) c_files
  )
