#ifndef _FPIL_INC
#define _FPIL_INC
#ifdef __cplusplus
extern "C" {
#endif

/*
 *+			F P I L

 * Include File name:
      fpil.h

 * Function:
      Header file for the Fibre Positioner Instrument Library.

 * Description:
      The Fibre Positioner Instrument Library provides interfaces which allow
      code to be written to run with any fibre positioner instrument, in 
      particular, things such as collision detection and fibre button 
      position validation, which are common to all instruments.

 * Language:
      C


 * Support: Tony Farrell, AAO

 *-


 * Sccs Id:     fpil.h, Release 1.16, 09/30/02

 * History:
     03-Dec-1999 - TJF - Original version
     28-Jan-1999 - TJF - Pass both unsquared and squared tolerance values
                         to each called routine.
     10-Mar-2000 - TJF - Add extra arguments to telescope model and
                         constant details access routines to enable
                         use of search paths to find files.
     13-Mar-2000 - TJF - Add FpilGetFieldRadius and FpilModelQuickAppToObs
     15-Mar-2000 - TJF - Add FpilGetFieldRadiusIncPark function and
                         FieldRadIncPark item to structure.
     16-Mar-2000 - TJF - Add FpilDrawButFibRoutineType and FpilCvtRoutineType
	                 and associated bits.
     17-Mar-2000 - TJF - Add NumFids to structure and routine
                         FpilGetNumFiducials.
     19-Apr-2000 - TJF - Support checking for the possiblity of park
                         collisions, and the specification of a plate
                         argument to FpilColInvPos().
     20-Apr-2000 - TJF - Remove dummy error code definitions, we now
                         have proper error codes.
     19-Jul-2000 -  KS - Add prototypes for new Fpil routines connected with
                         target and pivot classification.
     02-Feb-2001 -  KS - Add FpilFibreReallyExists().
     05-Feb-2001 -  KS - Add FpilFibreCombos() and FpilComboFibreCodes().
     08-Apr-2001 -  KS - Added modeNames argument to FpilFibreCombos().
     06-Jun-2001 -  KS - Added fibreCombo argument to FpilFibreColour().
     12-Jun-2001 -  KS - Added useGrey arguments.
     20-Jun-2001 -  KS - Added tscale argument to FpilDrawTarget().
     13-Aug-2001 -  KS - Added FpilSetPositionAngle().
     16-Nov-2001 -  KS - Added FpilGetPivotText(), FpilPivotTypeMatch() and
                         FpilPivotSharesSky().
     21-Mar-2002 -  KS - Added FpilWavelengthCombos(), FpilSetWavelengthCombo(),
                         FpilSetFibreCombo(), FpilSetObsWavelength() and
                         FpilGetFibreWavelength().
     03-Sep-2002 -  KS - Added FpilCompatiblePivotTypes().
     30-Sep-2002 - TJF - CvtInit needs an argument argument.
     17-Oct-2002 -  KS - Added FpilGetPointingWavelength().
     05-Nov-2002 -  KS - Added FpilGetFibreLength().
     21-Dec-2002 - RSC - fibreType arg. added to FpilColInvPos().
 */

/*
 *  Include files - note, don't use DRAMA include files - this module
 *  must be independent of DRAMA
 */

/* 
 * Constant definitions
 */
#define FPIL_VERSION 1
#define FPIL_COL_BUTBUT 2
#define FPIL_COL_BUTFIB 2
#define FPIL_COL_FIBBUT 3
#define FPIL_NOCOL 0
#define FPIL_NAMELEN 20
#define FPIL_MAXPIVOTS 1000
#define FPIL_MAXFIELDS 4
#define FPIL_MAXFIDS 40
/*
 * Macro function definitions
 */

/*
 * type definitions
 */


/*
 * I don't really want to include status.h and sds.h, and as I only need
 * pointers to StatusType and SdsType, check if I have them, and if not,
 * emulate what they do.  I do the checks by looking for macros which
 * are defined by the include files and are part of the definition of
 * the module.
 */
#ifndef STATUS__OK
typedef int FpilStatusType;
#else
typedef StatusType FpilStatusType;
#endif

#ifdef SDS_STRUCT
typedef SdsIdType FpilSdsIdType;
#else
typedef long FpilSdsIdType;
#endif


/*
 * The types used internally to store details and passed to each function.  
 * This is a hidden type.
 */
typedef struct { int dummy; } Fpil___DummyType;
typedef Fpil___DummyType  * FpilType;

/*
 *+			F p i l R p t R o u t i n e T y p e

 * Function name:
      FpilRptRoutineType

 * Function:
      A type for reporting callback routines.

 * Description:
      This is a description of a routine type, used when calling
      FpilConsAccess(3) and FpilModelAccess(3).   It is used for a
      routine invoked as a callback routine.  Also
      see the descriptions of the types FpilConsAccessRoutineType(3)
      and FpilModelAccessRoutineType(3).

 * Language:
      C

 * Call:
      (void) = FpilRtpRoutineType(clientData, string)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) clientData      (void *)    Was passed to FpilConsAccess or
                                      FpilModelAccess.
      (>) string          (const char *) A string to be reported.
      (!) status    (StatusType *) modified status.


 * Include files: fpi.h

 *-

 * History:
     10-Mar-2000 - TJF - Original version
 */
typedef void (*FpilRptRoutineType) (
        void * clientData,
        const char *string,
        FpilStatusType *status);
        

/*
 *+			F p i l C o l B u t F i b R o u t i n e T y p e

 * Function name:
      FpilColButFibRoutineType

 * Function:
      User callback routine for determing button to fibre collisions.

 * Description:
      This is a description of a routine type, used when calling
      FpilInit(3).  This routine type is used for the routine
      which checks for collisions between buttons and fibres in the user fibre
      positioner instrument.

 * Language:
      C

 * Call:
      (int) = FpilColBufFibRoutineType(clientData, butX, butY, theta, 
                                       fvpX, fvpY, pivX, pivY, fibreClear,
                                       fibreClearSqrd)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) clientData      (void *)    Was passed to FpilInit.
      (>) butX            (double)    Button X ordinate.
      (>) butY            (double)    Button Y ordinate.
      (>) theta           (double)    Button theta.
      (>) fvpX            (double)    Fibre virtual pivot X ordinate.
      (>) fvpY            (double)    Fibre virtual pivot Y ordinate.
      (>) pivX            (double)    Fibre pivot X ordinate.
      (>) pivY            (double)    Fibre pivot Y ordinate.
      (>) fibreClear      (long int)  Clearance to allow (microns)
      (>) fibreClearSqrd  (double)    The square of fibreClear

 * Returned value:
      true if a collision is detected.

 * Include files: fpi.h

 *-

 * History:
     09-Dec-1999 - TJF - Original version
 */
typedef int  (*FpilColButFibRoutineType) (
        void * clientData,
        double    butX, double    butY, double    theta,
        double    fvpX, double    fvpY, double    pivX, double    pivY,
        long int  fibreClear, double fibreClearSqrd);


/*
 *+			F p i l C o l B u t B u t R o u t i n e T y p e

 * Function name:
      FpilColButButRoutineType

 * Function:
      User callback routine for determing button collisions.

 * Description:
      This is a description of a routine type, used when calling
      FpilInit(3).  This routine type is used for the routine
      which checks for collisions between button in the user fibre
      positioner instrument.

 * Language:
      C

 * Call:
      (int) = FpilColButButRoutineType(clientData, ,butXa,butYa,thetaa,
                                       butXb,butYb,thetab, butClear,
                                       butClearSqrd)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) clientData      (void *)    Was passed to FpilInit.
      (>) butXa        (double)    Button A  X ordinate.
      (>) butYa        (double)    Button A  Y ordinate.
      (>) thetaa       (double)    Button A  theta.
      (>) butXb        (double)    Button B  X ordinate.
      (>) butYb        (double)    Button B  Y ordinate.
      (>) thetab       (double)    Button B  theta.
      (>) butClear     (long int)  Clearance to allow (microns)
      (>) butClearSqrd (double)    The square of butClear
      

 * Returned value:
      true if a collision is detected.


 * Include files: fpi.h

 *-

 * History:
     09-Dec-1999 - TJF - Original version
 */
typedef int  (*FpilColButButRoutineType) (
        void * clientData,
        double    butXa, double    butYa, double    thetaa,
        double    butXb, double    butYb, double    thetab,
        long int  butClear, double butClearSqrd);

/*
 *+			F p i l C o l F i d B u t R o u t i n e T y p e

 * Function name:
      FpilColFidButRoutineType

 * Function:
      User callback routine for determing button  -fiducial collisions.

 * Description:
      This is a description of a routine type, used when calling
      FpilInit(3).  This routine type is used for the routine
      which checks for collisions between buttons and fiducials in the 
      user fibre positioner instrument.

 * Language:
      C

 * Call:
      (int) = FpilColFidButRoutineType(clientData, 
                               butx, buty, theta
                               markx, marky,
                               fidTol, fidTolSqrd)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) clientData      (void *)    Was passed to FpilInit.
      (>) butx            (long int)   Fibre Button X ordinate.
      (>) buty            (long int)   Fibre Button Y ordinate.
      (>) theta           (double)     Fubre button theta orientation.
      (>) markx           (long int)   Fiducial X ordinate
      (>) marky           (long int)   Fiducial Y ordinate
      (>) fidTol          (long int)   Tolerance to allow
      (>) fidTolSqrd      (double)     Square of fidTol

 * Returned value:
      true if a collision is detected.

 * Include files: fpi.h

 *-

 * History:
     09-Dec-1999 - TJF - Original version
 */
typedef int  (*FpilColFidButRoutineType) (
        void * clientData,
        long int  butx, long int  buty, double    theta, long int  markx,
        long int  marky,
        long int  fidTol, double    fidTolSqrd);

/*
 *+			F p i l C o l I n v P o s R o u t i n e T y p e

 * Function name:
      FpilColInvPosRoutineType

 * Function:
      User callback routine for determing invalid position collisions.

 * Description:
      This is a description of a routine type, used when calling
      FpilInit(3).  This routine type is used for the routine
      which checks for collisions with an invalid position on the
      field plate (such as mounting screws etc.)

 * Language:
      C

 * Call:
      (int) = FpilColInvPosRoutineType(clientData, plate, fibreType,
      				       butx,buty,theta,butClear,butClearSqrd)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) clientData      (void *)    Was passed to FpilInit.
      (>) plate           (unsigned int) Plate to check for.
      (>) fibreType       (int)        Fibre type.
      (>) butx            (long int)   Button X ordinate.
      (>) buty            (long int)   Button Y ordinate.
      (>) theta           (double)     Button orientation.
      (>) butClear        (long int)  Clearance to allow (microns)
      (>) butClearSqrd    (double)    The square of butClear

 * Returned value:
      true if a collision is detected.

 * Include files: fpi.h

 *-

 * History:
     09-Dec-1999 - TJF - Original version
     19-Apr-2000 - TJF - Add plate argument.
     21-Dec-2002 - RSC - Add fibreType argument.
 */
typedef int  (*FpilColInvPosRoutineType) (
    void * clientData,
    unsigned int plate,
    int       fibreType,
    long int  butx,
    long int  buty,
    double    theta,
    long int  butClear, 
    double butClearSqrd);

/*
 *+		F p i l  G et V i r P i v R o u t i n e T y p e

 * Function name:
      FpilGetVirPivRoutineType

 * Function:
      User callback routine for determing a virtual pivot position.

 * Description:
      This is a description of a routine type, used when calling
      FpilInit(3).  This routine type is used for the routine
      which determines a virtual pivot position for a given button in the
      user fibre positioner instrument.

      Given the position of a button - ie the x,y position of its origin, 
      which is the position of the fibre end, and its theta value (the angle 
      measured anticlockwise from the vertical through that origin to the 
      fibre) the virtual pivot point  is the point 
      at where the fibre is assumed to emerge.


 * Language:
      C

 * Call:
      (void) = FpilRoutineType(clientData, originX, originY, theta, 
                              virtPivX, virtPivY)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) clientData      (void *)    Was passed to FpilInit.
      (>) originX         (double)    Position of fibre end - X ordinate, 
                                      microns
      (>) originY         (double)    Position of fibre end - Y ordinate,
                                      microns
      (>) theta           (double)    Button theta - radians
      (<) virtPivX        (double *)  The virtual fibre end - X ordinate.
      (<) virtPivy        (double *)  The virtual fibre end - Y ordinate.

 * Include files: fpi.h

 *-

 * History:
     09-Dec-1999 - TJF - Original version
 */
typedef void (*FpilGetVirPivRoutineType) (
        void * clientData,
        double OriginX,
        double OriginY,
        double Theta,
        double *VirtPivX,
        double *VirtPivY);



/*
 *+			F p i l C o n s A c c e s s R o u t i n e T y p e

 * Function name:
      FpilConsAccessRoutineType

 * Function:
      User callback routine accessing instrument constants details.

 * Description:
      This is a description of a routine type, used when calling
      FpilInit(3).  This routine type is used for the routine
      which accesses the "constants" details for the 
      user fibre positioner instrument.

      Routines which implement this function should first search for
      the specifie file in the directory specified by the
      dirOverride argument.  They can then (optionally) search
      in an instrument specific default directory.  Finally, they
      can search the the directory specified by the dirFallback argument.

      When they open the file, they should invoke the RptRoutine routine
      if it is non-null with RptArg as its first argument and the name 
      of the file which was actually opened as the second argument.

 * Language:
      C

 * Call:
      (void) = FpilConsAccessRoutineType(clientData, dirOverride, 
               dirFallback, RptRoutine, RptArg, id, status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) clientData      (void *)    Was passed to FpilInit.
      (>) dirOverride     (const char *)  If non-null, a directory to
                                   search for the constant details file.
      (>) dirFallback     (const char *)  If non-null, a fall back
                                   directory to search for the constants
                                   details file.
      (>) RptRoutine      (FpilRptRoutineType) A routine to be invoked
                                   to report which file has been opened.
      (>) RptArg          (void *) Argument passed to RptRoutine.
      (<) id        (SdsIdType *) The SDS id of the constant details should
                    be returned here.
      (!) status    (StatusType *) modified status.
      

 * Include files: fpi.h

 *-

 * History:
     09-Dec-1999 - TJF - Original version
 */
typedef void (*FpilConsAccessRoutineType) (
        void * clientData,
        const char *dirOverride,
        const char *dirFallback,
        FpilRptRoutineType RptRoutine,
        void * RptArg,
        FpilSdsIdType *id,
        FpilStatusType *status);



/*
 *+			F p i l W h i c h S p e c R o u t i n e T y p e

 * Function name:
      FpilWhichSpecRoutineType

 * Function:
      User callback routine for determining the spectrograph.

 * Description:
      This is a description of a routine type, used when calling
      FpilInit(3).  This routine type is used for the routine
      which determines the spectrograph assocaited with a pivot.


 * Language:
      C

 * Call:
      (void) = FpilWhichSpecRoutineType(clientData, field, pivot
               Constants, spec, SpecName, status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) clientData      (void *)    Was passed to FpilInit.
      (>) field           (unsigned) The field plate number.
      (>) pivot           (unsigned) The pivot number, starting from 1.
      (>) Constants  (SdsIdType) The SDS id of the constant details that
                    was returned by the instrument specific contants access  
                    routine (AccessCons) when called by FpilConsAccess().
      (<) spec      (unsigned *) The spectrograph, where 0 indicates a
                    guide fibre, 1 unwards indicates a specrograph.
      (<) SpecName (const char **) A descriptive name string
                    for the spectorgraph.  "Guide" should be used
	           for guide fibres.
      (!) status    (StatusType *) modified status.
      

 * Include files: fpi.h

 *-

 * History:
     09-Dec-1999 - TJF - Original version
 */
typedef void (*FpilWhichSpecRoutineType) (
        void * clientData,
        unsigned field,
        unsigned pivot,
        FpilSdsIdType Constants,
        unsigned *spec,
        const char ** SpecName,
        FpilStatusType *status);


/*
 *+			F p i l C v t R o u t i n e T y p e

 * Function name:
      FpilCvtRoutineType

 * Function:
      User callback routine to convert coorindates.

 * Description:
      This is a description of a routine type, used when calling
      FpilDrawButFib(3).  It is used to convert positions in microns
      on the field plate to pixels on a GUI.

 * Language:
      C

 * Call:
      (int) = FpilCvtRoutineType(clientData, Microns,Zoom)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) clientData      (void *)  Was passed to FpilDrawButFib().
      (>) Microns         (double)  The value to convert.
      (>) Zoom            (double)  The current zoom value, was passed
                                    to FpilDrawButFib().
      (!) status    (StatusType *) modified status.
      

 * Returned Value: The position in pixels.

 * Include files: fpi.h

 *-

 * History:
     16-Mar-2000 - TJF - Original version
 */
typedef int (*FpilCvtRoutineType)(
    void *clientData,
    double Microns, 
    double Zoom);
/*
 *+			F p i l D r a w R o u t i n e T y p e

 * Function name:
      FpilDrawRoutineType

 * Function:
      User callback routine to provide information about drawing a button and fibre.

 * Description:
      This is a description of a routine type, used when calling
      FpilInit(3).  This routine type is used for the routine
      which returns information about how to draw a button and fibre.

      Note that the fibre is drawn from the pivot position to the
      position returned by this routine.  

 * Language:
      C

 * Call:
      (void) = FpilDrawRoutineType(clientData,scaling, zoom, 
                            CvtX, CvtY, CvtData,
                            X, Y, Theta, Type, ButGrapLen, FibX, FibY,
                            ButGraphic1, ButGraphic2, status)


 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) clientData      (void *)    Was passed to FpilInit.
      (>) scaling         (double) The scaling from microns to pixels, 
                          excluding zoom factor. Note
                          that you should use the conversion routines CvtX
                          and CvtY to convert absolute positions on the
                          field plate to absolute positions on the GUI as
                          they take account of the field center etc.
      (>) zoom            (double)  Current zoom value.  
      (>) CvtX            (FpilCvtRoutineType) A routine for converting 
                          positions on the field plate in microns in X to
                          pixels on the GUI.
      (>) CvtY            (FpilCvtRoutineType) A routine for converting 
                          positions on the field plate in microns in X to
                          pixels on the GUI.
      (>) CvtData         (void *) Should be passed as the first argument
	                 to the conversion routines.
      (>) X               (double) Fibre position in X.
      (>) Y               (double) Fibre position in Y.
      (>) Theta           (double) Fibre theta.
      (>) Type            (int)  The button type string.  0 = Guide bundle
                          1 = program object.
      (>) ButGrapLen1     (unsigned) Length of the ButGraphic1 and ButGraphic2
                          string2.
      (<) FibX            (double *) The fibre end position in X.
      (<) FibY            (double *) The fibre end position in X.
      (<) ButGraphic1     (char *) The routine should place here
                          the arguments to a Tk canvas widget create command 
                          required to draw a graphic of the button.  Only 
                          graphics which accept a "-fill" option can be used 
                          since the Tcl code sets colours appropiately 
                          depending on the type of object and if it has been 
                          selected.  For example, this may contain
                          something like "polygon x1 y1 x2 y2 x2 y3..."
                          or             "oval 10 10 80 80"
      (<) ButGraphic2     (char *) As per ButGraphic1, to be used if
                          a second graphic item is required to represent
                          the one pivot.  If not required, this can be
                          ignored.
      
      (!) status    (StatusType *) modified status.
      

 * Include files: fpi.h

 *-

 * History:
     16-Mar-2000 - TJF - Original version
 */

typedef void (*FpilDrawRoutineType) (
        void * clientData,
        double scaling,
        double zoom,
        FpilCvtRoutineType CvtX,
        FpilCvtRoutineType CvtY,
	void * CvtData,
        double X,
        double Y,
        double Theta,
        int Type,
        unsigned ButGrapLen,
        double *FibX,
        double *FibY,
        char *ButGraphic1,
        char *ButGraphic2,
        FpilStatusType *status);


typedef FpilSdsIdType FpilConstantsType;

typedef struct FpilModelStruct FpilModelType;


/*
 *+	      	F p i l M o d e l A c c e s s R o u t i n e T y p e

 * Function name:
      FpilModelAccessRoutineType

 * Function:
      User callback routine for accessing telescope model details.

 * Description:
      This is a description of a routine type, used when calling
      FpilInit(3).  This routine type is used for the routine
      which accesses the telescope details for the
      user fibre positioner instrument.

      This routine should access the model and 
      malloc and return the address of a structure of type
      FpilModelType.  By actually mallocing a different structure
      which contains as its first item a structure of type
      FpilModelType, the telescope model implementation routines
      can keep there own information in this structure.  It just
      casts to and from FpilModelType as required (this can be considered
      deriving from FpilModelType, in the O-O sense).
      
      Routines which implement this function should first search for
      the specific files in the directory specified by the
      dirOverride argument.  They can then (optionally) search
      in an instrument specific default directory.  Finally, they
      can search the the directory specified by the dirFallback argument.

      When they open the file, they should invoke the RptRoutine routine
      if it is non-null with RptArg as its first argument and the name of 
      the file which was actually opened as the second argument.


 * Language:
      C

 * Call:
      (void) = FpilModelAccessRoutineType(clientData,  dirOverride, 
               dirFallback, RptRoutine, RptArg, field,  model, status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) clientData      (void *)    Was passed to FpilInit.
      (>) dirOverride     (const char *)  If non-null, a directory to
                                   search for the constant details file.
      (>) dirFallback     (const char *)  If non-null, a fall back
                                   directory to search for the constants
                                   details file.
      (>) RptRoutine      (FpilRptRoutineType) A routine to be invoked
                                   to report which file has been opened.
      (>) RptArg          (void *) Argument passed to RptRoutine.
      (>) field     (unsigned int) The field plate number.
      (<) model     (FpilModelType **) The address of the model is returned
                                   here.
      (!) status    (StatusType *) modified status.

 * Include files: fpi.h

 *-

 * History:
     09-Dec-1999 - TJF - Original version
 */
typedef void (*FpilModelAccessRoutineType) (
        void * clientData,
        const char *dirOverride,
        const char *dirFallback,
        FpilRptRoutineType RptRoutine,
        void * RptArg,
        unsigned int field,
        FpilModelType **model,
        FpilStatusType *status);



/*
 *+			F p i l M o d e l C v t I n i t R o u t i n e T y p e

 * Function name:
      FpilModelCvtInitRoutineType

 * Function:
      User callback routine initializing the model conversion for a 
      particular time etc.


 * Description:
      This is a description of a routine type, used when calling
      FpilInit(3).  This routine type is used for the routine
      which  initialises telescope model conversion for a particular time etc.
      for the user fibre positioner instrument.

 * Language:
      C

 * Call:
      (void) = FpilModelCvtInitRoutineType(clientData,  mjd, dut, temp, pres, 
                                   humid, cenWave, obsWave, tel, status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) model   (FpilModelType *model) Contains the telescope model details
                           obtained by FpilModelAccess(3).  This routine
                           may or may not have been previously invoked
                           on this model.
      (>) mjd     (double) UTC data and time as a modified julian date.
      (>) dut     (double) Delta UT (UT1 - UTC) seconds.
      (>) temp    (double) Atmospheric temperature (K).
      (>) press   (double) Atmospheric pressure (mB).
      (>) humid   (double) Atmospheric humidity (0 to 1.0).
      (>) cenWave (double) Wavelength telescope pointed using (micons).
      (>) obsWave (double) Wavelength of observation (micons).
      (>) params  (double *) Telescope Model details.  An array of double
                           items giving parameters for the telescope model.
                           The number of values required here must be
                           put in NumParams.  
      (!) status  (StatusType *) modified status.
                           

 * Returned value:

 * Include files: fpi.h

 *-

 * History:
     09-Dec-1999 - TJF - Original version
     30-Sep-2002 - TJF - Add cenWave argument.  Renamed wave to obsWave.
 */
typedef void (*FpilModelCvtInitRoutineType) (
        FpilModelType *model,
        double mjd,
        double dut,
        double temp,
        double pres,
        double humid,
        double cenWave,
        double obsWave,
        const double *params,
        FpilStatusType *status);

/*
 *+			F p i l M o d e l R d 2 X y R o u t i n e T y p e

 * Function name:
      FpilModelRd2XyRoutineType

 * Function:
      User callback routine for converting from RA/Dec to X/Y

 * Description:
      This is a description of a routine type, used when calling
      FpilInit(3).  This routine type is used for the routine
      which converts from RA/Dec to X/Y on the field plate for the
      user fibre positioner instrument.

 * Language:
      C

 * Call:
      (void) = FpilModelRd2XyRoutineType(clientData, cra, cdec, ra, dec, 
                                         mjd, x, y, status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) model   (const FpilModelType *model) Contains the telescope 
                           model details  obtained by FpilModelAccess(3).
                           It Can be assumed that the CvtInit routine has been 
                           invoked on this.
      (>) cra     (double) Apparent RA of field center.
      (>) cdec    (double) Apparent DEC of field center.
      (>) ra      (double) Apparent RA of source.
      (>) dec     (double) Apparent DEC of source.
      (>) mjd     (double) UTC time and date expressed as Modified Julian Date
      (<) x       (double *) X position of source on field plate (microns).
      (<) y       (double *) Y position of source on field plate (microns).
      (!) status  (StatusType *) modified status.
 
 * Include files: fpi.h

 *-

 * History:
     09-Dec-1999 - TJF - Original version
 */        
typedef void (*FpilModelRd2XyRoutineType) (
        const FpilModelType *model,
        double cra,
        double cdec,
        double ra,
        double dec,
        double mjd,
        double *x,
        double *y,
        FpilStatusType *status);

/*
 *+			F p i l M o d e l X y 2 P o s R o u t i n e T y p e

 * Function name:
      FpilModelXy2PosRoutineType

 * Function:
      User callback routine for converting field plate to positioner coordinates

 * Description:
      This is a description of a routine type, used when calling
      FpilInit(3).  This routine type is used for the routine
      which converts from field plate X/Y cordinates to positioner X/Y
      coordinates for the user fibre positioner instrument.

 * Language:
      C

 * Call:
      (void) = FpilModelXy2PosRoutineType(clientData, x, y, xp, yp, status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) model   (const FpilModelType *model) Contains the telescope 
                           model details  obtained by FpilModelAccess(3).
                           It Can be assumed that the CvtInit routine has been 
                           invoked on this.
      (>) x       (double) Field plate x coordinate of object (microns).
      (>) y       (double) Field plate y coordinate of object (microns).
      (>) *xp     (double) Positioner X coordinate of object (microns).
      (>) *yp     (double) Positioner Y coordinate of object (microns).
      (!) status  (StatusType *) modified status.
                        
 * Include files: fpi.h

 *-

 * History:
     09-Dec-1999 - TJF - Original version
 */
typedef void (*FpilModelXy2PosRoutineType) (
        const FpilModelType *model,
        double x,
        double y,
        double *xp,
        double *yp,
        FpilStatusType *status);

/*
 *+			F p i l M o d e l X y 2 R d R o u t i n e T y p e

 * Function name:
      FpilModelXy2RdRoutineType

 * Function:
      User callback routine for converting X/Y to Ra/Dec.

 * Description:
      This is a description of a routine type, used when calling
      FpilInit(3).  This routine type is used for the routine
      which converts from X/Y positions on the field plate to RA/Dec for
      the user fibre positioner instrument.

 * Language:
      C

 * Call:
      (void) = FpilModelXy2RdRoutineType(clientData, cra, cdec, x, y, 
                                         mjd, ra, dec, status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) model   (const FpilModelType *model) Contains the telescope 
                           model details  obtained by FpilModelAccess(3).
                           It Can be assumed that the CvtInit routine has been 
                           invoked on this.
      (>) cra     (double) Apparent RA of field center.
      (>) cdec    (double) Apparent DEC of field center.
      (>) x       (double) X position of source on field plate (microns).
      (>) y       (double) Y position of source on field plate (microns).
      (>) mjd     (double) UTC time and date expressed as Modified Julian Date
      (<) ra      (double *) Apparent RA of source.
      (<) dec     (double *) Apparent DEC of source.
      (!) status  (StatusType *) modified status.


 * Include files: fpi.h

 *-

 * History:
     09-Dec-1999 - TJF - Original version
 */
typedef void (*FpilModelXy2RdRoutineType) (
        const FpilModelType *model,
        double cra,
        double cdec,
        double x,
        double y,
        double mjd,
        double *ra,
        double *dec,
        FpilStatusType *status);

/*
 *+			F p i l M o d e l P o s 2 X y R o u t i n e T y p e

 * Function name:
      FpilModelPos2XyRoutineType

 * Function:
      User callback routine for converting positioner to field plate coorindates

 * Description:
      This is a description of a routine type, used when calling
      FpilInit(3).  This routine type is used for the routine
      which converts positioner X/Y coordinates to field plate coorindates for
      the user fibre positioner instrument.

 * Language:
      C

 * Call:
      (void) = FpilModelPos2XyRoutineType(clientData,  xp, yp, x, y, status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) model   (const FpilModelType *model) Contains the telescope 
                           model details  obtained by FpilModelAccess(3).
      (>) xp     (double) Positioenr X coordinate of object (microns).
      (>) yp     (double) Positioenr Y coordinate of object (microns).
      (>) *x      (double) Field plate x coordinate of object (microns).
      (>) *y      (double) Field plate y coordinate of object (microns).
      (!) status  (StatusType *) modified status.

 * Include files: fpi.h

 *-

 * History:
     09-Dec-1999 - TJF - Original version
 */
typedef void (*FpilModelPos2XyRoutineType) (
        const FpilModelType *model,
        double xp,
        double yp,
        double *x,
        double *y,
        FpilStatusType *status);

/*
 *+			F p i l M o d e l F r e e R o u t i n e T y p e

 * Function name:
      FpilModelFreeRoutineType

 * Function:
      User callback routine for free the telescope model

 * Description:
      This is a description of a routine type, used when calling
      FpilInit(3).  This routine type is used for the routine
      which frees the telescope model details acuired in a call
      to FpilModelAccess for the user fibre positioner instrument.

 * Language:
      C

 * Call:
      (void) = FpilModelFreeRoutineType(model,status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) model   (FpilModelType *model) Contains the telescope model details
                           obtained by FpilModelAccess(3).  The CvtInit routine
                           may or may not have been previously invoked
                           on this model.
      (!) status  (StatusType *) modified status.
                        

 * Include files: fpi.h

 *-

 * History:
     09-Dec-1999 - TJF - Original version
 */
typedef void (*FpilModelFreeRoutineType) (
        const FpilModelType *model,
        FpilStatusType *status);
/*
 *+			F p i l M o d e l Q A O R o u t i n e T y p e

 * Function name:
      FpilModelQAORoutineType

 * Function:
      User callback routine for quick conversion from apparent to observed.

 * Description:
      This is a description of a routine type, used when calling
      FpilInit(3).  This routine type is used for the routine
      which does a quick conversion from apparent place ot observed place.

 * Language:
      C

 * Call:
      (void) = FpilModelQAORoutineType(model, ra, dec,
                                        aob, zob, hob, dob, rob,status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) model   (FpilModelType *model) Contains the telescope model details
                           obtained by FpilModelAccess(3). 
      (>) ra      (double) Apparent center RA in radians.
      (>) dec     (double) Apparent Center Declination in radians.
      (<) aob     (double *) Observed azimuth (radians, N=0, E = 90degrees)
      (<) zob     (double *) Observed zenith distance (radians)
      (<) hob     (double *) Observed Hour Angle (radians)
      (<) dob     (double *) Observed Declination (radians)
      (<) rob     (double *) Observed Right Ascension (radians)
      (!) status  (StatusType *) modified status.
                        

 * Include files: fpi.h

 *-

 * History:
     09-Dec-1999 - TJF - Original version
 */
typedef void (*FpilModelQAORoutineType) (
        const FpilModelType *model,
        double ra,
        double dec,
        double *aob,
        double *zob,
        double *hob,
        double *dob,
        double *rob,
        FpilStatusType *status);

typedef  struct {
    
    FpilColButFibRoutineType ColButFib; /* For button/fibre collisions       */
    FpilColButButRoutineType ColButBut; /* For button/button collisions      */
    FpilColFidButRoutineType ColFidBut; /* For fiducial/button collisions    */
    FpilColInvPosRoutineType ColInvPos; /* For button against invalid 
                                           positions                         */
    FpilGetVirPivRoutineType GetVirPiv; /* Return the virtual pivot point for*/
                                        /* a given button                    */
    FpilConsAccessRoutineType AccessCons; /* Access instrument constants file*/
    FpilWhichSpecRoutineType WhichSpec; /* Determine the spectrograph */
    FpilDrawRoutineType      DrawButFib; /* Provide information to use
                                            to draw an image of a button and
                                            fibre */
    
} FpilCollisionChecksType;


typedef struct {
    FpilModelAccessRoutineType Access; /* For loading model */
    FpilModelFreeRoutineType   Free;   /* For freeing model structure */
    FpilModelCvtInitRoutineType CvtInit;/* For initialsing transformation*/
    FpilModelRd2XyRoutineType  Rd2Xy;  /* For converting ra/dec to X/Y */
    FpilModelXy2PosRoutineType Xy2Pos; /* For converting X/Y to positioner */
    FpilModelXy2RdRoutineType  Xy2Rd;  /* For converting X/Y to ra/dec */
    FpilModelPos2XyRoutineType Pos2Xy; /* For converting positioner to x/y */
    FpilModelQAORoutineType    QuickAppToObs;/* For converting Apparent place
                                                to observed place */

    unsigned int NumParams;            /* Number of model parameters */
    const char * const * ParamNames;   /* Pointer to array of parameter names*/
    const char * const * ParamDescrs;  /* Pointer to array of parameter descriptions */
    const double * ParamDefaults;      /* Pointer to array of parameter default
                                          values */
    
} FpilModelRoutinesType;


/*
 * A structure which is passed around the telescope model access routines.
 */
struct FpilModelStruct {
    FpilModelRoutinesType routines;
    char Telescope[FPIL_NAMELEN]; /* Telescope name */
};


typedef struct {
    unsigned long ButtonClear;  /* Min clearance between buttons (microns)   */
    unsigned long FibreClear;   /* Min button/fibre clearance      (microns) */
    unsigned long FidClear;     /* Min Clearance for a fiducal mark (microns)*/
    unsigned long FieldRad;     /* Field radius (microns)                    */
    unsigned long FieldRadIncPark;/* Field radius including space to park    
                                     the fibres (which is not part of the 
                                     allocatable field                       */
    unsigned long Extension;    /* Max fibre extension        (microns)      */
    double        MaxPivAngle;  /* Max allow piv/fibre angle (radians)      */
    double        MaxFibAngle;  /* Max allowed fibre/button angle (radians)  */
        
    int           FibAngleVar;  /*  Logical. If true, the fibre/button angle 
                                    can be varied to the limit MaxFibAngle.  
                                    OTherwise, the button rotates such that 
                                    this is always zero.                     */
    unsigned long MaxSize;      /*  Maximum size of button in any planner 
                                    dimension (microns)
                                    (Replaces use of BUTTON_FVP in config.c) */
    unsigned int NumFields;     /* Number of field plates */
    unsigned int NumPivots;     /* Number of pivots ona field */
    unsigned int NumFids;       /* Number of fiducials on a plate */
    unsigned int NumSpec;       /* Number of spectrographs */
    unsigned int ParkMayCollide;/* Logical.  If true, then when fibres are
                                   parked, they may collide with other fibres*/
    char         Telescope[FPIL_NAMELEN]; /* Telescope name */
    char         Instrument[FPIL_NAMELEN];/* Instrument name */
    const char * const * SpecNames;   /* Pointer to array of spectrograph
                                         names*/
} FpilInstParamsType;

/*  Note: (added KS, 21/2/2001) We need to be careful about the use of
 *  Extension and extenSqrd, given that some instruments have different
 *  maximum extensions for different fibres. I believe that extenSqrd is
 *  not actually used - it appears in FpilCheckExtension2(), but that
 *  routine isn't used any more. Extension is - I think - only used by
 *  FpilGetMaxExtension() and that is only used in places where the overall
 *  max extension is needed.
 */

typedef struct {
    int   version;                     /* Currently, value passed to FpilInit*/
    int   hasModel;                    /* Do we support the telescope model */
    void  * clientData;                /* Use clientData routine */
    double butTolSqrd;                 /* params.ButtonClear squared */
    double fidTolSqrd;                 /* params.FidClear squared */
    double fibreTolSqrd;               /* params.FibreClear squared */
    double fieldRadSqrd;               /* params.FieldRad squared */
    double extenSqrd;                  /* params.Extension squared */
    unsigned long maxButSize;          /* Various values added*/
    FpilCollisionChecksType routines;  /* The routines */
    FpilInstParamsType      params;    /* The parameters */
    FpilModelRoutinesType   model;     /* The telescope model routines */
    
} Fpil___InstType;
/*
 * exported function prototypes
 */


/* Initialise FpilType */
extern int FpilInit(
    void *clientData,
    int version,
    const FpilCollisionChecksType * routines,
    const FpilInstParamsType      * params,
    const FpilModelRoutinesType   * modelRoutines,
    FpilType *inst);




extern void FpilFree(FpilType inst);

/* Collision routines */

extern int  FpilColFibFib (
        const FpilType inst,
        double  xapiv,
        double  yapiv,
        double  xa,
        double  ya,
        double  xbpiv,
        double  ybpiv,
        double  xb,
        double  yb);

extern int  FpilColButFib (
        const FpilType inst,
        double    butX,
        double    butY,
        double    theta,
        double    fvpX,
        double    fvpY,
        double    pivX,
        double    pivY);

extern int  FpilColButBut (
        const FpilType inst,
        double    butXa,
        double    butYa,
        double    thetaa,
        double    butXb,
        double    butYb,
        double    thetab);


extern int  FpilColFiducial (
        const FpilType inst,
        long int  butx,
        long int  buty,
        double    theta,
        long int  end1x,
        long int  end1y,
        long int  end2x,
        long int  end2y,
        long int  markx,
        long int  marky);


extern int FpilCollision1(
        const FpilType inst,
        long int butXa,
        long int butYa,
        double thetaa,
        long int pivXa,
        long int pivYa,
        long int butXb,
        long int butYb,
        double thetab,
        long int pivXb,
        long int pivYb);

extern int FpilCollision2(
        const FpilType inst,
        long int butXa,
        long int butYa,
        double thetaa,
        long int fibXa,
        long int fibYa,
        long int pivXa,
        long int pivYa,
        long int butXb,
        long int butYb,
        double thetab,
        long int fibXb,
        long int fibYb,
        long int pivXb,
        long int pivYb);

extern int  FpilColInvPos (
        const FpilType inst,
        unsigned  plate,
	int       fibreType,
        long int  butx,
        long int  buty,
        double    theta);


extern double        FpilGetMaxPivAng(const FpilType inst);
extern double        FpilGetMaxFibAng(const FpilType inst);
extern int           FpilGetFibAngVar(const FpilType inst);
extern unsigned long FpilGetMaxButSize(const FpilType inst);
extern unsigned long FpilGetMaxExtension(const FpilType inst);
extern unsigned long FpilGetFibreLength(const FpilType inst, int fibreType);
extern unsigned long FpilGetFieldRadius(const FpilType inst);
extern unsigned long FpilGetFieldRadiusIncPark(const FpilType inst);

extern unsigned long FpilGetButClear(const FpilType inst);
extern unsigned long FpilGetFibClear(const FpilType inst);
extern unsigned long FpilGetFidClear(const FpilType inst);

extern void          FpilSetButClear(const FpilType inst, 
                                     unsigned long value);
extern void          FpilSetFibClear(const FpilType inst, 
                                     unsigned long value);
extern void          FpilSetFidClear(const FpilType inst, 
                                     unsigned long value);
extern void          FpilSetMaxPivAng(const FpilType inst, 
                                      double value);
extern void          FpilSetMaxFibAng(const FpilType inst, 
                                      double value);



extern int           FpilOnField(const FpilType inst, 
                                 long x, 
                                 long y);

extern int           FpilExtensionCheck1(const FpilType instrument, 
                                         double length);

extern int           FpilExtensionCheck2(const FpilType instrument, 
                                        double deltaX, double deltaY);


extern void          FpilGetVirtPiv (const FpilType instrument,
                                     double OriginX,
                                     double OriginY,
                                     double Theta,
                                     double *VirtPivX,
                                     double *VirtPivY);

extern unsigned int FpilGetNumFields(const FpilType instrument);
extern unsigned int FpilGetNumPivots(const FpilType instrument);
extern unsigned int FpilGetNumFiducials(const FpilType instrument);
extern const char*  FpilGetTelescope(const FpilType instrument);
extern const char*  FpilGetInstName(const FpilType instrument);

extern void         FpilConsAccess(const FpilType instrument,
                                   const char *dirOverride,
                                   const char *dirFallback,
                                   FpilRptRoutineType RptRoutine,
                                   void * RptArg,
                                   FpilConstantsType *details,
                                   FpilStatusType *status);


extern void         FpilConFree(FpilConstantsType details,
                                FpilStatusType *status);
    
extern void         FpilConId(FpilConstantsType details,
                              FpilSdsIdType *id,
                              FpilStatusType *status);
extern void         FpilConFieldId(FpilConstantsType details,
                                   unsigned int field,
                                   FpilSdsIdType *id,
                                   FpilStatusType *status);

extern void         FpilConWhichSpec(
    const FpilType instrument,
    FpilConstantsType details,
    unsigned field,
    unsigned pivot,
    unsigned *spec,
    const char ** SpecName,
    FpilStatusType *status);

extern void FpilConSpecInfo(
    const FpilType instrument,
    unsigned int *NumberSpec,
    const char * const *  *SpecNames,
    FpilStatusType *status);




extern void FpilDrawButFib (
    const FpilType instrument,
    double scaling,
    double zoom,
    FpilCvtRoutineType CvtX,
    FpilCvtRoutineType CvtY,
    void *CvtData,
    double X,
    double Y,
    double Theta,
    int Type,
    unsigned ButGrapLen,
    double *FibX,
    double *FibY,
    char *ButGraphic1,
    char *ButGraphic2,
    FpilStatusType *status);

extern void FpilModelAccess(
    const FpilType instrument,
    const char *dirOverride,
    const char *dirFallback,
    FpilRptRoutineType RptRoutine,
    void * RptArg,
    unsigned int field,
    FpilModelType **model,
    FpilStatusType *status);

extern void FpilModelFree(
    FpilModelType *model,
    FpilStatusType *status);

extern void FpilModelCvtInit(
    FpilModelType *model,
    double mjd,
    double dut,
    double temp,
    double pres,
    double humid,
    double cenWave,
    double obsWave,
    const double *params,
    FpilStatusType *status);

extern void FpilModelRd2Xy(
    const FpilModelType *model,
    double cra,
    double cdec,
    double ra,
    double dec,
    double mjd,
    double *x,
    double *y,
    FpilStatusType *status);
    
extern void FpilModelXy2Pos(
    const FpilModelType *model,
    double x,
    double y,
    double *xp,
    double *yp,
    FpilStatusType *status);


extern void FpilModelXy2Rd(
    const FpilModelType *model,
    double cra,
    double cdec,
    double x,
    double y,
    double mjd,
    double *ra,
    double *dec,
    FpilStatusType *status);

extern void FpilModelPos2Xy(
    const FpilModelType *model,
    double xp,
    double yp,
    double *x,
    double *y,
    FpilStatusType *status);
extern const char* FpilModelGetTelescope();

extern void FpilModelQuickAppToObs(
    const FpilModelType *model,
    double ra,
    double dec,
    double *aob,
    double *zob,
    double *hob,
    double *dob,
    double *rob,
    FpilStatusType *status);

extern void FpilModelParams(
    const FpilModelType *model,
    unsigned int *NumberParams,
    const char * const ** ParamNames,
    const char * const ** ParamDescrs,
    const double *  *       ParamDefaults,
    FpilStatusType *status);

extern int FpilParkMayCollide(
    const FpilType instrument);

extern int FpilEncodeTargetType (
    const FpilType instrument,
    char* targetString,
    char* targetType,
    char* targetSpect);
    
extern int FpilDecodeTargetType (
    const FpilType instrument,
    char  targetType,
    char  targetSpect,
    int   spectKnown,
    char* targetString,
    char* targetText);
    
extern int FpilIsTargetGuide (
    const FpilType instrument,
    char targetType);
    
extern int FpilIsTargetTelGuide (
    const FpilType instrument,
    char targetType);
    
extern int FpilIsTargetSky (
    const FpilType instrument,
    char targetType);
    
extern void FpilTargetSkyType (
    const FpilType instrument,
    char* targetType,
    char* spectType);
    
extern int FpilTargetPivotCompatible (
    const FpilType instrument,
    char targetType,
    char targetSpect,
    int  pivotType);
    
extern void FpilDrawTarget (
    const FpilType instrument,
    char  targetType,
    char  targetSpect,
    int   x,
    int   y,
    float zoom,
    float tscale,
    int   radius,
    int   offset,
    short useGrey,
    char* arguments);
    
extern char FpilUnallocatedType (
    const FpilType instrument);
    
extern char FpilDisabledTargetType (
    const FpilType instrument,
    char targetType);

extern char FpilEnabledTargetType (
    const FpilType instrument,
    char targetType);
    
extern int FpilBrokenFibreType (
    const FpilType instrument,
    int pivotType);

extern int FpilFibreReallyExists (
    const FpilType instrument,
    int pivotType);

extern int FpilIsFibreBroken (
    const FpilType instrument,
    int fibreType);
    
extern int FpilIsFibreGuide (
    const FpilType instrument,
    int fibreType);
    
extern void FpilFibreColour (
    const FpilType instrument,
    int fibreType,
    int fibreCombo,
    short useGrey,
    char* colour,
    int nChar);
    
extern void FpilTargetCriteria (
    const FpilType instrument,
    unsigned int* numCriteria,
    const char* const **criteria);
    
extern int FpilTargetCriteriaMet (
    const FpilType instrument,
    const int* flags,
    int numCriteria,
    char targetType,
    char targetSpect,
    int spectKnown);

extern void FpilPAFTypes (
    const FpilType instrument,
    int maxTypes,
    int *numTypes,
    char* typeNames[],
    char* abbrNames[]);

 extern void FpilPAFFibreCodes (
    const FpilType instrument,
    int PAFType,
    int maxTypeCodes,
    int *numTypeCodes,
    int TypeCodes[]);

extern void FpilFibreCombos (
    const FpilType instrument,
    int maxCombos,
    int *numCombos,
    char* comboNames[],
    char* modeNames[]);
    
extern void FpilComboFibreCodes (
    const FpilType instrument,
    int comboId,
    int maxTypeCodes,
    int *numTypeCodes,
    int TypeCodes[]);

extern char* FpilGetPivotText (
    const FpilType instrument,
    int pivotType);
    
extern int FpilPivotTypeMatch (
    const FpilType instrument,
    int pivotType,
    int secondType);
    
extern int FpilPivotSharesSky (
    const FpilType instrument,
    int pivotType);
    
extern void FpilSetPositionAngle (
    const FpilType instrument,
    short useDefault,
    double posangle,
    double *params);
    
extern void FpilWavelengthCombos (
    const FpilType instrument,
    int maxCombos,
    int *numCombos,
    char* comboNames[]);
    
extern void FpilSetWavelengthCombo (
    const FpilType instrument,
    int comboId);

extern void FpilSetFibreCombo (
    const FpilType instrument,
    int fibreComboId);

extern void FpilSetObsWavelength (
    const FpilType instrument,
    double wavelength);

extern double FpilGetFibreWavelength (
    const FpilType instrument,
    char targetType);
    
extern double FpilGetPointingWavelength (
    const FpilType instrument);
    
extern void FpilCompatiblePivotTypes (
    const FpilType instrument,
    int maxPivTypes,
    char targetType,
    char targetSpect,
    int *numPivTypes,
    int *pivTypes);
    
#ifdef __cplusplus
}
#endif
#endif


