/*******************************************************************************
* E.S.O. - VLT project
*
* "@(#) $Id: fpcolMdlFit.c 186402 2009-06-26 16:05:56Z mpruemm $
* 
*
* who       when      what
* --------  --------  ----------------------------------------------
* mpruemm   2009-06-23  Fix rcsId for VLT2009.
* tfarrell  18/10/02  Add extra argument used by powell function. 
* tfarrell  21/08/02  Don't output error table.
* kshortri  07/08/02  Some systems (Linux) define DBL_MAX in float.h.
* tfarrell  23/07/02  Add #define of _XOPEN_SOURCE .
*                     rather then _INCLUDE_XOPEN_SOURCE
* tfarrell  17/04/02  Restrain coefficents which might produce
*                     spuriously large coefficents.
* tfarrell  12/04/02  Include fpDefines and use fpMDL_NUM_PARS to define NPAR.
* tfarrell  11/07/01  Rename OzPos to OzPoz.
* tfarrell  10/07/01  Created ESO style header.
* jbailey   27/04/01  Created.
*/

/************************************************************************
*   NAME
*       fpcolMdlFit - Model fit for OzPoz.
*
*   SYNOPSIS
*       #include "fpcolMdl.h"
*
*
*        void fpcolMdlFit(
*                      int nobs, 
*                      FpilModelType *model,
*                      double tolerance,
*                      int maxIts,
*                      int mode, 
*                      double mjd, 
*                      double rc, 
*                      double dc,
*                      double *ra, 
*                      double *dec, 
*                      double *dra, double *ddec,
*                      int *pfit, 
*                      double *pvar, 
*                      double *rms, 
*                      double *psd, 
*                      StatusType *status);
*
*   DESCRIPTION
*     This module does a least squares fit of a model to a set of OzPoz
*     observations.
*
*     The routine fpcolMdlFit does the fit.  "nobs" is the number of fibres 
*     used.  "model" contains the telescope model. On input this must have 
*     the same parameters as that used to onfigure the fibres for the test. 
*     On output 
*     it will contain the fitted parameters.  "rc" should contain the field 
*     centre apparent RA (radians), and "dec" the Declination (radians).
*     "dra" and "ddec" are array sof size nobs containing the apparent RA
*     and Declination offsets needed to centre the star on the fibre.
*     "pfit" is an array specifying which parameters are to be fitted. 
*     It should have a value of zero if the corresponding parameter is
*     to be fitted. one to fix the parameter at its initial value.
*
*     "pvar" is an array of variances on the fitted parameters.
*     "rms" is the RMS of the fit in microns.
*     "psd " is the estimated Population Standard Deviation of the fit in microns.  
*           This is only valid if the number of objects exceeds the number of parameters being 
*           fited.  Otherwise it is set to 0
*           
*      "tolerance", "maxIts" and "mode" are ignored
*
*     "status" is a DRAMA style modifie status argument.
*
*     This module is based on AAO 2dF software and hence has formating
*     more appropiate for 2dF in some places rather then the ESO VLT
*     project.
***
*   ENVIRONMENT
*
*   CAUTIONS 
*
*   EXAMPLES
*
*   SEE ALSO
*       fpcol module, AAO FPIL Library.
*   BUGS   
*"
*------------------------------------------------------------------------
*/

/*
 * Define module id for error macros
 */

#define MODULE_ID     "fpcol"

#if !defined(ATTRIBUTE_UNUSED) && __GNUC__
#define ATTRIBUTE_UNUSED __attribute__((__unused__))
#else
#define ATTRIBUTE_UNUSED /*nothing*/
#endif

ATTRIBUTE_UNUSED static const char *rcsId="@(#) $Id: fpcolMdlFit.c 186402 2009-06-26 16:05:56Z mpruemm $";



#define _POSIX_SOURCE 1
#define _XOPEN_SOURCE 1

#include <stdlib.h>     /* For malloc and free */
#include <stdio.h>
#include <math.h>
#include <string.h>
#include <limits.h>
#include <float.h>
/*
 * Before including fpcolMdl.h we must define this macro to ensure
 * the correct include files are brought in.
 */
#define FPCOL_MDL_MODULE
#include "fpcolMdlCalLibs.h"
#include "fpcolMdlCalModel.h"
#include "fpDefines.h" /* For fpMDL_NUM_PARS */

/*
 * Slalib include files.
 */
#include "slalib.h"
#include "slamac.h"

/*#define DO_WANALYSIS*/

static void XyVec(int nobs, double rc, double dc, FpcalMdl *models, int *widx,
                double mjd, double *rd, double *xy, StatusType *status);
static void wanalysis(int npar, double *wmat, const double *vmat, const int *pfit);
static void rowsofbiggests(int npar, int col, const double *vmat, const int *pfit);

/*
 *+			  f p c o l M d l F i t

 * Function name:
      fpcolMdlFit

 * Function:
      Do a fit to a set of OzPoz astrometric test data


 * Description:
      This routine does a linear least squares fit to a set of OzPoz
      astrometric test data, in the form of a set of positions, and a 
      set of offsets required to center the stars.
      
      The fit is performed using the singular value decomposition
      technique.

 * Language:
      C

 * Call:
      (void) = fpcolMdlFit(nobs, model, mjd, rc, dc, ra, dec, dra, ddec,
                        pfit, pvar, rms, psd, status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) nobs    (int)           The number of fibres used
      (>) model   (FpcalMdl *model) Contains the telescope model. On input
                       this must have the same parameters as that used to
                       configure the fibres for the test. On output it will
                       contain the fitted parameters.
      (>) tolerance (double)     Ignored.
      (>) maxIts    (int)    Ignored.
      (>) mode      (int)    Ignored.
      (>) rc      (double)       Field centre apparent RA (radians)            
      (>) dc      (double)       Field centre apparent Dec (radians)
      (>) ra      (double *)     Array of size nobs containing the 
                                    apparent RA used to setup each fibre.
      (>) dec     (double *)     Array of size nobs containing the
                                    apparent Dec used to setup each fibre.
      (>) dra     (double *)     Array of size nobs containing the apparent
                                    RA offset needed to centre the star on
                                    the fibre.                                                            
      (>) ddec    (double *)     Array of size nobs containing the apparent
                                    Dec offset needed to centre the star on
                                    the fibre. 
      (>) widx    (int *)        Array of size nobs containing an index to the
                                    wavelength dependent telescope model for the
                                    corresponding offset.
      (>) pfit    (int *)        Array specifying which parameters are to
                                    be fitted. It should have a value of
                                    zero if the corresponding parameter is
                                    to be fitted. one to fix the parameter
                                    at its initial value.
      (<) pvar    (double *)     Array of variances on the fitted parameters.                              
      (<) rms     (double *)     RMS of the fit in microns.
      (<) psd     (double *)     Estimated Population Standard Deviation
                                    of the fit in microns.                                                                                                                          
      (!) status  (StatusType *) modified status.
                           

 * Returned value:

 * Include files: fpcolMdl.h

 *-

 */


void fpcolMdlFit(int nobs, FpcalMdl *models, 
                 double tolerance, int maxIts, int mode, 
                 double mjd, double rc, 
                 double dc,
                 double *ra, double *dec, double *dra, double *ddec, int *widx,
                 int *pfit, double *pvar, double *rms, double *psd, 
                 StatusType *status)
              
{
   const int NPAR = fpMDL_NUM_PARS;  /* Number of parameters in the model */
   double *aumat;
   double *bmat;
   double *wmat;
   double *vmat;
   double *wk;
   double *xy0;
   double *xyp;
   double *xy;
   double *rd;
   double *rd2;
   int nfit;
   int ifit;
   int i,j;
   double xvec[NPAR];
   int jstat;
   double cov[NPAR][NPAR];
   double sumsq,var;
   int jf;
   FpcalPars pars;
#if 0
   double Variation[NPAR]; 
#endif
   
/*  Get Work arrays   */

   aumat = (double*) calloc ((2*nobs*NPAR),  sizeof(double)); /* design matrix */
   bmat =  (double*) calloc ((2*nobs),       sizeof(double)); /* error vector */
   wmat =  (double*) calloc (NPAR,           sizeof(double)); /* SVD matrices */
   vmat =  (double*) calloc (NPAR*NPAR,      sizeof(double));   
   wk =    (double*) calloc (NPAR,           sizeof(double));
   xy0 =   (double*) calloc (2*nobs,         sizeof(double)); /* Initial XY */
   xyp =   (double*) calloc (2*nobs,         sizeof(double)); /* Current XY */
   xy =    (double*) calloc (2*nobs,         sizeof(double)); /* Offset XY */
   rd =    (double*) calloc (2*nobs,         sizeof(double)); /* RA, Dec */
   rd2 =   (double*) calloc (2*nobs,         sizeof(double)); /* RA, Dec + offset */

/*  Number of parameters to be fitted  */

   nfit = 0;
   for (i=0;i<NPAR;i++) if (pfit[i] == 0) nfit++;



#if 0
/* 
 * Set up Variation array - how we vary each value    
 *  NOTE - this requires knowledge of the use of each parameter, which
 *         should not be done at this level - this array should be passed down.
 *
 * CURRENTLY PRODUCING WRONG RESULTS?
 */
   pars = *fpcolMdlCalGetPars(models);

   Variation[0] = 10;     /* Origin X - 10 microns */
   Variation[3] = 10;     /* Origin Y - 10 microns */
   Variation[1] = 0.1*pars.p[1];      /* Linear model parameters b, c , e, f */
   Variation[2] = 0.1*pars.p[2];
   Variation[4] = 0.1*pars.p[4];
   Variation[5] = 0.1*pars.p[5];

   Variation[ROTPAR] = .1;  /* Rotation, .1 degree */
   Variation[SCALEPAR]  = .01; /* Scale, 1% of the typical value, which is 1. */
   for (i = NW_DIST_FIRST ; i < NPAR ; ++i)
       {
       if (pars.p[i] == 0)
           Variation[i] = 1;   /* There are probably not being used */
       else
           Variation[i] = 0.1 * pars.p[i]; /* 10% of the existing value */
       }
   
#endif
/*  Set up rd and rd2 vectors */

   for (i=0;i<nobs;i++)
   {
      rd[2*i] = ra[i];
      rd[2*i+1] = dec[i];   
      rd2[2*i] = ra[i] + dra[i];
      rd2[2*i+1] = dec[i] + ddec[i];
   }         
   
/*  Initial XY vector  */

    XyVec(nobs,rc,dc,models,widx,mjd,rd,xy0,status);
   
/*  Set up design matrix - this is done by varying each parameter
    in turn and recalculating the x,y with the modified model */

   ifit = 0;
   for (i=0;i<NPAR;i++)
   {
       if (pfit[i] == 0)
       {
           pars = *fpcolMdlCalGetPars(models);
           pars.p[i] = pars.p[i] + 1/*Variation[i]*/;
           fpcolMdlCalSetPars(&pars,models);
           XyVec(nobs,rc,dc,models,widx,mjd,rd,xyp,status);
           pars.p[i] = pars.p[i] - 1/*Variation[i]*/;
           fpcolMdlCalSetPars(&pars,models);
           jf = 0;
           for (j=0;j<nobs;j++)
           {
               aumat[NPAR*2*jf+ifit]     = xyp[2*j]   - xy0[2*j];
               aumat[NPAR*(2*jf+1)+ifit] = xyp[2*j+1] - xy0[2*j+1];
               jf++;
           }
           ifit++;
       }        
   }                         
   
/*  
 * Set up the error vector - this is the vector of differences between
 *    the observations and the model  
 */
    
   XyVec(nobs,rc,dc,models,widx,mjd,rd2,xy,status);
   jf = 0;
   for (j=0;j<nobs;j++)
   {
       bmat[2*jf] = xy[2*j] - xy0[2*j];
       bmat[2*jf+1] = xy[2*j+1] - xy0[2*j+1];
/*
       printf("%10.5f %10.5f %10.5f %10.5f\n",bmat[2*jf],bmat[2*jf+1],
                   xy[2*j],xy0[2*j]);
*/
       jf++;
   }        
   
/*
 * Do singular value decomposition of design matrix  
 */
   slaSvd(2*nobs,nfit,2*nobs,NPAR,aumat,wmat,vmat,wk,&jstat);


#ifdef DO_WANALYSIS
/*
 * Analyse the W matrix to see if we need to restrain
 * spuriously large coefficent values.
 */
   wanalysis(NPAR, wmat, vmat, pfit);
#endif
 
/*
 * Hendle possible errors from slaSvd().
 */
   if (jstat == -1)
     {
       if ((2*nobs) < nfit)
         {
           fprintf(stderr, 
                   "Insufficent observations (nobs) for number of elements being fitted (nfit)\n");
           fprintf(stderr, "%d observation, %d parameters being fitted\n", nobs, nfit);
           fprintf(stderr, "We need 2*nobs >= nfit\n");
         } 
       else
         {
           /*
            * This one really looks like a programming error.  It seems to indicate
            * that either (2*nobs > 2*nobs) or (NPARS > nfit), neither of which 
            * should be the case.
            */
           fprintf(stderr,
                   "fpcolMdlFit:Programming error - slaSvd call, array A is the wrong shape\n");
           fprintf(stderr,
                   "fpcolMdlFit:Relevant items are nobs=%d, nfit=%d, npar=%d\n",
                   nobs, nfit, NPAR);
         }
         *status = 1;
         return;
     }
   else if (jstat > 0)
     {
       fprintf(stderr, "fpcolMdlFit:Warning, slaSvd call status = %d \n",jstat);
       fprintf(stderr, "                     Results may not be trustworthy, see slaSvd manual\n");
     }
   
/*  Solve the least squares problem  */

   slaSvdsol(2*nobs,nfit,2*nobs,NPAR,bmat,aumat,wmat,vmat,wk,xvec);
   
/*  Calculate covariance matrix  */

   slaSvdcov(nfit,NPAR,NPAR,wmat,vmat,wk,(double*)cov);
   
/*  Update parameters  */   
   
   j=0;
   pars = *fpcolMdlCalGetPars(models);
   for (i=0;i<NPAR;i++)
   {
       if (pfit[i] == 0)
       {
           pars.p[i] = pars.p[i]+xvec[j];
           pvar[i] = cov[j][j];
           j++;
       }
       else
       {
           pvar[i] = 0.0;
       }
   }
   fpcolMdlCalSetPars(&pars,models);
   
/*  New xy vector  */   
    
   XyVec(nobs,rc,dc,models,widx,mjd,rd,xyp,status);
   
   sumsq = 0.0;
   for (i=0;i<nobs;i++)
   {
       sumsq = sumsq + (xyp[2*i]   - xy[2*i])   * (xyp[2*i]   - xy[2*i]);
       sumsq = sumsq + (xyp[2*i+1] - xy[2*i+1]) * (xyp[2*i+1] - xy[2*i+1]);
   }
   var = sumsq/(double)nobs;
   *rms = sqrt(var);
   if (nobs > nfit)
     *psd = sqrt(var*(double)nobs/((double)nobs-(double)nfit));
   else
     *psd = 0;
   
   for (i=0;i<NPAR;i++)
   {
       pvar[i] = pvar[i] * var;
   }    

   printf("Singular Value Decomposition Fit completed\n");
   
   free(aumat);
   free(bmat);
   free(wmat);
   free(vmat);
   free(wk);
   free(xy0);
   free(xyp);
   free(xy);
   free(rd);
   free(rd2);
}       
               
   
static void XyVec(int nobs, double rc, double dc, FpcalMdl *models, int *widx,
                double mjd, double *rd, double *xy, StatusType *status)
                
{
    int i;
    double r,d;
    double x,y;
    double x2,y2;

    for (i=0;i<nobs;i++)                         
    {
        /* Index to telescope model for wavelength assoicated with this offset*/
      FpilModelType *telModel = fpcolMdlCalWaveModel(models, widx[i]);
        r = rd[2*i];
        d = rd[2*i+1];
        fpcolMdlRd2Xy(telModel,rc,dc,r,d,mjd,&x,&y,status);
        fpcolMdlXy2Pos(telModel,x,y,&x2,&y2,status);
        xy[2*i] = x2;
        xy[2*i+1] = y2;
    }    
}        

#ifdef DO_WANALYSIS
/*
 * Analyse the W matrix. The W matrix represents a npar*npar diagonal matrix - 
 * only the diagional elements are present.
 *
 * Relatively small W elements indicate ill-conditioning.  By setting the
 * suspiciously small W elements to zero, we restrain the offending coefficents
 * from moving very far.  Furthermore, if element w[j] was set to zero, then
 * the largest element in column j of the v matrix identifies the most dependent
 * solution coefficent and we log this.  According to the slaLib documentation, 
 * the second largest element in column j of v is also significant, but it
 * was not when tested.
 *
 * See Sla document, particular the section in "Starlink User Note 67" on 
 * "Numerical Methods".
 */
static void wanalysis(
    int npar,          /* Number of parameters - size of wmat */
    double *wmat,      /* The w matrix */
    const double *vmat,/* The v matrix, size npar*npar  */ 
    const int *pfit)   /* Array indicate what we are fitting */
{     
  int elem;
  double largest = -1.0*DBL_MAX; /* Most negative double (is this correct?) */
  /*
   * Find the largest element
   */
  for (elem = 0; elem < npar ; ++elem)
    {
      if (wmat[elem] > largest)
        largest = wmat[elem];
    }
  
  largest = fabs(largest);
  /*
   * Now find any w elements which are much smaller than the
   * largest element.
   */
  for (elem = 0; elem < npar ; ++elem)
    {
    if ((wmat[elem])&&(fabs(wmat[elem]) < (largest/10000.0)))
        {
          /*
           * Set this element to zero and work out the rows
           * of the biggest element in vmat.
           */
          wmat[elem] = 0;
          rowsofbiggests(npar, elem, vmat, pfit);
        }
    }
}

/*
 * This is used by wanalysis above.  It is just use to indicate which
 * is coefficent dependent on the suspicious element found above.
 */
static void rowsofbiggests(
    int npar,          /* Number of parameters - size of wmat */
    int col,           /* The colument */
    const double *vmat,/* The v matrix, size npar*npar  */ 
    const int *pfit)   /* Array indicate what we are fitting */
{
  int row;
  double largestVals[2];  /* Only valid if corresponding entry in largestRows is nonzero */
  int    largestRows[2] = {-1, -1};
  /*
   * Find the row with the largest two values in.  Note - this only
   * worked when I used the absolute value rather then just the
   * value.  Possibly this is an error in the sla documentation.
   */
  for (row = 0; row < npar ; ++row)
    {
      double val = fabs(vmat[row*npar+col]);
      if ((largestRows[0] == -1)||(val > largestVals[0]))
        {
          /* Push down the largest into the second largest */
          largestVals[1] = largestVals[0];
          largestRows[1] = largestRows[0];

          /* SAve the largest values and its row */
          largestVals[0] = val;
          largestRows[0] = row;
          
        }
      else if ((largestRows[1] == -1)||(val > largestVals[1]))
        {
          /* Larger then the second largests */
          largestVals[1] = val;
          largestRows[1] = row;
        }

    } 

  /*
   *  If we were trying to fit the coefficent we found, then log
   *  a message saying we are restraining it.
   */
  if (pfit[largestRows[0]] == 0)
    {
      printf("WARNING:Spurious fit coefficents for parameter of index %d\n", largestRows[0]);
      printf("        Restraining the fit of this parameter - see documentation\n");
    }
}
#endif



