/*******************************************************************************
* E.S.O. - VLT project
*
* "@(#) $Id: configure.c 290518 2016-11-24 15:14:57Z pbaksai $"
*
* who       when        what
* --------  ----------  ----------------------------------------------
* jpr       2016/10/03  Support for Linux_x86_64 native, cleaned up DEBUG
*                       from previous devel work for Mac OS X porting
* jpr/pba   2016/06/22  PPRS-65719: Corrected error in proper motion.
* jpritcha  2016-04-20  Mac OS X-alizing
* mpruemm   2009-06-23  Fix a few signed/unsigned mismatches.
* tshen     2006/10/21  Linux Porting: prototype for putenv() was added to avoid warnings.
* lsanzana  2004-06-11  added crc32 checksum to PAF file.
* rschmutz  2004-05-22  adapted to APR2004 (Tcl/Tk 8.4).
* rschmutz  14/06/03  PPRS 9365: ConfParseComment: check numbers.
* rschmutz  29/01/03  PPRS 8670: load/save INS.OSS.CHECKHA in PAF file.
* rschmutz  28/01/03  PPRS 8651: display CMM version instead of date.
* akaufer   26/01/03  PPRS 8633: skycat object symbols and colors changed
*                                VLT guide star added to list 
* akaufer   26/01/03  PPRS 8632: multiple lists
*                     ConfCompressUnalloc updated acc. to patch by KS
*                     ConfRemoveAlloc updated according to patch by KS
* rschmutz  20/01/03  PAF checksum disabled.
* rschmutz  16/01/03  PAF checksum activated/updated.
* rschmutz  24/11/02  search configure.tcl in ../lib.
* rschmutz  24/11/02  search Tcl/Tk libraries in ../lib/tcl, ../lib/tk.
* rschmutz  23/11/02  this header created - previous history see below.
*/

/*
 *                    C O N F I G U R E   ( F P O S S )
 * 
 *  Program name:
 *     Configure
 * 
 *  Function:
 *     Main C code module for the AAO Fibre Positioner Configuration program.
 *  
 *  Description:
 *     The 'configure' program is part of the observation preparation
 *     software used by a number of instruments built by the AAO. These
 *     include the original '2dF', the more recent '6dF' and the 'OzPoz'
 *     fibre positioner built for the 'FLAMES' instrument for ESO. The
 *     ESO version of this program is known as FPOSS - Fibre Positioner
 *     Observation Support Software. For more details about the overall
 *     structure of the code and what the program has to achieve, see the
 *     comments at the start of the file 'configure.tcl' or the FPOSS
 *     internals manual written for ESO.
 *     
 *     The comments at the start of configure.tcl emphasise that the
 *     structure of the configuration program is that of a Tcl/Tk program
 *     with a large number of Tcl commands being implemented in C. This
 *     file - configure.c - is the location of the interface code for all
 *     these Tcl commands. In many cases, the work for these Tcl commands is
 *     done through calls to other layers of the system, but in all cases
 *     a C-implemented Tcl command ends up invoking one of the routines in
 *     this file.
 *     
 *     This file also includes the main() entry point for the program, and
 *     in that sense this is the main code module for the program. However,
 *     the main() code does little other than initialise Tcl/Tk, causing
 *     them to be initialised in such a way that the routine ConfAddCommands()
 *     is invoked. This adds all the C-implemented Tcl commands to the
 *     Tcl interpreter. Control then passes on to the main-line code in
 *     configure.tcl, which puts up the main GUI for the program. After that,
 *     the program operates like a conventional (if quite complex) Tcl/Tk
 *     program.
 *
 *  Synopsis:
 *     configure [options]
 *     
 *     For a full description of all the options, see the Configure/FPOSS
 *     user guide.  In general use, very few of these options are ever used,
 *     particularly when used in the ESO context as FPOSS.
 *
 *  Language:
 *     C
 *
 *  Authors:
 *     Jeremy Bailey, AAO. (JAB)
 *     Gavin Dalton, University of Oxford (GBD)
 *     Tony Farrell, AAO (TJF)
 *     Keith Shortridge, AAO (KS)  
 *
 *  Program overview: 
 *     A relatively full description of the operation of the configuration
 *     program is provided in the FPOSS Internals manual written as part of
 *     the OzPoz project. This should be consulted to provide an overall
 *     picture of the operation of this code. In particular, it contains
 *     details of the SDS data structures manipulated by the program, and
 *     a walk-through what the code does as it performs a basic configuration.
 *     
 *  Pre-processor variables:
 *     FLAMES_ONLY  is defined if the program is to be built with support only
 *                  for FLAMES. Normally, it will be built with support for
 *                  all of 2dF, 6dF and FLAMES.
 *    
 *     Some additional pre-processor variables are used to control details
 *     of the compilation under different operating systems, but these will
 *     all be set by the program development environments for these machines.
 *     (For example, macintosh is defined when compiling on a Macintosh, 
 *     WIN32 is defines when compiling for Windows.)
 *     
 *  History:
 *     The original version of this program (only supporting 2dF) dates from
 *     1993. The history section of the code that dates from 1993 to 2001
 *     is long and - given the substantial rewriting to support multiple
 *     instruments such as 6dF and FLAMES - somewhat irrelevant. It has been
 *     removed from this file and saved in the file configure.history, which
 *     can be consulted as a historical document. The following section
 *     contains only a brief summary of changes up to 2001, and the detailed
 *     history starts with the release of the program with support for all of 
 *     2dF, 6dF and FLAMES.
 *     
 *     18-Oct-1993 - JAB - Original version
 *     15-Nov-1996 - GBD - Released with changes to support the 'Oxford' 
 *                         allocation algorithm.
 *     04-Nov-1998 - TJF - Reworked as a Tcl program instead of a DRAMA
 *                         program.
 *     20-Apr-2000 - TJF - Added support for 6dF.
 *     03-Jul-2000 - KS  - Initial version with OzPoz support.
 *     26-Sep-2001 - KS  - First release with proper OzPoz support. Commenting
 *                         revised.
 *     30-Oct-2001 - KS  - Added TargetDescr{}. Instrument info now passed to
 *                         PAF input conversion routines.
 *     05-Nov-2001 - KS  - Implemented ExpertAllowed{}.
 *     09-Nov-2001 - KS  - Implemented UpdateFibreStats{} and GetFibreStats{}.
 *     20-Nov-2001 - KS  - Added SetSkyCounts{} and support for explicit control
 *                         over the number of sky targets allocated for each
 *                         fibre type.
 *     21-Nov-2001 - KS  - SetSkyCounts{} was returning an undefined Tcl code
 *                         when called with no arguments. Fixed.
 *     26-Nov-2001 - KS  - The confusing CONF_SPECIAL code discontinued. A 
 *                         listing of unallocated objects now includes sky
 *                         targets as well as fiducial targets.
 *     28-Nov-2001 - KS  - Minor changes to get a clean compilation under
 *                         Linux.
 *     10-Dec-2001 - KS  - Blanks in PAF file items that don't allow them
 *                         now changed to underscores. Leading spaces are
 *                         now allowed in the values for items such as B-V
 *                         included in comment strings, and such items with
 *                         null values are not written to PAF files.
 *     19-Dec-2001 - KS  - GetFibre{} now makes sure it doesn't return fibre
 *                         type descriptions with embedded blanks.
 *     03-Jan-2002 - KS  - DoCheck{} now accepts a 'positioner tolerances'
 *                         optional argument. PAF files now include the 
 *                         INS.OSS.VER keyword, generated using the new
 *                         routine RCSVersion().
 *     30-Jan-2002 - KS  - Added SkycatListing{}.
 *     05-Feb-2002 - KS  - Added support for PAF file checksums. This is
 *                         disabled at present. To enable, change the value
 *                         in the #define for USE_CHECKSUM. For FLAMES, the
 *                         default humidity is now 20% instead of 50%.
 *     06-Feb-2002 - KS  - Skycat listing format improved - now defines
 *                         plotting symbols properly.
 *     12-Feb-2002 - KS  - Use of local tcl and tk directories modified to 
 *                         make it easier to change tcl/tk versions.
 *     25-Mar-2002 - KS  - ConfDoConversion() now uses FpilGetFibreWavelength().
 *                         This supports use of different wavelength 
 *                         combinations when checking configurations.
 *     11-Apr-2002 - KS  - Added PivotInUse{}. Supports -use_all_fibres command
 *                         line option.
 *     04-Jun-2002 - KS  - PAFFileName and WritePAF now incorporate the changes
 *                         suggested by Andreas Kaufer to incorporate the
 *                         time and mode into the file name and the PAF.ID and
 *                         PAF.NAME fields.
 *     05-Jun-2002 - KS  - Declaration of putenv() removed to compile properly
 *                         under Linux RedHat 7.2.
 *     30-Jul-2002 - KS  - Modified vertex set used to draw the VLT guide 
 *                         probe. See comments in the code. KS.
 *     08-Aug-2002 - KS  - Corrected field number passed to FpilModelAccess()
 *                         by SetPlate{}. Revised use of 'macintosh' pre-
 *                         processor variable to allow compilation on OS X.
 *     21-Aug-2002 - KS  - TEL.TARG.NAME now included in PAF files.
 *     22-Aug-2002 - TJF - Modify WritePAF to write the SKYANGLE correctly.
 *                         (Originally made at Paranal 20/8/02, incorporated
 *                         into this version by KS.)
 *     28-Aug-2002 - KS  - ReadPAF{} now returns a code giving the fibre
 *                         combination.
 *      5-Sep-2002 - KS  - Fixed bug where VLT guide stars were not being
 *                         identified properly in WritePAF{} because the
 *                         positions had changes. Stars now identified by name.
 *      2-Oct-2002 - KS  - Modified ArgusOffset from 90 to 0. This probably 
 *                         still isn't the right answer, but it makes things
 *                         less complicated while we try to find the right
 *                         answer.
 *     17-Oct-2002 - KS  - FpilModelCvtInit() now takes two wavelength 
 *                         parameters. ArgusOffset set to -90, which is now
 *                         believed to be the correct value - although this
 *                         may change.
 *     19-Oct-2002 - KS  - Incorporated two changes coming from Paranal during
 *                         commissioning. Code to decide which VLT guide probe
 *                         position is 'POS' and which 'NEG' has been modified,
 *                         and the atmospheric pressure used for FLAMES has
 *                         been reduced. The Argus offset is no longer
 *                         subtracted from the angle written as INS.ARGS.ANGLE.
 *                         Unused variables removed.
 *     22-Oct-2002 - KS  - VLT guide probe code comments tidied up.
 *     05-Nov-2002 - KS  - Now uses proper motion information in pseudo-comments
 *                         in .fld files to correct apparent RA/DEC.  PAF file
 *                         no longer has TEL.ROT.OFFANGLE.
 */

/*  ------------------------------------------------------------------------- */

/*                       P r e l i m in a r i e s
 *
 *  The usual stuff in any file before we get to the code proper. Assorted
 *  definitions, include files, forward declarations of routines..  To
 *  find the start of the real code, search forward for "M a i n".
 */

/*  DEBUG is used to make diagnostic code stand out */

#ifdef DEBUG
#undef DEBUG
#endif
#define DEBUG printf

/*  By default, at present, PAF files do not have checksums included. To
 *  enable the writing of checksums change USE_CHECKSUM to a non-zero value.
 */
 
#define USE_CHECKSUM 0

/*  The following define the file names used for the initial Tcl script
 *  and the directory we try to read it from.
 */

#define CONF_TCLDIR "CONFIGURE_DIR"
#define CONF_TCLFILE "configure.tcl"
#define CONF_SCRIPT "CONFIGURE_SCRIPT"

/*  Standard include files  */

#if defined WIN32
#include <windows.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <time.h>
#include <ctype.h>
#include <limits.h>
#ifndef macintosh
#include <unistd.h>
#include <sys/param.h>
#include <X11/Xlib.h>
#endif

/*
 *  Needed for Macintosh.
 */
 
#ifdef macos9
#  include <unistd.h>
   extern int MacintoshInit _ANSI_ARGS_((void));
   extern int SetupMainInterp _ANSI_ARGS_((Tcl_Interp * interp));
#endif

/*  Not all systems seem to declare readlink() properly, so we do so here.
 *  Ditto strcasecmp.
 */

#ifndef __macosx__
int readlink (const char* path, char *buf, size_t bufsiz);
int strcasecmp(const char *s1, const char *s2);
#endif

/*  At the moment, the Mac OS 9 version only has support for FLAMES */

#ifdef macos9
#ifndef FLAMES_ONLY
#define FLAMES_ONLY 1
#endif
#endif

/*  Program-specific include files  */

#include "drama.h" 
#include "status.h"
#include "sds.h"
#include "Ers.h"
#include "mess.h"
#include "sdstcl.h"
#include "arg.h"
#include "arg_err.h" 
#include "tcl.h"
#include "tdFconfig.h"
#include "tk.h"
#include "tdFconvert.h"
#include "convertPAF.h"
#include "slalib.h"
#include "slamac.h"
#include "sds_err_msgt.h"
#include "arg_err_msgt.h"
#include "conf_err_msgt.h"
#include "config_err_msgt.h"
#include "tdfxy_err_msgt.h"
#include "sixdf_util_err_msgt.h"
#include "fpil_err_msgt.h"
#include "conf_err.h"
#include "pafchk.h"

/* Tcl_GetStringResult() forward compatibility
 * http://wiki.tcl.tk/1564
 */
#if (TCL_MAJOR_VERSION < 8)
#define Tcl_GetStringResult(interp) (interp->result)
#endif

/*
 *  Include FPIL interface headers for each supported instrument.
 */

#include "tdffpil.h"            /* 2dF - AAT */
#include "sixdffpil.h"          /* 6dF - UKSCHMIDT */
#include "fpcolFpil.h"          /* OzPoz - VLT-2 */

typedef void (*ConfFpilInitRoutineType) (FpilType * inst);

/*
 *  Assuming each supported instrument has a function with the
 *  following typedef which does the FPIL Full initialisation, then
 *  we can set up an array of supported instruments here and
 *  otherwise don't have to worry about them - FPIL gives us all
 *  the details.
 */

extern void TdfFpilFullInit(FpilType * inst);

/*  This should be made more general, but for the moment it's convenient
 *  to be able to build the system just for FLAMES. We should also be
 *  able to build one just for 2dF and one just for 6dF, I suppose - or
 *  maybe one for AAO use that supports both 2dF and 6dF but not FLAMES?
 */
 
#if defined(FLAMES_ONLY)
    static struct {
        FpilType Info;
        ConfFpilInitRoutineType InitRoutine;
    } Instruments[] = {
        {
        0, fpcolFpilInit}
    };
#else
    static struct {
        FpilType Info;
        ConfFpilInitRoutineType InitRoutine;
    } Instruments[] = {
        {
        0, TdfFpilFullInit},
        {
        0, fpcolFpilInit},
        {
        0, sixdfFpilFullInit}
    };
#endif

/*  A structure of type FibreStatStruct contains all the relevant statistics
 *  for a given fibre type. The global variable FibreTypeStats is a pointer
 *  to a malloc'd array of such a structure.
 */

typedef struct FibreStatStruct {
    char Name[80];                   /* Name describing fibre type */
    int Fibres;                      /* Total # fibres of this type */
    int AllocToSky;                  /* # fibres allocated to sky */
    int AllocToGuide;                /* # fibres allocated to guide stars */
    int AllocToObject;               /* # fibres allocated to objects */
    int Unallocated;                 /* # fibres not allocated to anything */
    int FibreSkyCount;               /* # fibres to be allocated to sky */
    int Targets;                     /* # compatible unallocated targets */
    int Skies;                       /* # compatible unallocated sky targets */
} FibreStatStruct;

/*  ------------------------------------------------------------------------- */

/*           G l o b a l  V a r i a b l e s  &  C o n s t a n t s
 */
 
/*
 *  When an instrument is selected, details are copied here. Note that most
 *  calls to Fpil routines take *Instrument as the first argument, so it is
 *  important that this be defined before any such calls are made. However,
 *  the way the program runs guarantees that the first thing that happens is
 *  that the instrument is selected.
 */

static FpilType *Instrument = 0;             /* Instrument details pointer */
static FpilConstantsType InstConstDetails;   /* Used to access constants   */
static FpilModelType *TelModel;              /* For telescope model        */
static SdsIdType InstFieldDetails;           /* Sds id of field details    */
static unsigned CurrentField;                /* Current field number       */
static double *TelModelParams;               /* Telescope model parameters */
static double *TelModelParamsUsed;           /* Telescope model parameters
                                              * used when the last coordinate
                                              * conversion was performed.  */
static unsigned int TelModelNumParams;       /* Number of tel model params */
unsigned int InstNumFibreTypes = 0;          /* Number of fibre types      */
static int *UseFibreType;                    /* Points to array of values  
                                                indicating if a fibre type
                                                should be configured.      */
static FibreStatStruct* FibreTypeStats;      /* Allocation statistics for  
                                                each fibre type.           */
time_t FileTime;                             /* Time used in file name     */

#define MAX_ERRORS 10           /* Maximum allocation errors */
#define ALLOCATE_NORMAL 0       /* ClientData item value to ConfDoAllocation */
#define ALLOCATE_BATCH  1       /* ClientData item value to ConfDoAllocation */

/*
 *  Note that of two below, CMTLEN is always used for buffers which may
 *  take either a comment or name item, hence if their sizes changes.
 *  CMTLEN must remain the largest.
 */
 
#define NAMELEN 80              /* Length of "name" items in structures */
#define CMTLEN  80              /* Length of "comment" items in structures */

/*       M i m i c  W i n d o w  S i z i n g  a n d  S c a l i n  g
 *
 *  Scaling is all controlled by the relationship between the field size
 *  in microns and the number of pixels used for the mimic window. At a
 *  zoom setting of 1, the mimic window shows just a little more than
 *  the whole available field.  The mimic window size in pixels is placed in the
 *  variable MimicWindowSize by the ConfInitialise() function, which
 *  determines it based on a percentage (MIMIC_PROPORTION) of the smallest 
 *  dimension of the display.
 *  
 *  A circle is drawn some pixels (MIMIC_CIRCLE_OUTSIDE) in from this to 
 *  represent the available field.  The radius of this circle is placed in 
 *  MimicCircleRadius and the center of this circle in relation to the mimic 
 *  window is placed in MimicCircleOffset. (For OzPoz, the code allows an
 *  additional number of pixels (MIMIC_GUIDE_EXTRA) to allow for the need to
 *  show the guide probe. For 6dF, we add an additional amount to allow for
 *  the pivot positions being some way outside the field (MIMIC_6DF_EXTRA)).
 *
 *  So all scaling is controlled by the relative values of the field
 *  size (fetched from FPIL using FpilGetFieldRadiusIncPark()) and the value
 *  of the variable MimicCircleRadius (the radius in pixels of the circle
 *  drawn in the   window to represent the field). 
 *
 *  Tcl commands are avaialble to fetch these values.
 */

unsigned int MimicWindowSize = 0;
unsigned int MimicCircleRadius = 0;
unsigned int MimicCircleOffset = 0;

#define MIMIC_PROPORTION 0.9
#define MIMIC_CIRCLE_OUTSIDE 10
#define MIMIC_GUIDE_EXTRA 60
#define MIMIC_6DF_EXTRA 10

/*  useGrey is a global flag that indicates that a grey-scale set of colours
 *  is being used. This is intended as an option mainy for colour-blind users,
 *  since more contrasting hues of grey can be selected if this flag is set.
 *  The tcl command useGrey gets the value of this flag, and the tcl
 *  command setGrey can set it.
 */
 
static short useGrey = 0;

/*  PAF flags are used to control the output for different objects when a
 *  PAF file is being written. The only one needed is PAF_NO_PRIO which
 *  is used to suppress the priority information for VLT guide stars.
 */
 
#define PAF_NO_PRIO 1

/*
 *  Imported from the configversion.c module.
 */
 
extern const char *const ConfigVersion;

/*
 * WIN32 compatibility stuff
 */
 
#ifdef WIN32

#define MAXPATHLEN MAX_PATH

/* Symbolic constants for the "access" routine: */

#define R_OK    4               /* Test for Read permission */
#define W_OK    2               /* Test for Write permission */
#define X_OK    1               /* Test for eXecute permission */
#define F_OK    0               /* Test for existence of File */

/*
 * WIN32 does not provide the "access" function. Here is a
 * cheap and nasty equivalent - just checks for existence of the file.
 */
 
static int access(const char *path, int amode)
{
    DWORD attributes = GetFileAttributes(path);
    if (attributes == 0xFFFFFFFF)
        return -1;              /* File does not exist */
    else {
        return 0;               /* File exists, but not sure how to check 
                                 * for readability or executability */
    }
}

#endif

/*  End of Win 32 compatibility stuff */

/*  Sds Ids - kept globally to speed up drawing of displays  */

static SdsIdType gxid = 0, gyid = 0, gthetaid = 0, gxPivid = 0, gyPivid =
    0, gtype = 0;

static SdsIdType gtopid = 0, gpivotid = 0, gxpid = 0, gypid = 0, gthetapid = 0;

/*
 *  XErrorCount is the number of X errors so far
 *  MAX_XERRORS is the maximum number we allow (to avoid error loops)
 */
 
static int XErrorCount = 0;
#define MAX_XERRORS 10

/*
 *  Flag to see if we should ignore some errors GBD 17/6/98
 */
 
static int ImportOnlyMode = 0;

/*  Flag to see if expert mode is allowed */

static int ExpertAllowed = 0;

/*  Flag to control enabling of all fibres, if requested by the -use_all_fibres
 *  command line option.
 */

static int EnableAllFibres = 0;

/*
 *  Executable directory.  Not all operating systems define MAXPATHLEN so
 *  define it here if needed.
 */
 
#ifndef MAXPATHLEN
#define MAXPATHLEN 1024
#endif
static char ExecutableDir[MAXPATHLEN];

/*
 *  Config files directory (value of the CONFIG_FILES parameter)
 */
 
static char ConfigFilesDir[MAXPATHLEN];

/*  Defaults for Temp, Pressure, Humidity and Wavelength. There are three
 *  wavelengths - the main one used for ordinary target objects, the one
 *  used for the guide stars (which will usually depend on any filters in
 *  the guiding system and the response of the guider - we assume that these
 *  effects dominate the colour of the guide star itself), and the wavelength
 *  used for the telescope guider.
 *
 *  At present, the user interface allows the temperature, pressure and
 *  humidity to be changed, and allows the main observing wavelength to
 *  be changed. However, at present, the guide wavelengths are fixed.
 *  Of the supported instruments, only FLAMES makes use of telescope guide
 *  targets. Note that Initialise{} modifies the humidity default if the
 *  instrument is FLAMES, in which case it uses 0.2. It also changes the
 *  pressure for FLAMES to 742.
 */
 
#define TEMP 285
#define PRESS 900
#define HUMID 0.5
#define WAVE 0.60

static double gtemp = TEMP;             /* Temperature in degrees kelvin */
static double gpress = PRESS;           /* Pressure in mm Hg */
static double ghumid = HUMID;           /* Humidity - as a fraction, ie 0 - 1 */
static double gwave = WAVE;             /* Wavelength in microns for main 
                                         * target objects (non-guide objects) */

/*  Flag to indicate batch mode operation */

static short batchMode = FALSE;

/*  ------------------------------------------------------------------------- */

/*                   F u n c t i o n  P r o t y p e s
 *
 *  Function prototypes for routines defined in this file.
 */

static int ConfAddCommands(Tcl_Interp * interp);
static void ConfDoXyConversion(SdsIdType id, double cra, double cdec,
    double mjd, double *gmap, StatusType * status);
static void ConfArrayGets(SdsIdType id, char *name, long num,
    short *s, StatusType * status);
static void ConfArrayGetc(SdsIdType id, char *name, long num,
    char *s, StatusType * status);
static void ConfArrayPuts(SdsIdType id, char *name, long num,
    short s, StatusType * status);
static void ConfArrayPutc(SdsIdType id, char *name, long num,
    char s, StatusType * status);
static void ConfArrayPuti(SdsIdType id, char *name, long num,
    int i, StatusType * status);
static void ConfArrayGeti(SdsIdType id, char *name, long num,
    int *i, StatusType * status);
static void ConfArrayPutd(SdsIdType id, char *name, long num,
    double d, StatusType * status);
static void ConfArrayPutString(SdsIdType id, char *name, long num,
    char *string, StatusType * status);
static void ConfArrayCopy(SdsIdType id1, SdsIdType id2, char *name, long size,
    long num1, long num2, StatusType * status);
static void ConfCopyAllItems(SdsIdType id, int j, int i, StatusType * status);
static void ConfResizeAllItems(SdsIdType id, int size, StatusType * status);
static void ConfRemoveAlloc(SdsIdType id, StatusType * status);
static void ConfListLine(FILE * listing, int i,
    SdsIdType ra_id, SdsIdType dec_id,
    SdsIdType name_id, SdsIdType type_id,
    SdsIdType spect_id, SdsIdType prio_id,
    SdsIdType mag_id, SdsIdType pid_id, SdsIdType comment_id,
    StatusType * status);
static void ConfListPAFTarget(FILE * listing, int i,
    char* prefix,
    SdsIdType ra_id,
    SdsIdType dec_id,
    SdsIdType name_id, SdsIdType type_id, SdsIdType spect_id,
    SdsIdType prio_id, SdsIdType mag_id, SdsIdType pid_id,
    SdsIdType comment_id, 
    int fibre,
    int PAF_Flags,
    StatusType * status);
static int ConfGetProperMotions ( char* Comment,double* PmRa,double* PaDec);                    /* Proper motion in Dec */
static void ConfListDSSLine(int skyonly,
    FILE * listing, int i, SdsIdType ra_id,
    SdsIdType dec_id,
    SdsIdType name_id, SdsIdType type_id, SdsIdType prio_id,
    SdsIdType mag_id, SdsIdType pid_id, SdsIdType comment_id,
    StatusType * status);
static void ConfGetIds(SdsIdType topid, SdsIdType * ra_id, SdsIdType * dec_id,
    SdsIdType * name_id, SdsIdType * type_id, SdsIdType * spect_id,
    SdsIdType * prio_id,
    SdsIdType * mag_id, SdsIdType * pid_id, SdsIdType * comment_id,
    StatusType * status);
static int ConfGetArgusData(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[]);
static int ConfSetArgusData(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[]);
static int ConfParseComment (char* Comment, char* Keyword, char* Value,
    char** NextChar);    
static int ConfWhichSpec(int IPiv, const char **Descr);
static void RCSVersion(char* Version, int Length);
static int XError(ClientData clientData, XErrorEvent * errorEventPtr);
static void ErsHandler(void *outArg DUNUSED, unsigned int count,
    const ErsMessageType messages[], StatusType * status);
static void ConfMessage(void *argument, const char *string);
static void ConfRptConstFile(void *argument, const char *string,
    StatusType * status);
static void ConfRptTelModelFile(void *argument, const char *string,
    StatusType * status);

/*  The following routines are made externally visible, so they can be
 *  used by modules such as the PAF input routines.
 */
 
void DoMakeAllocated (
   SdsIdType topid,
   SdsIdType pivid,
   StatusType* status);
   
void ConfCSetArgusData(
    short InUse,
    double Angle,
    char* Scale,
    double Offset);

void ConfCGetArgusData(
    short* InUse,
    double* Angle,
    char* Scale,
    double* Offset);

/*  ------------------------------------------------------------------------- */

/*                 U t i l i t y   R o u t i n e s
 *
 *  Utility routines for use by functions defined in this file. Some of these
 *  are declarations for routines defined later, other short routines are
 *  defined here.
 */

/*  checksum() calculates the checksum of a character array, and 
 *  char_encode() encodes an unsigned integer into a printable ASCII
 *  string. 
 */
 
static void checksum (char* buf, int length, unsigned short* sum16, 
                                              unsigned int* sum32);
                
static void char_encode (unsigned int value, char* ascii, int nbytes,
                                                          int permute);

/*  ImpZExeDir() returns the specification for the directory from which the
 *  executable program is run.  This is a direct copy of the internal DRAMA
 *  routine of the same name. It happens that this is a very low level routine
 *  and can be used stand-alone in this way.
 */

static void ImpZExeDir(char *Arg0, char *Directory, int LDir, int *Known);

/*  LinesIntersect() tests whether or not two vectors intersect. */

static int LinesIntersect(double VecX1, double VecY1, double VecX2,
    double VecY2, double X1, double Y1, double X2, double Y2);

/*  Sds Id leak checking routines. If Sds Ids are not released once they
 *  are no longer needed, subtle memory leaks and fragmentation can result.
 *  These routines can be used to test for Sds Id leaks. The basic principle
 *  used here is that if Sds is leaking Ids then Sds Id values (Sds Ids are
 *  just integers) will constantly increase - if Ids are not leaking then
 *  most will be reused as the program progresses.
 */
 
/*
 * Invoked to prepare for a Sds id leak test.  The value returned from
 * this function should be passed to LeakCheck to check for Sds leaks
 * between the two calls.
 */
 
typedef long int LeakCheckType;

static LeakCheckType LeakPrep(void)
{
    StatusType ignore = STATUS__OK;
    SdsIdType id = 0;
    ArgNew(&id, &ignore);
    ArgDelete(id, &ignore);
    if (ignore == STATUS__OK)
        return (LeakCheckType) id;
    else
        return -1;
}

/*
 * Check for Sds leaks.  The origVal should be a value returned
 * from LeakPrep() above prior to the sequences of Sds calls we
 * want to check for leaks.
 *
 * If a leak has occured, a message is written to stderr about it.
 *
 *  routine -> string output in message to indicate where the check was done.
 *  allow   -> We allow this many Sds id's to allocated in this routine
 *             as part of it's job.
 */
 
static void LeakCheck(const char *const routine,
    LeakCheckType origVal, int allowance)
{
    LeakCheckType newVal = LeakPrep() - allowance;
    if ((newVal == -1) || (origVal == -1)) {
        fprintf(stderr, "%s:Error checking for SDS leak\n", routine);
    } else if (newVal != origVal) {
        fprintf(stderr, "### %s:Leaked %ld Sds id's\07\n", routine,
            (newVal - origVal));
    } else {
        fprintf(stderr, "### %s:Leaked NO Sds id's\n", routine);
    }
}

/*  Tcl interfaces to LeakPrep() and LeakCheck() */

static int TclLeakPrep(ClientData clientdata DUNUSED, Tcl_Interp * interp,
    int argc DUNUSED, char *argv[]DUNUSED)
{
    sprintf(interp->result, "%ld", (long int) (LeakPrep()));
    /* Tcl_SetResult(interp, fprintf("%ld", (long int) (LeakPrep())), TCL_STATIC); */
    return TCL_OK;
}

static int TclLeakCheck(ClientData clientdata DUNUSED, Tcl_Interp * interp,
    int argc DUNUSED, char *argv[]DUNUSED)
{

    if (argc != 4) {
        sprintf(interp->result, "%s:Wrong number of arguments", argv[0]);
        return TCL_ERROR;
    }
    LeakCheck(argv[1], atol(argv[2]), atol(argv[3]));
    return TCL_OK;
}

/*
 * By setting a break point in this routine, we can cause the debugger
 * to be triggered from Tcl code.
 */
 
static int ConfInvokeDebugger(ClientData clientdata DUNUSED,
    Tcl_Interp * interp DUNUSED, int argc DUNUSED, char *argv[])
{
    printf("%s invoked\n", argv[0]);
    return TCL_OK;
}



#ifdef __linux__
   extern int putenv (char *__string) ;
#endif



/*  ------------------------------------------------------------------------- */
/*
 *                               M a i n
 *
 *  This is the main routine for Configure. It performs any necessary
 *  initialisations prior to starting up Tcl/Tk, and then passes control
 *  over to Tcl/Tk.  Note that Configure is best thought of as a Tcl/Tk
 *  program that makes use of a number of Tcl commands that are implemented
 *  in C. So this routine has the job of starting up Tcl/Tk in such a
 *  way that these additional commands are defined, and doing any necessary
 *  initialisation needed by the C routines that implement the Tcl commands.
 */

extern int main(int argc, char *argv[])
{
    /*  Local variables  */
    
    char filename[MAXPATHLEN];
    char library[MAXPATHLEN];
    char *configFilesValPnt;
    char tcl_library_env[MAXPATHLEN + 40] = "TCL_LIBRARY=";
    char tk_library_env[MAXPATHLEN + 40] = "TK_LIBRARY=";
    StatusType status = STATUS__OK;
    int known;
    short sysgenMode = FALSE;
    register unsigned i;

    char **newArgv;                      /* Modified command line arguments */
    
    /*  Initialise the Mess routines. These decode status return codes,
     *  and need to be told which sets of status codes are expected. 
     */
    
    MessPutFacility(&MessFac_SDS);
    MessPutFacility(&MessFac_ARG);
    MessPutFacility(&MessFac_CONF);
    MessPutFacility(&MessFac_CONFIG);
    MessPutFacility(&MessFac_TDFXY);
    MessPutFacility(&MessFac_SIXDF_UTIL);
    MessPutFacility(&MessFac_FPIL);
    
    /*  Assume we're not in batch mode. We'll check the command arguments
     *  later. Ditto system generation mode.
     */
     
    batchMode = FALSE;
    sysgenMode = FALSE;
    
#ifndef macos9

    /*  This is the usual start-up for Windows or Unix.  The Mac (as usual)
     *  is significantly different and is handled in the #else clause, at
     *  least in its old OS 9 version. OS X is more or less standard UNIX.
     */

    /*  See if we are being run from a program that includes the name
     *  'expert'. If we are, allow expert mode.
     */
     
    if (strstr(argv[0], "expert")) ExpertAllowed = 1;
    
    /*
     *  A quick pass through the command line arguments to see if we are
     *  running in batch mode - one of the -a or -d flags should be specified
     *  for batch operation.
     *  
     *  We also look to see if one of the tcl_library or tk_library flags
     *  is specified, in which case this is just being run as part of the
     *  system generation process. Ditto -tk_version and -tcl_version.
     *  -tk_version is reported here, as there isn't an [info tkversion]
     *  option.
     *
     *  We also look for the -use_all_fibres option. This is handled entirely
     *  by the C layer and allows an expert user to force all fibres to be
     *  regarded as in use. This is really only for use during commissioning,
     *  when the fibres available changes rapidly and many configurations
     *  are done by hand anyway.
     */
 
    for (i = 1; i < (unsigned) argc; ++i) {
        if (!(strncmp(argv[i],"-a",2)) || !(strncmp(argv[i],"-d",2))) {
           batchMode = TRUE;
        }
        if (!(strncmp(argv[i],"-tk_library",11))  || 
            !(strncmp(argv[i],"-tcl_library",12)) ||
            !(strncmp(argv[i],"-tcl_version",12))) {
           sysgenMode = TRUE;
        }
        if (!(strncmp(argv[i],"-tk_version",11))) {
           sysgenMode = TRUE;
           printf ("%s\n",TK_VERSION);
        }
        if (!(strncmp(argv[i],"-use_all_fibres",15))) {
           if (!ExpertAllowed) {
              printf (
                "-use_all_fibres is only allowed if expert mode is enabled\n");
              return (0);
           } else {
              EnableAllFibres = TRUE;
              printf ("All fibres will be treated as in use\n");
           }
        }
    }
        
    /*
     *  Find where are being run from so we can find other files.
     */
     
    ImpZExeDir(argv[0], ExecutableDir, sizeof(ExecutableDir), &known);

    /*  See if the CONFIG_FILE environment variable has a value and if
     *  so, grab it.
     */
     
    configFilesValPnt = getenv("CONFIG_FILES");
    if (configFilesValPnt) {
        strncpy(ConfigFilesDir, configFilesValPnt, sizeof(ConfigFilesDir));
        ConfigFilesDir[sizeof(ConfigFilesDir) - 1] = '\0';
    } else
        ConfigFilesDir[0] = '\0';

    /*  See if the Tcl library files are in the directory we are being
     *  run from.  If so, we must set TCL_LIBRARY and TK_LIBRARY. We expect
     *  these to be in the directories '../lib/tcl' and '../lib/tk'. As a check,
     *  the makefiles that create these should have included a file in
     *  each giving the version number. We check for that - it's important
     *  that it match the version we've been linked with.
     */
     
    strcpy(library, ExecutableDir);
    strcat(library, "../lib/tcl");
    if (access(library, R_OK | X_OK) == 0) {

        /*  Note, the strange putenv function requires that tcl_library_env
         *  be kept about until the variable is overwritten (or in this case
         *  until the program completes).
         */
         
        strcat(tcl_library_env, library);
        putenv(tcl_library_env);
        
        /*  Check the version - we are looking for a file called 
         *  TCL_VERSION_m.n where m.n is the current tcl version.
         */
        
        strcat(library,"/TCL_VERSION_");
        strcat(library,TCL_VERSION);
        if (access(library, R_OK) != 0) {
           printf ("Warning - local ../lib/tcl may not be for Tcl version %s\n",
                                                                TCL_VERSION);
        }
    }

    /*  Now tk. */
    
    strcpy(library, ExecutableDir);
    strcat(library, "../lib/tk");
    if (access(library, R_OK | X_OK) == 0) {
     
        /*  Note, the strange putenv function requires that tk_library_env
         *  be kept about until the variable is overwritten (or in this case
         *  until the program completes).
         */
         
        strcat(tk_library_env, library);
        putenv(tk_library_env);
        
        /*  Check the version - we are looking for a file called 
         *  TK_VERSION_m.n where m.n is the current tk version.
         */
        
        strcat(library,"/TK_VERSION_");
        strcat(library,TK_VERSION);
        if (access(library, R_OK) != 0) {
           printf ("Warning - local ../lib/tk may not be for Tk version %s\n",
                                                                TK_VERSION);
        }
    }

    /*  Find the configure tcl script.  We first look for the environment
     *  variable defined by CONF_SCRIPT.  If that is not defined, then we
     *  look for configure.tcl in the directory CONFIGURE_DIR.  Finally, we
     *  look for configure.tcl in the executable directory + "../lib".
     */
     
    if (getenv(CONF_SCRIPT)) {
        strcpy(filename, getenv(CONF_SCRIPT));
        printf("%s:Using script file \"%s\"\n", argv[0], filename);
    } else if (getenv(CONF_TCLDIR)) {
        strcpy(filename, getenv(CONF_TCLDIR));
        strcat(filename, "/");
        strcat(filename, CONF_TCLFILE);
    } else {
        strcpy(filename, ExecutableDir);
        strcat(filename, "../lib/");
        strcat(filename, CONF_TCLFILE);
    }

    /*  We have to add the Tcl script filename as the first argument.  This
     *  is required by Tk_Main.  Thus, we have to malloc space for a new
     *  argv array.
     */
     
    newArgv = malloc(sizeof(char *) * (argc + 1));
    newArgv[0] = argv[0];
    newArgv[1] = filename;
    for (i = 1; i < (unsigned) argc; ++i) {
        newArgv[i + 1] = argv[i];
    }
    
    /*  Make sure the global FileTime variable contains a sensible time */
    
    FileTime = time(0);
    
#else

    /*
     *  Macintosh is rather different from windows and unix. We can really
     *  only make use of the executable directory. We also need to run
     *  the Mac-specific initialisation code provided by Tcl.
     */
     
    if (MacintoshInit() != TCL_OK) Tcl_Exit(1);
    
    ExpertAllowed = 1;
    argc = 0;
    newArgv = malloc(sizeof(char *) * (argc + 1));
    newArgv[0] = "configure";
    newArgv[1] = NULL;
    if (getcwd(ExecutableDir, sizeof(ExecutableDir)) == NULL) {
        ExecutableDir[0] = '\0';
    } else {
        ExecutableDir[strlen(ExecutableDir) - 1] = '\0';
    }
    ConfigFilesDir[0] = '\0';
    
#endif                          /* macintosh */

    /*  Initialise each supported instrument. We don't have to do this in
     *  system generation mode, and it confuses the output if we do.
     */
    
    if (!sysgenMode) {
       for (i = 0; i < sizeof(Instruments) / sizeof(Instruments[0]); ++i) {
           (*Instruments[i].InitRoutine) (&Instruments[i].Info);
       }
    }
       
/*
 *  Initialise Tcl/Tk and run the program. From here on, Tcl/Tk takes
 *  control, and we only get back into C when ConfAddCommands() is called
 *  to define the C-implemented Tcl commands, and when one of those
 *  commands is invoked.
 */
    if (batchMode) {
       Tcl_Main(argc + 1, newArgv, ConfAddCommands);
    } else {
       Tk_Main(argc + 1, newArgv, ConfAddCommands);
    }

/*
 *  The Tk documentation says the above never returns, but just in case,
 *  do our tidying up.
 */
    free(newArgv);
    ErsStop(&status);

    return (0);
}

/*  ------------------------------------------------------------------------- */

/*
 *                        T c l  C o m m a n d s
 *
 *   The rest of the code in this file provides the implementation of the
 *   various new Tcl commands defined in ConfAddCommands().
 */

/*  ------------------------------------------------------------------------- */

/*
 *                       I n i t i a l i s e { }
 */
 
static int ConfInitialise(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])

/*+
 *  Command Name:
 *    Initialise
 *
 *  Function:
 *    Initialise configuration algorithms
 *
 *  Call:
 *    Initialise overrideDir defaultDir instrument field x y
 *
 *  Description:
 *    Initialise the configuration algorithm routines by calling selecting
 *    the instrument, the initial field and calling ConfigInitialise
 *
 *    Instrument details are read at this time.  You cannot change
 *    the instrument after this call.
 *
 *    The field details are read at this time, but this may be changed
 *
 *
 *  Parameters:
 *    (>)  instrument  (string)  Name of the instrument to enable
 *                               If supplied as an empty string, the first 
 *                               instrument known by the software is used.
 *    (>)  field      (int)      The initial plate/field to configure for.
 *    (>)  x          (int)      Display x dimension.
 *    (>)  y          (int)      Display y dimension.
 *
 *
 *   Support:
 *     Jeremy Bailey, AAO
 *     Tony Farrell, AAO
 *
 *   Version Date:
 *     15-Mar-2000    
 *-
 */
{

    StatusType status = STATUS__OK;
    unsigned InstrumentIndex = 0;
    int field;
    int x;
    int y;

    if (argc != 5) {
        sprintf(interp->result, "%s:Wrong number of arguments", argv[0]);
        return TCL_ERROR;
    }

/*
 *  Parse the integer arguments.
 */
    if (Tcl_GetInt(interp, argv[2], &field) != TCL_OK) {
        return TCL_ERROR;
    }
    if (Tcl_GetInt(interp, argv[3], &x) != TCL_OK) {
        return TCL_ERROR;
    }
    if (Tcl_GetInt(interp, argv[4], &y) != TCL_OK) {
        return TCL_ERROR;
    }

/*
 *  If an instrument name has been specified, attempt to find that instrument.
 */
    if (strlen(argv[1]) != 0) {
        register unsigned i;
        for (i = 0; i < sizeof(Instruments) / sizeof(Instruments[0]); ++i) {
            if (strcasecmp(argv[1], 
                                  FpilGetInstName(Instruments[i].Info)) == 0) {
                InstrumentIndex = i;
                break;
            }
        }
        if (i >= sizeof(Instruments) / sizeof(Instruments[0])) {
            sprintf(interp->result,
                "%s:Instrument \"%s\" unknown", argv[0], argv[1]);
            return TCL_ERROR;
        }
    }

/*
 *  Set up the instrument
 */
    Instrument = &Instruments[InstrumentIndex].Info;

    printf("Configuring fibre instrument \"%s\" for telescope \"%s\"\n",
        FpilGetInstName(*Instrument), FpilGetTelescope(*Instrument));

    FpilConsAccess(*Instrument, ConfigFilesDir, ExecutableDir,
        ConfRptConstFile, 0, &InstConstDetails, &status);

/*
 *  Modify the humidity and pressure defaults, now that we know the instrument.
 *  The VLT is in a somehat drier and higher location than are 2dF and 6dF.
 */
    if (!strcmp(FpilGetInstName(*Instrument), "FLAMES")) {
       ghumid = 0.2;
       gpress = 742;
    }

/*
 *  Validate initial value number.
 */
    if (status == STATUS__OK) {
        CurrentField = field;
        if (CurrentField >= FpilGetNumFields(*Instrument)) {
            sprintf(interp->result,
                "%s:Field %d not valid, %s has only %d field plates\n",
                argv[0],
                CurrentField,
                FpilGetInstName(*Instrument), FpilGetNumFields(*Instrument));
            return TCL_ERROR;
        } else {
            const char *const *ParamNames;
            const char *const *ParamDescrs;
            const double *ParamDefaults;
            const char *const *SpecNames;
            register unsigned i;

/*
 *          Field number valid.  Access details and read telescope model
 *          details for this field.
 */
            FpilConFieldId(InstConstDetails, CurrentField, &InstFieldDetails,
                &status);
            printf("Initial field plate is %d\n", CurrentField);
            
/*          At this point, if the -use_all_fibres command line flag was given
 *          and allowed, we enable all the fibres, irrespective of the actual
 *          inUse flag values. This code is rather nasty and low-level, but
 *          it sets all the inUse flags for all the fields.
 */
            if (EnableAllFibres) {
            
               int NumFields;         /* Number of fields in structure */
               int Field;             /* Loop index through fields */
               SdsIdType FieldID;     /* Sds id for field structure */
               SdsIdType inUseID;     /* Sds id for inUse array */
               char ItemName[20];     /* Name of item in structure - ignored */
               SdsCodeType Code;      /* Type code for item - ignored */
               unsigned long Dims[10];/* Dimensions of structure object */
               long NDims;            /* Number of dimensions for object */
               unsigned long Length;  /* Size of structure item in bytes */
               void *VPointer;        /* Receives pointer to array data */
               short *SPtr;           /* Start of inUse flags */
               int NPiv;              /* # of pivots - ie size of inUse array */
               int I;                 /* Loop index through inUse flags */
               
               NumFields = FpilGetNumFields(*Instrument);
               for (Field = 0; Field <  NumFields; Field++) {
                  FpilConFieldId(InstConstDetails, Field, &FieldID, &status);
                  SdsFind (FieldID,"inUse",&inUseID,&status);
                  SdsInfo (inUseID,ItemName,&Code,&NDims,Dims,&status);
                  SdsPointer (inUseID,&VPointer,&Length,&status);
                  NPiv = Dims[0];
                  SPtr = (short *) VPointer;
                  for (I = 0; I < NPiv; I++) {
                     *SPtr++ = 1;
                  }
                  SdsFreeId (inUseID,&status);
                  SdsFreeId (FieldID,&status);
               }
            }

/*          Allocate space and initialise the flags which indicate if
 *          given fibre types are to be used, and the structure that
 *          keeps the allocation statistics for each fibre type.
 */
            FpilConSpecInfo(*Instrument, &InstNumFibreTypes, &SpecNames,
                &status);
            UseFibreType = malloc((InstNumFibreTypes) * sizeof(*UseFibreType));
            FibreTypeStats = malloc((InstNumFibreTypes) * 
                                                  sizeof(FibreStatStruct));
            for (i = 0; i < InstNumFibreTypes; ++i) {
                UseFibreType[i] = 1;
                FibreTypeStats[i].Name[0] = '\0';
                FibreTypeStats[i].Fibres = 0;
                FibreTypeStats[i].AllocToSky = 0;
                FibreTypeStats[i].AllocToGuide = 0;
                FibreTypeStats[i].AllocToObject = 0;
                FibreTypeStats[i].Unallocated = 0;
                FibreTypeStats[i].FibreSkyCount = -1;
            }

            FpilModelAccess(*Instrument, ConfigFilesDir, ExecutableDir,
                ConfRptTelModelFile, 0, CurrentField, &TelModel, &status);

/*
 *          Get details of the telescope model parameters.
 */
            FpilModelParams(TelModel, &TelModelNumParams,
                &ParamNames, &ParamDescrs, &ParamDefaults, &status);

            if (status == STATUS__OK) {
/*
 *              Malloc two arrays to contain the parameter values and copy
 *              in the default parameter values. One copy is used for the
 *              current parameter values, one to record the values used
 *              the last time the XY-conversion of positions was performed
 *              (See ConvertXY).
 */
                if (TelModelNumParams) {
                    TelModelParams = malloc(sizeof(double) * TelModelNumParams);
                    TelModelParamsUsed = 
                                    malloc(sizeof(double) * TelModelNumParams);
                    for (i = 0; i < TelModelNumParams; ++i) {
                        TelModelParams[i] = ParamDefaults[i];
                        TelModelParamsUsed[i] = ParamDefaults[i];
/*
                        printf("Parameter %d, name %s, descr %s, default %g\n",
                               i, ParamNames[i], ParamDescrs[i], 
                               ParamDefaults[i]);
*/
                    }
                } else {
                    TelModelParams = 0;
                }
            }
        }
    }

/*
 *  Work out mimic sizes.  x and y represent the size of the display.
 *  We work out the smallest of the two values and then the mimic
 *  window size is set to MIMIC_PROPORTION of this.
 */
    if (x < y) {
        MimicWindowSize = (unsigned int) ((float) x * MIMIC_PROPORTION);
    } else {
        MimicWindowSize = (unsigned int) ((float) y * MIMIC_PROPORTION);
    }
/*
 *  Now work out the position of the circle center and the size of the circle.
 */
    MimicCircleOffset = MimicWindowSize / 2;
    MimicCircleRadius = MimicCircleOffset - MIMIC_CIRCLE_OUTSIDE;
    if (!strcmp(FpilGetInstName(*Instrument), "FLAMES")) {
        MimicCircleRadius -= MIMIC_GUIDE_EXTRA;
    }
    if (!strcmp(FpilGetInstName(*Instrument), "6dF")) {
        MimicCircleRadius -= MIMIC_6DF_EXTRA;
    } else {
        /*
          Setting MimicCircleRadius using absolute pixel offsets is a really bad idea
          especially as displays get more and more pixels
          What we actually need to do is set MimicCircleRadius (in pixels) which will
          be the radius at which the 25' diameter = 12.5' radius field of view is drawn at
          such that the Guide Probe radius (~16') is inside the view...
          Imperically I find the best size to use is 15.5
        */
        MimicCircleRadius=MimicWindowSize / 2 * 12.5/15.5;
    }

/*  
 *  Initialize Config package  
 */

    ConfigInitialise(*Instrument, InstConstDetails, ConfMessage,
        interp, &status);


    if (status != STATUS__OK) {
        char mbuf[100];
        MessGetMsg(status, 0, sizeof(mbuf), mbuf);
        sprintf(interp->result, "%s:Initialise error -\n", mbuf);
        return TCL_ERROR;
    }
/*  
 *  Ers had not yet been started, so any previous errors will goto
 *  to stderr.  We are now ready to start ERS if status is ok.
 *  Initialise ERS.  From now on, ERS errors will goto the GUI.
 */
    ErsStart(ErsHandler, interp, 0, 0, &status);

    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*              F i e l d  P l a t e  T o  D i s p  X / Y
 *
 *  Functions to convert a field plate position in x/y in microns to
 *  the corresponding x/y value on the display, taking account of 
 *  the zoom factor. Note that the function signature must match
 *  the FpilCvtRoutineType defined in fpil.h.
 */
 
static int FieldPlateToDispX(void *clientData DUNUSED,
    double Microns, double Zoom)
{
    double Result;
    double circleOffset = MimicCircleOffset; /* Just to convert int to double */
    double scale = (double) MimicCircleRadius /
        (double) FpilGetFieldRadiusIncPark(*Instrument);

    Result = (scale * Microns + circleOffset) * Zoom + 0.5;
    return ((int) Result);
}

static int FieldPlateToDispY(void *clientData DUNUSED,
    double Microns, double Zoom)
{
    double Result;
    double circleOffset = MimicCircleOffset; /* Just to convert int to double */
    double scale = (double) MimicCircleRadius /
        (double) FpilGetFieldRadiusIncPark(*Instrument);

    Result = (-1.0 * scale * Microns + circleOffset) * Zoom + 0.5;
    return ((int) Result);
}

static int FieldPlateToDispX_float(void *clientData DUNUSED,
    double Microns, double Zoom)
{
    double Result;
    double circleOffset = MimicCircleOffset; /* Just to convert int to double */
    double scale = (double) MimicCircleRadius /
        (double) FpilGetFieldRadiusIncPark(*Instrument);

    Result = (scale * Microns + circleOffset) * Zoom + 0.5;
    return ((double) Result);
}

static int FieldPlateToDispY_float(void *clientData DUNUSED,
    double Microns, double Zoom)
{
    double Result;
    double circleOffset = MimicCircleOffset; /* Just to convert int to double */
    double scale = (double) MimicCircleRadius /
        (double) FpilGetFieldRadiusIncPark(*Instrument);

    Result = (-1.0 * scale * Microns + circleOffset) * Zoom + 0.5;
    return ((double) Result);
}
/*  ------------------------------------------------------------------------- */

/*              D i s p  T o  F i e l d  P l a t e  X / Y
 *
 *  These are the reverse of the previous conversions. Note that these
 *  are not a mirror image of the previous routines, since these have an
 *  offset value in their parameters. This just happens to be more convenient
 *  given the way the rest of the code uses these routines.
 */

static int DispToFieldPlateX(double Pixels, /* Pixel units to convert */
    double Offset,           /* Offset of window in pixels (scroll position) */
    double Zoom)             /* Current zoom factor */
{ 
    double Result;
    double circleOffset = MimicCircleOffset - Offset;
    double scale = (double) FpilGetFieldRadiusIncPark(*Instrument) /
        (double) MimicCircleRadius;

    Result = ((Pixels / Zoom) - circleOffset) * scale + 0.5;
    return ((int) Result);
}

static int DispToFieldPlateY(double Pixels, /* Pixel units to convert */
    double Offset,           /* Offset of window in pixels (scroll position) */
    double Zoom)             /* Current zoom factor */
{
    double Result;
    double circleOffset = MimicCircleOffset - Offset;
    double scale = (double) FpilGetFieldRadiusIncPark(*Instrument) /
        (double) MimicCircleRadius;

    Result = (circleOffset - (Pixels / Zoom)) * scale + 0.5;
    return ((int) Result);
}

/*  ------------------------------------------------------------------------- */

/*                          C h a n g e  M j d
 *
 *  Invoked to set a new configuration MJD.  This involves changing the
 *  apparent position of the field center, so is convenient to have
 *  packaged as a separate routine.
 */
 
static void ChangeMjd (SdsIdType fieldId, double mjd, StatusType *status)
{
    double cenRa, cenDec;
    double appRa, appDec;

    if (*status != STATUS__OK) return;
    
    /*
     * Get the field centre mean position
     */
     
    ArgGetd(fieldId, "cenRa",  &cenRa,  status);
    ArgGetd(fieldId, "cenDec", &cenDec, status);

    /*
     * Convert the field centre mean position to apparent.
     */
     
    slaMap(cenRa, cenDec, 0 ,0,0,0,2000,  mjd, &appRa, &appDec);

    /*
     * Put all the new values in the SDS structure.
     */
     
    ArgPutd(fieldId, "configMjd", mjd,    status);
    ArgPutd(fieldId, "appRa",     appRa,  status);
    ArgPutd(fieldId, "appDec",    appDec, status);

}
 
/*  ------------------------------------------------------------------------- */

/*
 *                     I n s t r u m e n t s  { }
 */
 
static int ConfInstruments(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc DUNUSED, char *argv[]DUNUSED)


/*+
 *  Command Name:
 *    Instruments
 *
 *  Function:
 *    Returns the list of supported instruments.
 *
 *  Call:
 *   Instruments
 *
 *  Description:
 *   Returns the list of supported fibre positioner instrumetns.
 *
 *
 *  Parameters: none
 *
 *
 *   Support:
 *     Tony Farrell, AAO
 *
 *   Version Date:
 *     13-Mar-2000    
 *-
 */
{

    register unsigned i;

/*
 *  We assume there is at least one instrument.
 */
    Tcl_AppendResult(interp, FpilGetInstName(Instruments[0].Info), 0);

    for (i = 1; i < sizeof(Instruments) / sizeof(Instruments[0]); ++i) {
/*
 *      Append a space and then the next instrument name.
 */
        Tcl_AppendResult(interp, " ", FpilGetInstName(Instruments[i].Info), 0);
    }
    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*
 *                     I n s t r u m e n t  { }
 */
 
static int ConfInstrument(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc DUNUSED, char *argv[]DUNUSED)


/*+
 *  Command Name:
 *    Instrument
 *
 *  Function:
 *    Returns the name of the current instrument.
 *
 *  Call:
 *   Instrument
 *
 *  Description:
 *   Returns the name of the instrument being configure.
 *
 *
 *  Parameters: none
 *
 *
 *   Support:
 *     Tony Farrell, AAO
 *
 *   Version Date:
 *     13-Mar-2000    
 *-
 */
 
{

    if (Instrument == 0) {
        sprintf(interp->result, "%s:Configure not yet initialised", argv[0]);
        return TCL_ERROR;
    }
    sprintf(interp->result, "%s", FpilGetInstName(*Instrument));

    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*
 *                         T e l e s c o p e  { }
 */

static int ConfTelescope(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc DUNUSED, char *argv[]DUNUSED)


/*+
 *  Command Name:
 *    Telescope
 *
 *  Function:
 *    Returns the name of the Telescope..
 *
 *  Call:
 *   Telescope
 *
 *  Description:
 *   Returns the name of the Telescope for the instrument being configured.
 *
 *
 *  Parameters: none
 *
 *
 *   Support:
 *     Tony Farrell, AAO
 *
 *   Version Date:
 *     13-Mar-2000    
 *-
 */
{

    if (Instrument == 0) {
        sprintf(interp->result, "%s:Configure not yet initialised", argv[0]);
        return TCL_ERROR;
    }
    sprintf(interp->result, "%s", FpilGetTelescope(*Instrument));

    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*
 *                      N u m  P i v o t s  { }
 */

static int ConfNumPivots(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc DUNUSED, char *argv[]DUNUSED)


/*+
 *  Command Name:
 *    NumPivots
 *
 *  Function:
 *    Returns the number of pivots in the current instrument.
 *
 *  Call:
 *   NumPivots
 *
 *  Description:
 *   Returns the number of pivots in the current fibre instrument
 *
 *
 *  Parameters: none
 *
 *
 *   Support:
 *     Tony Farrell, AAO
 *
 *   Version Date:
 *     15-Mar-2000    
 *-
 */
{

    if (Instrument == 0) {
        sprintf(interp->result, "%s:Configure not yet initialised", argv[0]);
        return TCL_ERROR;
    }
    sprintf(interp->result, "%d", FpilGetNumPivots(*Instrument));

    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*
 *                      N u m  F i e l d s  { }
 */

static int ConfNumFields(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc DUNUSED, char *argv[]DUNUSED)


/*+
 *  Command Name:
 *    NumFields
 *
 *  Function:
 *    Returns the number of field plates in the current instrument.
 *
 *  Call:
 *    NumPivots
 *
 *  Description:
 *    Returns the number of field plates in the current fibre instrument
 *
 *  Parameters: none
 *
 *   Support:
 *     Tony Farrell, AAO
 *
 *   Version Date:
 *     15-Mar-2000    
 *-
 */
{

    if (Instrument == 0) {
        sprintf(interp->result, "%s:Configure not yet initialised", argv[0]);
        return TCL_ERROR;
    }
    sprintf(interp->result, "%d", FpilGetNumFields(*Instrument));

    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*
 *                      C l a s s i f y  F i l e  { }
 */

static int ConfClassifyFile (ClientData clientData DUNUSED,
                      Tcl_Interp * interp, int argc, char *argv[])

/*+
 *  Command Name:
 *     ClassifyFile
 *
 *  Function:
 *     Attempts to work out what sort of a file has been specified.
 *
 *  Call:
 *     ClassifyFile Filename
 *
 *  Description:
 *     Returns one of: "PAF", "SDS", "FLD" or "UNKNOWN"
 *
 *  Parameters: 
 *     Filename  (string) Full name of the file in question. 
 *    
 *  Support:
 *     Keith Shortridge, AAO
 *
 *  Version Date:
 *     17-May-2001    
 *-
 */
{
   FILE* theFile;
   char DataLine[256];
   int LineCount;
   char* FileType;
   char theChar;
   char* DotPtr;
   char* FilePtr;
   char* LinePtr;
   char* FoldPtr;
   short Known;

   if (argc != 2) {
       interp->result = "wrong # args";
       return TCL_ERROR;
   }

   Known = FALSE;
   FileType = "UNKNOWN";
   
   /*  If the file exists, we try to look at its contents rather than
    *  rely on the file name.
    */
    
   if (access(argv[1], R_OK) == 0) {
   
      if ((theFile = fopen(argv[1], "r")) == NULL) {
          sprintf(interp->result, "Cannot Open file %s", argv[1]);
          return TCL_ERROR;
      }

      LineCount = 0;

      /*  We look at the first 10 non-comment lines in the file.  By then
       *  we ought to have found either a "LABEL" line in an ASCII field file,
       *  or a "PAF.HDS.START" line in a PAF file, or something unprintable,
       *  in which case we assume an SDS file.
       */

      while (LineCount < 10) {

         if (fgets(DataLine,sizeof(DataLine),theFile) == NULL) break;

         /*  Look for the first character that isn't a blank.  If we hit an 
          *  end of line, then go on to the next line.
          */

         LinePtr = DataLine;
         while ((theChar = *LinePtr) != 0) {
            if (theChar == '\n') break;
            if (theChar == ' ') {
               LinePtr++;
               continue;
            }
            if (!isprint((int)theChar)) {

               /*  We've got a non-printable character. This can't be a
                *  PAF and standard ASCII file. It's a binary file, so we
                *  assume it's an SDS file.
                */

               FileType = "SDS";
               Known = TRUE;
               break;

            } else {

               /*  We've got a non-space, printable character. If this is
                *  a '#' we're looking at a comment in a PAF file. If it's
                *  a '*' we're looking at a comment in an ASCII field file.
                *  We don't judge the file on that basis, however. We want a
                *  real line to look at.
                */

               if ((theChar == '#') || (theChar == '*')) break;

               /*  We've now got the start of a non-comment line. Look
                *  for a characteristic line for a file. First we fold the
                *  rest of the line to upper case.
                */

               FoldPtr = LinePtr;
               while ((theChar = *FoldPtr) != 0) {
                  if (!isprint((int)theChar)) break;
                  if (islower((int)theChar)) *FoldPtr = toupper(theChar);
                  FoldPtr++;
               } 
               if (!strncmp(LinePtr,"LABEL",5)) {
                  FileType = "FLD";
                  Known = TRUE;
                  break;
               } else if (!strncmp(LinePtr,"PAF.HDR.START",13)) {
                  FileType = "PAF";
                  Known = TRUE;
                  break;
               } else {
                  LineCount++;
                  break;
               }
            }
            LinePtr++;
         }
         if (Known) break;
      }
      fclose(theFile);
   }
   
   /*  If after that, we don't have a classification - which will be the
    *  case if the file didn't exist - we fall back on looking at the
    *  file extension. Find the last '.' in the filename, and look at
    *  what follows.
    */
   
   if (!Known) {
      DotPtr = (char*) NULL;
      FilePtr = argv[1];
      while ((theChar = *FilePtr) != 0) {
         if (theChar == '.') DotPtr = FilePtr;
         FilePtr++;
      }
      if (DotPtr) {
         if (!strcmp(DotPtr,".sds")) {
            FileType = "SDS";
         } else if (!strcmp(DotPtr,".paf")) {
            FileType = "PAF";
         } else if (!strcmp(DotPtr,".ins")) {
            FileType = "PAF";
         } else if (!strcmp(DotPtr,".fld")) {
            FileType = "FLD";
         }
      }
   }

   strcpy(interp->result,FileType);

   return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*
 *                      T a r g e t  D e s c r  { }
 */

static int ConfTargetDescr (ClientData clientData DUNUSED,
                      Tcl_Interp * interp, int argc, char *argv[])

/*+
 *  Command Name:
 *     TargetDescr
 *
 *  Function:
 *     Given a target type code character, returns a fuller description.
 *
 *  Description:
 *     This routine is passed the single character code used to classify a
 *     target object, and returns the longer text description of the
 *     target type. For example, 'M' will return 'Medusa target'.
 *
 *  Call:
 *     TargetDescr Type
 *
 *  Parameters: 
 *     Type  (char) Single character giving the object type. 
 *
 *  Returns:
 *     Description of object type
 *   
 *  Support:
 *     Keith Shortridge, AAO
 *
 *  Version Date:
 *     30-Oct-2001    
 *-
 */
{
    char TargetString[32];
    char TargetText[128];
    char TargetType;

    if (argc != 2) {
        interp->result = "wrong # args";
        return TCL_ERROR;
    }
    if (Instrument == 0) {
        sprintf(interp->result, "%s:Configure not yet initialised", argv[0]);
        return TCL_ERROR;
    }

    TargetType = argv[1][0];
    if (FpilDecodeTargetType(*Instrument,TargetType,' ',0,TargetString,
                                                                TargetText)) {

       strcpy(interp->result,TargetText);
    } else {
       strcpy(interp->result,"Unknown type");
    }
    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*
 *                      F i b r e  T y p e s  { }
 */

static int ConfFibreTypes(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc DUNUSED, char *argv[]DUNUSED)

/*+
 *  Command Name:
 *    FibreTypes
 *
 *  Function:
 *    Returns the fibre type names.
 *
 *  Call:
 *   FibreTypes
 *
 *  Description:
 *   Returns a list giving the names of the the fibre types supported
 *   by the instrument.
 *
 *  Parameters: none
 *
 *   Support:
 *     Tony Farrell, AAO
 *
 *   Version Date:
 *     15-Mar-2000    
 *-
 */
{

    unsigned int NumFibreTypes;
    const char *const *FibreTypeNames;
    StatusType status = STATUS__OK;
    register unsigned i;
    Tcl_DString Names;

    if (Instrument == 0) {
        sprintf(interp->result, "%s:Configure not yet initialised", argv[0]);
        return TCL_ERROR;
    }

    FpilConSpecInfo(*Instrument, &NumFibreTypes, &FibreTypeNames, &status);
    if (status != STATUS__OK) {
        char mbuf[100];
        MessGetMsg(status, 0, sizeof(mbuf), mbuf);
        sprintf(interp->result,
            "%s:Error getting fibre type details - %s\n", argv[0], mbuf);
        return TCL_ERROR;
    }

    Tcl_DStringInit(&Names);
    for (i = 0; i < NumFibreTypes; ++i) {
        Tcl_DStringAppendElement(&Names, FibreTypeNames[i]);
    }
    Tcl_DStringResult(interp, &Names);
    Tcl_DStringFree(&Names);

    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*
 *                     T a r g e t  C r i t e r i a  { }
 */

static int ConfTargetCriteria(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc DUNUSED, char *argv[]DUNUSED)

/*+
 *  Command Name:
 *    TargetCriteria
 *
 *  Function:
 *    Returns the target selection criteria.
 *
 *  Call:
 *    TargetCriteria
 *
 *  Description:
 *   Returns a list giving the names of the the target selection criteria
 *   supported by the instrument.
 *
 *
 *  Parameters: none
 *
 *
 *   Support:
 *     Keith Shortridge, AAO
 *
 *   Version Date:
 *     21-Aug-2000    
 *-
 */
{

    unsigned int NumCriteria;
    const char *const *TargetCriteria;
    register unsigned i;
    Tcl_DString Names;

    if (Instrument == 0) {
        sprintf(interp->result, "%s:Configure not yet initialised", argv[0]);
        return TCL_ERROR;
    }

    FpilTargetCriteria(*Instrument, &NumCriteria, &TargetCriteria);

    Tcl_DStringInit(&Names);
    for (i = 0; i < NumCriteria; ++i) {
        Tcl_DStringAppendElement(&Names, TargetCriteria[i]);
    }
    Tcl_DStringResult(interp, &Names);
    Tcl_DStringFree(&Names);

    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*
 *                     F i b r e  C o m b o s  { }
 */

static int ConfFibreCombos(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc DUNUSED, char *argv[]DUNUSED)

/*+
 *  Command Name:
 *    FibreCombos
 *
 *  Function:
 *    Returns the set of possible fibre combinations.
 *
 *  Call:
 *    FibreCombos
 *
 *  Description:
 *    Returns a list giving the names of the the possible fibre combinations
 *    supported by the instrument.
 *
 *  Parameters: none
 *
 *   Support:
 *     Keith Shortridge, AAO
 *
 *   Version Date:
 *     05-Feb-2001
 *-
 */
{

    int NumCombos;
    char* ComboNames[16];
    char* ModeNames[16];
    register unsigned i;
    Tcl_DString Names;

    if (Instrument == 0) {
        sprintf(interp->result, "%s:Configure not yet initialised", argv[0]);
        return TCL_ERROR;
    }

    FpilFibreCombos(*Instrument, sizeof(ComboNames)/sizeof(char*),
                                          &NumCombos, ComboNames, ModeNames);

    Tcl_DStringInit(&Names);
    for (i = 0; i < NumCombos; ++i) {
        Tcl_DStringAppendElement(&Names, ComboNames[i]);
    }
    Tcl_DStringResult(interp, &Names);
    Tcl_DStringFree(&Names);

    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*
 *                  F i b r e  C o m b o  M o d e  { }
 */

static int ConfFibreComboMode(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])

/*+
 *  Command Name:
 *    FibreComboMode
 *
 *  Function:
 *    Returns the set of possible fibre combinations.
 *
 *  Call:
 *    FibreComboMode ComboId
 *
 *  Description:
 *   Returns a string giving the mode keyword associated with a
 *   given fibre combination.
 *
 *  Parameters:
 *     ComboId   (int) Index number of the combination in question. This 
 *                     should start from zero and should be the index into
 *                     the list of combinations returned by FibreCombos.
 *
 *   Support:
 *     Keith Shortridge, AAO
 *
 *   Version Date:
 *     08-Apr-2001
 *-
 */
{
    int NumCombos;
    int ComboId;
    char* ComboNames[16];
    char* ModeNames[16];

    if (Instrument == 0) {
        sprintf(interp->result, "%s:Configure not yet initialised", argv[0]);
        return TCL_ERROR;
    }
    if (argc != 2) {
        interp->result = "wrong # args";
        return TCL_ERROR;
    }
    ComboId = (atol(argv[1]));
    
    FpilFibreCombos(*Instrument, sizeof(ComboNames)/sizeof(char*),
                                          &NumCombos, ComboNames, ModeNames);
                                          
    if ((ComboId < 0) || (ComboId >= NumCombos)) {
        interp->result = "invalid combination index";
        return TCL_ERROR;
    }
    strcpy(interp->result,ModeNames[ComboId]);
    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*
 *                  F i b r e  C o m b o  T y p e s  { }
 */

static int ConfFibreComboTypes(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])

/*+
 *  Command Name:
 *    FibreComboTypes
 *
 *  Function:
 *    Returns the set of fibre types included in a given combination.
 *
 *  Call:
 *    FibreComboTypes ComboId
 *
 *  Description:
 *    Returns a list giving the fibre type codes for each fibre type included
 *    in the specified combination.
 *
 *  Parameters:
 *     ComboId   (int) Index number of the combination in question. This 
 *                     should start from zero and should be the index into
 *                     the list of combinations returned by FibreCombos.
 *
 *   Support:
 *     Keith Shortridge, AAO
 *
 *   Version Date:
 *     06-Feb-2001
 *-
 */
{

    int ComboId;
    int TypeCodes[16];
    int NumCodes;
    register unsigned i;
    Tcl_DString Codes;

    if (Instrument == 0) {
        sprintf(interp->result, "%s:Configure not yet initialised", argv[0]);
        return TCL_ERROR;
    }
    if (argc != 2) {
        interp->result = "wrong # args";
        return TCL_ERROR;
    }
    ComboId = (atol(argv[1]));

    FpilComboFibreCodes(*Instrument, ComboId,sizeof(TypeCodes)/sizeof(int),
                                                      &NumCodes, TypeCodes);

    Tcl_DStringInit(&Codes);
    for (i = 0; i < NumCodes; ++i) {
        char CodeString[16];
        sprintf (CodeString,"%d",TypeCodes[i]);
        Tcl_DStringAppendElement(&Codes, CodeString);
    }
    Tcl_DStringResult(interp, &Codes);
    Tcl_DStringFree(&Codes);

    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*
 *                  S e t  I m p o r t  O n l y  M o d e  { }
 */

static int ConfSetImportOnlyMode(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])
    
/*+
 *  Command Name:
 *    SetImportOnlyMode
 *
 *  Function:
 *    Sets a flag to suppress errors in 'import only' mode.
 *
 *  Call:
 *    SetImportOnlyMode Flag
 *
 *  Description:
 *    Sets a flag used by the allocation algorithm to suppress errors when
 *    used in 'import only' mode
 *
 *  Parameters:
 *     Flag   (int) Flag value to be used.
 *-
 */
{
    int value;
    if (argc != 2) {
        interp->result = "Wrong # args";
        return TCL_ERROR;
    }
    value = atoi(argv[1]);
    interp->result = "OK";
    ImportOnlyMode = value;
    ConfigSetImportOnlyMode(value);
    return TCL_OK;
}


/*  ------------------------------------------------------------------------- */

/*                      F r e e  A n d  F i n d
 *
 * This function does an Sds find, but before doing so, free's any old
 * SDS id being pointed to by the variable. 
 */
 
static void FreeAndFind(const SdsIdType source, const char *const name,
    SdsIdType * const id, StatusType * const status)
{
    if (*status != STATUS__OK)
        return;

    if (*id)
        SdsFreeId(*id, status);
    if (*status != STATUS__OK) {
        ErsRep(0, status, "Error freeing id (%ld) for item \"%s\"",
            (long int) *id, name);
    } else {
        *id = 0;
        SdsFind(source, name, id, status);
        if (*status != STATUS__OK)
            ErsRep(0, status, "Error finding SDS item \"%s\"", name);
    }
}

/*  ------------------------------------------------------------------------- */

/*
 *                      S t a r t  U p d a t e  { }
 */

static int ConfStartUpdate(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])

/*+
 *  Command Name:
 *    StartUpdate
 *
 *  Function:
 *    Start an update of the mimic display fibre information
 *
 *  Call:
 *    StartUpdate id pivid
 *
 *  Description:
 *    Called at the start of a mimic display update to get the Sds Ids
 *    of the various components - these are put in global variables
 *    for subsequent use by commands such as GetButton.
 *
 *  Parameters:
 *    (>)  id    (Sds Id)  Sds identifier of the structure
 *    (>)  pivid (Sds Id)  Sds identifier of the pivots structure
 *
 *   Support:
 *     Jeremy Bailey, AAO
 *
 *   Version Date:
 *     6-Oct-1994    
 *-
 */
{
    StatusType status;
    SdsIdType id;
    char mbuf[100];

    if (argc != 3) {
        interp->result = "wrong # args";
        return TCL_ERROR;
    }
    status = SDS__OK;
    id = (SdsIdType) (atol(argv[1]));
    FreeAndFind(id, "objects", &gtopid, &status);
    gpivotid = (SdsIdType) (atol(argv[2]));
    FreeAndFind(gpivotid, "xPiv", &gxPivid, &status);
    FreeAndFind(gpivotid, "yPiv", &gyPivid, &status);
    FreeAndFind(gtopid, "x", &gxid, &status);
    FreeAndFind(gpivotid, "xPark", &gxpid, &status);
    FreeAndFind(gpivotid, "yPark", &gypid, &status);
    FreeAndFind(gpivotid, "tPark", &gthetapid, &status);
    FreeAndFind(gtopid, "y", &gyid, &status);
    FreeAndFind(gtopid, "theta", &gthetaid, &status);
    FreeAndFind(gtopid, "type", &gtype, &status);

    if (status != SDS__OK) {
        MessGetMsg(status, 0, sizeof(mbuf), mbuf);
        ErsFlush(&status);
        sprintf(interp->result, "Error Starting Update\n%s", mbuf);
        return TCL_ERROR;
    } else
        return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*
 *                       G e t  B u t t o n  { }
 */

static int ConfGetButton(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])

/*+
 *
 *  Command name:
 *    GetButton
 *
 *
 *  Function:
 *    Get button information for mimic display
 *  
 *  Call:
 *    GetButton id pivid fibre index zoom button_graphic
 *
 *  Description:
 *    Return the information needed to draw a fibre and its button
 *    on the mimic display.
 *   
 *    This version does not use the id and pivid arguments, but rather
 *    uses ids previously set up with a call to StartUpdate.
 *
 *   Parameters:
 *
 *    (>) id      (Sds Id)  Sds identifier of the structure   
 *    (>) pivid   (Sds Id)  Sds identifier of the pivots structure
 *    (>) fibre   (int)     Fibre number
 *    (>) index   (int)     Index number in structure
 *    (>) zoom    (int)     Zoom factor
 *    (<) button_graphic1 (string)    A tcl variable in which will be placed
 *                 the arguments to a Tk canvas widget create command required
 *                 to draw a graphic of the button.  Only graphics which
 *                 accept a "-fill" option can be used since the Tcl code
 *                 sets colours appropiately depending on the type of object
 *                 and if it has been selected.  For example, this may contain
 *                 something like "polygon x1 y1 x2 y2 x2 y3..."
 *                 or             "oval 10 10 80 80"
 *    (<) button_graphic2 (string) A second such variable, used for the
 *                 same purpose if two graphics are required to represent
 *                 a button.  If not required, will be set to an empty string.
 *
 *   Returns:
 *     A list of 4 values giving the x,y values of the end points of the
 *     fibre line in a form suitable for use by the Tk canvas widget
 *     line command.
 *
 *   Support:
 *     Jeremy Bailey, AAO
 *     Tony Farrell, AAO
 *
 *   Version Date:
 *     16-Mar-2000
 *-
 */
{
    StatusType status;
    long fibre_no;
    long index_no;
    int x, y;                   /* Fibre position, microns              */
    int xa, ya;                 /* Fibre pivot end coordinates, microns */
    unsigned long actlen;
    double zoom;
    double xs, ys;              /* Fibre pivot end coordinates, pixels */
    double xf, yf;              /* Fibre button end coordinates, pixels */
    double theta;               /* Button theta                        */
    char graphic1[500];
    char graphic2[500];
    char mbuf[100];
    short type;                 /* Fibre Type                           */
    double scaling = (double) MimicCircleRadius /
        (double) FpilGetFieldRadiusIncPark(*Instrument);

    if (argc != 8) {
        interp->result = "wrong # args";
        return TCL_ERROR;
    }
    status = SDS__OK;
    fibre_no = atol(argv[3]) - 1;
    index_no = atol(argv[4]) - 1;
    zoom = atof(argv[5]);

/*
 * Validate the fibre number.
 */
    if ((fibre_no < 0) || (fibre_no >= (int) FpilGetNumPivots(*Instrument))) {
        sprintf(interp->result,
            "Illegal Pivot Number %ld\nMust be between 1 and %d",
            fibre_no + 1, FpilGetNumPivots(*Instrument));
        return TCL_ERROR;
    }

/*
 * Get button details
 *     Pivot position (x and y)
 *     Pivot Type
 *     Fibre Position (x, y and theta)
 *     
 */
    SdsGet(gxPivid, 4, fibre_no, &xa, &actlen, &status);
    SdsGet(gyPivid, 4, fibre_no, &ya, &actlen, &status);
    SdsGet(gtype, 2, fibre_no, &type, &actlen, &status);

    SdsGet(gxid, 4, index_no, &x, &actlen, &status);
    SdsGet(gyid, 4, index_no, &y, &actlen, &status);
    SdsGet(gthetaid, 8, index_no, &theta, &actlen, &status);

    if (status != SDS__OK) {
        MessGetMsg(status, 0, 100, mbuf);
        sprintf(interp->result, "Error Getting Fibre Data\n%s", mbuf);
        return TCL_ERROR;
    }

/*
 * Convert pivot position in microns on plate to pixels.
 */
    xs = FieldPlateToDispX(0, xa, zoom);
    ys = FieldPlateToDispY(0, ya, zoom);


    FpilDrawButFib(*Instrument, scaling, zoom,
        FieldPlateToDispX,
        FieldPlateToDispY,
        0,
        x, y, theta, type,
        sizeof(graphic1), &xf, &yf, graphic1, graphic2, &status);

    if (status != STATUS__OK) {
        MessGetMsg(status, 0, 100, mbuf);
        sprintf(interp->result, "Error Drawing Button\n%s", mbuf);
        return TCL_ERROR;
    }
/*
 * Write the line describing the fibre ends to result string.
 */
    sprintf(interp->result, "%f %f %f %f", xs, ys, xf, yf);
/*
 * Set the variables which are to contain the graphics
 */
    Tcl_SetVar(interp, argv[6], graphic1, 0);
    Tcl_SetVar(interp, argv[7], graphic2, 0);

    return TCL_OK;

}

/*  ------------------------------------------------------------------------- */

/*
 *                  M a k e  L i s t i n g  { }
 */

static int ConfMakeListing(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])

/*+
 *  Command Name:
 *    MakeListing
 *
 *  Function:
 *    Output listing of objects and allocations
 *
 *  Call:
 *    MakeListing id file type
 *
 *  Description:
 *    Output a text file listing of the allocations or objects
 *    or both in an Sds structure. The format of the listing of
 *    unallocated objects is the same as the input format for
 *    the configuration. If the allocations are listed this listing takes
 *    the form of comments in the configuration file so the file will
 *    still be a valid configuration input file. The unalocated objects
 *    listing includes all fiducial stars and sky targets whether allocated
 *    or not.
 *
 *
 *  Parameters:
 *    (>)  id    (Sds Id)  Sds identifier of the structure
 *    (>)  file  (char)    Name of the file to be created
 *    (>)  type  (char)    Information to output
 *                   alloc  =>  List Allocations
 *                   obj    =>  List unallocated objects
 *                   all    =>  List unallocated objects and allocations
 *
 *   Returns:
 *     "Error" - if the file could not be opened.
 *
 *   Support:
 *     Jeremy Bailey, AAO
 *
 *   Version Date:
 *     6-Oct-1994
 *    26-Nov-2001 Now includes allocated sky targets with unallocated 
 *                objects. KS.    
 *-
 */
{
    SdsIdType topid, objid, uid, fid;
    StatusType status;
    short alloc_array;
    FILE *listing;
    double mjd;
    int iy, im, id;
    double fd;
    int jstat;
    double ra, dec;
    int rhmsf[4];
    int ddmsf[4];
    char sign[1];
    char buf[80];
    char iname[16];
    SdsCodeType code;
    long ndims;
    unsigned long dims[7];
    SdsIdType ra_id, dec_id, name_id, type_id, spect_id, prio_id, mag_id,
        pid_id, comment_id, temp_id;
    unsigned int i;
    char mbuf[100];
    char c, s;

    if (argc != 4) {
        interp->result = "wrong # args";
        return TCL_ERROR;
    }
    status = STATUS__OK;
    topid = (SdsIdType) (atol(argv[1]));
    SdsFind(topid, "fieldData", &fid, &status);
    listing = fopen(argv[2], "w");
    if (listing == NULL) {
        ErsOut(0, &status, "Error Opening File %s", argv[2]);
        interp->result = "Error";
        return TCL_OK;
    }
    if (Instrument == NULL) {
        ErsOut(0, &status, "No instrument selected");
        interp->result = "Error";
        return TCL_OK;
    }

/*  Output the Header Section  */

    ArgGetString(fid, "label", sizeof(buf), buf, &status);
    fprintf(listing, "LABEL %s\n", buf);
    ArgGetd(fid, "configMjd", &mjd, &status);
    slaDjcl(mjd, &iy, &im, &id, &fd, &jstat);
    fprintf(listing, "UTDATE %4.4d %2.2d %2.2d\n", iy, im, id);
    ArgGetd(fid, "cenRa", &ra, &status);
    ArgGetd(fid, "cenDec", &dec, &status);
    slaDr2tf(3, ra, sign, rhmsf);
    slaDr2af(2, dec, sign, ddmsf);
    fprintf(listing,
        "CENTRE %2.2d %2.2d %2.2d.%3.3d  %c%2.2d %2.2d %2.2d.%2.2d\n",
        rhmsf[0], rhmsf[1], rhmsf[2], rhmsf[3], sign[0], ddmsf[0], ddmsf[1],
        ddmsf[2], ddmsf[3]);
    if (strcmp(FpilGetInstName(*Instrument), "FLAMES")) {
        fprintf(listing, "EQUINOX 2000.0\n");
    }

    if ((strcmp(argv[3], "alloc") == 0) || (strcmp(argv[3], "all") == 0)) {
        SdsFind(topid, "objects", &objid, &status);
        ConfGetIds(objid, &ra_id, &dec_id, &name_id, &type_id, &spect_id,
            &prio_id, &mag_id, &pid_id, &comment_id, &status);
        SdsInfo(ra_id, iname, &code, &ndims, dims, &status);
        fprintf(listing, "*\n");
        fprintf(listing, "*  Fibre Allocations\n");
        fprintf(listing, "*\n");
        fprintf(listing, "*%5s%12s%12s%12s\n", "Fibre", "Name", "R.A.", "Dec");
        for (i = 0; i < dims[0]; i++) {
            fprintf(listing, "*%4d ", i + 1);
            ConfListLine(listing, i, ra_id, dec_id, name_id, type_id, spect_id,
                prio_id, mag_id, pid_id, comment_id, &status);
            if (status != STATUS__OK)
                break;
        }
        SdsFreeId(name_id, &status);
        SdsFreeId(ra_id, &status);
        SdsFreeId(dec_id, &status);
        SdsFreeId(type_id, &status);
        if (spect_id)
            SdsFreeId(spect_id, &status);
        SdsFreeId(prio_id, &status);
        SdsFreeId(mag_id, &status);
        SdsFreeId(pid_id, &status);
        SdsFreeId(comment_id, &status);
        SdsFreeId(objid, &status);
    }
    if ((strcmp(argv[3], "obj") == 0) || (strcmp(argv[3], "all") == 0)) {
        SdsFind(topid, "unallocObject", &uid, &status);
        ConfGetIds(uid, &ra_id, &dec_id, &name_id, &type_id, &spect_id,
            &prio_id, &mag_id, &pid_id, &comment_id, &status);
        SdsInfo(ra_id, iname, &code, &ndims, dims, &status);
        fprintf(listing, "*\n");
        fprintf(listing, "*  Unallocated Objects\n");
        fprintf(listing, "*\n");
        fprintf(listing, "*%12s%12s%12s\n", "Name", "R.A.", "Dec");
        alloc_array = TRUE;
        SdsFind (uid,"allocated",&temp_id,&status);
        if (status != 0) {
            alloc_array = FALSE;
            status = 0;
        } else {
            SdsFreeId (temp_id, &status);
        }
        for (i = 0; i < dims[0]; i++) {
            if (alloc_array) {
                ConfArrayGetc(uid, "allocated", i, &c, &status);
            } else {
                c = 0;
            }
            if (c == 0) {
                ConfListLine(listing, i, ra_id, dec_id, name_id, type_id,
                    spect_id, prio_id, mag_id, pid_id, comment_id, &status);
            }
            if (status != STATUS__OK)
                break;
        }
        SdsFreeId(name_id, &status);
        SdsFreeId(ra_id, &status);
        SdsFreeId(dec_id, &status);
        SdsFreeId(type_id, &status);
        if (spect_id)
            SdsFreeId(spect_id, &status);
        SdsFreeId(prio_id, &status);
        SdsFreeId(mag_id, &status);
        SdsFreeId(pid_id, &status);
        SdsFreeId(comment_id, &status);
        SdsFreeId(uid, &status);
    }
    if ((strcmp(argv[3], "obj") == 0) || (strcmp(argv[3], "all") == 0)) {
        SdsFind(topid, "unallocGuide", &uid, &status);
        ConfGetIds(uid, &ra_id, &dec_id, &name_id, &type_id, &spect_id,
            &prio_id, &mag_id, &pid_id, &comment_id, &status);
        SdsInfo(ra_id, iname, &code, &ndims, dims, &status);
        fprintf(listing, "*\n");
        fprintf(listing, "*  Fiducial Stars\n");
        fprintf(listing, "*\n");
        fprintf(listing, "*%12s%12s%12s\n", "Name", "R.A.", "Dec");
        for (i = 0; i < dims[0]; i++) {
            ConfListLine(listing, i, ra_id, dec_id, name_id, type_id, spect_id,
                prio_id, mag_id, pid_id, comment_id, &status);
            if (status != STATUS__OK)
                break;
        }
        SdsFreeId(name_id, &status);
        SdsFreeId(ra_id, &status);
        SdsFreeId(dec_id, &status);
        SdsFreeId(type_id, &status);
        if (spect_id)
            SdsFreeId(spect_id, &status);
        SdsFreeId(prio_id, &status);
        SdsFreeId(mag_id, &status);
        SdsFreeId(pid_id, &status);
        SdsFreeId(comment_id, &status);
        SdsFreeId(uid, &status);

/*  We also need the fiducials and skies from the allocated objects section  */

        SdsFind(topid, "objects", &objid, &status);
        ConfGetIds(objid, &ra_id, &dec_id, &name_id, &type_id, &spect_id,
            &prio_id, &mag_id, &pid_id, &comment_id, &status);
        SdsInfo(ra_id, iname, &code, &ndims, dims, &status);
        for (i = 0; i < dims[0]; i++) {
            ConfArrayGetc(objid, "type", i, &c, &status);
            ConfArrayGetc(objid, "spectrograph", i, &s, &status);
            if (FpilIsTargetGuide(*Instrument, c) ||
                                     FpilIsTargetSky(*Instrument, c)) {
                ConfListLine(listing, i, ra_id, dec_id, name_id, type_id,
                    spect_id, prio_id, mag_id, pid_id, comment_id, &status);
            }
            if (status != STATUS__OK)
                break;
        }
        SdsFreeId(name_id, &status);
        SdsFreeId(ra_id, &status);
        SdsFreeId(dec_id, &status);
        SdsFreeId(type_id, &status);
        if (spect_id)
            SdsFreeId(spect_id, &status);
        SdsFreeId(prio_id, &status);
        SdsFreeId(mag_id, &status);
        SdsFreeId(pid_id, &status);
        SdsFreeId(comment_id, &status);
        SdsFreeId(objid, &status);
    }

    SdsFreeId(fid, &status);

    fclose(listing);
    if (status == STATUS__OK) {
        return TCL_OK;
    } else {
        MessGetMsg(status, 0, 100, mbuf);
        sprintf(interp->result, "Error in MakeListing\n%s", mbuf);
        return TCL_ERROR;
    }
}

/*  ------------------------------------------------------------------------- */

/*
 *                  S k y c a t   L i s t i n g  { }
 */

static int ConfSkycatListing(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])

/*+
 *  Command Name:
 *    SkycatListing
 *
 *  Function:
 *    Output listing of allocated objects in skycat format
 *
 *  Call:
 *    SkycatListing id file
 *
 *  Description:
 *    Output a listing in skycat catalogue format of the allocated objects. 
 *
 *  Parameters:
 *    (>)  id    (Sds Id)  Sds identifier of the structure
 *    (>)  file  (char)    Name of the file to be created
 *
 *   Returns:
 *     "Error" - if the file could not be opened.
 *
 *   Support:
 *     Keith Shortridge, AAO
 *
 *   Version Date:
 *    29-Jan-2001 Original version, KS.
 *    27-Jan-2003 skycat colors and symbols updated, AKA.
 *                VLT Guide star added to the listing, AKA.
 *-
 */
{
    SdsIdType topid, objid, fid, uid;
    StatusType status;
    FILE *listing;
    double mjd;
    int iy, im, id;
    double fd;
    int jstat;
    double ra, dec;
    int rhmsf[4];
    int ddmsf[4];
    char sign[1];
    char buf[80];
    char iname[16];
    SdsCodeType code;
    long ndims;
    unsigned long dims[7];
    SdsIdType ra_id, dec_id, name_id, type_id, spect_id, prio_id, mag_id,
        pid_id, comment_id;
    unsigned int i;
    char mbuf[100];
    unsigned long actlen;
    char typeString[32];
    char typeText[80];
    char name[NAMELEN];
    char otype;
    char ospect;
    char UnallocType;
    char c;                       
    int selGuide;

    if (argc != 5) {
        interp->result = "wrong # args";
        return TCL_ERROR;
    }
    status = STATUS__OK;
    topid = (SdsIdType) (atol(argv[1]));
    SdsFind(topid, "fieldData", &fid, &status);
    listing = fopen(argv[2], "w");
    if (listing == NULL) {
        ErsOut(0, &status, "Error Opening File %s", argv[2]);
        interp->result = "Error";
        return TCL_OK;
    }
    if (Instrument == NULL) {
        ErsOut(0, &status, "No instrument selected");
        interp->result = "Error";
        return TCL_OK;
    }

    /*  Output the Header Section. First, just the header information
     *  included in the original input file about the field itself
     */

    ArgGetString(fid, "label", sizeof(buf), buf, &status);
    fprintf(listing, "#LABEL %s\n", buf);
    ArgGetd(fid, "configMjd", &mjd, &status);
    slaDjcl(mjd, &iy, &im, &id, &fd, &jstat);
    fprintf(listing, "#UTDATE %4.4d %2.2d %2.2d\n", iy, im, id);
    ArgGetd(fid, "cenRa", &ra, &status);
    ArgGetd(fid, "cenDec", &dec, &status);
    slaDr2tf(3, ra, sign, rhmsf);
    slaDr2af(2, dec, sign, ddmsf);
    fprintf(listing,
        "#CENTRE %2.2d %2.2d %2.2d.%3.3d  %c%2.2d %2.2d %2.2d.%2.2d\n",
        rhmsf[0], rhmsf[1], rhmsf[2], rhmsf[3], sign[0], ddmsf[0], ddmsf[1],
        ddmsf[2], ddmsf[3]);
        
    /*  Now the catalogue description information that skycat will use
     *  to plot the data. The important bit here is the symbol definition
     *  section, because this indicates how the targets will be displayed -
     *  and is probably the bit most people will have aesthetic opinions
     *  about!
     */
     
    fprintf(listing,"# Skycat configuration information\n");
    fprintf(listing,"serv_type: local\n");
    fprintf(listing,"long_name: %s\n",argv[2]);
    fprintf(listing,"short_name: %s\n",argv[2]);
    fprintf(listing,"url: %s\n",argv[2]);
    fprintf(listing,"%s%s\n","symbol: {Sky Ref} {circle blue {} {} {} ",
                    "{$Sky==0 && $Ref==0}} {8 {}} \\" );
    fprintf(listing,"%s\n",
                    ": {Sky Ref} {cross blue {} {} {} {$Sky==1 && $Ref==0}} {8 {}} \\" );
    fprintf(listing,"%s\n",
                    ": {Sky Ref} {square red {} {} {} {$Sky==0 && $Ref==1}} {12 {}} \\" );
    fprintf(listing,"%s\n",
                    ": {Sky Ref} {diamond red {} {} {} {$Sky==0 && $Ref==2}} {15 {}}" );
    fprintf(listing,"# End Skycat configuration information\n");
    
    /*  Now the column headings used by Skycat. Note that the Ref and Sky 
     *  column values are used in the conditions in the symbol descriptions
     *  in the configuration information that's just been written to control
     *  the way targets of different types are displayed.
     */

    fprintf(listing,"ID\tRA\tDEC\tSky\tRef\tType\n");
    fprintf(listing,"--\t--\t---\t---\t---\t----\n");

    /*  The first entry in the catalogue is the field center - this is not
     *  a real target, of course, but having it first will cause skycat to
     *  use this as the center of its field. Note that ra and dec still
     *  contain its coordinates, albeit in radians.
     */
     
    ra *= 180.0 / DPI;
    dec *= 180.0 / DPI;
    fprintf(listing,"%s\t%f\t%f\t%d\t%d\t%s\n",
                  "Field center",ra,dec,0,0,"Field center");

       
    /*  
     *  Now, get the VLT guide star (added 2003-01-26)
     */

    SdsFind(topid, "unallocGuide", &uid, &status);
    ConfGetIds(uid, &ra_id, &dec_id, &name_id, &type_id, &spect_id,
            &prio_id, &mag_id, &pid_id, &comment_id, &status);
    SdsInfo(ra_id, iname, &code, &ndims, dims, &status);
    
    /*  The selected guide star, if there is one. We identify it by
     *  its name, as passed.
     */
    
    selGuide = atoi(argv[3]);
    if (selGuide > 0) {
        char* guideName = argv[4];
        char name[NAMELEN];
        for (i = 0; i < dims[0]; i++) {
            ConfArrayGetc(uid, "type", i, &c, &status);
            if (FpilIsTargetTelGuide(*Instrument, c)) {
	        SdsGet(name_id, NAMELEN, NAMELEN * i, name, &actlen, &status);
                if (!strcmp(name,guideName)) {
		  SdsGet(ra_id, sizeof(double), i, &ra, &actlen, &status);
		  SdsGet(dec_id, sizeof(double), i, &dec, &actlen, &status);
		  ra *= 180.0 / DPI;
		  dec *= 180.0 / DPI;
		  SdsGet(type_id, sizeof(char), i, &otype, &actlen, &status);
		  if (spect_id) {
		      SdsGet(type_id, sizeof(char), i, &ospect, &actlen, &status);
		  } else {
		      ospect = 0;
		  }
		  FpilDecodeTargetType(*Instrument, otype, ospect, TRUE, typeString,
				     typeText);
		  fprintf(listing,"%s\t%f\t%f\t%d\t%d\t%s\n",
			  name,ra,dec,0,2,typeText);
		  break;
                }
            }
        }
    }
    
       
    /*  Now, we work through all the allocated objects, and output their
     *  details to the catalogue.
     */
     
    UnallocType = FpilUnallocatedType(*Instrument);
    SdsFind(topid, "objects", &objid, &status);
    ConfGetIds(objid, &ra_id, &dec_id, &name_id, &type_id, &spect_id,
            &prio_id, &mag_id, &pid_id, &comment_id, &status);
    SdsInfo(ra_id, iname, &code, &ndims, dims, &status);
    for (i = 0; i < dims[0]; i++) {
        SdsGet(name_id, NAMELEN, NAMELEN * i, name, &actlen, &status);
        SdsGet(ra_id, sizeof(double), i, &ra, &actlen, &status);
        SdsGet(dec_id, sizeof(double), i, &dec, &actlen, &status);
        ra *= 180.0 / DPI;
        dec *= 180.0 / DPI;
        SdsGet(type_id, sizeof(char), i, &otype, &actlen, &status);
        if (spect_id) {
            SdsGet(type_id, sizeof(char), i, &ospect, &actlen, &status);
        } else {
            ospect = 0;
        }
        FpilDecodeTargetType(*Instrument, otype, ospect, TRUE, typeString,
                                                               typeText);
        if (status != STATUS__OK) break;
        if (otype != UnallocType) {
           fprintf(listing,"%s\t%f\t%f\t%d\t%d\t%s\n",
                 name,ra,dec,
                 FpilIsTargetSky(*Instrument,otype),
                 (FpilIsTargetGuide(*Instrument,otype) &&
                 !FpilIsTargetTelGuide(*Instrument,otype)),typeText);
	  }
    }
    SdsFreeId(name_id, &status);
    SdsFreeId(ra_id, &status);
    SdsFreeId(dec_id, &status);
    SdsFreeId(type_id, &status);
    if (spect_id) SdsFreeId(spect_id, &status);
    SdsFreeId(prio_id, &status);
    SdsFreeId(mag_id, &status);
    SdsFreeId(pid_id, &status);
    SdsFreeId(comment_id, &status);
    SdsFreeId(objid, &status);

    SdsFreeId(fid, &status);

    fclose(listing);
    if (status == STATUS__OK) {
        return TCL_OK;
    } else {
        MessGetMsg(status, 0, 100, mbuf);
        sprintf(interp->result, "Error in SkycatListing\n%s", mbuf);
        return TCL_ERROR;
    }
}



/*  ------------------------------------------------------------------------- */

/*
 *                        W r i t e  P A F  { }
 */

static int ConfWritePAF(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])

/*+
 *
 *  Tcl Command Name:
 *    WritePAF
 *
 *  Function:
 *    Output PAF file listing of objects and allocations
 *
 *  Call:
 *    WritePAF id file selGuide guideName posneg mode expert
 *
 *  Description:
 *    Output a text file listing of the allocations or objects
 *    or both in an Sds structure. The format of the listing of
 *    unallocated objects is the standard PAF file format defined for
 *    the FLAMES instrument. It includes field parameters, the VLT
 *    guide stars, the allocated and unallocated targets, and the
 *    FACB guide stars.
 *
 *  Parameters:
 *    (>)  id       (Sds Id)  Sds identifier of the top level structure
 *    (>)  file     (char)    Name of the file to be created
 *    (>)  selGuide (int)     Zero if no VLT guide star has been selected, 
 *                            positive if one has.
 *    (>)  guideName(char)    Name of selected VLT guide star, if any.
 *    (>)  posneg   (char)    Orientation for guide pivot, if a guide
 *                            object has been selected. Must be "POS" or "NEG".
 *    (>)  mode     (char)    Keyword describing the current fibre combination.
 *    (>)  expert   (int)     Non-zero if in expert mode. (Unallocated targets
 *                            are only written to the PAF file in expert mode.)
 *    (>)  checkha  (double)  +/- hour angle for which the config. has been
 *			      checked.
 *
 *   Returns:
 *     "Error" - if the file could not be opened.
 *
 *   Support:
 *     Keith Shortridge, AAO
 *
 *   History:
 *     25th Sept 2000  KS  Original version
 *     18th Aug  2001  KS  Added ARGUS keywords
 *     22nd Aug  2001  KS  Added use of expert mode to control listing of
 *                         unallocated targets.
 *     10th Dec  2001  KS  Blanks in fieldName are now changed to underscores.
 *      3rd Jan  2002  KS  Now includes INS.OSS.VER keyword. 
 *      5th Feb  2002  KS  Now supports a PAF checksum. Variables commented.
 *      4th Jun  2002  KS  Format of PAF.ID and PAF.NAME changed accoring to
 *                         suggestion by Andreas Kaufer (ESO).
 *      1st Aug  2002  KS  Modified for new use of mode string, as requested
 *                         by Paranal.
 *      2nd Aug  2002  KS  Null mode strings now represented by '-'.
 *     21st Aug  2002  KS  TEL.TARG.NAME added.
 *     22nd Aug  2002  TJF SKYANGLE now calculated properly.
 *      5th Sep  2002  KS  Guide star now identified by name rather than by
 *                         x and y position.
 *-
 */
{
    char* abbrNames[10];           /* Abbreviated PAF type names */
    SdsIdType topid;               /* Sds Id of top structure */
    SdsIdType objid;               /* Sds Id of objects structure */
    SdsIdType uid;                 /* Sds Id of unalloc guide structure */
    SdsIdType fid;                 /* Sds Id of field data structure */
    StatusType status;             /* Inherited status variable */
    FILE *listing;                 /* Used for the output PAF file */
    double mjd;                    /* Modified julian date */
    double ra;                     /* Central RA - mean */
    double dec;                    /* Central Dec - mean */
    int rhmsf[4];                  /* RA in hours minutes seconds */
    int ddmsf[4];                  /* Dec in degrees minutes seconds */
    char sign[1];                  /* Sign for RA and Dec */
    char iname[16];                /* Name of SDS item - ignored */
    SdsCodeType code;              /* SDS code type for an item */
    long ndims;                    /* Number of dimensions for SDS array */
    unsigned long dims[7];         /* Actual dimensions for SDS array */
    SdsIdType ra_id;               /* SDS id for RA array in structure */
    SdsIdType dec_id;              /* SDS id for Dec array in structure */
    SdsIdType name_id;             /* SDS id for Name array in structure */
    SdsIdType type_id;             /* SDS id for type array in structure */
    SdsIdType spect_id;            /* SDS id for spect array in structure */
    SdsIdType prio_id;             /* SDS id for priority array in structure */
    SdsIdType mag_id;              /* SDS id for magnitude array in structure */
    SdsIdType pid_id;              /* SDS id for pid array in structure */
    SdsIdType theta_id;            /* SDS id for theta array in structure */
    SdsIdType comment_id;          /* SDS id for comment array in structure */
    unsigned int i;                /* General index variable */
    int IGuide;                    /* Number of VLT guide targets */
    int unallocFACBTargets;        /* Number of unallocated reference stars */
    int Fibre;                     /* Allocated fibre for target -1 => none */
    char Prefix[16];               /* Prefix for PAF keywords for an object */
    char mbuf[100];                /* Error message buffer */
    char c;                        /* Single character used for target type */
    char s;                        /* Single char used for spectrograph */
    int numTypes;                  /* Number of different PAF types */
    char* typeNames[10];           /* Full names of PAF types */
    int iType;                     /* Index through types */
    int iFibre;                    /* Index through fibres */
    int fibreType;                 /* Fibre type code for a given fibre */
    short match;                   /* True if fibre type matches */
    int iFibType;                  /* Index through possible fibre types */
    int numTypeCodes;              /* Number of PAF fibre codes */
    int fibreTypeCodes[10];        /* PAF fibre codes; see FpilPAFFibreCodes()*/
    char insCode;                  /* Single letter for instrument */
    char unallocType;              /* Character used as "unallocated" code */
    int xusedCount;                /* Total unused targets */
    int usedCount;                 /* Total used non-sky targets */
    int totalUsedCount;            /* Total number of allocated targets */
    int skyCount;                  /* Total number of sky targets */
    int usedSkyCount;              /* Total number of allocated sky targets */
    int xTargetCount;              /* Total of unallocated targets */
    char fieldName[128];           /* Name associated with field */
    char timeString[32];           /* Formatted current time, ESO standard */
    char HhMmSs[8];                /* Formatted hours minutes secs, hhmmss. */
    struct tm* gmTimePtr;          /* Used to get current time */
    int selGuide;                  /* Value of selGuide parameter */
    int PAF_Flags;                 /* Used to pass info to ConfListPAFTarget */
    long nUnallocObj;              /* Number of unallocated objects */
    long nUnallocSky;              /* Number of unallocated sky targets */
    int nEntry;                    /* Entry count - used in PAF keyword names */
    double appEpoch;               /* Apparent epoch */
    double appRa;                  /* Field center RA -apparent */
    double appDec;                 /* Field center Dec -apparent */
    short InUse;                   /* True if ARGUS is being used */
    double Angle;                  /* ARGUS angle, if used */
    double Offset;                 /* ARGUS offset, if used */
    char Scale[32];                /* ARGUS scale, if used */
    short ExpertMode;              /* True if program is in expert mode */
    SdsIdType updateID  = 0;       /* ID for Update time in instrument file */
    StatusType localStatus;        /* Inherited status variable */
/* Linux x86_64 */
#if defined(__linux_x86_64__)
    long modTime;                   /* Instrument file modification time */
#else
    int modTime;                   /* Instrument file modification time */
#endif
    unsigned long actlen;          /* Actual length of SDS item */
    struct tm* tmPtr;              /* Used to convert modification time */
    int length;                    /* Number of characters in string */      
    char Version[32];              /* RCS version number for FPOSS */
    short ChecksumEnabled;         /* True if checksum is to be included */
    int Blanks;                    /* Number of blanks to align checksum */
    int ChecksumOffset;            /* File offset for checksum string */
    char ModeString[64];           /* Local copy of mode argument */
    char* FileMode;                /* File name component of mode argument */
    char* InsMode;                 /* INS.MODE component of mode */
    char* GirafMode;               /* INS.GIRAF.MODE component of mode */
    char* UvesSlit;                /* INS.UVES.SLIT component of mode */
   
    /*  This routine is a bit of an oddity, having been introduced purely
     *  for the FLAMES instrument used at ESO. ESO like all output files
     *  from programs such as this to be in PAF format, and we're happy
     *  to comply. This means, however, that although this routine may
     *  look like a fairly general one - and there is no reason why someone
     *  shouldn't write out a 2dF configuration using this - there are
     *  places where it has definite ESO-isms. For example, at ESO the
     *  configure program is known as FP OSS, and that appears in the
     *  header.
     */
    
    /*  Initial values */
    
    localStatus = STATUS__OK;
    ChecksumEnabled =  USE_CHECKSUM;
    ChecksumOffset = 0;
      
    /*  Initial checks. Make sure we've been invoked properly */
    
    if (argc != 9) {
        interp->result = "wrong # args";
        return TCL_ERROR;
    }
    
    /*  See if we are in expert mode or not */
    
    ExpertMode = (atoi(argv[7]) != 0);
    
    /*  We need to know the unallocated type code used in the 'objects'
     *  structure for this instrument.
     */
     
    unallocType = FpilUnallocatedType(*Instrument);
    
    /*  Make sure there's an instrument selected */
    
    if (Instrument == NULL) {
        ErsOut(0, &status, "No instrument selected");
        interp->result = "Error";
        return TCL_OK;
    }
    
    /*  Open the file specified in the second argument */
    
    listing = fopen(argv[2], "w");
    if (listing == NULL) {
        ErsOut(0, &status, "Error Opening File %s", argv[2]);
        interp->result = "Error";
        return TCL_OK;
    }
    
    /*  Access the field data in the structure, using the SDS ID passed
     *  in the first argument. Get the field name, which we need for the
     *  PAF header.
     */
    
    status = STATUS__OK;
    topid = (SdsIdType) (atol(argv[1]));
    SdsFind(topid, "fieldData", &fid, &status); 
    ArgGetString(fid, "label", sizeof(fieldName), fieldName, &status);
    nUnallocObj=0; nUnallocSky=0;
    ArgGeti(fid,"unallocObj",&nUnallocObj,&status);
    ArgGeti(fid,"unallocSky",&nUnallocSky,&status);
    
    /*  Get the date and time, and format those in the ESO standard way. Note
     *  that we get these from the global FileTime, which should have been
     *  set by PAFFileName, which will have been used to generate the file
     *  name. This ensures that we have a common view of the time in both
     *  the file name and its contents.
     */
    
    gmTimePtr = gmtime(&FileTime);
    sprintf (timeString,"%4.4d-%2.2d-%2.2dT%2.2d:%2.2d:%2.2d",
         gmTimePtr->tm_year + 1900,gmTimePtr->tm_mon + 1,gmTimePtr->tm_mday,
         gmTimePtr->tm_hour,gmTimePtr->tm_min,gmTimePtr->tm_sec);
    sprintf (HhMmSs,"%2.2d%2.2d%2.2d",
         gmTimePtr->tm_hour,gmTimePtr->tm_min,gmTimePtr->tm_sec);
         
    /*  The strings written into PAF.ID and PAF.NAME must not contain
     *  spaces. So we have to pass through fieldName changing spaces etc to
     *  underscores. timeString won't contain spaces. Nor will the mode.
     *  we
     */
   
    length = strlen(fieldName);
    for (i = 0; i < length; i++) {
        char TheChar = fieldName[i];
        if ((TheChar < 'a') || (TheChar > 'z')) {
            if ((TheChar < 'A') || (TheChar > 'Z')) {
                if ((TheChar < '0') || (TheChar > '9')) {
                    fieldName[i] = '_';
                }
            }
        }
    }
    
    /*  The 'mode' string - which is generated by FpilFibreCombos() - has
     *  four separate slash-separated parts. We need to take these apart.
     *  Note that strtok() modifies its search string, so we take a local
     *  copy in ModeString.
     */
    
    FileMode = InsMode = GirafMode = UvesSlit = (char*) NULL;
    strncpy (ModeString,argv[6],sizeof(ModeString));
    ModeString[sizeof(ModeString) - 1] = '\0';
    FileMode = strtok(ModeString,"/");
    InsMode = strtok((char*)NULL,"/");
    GirafMode = strtok((char*)NULL,"/");
    UvesSlit = strtok((char*)NULL,"/");
    if (FileMode == (char*) NULL) FileMode = "";
    if (InsMode == (char*) NULL) InsMode = "";
    if (GirafMode == (char*) NULL) GirafMode = "";
    if (UvesSlit == (char*) NULL) UvesSlit = "";
    
    /*  Insert a warning at the start of the file */
    
    fprintf (listing,
               "#\n# This file has been generated by the FPOSS program.\n");
    fprintf (listing,
               "#\n# Do NOT attempt to edit it by hand.\n");
    fprintf (listing,
                  "# Modified files will almost certainly be rejected\n");
    fprintf (listing,
                  "# by the observing system.\n\n");
    
    /*  Write out the standard PAF header. 
     */
    
    fprintf (listing,"# PAF Header;\n");
    fprintf (listing,"PAF.HDR.START;\n");
    fprintf (listing,"PAF.TYPE          \"Paramfile\";\n");
    fprintf (listing,"PAF.ID            \"%s.%s.%s\";\n",fieldName,FileMode,
                                                                     HhMmSs);
    fprintf (listing,"PAF.NAME          \"%s.%s.%s\";\n",fieldName,FileMode,
                                                                     HhMmSs);
    fprintf (listing,"PAF.DESC          \"Output from FP OSS\";\n");
    fprintf (listing,"PAF.CRTE.NAME     \"FP OSS\";\n");
    fprintf (listing,"PAF.CRTE.DAYTIM   \"%s\";\n",timeString);
    fprintf (listing,"PAF.LCHG.NAME     \"\";\n");
    fprintf (listing,"PAF.LCHG.DAYTIM   \"\";\n");
    if (ChecksumEnabled) {
        fprintf (listing,"PAF.CHCK.NAME     \"FP OSS\";\n");
        fprintf (listing,"PAF.CHCK.DAYTIM   \"%s\";\n",timeString);
    } else {
        fprintf (listing,"PAF.CHCK.NAME     \"\";\n");
        fprintf (listing,"PAF.CHCK.DAYTIM   \"\";\n");
    }
    
    /*  We treat the checksum line differently, depending on whether or not
     *  we are going to include a checksum. If we aren't, we just write a
     *  null entry which the reading software will ignore. If we are including
     *  a proper checksum, we write a zero value at this point and record
     *  its location in the file so that we can overwrite it with the
     *  calculated calue later. The checksum string has to start on a 4-byte
     *  boundary in the file, so we insert spaces as needed.
     */
     
    if (ChecksumEnabled) {
        fprintf (listing,"PAF.CHCK.CHECKSUM ");
        ChecksumOffset = ftell(listing) + 1;
        Blanks = 4 - (ChecksumOffset % 4);
        if (Blanks) {
            for (i = 0; i < Blanks; i++) fprintf (listing," ");
            ChecksumOffset += Blanks;
        }
        fprintf (listing,"\"0000000000000000\"; \n");
    } else {
        fprintf (listing,"PAF.CHCK.CHECKSUM \"\";\n");
    }
    fprintf (listing,"PAF.HDR.END;\n");
    
    fprintf (listing,"\n");
    fprintf (listing,"TPL.FILE.DIRNAME  \"$INS_ROOT/SYSTEM/MISC\";\n");
    
    /*  Now write the various keywords that describe the field in general.
     *  If INS.ARG.USED is true, we also need INS.ARGS.ANGLE (0 - 360 deg)
     *  and INS.ARGS.SCALE (a string set to either "1:1.67" or "1:1"). The
     *  angle written to the PAF file is the same is the angle returned
     *  by ConfCGetArgusData - the same as the angle passed to the 
     *  astrometric code as the first telescope model parameter. (Earlier
     *  versions of this code subtracted the offset first, but this is
     *  not correct, given the way the first telescope model parameter happens
     *  to have been defined.)
     */
         
    fprintf (listing,"\n");
    fprintf (listing,"#Primary Keywords\n");
    fprintf (listing,"INS.EQUINOX       2000.0;\n");
    
    RCSVersion(Version,sizeof(Version));
    fprintf (listing,"INS.OSS.VER       \"%s\";\n",Version);
    fprintf (listing,"INS.OSS.CHECKHA   %.1f;\n", atof(argv[8]));
       
    ConfCGetArgusData (&InUse,&Angle,Scale,&Offset);
    if (InUse) {
       fprintf (listing,"INS.ARGS.USED     T;\n");
       fprintf (listing,"INS.ARGS.ANGLE    %f;\n",Angle);
       fprintf (listing,"INS.ARGS.SCALE    \"%s\";\n",Scale);
    } else {
       fprintf (listing,"INS.ARGS.USED     F;\n");
    }
    fprintf (listing,"\n");
    fprintf (listing,"#Telescope Category\n");
    
    ArgGetd(fid, "appRa", &appRa, &status);
    ArgGetd(fid, "appDec", &appDec, &status);
    ArgGetd(fid, "configMjd", &mjd, &status);
    ArgGetd(fid, "appEpoch", &appEpoch, &status);
    ArgGetd(fid, "cenRa", &ra, &status);
    ArgGetd(fid, "cenDec", &dec, &status);
    slaDr2tf(3, ra, sign, rhmsf);
    slaDr2af(2, dec, sign, ddmsf);
    
    /*  We duplicate the RA and DEC in both TEL and INS. This is
     *  deliberate, because the Telescope system needs this and so does
     *  the positioner software, and the easiest thing is to have them
     *  in twice.
     */
     
    fprintf (listing,"TEL.TARG.NAME     \"%s\";\n",fieldName);
    fprintf (listing,"TEL.TARG.RA       %2.2d%2.2d%2.2d.%2.2d;\n",
                    rhmsf[0], rhmsf[1], rhmsf[2], rhmsf[3] / 10);
    if (sign[0] == '-') {
       fprintf (listing,"TEL.TARG.DEC      -%2.2d%2.2d%2.2d.%1.1d;\n",
                    ddmsf[0], ddmsf[1], ddmsf[2], ddmsf[3] / 10);
    } else {
       fprintf (listing,"TEL.TARG.DEC      %2.2d%2.2d%2.2d.%1.1d;\n",
                    ddmsf[0], ddmsf[1], ddmsf[2], ddmsf[3] / 10);
    }
    fprintf (listing,"INS.TARG.RA       %2.2d%2.2d%2.2d.%2.2d;\n",
                    rhmsf[0], rhmsf[1], rhmsf[2], rhmsf[3] / 10);
    if (sign[0] == '-') {
       fprintf (listing,"INS.TARG.DEC      -%2.2d%2.2d%2.2d.%1.1d;\n",
                    ddmsf[0], ddmsf[1], ddmsf[2], ddmsf[3] / 10);
    } else {
       fprintf (listing,"INS.TARG.DEC      %2.2d%2.2d%2.2d.%1.1d;\n",
                    ddmsf[0], ddmsf[1], ddmsf[2], ddmsf[3] / 10);
    }
    
    /*  If we read the PAF file back into configure, we need the
     *  APPRA, APPDEC and MJD and EPOCH - actually, we need to think
     *  carefully about the way these are used, but they're in the
     *  structure so it makes sense to have them. Since they aren't
     *  in the FPOSS dictionary at the moment, we put them out as
     *  pseudo-comments.
     */
     
    slaDr2tf(3, appRa, sign, rhmsf);
    slaDr2af(2, appDec, sign, ddmsf);
    fprintf (listing,"#!INS.TARG.APPRA  %2.2d%2.2d%2.2d.%2.2d;\n",
                    rhmsf[0], rhmsf[1], rhmsf[2], rhmsf[3] / 10);
    if (sign[0] == '-') {
       fprintf (listing,"#!INS.TARG.APPDEC -%2.2d%2.2d%2.2d.%1.1d;\n",
                    ddmsf[0], ddmsf[1], ddmsf[2], ddmsf[3] / 10);
    } else {
       fprintf (listing,"#!INS.TARG.APPDEC %2.2d%2.2d%2.2d.%1.1d;\n",
                    ddmsf[0], ddmsf[1], ddmsf[2], ddmsf[3] / 10);
    }
    fprintf (listing,"#!INS.TARG.MJD    %f;\n",mjd);
    fprintf (listing,"#!INS.TARG.APPEPOCH  %f;\n",appEpoch);
    
    /*  Now the time stamp for the instrument status file. We look for
     *  an UPDATE_TIME item in the pivot details structure. If this is
     *  FLAMES, then such an item should be present.
     */
    
    localStatus = STATUS__OK;
    modTime = 0;
    SdsFind(InstFieldDetails, "UPDATE_TIME", &updateID, &localStatus);
    SdsGet(updateID,sizeof(int),0,&modTime,&actlen,&localStatus);
    SdsFreeId(updateID,&localStatus);
    tmPtr = gmtime((time_t*)&modTime);
    
    fprintf (listing,"\n");
    fprintf (listing,"#Instrument status file time stamp\n");
    fprintf (listing,"INS.STAT.SFSTAMP  \"%4d-%02d-%02dT%02d:%02d:%02d\";\n",
               1900 + tmPtr->tm_year, 1 + tmPtr->tm_mon,tmPtr->tm_mday,
                              tmPtr->tm_hour,tmPtr->tm_min,tmPtr->tm_sec);
    
    /*  We need to set the keywords for the current fibre combination mode */
    
    fprintf (listing,"INS.MODE          \"%s\";\n",InsMode);
    if ((GirafMode[0] != '\0') && (GirafMode[0] != '-')) {
        fprintf (listing,"INS.GIRAF.MODE    \"%s\";\n",GirafMode);
    }
    if ((UvesSlit[0] != '\0') && (UvesSlit[0] != '-')) {
        fprintf (listing,"INS.UVES.SLIT     \"%s\";\n",UvesSlit);
    }
        
    /*  The potential guide stars refer to those targets that can be used
     *  as VLT guide stars. These cannot be allocated, so will be found
     *  in the unallocated guide section of the structure. We search that
     *  structure for them. Note that one of these may have been selected
     *  and the guide probe position for it may also have been selected. In
     *  this case, the arguments used to invoke this command show this, and
     *  we can create an entry for the actual guide star to be used.
     */

    SdsFind(topid, "unallocGuide", &uid, &status);
    ConfGetIds(uid, &ra_id, &dec_id, &name_id, &type_id, &spect_id,
            &prio_id, &mag_id, &pid_id, &comment_id, &status);
    SdsInfo(ra_id, iname, &code, &ndims, dims, &status);
    
    /*  The selected guide star, if there is one. We identify it by
     *  its name, as passed.
     */
    
    selGuide = atoi(argv[3]);
    if (selGuide > 0) {
        char* guideName = argv[4];
        char name[NAMELEN];
        fprintf (listing,"\n");
        fprintf (listing,"# Selected Instrument Guide Star\n");
        for (i = 0; i < dims[0]; i++) {
            ConfArrayGetc(uid, "type", i, &c, &status);
            if (FpilIsTargetTelGuide(*Instrument, c)) {
                SdsGet(name_id, NAMELEN, NAMELEN * i, name, &actlen, &status);
                if (!strcmp(name,guideName)) {
                    Fibre = -1;
                    PAF_Flags = PAF_NO_PRIO;
                    ConfListPAFTarget(listing, i, "TEL.GS.", ra_id, 
                        dec_id, name_id, type_id, spect_id, prio_id, mag_id,
                        pid_id, comment_id, Fibre, PAF_Flags, &status);
                    fprintf (listing,"TEL.GS.PPOS            \"%s\";\n",
                                                                   argv[5]);
                    break;
                }
            }
        }
    }
    
    /*  And now all the others, the potential guide stars - note that these
     *  include the actual guide star selected, if there is one.
     *  As we go through, we note how many non-VLT guide targets there are
     *  in the unallocated guide section, since we need that later.
     */
     
    fprintf (listing,"\n");
    fprintf (listing,"# Instrument Potential Guide Stars\n");    
    IGuide = 0;
    unallocFACBTargets = 0;
    for (i = 0; i < dims[0]; i++) {
        ConfArrayGetc(uid, "type", i, &c, &status);
        if (FpilIsTargetTelGuide(*Instrument, c)) {
            IGuide++;
            Fibre = -1;
            sprintf (Prefix,"TEL.PGS%d.",IGuide);
            PAF_Flags = 0;
            ConfListPAFTarget(listing, i, Prefix, ra_id, dec_id, name_id,
                    type_id, spect_id, prio_id, mag_id, pid_id, comment_id,
                    Fibre, PAF_Flags, &status);
        } else {
           unallocFACBTargets++;
        }
        if (status != STATUS__OK) break;
    }
    SdsFreeId(name_id, &status);
    SdsFreeId(ra_id, &status);
    SdsFreeId(dec_id, &status);
    SdsFreeId(type_id, &status);
    if (spect_id) {
        SdsFreeId(spect_id, &status);
    }
    SdsFreeId(prio_id, &status);
    SdsFreeId(mag_id, &status);
    SdsFreeId(pid_id, &status);
    SdsFreeId(comment_id, &status);
    SdsFreeId(uid, &status);
    
    /*  And we need the central wavelength. This is maintained by the
     *  C layer in the global variable gwave, and is a value in microns.
     *  We convert it to Angstroms - is this the right unit?
     */
   
    fprintf (listing,"TEL.PCC.WLEN           %f;\n",gwave * 10000.0);
      
    /*  For each type of fibre, we want to put out a summary containing the
     *  following information:
     *  INS.type.USED.OBJECTS   is # of fibres allocated to non-sky targets
     *  INS.type.USED.SKY       is # of fibres allocated to sky targets
     *  INS.type.XOBJ           is # of potential targets this type of fibre
     *                          could have been allocated to, but which ended
     *                          up unallocated.
     *  INS.type.XUSE           is # of fibres of this type unallocated.
     *
     *  We also want to put out the details of each fibre that has been
     *  allocated.
     */
    
    /*  First of all we do this for the FACB guide stars. These have to be
     *  handled separately, since the information about them is scattered
     *  in different sections of the structure than that for the other types.
     *  Some of these will be unallocated, and will be found in the unallocated
     *  guide section of the structure. Some will have been allocated, and
     *  these will be found in the objects structure. We list here the ones
     *  that have been allocated, and leave the ones that haven't for the
     *  end of the file where we list all the unallocated targets.
     */
                                     
    fprintf (listing,"\n");
    fprintf (listing,"# FACBs - Allocated Reference Stars\n");    
    IGuide = 0;
    usedCount = usedSkyCount = xusedCount = 0;
    SdsFind(topid, "objects", &objid, &status);
    ConfGetIds(objid, &ra_id, &dec_id, &name_id, &type_id, &spect_id,
            &prio_id, &mag_id, &pid_id, &comment_id, &status);
            

    /*  Also pick up "theta" - which is not returned by ConfGetIds. */
     
    SdsFind(objid, "theta", &theta_id, &status);

    SdsInfo(ra_id, iname, &code, &ndims, dims, &status);
    for (i = 0; i < dims[0]; i++) {
    
        /*  What we are doing here is looking at each entry in the 'objects'
         *  structure. There is one entry for each fibre. For each, we see if
         *  this is a guide fibre. If it is, we see if it is allocated. If
         *  so, we see if it is allocated to sky or to something else. 
         *  (Actually, guide fibres can only be allocated to guide targets,
         *  as things stand, but that's something hidden in the Fpil
         *  compatability routine, so we shouldn't assume it here.) If it's
         *  allocated, we list it.
         */
         
        fibreType = ConfWhichSpec (i, 0);
        if (FpilIsFibreGuide(*Instrument,fibreType)) {
            ConfArrayGetc(objid, "type", i, &c, &status);
            if (c == unallocType) {
                xusedCount++;
            } else {
            
               /*  Get theta value - the button orientation */
               
                double theta;
                SdsGet(theta_id, sizeof(double), i, &theta, &actlen, &status);
                
                if (FpilIsTargetSky(*Instrument,c)) {
                    usedSkyCount++;
                } else {
                    usedCount++;
                }
                IGuide++;
                sprintf (Prefix,"INS.REF%d.",IGuide);
                Fibre = i + 1;
                PAF_Flags = 0;
                ConfListPAFTarget(listing, i, Prefix, ra_id, dec_id, name_id,
                    type_id, spect_id, prio_id, mag_id, pid_id, comment_id,
                    Fibre, PAF_Flags, &status);
                    
                /*  We need to calculate the sky angle properly - this can be
                 *  derived from the button theta by adding PI/2 to it.
                 */
                 
                fprintf (listing,"%sSKYANGLE        %f;\n",
                                            Prefix, slaDranrm(theta+DPIBY2));
            }
        }
        if (status != STATUS__OK) break;
    }
    
    /*  At the end of that loop, IGuide is the number of guide fibres that
     *  have been allocated. usedCount is the number allocated to non-sky
     *  targets, usedSkyCount is the number (probably none) allocated to
     *  sky targets. unallocFACBTargets is the number of unallocated guide 
     *  targets, determined earlier. We now have all we need to output the
     *  FACB summary. 
     */
     
    fprintf (listing,"\n");
    fprintf (listing,"# FACB Fibre counts\n");
    fprintf (listing,"INS.FACB.USED.OBJECTS %d;\n",usedCount);
    fprintf (listing,"INS.FACB.USED.SKY     %d;\n",usedSkyCount);
    fprintf (listing,"INS.FACB.XOBJ         %d;\n",unallocFACBTargets);
    fprintf (listing,"INS.FACB.XUSE         %d;\n",xusedCount);
                                                                   
    
    /*  Release all the SDS Id we gathered in the process. */
                                                                 
    SdsFreeId(name_id, &status);
    SdsFreeId(ra_id, &status);
    SdsFreeId(dec_id, &status);
    SdsFreeId(type_id, &status);
    if (spect_id) {
        SdsFreeId(spect_id, &status);
    }
    SdsFreeId(prio_id, &status);
    SdsFreeId(mag_id, &status);
    SdsFreeId(pid_id, &status);
    SdsFreeId(comment_id, &status);
    SdsFreeId(objid, &status);
        
    /*  Now we have to handle the targets for the various other fibre types.
     *  The allocated fibres are all in the 'objects' sub-structure. In
     *  an attempt to keep all this relatively instrument-independent -
     *  ie we don't want too much FLAMES-specific stuff here if we can
     *  help it - we do this by getting the number of different PAF types
     *  to use and then we go through each fibre in turn to see which
     *  PAF type it is.
     */ 
     
    SdsFind(topid, "objects", &objid, &status);
    ConfGetIds(objid, &ra_id, &dec_id, &name_id, &type_id, &spect_id,
            &prio_id, &mag_id, &pid_id, &comment_id, &status);
    SdsInfo(ra_id, iname, &code, &ndims, dims, &status);
    
    FpilPAFTypes (*Instrument,10,&numTypes,typeNames,abbrNames);
    totalUsedCount = 0;
    for (iType = 0; iType < numTypes; iType++) {    
        insCode = *(abbrNames[iType]);
        usedCount = usedSkyCount = xusedCount = 0;
        fprintf (listing,"\n");
        fprintf (listing,"# %s Allocated Fibres\n",typeNames[iType]);
        FpilPAFFibreCodes (*Instrument,iType,10,&numTypeCodes,fibreTypeCodes);
        nEntry = 0;
        for (iFibre = 0; iFibre < dims[0]; iFibre++) {
            ConfArrayGetc(objid, "type", iFibre, &c, &status);
            fibreType = ConfWhichSpec (iFibre, 0);
            match = FALSE;
            for (iFibType = 0; iFibType < numTypeCodes; iFibType++) {
                if (fibreTypeCodes[iFibType] == fibreType) {
                    match = TRUE;
                    break;
                }
            }
            if (match) {
                if (c == unallocType) {
                    xusedCount++;
                } else {
                    if (FpilIsTargetSky(*Instrument,c)) {
                        usedSkyCount++;
                    } else {
                        usedCount++;
                    }
                    Fibre = iFibre + 1;
                    PAF_Flags = 0;
                    nEntry++;
                    sprintf (Prefix,"INS.%cFIB%d.",insCode,nEntry);
                    ConfListPAFTarget(listing, iFibre, Prefix, ra_id, dec_id,
                        name_id, type_id, spect_id, prio_id, mag_id, pid_id, 
                        comment_id, Fibre, PAF_Flags, &status);
                }
                if (status != STATUS__OK) break;
            }
        }
        fprintf (listing,"\n");
        fprintf (listing,"# %s Fibre counts\n",typeNames[iType]);
        fprintf (listing,"INS.%s.USED.OBJECTS %d;\n",abbrNames[iType],
                                                                 usedCount);
        fprintf (listing,"INS.%s.USED.SKY     %d;\n",abbrNames[iType],
                                                                 usedSkyCount);
        totalUsedCount += (usedCount + usedSkyCount);
        
        SdsFind(topid, "unallocObject", &uid, &status);
        xTargetCount = 0;
        for (i = 0; i < (nUnallocObj + nUnallocSky); i++) {
            ConfArrayGetc(uid, "type", i, &c, &status);
            c = FpilEnabledTargetType (*Instrument,c);
            ConfArrayGetc(uid, "spectrograph", i, &s, &status);
            if (!FpilIsTargetSky(*Instrument,c)) {
                match = FALSE;
                for (iFibType = 0; iFibType < numTypeCodes; iFibType++) {
                    if (FpilTargetPivotCompatible(*Instrument, c, s,
                                        fibreTypeCodes[iFibType])) {
                        match = TRUE;
                        break;
                    }
                }
                if (match) xTargetCount++;
            }          
            if (status != STATUS__OK) break;
        }
        SdsFreeId(uid, &status);
        fprintf (listing,"INS.%s.XOBJ         %d;\n",abbrNames[iType],
                                                                 xTargetCount);
        fprintf (listing,"INS.%s.XUSE         %d;\n",abbrNames[iType],
                                                                   xusedCount);
    }
    SdsFreeId(name_id, &status);
    SdsFreeId(ra_id, &status);
    SdsFreeId(dec_id, &status);
    SdsFreeId(type_id, &status);
    if (spect_id) {
        SdsFreeId(spect_id, &status);
    }
    SdsFreeId(prio_id, &status);
    SdsFreeId(mag_id, &status);
    SdsFreeId(pid_id, &status);
    SdsFreeId(comment_id, &status);
    SdsFreeId(objid, &status);
    
    /*  Total number of used fibres */
    
    fprintf (listing,"\n");
    fprintf (listing,"# TOTAL number of fibres in the configuration\n");    
    fprintf (listing,"# INS.PCC.NUMFIB =\n");    
    for (iType = 0; iType < numTypes; iType++) {
    
        /*  The only difference between these two cases is the final '+' in
         *  the line!
         */
         
        if (iType < (numTypes - 1)) {    
            fprintf (listing,"#   INS.%s.USED.OBJECTS + INS.%s.USED.SKY +\n",
                                         abbrNames[iType],abbrNames[iType]);
        } else {
            fprintf (listing,"#   INS.%s.USED.OBJECTS + INS.%s.USED.SKY\n",
                                         abbrNames[iType],abbrNames[iType]);
        }
    }
    fprintf (listing,"INS.PCC.NUMFIB         %d;\n",totalUsedCount);
       
    /*  And finally the unallocated targets. The unallocated program objects
     *  are all in the 'unallocObject' structure, and all the targets in there
     *  should be included in the listing. These are put out as pseudo-
     *  comments, with a #! in front of them - this means that standard ESO 
     *  PAF readers won't be overwhelmed by them, but configure itself can
     *  still read them. (This is a bit of a fudge, but it's effective.)
     *  This is only done in expert mode, to keep the size of the PAF file
     *  down. (This was a compromise reached with ESO, once it was realised
     *  that FPOSS probably would not usually be used to re-read a PAF file
     *  at the telescope.) We do the pass through, even if not in expert mode
     *  to get the counts right, but only write out the pseudo-comment 
     *  keyword in expert mode.
     */
     
    skyCount = 0;
    fprintf (listing,"\n");
    if (ExpertMode) fprintf (listing,"# Unallocated target objects\n");    
    SdsFind(topid, "unallocObject", &uid, &status);
    ConfGetIds(uid, &ra_id, &dec_id, &name_id, &type_id, &spect_id,
            &prio_id, &mag_id, &pid_id, &comment_id, &status);
    SdsInfo(ra_id, iname, &code, &ndims, dims, &status);
    for (i = 0; i < dims[0]; i++) {
        ConfArrayGetc(uid, "type", i, &c, &status);
        if (FpilIsTargetSky(*Instrument,c)) {
           skyCount++;
        }
        if (ExpertMode) {
            Fibre = -1;
            PAF_Flags = 0;
            sprintf (Prefix,"#!INS.XOBJ%d.",i + 1);
            ConfListPAFTarget(listing, i, Prefix, ra_id, dec_id, name_id,
                    type_id, spect_id, prio_id, mag_id, pid_id, comment_id,
                    Fibre, PAF_Flags, &status);
        }
        if (status != STATUS__OK) break;
    }
    fprintf (listing,"\n");
    fprintf (listing,"INS.SKY.XOBJ        %d;\n",skyCount);
    
    SdsFreeId(name_id, &status);
    SdsFreeId(ra_id, &status);
    SdsFreeId(dec_id, &status);
    SdsFreeId(type_id, &status);
    if (spect_id) {
        SdsFreeId(spect_id, &status);
    }
    SdsFreeId(prio_id, &status);
    SdsFreeId(mag_id, &status);
    SdsFreeId(pid_id, &status);
    SdsFreeId(comment_id, &status);
    SdsFreeId(uid, &status);

    /*  Then we have the unallocated guide objects. These will be in the
     *  'unallocGuide' section of the structure. We don't include the
     *  guide stars intended for use with the telescope guider, as these
     *  have already been listed.  Again, note the use of pseudo-comments.
     */
     
    if (ExpertMode) {
        fprintf (listing,"\n");
        fprintf (listing,"# Unallocated guide objects\n");    
        SdsFind(topid, "unallocGuide", &uid, &status);
        ConfGetIds(uid, &ra_id, &dec_id, &name_id, &type_id, &spect_id,
            &prio_id, &mag_id, &pid_id, &comment_id, &status);
        SdsInfo(ra_id, iname, &code, &ndims, dims, &status);
        IGuide = 0;
        for (i = 0; i < dims[0]; i++) {
            ConfArrayGetc(uid, "type", i, &c, &status);
            if ((FpilIsTargetGuide(*Instrument, c)) && 
                   !FpilIsTargetTelGuide(*Instrument, c)) {
                IGuide++;
                sprintf (Prefix,"#!INS.XREF%d.",IGuide);
                Fibre = -1;
                PAF_Flags = 0;
                ConfListPAFTarget(listing, i, Prefix, ra_id, dec_id, name_id,
                    type_id, spect_id, prio_id, mag_id, pid_id, comment_id,
                    Fibre, PAF_Flags, &status);
            }
            if (status != STATUS__OK) break;
        }
        SdsFreeId(name_id, &status);
        SdsFreeId(ra_id, &status);
        SdsFreeId(dec_id, &status);
        SdsFreeId(type_id, &status);
        if (spect_id) {
            SdsFreeId(spect_id, &status);
        }
        SdsFreeId(prio_id, &status);
        SdsFreeId(mag_id, &status);
        SdsFreeId(pid_id, &status);
        SdsFreeId(comment_id, &status);
        SdsFreeId(uid, &status);
    }
   
    SdsFreeId(fid, &status);
    
    fclose(listing);
    
    pafchkChecksumCreate(argv[2], "FP OSS", 0);         

    /*  Now, if we are to write a checksum into the file, we rewind it,
     *  and re-read it, calculating the checksum for it. We then overwrite
     *  the "0000000000000000" zero checksum it contains with the complement
     *  of the checksum calculated for it with that zero checksum included.
     *  The result will be a file that has a zero checksum (strictly, it
     *  has a -ve zero checksum of all 1's, which is a 1's complement
     *  -ve zero).
     */
     
    if (ChecksumEnabled) {
    
        char record[28800];
        int recsize;
        char ascii[32];

        unsigned short sum16 = 0;
        unsigned int sum32 = 0;

        /*  Re-read, accumulating the current checksum for the file */
        
        listing = fopen(argv[2], "r+");
        while (! feof (listing)) {
            if ((recsize = fread (record, sizeof(char), 
                                   sizeof(record)/sizeof(char), listing))) {
                checksum (record, recsize, &sum16, &sum32);
            }
        }
        
        /*  Calculate the required complement as an encoded ascii string */
        
        char_encode (~sum32, ascii, 4, 0);
        ascii[16] = '\0';
        
        /*  And overwrite the original zero checksum with the complement */
        
        fseek (listing,ChecksumOffset,SEEK_SET);
        fwrite (ascii,1,16,listing);
        fclose(listing);
	 }
        
    
    if (status == STATUS__OK) {
        return TCL_OK;
    } else {
        MessGetMsg(status, 0, 100, mbuf);
        sprintf(interp->result, "Error in WritePAF\n%s", mbuf);
        return TCL_ERROR;
    }
}

/*  ------------------------------------------------------------------------- */

/*
 *                        P A F  F i l e  N a m e { }
 */

static int ConfPAFFileName (ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])

/*+
 *
 *  Tcl Command Name:
 *    PAFFileName
 *
 *  Function:
 *    Generate the name to be used for an output PAF file.
 *
 *  Call:
 *    PAFFileName id mode
 *
 *  Description:
 *    For FLAMES, ESO mandate a form to be used for the PAF file name.
 *    This takes the form <label>.<mode>.<time>.ins where <label> is the
 *    label supplied by the astronomer for the field, <mode> is the 
 *    observing mode and <time> is the time the file was written in the
 *    form HHMMSS.
 *
 *  Parameters:
 *    (>)  id       (Sds Id)  Sds identifier of the top level structure
 *    (>)  mode     (char)    Keyword describing the current fibre combination.
 *
 *   Returns:
 *     The file name to be used.
 *
 *   Support:
 *     Keith Shortridge, AAO
 *
 *   History:
 *      4th Jun  2002  KS  Original version
 *      1st Aug  2002  KS  Modified for new use of mode string, as requested
 *                         by Paranal.
 *-
 */
{
          
    SdsIdType fid;                 /* Sds Id of field data structure */
    char fieldName[128];           /* Name associated with field */
    char* FileMode;                /* File name component of mode argument */
    struct tm* gmTimePtr;          /* Used to get current time */
    unsigned int i;                /* General index variable */
    int length;                    /* Number of characters in string */      
    char ModeString[64];           /* Local copy of mode argument */
    StatusType status;             /* Inherited status variable */
    SdsIdType topid;               /* Sds Id of top structure */
    
    /*  Initial checks. Make sure we've been invoked properly */
    
    if (argc != 3) {
        interp->result = "wrong # args";
        return TCL_ERROR;
    }
    
    /*  Make sure there's an instrument selected */
    
    if (Instrument == NULL) {
        ErsOut(0, &status, "No instrument selected");
        interp->result = "Error";
        return TCL_OK;
    }
        
    /*  Access the field data in the structure, using the SDS ID passed
     *  in the first argument. Get the field name.
     */
    
    status = STATUS__OK;
    topid = (SdsIdType) (atol(argv[1]));
    SdsFind(topid, "fieldData", &fid, &status); 
    ArgGetString(fid, "label", sizeof(fieldName), fieldName, &status);
    
    /*  The filed name must not contain spaces. So we have to pass through
     *  fieldName changing spaces and other non-alphabetic non-numeric
     *  characters to underscores.
     */
   
    length = strlen(fieldName);
    for (i = 0; i < length; i++) {
        char TheChar = fieldName[i];
        if ((TheChar < 'a') || (TheChar > 'z')) {
            if ((TheChar < 'A') || (TheChar > 'Z')) {
                if ((TheChar < '0') || (TheChar > '9')) {
                    fieldName[i] = '_';
                }
            }
        }
    }
    
    /*  Get the date and time. We get this into the global variable FileTime,
     *  so that WritePAF can be sure it is using the same time.
     */
    
    FileTime = time(0);
    gmTimePtr = gmtime(&FileTime);
    
    /*  The current instrument mode is included in the file name, but this
     *  is now (since 1/8/02) a set of components as formatted by the
     *  routine FpilFibreCombos(). We now only need the first component. 
     */    
    
    FileMode = (char*) NULL;
    strncpy (ModeString,argv[2],sizeof(ModeString));
    ModeString[sizeof(ModeString) - 1] = '\0';
    FileMode = strtok(ModeString,"/");
    if (FileMode == (char*) NULL) FileMode = "";
    
    /*  Now format the resulting file name */
    /*
    sprintf (interp->result,"%s.%s.%4.4d-%2.2d-%2.2dT%2.2d:%2.2d:%2.2d.ins",fieldName,FileMode,
                  gmTimePtr->tm_year,gmTimePtr->tm_mon+1,gmTimePtr->tm_mday,
                  gmTimePtr->tm_hour,gmTimePtr->tm_min,gmTimePtr->tm_sec); 
    */
    sprintf (interp->result,"%s.%s.%2.2d%2.2d%2.2d.ins",fieldName,FileMode,
             gmTimePtr->tm_hour,gmTimePtr->tm_min,gmTimePtr->tm_sec);
    return TCL_OK;
} 
    
/*  ------------------------------------------------------------------------- */

/*
 *                C o n f  G e t  P r o p e r  M o t i o n s
 *
 *  This routine looks through the comment string for a target object,
 *  looking for any proper-motion information encoded as a pseudo-comment.
 *  It is looking specifically for fields of the form PM-RA=value
 *  and PM-DEC=value. See ConfParseComment() for more detail on which
 *  formats are actually accepted. The proper motions are specified in
 *  the pseudo-comments in units of arcsec/year, but are returned by this
 *  routine in units of radians/year which is as required by the SLALIB 
 *  routine sla_mapqk().
 */
 
static int ConfGetProperMotions (     /* True if proper motion data found */
    char* Comment,                    /* Address at which to start parsing */
    double* PmRa,                     /* Proper motion in RA */
    double* PmDec)                    /* Proper motion in Dec */
{
    int Found = FALSE;
    char Keyword[80];
    char Value[80];
    char *NextChar;
    int Key;
    char* dstr;

    *PmRa = 0.0;
    *PmDec = 0.0;
    
    /*  For efficiency, we do a quick scan for the string "PM-", which
     *  both keywords have in common. If this isn't there, there's no
     *  point doing the more exhaustive scan of the string.
     */
   
    if (strstr(Comment,"PM-")) {

       /*  This looks as if it might have proper motion pseudo-comments.
        *  We scan it in detail using ConfParseComment(). If this finds one
        *  of the proper motion keywords, we get the value associated
        *  with it. Note that ConfParseComment() returns the PAF keywords
        *  associated with the specification, which for PM-RA and PM-DEC
        *  are PM_RA and PM_DEC - underscores, not dashes.
        */
        
       NextChar = Comment;
       for (;;) {
           char* CommentPtr = NextChar;
           Key = ConfParseComment (CommentPtr, Keyword, Value, &NextChar);
           if (!Key) break;
           if (!strncmp(Keyword,"PM_RA",5)) {
               *PmRa = atof(Value) * PI / (60.0 * 60.0 * 180.0);
               Found = TRUE;
           } else if (!strncmp(Keyword,"PM_DEC",6)) {
               *PmDec = atof(Value) * PI / (60.0 * 60.0 * 180.0);
               Found = TRUE;
           }
       }
    }
    
    return (Found);
}

/*  ------------------------------------------------------------------------- */

/*
 *                    C o n f  P a r s e  C o m m e n t
 *
 *  This routine parses a comment field for a target object, looking for any
 *  of the auxiliary keywords that can be embedded in a comment string,
 *  particularly for FLAMES. It is passed the address of a starting character
 *  in the comment string, and - if it finds one of the auxiliary keywords -
 *  returns the auxiliary keyword, a value string for it, suitable for output
 *  to a PAF file, and the address of the next character in the comment string.
 *  The calling routine should continue to call this routine, each time passing
 *  it the address it returns, until it returns indicating that no keyword
 *  was found. The keyword is returned padded on the right to 8 characters.
 */
 
static int ConfParseComment (         /* True if there was a keyword found */
    char* Comment,                    /* Address at which to start parsing */
    char* Keyword,                    /* Returned with Keyword string */
    char* Value,                      /* Returned with value string */
    char** NextChar)                  /* Address at which to start next */
{
    char* CharPtr;
    char* ValuePtr;
    int I;
    int KeyIndex;
    int Length;
    short PossibleKey;
    short Leading;
    
    /*  The parsing provided by this routine is a little crude. It will
     *  correctly parse any correctly formatted strings, but may not
     *  pick up some incorrectly formatted ones. It's possible the 
     *  interface should actually include an error flag that can be set.
     */
    
    static char* Keywords[] = {
       "BAND", "B-V", "V-R", "PM-RA", "PM-DEC", "SNR", "CATEGORY", "DIAMETER" };
    static char* PAFKeywords[] = {
       "BAND", "B_V", "V_R", "PM_RA", "PM_DEC", "SNR", "CATEGORY", "DIAMETER" };
    static short KeywordIsChar[] = {
        TRUE,   FALSE, FALSE, FALSE,   FALSE,    FALSE, TRUE,       FALSE };
       
    static int NumKeywords = sizeof(Keywords)/sizeof(char*);
    
    /*  We allow both formats (Name)=value and (Name=value). In both cases,
     *  these begin with a '('. We skip any leading blanks, commas, and
     *  semi-colons.
     */
    
    CharPtr = Comment;
    PossibleKey = FALSE;
    KeyIndex = 0;
    for (;;) {
       char TheChar = *CharPtr;
       if (TheChar == '(') {
           CharPtr++;
           PossibleKey = TRUE;
           break;
       } else if ((TheChar == ' ') || (TheChar == ',') || (TheChar == ';')) {
           CharPtr++;
       } else {
           break;
       }
    }
    
    /*  So, if we have a startling '(', we can look through all the possible
     *  keywords. If we find a match, we check that the next characters are
     *  '=' or ')='. If they are, we have a match.
     */
    
    if (PossibleKey) {
        PossibleKey = FALSE;
        for (KeyIndex = 0; KeyIndex < NumKeywords; KeyIndex++) {
            Length = strlen(Keywords[KeyIndex]);
            if (!strncmp(CharPtr,Keywords[KeyIndex],Length)) {
                CharPtr += Length;
                if (*CharPtr == ')') CharPtr++;
                if (*CharPtr == '=') {
                    PossibleKey = TRUE;
                    CharPtr++;
                    break;
                }   
            }
        }
    }
        
    /*  If we do have a match, KeyIndex will have the index of the keyword
     *  that matches. We now need to locate the value. We ignore leading
     *  blanks in the value string.
     */
    
    if (PossibleKey) {
        strcpy (Keyword,PAFKeywords[KeyIndex]);
        Length = strlen(Keyword);
        for (I = 0; I < (8 - Length); I++) strcat(Keyword," ");
        ValuePtr = Value;
        if (KeywordIsChar[KeyIndex]) *ValuePtr++ = '"';
        Leading = TRUE;
        for (;;) {
            char TheChar = *CharPtr++;
            short Ignore = FALSE;            
            if (Leading && (TheChar == ' ')) Ignore = TRUE;
            if (TheChar != ' ') Leading = FALSE;
	    if (Ignore)
        	*NextChar = (TheChar == '\0') ? CharPtr - 1 : CharPtr;
            if (!Ignore) {
                if ((TheChar == ' ') || (TheChar == ')') || (TheChar == '\0') ||
                                         (TheChar == ';') || (TheChar == ',')) {
                    if (!(Leading && (TheChar == ' '))) {
                        if (KeywordIsChar[KeyIndex]) *ValuePtr++ = '"';
                        *ValuePtr = '\0';
                        if (KeywordIsChar[KeyIndex] == 0 && ValuePtr == Value)
		    	    strcpy(Value, "-999");
                        if (TheChar == '\0') CharPtr--;
                        *NextChar = CharPtr;
                        break;
                    }
                }
        	if (KeywordIsChar[KeyIndex] == 0 && ((TheChar < '0' || TheChar > '9') && 
			TheChar != '.') && TheChar != '+' && TheChar != '-') {
		    strcpy(Value, "-999");
		    break;
		}
                *ValuePtr++ = TheChar;
            }
        }
        
    }
    if (!PossibleKey) *NextChar = Comment;
    
    return (PossibleKey);    
}        



/*  ------------------------------------------------------------------------- */

/*              PAF checksum verify {}
 *
 */

static int PafChkVerify(void *clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])
{
    int Result;
    const char *filename;

    if (argc != 2)
	{
        interp->result = "wrong # args";
        return TCL_ERROR;
	}

    filename = argv[1];

    Result =  pafchkChecksumVerify(filename);

    sprintf(interp->result, "%d", Result);
    return TCL_OK;
}


/*  ------------------------------------------------------------------------- */

/*
 *                         R e a d  P A F  { }
 */

static int ConfReadPAF(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])

/*+
 *  Tcl Command Name:
 *    ReadPAF
 *
 *  Function:
 *    Read a PAF file listing of objects and allocations
 *
 *  Call:
 *    ReadPAF
 *
 *  Description:
 *
 *  Parameters:
 *    (>)  file     (char)    Name of the file to be read
 *    (<)  selGuide (int)     Selected guide object - index into the 
 *                            unallocGuide structure, starting from 1. Zero
 *                            if no guide object selected. 
 *    (<)  posneg   (char)    Orientation for guide pivot, if a guide
 *                            object has been selected. Will be "POS" or "NEG".
 *    (<)  modeCode (int)     Integer code giving the mode - the fibre 
 *                            combination in use - for the file. If this 
 *                            cannot be determined, -1 is returned. Otherwise
 *                            this is an index into the set of combinations
 *                            returned by FpilFibreCombos().
 *    (<)  checkha (double)   +/- hour angle for which the config. is valid.
 *
 *  Returns:
 *    An SDS id refering to the structure which contains the
 *    contants details for the current field. Returns an error
 *    message if the file could not be opened (in which case the
 *    C function returns TCL_ERROR).
 *
 *   Support:
 *     Keith Shortridge, AAO
 *
 *-
 *   History:
 *     25th Sept 2000.  Original version. KS.
 *     28th Aug  2002.  Mode code now returned. KS.
 */
{
    SdsIdType FileId;
    StatusType status;
    PAF_VLTGuideInfoType GuideInfo;
    int ModeCode;
    double checkHA;
        
    /*  Initial checks. Make sure we've been invoked properly */
    
    if (argc != 6) {
        interp->result = "wrong # args";
        return TCL_ERROR;
    }
    if (Instrument == NULL) {
        interp->result = "No instrument specified";
        return TCL_ERROR;
    }
      
    /*  Most of the work is done by the routines defined in convertPAF.c */
    
    status = STATUS__OK;
    PAF_CreateSds (Instrument,argv[1],InstFieldDetails,&FileId,&GuideInfo,
                   &ModeCode,&checkHA,&status);
    if (status == STATUS__OK) {
        char Buffer[64];
        sprintf(Buffer, "%d", GuideInfo.SelGuideIndex);
        Tcl_SetVar(interp, argv[2], Buffer, 0);
        Tcl_SetVar(interp, argv[3], GuideInfo.Posneg, 0);
        sprintf(Buffer, "%d", ModeCode);
        Tcl_SetVar(interp, argv[4], Buffer, 0);
        sprintf(Buffer, "%.1f", checkHA);
        Tcl_SetVar(interp, argv[5], Buffer, 0);
        sprintf(interp->result, "%ld", (long int) FileId);
        return TCL_OK;
    } else {
        char Buff[100];
        MessGetMsg(status, 0, sizeof(Buff), Buff);
        sprintf(interp->result, "Error in readPAF\n%s", Buff);
        return TCL_ERROR;
    }
}


/*  ------------------------------------------------------------------------- */

/*
 *                  M a k e  D S S  L i s t i n g  { }
 */

static int ConfMakeDSSListing(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])

/*+
 *  Command Name:
 *    MakeDSSListing
 *
 *  Function:
 *    Output listing positions in DSS format
 *
 *  Call:
 *    MakeDSSListing id file type
 *
 *  Description:
 *    Output a text file listing of the allocations or objects
 *    or both in an Sds structure. The format of the listing of
 *    unallocated objects is in DSS format.
 *
 *  Parameters:
 *    (>)  id    (Sds Id)  Sds identifier of the structure
 *    (>)  file  (char)    Name of the file to be created
 *    (>)  type  (char)    Information to output
 *                         dsssky =>  Allocated Skys only
 *                         dssall =>  All allocated objects
 *
 *   Returns:
 *     "Error" - if the file could not be opened.
 *
 *   Support:
 *     Tony Farrell, AAO
 *-
 */
{
    SdsIdType topid, objid, fid;
    StatusType status;
    FILE *listing;
    char iname[16];
    SdsCodeType code;
    long ndims;
    unsigned long dims[7];
    int skyonly;
    SdsIdType ra_id, dec_id, name_id, type_id, spect_id, prio_id, mag_id,
        pid_id, comment_id;
    unsigned long i;
    char mbuf[100];

    if (argc != 4) {
        interp->result = "wrong # args";
        return TCL_ERROR;
    }
    if (strcmp(argv[3], "dsssky") == 0)
        skyonly = 1;
    else
        skyonly = 0;

    status = STATUS__OK;
    topid = (SdsIdType) (atol(argv[1]));
    SdsFind(topid, "fieldData", &fid, &status);
    listing = fopen(argv[2], "w");
    if (listing == NULL) {
        ErsOut(0, &status, "Error Opening File %s", argv[2]);
        interp->result = "Error";
        return TCL_OK;
    }

    SdsFind(topid, "objects", &objid, &status);
    ConfGetIds(objid, &ra_id, &dec_id, &name_id, &type_id, &spect_id, &prio_id,
        &mag_id, &pid_id, &comment_id, &status);
    SdsInfo(ra_id, iname, &code, &ndims, dims, &status);
    for (i = 0; i < dims[0]; i++) {
        ConfListDSSLine(skyonly, listing, i, ra_id, dec_id, name_id, type_id,
            prio_id, mag_id, pid_id, comment_id, &status);
        if (status != STATUS__OK)
            break;
    }
    SdsFreeId(name_id, &status);
    SdsFreeId(ra_id, &status);
    SdsFreeId(dec_id, &status);
    SdsFreeId(type_id, &status);
    if (spect_id)
        SdsFreeId(spect_id, &status);
    SdsFreeId(prio_id, &status);
    SdsFreeId(mag_id, &status);
    SdsFreeId(pid_id, &status);
    SdsFreeId(comment_id, &status);
    SdsFreeId(objid, &status);

    SdsFreeId(fid, &status);
    fclose(listing);
    if (status == STATUS__OK)
        return TCL_OK;
    else {
        MessGetMsg(status, 0, 100, mbuf);
        sprintf(interp->result, "Error in MakeDSSListing\n%s", mbuf);
        return TCL_ERROR;
    }
}



/*  ------------------------------------------------------------------------- */

/*                        C o n f  G e t  I d s
 *
 *   Many routines in this file are passed the Sds Id of a substructure
 *   of the configuration data and then need to access all the items in
 *   that structure. This routine packages up access to the Ra, Dec, Name,
 *   Type, Spectrograph, Priority, Pid and Comment array items. Of these,
 *   not all sub-structures have a Spectrograph item, so the calling routine
 *   should be prepared to have an Sds Id of zero returned for Spectrograph,
 *   which indicates that it was not found in the sub-structure.
 */
 
static void ConfGetIds(SdsIdType topid, SdsIdType * ra_id, SdsIdType * dec_id,
    SdsIdType * name_id, SdsIdType * type_id,
    SdsIdType * spect_id, SdsIdType * prio_id,
    SdsIdType * mag_id, SdsIdType * pid_id, SdsIdType * comment_id,
    StatusType * status)
 {
    StatusType spectStatus;

    SdsFind(topid, "ra", ra_id, status);
    SdsFind(topid, "dec", dec_id, status);
    SdsFind(topid, "name", name_id, status);
    SdsFind(topid, "type", type_id, status);
    SdsFind(topid, "priority", prio_id, status);
    SdsFind(topid, "magnitude", mag_id, status);
    spectStatus = STATUS__OK;
    SdsFind(topid, "spectrograph", spect_id, &spectStatus);
    if (spectStatus != STATUS__OK) *spect_id = 0;
    SdsFind(topid, "pId", pid_id, status);
    SdsFind(topid, "comment", comment_id, status);
}

/*  ------------------------------------------------------------------------- */

/*                   C o n f  L i s t  P A F  T a r g e t
 *
 *  This routine outputs the details of one target object in the current
 *  sub-structure.  This just packages up a piece of code that would
 *  otherwise reccur in the code for WritePAF().
 */

static void ConfListPAFTarget(FILE * listing, int i,
    char* prefix,
    SdsIdType ra_id,
    SdsIdType dec_id,
    SdsIdType name_id, SdsIdType type_id, SdsIdType spect_id,
    SdsIdType prio_id, SdsIdType mag_id, SdsIdType pid_id,
    SdsIdType comment_id, 
    int fibre,
    int PAF_Flags,
    StatusType * status)
 {

    char name[NAMELEN], comment[CMTLEN];
    double mag;
    char otype;
    char ospect;
    short prio, pid;
    unsigned long actlen;
    double ra, dec;
    int rhmsf[4];
    int ddmsf[4];
    char sign[1];
    char typeString[32];
    char typeText[80];
    char keyword[80];
    char value[80];
    char *nextChar;
    int key;
    
    if (*status != STATUS__OK) return;
    
    SdsGet(name_id, NAMELEN, NAMELEN * i, name, &actlen, status);
    SdsGet(ra_id, sizeof(double), i, &ra, &actlen, status);
    SdsGet(dec_id, sizeof(double), i, &dec, &actlen, status);
    slaDr2tf(3, ra, sign, rhmsf);
    slaDr2af(2, dec, sign, ddmsf);
    SdsGet(type_id, sizeof(char), i, &otype, &actlen, status);
    if (spect_id) {
        SdsGet(type_id, sizeof(char), i, &ospect, &actlen, status);
    } else {
        ospect = 0;
    }
    FpilDecodeTargetType(*Instrument, otype, ospect, TRUE, typeString,
                                                               typeText);
    SdsGet(prio_id, sizeof(short), i, &prio, &actlen, status);
    SdsGet(mag_id, sizeof(double), i, &mag, &actlen, status);
    SdsGet(pid_id, sizeof(short), i, &pid, &actlen, status);
    SdsGet(comment_id, CMTLEN, CMTLEN * i, comment, &actlen, status);
    if (*status != STATUS__OK) return;

    fprintf (listing,"%sNAME            \"%s\";\n",prefix,name);
    fprintf (listing,"%sRA              %2.2d%2.2d%2.2d.%2.2d;\n",
                    prefix,rhmsf[0], rhmsf[1], rhmsf[2], rhmsf[3] / 10);
    if (sign[0] == '-') {
       fprintf (listing,"%sDEC             -%2.2d%2.2d%2.2d.%1.1d;\n",
                    prefix,ddmsf[0], ddmsf[1], ddmsf[2], ddmsf[3] / 10);
    } else {
       fprintf (listing,"%sDEC             %2.2d%2.2d%2.2d.%1.1d;\n",
                    prefix,ddmsf[0], ddmsf[1], ddmsf[2], ddmsf[3] / 10);
    }  
    fprintf (listing,"%sTYPE            \"%s\";\n",prefix,typeString);
    if ((PAF_Flags & PAF_NO_PRIO) == 0) {
        fprintf (listing,"%sPRIOR           %d;\n",prefix,prio);
    }
    fprintf (listing,"%sMAG             %f;\n",prefix,mag);
    fprintf (listing,"%sPRGID           %d;\n",prefix,pid);
    if (fibre >= 0) {
        fprintf (listing,"%sFIBRE           %d;\n",prefix,fibre);
    }
    
    /*  Now we look in the comment string for any of the auxiliary keywords.
     *  We cannot have a null value, so if we're passed one, ignore the 
     *  keyword.
     */
    
    nextChar = comment;
    do {
        char* commentPtr = nextChar;
        key = ConfParseComment (commentPtr, keyword, value, &nextChar);
        if (key) {
            if (strlen(value) > 0) {
                if (!strncmp(keyword,"PM_RA",5)) {
                    printf("correcting PM_RA with value %s for %s\n",value,name);
                    sprintf(value,"%*f",(int) strlen(value),atof(value)/cos(dec));
                }
                fprintf (listing,"%s%s        %s;\n",prefix,keyword,value);
            }
        }
    } while (key);
    fprintf (listing,"%sCOMMENT         \"%s\";\n",prefix,nextChar);
   
    
}

/*  ------------------------------------------------------------------------- */

/*                     C o n f  L i s t  L i n e
 *
 *  This routine outputs the details of one target object in the current
 *  sub-structure to the listing file.  This is a utility routine for use
 *  by ConfMakeListing().
 */

static void ConfListLine(FILE * listing, int i, SdsIdType ra_id,
    SdsIdType dec_id,
    SdsIdType name_id, SdsIdType type_id, SdsIdType spect_id,
    SdsIdType prio_id, SdsIdType mag_id, SdsIdType pid_id,
    SdsIdType comment_id, StatusType * status)
 {

    char name[NAMELEN], comment[CMTLEN];
    double mag;
    char otype;
    char ospect;
    short prio, pid;
    unsigned long actlen;
    double ra, dec;
    int rhmsf[4];
    int ddmsf[4];
    char sign[1];
    char typeString[32];
    char typeText[80];

    SdsGet(name_id, NAMELEN, NAMELEN * i, name, &actlen, status);
    SdsGet(ra_id, sizeof(double), i, &ra, &actlen, status);
    SdsGet(dec_id, sizeof(double), i, &dec, &actlen, status);
    slaDr2tf(3, ra, sign, rhmsf);
    slaDr2af(2, dec, sign, ddmsf);
    SdsGet(type_id, sizeof(char), i, &otype, &actlen, status);
    if (spect_id) {
        SdsGet(type_id, sizeof(char), i, &ospect, &actlen, status);
    } else {
        ospect = 0;
    }
    FpilDecodeTargetType(*Instrument, otype, ospect, TRUE, typeString,
                                                               typeText);
    SdsGet(prio_id, sizeof(short), i, &prio, &actlen, status);
    SdsGet(mag_id, sizeof(double), i, &mag, &actlen, status);
    SdsGet(pid_id, sizeof(short), i, &pid, &actlen, status);
    SdsGet(comment_id, CMTLEN, CMTLEN * i, comment, &actlen, status);
    if (*status != STATUS__OK)
        return;

    fprintf(listing,"%12s %2.2d %2.2d %2.2d.%3.3d  %c%2.2d %2.2d %2.2d.%2.2d ",
        name, rhmsf[0], rhmsf[1], rhmsf[2], rhmsf[3], sign[0],
        ddmsf[0], ddmsf[1], ddmsf[2], ddmsf[3]);
    fprintf(listing,"%s %d %6.2f %d %s\n",typeString, prio, mag, pid, comment);
}

/*  ------------------------------------------------------------------------- */

/*                     C o n f  L i s t  D S S  L i n e
 *
 *  This routine outputs the details of one target object in the current
 *  sub-structure to the listing file as required by the DSS format.  This
 *  is a utility routine for use by ConfMakeDSSListing().
 */

static void ConfListDSSLine(int skyonly,
    FILE * listing, int i, SdsIdType ra_id,
    SdsIdType dec_id,
    SdsIdType name_id, SdsIdType type_id, SdsIdType prio_id,
    SdsIdType mag_id, SdsIdType pid_id, SdsIdType comment_id,
    StatusType * status)
 {

    char name[NAMELEN], comment[CMTLEN];
    double mag;
    char otype;
    short prio, pid;
    unsigned long actlen;
    double ra, dec;
    int rhmsf[4];
    int ddmsf[4];
    char sign[1];


    SdsGet(type_id, sizeof(char), i, &otype, &actlen, status);
    if ((skyonly) && (!FpilIsTargetSky(*Instrument, otype)))
        return;

    SdsGet(name_id, NAMELEN, NAMELEN * i, name, &actlen, status);
    SdsGet(ra_id, sizeof(double), i, &ra, &actlen, status);
    SdsGet(dec_id, sizeof(double), i, &dec, &actlen, status);
    slaDr2tf(3, ra, sign, rhmsf);
    slaDr2af(2, dec, sign, ddmsf);
    SdsGet(prio_id, sizeof(short), i, &prio, &actlen, status);
    SdsGet(mag_id, sizeof(double), i, &mag, &actlen, status);
    SdsGet(pid_id, sizeof(short), i, &pid, &actlen, status);
    SdsGet(comment_id, CMTLEN, CMTLEN * i, comment, &actlen, status);
    if (*status != STATUS__OK)
        return;

    if (i >= 99) {
        fprintf(listing, "S%3d ", i + 1);
    } else {
        if (i >= 9) {
            fprintf(listing, "S0%2d ", i + 1);
        } else {
            fprintf(listing, "S00%1d ", i + 1);
        }
    }
    fprintf(listing,
        "%2.2d %2.2d %2.2d.%3.3d  %c%2.2d %2.2d %2.2d.%2.2d 2. 2.\n",
        rhmsf[0], rhmsf[1], rhmsf[2], rhmsf[3], sign[0],
        ddmsf[0], ddmsf[1], ddmsf[2], ddmsf[3]);
}


/*  ------------------------------------------------------------------------- */

/*
 *                        G e t  F i b r e  { }
 */

static int ConfGetFibre(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])
    
/*+
 *  Command Name:
 *    GetFibre
 *
 *  Function:
 *    Return Information about a fibre
 *
 *  Call:
 *    GetFibre id fibre
 *
 *  Description:
 *    Return information associated with a fibre. 9 items are returned in the
 *    result string. These are as follows - the index numbers given are those
 *    that Tcl will use for lindex when it extracts these from the string.
 *
 *    0     x coordinate
 *    1     y coordindate
 *    2     Button angle (theta)
 *    3     The object type (target type code character, or the unallocated
 *                                                                    code)
 *    4     Fibre Type (the numeric code for the fibre type)
 *    5     Object Priority
 *    6     Object Name
 *    7     Fibre type (a descriptive string for the fibre type)
 *    8     Integer 0/1 indicating if the we are allowed to allocate/unallocate
 *                                                                  this fibre.
 *    9     Pivot position in x
 *   10     Pivot position in y
 *   11     Park position in x
 *   12     Park position in y
 *
 *  Note that the 'spect' code associated with the target to which the fibre
 *  is allocated (if it is indeed allocated) is not supplied by this code.
 *  This is because this is not included in the "objects" structure that is
 *  used by this routine.  (Once the target is allocated, this is regarded as
 *  being irrelevant, but it does mean that this routine strictly doesn't
 *  return enough information to reproduce the original object type exactly -
 *  for example, in 2dF the 'spect' field might have specified the spectrograph
 *  that matches the fibre type, or it might have been a zero code to indicate
 *  that either spectrograph would do - we can't tell this. 
 *
 *  Parameters:
 *    (>)  id    (Sds Id)  Sds identifier of the structure
 *    (>)  fibre (Sds Id)  Fibre (pivot) number - starting at one
 *
 *  Returns:
 *    List contaning the values.
 *
 *   Support:
 *     Jeremy Bailey, AAO
 *
 *   Version Date:
 *     6-Oct-1994
 *    13-Apr-2001  KS  Added pivot and park positions to the end of the list.
 *    19-Dec-2001  KS  Ensured that the type description doesn't contain blanks
 *                     as this confuses the tcl lindex operation.   
 *-
 */
{
    StatusType status;
    long fibre_no;
    SdsIdType xid = 0, yid = 0, thetaid = 0, typeid = 0;
    SdsIdType topid = 0;
    SdsIdType idf = 0;
    int x, y, fibreType;
    int xp, yp, xpark, ypark;
    short oprio;
    char oname[NAMELEN];
    char FibreTypeString[64];
    int length, i;
    double theta;
    unsigned long actlen;
    char mbuf[100];
    char type;
    const char *FibreTypeDescr;
    StatusType ignore = STATUS__OK;

    if (argc != 3) {
        interp->result = "wrong # args";
        return TCL_ERROR;
    }
    status = SDS__OK;
    idf = (SdsIdType) (atol(argv[1]));
    SdsFind(idf, "objects", &topid, &status);
    fibre_no = atol(argv[2]) - 1;

    if ((fibre_no < 0) || (fibre_no >= (int) FpilGetNumPivots(*Instrument))) {
        sprintf(interp->result,
            "Illegal Pivot Number %ld\nMust be between 1 and %d",
            fibre_no + 1, FpilGetNumPivots(*Instrument));
        return TCL_ERROR;
    }
    
    /*  We can get pivot information directly from the instrument field
     *  details structure.
     */
     
    ConfArrayGeti(InstFieldDetails, "xPiv", fibre_no, &xp, &status);
    ConfArrayGeti(InstFieldDetails, "yPiv", fibre_no, &yp, &status);
    ConfArrayGeti(InstFieldDetails, "xPark", fibre_no, &xpark, &status);
    ConfArrayGeti(InstFieldDetails, "yPark", fibre_no, &ypark, &status);

    /*  Now get the rest of the information needed */
    
    SdsFind(topid, "x", &xid, &status);
    SdsFind(topid, "y", &yid, &status);
    SdsFind(topid, "type", &typeid, &status);
    if (status != SDS__OK) {
        status = SDS__OK;
        SdsFind(topid, "xf", &xid, &status);
        SdsFind(topid, "yf", &yid, &status);
    }
    SdsFind(topid, "theta", &thetaid, &status);
    SdsGet(xid, 4, fibre_no, &x, &actlen, &status);
    SdsGet(yid, 4, fibre_no, &y, &actlen, &status);
    SdsGet(thetaid, 8, fibre_no, &theta, &actlen, &status);
    SdsGet(typeid, 1, fibre_no, &type, &actlen, &status);
    if (type != FpilUnallocatedType(*Instrument)) {
        SdsFind(topid, "name", &xid, &status);
        SdsGet(xid, NAMELEN, fibre_no * NAMELEN, &oname, &actlen, &status);
        SdsFind(topid, "priority", &xid, &status);
        SdsGet(xid, sizeof(short), fibre_no, &oprio, &actlen, &status);
    } else {
        oprio = 0;
        sprintf(oname, "\"\"");
    }
    fibreType = 0;
    fibreType = ConfWhichSpec((int) fibre_no, &FibreTypeDescr);

    SdsFreeId(topid, &ignore);
    SdsFreeId(xid, &ignore);
    SdsFreeId(yid, &ignore);
    SdsFreeId(typeid, &ignore);
    SdsFreeId(thetaid, &ignore);
    
    strncpy(FibreTypeString,FibreTypeDescr,sizeof(FibreTypeString));
    FibreTypeString[sizeof(FibreTypeString) - 1] = '\0';
    length = strlen(FibreTypeString);
    for (i = 0; i < length; i++) {
       if (FibreTypeString[i] == ' ') FibreTypeString[i] = '_';
    }

    if (status == SDS__OK) {
        sprintf(interp->result, "%d %d %f %c %d %d %s %s %d %d %d %d %d",
            x, y, theta, type, fibreType, oprio, oname, FibreTypeString,
            UseFibreType[fibreType],xp,yp,xpark,ypark);
        return TCL_OK;
    } else {
        MessGetMsg(status, 0, 100, mbuf);
        sprintf(interp->result, "Error Getting Fibre Data\n%s", mbuf);
        return TCL_ERROR;
    }
}

/*  ------------------------------------------------------------------------- */

/*
 *                        S e t  F i b r e  { }
 */
 
static int ConfSetFibre(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])

/*+
 *  Command Name:
 *     SetFibre
 *
 *  Function:
 *     Set Information about a fibre
 *
 *  Call:
 *     SetFibre cid sid
 *
 *  Description:
 *     This routine was uncommented in the original code, and I (KS) believe
 *     that it is no longer used. It seems to expect a specific structure
 *     as a second argument with 'pivot', 'x', 'y' and 'theta' items. It 
 *     uses the pivot value as an index and sets the corresponding item in
 *     the 'objects' sub-structure of the configuration data structure to
 *     the x, y and theta values supplied in the second argument.
 *
 *  Parameters:
 *    (>)  icd    (Sds Id)  Sds identifier of the configuration data structure
 *    (>)  sid    (Sds Id)  Sds identifier of a structure containing 'pivot',
 *                          'x', 'y' and 'theta' items.
 *-
 */
{
    StatusType status;
    long fibre_no;
    SdsIdType xid = 0, yid = 0, thetaid = 0;
    SdsIdType topid = 0, idf = 0;
    SdsIdType arg;
    long x, y;
    int xi, yi;
    double theta;
    char mbuf[100];
    StatusType ignore = STATUS__OK;
    LeakCheckType LL;
    LL = LeakPrep();

    if (argc != 3) {
        interp->result = "wrong # args";
        return TCL_ERROR;
    }
    status = SDS__OK;
    idf = (SdsIdType) (atol(argv[1]));
    SdsFind(idf, "objects", &topid, &status);
    arg = atol(argv[2]);
    ArgGeti(arg, "pivot", &fibre_no, &status);
    ArgGeti(arg, "x", &x, &status);
    ArgGeti(arg, "y", &y, &status);
    xi = (int) x;
    yi = (int) y;
    fibre_no = fibre_no - 1;
    if ((fibre_no < 0) || (fibre_no >= (int) FpilGetNumPivots(*Instrument))) {
        sprintf(interp->result,
            "Illegal Pivot Number %ld\nMust be between 1 and %d",
            fibre_no + 1, FpilGetNumPivots(*Instrument));
        return TCL_ERROR;
    }

    ArgGetd(arg, "theta", &theta, &status);
    SdsFind(topid, "xf", &xid, &status);
    SdsFind(topid, "yf", &yid, &status);
    SdsFind(topid, "theta", &thetaid, &status);
    SdsPut(xid, 4, fibre_no, &xi, &status);
    SdsPut(yid, 4, fibre_no, &yi, &status);
    SdsPut(thetaid, 8, fibre_no, &theta, &status);

    SdsFreeId(topid, &ignore);
    SdsFreeId(xid, &ignore);
    SdsFreeId(yid, &ignore);
    SdsFreeId(thetaid, &ignore);
    LeakCheck("ConfSetFibre", LL, 0);

    if (status == SDS__OK) {
        return TCL_OK;
    } else {
        MessGetMsg(status, 0, 100, mbuf);
        sprintf(interp->result, "Error Setting Fibre Data\n%s", mbuf);
        return TCL_ERROR;
    }
}

/*  ------------------------------------------------------------------------- */

/*
 *                 S e t  B u t t o n  A n g l e  { }
 */
 
static int ConfSetButtonAngle(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])

/*+
 *  Command Name:
 *    SetButtonAngle
 *
 *  Function:
 *    Set the angle of a button
 *
 *  Call:
 *    SetButtonAngle id pivid fibre angle
 *
 *  Description:
 *    Set the angle of a button to a specified value, or to the radial
 *    value.
 *
 *
 *  Parameters:
 *    (>)  id    (Sds Id)  Sds identifier of the structure
 *    (>)  pivid (Sds Id)  Sds identifier of the pivot structure
 *    (>)  fibre (int)     Fibre (pivot) number - starting at one
 *    (>)  theta (char)    Either the required button angle in radians,
 *                         or the word "radial".
 *
 *   Support:
 *     Jeremy Bailey, AAO
 *
 *   Version Date:
 *     6-Oct-1994    
 *-
 */
{
    StatusType status;
    long fibre_no;
    SdsIdType topid, idf, pivid;
    int xp, yp, x, y;
    double theta;
    char mbuf[100];

    if (argc != 5) {
        interp->result = "wrong # args";
        return TCL_ERROR;
    }
    status = SDS__OK;
    idf = (SdsIdType) (atol(argv[1]));
    pivid = (SdsIdType) (atol(argv[2]));
    SdsFind(idf, "objects", &topid, &status);
    fibre_no = atol(argv[3]);
    fibre_no = fibre_no - 1;
    if ((fibre_no < 0) || (fibre_no >= (int) FpilGetNumPivots(*Instrument))) {
        sprintf(interp->result,
            "Illegal Pivot Number %ld\nMust be between 1 and %d",
            fibre_no + 1, FpilGetNumPivots(*Instrument));
        return TCL_ERROR;
    }
    if (strcmp(argv[4], "radial") == 0) {

/*  Calculate theta so that the button is parallel to the fibre  */

        ConfArrayGeti(pivid, "xPiv", fibre_no, &xp, &status);
        ConfArrayGeti(pivid, "yPiv", fibre_no, &yp, &status);
        ConfArrayGeti(topid, "x", fibre_no, &x, &status);
        ConfArrayGeti(topid, "y", fibre_no, &y, &status);
        theta = slaDranrm(atan2((double) (yp - y), (double) (xp - x)) - DPIBY2);
    } else
        theta = atof(argv[4]);

    ConfArrayPutd(topid, "theta", fibre_no, theta, &status);
    SdsFreeId(topid, &status);

    if (status == SDS__OK) {
        return TCL_OK;
    } else {
        MessGetMsg(status, 0, 100, mbuf);
        sprintf(interp->result, "Error Setting Button Angle\n%s", mbuf);
        return TCL_ERROR;
    }
}


/*  ------------------------------------------------------------------------- */

/*
 *                        G e t  O b j e c t  D a t a  { }
 */
 
static int ConfGetObjectData(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])

/*+
 *  Command Name:
 *    GetObjectData
 *
 *  Function:
 *    Get information about an object
 *
 *  Call:
 *    GetObjectData id index name ra dec type spect priority mag pid comment
 *
 *  Description:
 *    Return information about an object in any of the three sections
 *    of the SDS structure (allocated objects, unallocated objects,
 *    unallocated guides).
 *
 *
 *  Parameters:
 *    (>)  id    (Sds Id)  Sds identifier of the substructure
 *    (>)  index (int)     index number - starting at one
 *    (<)  name  (char)    The object name.
 *    (<)  ra    (double)  RA in radians.
 *    (<)  dec   (double)  Dec in radians.
 *    (<)  type  (char)    Type of object. See below for possible values.
 *    (<)  spect (char)    The spect code associated with the object. See below.
 *    (<)  priority (int)  Priority of object.
 *    (<)  mag   (double)  Magnitude of object.
 *    (<)  pid   (int)     Program id.
 *    (<)  text  (char)    Text describing the object type.
 *    (<)  comment (char)  Comment String.
 *    (<)  x     (int)     Position in X.
 *    (<)  y     (int)     Position in Y.
 *
 *    The value returned in the type field is one of the characters
 *    defined as possible target type codes for the instrument. This will
 *    be one of the type codes returned by FpilEncodeTargetType(), or (for
 *    items in the "objects" section, may be the 'unallocated' type code
 *    as returned by FpilUnallocatedType(). The value returned in the type
 *    field is the spect code returned by FpilEncodeTargetType() for the
 *    target.
 *
 *   Support:
 *     Jeremy Bailey, AAO
 *
 *   Version Date:
 *     6-Oct-1994
 *    13-Oct-2001  KS  Added Y and Y positions to argument list.  
 *-
 */
{
    StatusType status, notfound;
    long index_no;
    SdsIdType id;
    SdsIdType topid;
    char name[NAMELEN];
    double ra, dec;
    char type;
    char spect;
    short priority;
    double mag;
    short pid;
    int spectKnown;
    char comment[CMTLEN];
    unsigned long actlen;
    char mbuf[100];
    char buffer[80];
    char targetString[32];
    char targetText[80];
    int x,y;

    if (argc != 15) {
        interp->result = "wrong # args";
        return TCL_ERROR;
    }
    status = SDS__OK;
    topid = (SdsIdType) (atol(argv[1]));
    index_no = atol(argv[2]) - 1;

    SdsFind(topid, "name", &id, &status);
    SdsGet(id, NAMELEN, index_no * NAMELEN, name, &actlen, &status);
    SdsFreeId(id, &status);
    SdsFind(topid, "ra", &id, &status);
    SdsGet(id, sizeof(double), index_no, &ra, &actlen, &status);
    SdsFreeId(id, &status);
    SdsFind(topid, "dec", &id, &status);
    SdsGet(id, sizeof(double), index_no, &dec, &actlen, &status);
    SdsFreeId(id, &status);
    SdsFind(topid, "type", &id, &status);
    SdsGet(id, sizeof(char), index_no, &type, &actlen, &status);
    SdsFreeId(id, &status);
    
    notfound = 0;
    spectKnown = FALSE;
    SdsFind(topid, "spect", &id, &notfound);
    if (notfound)
        spect = '\0';
    else {
        SdsGet(id, sizeof(char), index_no, &spect, &actlen, &status);
        SdsFreeId(id, &status);
        spectKnown = TRUE;
    }
    
    notfound = 0;
    SdsFind(topid, "priority", &id, &notfound);
    if (notfound)
        priority = 0;
    else {
        SdsGet(id, sizeof(short), index_no, &priority, &actlen, &status);
        SdsFreeId(id, &status);
    }

    notfound = 0;
    SdsFind(topid, "magnitude", &id, &notfound);
    if (notfound)
        mag = 0;
    else {
        SdsGet(id, sizeof(double), index_no, &mag, &actlen, &status);
        SdsFreeId(id, &status);
    }

    notfound = 0;
    SdsFind(topid, "pId", &id, &notfound);
    if (notfound)
        pid = 0;
    else {
        SdsGet(id, sizeof(short), index_no, &pid, &actlen, &status);
        SdsFreeId(id, &status);
    }

    notfound = 0;
    SdsFind(topid, "comment", &id, &notfound);
    if (notfound) {
        strcpy(comment, "No comment");
        actlen = 10;
    } else {
        SdsGet(id, CMTLEN, index_no * CMTLEN, comment, &actlen, &status);
        SdsFreeId(id, &status);
    }
    
    notfound = 0;
    SdsFind(topid, "x", &id, &notfound);
    if (notfound)
        x = 0;
    else {
        SdsGet(id, sizeof(x), index_no, &x, &actlen, &status);
        SdsFreeId(id, &status);
    }
    
    notfound = 0;
    SdsFind(topid, "y", &id, &notfound);
    if (notfound)
        y = 0;
    else {
        SdsGet(id, sizeof(y), index_no, &y, &actlen, &status);
        SdsFreeId(id, &status);
    }
    
    FpilDecodeTargetType (*Instrument,type,spect,spectKnown,targetString,
                                                                 targetText);

    if (status == SDS__OK) {
        Tcl_SetVar(interp, argv[3], name, 0);
        sprintf(buffer, "%10f", ra);
        Tcl_SetVar(interp, argv[4], buffer, 0);
        sprintf(buffer, "%10f", dec);
        Tcl_SetVar(interp, argv[5], buffer, 0);
        sprintf(buffer, "%c", type);
        Tcl_SetVar(interp, argv[6], buffer, 0);
        sprintf(buffer, "%d", spect);
        Tcl_SetVar(interp, argv[7], buffer, 0);
        sprintf(buffer, "%d", priority);
        Tcl_SetVar(interp, argv[8], buffer, 0);
        sprintf(buffer, "%.2f", mag);
        Tcl_SetVar(interp, argv[9], buffer, 0);
        sprintf(buffer, "%d", pid);
        Tcl_SetVar(interp, argv[10], buffer, 0);
        Tcl_SetVar(interp, argv[11], comment, 0);
        Tcl_SetVar(interp, argv[12], targetText, 0);
        sprintf(buffer, "%d", x);
        Tcl_SetVar(interp, argv[13], buffer, 0);
        sprintf(buffer, "%d", y);
        Tcl_SetVar(interp, argv[14], buffer, 0);

        return TCL_OK;
    } else {
        MessGetMsg(status, 0, 100, mbuf);
        sprintf(interp->result, "Error Getting Object Data\n%s", mbuf);
        return TCL_ERROR;
    }
}

/*  ------------------------------------------------------------------------- */

/*
 *                        G e t  O b j e c t  { }
 */
 
static int ConfGetObject(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])

/*+
 *
 *  Command name:
 *    GetObject
 *
 *
 *  Function:
 *    Get object information for mimic display
 *  
 *  Call:
 *    GetObject id index zoom [tscale]
 *
 *  Description:
 *    Return the information needed to draw an object
 *    on the mimic display.
 *   
 *
 *   Parameters:
 *
 *    (>) id      (Sds Id)  Sds identifier of the substructure   
 *    (>) index   (int)     Index number in structure
 *    (>) zoom    (float)   Zoom factor for overall display
 *    (>) tscale  (float)   Scale factor for targets (optional -
 *                          defaults to 1.0).
 *
 *   Returns:
 *    The argument string of a Tk canvas create command which will
 *    draw the object on the mimic display.
 *
 *   Support:
 *     Jeremy Bailey, AAO
 *
 *   Version Date:
 *     6-Oct-1994    
 *-
 */
{
    StatusType status;
    StatusType spectStatus;
    long index_no;
    SdsIdType xid = 0, yid = 0, tid = 0, sid = 0;
    SdsIdType topid = 0;
    int x, y;
    unsigned long actlen;
    float zoom;
    char mbuf[100];
    char type;
    char spect;
    float tscale;
    SdsIdType allid = 0;
    char c;

    if (argc != 4 && argc != 5) {
        interp->result = "wrong # args";
        return TCL_ERROR;
    }
    if (Instrument == NULL) {
        interp->result = "No instrument specified";
        return TCL_ERROR;
    }
    
    status = SDS__OK;
    topid = (SdsIdType) (atol(argv[1]));
    index_no = atol(argv[2]) - 1;
    zoom = atof(argv[3]);
    tscale = 1.0;
    if (argc == 5) tscale = atof(argv[4]);
    SdsFind(topid, "allocated", &allid, &status);
    if (status == SDS__NOITEM)
        ErsAnnul(&status);
    else {
        SdsGet(allid, 1, index_no, &c, &actlen, &status);
        SdsFreeId(allid, &status);
        allid = 0;
        if (c == 1) {
            sprintf(interp->result, "unused");
            return TCL_OK;
        }
    }

    SdsFind(topid, "x", &xid, &status);
    SdsFind(topid, "y", &yid, &status);
    if (status == SDS__NOITEM) {
        ErsAnnul(&status);
        SdsFind(topid, "xf", &xid, &status);
        SdsFind(topid, "yf", &yid, &status);
    }
    SdsFind(topid, "type", &tid, &status);
    SdsGet(xid, 4, index_no, &x, &actlen, &status);
    SdsGet(yid, 4, index_no, &y, &actlen, &status);
    SdsGet(tid, 1, index_no, &type, &actlen, &status);
    if (status != SDS__OK) {
        StatusType ignore = STATUS__OK;
        MessGetMsg(status, 0, 100, mbuf);
        if (xid != 0)
            SdsFreeId(xid, &ignore);
        if (xid != 0)
            SdsFreeId(yid, &ignore);
        if (tid != 0)
            SdsFreeId(tid, &ignore);
        sprintf(interp->result, "Error Getting Fibre Data\n%s", mbuf);
        return TCL_ERROR;
    }

    /*  Not all targets have spectrograph info - guide targets don't, so we
     *  aren't surprised if we can't find a spectrograph code.
     */

    spectStatus = 0;
    SdsFind(topid, "spectrograph", &sid, &spectStatus);
    if (spectStatus == 0) {
        SdsGet(sid, 1, index_no, &spect, &actlen, &spectStatus);
        if (spectStatus != 0)
            spect = 0;
        SdsFreeId(sid, &spectStatus);
    }

    if (type == FpilUnallocatedType(*Instrument)) {
        sprintf(interp->result, "unused");
    } else {   

        FpilDrawTarget(*Instrument, type, spect, x, y, zoom, tscale,
              MimicCircleRadius, MimicCircleOffset, useGrey, interp->result);
                          
    }

    SdsFreeId(xid, &status);
    SdsFreeId(yid, &status);
    SdsFreeId(tid, &status);

    return TCL_OK;

}

/*  ------------------------------------------------------------------------- */

/*
 *                        C o n v e r t  A s c i i  { }
 */
 
static int ConfConvertAscii(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])

/*+
 *
 *  Command name:
 *    ConvertAscii
 *
 *
 *  Function:
 *    Convert input text file to SDS
 *  
 *  Call:
 *    ConvertAscii file fracDayFlag [lowmag highmag]
 *
 *  Description:
 *    Convert a text file description of a configuration to its
 *    SDS representation and return the SDS identifier
 *   
 *   Parameters:
 *
 *    (>) file        (Char)  Name of the text file 
 *    (<) fracDayFlag (int)   True (1) if UTDATE in file specified a fractional
 *                            day value - ie if a specific day AND time was
 *                            made explicit in the file.
 *    (>) lowmag      (float) Lower magnitude limit
 *    (>) highmag     (float) Higher magnitude limit
 *
 *   Lowmag and highmag are optional, but if one is given both are required.
 *   If specified, the input targets will be filtered and only those within
 *   the magnitude limits will be included. 
 *
 *   Returns:
 *     The SDS Id of the created structure.
 *
 *   Support:
 *     Jeremy Bailey, AAO
 *
 *   Version Date:
 *     6-Oct-1994
 *    21-May-2001  KS  Added lowmag, highmag and fracDayFlag.
 *    14-Aug-2001  KS  Added position angle code.   
 *-
 */
{
    tdFfieldType field;
    StatusType status = STATUS__OK;
    SdsIdType fileId = 0;
    short fracDayFlag = FALSE;
    short options = 0;
    double longit, lat, hm;
    FILE *asciiFile;
    char oname[NAMELEN];
    double uttime = 0.5;
    char buff[80];
    objectType *list, *next;
    float maglimits[2];
    int lineCount = 0;
    double Angle;
    double Offset;
    short InUse;
    char Scale[16];

    maglimits[0] = maglimits[1] = 0.0;
    if (argc == 5) {
        maglimits[0] = atof(argv[3]);
        maglimits[1] = atof(argv[4]);
    } else {
        if (argc != 3) {
            interp->result = "wrong # args";
            return TCL_ERROR;
        }
    }

    /*  We need to work out the uttime correction to allow for the difference
     *  in midnight UT and local time.
     */
     
    slaObs(-1, (char *) FpilGetTelescope(*Instrument), oname, &longit, &lat,
                                                                         &hm);
    if (strcmp(oname, "?") == 0) {
       uttime = 0.0;
    } else {
       uttime = -longit / (2.0 * 3.141593);
    }
    
    /*  Now open file and process the field data at the start of the file */
    
    if ((asciiFile = fopen(argv[1], "r")) == NULL) {
        sprintf(interp->result, "Cannot Open file %s", argv[1]);
        return TCL_ERROR;
    }
    tdFparseFieldData(Instrument, asciiFile, &field.fieldData, uttime,
                                          &fracDayFlag, &lineCount, &status);
                                          
    /*  If field.fieldData.argusSpec has been set, then ARGUS information 
     *  has been specified. We handle this through ConfCSetArgusData(), which
     *  will arrange to set the position angle as required.
     */
    
    ConfCGetArgusData(&InUse,&Angle,Scale,&Offset);
    Angle = field.fieldData.posangle + Offset;
    ConfCSetArgusData(field.fieldData.argusSpec,Angle,
                                        field.fieldData.scale,Offset);
        

    field.unallocGuide = field.unallocObject = NULL;
    tdFparseObjectData(Instrument, asciiFile, &field, &options, maglimits,
                                                         &lineCount, &status);

    fclose(asciiFile);

    tdFconvertToSds("2df_field", &field, options, &fileId, &status);

    /*
     * Free the two lists.  This is done in spite of the status, as they
     * were initialised above and will need to be cleaned up if they
     * are no longer null
     */
     
    list = field.unallocObject;
    while (list) {
        next = list->next;
        free(list);
        list = next;
    }
    list = field.unallocGuide;
    while (list) {
        next = list->next;
        free(list);
        list = next;
    }

    if (status == STATUS__OK) {
        if (fracDayFlag) {
            Tcl_SetVar(interp, argv[2], "1", 0);
        } else {
            Tcl_SetVar(interp, argv[2], "0", 0);
        }   
        sprintf(interp->result, "%d", (int) fileId);
        return TCL_OK;
    } else {
        ErsRep (0,&status,"Error reading file '%s'",argv[1]);
        MessGetMsg(status, 0, sizeof(buff), buff);
        Tcl_AppendResult(interp, "Error Converting Ascii File\n",
            (char *) NULL);
        Tcl_AppendResult(interp, buff, (char *) NULL);
        return TCL_ERROR;
    }
}

/*  ------------------------------------------------------------------------- */

/*
 *                  C o u n t  A s c i i  O b j e c t s  { }
 */
 
static int ConfCountAsciiObjects(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])

/*+
 *
 *  Command name:
 *    CountAsciiObjects
 *
 *
 *  Function:
 *    Pass through input text file counting objects
 *  
 *  Call:
 *    ConvertAscii file [lowmag highmag]
 *
 *  Description:
 *    This routine performs processing on an ASCII input file similar to
 *    that performed by ConvertAscii(), but only returns the number of
 *    objects that will be read from the file, given the magnitude
 *    limits (if any).  
 *
 *   Parameters:
 *
 *    (>) file        (Char)  Name of the text file 
 *    (>) lowmag      (float) Lower magnitude limit
 *    (>) highmag     (float) Higher magnitude limit
 *
 *   Lowmag and highmag are optional, but if one is given both are required.
 *   If specified, the input targets will be filtered and only those within
 *   the magnitude limits will be included. 
 *
 *   Returns:
 *     The number of objects that will be read from the file.
 *
 *   Support:
 *     Keith Shortridge, AAO
 *
 *   Version Date:
 *    10-Jun-2001  KS  Original version, based on ConfConvertAscii().    
 *-
 */
{
    tdFfieldType field;
    StatusType status = STATUS__OK;
    short fracDayFlag = FALSE;
    FILE *asciiFile;
    double uttime = 0.0;
    char buff[80];
    float maglimits[2];
    int Count;
    int lineCount = 0;

    maglimits[0] = maglimits[1] = 0.0;
    if (argc == 4) {
        maglimits[0] = atof(argv[2]);
        maglimits[1] = atof(argv[3]);
    } else {
        if (argc != 2) {
            interp->result = "wrong # args";
            return TCL_ERROR;
        }
    }

    if ((asciiFile = fopen(argv[1], "r")) == NULL) {
        sprintf(interp->result, "Cannot Open file %s", argv[1]);
        return TCL_ERROR;
    }
    tdFparseFieldData(Instrument, asciiFile, &field.fieldData, uttime,
                                           &fracDayFlag, &lineCount, &status);
    Count = tdFparseObjectCount(Instrument, asciiFile, maglimits, &status);

    fclose(asciiFile);

    if (status == STATUS__OK) {
        sprintf(interp->result, "%d", Count);
        return TCL_OK;
    } else {
        ErsRep (0,&status,"Error reading file '%s'",argv[1]);
        MessGetMsg(status, 0, sizeof(buff), buff);
        Tcl_AppendResult(interp, "Error Converting Ascii File\n",
            (char *) NULL);
        Tcl_AppendResult(interp, buff, (char *) NULL);
        return TCL_ERROR;
    }
}


/*  ------------------------------------------------------------------------- */

/*
 *                     S e t  U T  M e r i d i a n  { }
 */
 
static int ConfSetUTMeridian(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])
/*+
 *
 *  Command name:
 *    SetUTMeridian
 *
 *
 *  Function:
 *    Set the UT of a structure for a given Hour Angle
 *  
 *  Call:
 *    SetUTMeridian id [ha]
 *
 *  Description:
 *    Set the UT in a configuration structure to the value required to
 *    put the field at a specified hour angle for its configuration date.
 *   
 *
 *   Parameters:
 *
 *    (>) id   (Sds id)  Sds Id of configuration structure.
 *    (>) ha   (double)  Required hour angle - if omitted set to the meridian. 
 *
 *
 *   Support:
 *     Jeremy Bailey, AAO
 *
 *   Version Date:
 *     6-Oct-1994
 *    21-May-2001. Now uses ChangeMjd(). KS  
 *-
 */
{
    StatusType status = STATUS__OK;
    SdsIdType id = 0, idf = 0;
    double cra, mjd, lst;
    double longit, lat, hm;
    char oname[NAMELEN];
    char sign[1];
    int ihmsf[4];
    char buff[80];
    double ha;
    int iy, im, idd;
    double f;
    int jstat;
    char buffer[200];
    StatusType ignore = STATUS__OK;

    if (argc < 2) {
        sprintf(interp->result, "%s:Wrong number of arguments", argv[0]);
        return TCL_ERROR;
    }

    id = atol(argv[1]);
    if (argc < 3)
        ha = 0.0;
    else
        ha = DPI / 12 * (atof(argv[2]));


    /* The discard of const (FpilGetTelescope) is safe as when
       the first argument is -1, this address is not written too
     */
    slaObs(-1, (char *) FpilGetTelescope(*Instrument), oname, &longit, &lat,
        &hm);
    if (strcmp(oname, "?") == 0) {
        sprintf(interp->result,
            "%s:Telescope \"%s\" is not recognized by the slaObs routine.",
            argv[0], FpilGetTelescope(*Instrument));
        strcat(interp->result, "  This must be fixed");
        return TCL_ERROR;
    }

    SdsFind(id, "fieldData", &idf, &status);
    ArgGetd(idf, "appRa", &cra, &status);
    ArgGetd(idf, "configMjd", &mjd, &status);

    lst = slaGmst(mjd) - longit + slaEqeqx(mjd);
    mjd = mjd + (slaDrange(cra - lst + ha)) / D2PI / 1.0027379;
    ChangeMjd (idf, mjd, &status);
    slaDd2tf(0, mjd - floor(mjd), sign, ihmsf);
    slaDjcl(mjd, &iy, &im, &idd, &f, &jstat);

    if (argc < 3) {
        sprintf(buffer,
            "UT set to %.4d/%.2d/%.2d  %.2d:%.2d:%.2d to put field on meridian",
            iy, im, idd, ihmsf[0], ihmsf[1], ihmsf[2]);
    } else {
        sprintf(buffer,
          "UT set to %.4d/%.2d/%.2d %.2d:%.2d:%.2d to put field at HA %f hours",
            iy, im, idd, ihmsf[0], ihmsf[1], ihmsf[2], atof(argv[2]));
    }
    ConfMessage(interp, buffer);

    SdsFreeId(idf, &ignore);

    if (status == STATUS__OK) {
        return TCL_OK;
    } else {
        MessGetMsg(status, 0, sizeof(buff), buff);
        Tcl_AppendResult(interp, "Error setting UT to meridian value\n",
            (char *) NULL);
        Tcl_AppendResult(interp, buff, (char *) NULL);
        return TCL_ERROR;
    }
}


/*  ------------------------------------------------------------------------- */

/*
 *                        S e t  U T  { }
 */
 
static int ConfSetUT(ClientData clientData DUNUSED, Tcl_Interp * interp,
    int argc DUNUSED, char *argv[])
    
/*+
 *
 *  Command name:
 *    SetUT
 *
 *  Function:
 *    Set the UT Date of a configuration structure
 *  
 *  Call:
 *    SetUT id year month day
 *
 *  Description:
 *    Set the UT date in a configuration structure to a specified value.
 *   
 *   Parameters:
 *
 *    (>) id    (Sds id)  Sds Id of configuration structure.
 *    (>) year  (int)  Required date - year
 *    (>) month (int)  Required date - month 
 *    (>) day   (int/float) Required date - day. Note, this can have a
 *              fractional part, to allow a time of day to be specified.
 *
 *
 *   Support:
 *     Jeremy Bailey, AAO
 *
 *   Version Date:
 *     6-Oct-1994    
 *    21-May-2001. Now uses ChangeMjd(), and allows day to specify
 *                 a fractional part. KS.    
 *-
 */
{
    StatusType status = STATUS__OK;
    SdsIdType id, idf = 0;
    double mjd;
    double dd;
    int jstat;
    int iy, im, idd;
    char buff[80];

    id = atol(argv[1]);

    SdsFind(id, "fieldData", &idf, &status);
    iy = atoi(argv[2]);
    im = atoi(argv[3]);
    dd = atof(argv[4]);
    idd = (int) dd;
    slaCldj(iy, im, idd, &mjd, &jstat);
    mjd += dd - (double)idd;
    ChangeMjd (idf, mjd, &status);
    /*  Note - this used to have a + 0.5 added to mjd */

    SdsFreeId(idf, &status);

    if ((status == STATUS__OK) && (jstat == 0)) {
        return TCL_OK;
    } else {
        MessGetMsg(status, 0, sizeof(buff), buff);
        Tcl_AppendResult(interp, "Error setting UT\n", (char *) NULL);
        Tcl_AppendResult(interp, buff, (char *) NULL);
        return TCL_ERROR;
    }
}

/*  ------------------------------------------------------------------------- */

/*
 *                        N e x t  D a y  { }
 */
 
static int ConfNextDay(ClientData clientData DUNUSED, Tcl_Interp * interp,
    int argc DUNUSED, char *argv[])

/*+
 *  Command name:
 *    NextDay
 *
 *  Function:
 *    Set the Date of a configuration structure to the next day
 *  
 *  Call:
 *    NextDay id 
 *
 *  Description:
 *    Set the UT date in a configuration structure the next day
 *    (i.e. increment it by one) .
 *   
 *   Parameters:
 *
 *    (>) id   (Sds id)  Sds Id of configuration structure.
 *
 *   Support:
 *     Jeremy Bailey, AAO
 *
 *   Version Date:
 *     6-Oct-1994    
 *    21-May-2001. Now uses ChangeMjd(). KS  
 *-
 */
{
    StatusType status = STATUS__OK;
    SdsIdType id, idf = 0;
    double mjd;
    char buff[80];

    id = atol(argv[1]);

    SdsFind(id, "fieldData", &idf, &status);
    ArgGetd(idf, "configMjd", &mjd, &status);
    mjd = mjd + 1;
    ChangeMjd (idf, mjd, &status);

    SdsFreeId(idf, &status);

    if (status == STATUS__OK) {
        return TCL_OK;
    } else {
        MessGetMsg(status, 0, sizeof(buff), buff);
        Tcl_AppendResult(interp, "Error setting UT\n", (char *) NULL);
        Tcl_AppendResult(interp, buff, (char *) NULL);
        return TCL_ERROR;
    }
}


/*  ------------------------------------------------------------------------- */

/*
 *                        G e t  P l a t e  { }
 */
 
static int ConfGetPlate(ClientData clientData DUNUSED, Tcl_Interp * interp,
    int argc DUNUSED, char *argv[]DUNUSED)

/*+
 *  Command name:
 *    GetPlate
 *
 *  Function:
 *    Get Field Plate
 *  
 *  Call:
 *    SetPlate
 *
 *  Description:
 *    Return thenumber of the field plate currently being configured
 *   
 *   Parameters: None
 *
 *   Returns:
 *      The field plate (0 or 1)
 *
 *   Support:
 *     Jeremy Bailey, AAO
 *
 *   Version Date:
 *     17-Nov-1994    
 *-
 */
{
    sprintf(interp->result, "%d", CurrentField);
    return TCL_OK;
}


/*  ------------------------------------------------------------------------- */

/*
 *                        G e t  W a v e l e n g t h  { }
 */
 
static int ConfGetWavelength(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc DUNUSED, char *argv[]DUNUSED)

/*+
 *  Command name:
 *    GetWavelength
 *
 *  Function:
 *    Get configuration wavelength
 *  
 *  Call:
 *    GetWavelength
 *
 *  Description:
 *    Return the current configuration wavelength in Angstroms
 *   
 *   Parameters: None
 *
 *   Returns:
 *      The configuration wavelength
 *
 *   Support:
 *     Jeremy Bailey, AAO
 *
 *   Version Date:
 *     1-Nov-1994    
 *-
 */
{
    sprintf(interp->result, "%8.2f", 10000 * gwave);
    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*
 *                   S e t  W a v e l e n g t h  { }
 */
 
static int ConfSetWavelength(ClientData clientData DUNUSED,
    Tcl_Interp * interp DUNUSED, int argc DUNUSED, char *argv[])

/*+
 *  Command name:
 *    SetWavelength
 *
 *  Function:
 *    Set configuration wavelength
 *  
 *  Call:
 *    SetWavelength lambda
 *
 *  Description:
 *    Set the current configuration wavelength in Angstroms
 *   
 *   Parameters:
 *    (>) lambda   (double)  Configuration wavelength in Angstroms
 *
 *   Support:
 *     Jeremy Bailey, AAO
 *
 *   Version Date:
 *     15-Nov-1994    
 *-
 */
{
    /*  gwave is the global variable holding the current wavelength */
    
    gwave = (atof(argv[1])) / 10000.0;
    return TCL_OK;
}


/*  ------------------------------------------------------------------------- */

/*
 *                     S e t  R e f  P a r s  { }
 */
 
static int ConfSetRefPars(ClientData clientData DUNUSED,
    Tcl_Interp * interp DUNUSED, int argc DUNUSED, char *argv[])

/*+
 *  Command name:
 *    SetRefPars
 *
 *  Function:
 *    Set Refraction Parameters
 *  
 *  Call:
 *    SetRefPars temp press humid
 *
 *  Description:
 *    Return the current values of the refraction parameters
 *   
 *   Parameters:
 *        temp    (double)  Temperature (K)
 *        press   (double)  Atmospheric pressure (mb)
 *        humid   (double)  Relative humidity 
 *
 *   Support:
 *     Jeremy Bailey, AAO
 *
 *   Version Date:
 *     16-Nov-1994    
 *-
 */
{
    /*  Set the global variables holding the parameters */
    
    gtemp = atof(argv[1]);
    gpress = atof(argv[2]);
    ghumid = atof(argv[3]);

    return TCL_OK;
}


/*  ------------------------------------------------------------------------- */

/*
 *                     G e t  R e f  P a r s  { }
 */
 
static int ConfGetRefPars(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc DUNUSED, char *argv[])

/*+
 *
 *  Command name:
 *    GetRefPars
 *
 *  Function:
 *    Get Refraction Parameters
 *  
 *  Call:
 *    GetRefPars temp press humid
 *
 *  Description:
 *    Return the current values of the refraction parameters
 *   
 *
 *   Parameters:
 *        temp    (Char)  Name of variable to receive the temperature
 *        press   (Char)  Name of variable to receive the atmospheric pressure
 *        humid   (Char)  Name of variable to receive the humidity
 *
 *   Support:
 *     Jeremy Bailey, AAO
 *
 *   Version Date:
 *     16-Nov-1994    
 *-
 */
{
    char buffer[40];

    sprintf(buffer, "%6.2f", gtemp);
    Tcl_SetVar(interp, argv[1], buffer, 0);
    if (gpress >= 1000.0)
        sprintf(buffer, "%7.2f", gpress);
    else
        sprintf(buffer, "%6.2f", gpress);
    Tcl_SetVar(interp, argv[2], buffer, 0);
    sprintf(buffer, "%8.4f", ghumid);
    Tcl_SetVar(interp, argv[3], buffer, 0);

    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*
 *                     S e t  T e l  P a r s  { }
 */
 
static int ConfSetTelPars(ClientData clientData DUNUSED,
    Tcl_Interp * interp DUNUSED, int argc DUNUSED, char *argv[])

/*+
 *
 *  Command name:
 *    SetTelPars
 *
 *  Function:
 *    Set Telescope Parameters
 *  
 *  Call:
 *    SetTelPars params
 *
 *  Description:
 *    Set the current values of the telescope pointing model parameters
 *   
 *   Parameters:
 *       (>) params (list) A list giving the telescope parameter values.  This
 *                         should be a list of real numbers, the number of
 *                         items being the same as the number returned in each
 *                         sub-list by GetTelPars.
 *
 *   Support:
 *     Tony Farrell, AAO
 *
 *   Version Date:
 *     14-Mar-2000
 *-
 */
{
    int listArgc;
    char **listArgv;
    unsigned int i;
    double *data;

    if (argc != 2) {
        sprintf(interp->result, "%s:Wrong number of arguments", argv[0]);
        return TCL_ERROR;
    }

/*
 *  Split the argument up into a list elements and confirm we have the
 *  correct number of elements.
 */
    if (Tcl_SplitList(interp, argv[1], &listArgc, &listArgv) != TCL_OK) {
        return TCL_ERROR;
    }
    if ((unsigned) listArgc != TelModelNumParams) {
        sprintf(interp->result,
            "%s:List has wrong number list elements, %d instead of %d",
            argv[0], listArgc, TelModelNumParams);
        return TCL_ERROR;
    }

/*
 *  Allocate temp storage space for the data, so that we can catch conversion 
 *  errors without clobbering the existing data items.
 */
    data = malloc(sizeof(double) * TelModelNumParams);

/*
 *  Work through converting each element.  IF we have an error, fall out
 *  at this point.
 */
    for (i = 0; i < TelModelNumParams; ++i) {
        if (Tcl_GetDouble(interp, listArgv[i], &data[i]) != TCL_OK) {
            free(data);
            return TCL_ERROR;
        }
    }
/*
 *  Copy the data into the correct spot.
 */
    for (i = 0; i < TelModelNumParams; ++i) {
        TelModelParams[i] = data[i];
    }

    free(data);
    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*
 *                     G e t  T e l  P a r s  { }
 */
 
static int ConfGetTelPars(ClientData clientData DUNUSED, Tcl_Interp * interp,
    int argc DUNUSED, char *argv[])

/*+
 *
 *  Command name:
 *    GetTelPars
 *
 *  Function:
 *    Get Telescope Parameters
 *  
 *  Call:
 *    GetTelPars 
 *
 *  Description:
 *    Return the current values of the telescope pointing model parameters
 *    as a list containing three lists.  The first of the three lists 
 *    contains the current telescope parameter values.  The second contains
 *    the simple names of each telescope parameter.  The third contains 
 *    a more descriptive name.  All three lists have the same number
 *    of elements.
 *
 *   Returns:
 *
 *   Support:
 *     Tony Farrell, AAO
 *
 *   Version Date:
 *     14-Mar-2000
 *-
 */
{
    unsigned int NumParams;
    const char *const *ParamNames;
    const char *const *ParamDescrs;
    const double *ParamDefaults;
    StatusType status = STATUS__OK;
    register unsigned i;

    Tcl_DString Names;
    Tcl_DString Descrs;
    Tcl_DString Values;
    Tcl_DString CommandResult;



    FpilModelParams(TelModel, &NumParams,
        &ParamNames, &ParamDescrs, &ParamDefaults, &status);

    if (status != STATUS__OK) {
        char mbuf[100];
        MessGetMsg(status, 0, sizeof(mbuf), mbuf);
        sprintf(interp->result,
            "%s:Error getting telescope model parameter details - %s\n",
            argv[0], mbuf);
        return TCL_ERROR;
    }

/*
 *  Validate number of telescope model parameters.
 */
    if (NumParams != TelModelNumParams) {
        fprintf(stderr, "%s:%d Assertion failed\n", __FILE__, __LINE__);
        fprintf(stderr, "Telescope model parameter count invalid\n");
        fprintf(stderr, "Talk to tjf@aaoepp.aao.gov.au\n");
        exit(-1);
    }

/*
 *  Initialise the various value strings
 */
    Tcl_DStringInit(&Names);
    Tcl_DStringInit(&Descrs);
    Tcl_DStringInit(&Values);
    Tcl_DStringInit(&CommandResult);
/*
 *  Put all the values into the strings
 */
    for (i = 0; i < TelModelNumParams; ++i) {
        char buffer[30];
        sprintf(buffer, "%g", TelModelParams[i]);
        Tcl_DStringAppendElement(&Values, buffer);
        Tcl_DStringAppendElement(&Names, ParamNames[i]);
        Tcl_DStringAppendElement(&Descrs, ParamDescrs[i]);
    }

/*
 *  Append the three sub-lists to the result string
 */
    Tcl_DStringAppendElement(&CommandResult, Tcl_DStringValue(&Values));
    Tcl_DStringAppendElement(&CommandResult, Tcl_DStringValue(&Names));
    Tcl_DStringAppendElement(&CommandResult, Tcl_DStringValue(&Descrs));
/*
 *  And put the result string  in the correct place.
 */
    Tcl_DStringResult(interp, &CommandResult);

/*
 *  Free resources.
 */
    Tcl_DStringFree(&Names);
    Tcl_DStringFree(&Descrs);
    Tcl_DStringFree(&Values);
    Tcl_DStringFree(&CommandResult);


    return TCL_OK;
}


/*  ------------------------------------------------------------------------- */

/*
 *                     S e t  P l a t e  { }
 */
 
static int ConfSetPlate(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc DUNUSED, char *argv[])

/*+
 *
 *  Command name:
 *    SetPlate
 *
 *  Function:
 *    Select the a new field plate
 *  
 *  Call:
 *    SetField field
 *
 *  Description:
 *    Select the field plate to operate on.  We validate the plate number and
 *    then access the constants details and telescope model appropiate for 
 *    this field plate
 *   
 *   Parameters:
 *    (>) plate       (int) Field Plate number, 0 to the number
 *                          supported by the instrument.
 *
 *   Support:
 *     Tony Farrell, AAO
 *
 *-
 */
{
    unsigned NewField;
    /* char buffer[100]; */
    StatusType status = STATUS__OK;

    SdsIdType NewFieldDetails = 0;
    FpilModelType *NewTelModel = 0;

    if (argc != 2) {
        sprintf(interp->result, "%s:Wrong number of arguments", argv[0]);
        return TCL_ERROR;
    }

    NewField = atol(argv[1]);
    if (NewField >= FpilGetNumFields(*Instrument)) {
        sprintf(interp->result,
            "%s:Field %d not valid, %s has only %d field plates\n",
            argv[0],
            NewField,
            FpilGetInstName(*Instrument), FpilGetNumFields(*Instrument));
        return TCL_ERROR;
    }
/*
 *  Field number valid.  Access details and read telescope model
 *  details for this field.
 */
    FpilConFieldId(InstConstDetails, NewField, &NewFieldDetails, &status);


    FpilModelAccess(*Instrument, ConfigFilesDir, ExecutableDir,
        ConfRptTelModelFile, interp, NewField, &NewTelModel, &status);


    if (status == STATUS__OK) {
/*
 *      Success.  Delete the old field details and copy in the new to
 *      the module level variables.
 */
/*       No Plates for FLAMES needed
 *       sprintf(buffer, "New field plate is %d", NewField);
 *       ConfMessage(interp, buffer);
 */
        FpilModelFree(TelModel, &status);
        TelModel = NewTelModel;

        SdsFreeId(InstFieldDetails, &status);
        InstFieldDetails = NewFieldDetails;

        CurrentField = NewField;
        return TCL_OK;
    } else {
/*
 *      Error - tidy up, leaving current field as the valid field
 */
        StatusType ignore = STATUS__OK;
        char mbuf[100];

        if (NewTelModel)
            FpilModelFree(NewTelModel, &ignore);
        if (NewFieldDetails)
            SdsFreeId(NewFieldDetails, &ignore);

        MessGetMsg(status, 0, sizeof(mbuf), mbuf);
        sprintf(interp->result, "%s:Error select field plate %d - %s\n",
            argv[0], NewField, mbuf);
        return TCL_ERROR;

    }

}

/*  ------------------------------------------------------------------------- */

/*
 *                     P i v o t s  I d  { }
 */
 
static int ConfPivotsId(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc DUNUSED, char *argv[]DUNUSED)
/*+
 *
 *  Command name:
 *    PivotsId
 *
 *  Function:
 *    Return an Sds identifier for the field constants.
 *
 *  Call:
 *    PivotsId
 *
 *  Description:
 *    Returns an SDS id refering to the structure which contains the
 *    contants details for the current field.
 *   
 *    This id must not be free-ed, the structure must not be modified 
 *    or deleted.
 *
 *   Parameters:  none.
 *
 *   Support:
 *     Tony Farrell, AAO
 *
 *-
 */
{
    sprintf(interp->result, "%ld", (long int) InstFieldDetails);
    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*
 *  MakeSkyvals 
 * 
 * Comment by TJF, 17-Mar-2000.  I think this command counts the number
 * of sky values allocated to each spectrograph, and puts the
 * items allocSkyn (n = spectrograph number) into the field data with 
 * these values.
 *
 * Note -> Currently called directly by ConfDoAllocation(), which assumes
 * we have the same arguments as ConfDoAllocation().
 *
 */
static int ConfMakeSkyvals(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc DUNUSED, char *argv[])
{
    SdsIdType id, oid, idf;
    unsigned i;
    short *specSkys;
    char type;
    unsigned NumPivots = FpilGetNumPivots(*Instrument);
    StatusType status = STATUS__OK;


    id = atol(argv[1]);

/*
 *  Allocate space for counting the sky values.  Initialise them to zero.
 */
    specSkys = malloc(sizeof(*specSkys) * InstNumFibreTypes);
    for (i = 0; i < InstNumFibreTypes; ++i)
        specSkys[i] = 0;

/*
 *  Find the object data.
 */
    SdsFind(id, "fieldData", &idf, &status);

    SdsFind(id, "objects", &oid, &status);

/*
 *  Search through the pivots, increment the number of skys assocated
 *  with each fibre type. Note that the type will be the type of the
 *  target allocated to that pivot, or will be the special 'unallocated'
 *  code.
 */
    for (i = 0; i < NumPivots; i++) {
        ConfArrayGetc(oid, "type", i, &type, &status);
        if (FpilIsTargetSky(*Instrument, type)) {
            specSkys[ConfWhichSpec((int) i, 0)]++;
        }
    }
/*
 *  Now add the data to the field data.
 */
    for (i = 0; i < InstNumFibreTypes; ++i) {
        char buffer[20];
        sprintf(buffer, "allocSky%d", i + 1);
        ArgPuts(idf, buffer, specSkys[i], &status);
    }
/*
 *  Tidy up
 */

    SdsFreeId(idf, &status);
    SdsFreeId(oid, &status);
    free(specSkys);

    sprintf(interp->result, "OK");


    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*
 *                     C o n v e r t  X y  { }
 */
 
static int ConfConvertXy(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc DUNUSED, char *argv[])

/*+
 *
 *  Command name:
 *    ConvertXy
 *
 *  Function:
 *    Calculate xy from Ra and Dec
 *  
 *  Call:
 *    ConvertXy id 
 *
 *  Description:
 *    Calculate the x,y corresponding to the RA and Dec values in
 *    a configuration structure, for the current values of the telescope
 *    and refraction parameters.
 *   
 *   Parameters:
 *
 *    (>) id   (Sds id)  Sds Id of configuration structure.
 *
 *   Returns:
 *      "OK" if succesful, "error" if the conversion failed
 *
 *   Support:
 *     Jeremy Bailey, AAO
 *
 *   Version Date:
 *     6-Oct-1994
 *    21-May-2001. Removed epoch and getting of appEpoch. Made gmap a local
 *                 variable. KS.
 *    16-Aug-2001. Now records TelModelParams in TelModelParamsUsed. KS. 
 *    17-Oct-2002. FpilModelCvtInit() now takes two wavelength parameters. KS.  
 *-
 */
{
    SdsIdType id, idf, idtemp;
    double cra, cdec, mjd;
    double gmap[21];                   /* Mean to apparent parameters  */
    StatusType status = STATUS__OK;
    int i;

    id = atol(argv[1]);
    
    /*
     *  Get field centre details
     */

    SdsFind(id, "fieldData", &idf, &status);
    ArgGetd(idf, "appRa", &cra, &status);
    ArgGetd(idf, "appDec", &cdec, &status);
    ArgGetd(idf, "configMjd", &mjd, &status);
    SdsFreeId(idf, &status);
    
    /*  We sometimes need to know if the parameters we may be about to use
     *  are different to those used the last time the calculation was done.
     *  (This is because if so, it may invalidate any allocation, and the
     *  user may have to confirm this.) So we record the parameters used.
     *  At the moment, we only record the telescope model parameters, but
     *  perhaps we should record everything - this was introduced with
     *  the ARGUS position angle code, where it was important to know if
     *  the position angle had changed from the last time.
     */
   
    for (i = 0; i < TelModelNumParams; i++) {
        TelModelParamsUsed[i] = TelModelParams[i];
    }

    /*
     *  Set up x,y conversion parameters. The wavelength values in this
     *  call shouldn't matter because ConfDoXYConversion() always resets
     *  them to the values appropriate to the fibre type. Indeed, this
     *  FpilModelCvtInit() initialisation call may be unnecessary.
     */

    slaMappa(2000.0, mjd, gmap);
    /*
    DEBUG("gmap=pal.mappa(2000.0, %18.15e)\n",mjd);
     */

    FpilSetObsWavelength(*Instrument,gwave);
    FpilModelCvtInit(TelModel, mjd, 0.0, gtemp, gpress, ghumid, gwave, gwave,
        TelModelParams, &status);

    
    SdsFind(id, "unallocGuide", &idtemp, &status);

    if (status == SDS__NOITEM) {
        ErsAnnul(&status);
    } else {
        ConfDoXyConversion(idtemp, cra, cdec, mjd, gmap, &status);
        SdsFreeId(idtemp, &status);
    }

    if (status == STATUS__OK) {
        SdsFind(id, "unallocObject", &idtemp, &status);
        if (status == SDS__NOITEM) {
            ErsAnnul(&status);
        } else {
            ConfDoXyConversion(idtemp, cra, cdec, mjd, gmap, &status);
            SdsFreeId(idtemp, &status);
        }
    }

    if (status == STATUS__OK) {
        SdsFind(id, "objects", &idtemp, &status);
        if (status == SDS__NOITEM) {
            ErsAnnul(&status);
        } else {
            ConfDoXyConversion(idtemp, cra, cdec, mjd, gmap, &status);
            SdsFreeId(idtemp, &status);
        }
    }

    if (status == STATUS__OK) {
        sprintf(interp->result, "OK");
        return TCL_OK;
    } else {
        char buff[200];
        MessGetMsg(status, 0, sizeof(buff), buff);
        Tcl_AppendResult(interp, "\nError in Ra,Dec to x,y conversion\n",
            (char *) NULL);
        Tcl_AppendResult(interp, buff, (char *) NULL);
        return TCL_ERROR;

    }
}

/*  ------------------------------------------------------------------------- */

/*
 *                     P a r a m s  C h a n g e d  { }
 */
 
static int ConfParamsChanged(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc DUNUSED, char *argv[] DUNUSED)

/*+
 *
 *  Command name:
 *    ParamsChanged
 *
 *  Function:
 *    See if parameters used by ConvertXY have changed.
 *  
 *  Call:
 *    ParamsChanged id
 *
 *  Description:
 *    Checks to see if the telescope model parameters used by ConvertXY
 *    have changed since the last time ConvertXY was used. If this is the
 *    case, using ConvertXY again may invalidate any allocation that has
 *    been made, and the user may need to be warned of this.
 *   
 *   Parameters: 
 *      (>) id   (Sds id)  Sds Id of configuration structure.
 *
 *   Returns:
 *      "1" if the parameters have changed, "0" if not.
 *
 *   Support:
 *     Keith Shortridge, AAO
 *
 *   History:
 *    16-Aug-2001. Original version KS.   
 *-
 */
{
    SdsIdType id, idf;
    double cra, cdec, mjd;
    StatusType status = STATUS__OK;
    int i;
    short Changed;

    /*  This code gets the central ra and dec and mjd, on the grounds
     *  that perhaps we ought to test them as well. However, for the 
     *  moment we don't actually use them. We should perhaps also
     *  use the humidity and other parameters?
     */
     
    id = atol(argv[1]);
    
    /*
     *  Get field centre details
     */

    SdsFind(id, "fieldData", &idf, &status);
    ArgGetd(idf, "appRa", &cra, &status);
    ArgGetd(idf, "appDec", &cdec, &status);
    ArgGetd(idf, "configMjd", &mjd, &status);
    SdsFreeId(idf, &status);
    
    /*  For the moment, we content ourselves with testing the telescope
     *  model parameters.
     */
    
    Changed = 0;
    for (i = 0; i < TelModelNumParams; i++) {
        if (TelModelParamsUsed[i] != TelModelParams[i]) Changed = 1;
    }

    sprintf(interp->result, "%d",Changed);
    
    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*                      C o n f  S d s  C h e c k
 *
 *  This is a small utility that packages up a common sequence of calls
 *  used to report an error accessing an item in an Sds structure. It
 *  converts the Sds status code passed into a text message, the outputs
 *  that message and the text passed in the call.
 */
 
static void ConfSdsCheck(char *text, StatusType * status)
 {
    char buff[100];

    if (*status != STATUS__OK) {
        MessGetMsg(*status, 0, 100, buff);
        ErsRep(0, status, buff);
        ErsRep(0, status, text);
    }
}

/*  ------------------------------------------------------------------------- */

/*                   C o n f  D o  X Y  C o n v e r s i o n
*
 *  This is a utility routine used only by ConfConvertXy().  It can be
 *  called to work on the 'unallocated guide' structure (in which case each
 *  entry will be a target object classified as a guide), or on the 
 *  'unallocated object' structure (in which case each entry will be a target
 *  object classified as non-guide), or on the 'objects' structure. In the 
 *  last case, the fields in the structure relating to position realy refer 
 *  both to the targets and the fibres allocated to them, and in particular 
 *  note that the 'type' field can be either the unallocated type code or the
 *  type of the object to which the fibre has been allocated.
 *  gmap contains the mean to apparent parameters, and is now a parameter
 *  rather than a global variable (which it was in earlier versions of the
 *  code).
 *
 *  Modified:
 *    17-Oct-2002. FpilModelCvtInit() now takes two wavelength parameters. KS.
 *     4-Nov-2002. Now allows for proper motions included in pseudo-comments.
 *                 It would be better to have these in the SDS structure, but
 *                 that's too dramatic a change to make at this stage. KS.  
 */

static void ConfDoXyConversion(SdsIdType id, double cra, double cdec,
    double mjd, double* gmap, StatusType * status)

{
    char name[16];
    SdsCodeType code;
    long ndims;
    unsigned long dims[7];
    SdsIdType rid, did, tid, xid, yid, cid;
    double ra, dec;
    unsigned long actlen;
    double xp, yp;
    int xp2, yp2;
    char type;
    double dhdx, dhdy, dddx, dddy;
    SdsIdType tempid;
    char obj[NAMELEN];
    char comment[CMTLEN];
    StatusType stat;
    double Wavelength;
    double PrevWavelength;
    double PointingWavelength;
    unsigned int i;
    double ara, adec, xf, yf;
    double ara2, adec2, ara3, adec3;
    double pmra, pmdec;
    SdsIdType nid;
    char text[80];
    StatusType LocalStatus;

    if (*status != STATUS__OK)
        return;
    SdsFind(id, "ra", &rid, status);
    ConfSdsCheck("SDS Item ra not found", status);
    SdsFind(id, "dec", &did, status);
    ConfSdsCheck("SDS Item dec not found", status);
    SdsFind(id, "type", &tid, status);
    ConfSdsCheck("SDS Item type not found", status);
    SdsFind(id, "x", &xid, status);
    ConfSdsCheck("SDS Item x not found", status);
    SdsFind(id, "y", &yid, status);
    ConfSdsCheck("SDS Item y not found", status);
    SdsFind(id, "comment", &cid, status);
    ConfSdsCheck("SDS Item comment not found", status);
    
    /*  Get the main 'pointing' wavelength, which is the same for all
     *  fibre types.
     */
     
    PointingWavelength = FpilGetPointingWavelength (*Instrument);
    
    /*  Reset to an unlikely wavelength, to force initialisation of the
     *  conversion routines.
     */
   
    PrevWavelength = 0.0;

    /*  We get the number of items in the structure from the dimensions of
     *  one of the arrays ("ra" will do), and process each item in turn.
     */

    SdsInfo(rid, name, &code, &ndims, dims, status);
    for (i = 0; (i < dims[0]) && (*status == STATUS__OK); i++) {
        SdsGet(tid, sizeof(char), i, &type, &actlen, status);
        
        /*  If this is a guide star, we need to do the coordinate conversion
         *  for a different wavelength to that used for the ordinary
         *  target stars. So we need to see what wavelength to use, and
         *  if this has changed, we need to re-initialise the coordinate
         *  conversion.
         */
        
        Wavelength = FpilGetFibreWavelength(*Instrument,type);
        if (Wavelength != PrevWavelength) {
            FpilModelCvtInit(TelModel, mjd, 0.0, gtemp, gpress, ghumid, 
                      PointingWavelength, Wavelength, TelModelParams, status);
            PrevWavelength = Wavelength;
        }
        
        /*  See if this is an unallocated pivot (only when we process the
         *  "objects" structure will this be the case).
         */

        if (type != FpilUnallocatedType(*Instrument)) {
        
            /*  This is an allocated pivot, so we calculate the X,Y
             *  position from the Ra,Dec of the target to which it is
             *  allocated. We have to allow for objects with proper motions
             *  specified as pseudo-comments, so we need to look at the
             *  comment field as well.
             */
             
            SdsGet(rid, sizeof(double), i, &ra, &actlen, status);
            SdsGet(did, sizeof(double), i, &dec, &actlen, status);
            
            /*  If proper motions are specified in pseudo-comments, we
             *  need to convert to apparent RA/Dec using slaMapqk(). If
             *  not, we can use the simpler slaMapqkz().
             */
             
            SdsGet(cid, CMTLEN, i * CMTLEN, comment, &actlen, status);
            if (ConfGetProperMotions (comment,&pmra,&pmdec)) {
               /*
		DEBUG("retruned PMs: %f,%f\n",pmra / PI * (60. * 60. * 180.),pmdec / PI * (60. * 60. * 180.));
		*/
               slaMapqk(ra,dec,pmra/cos(dec),pmdec,0.0,0.0,gmap,&ara,&adec);
/*
                DEBUG("numpy.array(pal.mapqk(%18.15e,%18.15e,%18.15e,%18.15e,0.0,0.0,gmap))/pi*180.\n",ra,dec,pmra,pmdec);
                DEBUG("returned Coords: %s :: %f %f :: %f %f\n",
                       "with" ,
                       ra / PI * 180.,
                       dec / PI * 180.,
                       ara / PI * 180.,
                       adec / PI * 180.
                       );
 */
/*               slaMapqk(ra,dec,pmra/cos(dec),pmdec,0.76813,-22.4,gmap,&ara,&adec);
 *               DEBUG("returned Coords [WITH Px+RV]: %s :: %f %f :: %f %f\n",
 *                      "with" ,
 *                      ra / PI * 180.,
 *                      dec / PI * 180.,
 *                      ara / PI * 180.,
 *                      adec / PI * 180.
 *                      );
 */
            } else {
               slaMapqkz(ra, dec, gmap, &ara, &adec);
/*
                DEBUG("numpy.array(pal.mapqkz(%18.15e,%18.15e,gmap))/pi*180.\n",ra,dec);
                DEBUG("returned Coords: %s :: %f %f :: %f %f\n",
                       "w/o ",
                       ra / PI * 180.,
                       dec / PI * 180.,
                       ara / PI * 180.,
                       adec / PI * 180.
                       );
 */
            }
            FpilModelRd2Xy(TelModel, cra, cdec, ara, adec, mjd, &xf, &yf,
                status);
            if (*status != STATUS__OK) {
                goto Exit;
            }
            FpilModelXy2Pos(TelModel, xf, yf, &xp, &yp, status);
            xp2 = (int) xp;
            yp2 = (int) yp;
            SdsPut(xid, sizeof(int), i, &xp2, status);
            SdsPut(yid, sizeof(int), i, &yp2, status);

            /*  If this is a guide object (which can be the case when we
             *  process either the unallocated guide structure or the
             *  "objects" structure), work out the local derivatives for the
             *  ra and dec with respect to x and y.
             */

            if (FpilIsTargetGuide(*Instrument, type)) {
                FpilModelXy2Rd(TelModel, cra, cdec, xf, yf, mjd, &ara, &adec,
                    status);
                FpilModelXy2Rd(TelModel, cra, cdec, xf + 1000.0, yf, mjd, &ara2,
                    &adec2, status);
                FpilModelXy2Rd(TelModel, cra, cdec, xf, yf + 1000.0, mjd, &ara3,
                    &adec3, status);
                if (*status == STATUS__OK) {
                    dhdx = (ara - ara2) / 1000.0;
                    dddx = (adec2 - adec) / 1000.0;
                    dhdy = (ara - ara3) / 1000.0;
                    dddy = (adec3 - adec) / 1000.0;
                    SdsFind(id, "dhdx", &tempid, status);
                    ConfSdsCheck("SDS Item dhdx not found", status);
                    SdsPut(tempid, sizeof(double), i, &dhdx, status);
                    SdsFreeId(tempid, status);
                    SdsFind(id, "dddx", &tempid, status);
                    ConfSdsCheck("SDS Item dddx not found", status);
                    SdsPut(tempid, sizeof(double), i, &dddx, status);
                    SdsFreeId(tempid, status);
                    SdsFind(id, "dhdy", &tempid, status);
                    ConfSdsCheck("SDS Item dhdy not found", status);
                    SdsPut(tempid, sizeof(double), i, &dhdy, status);
                    SdsFreeId(tempid, status);
                    SdsFind(id, "dddy", &tempid, status);
                    ConfSdsCheck("SDS Item dddy not found", status);
                    SdsPut(tempid, sizeof(double), i, &dddy, status);
                    SdsFreeId(tempid, status);
                }
            }
            if (*status != STATUS__OK) {
                stat = STATUS__OK;
                SdsFind(id, "name", &nid, &stat);
                SdsGet(nid, NAMELEN, NAMELEN * i, obj, &actlen, &stat);
                SdsFreeId(nid, &stat);
                sprintf(text, "Error converting to xy for object %d - %s", i,
                    obj);
                ConfSdsCheck(text, status);
                goto Exit;
            }
        } else {
        
            /*  This is an unallocated pivot.  In this case, we do the 
             *  reverse, and calculate the Ra,Dec from the parked position
             *  in X,Y. (This might seem to be something we shouldn't care
             *  about, but for some instruments - 6dF - parked fibres are
             *  in the field, and if they record data an observer might want
             *  to know where it came from.)
             */
             
            SdsGet(xid, sizeof(int), i, &xp2, &actlen, status);
            SdsGet(yid, sizeof(int), i, &yp2, &actlen, status);
            xp = (double) xp2;
            yp = (double) yp2;
            FpilModelXy2Rd(TelModel, cra, cdec, xp, yp, mjd, &ra, &dec,status);
            SdsPut(rid, sizeof(double), i, &ra, status);
            SdsPut(did, sizeof(double), i, &dec, status);
            if (*status != STATUS__OK) {
                sprintf(text, "Error converting from xy for fibre %d", i);
                ConfSdsCheck(text, status);
                goto Exit;
            }
        }
            
    }
    
Exit:;

    /*  On exit, play safe and reset conversions to the main observing
     *  wavelength.
     */
     
    LocalStatus = 0;
    FpilModelCvtInit(TelModel, mjd, 0.0, gtemp, gpress, ghumid, gwave, gwave,
                                               TelModelParams, &LocalStatus);
   
    SdsFreeId(rid, status);
    SdsFreeId(did, status);
    SdsFreeId(tid, status);
    SdsFreeId(xid, status);
    SdsFreeId(yid, status);
    SdsFreeId(cid, status);
}

/*  ------------------------------------------------------------------------- */

/*
 *                     R a  2  S t r i n g  { }
 */
 
static int ConfRa2string(ClientData clientData DUNUSED, Tcl_Interp * interp,
    int argc DUNUSED, char *argv[])

/*+
 *
 *  Command name:
 *    Ra2string
 *
 *  Function:
 *    Convert an RA in radians to a string 
 *  
 *  Call:
 *    Ra2string ra
 *
 *  Description:
 *    Convert an RA in radians to a string in Hours, Minutes and Seconds
 *   
 *   Parameters:
 *
 *    (>) ra   (double)  RA in radians.
 *
 *   Returns:
 *     The string representation of the RA in hours, minutes and seconds
 *
 *   Support:
 *     Jeremy Bailey, AAO
 *
 *   Version Date:
 *     6-Oct-1994    
 *-
 */
{
    double d;
    int ihmsf[4];
    char sign[1];

    d = atof(argv[1]);
    slaDr2tf(2, d, sign, ihmsf);
    sprintf(interp->result, "%.2d %.2d %.2d.%.2d", ihmsf[0], ihmsf[1], ihmsf[2],
        ihmsf[3]);
    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*
 *                     D e c  2  S t r i n g  { }
 */
 
static int ConfDec2string(ClientData clientData DUNUSED, Tcl_Interp * interp,
    int argc DUNUSED, char *argv[])

/*+
 *
 *  Command name:
 *    Dec2string
 *
 *  Function:
 *    Convert a Dec in radians to a string 
 *  
 *  Call:
 *    Dec2string dec
 *
 *  Description:
 *    Convert a Dec in radians to a string in Degrees, Minutes and Seconds
 *   
 *   Parameters:
 *
 *    (>) dec  (double)  Declination in radians
 *
 *   Returns:
 *     The string representation of the Dec in degrees, minutes and seconds
 *
 *   Support:
 *     Jeremy Bailey, AAO
 *
 *   Version Date:
 *     6-Oct-1994    
 *-
 */
{
    double d;
    int ihmsf[4];
    char sign[1];

    d = atof(argv[1]);
    slaDr2af(1, d, sign, ihmsf);
    sprintf(interp->result, "%c%.2d %.2d %.2d.%.1d", sign[0],
        ihmsf[0], ihmsf[1], ihmsf[2], ihmsf[3]);
    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*
 *                     M j d  2  D a t e  { }
 */
 
static int ConfMjd2Date(ClientData clientData DUNUSED, Tcl_Interp * interp,
    int argc DUNUSED, char *argv[])

/*+
 *  Command name:
 *    Mjd2Date
 *
 *  Function:
 *    Convert an MJD to a string 
 *  
 *  Call:
 *    Mjd2Date mjd
 *
 *  Description:
 *    Convert a Modified Julian Date to a string giving the date in
 *     the form Year, Month, Day
 *   
 *   Parameters:
 *
 *    (>) mjd  (double)  Modified Julian Date
 *
 *   Returns:
 *     The string representation of the MJD as year, month, day
 *
 *   Support:
 *     Jeremy Bailey, AAO
 *
 *   Version Date:
 *     6-Oct-1994    
 *-
 */
{
    double d;
    int iy, im, id;
    double f;
    int j;

    d = atof(argv[1]);
    slaDjcl(d, &iy, &im, &id, &f, &j);
    sprintf(interp->result, "%.4d/%.2d/%.2d", iy, im, id);
    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*
 *                     M j d  2  T i m e  { }
 */
 
static int ConfMjd2Time(ClientData clientData DUNUSED, Tcl_Interp * interp,
    int argc DUNUSED, char *argv[])

/*+
 *  Command name:
 *    Mjd2Time
 *
 *  Function:
 *    Return UT time from an MJD 
 *  
 *  Call:
 *    Mjd2Time mjd
 *
 *  Description:
 *    Convert a Modified Julian Date to a string giving the fractional
 *    part of the MJD expressed as a UT time in hours, minutes and seconds.
 *   
 *   Parameters:
 *
 *    (>) mjd  (double)  Modified Julian Date
 *
 *   Returns:
 *     The string representation of the UT as hours, minutes, seconds
 *
 *   Support:
 *     Jeremy Bailey, AAO
 *
 *   Version Date:
 *     6-Oct-1994    
 *-
 */
{
    double d;
    int ihmsf[4];
    char sign[1];

    d = atof(argv[1]);
    slaDd2tf(0, d - floor(d), sign, ihmsf);
    sprintf(interp->result, "%.2d:%.2d:%.2d", ihmsf[0], ihmsf[1], ihmsf[2]);
    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

static int ConfRaDec2Ha(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc DUNUSED, char *argv[])

/*+
 *  Command name:
 *    RaDec2Ha
 *
 *  Function:
 *    Calculate HA from RA and Dec 
 *  
 *  Call:
 *    RaDec2Ha ra dec
 *
 *  Description:
 *    Calculate the observed Hour Angle corresponding to an apparent
 *    RA and Dec. The conversion is for the time and other parameters
 *    used in the last call to ConvertXy
 *   
 *   Parameters:
 *
 *    (>) ra  (double)  Apparent RA (radians)
 *    (>) dec (double)  Apparent Dec (radians)
 *
 *   Returns:
 *     The string representation of the observed HA as hours, minutes, seconds
 *
 *   Support:
 *     Jeremy Bailey, AAO
 *
 *   Version Date:
 *     6-Oct-1994    
 *-
 */
{
    double r, d;
    int ihmsf[4];
    char sign[1];
    double a, zd, ha, dd, rr;
    StatusType status = STATUS__OK;

    r = atof(argv[1]);
    d = atof(argv[2]);
    FpilModelQuickAppToObs(TelModel, r, d, &a, &zd, &ha, &dd, &rr, &status);
    slaDr2tf(0, ha, sign, ihmsf);
    sprintf(interp->result, "%c%.2d:%.2d:%.2d", sign[0],
        ihmsf[0], ihmsf[1], ihmsf[2]);
    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*
 *                     R a  D e c  2  Z d  { }
 */
 
static int ConfRaDec2Zd(ClientData clientData DUNUSED, Tcl_Interp * interp,
    int argc DUNUSED, char *argv[])

/*+
 *  Command name:
 *    RaDec2Zd
 *
 *  Function:
 *    Calculate ZD from RA and Dec 
 *  
 *  Call:
 *    RaDec2Zd ra dec
 *
 *  Description:
 *    Calculate the observed Zenith Distance corresponding to an apparent
 *    RA and Dec. The conversion is for the time and other parameters
 *    used in the last call to ConvertXy
 *   
 *   Parameters:
 *
 *    (>) ra  (double)  Apparent RA (radians)
 *    (>) dec (double)  Apparent Dec (radians)
 *
 *   Returns:
 *     The string representation of the observed ZD as degrees, minutes, seconds
 *
 *   Support:
 *     Jeremy Bailey, AAO
 *
 *   Version Date:
 *     6-Oct-1994    
 *-
 */
{
    double r, d;
    int ihmsf[4];
    char sign[1];
    double a, zd, ha, dd, rr;
    StatusType status = STATUS__OK;

    r = atof(argv[1]);
    d = atof(argv[2]);
    FpilModelQuickAppToObs(TelModel, r, d, &a, &zd, &ha, &dd, &rr, &status);
    slaDr2af(0, zd, sign, ihmsf);
    sprintf(interp->result, "%.2d:%.2d:%.2d", ihmsf[0], ihmsf[1], ihmsf[2]);
    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*
 *                     M a k e  A l l o c a t e d  { }
 */
 
static int ConfMakeAllocated(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc DUNUSED, char *argv[])

/*+
 *  Command name:
 *    MakeAllocated
 *
 *  Function:
 *    Create the Allocated objects section of a configuration structure 
 *  
 *  Call:
 *    MakeAllocated id pivid
 *
 *  Description:
 *    Create the allocated objects section of a configuration structure
 *    if it does not already exist. The new structure is initialized by
 *    setting all fibres to their parked positions.
 *   
 *   Parameters:
 *
 *    (>) id  (Sds Id)  SDS identifier of configuration structure.
 *    (>) pivid (Sds Id)  SDS identifier of pivots structure.
 *
 *   Support:
 *     Jeremy Bailey, AAO
 *
 *   Version Date:
 *     6-Oct-1994    
 *-
 */
{
    SdsIdType topid, pivid;
    StatusType status = STATUS__OK;
    char buff[100];
    
    topid = atol(argv[1]);
    pivid = atol(argv[2]);
    DoMakeAllocated (topid,pivid,&status);
    if (status == STATUS__OK) {
        return TCL_OK;
    } else {
        MessGetMsg(status, 0, sizeof(buff), buff);
        Tcl_AppendResult(interp, "\nError Making Allocated Objects Structure\n",
            (char *) NULL);
        Tcl_AppendResult(interp, buff, (char *) NULL);
        return TCL_ERROR;
    }
}
    

/*  ------------------------------------------------------------------------- */

/*+
 *                     D o  M a k e  A l l o c a t e d
 *
 *   DoMakeAllocated() is the main routine used by ConfMakeAllocated() to
 *   create the 'objects' structure. It is packaged so that it can also be
 *   called directly from other C routines such as the PAF input routines.
 *   It is passed the Sds Id's for the configuration data structure and
 *   for the pivot data structure.
 *-
 */
 
void DoMakeAllocated (
   SdsIdType topid,
   SdsIdType pivid,
   StatusType* status)
{   
    SdsIdType objid;
    SdsIdType name_id, ra_id, dec_id, fibre_id, x_id, y_id, theta_id, type_id;
    SdsIdType prio_id, mag_id, spect_id, pid_id, dhdx_id, dhdy_id, dddx_id,
        dddy_id;
    SdsIdType comment_id;
    SdsIdType xpid, ypid, tpid;
    char name[NAMELEN];
    SdsCodeType code;
    long ndims;
    unsigned long dims[7], dims2[7];
    unsigned int i;
    double d;
    int l;
    short s;
    char c;
    unsigned long actlen;

    if (*status != STATUS__OK) return;

    SdsFind(topid, "objects", &objid, status);
    ConfSdsCheck("Error finding objects structure", status);

    if (status == STATUS__OK) return;

    ErsAnnul(status);

    SdsNew(topid, "objects", 0, NULL, SDS_STRUCT, 0, NULL, &objid, status);

    SdsFind(pivid, "xPark", &xpid, status);
    SdsInfo(xpid, name, &code, &ndims, dims, status);
    dims2[0] = NAMELEN;
    dims2[1] = dims[0];
    SdsNew(objid, "name", 0, NULL, SDS_CHAR, 2, dims2, &name_id, status);
    SdsNew(objid, "ra", 0, NULL, SDS_DOUBLE, 1, dims, &ra_id, status);
    SdsNew(objid, "dec", 0, NULL, SDS_DOUBLE, 1, dims, &dec_id, status);
    SdsNew(objid, "fibre", 0, NULL, SDS_SHORT, 1, dims, &fibre_id, status);
    SdsNew(objid, "x", 0, NULL, SDS_INT, 1, dims, &x_id, status);
    SdsNew(objid, "y", 0, NULL, SDS_INT, 1, dims, &y_id, status);
    SdsNew(objid, "theta", 0, NULL, SDS_DOUBLE, 1, dims, &theta_id, status);
    SdsNew(objid, "type", 0, NULL, SDS_CHAR, 1, dims, &type_id, status);
    SdsNew(objid, "priority", 0, NULL, SDS_SHORT, 1, dims, &prio_id, status);
    SdsNew(objid, "magnitude", 0, NULL, SDS_DOUBLE, 1, dims, &mag_id, status);
    SdsNew(objid, "spectrograph", 0, NULL, SDS_CHAR, 1, dims, &spect_id,
        status);
    SdsNew(objid, "pId", 0, NULL, SDS_SHORT, 1, dims, &pid_id, status);
    SdsNew(objid, "dhdx", 0, NULL, SDS_DOUBLE, 1, dims, &dhdx_id, status);
    SdsNew(objid, "dhdy", 0, NULL, SDS_DOUBLE, 1, dims, &dhdy_id, status);
    SdsNew(objid, "dddx", 0, NULL, SDS_DOUBLE, 1, dims, &dddx_id, status);
    SdsNew(objid, "dddy", 0, NULL, SDS_DOUBLE, 1, dims, &dddy_id, status);
    SdsNew(objid, "comment", 0, NULL, SDS_CHAR, 2, dims2, &comment_id, status);

    SdsFind(pivid, "yPark", &ypid, status);
    SdsFind(pivid, "tPark", &tpid, status);

    if (*status == STATUS__OK) {

        for (i = 0; i < dims[0]; i++) {
            strcpy(name, "Parked");
            SdsPut(name_id, NAMELEN, NAMELEN * i, name, status);
            d = 0.0;
            SdsPut(ra_id, sizeof(double), i, &d, status);
            SdsPut(dec_id, sizeof(double), i, &d, status);
            s = 0;
            SdsPut(fibre_id, sizeof(short), i, &s, status);
            SdsGet(xpid, sizeof(int), i, &l, &actlen, status);
            SdsPut(x_id, sizeof(int), i, &l, status);
            SdsGet(ypid, sizeof(int), i, &l, &actlen, status);
            SdsPut(y_id, sizeof(int), i, &l, status);
            SdsGet(tpid, sizeof(double), i, &d, &actlen, status);
            SdsPut(theta_id, sizeof(double), i, &d, status);
            c = FpilUnallocatedType(*Instrument);
            SdsPut(type_id, sizeof(char), i, &c, status);
            s = 1;
            SdsPut(prio_id, sizeof(short), i, &s, status);
            d = 0.0;
            SdsPut(mag_id, sizeof(double), i, &d, status);
            c = ' ';
            SdsPut(spect_id, sizeof(char), i, &c, status);
            s = 0;
            SdsPut(pid_id, sizeof(short), i, &s, status);
            SdsPut(dhdx_id, sizeof(double), i, &d, status);
            SdsPut(dhdy_id, sizeof(double), i, &d, status);
            SdsPut(dddx_id, sizeof(double), i, &d, status);
            SdsPut(dddy_id, sizeof(double), i, &d, status);
            SdsPut(comment_id, CMTLEN, CMTLEN * i, name, status);
        }
    }

    SdsFreeId(xpid, status);
    SdsFreeId(ypid, status);
    SdsFreeId(tpid, status);
    SdsFreeId(name_id, status);
    SdsFreeId(ra_id, status);
    SdsFreeId(dec_id, status);
    SdsFreeId(fibre_id, status);
    SdsFreeId(x_id, status);
    SdsFreeId(y_id, status);
    SdsFreeId(theta_id, status);
    SdsFreeId(type_id, status);
    SdsFreeId(prio_id, status);
    SdsFreeId(mag_id, status);
    SdsFreeId(spect_id, status);
    SdsFreeId(pid_id, status);
    SdsFreeId(dhdx_id, status);
    SdsFreeId(dhdy_id, status);
    SdsFreeId(dddx_id, status);
    SdsFreeId(dddy_id, status);
    SdsFreeId(comment_id, status);
    SdsFreeId(objid, status);

}

/*  ------------------------------------------------------------------------- */

/*        P r e l i m i n a r y   D e f i n i t i o n s   f o r
 *
 *                         C o n f  M a k e  A l l o c a t e d  I n d e x
 *
 *  ConfMakeAllocatedIndex is invoked if the objects->index, 
 *  unallocGuide->allocated and unallocObject->allocated items have just 
 *  been created by  ConfAddIndex().  It is possible that some items in the
 *  "objects" array are in the unallocObj/unallocGui arrays, as the
 *  structure may be the result of a previous run of the program - but
 *  the "index" item may or may not be correct.
 *
 *  In addition, we should add any objects not in unallocObject/unallocGuide
 *  to the appropiate component to ensure the structure is consistent with
 *  the way the config library will use it.
 * 
 *  We use a Tcl Hash table to help us here.  We first get access to
 *  the appropiate components of the unallocGuide and unallocObject structures
 *  (name, ra, dec, allocated).  We use the name as the hash table key.  The
 *  hash table value is the index for unallocObject objects and
 *  the negative of the index+1 for unallocGuide objects.  This avoids us
 *  setting up structures to maintain this information and should make
 *  this quite efficent.  Each object must have a unique name.
 *
 *  Then, we walk through the objects structure.  For every allocated
 *  object, we look for in the hash table for an object of the same 
 *  name.  
 *
 *  If the object exists in the hash table, then we ensure it is marked
 *  as allocated.   If not, then our structure is corrupted.  IF the
 *  existing index for the object is -1, then we set that index.  If the
 *  index for an allocated object was not -1, then is must correspond to the
 *  index for correspnding the unallocObject/unallocGuide item, otherwise
 *  our structure is corrupted.
 *
 *  If the object does not exist in the hash table, then its index should
 *  be minus 1.  We  create a list of such objects.
 *
 *  After the above, we will have the number of indexes of each object
 *  in the allocated are which are not in the unallocate area.  We then
 *  expand those compoenets and copy the details in.  We set the index
 *  for each object and the allocated area appropiately.
 */
 
typedef struct {
    unsigned size;
    SdsIdType name;
    char *namePtr;
    SdsIdType ra;
    double *raPtr;
    SdsIdType dec;
    double *decPtr;
    SdsIdType type;
    char *typePtr;
    SdsIdType other;            /* index/allocated */
    void *otherPtr;
} structIds;

const structIds IdsInitVal = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };

/*
 * Macros used to converting the guide array lists index to a hash table
 * value and the reverse.  The idea is to represent the guide array index
 * as a negative number, to avoid confusion with the object array lists, which
 * use the positive value as the hash value.
 */
 
#define GuideIdToHash(_i_) ((((int)(_i_))+1)*-1)
#define HashToGuideId(_i_) ((unsigned)(((_i_)*-1)-1))

/*
 * We use this hash table value to indicate items added to the hash table
 * as they were found in the "objects" component.  
 */

#define EXTRA_VALUE INT_MIN

#define CONF_C_ALLOC_EXISTS 0   /* Object exists in unallocGuide or
                                   unallocObject array or is not
                                   allocated */
#define CONF_C_ALLOC_GUIDE 1    /* Allocated guide object which does
                                   not exist in unallocGuide array */
#define CONF_C_ALLOC_OBJECT 2   /* Allocated program object which 
                                   does not exist in unallocObject
                                   array */

/*  ------------------------------------------------------------------------- */

/*        U t i l i t y  R o u t i n e s   f o r
 *
 *                         C o n f  M a k e  A l l o c a t e d  I n d e x
 */
 
/*  ------------------------------------------------------------------------- */

/*                   C o n f  I n d e x  I d s  G e t
 *
 *  Access each of the items of interest in a object array structure.
 *  If "alloc" is ture, we are dealing with the array of allocated
 *  structures ("objects").  Otherwise, it is one of the unallocated
 *  item arrays.
 */
 
static void ConfIndexIdsGet(SdsIdType topid,
    int alloc, structIds * ids, StatusType * status)
{
    unsigned long length;
    char name[SDS_C_NAMELEN];
    SdsCodeType code;
    long ndims;
    unsigned long dims[7];

/*
 *  Set initial values.
 */
    *ids = IdsInitVal;

    if (*status != STATUS__OK)
        return;

    SdsFind(topid, "ra", &ids->ra, status);
    SdsFind(topid, "dec", &ids->dec, status);
    SdsFind(topid, "name", &ids->name, status);
    SdsFind(topid, "type", &ids->type, status);
/*
 *  If this is from the allocated (objects) array, find the index.  If this
 *  is from the unallocGuide/unallocObject arrays, find the allocated item.
 */
    if (alloc)
        SdsFind(topid, "index", &ids->other, status);
    else
        SdsFind(topid, "allocated", &ids->other, status);

/*
 *  Get points to the data for each item.
 */
    SdsPointer(ids->name, (void **) &ids->namePtr, &length, status);
    SdsPointer(ids->ra, (void **) &ids->raPtr, &length, status);
    SdsPointer(ids->dec, (void **) &ids->decPtr, &length, status);
    SdsPointer(ids->type, (void **) &ids->typePtr, &length, status);
    SdsPointer(ids->other, (void **) &ids->otherPtr, &length, status);

/*
 *  Get the length of each array.
 */
    SdsInfo(ids->ra, name, &code, &ndims, dims, status);
    ids->size = dims[0];

}

/*  ------------------------------------------------------------------------- */

/*                   C o n f  I n d e x  I d s  F r e e
 *
 *   Free the ids created by ConfIndexIdsGet().
 */
 
static void ConfIndexIdsFree(structIds * ids)
{
    StatusType ignore = STATUS__OK;
    if (ids->ra)
        SdsFreeId(ids->ra, &ignore);
    if (ids->dec)
        SdsFreeId(ids->dec, &ignore);
    if (ids->name)
        SdsFreeId(ids->name, &ignore);
    if (ids->type)
        SdsFreeId(ids->type, &ignore);
    if (ids->other)
        SdsFreeId(ids->other, &ignore);

    *ids = IdsInitVal;
}


/*  ------------------------------------------------------------------------- */

/*            C o n f  M a k e  A l l o c a t e d  I n d e x
 *
 *  This is the routine called from ConfAddIndex() (which implements 
 *  AddIndex{}) to generate the index and allocated cross-reference items.
 */
 
static void ConfMakeAllocatedIndex(SdsIdType topid, StatusType * status)
{
    SdsIdType objectsId = 0;    /* Id to "objects" array    */
    SdsIdType unallocGuiId = 0; /* Id to "unallocGui" array */
    SdsIdType unallocObjId = 0; /* Id to "unallocObj" array */

    structIds objectsStruct;    /* Parts of objects array    */
    structIds unallocGuiStruct; /* Parts of unallocGui array */
    structIds unallocObjStruct; /* Parts of unallocObj array */

    Tcl_HashTable hashTable;    /* The hash table of unallocGui and
                                   unallocObj names  */

    register unsigned i;

    int *objectNotExist = 0;    /* Points to array of malloc data
                                   which are flags.  with one of the
                                   three values above
                                 */

    unsigned allocNEGuideCnt = 0;
                                /* Number of allocated guide not
                                   in unallocGuide array */
    unsigned allocNEObjCnt = 0; /* Number of allocated program objects
                                   not in unallocObject array */

    int duplicates = 0;         /* Do we have duplicate objects */


    StatusType ignore = STATUS__OK;
    if (*status != STATUS__OK)
        return;

/*
 *  Access the structures
 */
    SdsFind(topid, "objects", &objectsId, status);
    SdsFind(topid, "unallocGuide", &unallocGuiId, status);
    SdsFind(topid, "unallocObject", &unallocObjId, status);

/*
 *  And the components of the structures
 */
    ConfIndexIdsGet(objectsId, 1, &objectsStruct, status);
    ConfIndexIdsGet(unallocGuiId, 0, &unallocGuiStruct, status);
    ConfIndexIdsGet(unallocObjId, 0, &unallocObjStruct, status);

/*
 *  Initialise the TCL has table.  The key is a string (the name).
 *  
 */
    Tcl_InitHashTable(&hashTable, TCL_STRING_KEYS);

    if (*status != STATUS__OK)
        goto Exit;

/*
 *  Add the unallocObject array to the hash table.  The hash value is just
 *  the index, the key is the name.
 */

    for (i = 0; i < unallocObjStruct.size; ++i) {
        Tcl_HashEntry *entryPtr;
        int new;
        char *name = &unallocObjStruct.namePtr[NAMELEN * i];
        entryPtr = Tcl_CreateHashEntry(&hashTable, name, &new);


        if (!new) {
            ErsRep(0, status,
                "Unallocated Program Object %s is not unique.", name);
            duplicates = 1;
        } else
            Tcl_SetHashValue(entryPtr, i);

    }
/*
 *  likewise for the unallocGuide array.  Here the hash value is different.
 */
    for (i = 0; i < unallocGuiStruct.size; ++i) {
        Tcl_HashEntry *entryPtr;
        int new;
        char *name = &unallocGuiStruct.namePtr[NAMELEN * i];
        int value = GuideIdToHash(i);
        entryPtr = Tcl_CreateHashEntry(&hashTable, name, &new);


        if (!new) {
            ErsRep(0, status,
                "Unallocated Guide Object %s is not unique.", name);
            duplicates = 1;
        } else
            Tcl_SetHashValue(entryPtr, value);
    }

    if (duplicates) {
        ErsRep(0, status, "Configure cannot handle duplicate object names.");
        ErsRep(0, status, "Please remove them and try again.");
        *status = CONF__DUPLICATES;
        goto Exit;
    }
    
/*
 *  The hash table is now set up.  
 * 
 *  Malloc some space to keep track of which allocated objects do not
 *  exist in unallocGuide/unallocObject
 */
 
    objectNotExist = malloc(sizeof(*objectNotExist) * objectsStruct.size);

/*
 *  Walk through the objects array
 */
 
    for (i = 0; i < objectsStruct.size; ++i) {
        Tcl_HashEntry *entryPtr;
/*
 *      By default, assume that if the object is allocated, it exists in
 *      the other parts of the structure.
 */
        objectNotExist[i] = CONF_C_ALLOC_EXISTS;
/*
 *      If this object is allocated, See if this object has a hash table entry
 */
        if (objectsStruct.typePtr[i] != FpilUnallocatedType(*Instrument)) {
            char *name = &objectsStruct.namePtr[NAMELEN * i];
            short *indexPtr = objectsStruct.otherPtr;

            entryPtr = Tcl_FindHashEntry(&hashTable, name);
            if (entryPtr) {
                structIds *ids;
                int j;
                char *type;
                char *allocated;
/*
 *              We have an entry for this name.  Get details of the
 *              structure we are dealing with and validate the type.
 */
                j = (int) Tcl_GetHashValue(entryPtr);
                if (j == EXTRA_VALUE) {
                    fprintf(stderr,
                        "Multiple definition of object of name %s ", name);
                    fprintf(stderr, "in \"objects\" structure\n");
                    fprintf(stderr,
                        "%s:%d Assertion failed\n", __FILE__, __LINE__);
                    fprintf(stderr, "Talk to tjf@aaoepp.aao.gov.au\n");
                    exit(-1);
                } else if (j < 0) {
                    type = "unallocGuide";
                    j = HashToGuideId(j);
                    ids = &unallocGuiStruct;

                    if (!FpilIsTargetGuide(*Instrument,
                            objectsStruct.typePtr[i])) {
                        fprintf(stderr,
                            "Object %d, name %s, name corresponds to guide ",
                            i, name);
                        fprintf(stderr,
                            "object %d, but the allocation does not\n", j);
                        fprintf(stderr,
                            "Configure's data structures are corrupt\n");
                        fprintf(stderr,
                            "%s:%d Assertion failed\n", __FILE__, __LINE__);
                        fprintf(stderr, "Talk to tjf@aaoepp.aao.gov.au\n");
                        exit(-1);
                    }
                } else {
                    type = "unallocObject";
                    ids = &unallocObjStruct;
                    if (FpilIsTargetGuide(*Instrument,
                            objectsStruct.typePtr[i])) {
                        fprintf(stderr,
                            "Object %d, name %s, name corresponds to program ",
                            i, name);
                        fprintf(stderr,
                            "object %d, but the allocation does not\n", j);
                        fprintf(stderr,
                            "Configure's data structures are corrupt\n");
                        fprintf(stderr,
                            "%s:%d Assertion failed\n", __FILE__, __LINE__);
                        fprintf(stderr, "Talk to tjf@aaoepp.aao.gov.au\n");
                        exit(-1);
                    }

                }
/*
 *              ids now points to the correct one of unallocObjStruct/
 *              unallocGuiStruct.  The type is correct
 *
 *              Validate the same ra and dec.  There are other things we
 *              could check but this would seem to be sufficent.
 */
                if (objectsStruct.raPtr[i] != ids->raPtr[j]) {
                    fprintf(stderr, "Object %d - %s, %s %d.  RA not the same\n",
                        i, name, type, j);
                    fprintf(stderr,
                        "Configure's data structures are corrupt\n");
                    fprintf(stderr,
                        "%s:%d Assertion failed\n", __FILE__, __LINE__);
                    fprintf(stderr, "Talk to tjf@aaoepp.aao.gov.au\n");
                    exit(-1);
                }

                if (objectsStruct.decPtr[i] != ids->decPtr[j]) {
                    fprintf(stderr,
                        "Object %d - %s, %s %d.  DEC not the same\n",
                        i, name, type, j);
                    fprintf(stderr,
                        "Configure's data structures are corrupt\n");
                    fprintf(stderr,
                        "%s:%d Assertion failed\n", __FILE__, __LINE__);
                    fprintf(stderr, "Talk to tjf@aaoepp.aao.gov.au\n");
                    exit(-1);
                }

/*
 *              The unallocGuide/unallocObject array entry for this
 *              object should have the "allocated" flag set.
 */
                allocated = ids->otherPtr;
                if (!allocated[j]) {
                    fprintf(stderr,
                        "Object %d - %s, %s %d.  allocated flag not set\n",
                        i, name, type, j);
                    fprintf(stderr,
                        "Configure's data structures are corrupt\n");
                    fprintf(stderr,
                        "%s:%d Assertion failed\n", __FILE__, __LINE__);
                    fprintf(stderr, "Talk to tjf@aaoepp.aao.gov.au\n");
                    exit(-1);

                }
/*
 *              Now if the index entry is mimus 1, we should set it.  Otherwise
 *              it should be the same value as j.
 */
                if (indexPtr[i] == -1) {
                    indexPtr[i] = j;
                } else if (indexPtr[i] != j) {
                    fprintf(stderr,
                        "Object %d - %s, %s %d.  Index flag invalid - %d\n",
                        i, name, type, j, indexPtr[i]);
                    fprintf(stderr,
                        "Configure's data structures are corrupt\n");
                    fprintf(stderr,
                        "%s:%d Assertion failed\n", __FILE__, __LINE__);
                    fprintf(stderr, "Talk to tjf@aaoepp.aao.gov.au\n");
                    exit(-1);
                }

            } else {

                int new;
/*
 *              No hash table entry.  Validate index before noting this object
 *              so that later we can add it to the appropiate array.
 *              We also add it to the hash table to help us in picking
 *              up duplicates.
 *
 *              index should be minus -1.  Otherwise
 */
                if (indexPtr[i] != -1) {
                    fprintf(stderr,
                        "Allocated object %d (%s) is not in the hash table,",
                        i, name);
                    fprintf(stderr, " but has a index of %d\n", indexPtr[i]);
                    fprintf(stderr,
                        "Configure's data structures are corrupt\n");
                    fprintf(stderr, "%s:%d Assertion failed\n", __FILE__,
                        __LINE__);
                    fprintf(stderr, "Talk to tjf@aaoepp.aao.gov.au\n");
                    exit(-1);
                }
                if (FpilIsTargetGuide(*Instrument, objectsStruct.typePtr[i])) {
                    objectNotExist[i] = CONF_C_ALLOC_GUIDE;
                    ++allocNEGuideCnt;
                } else {
                    objectNotExist[i] = CONF_C_ALLOC_OBJECT;
                    ++allocNEObjCnt;
                }

                entryPtr = Tcl_CreateHashEntry(&hashTable, name, &new);


                if (!new) {
                    fprintf(stderr,
                        "CreateHashEntry indicated object %s is not unique,",
                        name);
                    fprintf(stderr,
                        " but it was not found so should be unique\n");
                    fprintf(stderr,
                        "%s:%d Assertion failed\n", __FILE__, __LINE__);
                    fprintf(stderr, "Talk to tjf@aaoepp.aao.gov.au\n");
                    exit(-1);
                }
                Tcl_SetHashValue(entryPtr, EXTRA_VALUE);

            }
        }
    }

/*
 *  Now I need to add each of the allocated objects not in the appropiate
 *  array to the end of the array.
 */
    if (allocNEObjCnt || allocNEGuideCnt) {
        if (allocNEObjCnt)
            ConfResizeAllItems(unallocObjId,
                unallocObjStruct.size + allocNEObjCnt, status);

        if (allocNEGuideCnt)
            ConfResizeAllItems(unallocGuiId,
                unallocGuiStruct.size + allocNEGuideCnt, status);

        for (i = 0; i < objectsStruct.size; ++i) {
            SdsIdType uid;
            int obj_no;
            short *indexPtr = objectsStruct.otherPtr;

            if (objectNotExist[i] == CONF_C_ALLOC_EXISTS)
                continue;

/*
 *          Copy in the details, code based on code in ConfDeallocateFibre
 */
            if (objectNotExist[i] == CONF_C_ALLOC_OBJECT) {
                obj_no = unallocObjStruct.size++;
                uid = unallocObjId;
                ConfArrayCopy(objectsId, uid, "spectrograph", sizeof(char),
                    i, obj_no, status);

            } else {            /* (objectNotExist[i] == CONF_C_ALLOC_GUIDE) */

                obj_no = unallocGuiStruct.size++;
                uid = unallocGuiId;
                ConfArrayCopy(objectsId, uid, "dhdx", sizeof(double), i,
                    obj_no, status);
                ConfArrayCopy(objectsId, uid, "dhdy", sizeof(double), i,
                    obj_no, status);
                ConfArrayCopy(objectsId, uid, "dddx", sizeof(double), i,
                    obj_no, status);
                ConfArrayCopy(objectsId, uid, "dddy", sizeof(double), i,
                    obj_no, status);

            }

            ConfArrayCopy(objectsId, uid, "name", NAMELEN, NAMELEN * i,
                NAMELEN * obj_no, status);
            ConfArrayCopy(objectsId, uid, "ra", sizeof(double), i, obj_no,
                status);
            ConfArrayCopy(objectsId, uid, "dec", sizeof(double), i, obj_no,
                status);
            ConfArrayCopy(objectsId, uid, "x", sizeof(int), i, obj_no, status);
            ConfArrayCopy(objectsId, uid, "y", sizeof(int), i, obj_no, status);
            ConfArrayCopy(objectsId, uid, "type", sizeof(char), i, obj_no,
                status);
            ConfArrayCopy(objectsId, uid, "priority", sizeof(short), i, obj_no,
                status);
            ConfArrayCopy(objectsId, uid, "magnitude", sizeof(double), i,
                obj_no, status);
            ConfArrayCopy(objectsId, uid, "pId", sizeof(short), i, obj_no,
                status);
            ConfArrayCopy(objectsId, uid, "comment", CMTLEN, CMTLEN * i,
                CMTLEN * obj_no, status);

/*
 *          Indicate object is allocated and set index.
 */
            ConfArrayPutc(uid, "allocated", obj_no, 1, status);
            indexPtr[i] = obj_no;


        }
    }

/*
 *  Clean up.
 */

  Exit:
    if (objectNotExist)
        free(objectNotExist);
    Tcl_DeleteHashTable(&hashTable);

    if (objectsId)
        SdsFreeId(objectsId, &ignore);

    if (unallocGuiId)
        SdsFreeId(unallocGuiId, &ignore);

    if (unallocObjId)
        SdsFreeId(unallocObjId, &ignore);

    ConfIndexIdsFree(&objectsStruct);
    ConfIndexIdsFree(&unallocGuiStruct);
    ConfIndexIdsFree(&unallocObjStruct);

}

/*  ------------------------------------------------------------------------- */

/*
 *                     A d d  I n d e x  { }
 */
 
static int ConfAddIndex(ClientData clientData DUNUSED, Tcl_Interp * interp,
    int argc DUNUSED, char *argv[])

/*+
 *  Command name:
 *    AddIndex
 *
 *  Function:
 *    Add the index and allocated components to a configuration structure
 *  
 *  Call:
 *    AddIndex id 
 *
 *  Description:
 *    Create the "index" component in the allocated objects section
 *    and the "allocated" component in the unallocated objects and
 *    guide sections of a configuration structure, if they do not
 *    already exist.
 *   
 *   Parameters:
 *
 *    (>) id  (Sds Id)  SDS identifier of configuration structure.
 *
 *   Support:
 *     Jeremy Bailey, AAO
 *
 *   Version Date:
 *     6-Oct-1994    
 *-
 */
{
    SdsIdType topid, objid, indid, uid, allid, ra_id;
    StatusType status = STATUS__OK;
    char name[NAMELEN];
    SdsCodeType code;
    long ndims;
    unsigned long dims[7];
    unsigned int i;
    short s;
    char c;
    int indexFound = 0;         /* Indicates if we found the "index" item */


    topid = atol(argv[1]);

/*
 * Find the objects structure, and look for "index".
 */
    SdsFind(topid, "objects", &objid, &status);
    if (status != STATUS__OK) {
        sprintf(interp->result, "Error finding objects structure");
        return TCL_ERROR;
    }
    SdsFind(objid, "index", &indid, &status);
    if (status == SDS__NOITEM) {
/*
 *     "index" does not exist.  Create it and initialise the values
 *     to -1.
 */
        ErsAnnul(&status);
        SdsFind(objid, "ra", &ra_id, &status);
        SdsInfo(ra_id, name, &code, &ndims, dims, &status);
        SdsNew(objid, "index", 0, NULL, SDS_SHORT, 1, dims, &indid, &status);
        for (i = 0; i < dims[0]; i++) {
            s = -1;
            SdsPut(indid, sizeof(short), i, &s, &status);
        }
        SdsFreeId(indid, &status);
        SdsFreeId(ra_id, &status);
    } else {
        indexFound = 1;
    }
    SdsFreeId(objid, &status);

/*
 * Find "unallocGuide" and the "allocated" component within.
 */
    SdsFind(topid, "unallocGuide", &uid, &status);
    if (status != STATUS__OK) {
        sprintf(interp->result, "Error finding unallocGuide structure");
        return TCL_ERROR;
    }
    SdsFind(uid, "allocated", &allid, &status);
    if ((status == SDS__NOITEM) && (!indexFound)) {
/*
 *     "allocated" does not exist.  Create it and initialise to 0, indicating
 *     objects are not allocated.
 */
        ErsAnnul(&status);
        SdsFind(uid, "ra", &ra_id, &status);
        SdsInfo(ra_id, name, &code, &ndims, dims, &status);
        SdsNew(uid, "allocated", 0, NULL, SDS_CHAR, 1, dims, &allid, &status);
        for (i = 0; i < dims[0]; i++) {
            c = 0;
            SdsPut(allid, sizeof(char), i, &c, &status);
        }
        SdsFreeId(allid, &status);
        SdsFreeId(ra_id, &status);
    }
    SdsFreeId(uid, &status);
/*
 * Find "unallocObject" and the "allocated" component within.
 */
    SdsFind(topid, "unallocObject", &uid, &status);
    if (status != STATUS__OK) {
        sprintf(interp->result, "Error finding unallocObject structure");
        return TCL_ERROR;
    }
    SdsFind(uid, "allocated", &allid, &status);
    if ((status == SDS__NOITEM) && (!indexFound)) {
/*
 *     "allocated" does not exist.  Create it and initialise to 0, indicating
 *     objects are not allocated.
 */ 
        ErsAnnul(&status);
        SdsFind(uid, "ra", &ra_id, &status);
        SdsInfo(ra_id, name, &code, &ndims, dims, &status);
        SdsNew(uid, "allocated", 0, NULL, SDS_CHAR, 1, dims, &allid, &status);
        for (i = 0; i < dims[0]; i++) {
            c = 0;
            SdsPut(allid, sizeof(char), i, &c, &status);
        }
        SdsFreeId(allid, &status);
        SdsFreeId(ra_id, &status);
    }
    SdsFreeId(uid, &status);

/*
 * If the index and allocated items were not found in the original
 * structure, we need to set them up appropiately.
 */
    if (!indexFound)
        ConfMakeAllocatedIndex(topid, &status);

    if (status == STATUS__OK) {
        return TCL_OK;
    } else {
        char buff[200];
        MessGetMsg(status, 0, sizeof(buff), buff);
        Tcl_AppendResult(interp, "\nError Making Allocated Objects Structure\n",
            (char *) NULL);
        Tcl_AppendResult(interp, buff, (char *) NULL);
        return TCL_ERROR;

    }
}

/*  ------------------------------------------------------------------------- */

/*                    C o n f  W h i c h  S p e c
 *
 *  Determine which fibre type a specified pivot is associated with.
 *  If the Descr argument is non-null, it will be set to a string 
 *  description of the fibre type. This is mainly a wrap-up of the
 *  FPIL routine FpilConWhichSpec(). The name is now slightly misleading.
 *  Originally (in 2dF) the only important distinctions between fibres
 *  (other than guide fibres, which were handled separately) was which
 *  of two possible spectrographs they were associated with. Now, particularly
 *  with FLAMES, where there are a number of different fibre types (not
 *  necessarily all associated with different spectrographs as such),
 *  this routine is really concerned with the general case of what type
 *  a fibre is, and this is no longer just a simple case of having
 *  different spectrographs. However, the name has persisted.
 *
 *  Note. this function takes pivot from 0 to NPiv-1, whilst FPIL uses
 *  1 to NPiv.
 */
 
static int ConfWhichSpec(int IPiv, const char **Descr)
{
    unsigned Spec;
    const char *SpecDescr;
    StatusType status = STATUS__OK;

    FpilConWhichSpec(*Instrument, InstConstDetails, CurrentField, IPiv + 1,
        &Spec, &SpecDescr, &status);

    if (Descr)
        *Descr = SpecDescr;

    return Spec;

}

/*  ------------------------------------------------------------------------- */

/*
 *                     M a r k  F i b r e  T y p e  { }
 */
 
static int ConfMarkFibreType(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])

/*+
 *  Command name:
 *    MarkFibreType
 *
 *  Function:
 *    Mark a fibre type as in use or not in use.
 *  
 *  Call:
 *    MarkFibreType flagArray
 *
 *  Description:
 *    The C layer (this file, configure.c) maintains a global array that
 *    flags which fibre type is to be used in configurations. This is the
 *    UseFibreTypes array. This is used when an allocation is performed.
 *    However, it is the Tcl/Tk layer that really knows which fibre types
 *    are to be used, and it maintains its own array of flags for each
 *    fibre type. When these are changed, MarkFibreType{} should be 
 *    invoked to keep the C layer's global array (UseFibreTypes) in sync
 *    with the Tcl layer.
 *   
 *   Parameters:
 *
 *    (>) flagArray  (Tcl array variable) This is the name of a Tcl array that
 *                   contains an entry for each fibre type (0 .. 
 *                   InstNumFibreTypes-1).  These entries are logical
 *                   values used to set the values in the UseFibreType array.
 *-
 */
 {
    register unsigned i;
    int *localUseFibreType;
    if (argc != 2) {
        sprintf(interp->result, "%s:Wrong # args", argv[0]);
        return TCL_ERROR;
    }

    /*
     * Malloc space for a local copy of the values.
     */
     
    localUseFibreType = malloc(InstNumFibreTypes * sizeof(*localUseFibreType));

    /*
     * Read each array element.
     */
    for (i = 0; i < InstNumFibreTypes; ++i) {
        char element[10];
        const char *stringValue;
        sprintf(element, "%d", i);
        /*
         *  Read the array element
         */
        stringValue = Tcl_GetVar2(interp, argv[1], element, TCL_LEAVE_ERR_MSG);
        if (stringValue == NULL) {
            free(localUseFibreType);
            return TCL_ERROR;
        }

        /*
         * Get the value as a boolean.
         */
        if (Tcl_GetBoolean(interp, stringValue,
                &localUseFibreType[i]) == TCL_ERROR) {
            free(localUseFibreType);
            return TCL_ERROR;
        }
    }

    /*
     * Everything ok, make the value valid.  
     */
    for (i = 0; i < InstNumFibreTypes; ++i) {
        UseFibreType[i] = localUseFibreType[i];
    }
    /*
     * Tidy up and return.
     */
    free(localUseFibreType);
    return TCL_OK;

}

/*  ------------------------------------------------------------------------- */

/*
 *                 C h e c k  T a r g e t  C r i t e r i a  { }
 */
 
static int ConfCheckTargetCriteria(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])

/*+
 *  Command name:
 *    CheckTargetCriteria
 *
 *  Function:
 *    Sees if a target meets a supplied set of criteria.
 *  
 *  Call:
 *    CheckTargetCriteria Flags NumCriteria TargetType TargetSpect SpectKnown 
 *
 *  Description:
 *    This routine is used as part of the code that highlights targets in
 *    the mimic display if they meet a user-specified set of criteria. It
 *    is little more than an interface to FpilTargetCriteriaMet() and that
 *    routine's comments should be consulted for more details. Note that
 *    the test is based purely on the type of the target. Additional
 *    criteria such as the target priority and whether or not it is allocated
 *    are handled separately. This routine is merely used to abstract the
 *    issues connected with target type.
 *   
 *  Parameters:
 *
 *    (>) Flags       (Tcl array)  Array of criteria flag values
 *    (>) NumCriteria (int)        Number of instrument-dependent criteria
 *    (>) TargetType  (string)     Type code for target to be checked
 *    (>) TargetSpect (string)     Spectrograph code for target to be checked
 *    (>) SpectKnown  (int)        True if the TargetSpect code can be used
 *
 *   Returns:
 *     The number of criteria met by the target.
 *
 *   Support:
 *     Keith Shortridge, AAO
 *
 *-
 */
 
{
    int     CriteriaMet;
    int*    Flags;
    short   FlagsMalloced;
    int     Icrit;
    int     LocalFlagArray[20];
    int     NumCriteria;
    int     ReturnValue;
    int     SpectKnown;
    char    TargetType;
    char    TargetSpect;

    Flags = (int*) NULL;
    FlagsMalloced = FALSE;
    ReturnValue = TCL_OK;
    CriteriaMet = 0;

    if (argc != 6) {
        sprintf(interp->result, "%s:Wrong # args", argv[0]);
        ReturnValue = TCL_ERROR;
        goto Exit;
    }
    NumCriteria = atoi(argv[2]);
    if (NumCriteria > 0) {
        if (NumCriteria <= (int)(sizeof(LocalFlagArray)/sizeof(int))) {
            Flags = LocalFlagArray;
            FlagsMalloced = FALSE;
        } else {

            /*  Malloc space for a local copy of the flag values if the
             *  predefined local flag array is too small.
             */
     
            Flags = malloc(NumCriteria * sizeof(int));
            if (Flags == (int*) 0) {
                sprintf(interp->result, "%s: Unable to malloc flag memory",
                                                                  argv[0]);
                ReturnValue = TCL_ERROR;
                goto Exit;
            }
            FlagsMalloced = TRUE;
        }

        /*  Get the value of each flag array element. */

        for (Icrit = 0; Icrit < NumCriteria; Icrit++) {
            char Element[10];
            const char *StringValue;
            sprintf(Element, "%d", Icrit);

            /*  Read the Tcl array element  */

            StringValue = Tcl_GetVar2(interp, argv[1], Element,
                                                   TCL_LEAVE_ERR_MSG);
            if (StringValue == NULL) {
                ReturnValue = TCL_ERROR;
                goto Exit;
            }

            /*  Get the value as a boolean.  */

            if (Tcl_GetBoolean(interp, StringValue,
                                     &Flags[Icrit]) == TCL_ERROR) {
                ReturnValue = TCL_ERROR;
                goto Exit;
            }
        }

        /*  Now get the TargetType, TargetSpect and SpectKnown parameters */

        TargetType = *(argv[3]);
        TargetSpect = *(argv[4]);
        SpectKnown = atoi(argv[5]);

        CriteriaMet = FpilTargetCriteriaMet (*Instrument,Flags,NumCriteria,
                                         TargetType,TargetSpect,SpectKnown);
        
        sprintf(interp->result, "%d",CriteriaMet);
    }

Exit:;

    if (FlagsMalloced) free(Flags);
    return (ReturnValue);

}

/*  ------------------------------------------------------------------------- */

/*
 *                     A l l o c a t e  F i b r e  { }
 */
 
static int ConfAllocateFibre(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])

/*+
 *  Command name:
 *    AllocateFibre
 *
 *  Function:
 *    Allocate an object to a fibre
 *  
 *  Call:
 *    AllocateFibre id pivid fibre type object
 *
 *  Description:
 *    Allocate an object to a fibre. This command checks that the fibre
 *    and object are not already allocated, and if not copies the object
 *    information to the allocated objects section of the structure. It sets
 *    the index entry for the fibre to point to the object location in the
 *    unallocated structure, and it sets the allocated flag in that structure.
 *
 *    The command does not check the validity of the resulting configuration.
 *    That can be done by a call to DoCheck.
 *   
 *   Parameters:
 *
 *    (>) id  (Sds Id)   SDS identifier of configuration structure.
 *    (>) pivid (Sds Id) SDS identifier of the pivots structure.
 *    (>) fibre (int)    Fibre (pivot) number - starting from one.
 *    (>) type  (char)   "object" or "guide".
 *    (>) object (int)   Object number - starting from one.
 *
 *   Returns:
 *     "OK" if succesful, "Error" otherwise
 *
 *   Support:
 *     Jeremy Bailey, AAO
 *
 *   Version Date:
 *      6-Oct-1994  Original version. JAB.
 *      8-Nov-2001  Now updates fibre type allocation statistics. KS.
 *      3-Sep-2002  Now updates available target allocation statistics. KS.   
 *-
 */
{
    SdsIdType topid, objid, uid, pivid, fid;
    long fibre_no;
    long obj_no;
    StatusType status = STATUS__OK;
    short s;
    char c;
    short IsSky;
    char guide;
    int x, y, xp, yp;
    double theta;
    int fibreType;
    int PivIndex;
    int NumPivTypes;
    int PivTypes[10];

    if (argc != 6) {
        interp->result = "Wrong # args";
        return TCL_ERROR;
    }

    topid = (SdsIdType) (atol(argv[1]));
    pivid = (SdsIdType) (atol(argv[2]));
    fibre_no = atol(argv[3]) - 1;
    obj_no = atol(argv[5]) - 1;
    fibreType = ConfWhichSpec((int) fibre_no, 0);

    if ((fibre_no < 0) || (fibre_no >= (int) FpilGetNumPivots(*Instrument))) {
        sprintf(interp->result,
            "Illegal Pivot Number %ld\nMust be between 1 and %d",
            fibre_no + 1, FpilGetNumPivots(*Instrument));
        return TCL_ERROR;
    } else if (!UseFibreType[fibreType]) {
        const char *FibreTypeDescr;
        ConfWhichSpec((int) fibre_no, &FibreTypeDescr);
        sprintf(interp->result,
            "Attempt to allocate pivot %ld, with a fibre type (%s) ",
            fibre_no + 1, FibreTypeDescr);
        strcat(interp->result, " whose allocation is disabled");
        DEBUG("Allocate rejected: fibre type %d disabled\n", fibreType);
        return TCL_ERROR;
    } else if (strcmp(argv[4], "object") == 0) {
        SdsFind(topid, "unallocObject", &uid, &status);
        ConfSdsCheck("Error finding unallocObject structure", &status);
        guide = 0;
    } else if (strcmp(argv[4], "guide") == 0) {
        SdsFind(topid, "unallocGuide", &uid, &status);
        ConfSdsCheck("Error finding unallocGuide structure", &status);
        guide = 1;
    } else {
        interp->result = "Arg 4 must be object or guide";
        return TCL_ERROR;
    }
/*      This DEBUG section provides additional details about the object.
    {
        int x, y;
        SdsIdType tid;
        char string[128];
        unsigned long actlen;
        ConfArrayGeti(uid, "x", obj_no, &x, &status);
        ConfArrayGeti(uid, "y", obj_no, &y, &status);
        SdsFind(uid, "name", &tid, &status);
        SdsGet(tid, 80, 80 * obj_no, string, &actlen, &status);
        SdsFreeId(tid, &status);

        DEBUG("AllocateFibre:Object %d (%s) position is at %d,%d\n",
           obj_no,string,x,y);

    }
*/
    SdsFind(topid, "objects", &objid, &status);
    ConfSdsCheck("Error finding objects structure", &status);

/*  Check fibre is not already allocated  */

    ConfArrayGetc(objid, "type", fibre_no, &c, &status);
    if (c != FpilUnallocatedType(*Instrument)) {

        sprintf(interp->result,
            "Fibre at pivot number %ld is already allocated", fibre_no + 1);
        return TCL_ERROR;
    }

/*  Check fibre is useable  */

    ConfArrayGets(pivid, "inUse", fibre_no, &s, &status);
    if (s != 1 && ImportOnlyMode == 0) {
        sprintf(interp->result,
            "Fibre at pivot number %ld is not useable", fibre_no + 1);

        return TCL_ERROR;
    }

/*  Check object is not already allocated  */

    ConfArrayGetc(uid, "allocated", obj_no, &c, &status);
    if (c != 0) {
        sprintf(interp->result, "Object %ld is already allocated", obj_no + 1);
        return TCL_ERROR;
    }

/*  Copy data from the unallocated object section to the objects
    section of the structure  */

    ConfArrayCopy(uid, objid, "name", NAMELEN, NAMELEN * obj_no,
        NAMELEN * fibre_no, &status);
    ConfArrayCopy(uid, objid, "ra", sizeof(double), obj_no, fibre_no, &status);
    ConfArrayCopy(uid, objid, "dec", sizeof(double), obj_no, fibre_no, &status);


    ConfArrayCopy(uid, objid, "x", sizeof(int), obj_no, fibre_no, &status);
    ConfArrayCopy(uid, objid, "y", sizeof(int), obj_no, fibre_no, &status);

    ConfArrayCopy(uid, objid, "type", sizeof(char), obj_no, fibre_no, &status);

    ConfArrayCopy(uid, objid, "priority", sizeof(short), obj_no, fibre_no,
        &status);

    ConfArrayCopy(uid, objid, "magnitude", sizeof(double), obj_no, fibre_no,
        &status);

    ConfArrayCopy(uid, objid, "pId", sizeof(short), obj_no, fibre_no, &status);

    ConfArrayCopy(uid, objid, "comment", CMTLEN, CMTLEN * obj_no,
        CMTLEN * fibre_no, &status);

    if (guide) {
        ConfArrayCopy(uid, objid, "dhdx", sizeof(double), obj_no, fibre_no,
            &status);
        ConfArrayCopy(uid, objid, "dhdy", sizeof(double), obj_no, fibre_no,
            &status);
        ConfArrayCopy(uid, objid, "dddx", sizeof(double), obj_no, fibre_no,
            &status);
        ConfArrayCopy(uid, objid, "dddy", sizeof(double), obj_no, fibre_no,
            &status);
    }

/*  Calculate theta so that the button is parallel to the fibre  */

    ConfArrayGeti(pivid, "xPiv", fibre_no, &xp, &status);
    ConfArrayGeti(pivid, "yPiv", fibre_no, &yp, &status);
    ConfArrayGeti(objid, "x", fibre_no, &x, &status);
    ConfArrayGeti(objid, "y", fibre_no, &y, &status);

    theta = slaDranrm(atan2((double) (yp - y), (double) (xp - x)) - DPIBY2);

    ConfArrayPutd(objid, "theta", fibre_no, theta, &status);
    
/* We need to update the allocation statistics held for this fibre type */

    ConfArrayGetc(objid, "type", fibre_no, &c, &status);
    IsSky = FpilIsTargetSky(*Instrument, c);
    if ((fibreType >= 0) && (fibreType < InstNumFibreTypes)) {
        FibreTypeStats[fibreType].Unallocated--;
        if (FpilIsTargetGuide(*Instrument, c)) {
            FibreTypeStats[fibreType].AllocToGuide++;
        } else if (IsSky) {
            FibreTypeStats[fibreType].AllocToSky++;
        } else {
            FibreTypeStats[fibreType].AllocToObject++;
        }
    }
    
/* Also, given this target type (in 'c') we need to calculate how this
 * allocation affects the possible sky and object targets for the different
 * fibre types - it might have been a target that could be allocated to a
 * number of different fibre types
 */
 
    FpilCompatiblePivotTypes (*Instrument,10,c,0,&NumPivTypes,PivTypes);
    for (PivIndex = 0; PivIndex < NumPivTypes; PivIndex++) {
        fibreType = PivTypes[PivIndex];
        if (IsSky) {
            FibreTypeStats[fibreType].Skies--;
        } else {
            FibreTypeStats[fibreType].Targets--;
        }
    }

/* We now need to update the field data structure */

    SdsFind(topid, "fieldData", &fid, &status);
    ConfSdsCheck("Error finding fieldData structure", &status);


    if (status == STATUS__OK) {
        if (guide) {
            ArgGets(fid, "unallocGui", &s, &status);
            s--;
            ArgPuts(fid, "unallocGui", s, &status);
            ArgGets(fid, "allocGui", &s, &status);
            if (status == SDS__NOITEM) {
                ErsAnnul(&status);
                s = 1;
            } else
                s++;
            ArgPuts(fid, "allocGui", s, &status);
        } else {
            ConfArrayGetc(objid, "type", fibre_no, &c, &status);
            if (!FpilIsTargetGuide(*Instrument, c) &&
                !FpilIsTargetSky(*Instrument, c)) {
                ArgGets(fid, "unallocObj", &s, &status);
                s--;
                ArgPuts(fid, "unallocObj", s, &status);
                ArgGets(fid, "allocObj", &s, &status);
                if (status == SDS__NOITEM) {
                    ErsAnnul(&status);
                    s = 1;
                } else
                    s++;
                ArgPuts(fid, "allocObj", s, &status);
            } else if (FpilIsTargetSky(*Instrument, c)) {
                char name[20];

                /* Update unallocSky and allocSky */
                
                ArgGets(fid, "unallocSky", &s, &status);
                ArgPuts(fid, "unallocSky", s - 1, &status);
                ArgGets(fid, "allocSky", &s, &status);
                if (status == SDS__NOITEM) {
                    ErsAnnul(&status);
                    s = 1;
                } else
                    s++;
                ArgPuts(fid, "allocSky", s, &status);

                /* Update allocSkyn where n = fibre type number */
                
                sprintf(name, "allocSky%d", fibreType);
                ArgGets(fid, "name", &s, &status);
                if (status == SDS__NOITEM) {
                    ErsAnnul(&status);
                    s = 1;
                } else
                    s++;
                ArgPuts(fid, name, s, &status);

            }
        }
    }
    ConfArrayPuts(objid, "index", fibre_no, obj_no, &status);
    ConfArrayPutc(uid, "allocated", obj_no, (char) 1, &status);
    SdsFreeId(objid, &status);
    SdsFreeId(uid, &status);
    SdsFreeId(fid, &status);

    if (status == STATUS__OK) {
        interp->result = "OK";
        return TCL_OK;
    } else {
        char buff[200];
        snprintf( interp->result, TCL_RESULT_SIZE, 
            "\nError Allocating Fibre %ld\n", fibre_no);
        Tcl_AppendResult(interp, buff, (char *) NULL);

        MessGetMsg(status, 0, sizeof(buff), buff);
        Tcl_AppendResult(interp, buff, (char *) NULL);
        return TCL_ERROR;

    }

}

/*  ------------------------------------------------------------------------- */

/*
 *                     P i v o t  I n  U s e  { }
 */
 
static int ConfPivotInUse(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])

/*+
 *  Command name:
 *    PivotInUse
 *
 *  Function:
 *    Examines the 'inUse' flag for a given pivot/fibre
 *  
 *  Call:
 *    PivotInUse pivid fibre
 *
 *  Description:
 *    This command returns the value of the 'inUse' flag for a given
 *    fibre/pivot.
 *
 *   Parameters:
 *
 *    (>) pivid (Sds Id) SDS identifier of the pivots structure.
 *    (>) fibre (int)    Fibre (pivot) number - starting from one.
 *
 *   Returns:
 *     1 if pivot is flagged as in use, 0 if not.
 *
 *   Support:
 *     Keith Shortridge, AAO
 *
 *   Version Date:
 *     11-Apr-2002  Original version. JAB.
 *-
 */
{
    SdsIdType pivid;
    long fibre_no;
    StatusType status = STATUS__OK;
    short s;

    if (argc != 3) {
        interp->result = "Wrong # args";
        return TCL_ERROR;
    }

    pivid = (SdsIdType) (atol(argv[1]));
    fibre_no = atol(argv[2]) - 1;

    if ((fibre_no < 0) || (fibre_no >= (int) FpilGetNumPivots(*Instrument))) {
        sprintf(interp->result,
            "Illegal Pivot Number %ld\nMust be between 1 and %d",
            fibre_no + 1, FpilGetNumPivots(*Instrument));
        return TCL_ERROR;
    }
    
    ConfArrayGets(pivid, "inUse", fibre_no, &s, &status);
    if (s != 1) {
        strcpy(interp->result,"0");
    } else {
        strcpy(interp->result,"1");
    }
    return TCL_OK;

}

/*  ------------------------------------------------------------------------- */

/*  The following global variables are used by ReportRoutine(), ConfAllocRpt()
 *  and ReportRoutineBatch(). gecount is a global error count, used to abort
 *  the allocation if errors become excessive. ginterp is used to keep a
 *  pointer to the current Tcl interpreter (but actually no longer seems
 *  to be used).
 */
 
static Tcl_Interp *ginterp;
static int gecount;

/*  ------------------------------------------------------------------------- */

/*                         R e p o r t  R o u t i n e
 *
 *  Called by  ConfSetTargets and ConfAllocate, within ConfDoAllocation, 
 *  to report on allocation progress.  Note that this has the same calling
 *  sequence as ConfAllocRpt() but since it is now used only to report
 *  the progress rather than actual allocation details, most of the arguments
 *  are unused.
 */
 
static int ReportRoutine(void *clientData, char Type DUNUSED, char *Description,
    int pivot DUNUSED, int guide DUNUSED, int target DUNUSED,
    long TargetX DUNUSED, long TargetY DUNUSED,
    double theta DUNUSED, double PercentDone)
 {
    Tcl_Interp *interp = clientData;
    char command[300];

    sprintf(command, "AllUpdate \"%s\" %f",Description, PercentDone);

/*
 *  Evaluate a command to report progress.  If we get an error, report
 *  it, but only abort if number of errors is above MAX_ERRORS.
 *
 *  Note, we only send the abort message once, after which we stop
 *  reporting messages to avoid flooding the screen.
 *
 */
    if (Tcl_Eval(interp, command) == TCL_ERROR) {

        if (gecount <= MAX_ERRORS) {
            char buffer[80];
            sprintf(buffer,
                "\n(Error when reporting allocation progress (%d))",
                gecount + 1);
            Tcl_AddErrorInfo(interp, buffer);
            Tcl_BackgroundError(interp);
        }
        gecount++;
        if (gecount == MAX_ERRORS) {
            StatusType status = STATUS__OK;
            ErsRep(ERS_M_BELL, &status,
                "Allocation Aborted due to too many (%d) errors", gecount);
            return 1;
        }
/*
 *  Command returned 1, this indicates a cancel is required.
 */
    } else if (strcmp(interp->result, "1") == 0) {
        return 1;
    }
    return 0;
}

/*  ------------------------------------------------------------------------- */

/*                   R e p o r t  R o u t i n e  B a t c h
 *
 *  Called by  ConfSetTargets and ConfAllocate, within ConfDoAllocationBatch, 
 *  to report on allocation progress.
 */
 
static int ReportRoutineBatch(void *clientData, char Type,
    char *Description,
    int pivot, int guide, int target,
    long TargetX DUNUSED, long TargetY DUNUSED,
    double theta, double PercentDone)
 {
    Tcl_Interp *interp = clientData;
    char command[300];

    if (guide)
        sprintf(command, "AllUpdateBatch %c %d %d uguide %f \"%s\" %f", Type,
            pivot + 1, target + 1, theta, Description, PercentDone);
    else
        sprintf(command, "AllUpdateBatch %c %d %d uobj %f \"%s\" %f", Type,
            pivot + 1, target + 1, theta, Description, PercentDone);

/*
 *  Evaluate a command to report progress.  If we get an error, report
 *  it, but only abort if number of errors is above MAX_ERRORS.
 *
 *  Note, we only send the abort message once, after which we stop
 *  reporting messages to avoid flooding the screen.
 *
 */
    if (Tcl_Eval(interp, command) == TCL_ERROR) {

        if (gecount <= MAX_ERRORS) {
            char buffer[80];
            sprintf(buffer,
                "\n(Error when reporting batch allocation progress (%d))",
                gecount + 1);
            Tcl_AddErrorInfo(interp, buffer);
            Tcl_BackgroundError(interp);
        }
        gecount++;
        if (gecount == MAX_ERRORS) {
            StatusType status = STATUS__OK;
            ErsRep(ERS_M_BELL, &status,
                "Allocation Aborted due to too many (%d) errors", gecount);
            return 1;
        }

/*
 *  Command returned 1, this indicates a cancel is required.
 */
    } else if (strcmp(interp->result, "1") == 0) {
        return 1;
    }
    return 0;
}


/*  ------------------------------------------------------------------------- */

/*                        C o n f  A l l o c  R p t
 *
 *  Called by ConfigAllocate(), within ConfDoAllocation() to indicate 
 *  allocation/deallocation of fibres.  We use the Tcl command AllocRpt to 
 *  do this. Note that it is this mechanism that is responsible for keeping the
 *  configuration data structures updated during an allocation.
 */
 
static void ConfAllocRpt(void *clientData, int Alloc,
    int pivot, int guide, int target,
    long TargetX DUNUSED, long TargetY DUNUSED,
    double theta, StatusType * status)
 {
    Tcl_Interp *interp = clientData;

    char command[300];

    if (*status != STATUS__OK)
        return;

    if (guide)
        sprintf(command, "AllocRpt %d %d %d uguide %f", Alloc,
            pivot + 1, target + 1, theta);
    else
        sprintf(command, "AllocRpt %d %d %d uobj %f", Alloc,
            pivot + 1, target + 1, theta);

/*
 *  Evaluate a command to report allocation/deallocation.  If we get an 
 *  error report it but only abort if number of errors is above MAX_ERRORS
 *
 *  Note, we only send the abort message once, after which we stop
 *  reporting messages to avoid flooding the screen.
 *
 */
    if (Tcl_Eval(interp, command) == TCL_ERROR) {

        if (gecount <= MAX_ERRORS) {
            char buffer[80];
            sprintf(buffer,
                "\n(Error when reporting allocations/deallocations (%d))",
                gecount + 1);
            Tcl_AddErrorInfo(interp, buffer);
            Tcl_BackgroundError(interp);
        }
        gecount++;
        if (gecount == MAX_ERRORS) {
            *status = CONF__RPTERR;
            ErsRep(ERS_M_BELL, status,
                "Allocation Aborted due to too many (%d) errors", gecount);
        }
    }
}


/*  ------------------------------------------------------------------------- */

/*
 *                     D o  A l l o c a t i o n  { }
 *
 *                 D o  A l l o c a t i o n  B a t c h  { }
 */
 
static int ConfDoAllocation(ClientData clientData DUNUSED, Tcl_Interp * interp,
    int argc DUNUSED, char *argv[])

/*+
 *  Command name:
 *    DoAllocation
 *    DoAllocationBatch
 *
 *  Function:
 *    Perform automatic allocation. The batch version does it in Batch mode
 *  
 *  Call:
 *    DoAllocation id
 *    DoAllocationBatch id
 *
 *  Description:
 *    Perform automatic allocation of objects to fibres by calling the
 *    routines ConfigSetTargets and ConfigAllocate. The Tcl procedure
 *    AllUpdate will be called to report the progress of the allocation.
 *
 *    Note that we make no change to the SDS structure, this is done
 *    by the AllUpdate command invoking commands such as AllocateFibre   
 *    SetButtonAngle and DeallocateFibre.
 *
 *    The clientData item is either ALLOCATE_NORMAL or ALLOCATE_BATCH 
 *    to indicate the mode.
 *
 *   Parameters:
 *
 *    (>) id  (Sds Id)   SDS identifier of configuration structure.
 *
 *   Support:
 *     Jeremy Bailey, AAO
 *
 *   Version Date:
 *     6-Oct-1994    
 *-
 */
{

    StatusType status = STATUS__OK;
    char mbuf[100];
    register unsigned i;
    CONFIG_ReportFunction RptFunction;

    if (clientData == (void *) ALLOCATE_BATCH)
        RptFunction = ReportRoutineBatch;
    else
        RptFunction = ReportRoutine;

    sprintf(mbuf, "Doing allocation for field %d", CurrentField);
    ConfMessage(interp, mbuf);

/*  Set Target SDS structure  */

    ConfigSetTargets(NULL, atol(argv[1]), CurrentField, RptFunction,
        (void *) interp, &status);

/*  Indicate which fibre type we want to use */

    for (i = 0; i < InstNumFibreTypes; ++i) {
        ConfigSetSpecPivotsFlag(i, UseFibreType[i], &status);
    }
    
/*  Set sky counts for each fibre type - these are the number of fibres of
 *  this type that are to be allocated to sky targets (if available) or
 *  left unallocated for later allocation to sky targets. We clear out all
 *  the sky counts, then set those for the fibre types for which a value
 *  was set using SetSkyCounts{}.
 */
    
    ConfigSetSky(-1,-1,&status);
    for (i = 0; i < InstNumFibreTypes; ++i) {
        if (FibreTypeStats[i].FibreSkyCount >= 0) {
            ConfigSetSky(i,FibreTypeStats[i].FibreSkyCount,&status);
        }
    }

/*  Do the allocation  */

    ginterp = interp;
    gecount = 0;

    ConfigAllocate(RptFunction, (void *) interp,
        ConfAllocRpt, (void *) interp, &status);

    if ((status == STATUS__OK) && (gecount < MAX_ERRORS)) {
/*
 *      Allocation ok.
 *      We need to update the sky values, which we can do by invoking
 *      the command used to implement MakeSkyvals.
 */
        return (ConfMakeSkyvals(clientData, interp, argc, argv));
    } else {
        char buff[200];
/*
 *   Note, we must return OK on error - just report details, as
 *   otherwise you are not allowed to fix what goes wrong, as 
 *   various bits of the code allow.  E.q. autorealocate (-q switch).
 */
        MessGetMsg(status, 0, sizeof(buff), buff);
        ErsRep(ERS_M_BELL, &status, "Error Performing Allocation:-");
        if (status != STATUS__OK)
            ErsRep(ERS_M_BELL, &status, buff);
        ErsFlush(&status);
        return TCL_OK;

    }

}

/*  ------------------------------------------------------------------------- */

/*
 *                          D o  C h e c k  { }
 */
 
static int ConfDoCheck(ClientData clientData DUNUSED, Tcl_Interp * interp,
    int argc, char *argv[])

/*+
 *  Command name:
 *    DoCheck
 *
 *  Function:
 *    Check validity of a configuration
 *  
 *  Call:
 *    DoCheck id positionerTolerances
 *
 *  Description:
 *    Perform a check of the validity of a configurastion by calling the
 *    routines ConfigSetTargets. The Tcl procedure AllUpdate will be called 
 *    to report the progress of the checking.
 *   
 *   Parameters:
 *
 *    (>) id         Sds Id)   SDS identifier of configuration structure.
 *    (>) positionerTolerances (int) True (non-zero) if the check is to use
 *                   the same tolerances as the positioner itself. This is
 *                   the default. If the value is explicitly passed as zero
 *                   then the current tolerances (usually set through the
 *                   allocation dialogue) will be used.
 *
 *   Returns:
 *      "OK" if the configuration is valid, "Error" otherwise.
 *
 *   Support:
 *     Jeremy Bailey, AAO
 *
 *   Version Date:
 *     6-Oct-1994 
 *     3-Jan-2002. Added PositionerTolerances optional argument.   
 *-
 */
{
    StatusType status = STATUS__OK;
    int PositionerTolerances = 1;
    
/*  Check on positioner tolerances argument, if supplied */

    if (argc >= 3) {
        if (atoi(argv[2]) == 0) PositionerTolerances = 0;
    }

/*  Set Target SDS structure  */

    ConfigSetTargets(NULL, atol(argv[1]), CurrentField,
        (CONFIG_ReportFunction) ReportRoutine, interp, &status);

/*  Do the check here explicitly, since we took the check line out of
 *  ConfigInitTargets to allow auto-reallocation ******GBD***** */

    ConfigFieldCheck((CONFIG_ReportFunction) ReportRoutine, interp, 
                                               PositionerTolerances,&status);

    if (status == STATUS__OK) {
        interp->result = "OK";
        return TCL_OK;
    } else {
        ErsFlush(&status);
        interp->result = "Error";
        return TCL_OK;
    }

}


/*  ------------------------------------------------------------------------- */

/*
 *                          D o  C h e c k  1  { }
 */
 
static int ConfDoCheck1(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc DUNUSED, char *argv[])

/*+
 *  Command name:
 *    DoCheck1
 *
 *  Function:
 *    Check validity of a configuration
 *  
 *  Call:
 *    DoCheck1 id
 *
 *  Description:
 *    Perform a check of the validity of a configurastion by calling the
 *    routines ConfigSetTargets. This is similar to DoCheck{} but is intended
 *    to be used when an initial allocation and check is run in batch mode.
 *    In fact the distinction seems to have been lost at some point in the
 *    development of the code, and this and ConfDoCheck() are now identical,
 *    except that this routine always uses the positioner tolerances. 
 *   
 *   Parameters:
 *
 *    (>) id  (Sds Id)   SDS identifier of configuration structure.
 *
 *   Returns:
 *      "OK" if the configuration is valid, "Error" otherwise.
 *
 *   Support:
 *     Jeremy Bailey, AAO
 *
 *   Version Date:
 *     6-Oct-1994    
 *-
 */
{
    StatusType status = STATUS__OK;

/*  Set Target SDS structure  */

    ConfigSetTargets(NULL, atol(argv[1]), CurrentField,
        (CONFIG_ReportFunction) ReportRoutine, interp, &status);

    ConfigFieldCheck((CONFIG_ReportFunction) ReportRoutine, interp, 1, &status);

    if (status == STATUS__OK) {
        interp->result = "OK";
        return TCL_OK;
    } else {
        ErsFlush(&status);
        interp->result = "Error";
        return TCL_OK;
    }

}

/*  ------------------------------------------------------------------------- */

/*
 *                    D e a l l o c a t e  F i b r e  { }
 */
 
static int ConfDeallocateFibre(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])

/*+
 *  Command name:
 *    DeAllocateFibre
 *
 *  Function:
 *    Deallocate a fibre
 *  
 *  Call:
 *    DeAllocateFibre id pivid fibre type object
 *
 *  Description:
 *    Deallocate a fibre. Return the index of the deallocated object
 *    in the unallocated objects or guide structure. 
 *
 *    If the fibre has an index entry set, this operation simply
 *    involves removing the index entry and unsetting the allocated
 *    flag for the object. However if there is no index entry, it
 *    is necessary to enlarge the unallocated objects structure and
 *    copy the object data into the new entry.
 *
 *    In either case the fibre position is set back to the parked
 *    position.
 *   
 *   Parameters:
 *
 *    (>) id  (Sds Id)   SDS identifier of configuration structure.
 *    (>) pivid (Sds Id) SDS identifier of the pivots structure.
 *    (>) fibre (int)    Fibre (pivot) number - starting from one.
 *    (<) type  (char)   "uobj" or "uguide".
 *    (<) object (int)   Object number - starting from one.
 *
 *   Returns:
 *     "OK" if succesful, "Error" otherwise
 *
 *   Support:
 *     Jeremy Bailey, AAO
 *
 *   Version Date:
 *      6-Oct-1994  Original version. JAB.    
 *      8-Nov-2001  Now updates fibre type allocation statistics. KS.   
 *      3-Sep-2002  Now updates available target allocation statistics. KS.   
 *-
 */
{
    SdsIdType topid, objid, uid, pivid, fid;
    long fibre_no;
    int fibreType;
    long obj_no;
    StatusType status = STATUS__OK;
    short s;
    char c;
    char guide;
    SdsIdType xpid, ypid, tpid;
    char name[NAMELEN];
    SdsCodeType code;
    long ndims;
    unsigned long dims[7];
    int l;
    double d;
    SdsIdType ra_id;
    unsigned long actlen;
    char ctype;
    char buf[20];
    int spec;
    int NumPivTypes;
    int PivTypes[10];
    short IsSky;
    int PivIndex;

    if (argc != 6) {
        interp->result = "Wrong # args";
        return TCL_ERROR;
    }

    topid = (SdsIdType) (atol(argv[1]));
    pivid = (SdsIdType) (atol(argv[2]));
    fibre_no = atol(argv[3]) - 1;
    spec = ConfWhichSpec((int) fibre_no, 0);

    if ((fibre_no < 0) || (fibre_no >= (int) FpilGetNumPivots(*Instrument))) {
        sprintf(interp->result,
            "Illegal Pivot Number %ld\nMust be between 1 and %d",
            fibre_no + 1, FpilGetNumPivots(*Instrument));
        return TCL_ERROR;
    } else if (!UseFibreType[spec]) {
        if (spec == 0) {

            sprintf(interp->result,
                "Attempt to deallocate guide pivot %ld,", fibre_no + 1);
            strcat(interp->result,
                " when guide pivot deallocation is disabled");
        } else {
            const char *SpecDescr;
            ConfWhichSpec((int) fibre_no, &SpecDescr);
            sprintf(interp->result,
                "Attempt to deallocate pivot %ld, from spectrograph (%s)",
                fibre_no + 1, SpecDescr);
            strcat(interp->result, " marked as not to be deallocated.");
        }
        return TCL_ERROR;
    }


    SdsFind(topid, "objects", &objid, &status);
    ConfSdsCheck("Error finding objects structure", &status);

/*  Check fibre is allocated  */

    ConfArrayGetc(objid, "type", fibre_no, &ctype, &status);

    if (status != STATUS__OK) {
        char buff[200];
        MessGetMsg(status, 0, sizeof(buff), buff);
        Tcl_AppendResult(interp, "\nError getting fibre details",
            (char *) NULL);
        Tcl_AppendResult(interp, buff, (char *) NULL);
        ErsFlush(&status);
        return TCL_ERROR;
    }


    if (ctype == FpilUnallocatedType(*Instrument)) {
        sprintf(interp->result,
            "Fibre at pivot number %ld is not allocated", fibre_no);
        return TCL_ERROR;
    }

    if (FpilIsTargetGuide(*Instrument, ctype)) {
        guide = 1;
        SdsFind(topid, "unallocGuide", &uid, &status);
        Tcl_SetVar(interp, argv[4], "uguide", 0);
    } else {
        guide = 0;
        SdsFind(topid, "unallocObject", &uid, &status);
        Tcl_SetVar(interp, argv[4], "uobj", 0);
    }

/*  Get fibre index  */

    ConfArrayGets(objid, "index", fibre_no, &s, &status);
    if (s != -1) {

/*  The object has an index specified - This is the easy case.
    We just unset the allocated flag  */
        obj_no = s;
        c = 0;
        ConfArrayPutc(uid, "allocated", obj_no, c, &status);
        sprintf(buf, "%ld", obj_no + 1);
        Tcl_SetVar(interp, argv[5], buf, 0);
    }

/*  Otherwise we have to enlarge the unallocated objects section
    and copy the data back to it  */

    else {

        printf("Enlarging unalloctated objects section ");
        printf("should not be needed now)\n");

        SdsFind(uid, "ra", &ra_id, &status);
        SdsInfo(ra_id, name, &code, &ndims, dims, &status);


        SdsFreeId(ra_id, &status);
        ConfResizeAllItems(uid, dims[0] + 1, &status);
        obj_no = dims[0];
        sprintf(buf, "%ld", obj_no + 1);
        Tcl_SetVar(interp, argv[5], buf, 0);


        ConfArrayCopy(objid, uid, "name", NAMELEN, NAMELEN * fibre_no,
            NAMELEN * obj_no, &status);
        ConfArrayCopy(objid, uid, "ra", sizeof(double), fibre_no, obj_no,
            &status);
        ConfArrayCopy(objid, uid, "dec", sizeof(double), fibre_no, obj_no,
            &status);
        ConfArrayCopy(objid, uid, "x", sizeof(int), fibre_no, obj_no, &status);
        ConfArrayCopy(objid, uid, "y", sizeof(int), fibre_no, obj_no, &status);
        ConfArrayCopy(objid, uid, "type", sizeof(char), fibre_no, obj_no,
            &status);
        ConfArrayCopy(objid, uid, "priority", sizeof(short), fibre_no, obj_no,
            &status);
        ConfArrayCopy(objid, uid, "magnitude", sizeof(double), fibre_no, obj_no,
            &status);
        ConfArrayCopy(objid, uid, "pId", sizeof(short), fibre_no, obj_no,
            &status);
        ConfArrayCopy(objid, uid, "comment", CMTLEN, CMTLEN * fibre_no,
            CMTLEN * obj_no, &status);
        if (guide) {
            ConfArrayCopy(objid, uid, "dhdx", sizeof(double), fibre_no,
                obj_no, &status);
            ConfArrayCopy(objid, uid, "dhdy", sizeof(double), fibre_no,
                obj_no, &status);
            ConfArrayCopy(objid, uid, "dddx", sizeof(double), fibre_no,
                obj_no, &status);
            ConfArrayCopy(objid, uid, "dddy", sizeof(double), fibre_no,
                obj_no, &status);
        } else
            ConfArrayCopy(objid, uid, "spectrograph", sizeof(char),
                fibre_no, obj_no, &status);
        ConfArrayPutc(uid, "allocated", obj_no, 0, &status);

    }

/*  Now set the objects section entry back to the parked position */

    SdsFind(pivid, "xPark", &xpid, &status);
    SdsFind(pivid, "yPark", &ypid, &status);
    SdsFind(pivid, "tPark", &tpid, &status);

    strcpy(name, "Parked");
    ConfArrayPutString(objid, "name", fibre_no, name, &status);
    ConfArrayPutd(objid, "ra", fibre_no, 0.0, &status);
    ConfArrayPutd(objid, "dec", fibre_no, 0.0, &status);
    ConfArrayPuts(objid, "fibre", fibre_no, 0, &status);
    SdsGet(xpid, sizeof(int), fibre_no, &l, &actlen, &status);
    ConfArrayPuti(objid, "x", fibre_no, l, &status);
    SdsGet(ypid, sizeof(int), fibre_no, &l, &actlen, &status);
    ConfArrayPuti(objid, "y", fibre_no, l, &status);
    SdsGet(tpid, sizeof(double), fibre_no, &d, &actlen, &status);
    ConfArrayPutd(objid, "theta", fibre_no, d, &status);
    ConfArrayPutc(objid, "type", fibre_no,
        FpilUnallocatedType(*Instrument), &status);
    ConfArrayPuts(objid, "priority", fibre_no, 1, &status);
    ConfArrayPutd(objid, "magnitude", fibre_no, 0.0, &status);
    ConfArrayPutc(objid, "spectrograph", fibre_no, ' ', &status);
    ConfArrayPuts(objid, "pId", fibre_no, 1, &status);
    ConfArrayPutd(objid, "dhdx", fibre_no, 0.0, &status);
    ConfArrayPutd(objid, "dhdy", fibre_no, 0.0, &status);
    ConfArrayPutd(objid, "dddx", fibre_no, 0.0, &status);
    ConfArrayPutd(objid, "dddy", fibre_no, 0.0, &status);
    ConfArrayPutString(objid, "comment", fibre_no, name, &status);
    ConfArrayPuts(objid, "index", fibre_no, -1, &status);

    SdsFreeId(xpid, &status);
    SdsFreeId(ypid, &status);
    SdsFreeId(tpid, &status);

/* We need to update the allocation statistics held for this fibre type */

    fibreType = ConfWhichSpec((int) fibre_no, 0);
    IsSky = FpilIsTargetSky(*Instrument, ctype);
    if ((fibreType >= 0) && (fibreType < InstNumFibreTypes)) {
        FibreTypeStats[fibreType].Unallocated++;
        if (FpilIsTargetGuide(*Instrument, ctype)) {
            FibreTypeStats[fibreType].AllocToGuide--;
        } else if (IsSky) {
            FibreTypeStats[fibreType].AllocToSky--;
        } else {
            FibreTypeStats[fibreType].AllocToObject--;
        }
    }

/* Also, given this target type (in 'ctype') we need to calculate how this
 * deallocation affects the possible sky and object targets for the different
 * fibre types - it might be a target that can be allocated to a
 * number of different fibre types
 */
 
    FpilCompatiblePivotTypes (*Instrument,10,ctype,0,&NumPivTypes,PivTypes);
    for (PivIndex = 0; PivIndex < NumPivTypes; PivIndex++) {
        fibreType = PivTypes[PivIndex];
        if (IsSky) {
            FibreTypeStats[fibreType].Skies++;
        } else {
            FibreTypeStats[fibreType].Targets++;
        }
    }
    
    if (status == STATUS__OK) {
        SdsFind(topid, "fieldData", &fid, &status);
        ConfSdsCheck("Error finding fieldData structure", &status);
    }
    if (status == STATUS__OK) {
        if (guide) {
            ArgGets(fid, "unallocGui", &s, &status);
            s++;
            ArgPuts(fid, "unallocGui", s, &status);
            ArgGets(fid, "allocGui", &s, &status);
            if (status == SDS__NOITEM) {
                ErsAnnul(&status);
                s = 1;
            } else
                s--;
            ArgPuts(fid, "allocGui", s, &status);
        } else {
            if (!FpilIsTargetGuide(*Instrument, ctype) &&
                !FpilIsTargetSky(*Instrument, ctype)) {
                /*  Not sky and not a guide object, so this is a 'program'
                 *  object.
                 */

                ArgGets(fid, "unallocObj", &s, &status);
                s++;
                ArgPuts(fid, "unallocObj", s, &status);
                ArgGets(fid, "allocObj", &s, &status);
                if (status == SDS__NOITEM) {
                    ErsAnnul(&status);
                    s = 1;
                } else
                    s--;
                ArgPuts(fid, "allocObj", s, &status);
            } else if (FpilIsTargetSky(*Instrument, ctype)) {
                char name[20];

                /* Update unallocSky and allocSky */
                ArgGets(fid, "unallocSky", &s, &status);
                ArgPuts(fid, "unallocSky", s + 1, &status);
                ArgGets(fid, "allocSky", &s, &status);
                if (status == SDS__NOITEM) {
                    ErsAnnul(&status);
                    s = 0;
                } else
                    s--;
                ArgPuts(fid, "allocSky", s, &status);

                /* Update allocSkyn where n = spectrograph number */
                sprintf(name, "allocSky%d", spec);
                ArgGets(fid, "name", &s, &status);
                if (status == SDS__NOITEM) {
                    ErsAnnul(&status);
                    s = 0;
                } else
                    s--;
                ArgPuts(fid, name, s, &status);
            }
        }
    }
    SdsFreeId(objid, &status);
    SdsFreeId(uid, &status);
    SdsFreeId(fid, &status);

    if (status == STATUS__OK) {
        interp->result = "OK";
        return TCL_OK;
    } else {

        char buff[200];
        sprintf(buff, "\nError Deallocating Fibre %ld\n", fibre_no);
        Tcl_AppendResult(interp, buff, (char *) NULL);

        MessGetMsg(status, 0, sizeof(buff), buff);
        Tcl_AppendResult(interp, buff, (char *) NULL);
        return TCL_ERROR;
    }

}

/*  ------------------------------------------------------------------------- */

/*
 *                  U p d a t e  F i b r e  S t a t s  { }
 */
 
static int ConfUpdateFibreStats(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])

/*+
 *  Command name:
 *    UpdateFibreStats
 *
 *  Function:
 *    Makes sure the fibre allocation statistics reflect the configuration.
 *  
 *  Call:
 *    UpdateFibreStats id
 *
 *  Description:
 *    The global array FibreTypeStats is used to maintain allocation 
 *    statistics for all the different fibre types. It is updated when
 *    a fibre is allocated or deallocated, but needs to be initialised
 *    both at the start of the program or when a configuration is read
 *    in. This routine performs this initialisation.
 *   
 *   Parameters:
 *
 *    (>) id  (Sds Id)   SDS identifier of configuration structure.
 *
 *   Returns:
 *     "OK" if succesful, some error indication otherwise
 *
 *   Support:
 *     Keith Shortridge, AAO
 *
 *   Version Date:
 *      6th Nov 2001. Original version. KS. 
 *     28th Nov 2001. Removed unused variable FibreInUse. KS.
 *      3rd Sep 2002. Now calculates available sky and object statistics for
 *                    each fibre type. KS.   
 *-
 */
{
    const char *const *SpecNames;
    unsigned int NumFibreTypes;
    StatusType status = STATUS__OK;
    char name[NAMELEN];
    int Index;
    SdsIdType topid, objid, typeId, allocId, spectId, xId, yId;
    SdsCodeType code;
    short AllocArray;
    long ndims;
    unsigned long dims[7];
    int XPosn,YPosn;
    int NumFibres;
    int NumObjects;
    char Alloc;
    char Spect;
    short SpectKnown;
    unsigned long actlen;
    unsigned int FibreType;
    const char *FibreTypeDescr;
    char Type;
    char UnallocTypeCode;
    int ReturnCode;
    int PivIndex;
    int NumPivTypes;
    int PivTypes[10];
    short IsSky;
    
    if (argc != 2) {
        interp->result = "Wrong # args";
        return TCL_ERROR;
    }
    
    objid = 0;
    typeId = 0;

    /*  Clear out the fibre type statistics array. Clear out the counts.
     */
    
    FpilConSpecInfo(*Instrument, &NumFibreTypes, &SpecNames, &status);
    for (Index = 0; Index < NumFibreTypes; Index++) {
       FibreTypeStats[Index].Name[0] = '\0';
       FibreTypeStats[Index].Fibres = 0;
       FibreTypeStats[Index].AllocToSky = 0;
       FibreTypeStats[Index].AllocToGuide = 0;
       FibreTypeStats[Index].AllocToObject = 0;
       FibreTypeStats[Index].Unallocated = 0;
       FibreTypeStats[Index].Targets = 0;
       FibreTypeStats[Index].Skies = 0;
    }
       
    /*  For all the types initialise the counts for things that are nothing
     *  to do with allocation, such as the total number of fibres of 
     *  each type, and the number of each type that are not in use (probably
     *  because they're broken). We start by assuming all fibres are 
     *  unallocated. Note that ConfWhichSpec() can return a code that is
     *  outside the range (0..NumFibreTypes-1) - it uses such codes for
     *  disabled fibres.
     */

    NumFibres = FpilGetNumPivots(*Instrument);
    for (Index = 0; Index < NumFibres; Index++) {
        FibreType = ConfWhichSpec(Index, &FibreTypeDescr);
        if ((FibreType >= 0) &&(FibreType < NumFibreTypes)) {
            if (FibreTypeStats[FibreType].Name[0] == '\0') {
                strncpy (FibreTypeStats[FibreType].Name,FibreTypeDescr,
                                     sizeof(FibreTypeStats[FibreType].Name));
                FibreTypeStats[FibreType].Name[
                       sizeof(FibreTypeStats[FibreType].Name) - 1] = '\0';
            }
            FibreTypeStats[FibreType].Fibres++;
            FibreTypeStats[FibreType].Unallocated++;
        }
    }
    
    /*  Get the Sds Id for the configuration data from the passed argument,
     *  then see if there is an objects section. If not, no targets have
     *  been allocated.
     */
   
    objid = typeId = allocId = spectId =  xId = yId = 0;
    topid = (SdsIdType) (atol(argv[1]));
    
    SdsFind(topid, "objects", &objid, &status);
    if (status == STATUS__OK) {
    
        /*  There is an "objects" section, so we can start to work through 
         *  it, building up the counts. Get the Sds Ids for the one array that
         *  we are going to make use of, namely the 'type' array.
         */

        SdsFind(objid, "type", &typeId, &status);
        SdsInfo(typeId, name, &code, &ndims, dims, &status);
        NumFibres = dims[0];

        UnallocTypeCode = FpilUnallocatedType(*Instrument);

        /*  Loop through each item in the objects structure. Remember that
         *  this contains information for one fibre, since there are is one
         *  item in the objects structure for each fibre on the instrument.
         */

        for (Index = 0; Index < NumFibres; Index++) {

            /*  For this fibre, we need to know which fibre type it represents.
             *  We could use the information in the pivot structure, assuming
             *  (knowing!) that there is a one to one mapping between items in
             *  there and in the objects section. However, it's probably better
             *  to use the usual routine for this purpose, namely a call to
             *  ConfWhichSpec().
             */

            FibreType = ConfWhichSpec(Index, &FibreTypeDescr);
            if ((FibreType >= 0) &&(FibreType <= NumFibreTypes)) {
                SdsGet(typeId, sizeof(char), Index, &Type, &actlen, &status);
                if (Type != UnallocTypeCode) {

                    /*  If the fibre is allocated, see to what sort of object 
                     *  it is allocated. The "type" information is enough 
                     *  here. If we wanted a more complex set of statistics - 
                     *  some magnitude details, for example - we would have 
                     *  to use other items in the objects structure.
                     */

                    FibreTypeStats[FibreType].Unallocated--;
                    if (FpilIsTargetGuide(*Instrument,Type)) {
                        FibreTypeStats[FibreType].AllocToGuide++;
                    } else if (FpilIsTargetSky(*Instrument,Type)) {
                        FibreTypeStats[FibreType].AllocToSky++;
                    } else {
                        FibreTypeStats[FibreType].AllocToObject++;
                    }
                }
            }
        }
    }
    
    /*  Release Sds Ids */
    
    if (objid) SdsFreeId(objid, &status);
    if (typeId) SdsFreeId(typeId, &status);
    objid = typeId = allocId = spectId = xId = yId = 0;
    
    /*  Now, we need to work out the unallocated targets compatible with 
     *  each fibre type. We have to look through the unallocated objects 
     *  and the unallocated guides to build up these statistics.
     */
     
    SdsFind(topid, "unallocObject", &objid, &status);
    if (status == STATUS__OK) {
    
        /*  There is an "unallocObject" section, so we can start to work through 
         *  it, building up the counts. Get the Sds Ids for the arrays that
         *  we are going to make use of, namely the 'type', 'spect' and
         *  'allocated' arrays - the allocated array may not exist, nor may
         *  the spect array. We also need the x and y positions to see if the
         *  target is in the field or not.
         */

        SdsFind(objid, "type", &typeId, &status);
        SdsInfo(typeId, name, &code, &ndims, dims, &status);
        NumObjects = dims[0];
        AllocArray = TRUE;
        SdsFind(objid, "allocated", &allocId, &status);
        if (status != 0) {
           AllocArray = FALSE;
           status = 0;
        }
        SpectKnown = TRUE;
        SdsFind(objid, "spectrograph", &spectId, &status);
        if (status != 0) {
           SpectKnown = FALSE;
           status = 0;
        }
        SdsFind(objid, "x", &xId, &status);
        SdsFind(objid, "y", &yId, &status);

        for (Index = 0; Index < NumObjects; Index++) {

            /*  For this target, we want to check that it isn't allocated,
             *  and then find out which pivot types it is compatible with.
             */

            SdsGet(typeId, sizeof(char), Index, &Type, &actlen, &status);
            SdsGet(xId, sizeof(int), Index, &XPosn, &actlen, &status);
            SdsGet(yId, sizeof(int), Index, &YPosn, &actlen, &status);
            Alloc = 0;
            if (AllocArray) {
               SdsGet(allocId, sizeof(char), Index, &Alloc, &actlen, &status);
            }
            Spect = 0;
            if (SpectKnown) {
               SdsGet(spectId, sizeof(char), Index, &Spect, &actlen, &status);
            }
            if (FpilOnField (*Instrument,XPosn,YPosn)) {
               if (!Alloc) {
                  IsSky = FpilIsTargetSky (*Instrument,Type);
                  FpilCompatiblePivotTypes (*Instrument,10,Type,Spect,
                                                     &NumPivTypes,PivTypes);
                  for (PivIndex = 0; PivIndex < NumPivTypes; PivIndex++) {
                     FibreType = PivTypes[PivIndex];
                     if (IsSky) {
                        FibreTypeStats[FibreType].Skies++;
                     } else {
                        FibreTypeStats[FibreType].Targets++;
                     }
                  }
               }
            }
        }
        if (typeId) SdsFreeId(typeId, &status);
        if (spectId) SdsFreeId(spectId, &status);
        if (AllocArray) SdsFreeId(allocId, &status);
    }
    if (objid) SdsFreeId(objid, &status);
    objid = typeId = allocId = spectId = xId = yId = 0;
    
    SdsFind(topid, "unallocGuide", &objid, &status);
    if (status == STATUS__OK) {
    
        /*  There is an "unallocGuide" section, so we can start to work through 
         *  it, just as for the "unallocObject" section.
         */

        SdsFind(objid, "type", &typeId, &status);
        SdsInfo(typeId, name, &code, &ndims, dims, &status);
        NumObjects = dims[0];
        AllocArray = TRUE;
        SdsFind(objid, "allocated", &allocId, &status);
        if (status != 0) {
           AllocArray = FALSE;
           status = 0;
        }
        SpectKnown = TRUE;
        SdsFind(objid, "spectrograph", &spectId, &status);
        if (status != 0) {
           SpectKnown = FALSE;
           status = 0;
        }
        SdsFind(objid, "x", &xId, &status);
        SdsFind(objid, "y", &yId, &status);

        for (Index = 0; Index < NumObjects; Index++) {

            /*  For this target, we want to check that it isn't allocated,
             *  and then find out which pivot types it is compatible with.
             */

            SdsGet(typeId, sizeof(char), Index, &Type, &actlen, &status);
            SdsGet(xId, sizeof(int), Index, &XPosn, &actlen, &status);
            SdsGet(yId, sizeof(int), Index, &YPosn, &actlen, &status);
            Alloc = 0;
            if (AllocArray) {
               SdsGet(allocId, sizeof(char), Index, &Alloc, &actlen, &status);
            }
            Spect = 0;
            if (SpectKnown) {
               SdsGet(spectId, sizeof(char), Index, &Spect, &actlen, &status);
            }
            if (FpilOnField (*Instrument,XPosn,YPosn)) {
               if (!Alloc) {
                  FpilCompatiblePivotTypes (*Instrument,10,Type,Spect,
                                                     &NumPivTypes,PivTypes);
                  for (PivIndex = 0; PivIndex < NumPivTypes; PivIndex++) {
                     FibreType = PivTypes[PivIndex];
                     FibreTypeStats[FibreType].Targets++;
                  }
               }
            }
        }
       if (typeId) SdsFreeId(typeId, &status);
       if (spectId) SdsFreeId(spectId, &status);
       if (AllocArray) SdsFreeId(allocId, &status);
    }
    if (objid) SdsFreeId(objid, &status);
    
    /*  Report status */
    
    if (status == STATUS__OK) {
        interp->result = "OK";
        ReturnCode = TCL_OK;
    } else {
        interp->result = "Error";
        ReturnCode = TCL_ERROR;
    }
    
    return ReturnCode;
}


/*  ------------------------------------------------------------------------- */

/*
 *                    G e t  F i b r e  S t a t s  { }
 */
 
static int ConfGetFibreStats(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])

/*+
 *  Command Name:
 *    GetFibreStats
 *
 *  Function:
 *    Get allocation statistics for a fibre type
 *
 *  Call:
 *    GetFibreStats index name enabled fibres skies guides 
 *                           objects unalloc sameAs mayShare targets skies
 *
 *  Description:
 *    Return information about the allocation statistics for a fibre type.
 *    Each fibre type has an index number, starting at one, corresponding to
 *    the index for the corresponding fibre type returned by FibreTypes{}.
 *
 *  Parameters:
 *    (>)  index (int)     index number - starting at zero
 *    (<)  name  (char)    The fibre type name.
 *    (<)  enabled (int)   True (non-zero) if the fibre type is currently
 *                         allocated.
 *    (<)  fibres (int)    Total number of fibres of this type.
 *    (<)  skies (int)     Number of fibres of this type allocated to sky.
 *    (<)  guides (int)    Number of fibres of this type allocated to guide
 *                         targets.
 *    (<)  objects (int)   Number of fibres of this type allocated to program
 *                         targets - ie not to sky or guide targets.
 *    (<)  unalloc (int)   Number of fibres of this type unallocated.
 *    (<)  sameAs (int)    If the fibre type is treated as being the same as
 *                         a lower-numbered fibre type for the purposes of
 *                         classifying allocation results, this is the index
 *                         of that lower-numbered type. If negative, there
 *                         is no such lower index.
 *    (<)  mayShare (int)  If the fibre type is one that can be allocated to
 *                         both sky and non-sky targets, this is set to 1.
 *                         Otherwise, it is set to zero.
 *    (<)  targets (int)   Number of non-sky targets available to this fibre
 *                         type - these are unallocated non-sky targets 
 *                         within the field that are compatible with this type.
 *                         Note that many targets are compatible with a number
 *                         of different fibre types.
 *    (<)  skies (int)     Number of sky targets available to this fibre
 *                         type - these are unallocated sky targets 
 *                         within the field that are compatible with this type.
 *                         Note that many targets are compatible with a number
 *                         of different fibre types.
 *
 *
 *   Support:
 *     Keith Shortridge, AAO
 *
 *   Version Date:
 *      8-Nov-2001  KS  Original version.
 *     19-Nov-2001  KS  Added sameAs and mayShare. 
 *      3-Sep-2002  KS  Added targets and skies. 
 *-
 */
{
    char buffer[80];
    int fibreIndex;
    int Index;

    if (argc != 13) {
        interp->result = "wrong # args";
        return TCL_ERROR;
    }
    fibreIndex = atol(argv[1]);
    if ((fibreIndex < 0) || (fibreIndex >= InstNumFibreTypes)) {
        interp->result = "invalid index for GetFibreStats";
        return TCL_ERROR;
    }

    Tcl_SetVar(interp, argv[2], FibreTypeStats[fibreIndex].Name, 0);
    sprintf(buffer, "%d", UseFibreType[fibreIndex]);
    Tcl_SetVar(interp, argv[3], buffer, 0);
    sprintf(buffer, "%d", FibreTypeStats[fibreIndex].Fibres);
    Tcl_SetVar(interp, argv[4], buffer, 0);
    sprintf(buffer, "%d", FibreTypeStats[fibreIndex].AllocToSky);
    Tcl_SetVar(interp, argv[5], buffer, 0);
    sprintf(buffer, "%d", FibreTypeStats[fibreIndex].AllocToGuide);
    Tcl_SetVar(interp, argv[6], buffer, 0);
    sprintf(buffer, "%d", FibreTypeStats[fibreIndex].AllocToObject);
    Tcl_SetVar(interp, argv[7], buffer, 0);
    sprintf(buffer, "%d", FibreTypeStats[fibreIndex].Unallocated);
    Tcl_SetVar(interp, argv[8], buffer, 0);
    strcpy (buffer,"-1");
    for (Index = 0; Index < fibreIndex; Index++) {
        if (FpilPivotTypeMatch(*Instrument,Index,fibreIndex)) {
            sprintf(buffer, "%d", Index);
            break;
        }
    }
    Tcl_SetVar(interp, argv[9], buffer, 0);
    if (FpilPivotSharesSky(*Instrument,fibreIndex)) {
       Tcl_SetVar(interp, argv[10], "1", 0);
    } else {
        Tcl_SetVar(interp, argv[10], "0", 0);
    }      
    sprintf(buffer, "%d", FibreTypeStats[fibreIndex].Targets);
    Tcl_SetVar(interp, argv[11], buffer, 0);
    sprintf(buffer, "%d", FibreTypeStats[fibreIndex].Skies);
    Tcl_SetVar(interp, argv[12], buffer, 0);
    
    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*
 *                  S e t  S k y  C o u n t s  { }
 */
 
static int ConfSetSkyCounts(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])

/*+
 *  Command name:
 *    SetSkyCounts
 *
 *  Function:
 *    Sets the number of sky targets to be allocated to a given fibre type.
 *  
 *  Call:
 *    SetSkyCounts
 *    SetSkyCounts Index Count
 *
 *  Description:
 *    The global array FibreTypeStats is used to maintain allocation 
 *    statistics for all the different fibre types. It also contains a 
 *    field FibreSkyCount which gives the number of fibres of this (or
 *    equivalent) types to be allocated to sky targets. (Or left
 *    unallocated for later allocation to sky, if there are no sky targets
 *    available.) If the form with no arguments is used, the counts for all
 *    types are set to -1, which indicates that no control is to be 
 *    exercised over the number of sky targets for the type in question.
 *   
 *   Parameters:
 *
 *    (>) Index  (int)   Fibre index, from 0 up. This should match a value
 *                       returned by FpilConfWhichSpect().
 *    (>) Count  (int)   Number of sky targets for this fibre type.
 *
 *   Returns:
 *     "OK" if succesful, some error indication otherwise
 *
 *   Support:
 *     Keith Shortridge, AAO
 *
 *   Version Date:
 *     19th Nov 2001. Original version. KS.    
 *-
 */
{
    const char *const *SpecNames;
    unsigned int NumFibreTypes;
    StatusType status = STATUS__OK;
    int Index;
    int ReturnCode;
    
    if ((argc != 1) && (argc != 3)) {
        interp->result = "Wrong # args";
        ReturnCode = TCL_ERROR;
    } else {
    
        FpilConSpecInfo(*Instrument, &NumFibreTypes, &SpecNames, &status);
        if (argc == 1) {

            /*  This is the form in which we simply clear out all the sky
             *  counts for each fibre type.
             */

            for (Index = 0; Index < NumFibreTypes; Index++) {
                FibreTypeStats[Index].FibreSkyCount = -1;
            }
            ReturnCode = TCL_OK;

        } else {

            /*  This is the form where we set the count for a given type */

            Index = atoi(argv[1]);
            if ((Index < 0) || (Index >= NumFibreTypes)) {
                sprintf(interp->result,"Index %d is invalid",Index);
                ReturnCode = TCL_ERROR;
            } else {
                FibreTypeStats[Index].FibreSkyCount = atoi(argv[2]);
                interp->result = "OK";
                ReturnCode = TCL_OK;
            }
        }
    }   
    
    return ReturnCode;
}

/*  ------------------------------------------------------------------------- */

/*
 *                     S d s  F i l e  N a m e  { }
 *
 *                  N e w  S d s  F i l e  N a m e  { }
 *
 *                   A s c i i  F i l e  N a m e  { }
 *
 *                  I m p o r t  F i l e  N a m e  { }
 *
 *                     D S S  F i l e  N a m e  { }
 *
 *                     L i s t  F i l e  N a m e  { }
 *
 */
 
static int ConfFileName(ClientData clientData, Tcl_Interp * interp,
    int argc DUNUSED, char *argv[])

/*+
 *
 *  Command names:
 *    SdsFileName
 *    NewSdsFileName
 *    AsciiFileName
 *    ImportFileName
 *    DSSFileName
 *    ListFileName
 *
 *  Function:
 *    Return name of Sds/Fld/Imp/Dss/list file corresponding to a 
 *    configuration file.
 *  
 *  Call:
 *    {cmd} name
 *
 *  Description:
 *    Return the name of file corresponding to the specified
 *    configuration file. This simply involves replacing whatever
 *    file extension is present with one of /sds/fld/imp/dss/lis, as
 *    specified in the client data.  For NewSdsFileName, it involves
 *    add _new.sds to the file name. So the differences between the way
 *    these different routines work is all 'hidden' in the way the commands
 *    are defined in ConfAddCommands().
 *   
 *   Parameters:
 *
 *    (<) name  (char)   Name of configuration file
 *
 *
 *   Support:
 *     Tony Farrell, AAO
 *
 *   Version Date:
 *     6-Oct-1994    
 *-
 */
{
    int l;
    if (argc != 2) {
        sprintf(interp->result, "%s:wrong # args", argv[0]);
        return TCL_ERROR;
    }
/*
 * Copy the input string to the result.
 */
    Tcl_SetResult(interp, argv[1], TCL_VOLATILE);

/*
 *  Find the period, indicating where the file type is.
 */
    l = strlen(argv[1]);
    for (l = strlen(argv[1]); l > 0; l--) {
        if (argv[1][l] == '.')
            break;
    }
/*
 *  Terminate the result so that we can add our desired type.
 *  (it is a bit naughty to assume we can write to interp->result[]
 *  here, it relies on an old behaviour of Tcl.  But is so hard to
 *  do it right, and the old behaviour really should keep working, there
 *  is too much old code out there.). In any case, this is done in many
 *  places in the code - although it would not be a huge job to update
 *  them to use a more strictly correct mechanism.
 */
    if (l > 0)
        interp->result[l] = 0;
/*
 * Append the desired file type.
 */
    Tcl_AppendResult(interp, (char *) (clientData), NULL);
    return TCL_OK;
}


/*  ------------------------------------------------------------------------- */

/*
 *                 C o m p r e s s  U n a l l o c  { }
 */
 
static int ConfCompressUnalloc(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc DUNUSED, char *argv[])

/*+
 *  Command name:
 *    CompressUnalloc
 *
 *  Function:
 *    Compress the Unallocated objects section
 *  
 *  Call:
 *    CompressUnalloc id
 *
 *  Description:
 *    Compress the unallocated objects and guide sections of a 
 *    configuration structure by removing any objects which have
 *    their "allocated" flag set. Also removes the corresponding "index"
 *    section from the allocated objects structure.
 *   
 *
 *   Parameters:
 *
 *    (<) id  (Sds Id)   SDS id of configuration structure
 *
 *   Support:
 *     Jeremy Bailey, AAO
 *
 *   Version Date:
 *     6-Oct-1994 
 *
 *   Modified:
 *     20th Jan 2003. KS. Failure to find "index" no longer treated
 *                        as an error - just indicates this routine
 *                        has already been run on this structure.   
 *-
 */
{
    SdsIdType topid, uid, objid, indid;
    StatusType status = STATUS__OK;
    StatusType indexStatus = STATUS__OK;

    topid = atol(argv[1]);
    SdsFind(topid, "unallocObject", &uid, &status);
    ConfRemoveAlloc(uid, &status);
    SdsFreeId(uid, &status);
    SdsFind(topid, "unallocGuide", &uid, &status);
    ConfRemoveAlloc(uid, &status);
    SdsFreeId(uid, &status);
    SdsFind(topid, "objects", &objid, &status);
    SdsFind(objid, "index", &indid, &indexStatus);
    if (indexStatus == STATUS__OK) {
        SdsDelete(indid, &status);
        SdsFreeId(indid, &status);
    }
    SdsFreeId(objid, &status);

    if (status == STATUS__OK) {
        return TCL_OK;
    } else {
        sprintf(interp->result,
            "Error Compressing the Unallocated Object Structures");
        return TCL_ERROR;
    }

}


/*  ------------------------------------------------------------------------- */

/*
 *                  N u m b e r  P a r a m e t e r s  { }
 */
 
static int ConfNumberParameters(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc DUNUSED, char *argv[]DUNUSED)

/*+
 *  Command name:
 *    NumberParameters
 *
 *  Function:
 *    Return the number of allocation parameters
 *  
 *  Call:
 *    NumberParameters
 *
 *  Description:
 *    Return the number of allocation parameters. This is a wrap-up routine
 *    to make ConfigNumberParameters() available to Tcl.
 *   
 *   Parameters:
 *
 *   Returns:
 *     The number of allocation parameters
 *
 *   Support:
 *     Jeremy Bailey, AAO
 *
 *   Version Date:
 *     6-Oct-1994    
 *-
 */
{
    int n;

    n = ConfigNumberParameters();
    sprintf(interp->result, "%d", n);
    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*
 *                     G e t  P a r a m e t e r  { }
 */
 
static int ConfGetParameter(ClientData clientData DUNUSED, Tcl_Interp * interp,
    int argc DUNUSED, char *argv[])

/*+
 *  Command name:
 *    GetParameter
 *
 *  Function:
 *    Return the value of an allocation parameter
 *  
 *  Call:
 *    GetParameter name
 *
 *  Description:
 *    Return the value of an allocation parameter.
 *   
 *   Parameters:
 *      (>) name (char)   The parameter name
 *
 *   Returns:
 *     The value of the allocation parameter
 *
 *   Support:
 *     Jeremy Bailey, AAO
 *
 *   Version Date:
 *     6-Oct-1994    
 *-
 */
{
    char value[100];
    StatusType status = STATUS__OK;

    ConfigGetParameter(argv[1], value, 100, &status);

    if (status == STATUS__OK) {
        sprintf(interp->result, "%s", value);
        return TCL_OK;
    } else {
        sprintf(interp->result, "Error Getting Parameter %s", argv[1]);
        return TCL_ERROR;
    }
}

/*  ------------------------------------------------------------------------- */

/*
 *                     S e t  P a r a m e t e r  { }
 */
 
static int ConfSetParameter(ClientData clientData DUNUSED, Tcl_Interp * interp,
    int argc DUNUSED, char *argv[])

/*+
 *  Command name:
 *    SetParameter
 *
 *  Function:
 *    Set the value of an allocation parameter
 *  
 *  Call:
 *    SetParameter name value
 *
 *  Description:
 *    Set the value of an allocation parameter.
 *   
 *   Parameters:
 *      (>) name (char)   The parameter name
 *      (>) value (char)  The parameter value
 *
 *   Support:
 *     Jeremy Bailey, AAO
 *
 *   Version Date:
 *     6-Oct-1994    
 *-
 */
{
    StatusType status = STATUS__OK;

    ConfigSetParameter(argv[1], argv[2], &status);

    if (status == STATUS__OK) {
        return TCL_OK;
    } else {
        sprintf(interp->result,
            "Error Setting Parameter %s to value %s", argv[1], argv[2]);
        return TCL_ERROR;
    }
}


/*  ------------------------------------------------------------------------- */

/*
 *                  P a r a m e t e r  D e t a i l s  { }
 */
 
static int ConfParameterDetails(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc DUNUSED, char *argv[])

/*+
 *  Command name:
 *    ParameterDetails
 *
 *  Function:
 *    Return details of an allocation parameter
 *  
 *  Call:
 *    ParameterDetails number name type low high nvalues values
 *
 *  Description:
 *    Set the value of an allocation parameter.
 *   
 *   Parameters:
 *      (>) number (char) The parameter number
 *      (<) name (char)   variable to receive the parameter name
 *      (<) type (char)   variable to receive the parameter type
 *      (<) low  (char)   variable to receive the low limit on the value
 *      (<) high (char)   variable to receive the high limit on the value
 *      (<) nvalues (char) variable to receive the number of possible values
 *      (<) values (char) variable to recive the list of possible values
 *
 *   Support:
 *     Jeremy Bailey, AAO
 *
 *   Version Date:
 *     6-Oct-1994    
 *-
 */
{
    StatusType status = STATUS__OK;
    int nvalues;
    double limits[2];
    char *name, type[2], **values;
    char buffer[80];
    int i;

    ConfigParameterDetails(atoi(argv[1]), &name, &(type[0]), limits,
        &nvalues, &values, &status);
    type[1] = 0;

    if (status == STATUS__OK) {
        Tcl_SetVar(interp, argv[2], name, 0);
        Tcl_SetVar(interp, argv[3], type, 0);
        if (type[0] == 'I') {
            sprintf(buffer, "%d", (int) (limits[0]));
            Tcl_SetVar(interp, argv[4], buffer, 0);
            sprintf(buffer, "%d", (int) (limits[1]));
            Tcl_SetVar(interp, argv[5], buffer, 0);
        } else {
            sprintf(buffer, "%f", limits[0]);
            Tcl_SetVar(interp, argv[4], buffer, 0);
            sprintf(buffer, "%f", limits[1]);
            Tcl_SetVar(interp, argv[5], buffer, 0);
        }
        sprintf(buffer, "%d", nvalues);
        Tcl_SetVar(interp, argv[6], buffer, 0);
        Tcl_UnsetVar(interp, argv[7], 0);
        for (i = 0; i < nvalues; i++) {
            Tcl_SetVar(interp, argv[7], values[i],
                TCL_LIST_ELEMENT | TCL_APPEND_VALUE);
        }
        return TCL_OK;
    } else {
        sprintf(interp->result, "Error Getting Parameter %s Details", argv[1]);
        return TCL_ERROR;
    }
}


/*  ------------------------------------------------------------------------- */

/*                    C o n f  R e m o v e  A l l o c
 *
 *   This is a utility routine used by ConfCompressUnalloc(). It removes
 *   the items flagged as 'allocated' from the unallocated sub-structures
 *   of the configuration data structure.
 *
 *   Modified:
 *      20th Jan 2003. KS. No longer returns bad status if allocated 
 *                         array not present.
 */
 
static void ConfRemoveAlloc(SdsIdType id, StatusType * status)
 {
    SdsIdType allid;
    char c;
    unsigned int i;
    unsigned int j;
    unsigned int nalloc;
    SdsCodeType code;
    char name[16];
    long ndims;
    unsigned long dims[7];
    unsigned long actlen;

    /*  Failing to find an "allocated" array should not be regarded as a 
     *  problem. It just indicates that this structure has already been
     *  compressed by this routine.
     */
     
    SdsFind(id, "allocated", &allid, status);
    if (*status != STATUS__OK) {
        *status = STATUS__OK;
        return;
    }
    SdsInfo(allid, name, &code, &ndims, dims, status);
    nalloc = 0;
    for (i = 0; i < dims[0]; i++) {
        SdsGet(allid, sizeof(char), i, &c, &actlen, status);
        if (c == 0)
            nalloc++;
    }
    j = nalloc - 1;
    for (i = 0; i < nalloc; i++) {
        SdsGet(allid, sizeof(char), i, &c, &actlen, status);
        if (c == 1) {
            do {
                j++;
                SdsGet(allid, sizeof(char), j, &c, &actlen, status);
            }
            while (c == 1);
            ConfCopyAllItems(id, j, i, status);
        }
    }
    SdsDelete(allid, status);
    SdsFreeId(allid, status);
    ConfResizeAllItems(id, nalloc, status);

}


/*  ------------------------------------------------------------------------- */

/*                      C o n f  C o p y  A l l  I t e m s
 *
 *  Within the specified SDS structure, we have a series of arrays.
 *  We want to copy item "source" to item "destin" in each element
 *  of the array. This is a utility routine used by ConfRemoveAlloc().
 */
 
static void ConfCopyAllItems(SdsIdType id,
    int source, int destin, StatusType * status)
 {
    SdsIdType array_id;
    int index;
    char value[CMTLEN];
    SdsCodeType code;
    char name[16];
    long ndims;
    unsigned long dims[7];
    unsigned long actlen;
    unsigned long size = 0;

    index = 1;
    do {
        /*
         * Find the next item in the structure and get its details.
         */
        SdsIndex(id, index, &array_id, status);
        SdsInfo(array_id, name, &code, &ndims, dims, status);
        switch (code) {
        case SDS_CHAR:
        case SDS_BYTE:
        case SDS_UBYTE:
            size = sizeof(char);
            break;
        case SDS_USHORT:
        case SDS_SHORT:
            size = sizeof(short);
            break;
        case SDS_UINT:
        case SDS_INT:
            size = sizeof(INT32);
            break;
        case SDS_FLOAT:
            size = sizeof(float);
            break;
        case SDS_DOUBLE:
            size = sizeof(double);
            break;
        }
        if (ndims == 1) {
            /*
             *  There is only one dimension to this item.  We are just
             *  copying a scalar item.
             */
            SdsGet(array_id, size, source, value, &actlen, status);
            SdsPut(array_id, size, destin, value, status);
        } else {                /* ndims == 2 */

            /*
             *  There are two dimensions, dims[0] specifies the size of
             *  each dimension.  We assume CMTLEN is the largets value 
             *  for dims[0], but don't check this
             */
            SdsGet(array_id, size * dims[0], source * dims[0], value, &actlen,
                status);
            SdsPut(array_id, size * dims[0], destin * dims[0], value, status);
        }
        index++;
        /*
         * Finished with this id.
         */
        SdsFreeId(array_id, status);
    }
    while (*status == STATUS__OK);
    if (*status == SDS__NOITEM)
        ErsAnnul(status);
}



/*  ------------------------------------------------------------------------- */

/*                 C o n f  R e s i z e  A l l  I t e m s
 *
 *   This is a utility routine used by ConfRemoveAlloc().  When allocated
 *   items have been removed from a sub-structure of the configuration
 *   data structure, this is called to resize the SDS arrays to their
 *   new, smaller, size.
 */
 
static void ConfResizeAllItems(SdsIdType id, int size, StatusType * status)
 {
    SdsIdType id2;
    int index;
    SdsCodeType code;
    char name[16];
    long ndims;
    unsigned long dims[7];

    index = 1;
    do {
        SdsIndex(id, index, &id2, status);
        SdsInfo(id2, name, &code, &ndims, dims, status);
        if (ndims == 1) {
            dims[0] = size;
            SdsResize(id2, ndims, dims, status);
        } else {
            dims[1] = size;
            SdsResize(id2, ndims, dims, status);
        }
        index++;
        SdsFreeId(id2, status);
    }
    while (*status == STATUS__OK);
    if (*status == SDS__NOITEM)
        ErsAnnul(status);
}


/*  ------------------------------------------------------------------------- */

/*                    C o n f  A r r a y  G e t s
 *
 *   This is a utility routine that gets the num'th item from the array
 *   of shorts given by the name parameter within the SDS structure whose
 *   id is supplied.
 */
 
static void ConfArrayGets(SdsIdType id, char *name, long num,
    short *s, StatusType * status)
 {
    SdsIdType tid;
    unsigned long actlen;

    if (*status != STATUS__OK)
        return;
    SdsFind(id, name, &tid, status);
    SdsGet(tid, sizeof(short), num, s, &actlen, status);
    SdsFreeId(tid, status);
    if (*status != STATUS__OK) {
        char buff[80];
        sprintf(buff, "Error getting array item %s", name);
        ConfSdsCheck(buff, status);
    }
}

/*  ------------------------------------------------------------------------- */

/*                    C o n f  A r r a y  P u t s
 *
 *   This is a utility routine that sets the num'th item from the array
 *   of shorts given by the name parameter within the SDS structure whose
 *   id is supplied to the supplied value.
 */
 
static void ConfArrayPuts(SdsIdType id, char *name, long num,
    short s, StatusType * status)
 {
    SdsIdType tid;

    if (*status != STATUS__OK)
        return;
    SdsFind(id, name, &tid, status);
    SdsPut(tid, sizeof(short), num, &s, status);
    SdsFreeId(tid, status);
    if (*status != STATUS__OK) {
        char buff[80];
        sprintf(buff, "Error putting array item %s", name);
        ConfSdsCheck(buff, status);
    }
}

/*  ------------------------------------------------------------------------- */

/*                    C o n f  A r r a y  G e t c
 *
 *   This is a utility routine that gets the num'th item from the array
 *   of char given by the name parameter within the SDS structure whose
 *   id is supplied.
 */

static void ConfArrayGetc(SdsIdType id, char *name, long num,
    char *c, StatusType * status)
 {
    SdsIdType tid;
    unsigned long actlen;

    if (*status != STATUS__OK)
        return;
    SdsFind(id, name, &tid, status);
    if (*status != STATUS__OK) {
         char buff[80];
         sprintf(buff, "Error in SdsFind getting array item %s", name);
         ConfSdsCheck(buff, status);
    }
    SdsGet(tid, sizeof(char), num, c, &actlen, status);
    if (*status != STATUS__OK) {
             char buff[80];
             sprintf(buff, "Error in SdsGet getting array item %s", name);
             ConfSdsCheck(buff, status);
    }
    SdsFreeId(tid, status);
    if (*status != STATUS__OK) {
        char buff[80];
        sprintf(buff, "Error in  SdsFreeId getting array item %s", name);
        ConfSdsCheck(buff, status);
    }
}

/*  ------------------------------------------------------------------------- */

/*                    C o n f  A r r a y  P u t c
 *
 *   This is a utility routine that sets the num'th item from the array
 *   of char given by the name parameter within the SDS structure whose
 *   id is supplied to the supplied value.
 */
 
static void ConfArrayPutc(SdsIdType id, char *name, long num,
    char c, StatusType * status)
 {
    SdsIdType tid;

    if (*status != STATUS__OK)
        return;
    SdsFind(id, name, &tid, status);
    SdsPut(tid, sizeof(char), num, &c, status);
    SdsFreeId(tid, status);
    if (*status != STATUS__OK) {
        char buff[80];
        sprintf(buff, "Error putting array item %s", name);
        ConfSdsCheck(buff, status);
    }
}

/*  ------------------------------------------------------------------------- */

/*                    C o n f  A r r a y  G e t i
 *
 *   This is a utility routine that gets the num'th item from the array
 *   of ints given by the name parameter within the SDS structure whose
 *   id is supplied.
 */
 
static void ConfArrayGeti(SdsIdType id, char *name, long num,
    int *i, StatusType * status)
 {
    SdsIdType tid;
    unsigned long actlen;

    if (*status != STATUS__OK)
        return;
    SdsFind(id, name, &tid, status);
    SdsGet(tid, sizeof(int), num, i, &actlen, status);
    SdsFreeId(tid, status);
    if (*status != STATUS__OK) {
        char buff[80];
        sprintf(buff, "Error getting array item %s", name);
        ConfSdsCheck(buff, status);
    }
}

/*  ------------------------------------------------------------------------- */

/*                    C o n f  A r r a y  P u t i
 *
 *   This is a utility routine that sets the num'th item from the array
 *   of ints given by the name parameter within the SDS structure whose
 *   id is supplied to the supplied value.
 */
 
static void ConfArrayPuti(SdsIdType id, char *name, long num,
    int i, StatusType * status)
 {
    SdsIdType tid;

    if (*status != STATUS__OK)
        return;
    SdsFind(id, name, &tid, status);
    SdsPut(tid, sizeof(int), num, &i, status);
    SdsFreeId(tid, status);
    if (*status != STATUS__OK) {
        char buff[80];
        sprintf(buff, "Error putting array item %s", name);
        ConfSdsCheck(buff, status);
    }
}

/*  ------------------------------------------------------------------------- */

/*                    C o n f  A r r a y  P u t d
 *
 *   This is a utility routine that sets the num'th item from the array
 *   of doubles given by the name parameter within the SDS structure whose
 *   id is supplied to the supplied value.
 */
 
static void ConfArrayPutd(SdsIdType id, char *name, long num,
    double d, StatusType * status)
 {
    SdsIdType tid;

    if (*status != STATUS__OK)
        return;
    SdsFind(id, name, &tid, status);
    SdsPut(tid, sizeof(double), num, &d, status);
    SdsFreeId(tid, status);
    if (*status != STATUS__OK) {
        char buff[80];
        sprintf(buff, "Error putting array item %s", name);
        ConfSdsCheck(buff, status);
    }
}

/*  ------------------------------------------------------------------------- */

/*                C o n f  A r r a y  P u t  S t r i n g
 *
 *   This is a utility routine that sets the num'th item from the array
 *   of strings given by the name parameter within the SDS structure whose
 *   id is supplied to the supplied string. Note the assumption that the
 *   strings in question are all 80 characters long.
 */
 
static void ConfArrayPutString(SdsIdType id, char *name, long num,
    char *string, StatusType * status)
 {
    SdsIdType tid;

    if (*status != STATUS__OK)
        return;
    SdsFind(id, name, &tid, status);
    SdsPut(tid, 80, 80 * num, string, status);
    SdsFreeId(tid, status);
    if (*status != STATUS__OK) {
        char buff[80];
        sprintf(buff, "Error putting array item %s", name);
        ConfSdsCheck(buff, status);
    }
}

/*  ------------------------------------------------------------------------- */

/*                    C o n f  A r r a y  C o p y
 *
 *  Function to copy data from one SDS array element to another.  The
 *  two elements must have equivalent types and names within their parent
 *  SDS structure.
 */

static void ConfArrayCopy(
    SdsIdType id1,       /* SDS id of structure containing array to be copied */
    SdsIdType id2,       /* SDS id of structure into which array is copied */
    char *name,          /* Name of item in the above structures to be copied */
    long size,           /* Amount of data to copy (bytes) */
    long offset1,        /* Offset (array element units) in from array */
    long offset2,        /* Offset (array element units) in to array */
    StatusType* status)
 {
    SdsIdType tid1, tid2;
    unsigned long actlen;
    char buffer[CMTLEN];
    char *bufpnt;

    if (*status != STATUS__OK)
        return;

/*
 * We have an internal buffer which is normally big enough, but
 * if it is not, malloc space for the buffer.
 */
    if (size > (long) sizeof(buffer))
        bufpnt = malloc(size);
    else
        bufpnt = buffer;

/*
 * Find source and destination items.
 */
    SdsFind(id1, name, &tid1, status);
    SdsFind(id2, name, &tid2, status);
/*
 * Copy the data.
 */
    SdsGet(tid1, size, offset1, bufpnt, &actlen, status);
    SdsPut(tid2, size, offset2, bufpnt, status);

/*
 * Tidy up.
 */
    SdsFreeId(tid1, status);
    SdsFreeId(tid2, status);

    if (size > (long) sizeof(buffer))
        free(bufpnt);

    if (*status != STATUS__OK) {
        char buff[80];
        sprintf(buff, "Error copying item %s", name);
        ConfSdsCheck(buff, status);
    }
}

/*  ------------------------------------------------------------------------- */

/*
 *                          G e t  T y p e  { }
 */
 
static int ConfGetType(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])

/*+
 *  Command name:
 *    GetType
 *
 *  Function:
 *    Determines the colour to use to display a button based on its fibre type
 *  
 *  Description:
 *    This routine returns (to tcl, through the interpreter's result string)
 *    the colour to use when drawing a button. It does this on the basis of
 *    the fibre type associated with the pivot. If the fibre is disabled, it
 *    returns "Blank" and leaves the choice of how to display the fibre to
 *    the Tcl/Tk layer. The name is perhaps misleading - it gets a colour
 *    based on the type, not the type itself.
 *
 *   Call:
 *      GetType pivid fibre_no fibre_combo
 *
 *   Parameters:
 *
 *    (>) pivid  (Sds Id)   SDS identifier of pivot details structure. (This
 *                          is ignored now, but was needed in earlier versions
 *                          of the routine. Fpil now handles this structure.)
 *    (>) fibre_no (int)    Number of pivot in question - from 0 up.
 *    (>) fibre_combo (int) Index number of the selected fibre combination,
 *                          if such a selection has been made. -1 indicates
 *                          that no known configuration has been selected.
 *                          Values from 0 up are index values that match the
 *                          order returned by ConfFibreCombos().
 *
 *   Returns:
 *     "OK" if succesful, "Error" otherwise.
 *-
 */
{
    long fibre_no;
    int fibreType;
    int fibreCombo;
    char colourString[32];

    /*  Validate the arguments */

    if (argc != 4) {
        interp->result = "Wrong # args";
        return TCL_ERROR;
    }
    fibre_no = atol(argv[2]) - 1;
    if (fibre_no == -1)
        return TCL_OK;
    fibreCombo = atol(argv[3]);
    if ((fibre_no < 0) || (fibre_no >= (int) FpilGetNumPivots(*Instrument))) {
        sprintf(interp->result,
            "Illegal Pivot Number %ld\nMust be between 0 and %d",
            fibre_no + 1, FpilGetNumPivots(*Instrument));
        return TCL_ERROR;
    }

    /*  The colour choice is made by FpilFibreColour, and is based on the
     *  fibre type returned by ConfWhichSpec().  If the type is one that we're 
     *  not currently allocating (something Fpil doesn't know about) we 
     *  explicitly set it to Blank first.
     */

    fibreType = ConfWhichSpec((int) fibre_no, 0);

    if (!UseFibreType[fibreType]) {
        sprintf(interp->result, "Blank");
    } else {
        FpilFibreColour(*Instrument, fibreType, fibreCombo, useGrey,
                                       colourString, sizeof(colourString));
        sprintf(interp->result, colourString);
    }
    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*
 *                   R e a l l y  E x i s t s  { }
 */
 
static int ConfReallyExists(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])

/*+
 *  Command name:
 *    ReallyExists
 *
 *  Function:
 *    Determines if a button really exists based on its fibre type
 *  
 *  Description:
 *    This routine returns (to tcl, through the interpreter's result string)
 *    a flag (0 for false, 1 for true) that indicates whether the button
 *    in question really exists. (Some instruments, such as FLAMES, have
 *    fibres included in the instrument descriptions that are just 
 *    placeholders for fibres that may be installed some time in the future.
 *
 *   Parameters:
 *
 *    (>) pivid  (Sds Id)   SDS identifier of pivot details structure. (This
 *                          is ignored now, but was needed in earlier versions
 *                          of the routine. Fpil now handles this structure.)
 *    (>) fibre_no (int)    Number of pivot in question - from 0 up.
 *
 *   Returns:
 *     "OK" if succesful, "Error" otherwise.
 *-
 */
{
    long fibre_no;
    int fibreType;

    /*  Validate the arguments */

    if (argc != 3) {
        interp->result = "Wrong # args";
        return TCL_ERROR;
    }
    fibre_no = atol(argv[2]) - 1;
    if (fibre_no == -1)
        return TCL_OK;
    if ((fibre_no < 0) || (fibre_no >= (int) FpilGetNumPivots(*Instrument))) {
        sprintf(interp->result,
            "Illegal Pivot Number %ld\nMust be between 0 and %d",
            fibre_no + 1, FpilGetNumPivots(*Instrument));
        return TCL_ERROR;
    }

    /*  The check for existence is made by FpilFibreReallyExists, and is
     *  based entirely on the fibre type.
     */

    fibreType = ConfWhichSpec((int) fibre_no, 0);

    if (FpilFibreReallyExists(*Instrument, fibreType)) {
        strcpy (interp->result,"1");
    } else {
        strcpy (interp->result,"0");
    }
    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*
 *                C o n f  D o  A d d  S k y  P o s i t i o n
 *
 *  This is a utility routine that is used to implement AddSkyPosition{},
 *  but is also called directly from other code in this file. It adds
 *  a sky target as specified in the calling parameters.
 */
  
static void ConfDoAddSkyPosition(SdsIdType topid, double xsky, double ysky,
    char title[], long *object, StatusType * status)

{
    SdsIdType uid, fid;
    short s;
    char c;
    char name[NAMELEN];
    SdsCodeType code;
    long ndims, obj_no;
    short l;
    int x, y;
    double xx, yy;
    double appra, appdec, mjd;
    double rasky, decsky, ara, adec;
    char targetType;
    char spectType;
    SdsIdType ra_id;
    double gmap[21];         /* Mean to apparent parameters  */
    unsigned long dims[7];

    double lx, ly, jx, jy, jxx, jyy;

    SdsFind(topid, "fieldData", &fid, status);

    ArgGetd(fid, "appRa", &appra, status);
    ArgGetd(fid, "appDec", &appdec, status);
    ArgGetd(fid, "configMjd", &mjd, status);

    FpilModelPos2Xy(TelModel, xsky, ysky, &xx, &yy, status);
    FpilModelXy2Rd(TelModel, appra, appdec, xx, yy, mjd, &ara, &adec, status);

    FpilModelXy2Pos(TelModel, xx, yy, &lx, &ly, status);
    FpilModelRd2Xy(TelModel, appra, appdec, ara, adec, mjd, &jxx, &jyy, status);
    FpilModelXy2Pos(TelModel, jxx, jyy, &jx, &jy, status);

    slaMappa(2000.0, mjd, gmap);

    slaAmpqk(ara, adec, gmap, &rasky, &decsky);

    SdsFind(topid, "unallocObject", &uid, status);

/*  Now we have to enlarge the unallocated objects section
    and put the new data into it  */

    FpilTargetSkyType(*Instrument, &targetType, &spectType);

    SdsFind(uid, "ra", &ra_id, status);
    SdsInfo(ra_id, name, &code, &ndims, dims, status);
    SdsFreeId(ra_id, status);
    ConfResizeAllItems(uid, dims[0] + 1, status);
    obj_no = dims[0];
    *object = obj_no;
    SdsFind(uid, "name", &ra_id, status);
    SdsPut(ra_id, NAMELEN, NAMELEN * obj_no, title, status);
    SdsFreeId(ra_id, status);
    SdsFind(uid, "ra", &ra_id, status);
    SdsPut(ra_id, sizeof(double), obj_no, &rasky, status);
    SdsFreeId(ra_id, status);
    SdsFind(uid, "dec", &ra_id, status);
    SdsPut(ra_id, sizeof(double), obj_no, &decsky, status);
    SdsFreeId(ra_id, status);
    SdsFind(uid, "x", &ra_id, status);
    x = (int) xsky;
    SdsPut(ra_id, sizeof(int), obj_no, &x, status);
    SdsFreeId(ra_id, status);
    SdsFind(uid, "y", &ra_id, status);
    y = (int) ysky;
    SdsPut(ra_id, sizeof(int), obj_no, &y, status);
    SdsFreeId(ra_id, status);
    SdsFind(uid, "type", &ra_id, status);
    c = targetType;
    SdsPut(ra_id, sizeof(char), obj_no, &c, status);
    SdsFreeId(ra_id, status);
    SdsFind(uid, "priority", &ra_id, status);
    l = 9;
    SdsPut(ra_id, sizeof(short), obj_no, &l, status);
    SdsFreeId(ra_id, status);
    SdsFind(uid, "magnitude", &ra_id, status);
    xx = 23.0;
    SdsPut(ra_id, sizeof(double), obj_no, &xx, status);
    SdsFreeId(ra_id, status);
    SdsFind(uid, "pId", &ra_id, status);
    l = 0;
    SdsPut(ra_id, sizeof(short), obj_no, &l, status);
    SdsFreeId(ra_id, status);
    SdsFind(uid, "comment", &ra_id, status);
    SdsPut(ra_id, 80, 80 * obj_no, "Additional Sky Position", status);
    SdsFreeId(ra_id, status);
    SdsFind(uid, "spectrograph", &ra_id, status);
    c = spectType;
    SdsPut(ra_id, sizeof(char), obj_no, &c, status);
    SdsFreeId(ra_id, status);
    ConfArrayPutc(uid, "allocated", obj_no, 0, status);
    if (*status != STATUS__OK) {
        ConfSdsCheck("Error adding to unalloc objects structure", status);
    }

    /* Now update the numbers for the GUI */
    if (*status == STATUS__OK) {
        ArgGets(fid, "unallocSky", &s, status);
        s++;
        ArgPuts(fid, "unallocSky", s, status);
    }


    {
        int x, y;
        ConfArrayGeti(uid, "x", obj_no, &x, status);
        ConfArrayGeti(uid, "y", obj_no, &y, status);

        /*  printf("DoAddSky:new object %d position is at %d,%d\n",
           obj_no,x,y); */

    }


    SdsFreeId(uid, status);
    SdsFreeId(fid, status);

}

/*  ------------------------------------------------------------------------- */

/*
 *                A d d  S k y  P o s i t i o n  { }
 */
 
static int ConfAddSkyPosition(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])
    
/*+
 *  Command name:
 *    AddSkyPosition
 *
 *  Function:
 *    Add a new (user/method supplied) sky position to the unallocated
 *    objects sub-struct
 *  
 *  Description:
 *    Generate a new ra,dec,pair from a user-supplied x,y position
 *    Allocate the specified fibre to this new target. It returns the
 *    index of the new entry in the unallocated objects structure, which
 *    will have been enlarged in the process.
 *
 *  Call:
 *    AddSkyPosition id xsky ysky object zoom xoff yoff
 *
 *  Parameters:
 *
 *    (>) id   (Sds Id)  SDS identifier of configuration structure.
 *    (>) xsky (double)  input x-sky;
 *    (>) ysky (double)  input y-sky
 *    (<) object (int)   Object number - starting from one.
 *    (>) zoom (double)  The current zoom factor.
 *    (>) xoff (double)  Left window scroll position, range 0 to 1.
 *    (>) yoff (double)  Top window scroll position, range 0 to 1.
 *
 *   Returns:
 *     "OK" if succesful, "Error" otherwise
 *
 *   Support:
 *     Gavin Dalton, Oxford
 *
 *   Version Date:
 *     6-Oct-1994    
 *-
 */
 
{
    double xsky, ysky;
    double xoff, yoff;
    long object;
    SdsIdType topid;
    double zoom;
    char mbuf[100], label[NAMELEN];
    StatusType status = STATUS__OK;
    short allocSkyCnt, unallocSkyCnt;
    SdsIdType field;

    if (argc != 8) {
        interp->result = "Wrong # args";
        return TCL_ERROR;
    }
    topid = (SdsIdType) (atol(argv[1]));
    xsky = atof(argv[2]);
    ysky = atof(argv[3]);
    zoom = atof(argv[5]);
    xoff = atof(argv[6]);
    yoff = atof(argv[7]);

/*
 * Convert scroll position to pixel offsets. 
 */
    xoff *= MimicWindowSize;
    yoff *= MimicWindowSize;

/*
 *  Need the reverse of FieldPlateToDispX/Y ??
 */
    xsky = DispToFieldPlateX(xsky, xoff, zoom);
    ysky = DispToFieldPlateY(ysky, yoff, zoom);


/*
 * Need a unique name for the sky.
 */
    SdsFind(topid, "fieldData", &field, &status);
    ArgGets(field, "unallocSky", &unallocSkyCnt, &status);
    if (status == STATUS__OK) {
        allocSkyCnt = 0;
        ArgGets(field, "allocSky", &allocSkyCnt, &status);
        if (status == SDS__NOITEM)
            ErsAnnul(&status);
    }
    SdsFreeId(field, &status);

    sprintf(label, "User_Sky_%d", unallocSkyCnt + allocSkyCnt + 1);

/*
printf("Adding user sky %s, unallocSkyCnt = %d, allocSkyCnt = %d\n",
       label, unallocSkyCnt, allocSkyCnt);
*/

    ConfDoAddSkyPosition(topid, xsky, ysky, label, &object, &status);
    sprintf(mbuf, "%ld", object + 1);


    Tcl_SetVar(interp, argv[4], mbuf, 0);

    if (status != STATUS__OK) {
        char buff[200];
        MessGetMsg(status, 0, sizeof(buff), buff);
        Tcl_AppendResult(interp, "\nError Setting Sky Position\n",
            (char *) NULL);
        Tcl_AppendResult(interp, buff, (char *) NULL);
        return TCL_ERROR;


    }
    return TCL_OK;

}

/*  ------------------------------------------------------------------------- */

/*
 *                S e t  I m p o r t  F i l e  { }
 */
 
static int ConfSetImportFile(ClientData clientData DUNUSED,
    Tcl_Interp * interp DUNUSED, int argc DUNUSED, char *argv[]DUNUSED)
    
/*+
 *  Command name:
 *    SetImportFile
 *
 *  Function:
 *    Sets the file to be imported by the allocation algorithm.
 *  
 *  Description:
 *    This routine provides an interface from Tcl to the allocation
 *    algorithm routine ConfigSetImportFile(). It specifies the name
 *    of the file to be used to import a set of allocations.
 *
 *  Call:
 *    SetImportFile filename
 *
 *  Parameters:
 *
 *    (>) filename  (string)  Name of import file.
 *
 *   Support:
 *     Gavin Dalton, Oxford
 *-
 */
{
    ConfigSetImportFile(argv[1]);
    return (0);
}

/*  ------------------------------------------------------------------------- */

/*
 *                    S e t  S k y  G r i d  { }
 */
 
static int ConfSetSkyGrid(ClientData clientData DUNUSED, Tcl_Interp * interp,
    int argc DUNUSED, char *argv[])
    
/*+
 *  Command name:
 *    SetSkyGrid
 *
 *  Function:
 *    Sets up a grid of sky positions.
 *  
 *  Description:
 *    This routine creates a set of sky targets in the unallocated objects
 *    section of the configuration data, based on a rectangular grid.
 *
 *  Call:
 *    SetSkyGrid id
 *
 *  Parameters:
 *
 *    (>) id  (Sds Id)  Id of top level of the configuration data structure.
 *
 *   Support:
 *     Gavin Dalton, Oxford
 *-
 */
 {
    double xx, yy, th, thint, tp1, s, c;
    double xrad1, xrad2, xrad3;
    double fieldrad;
    SdsIdType topid, oid, rid;
    SdsCodeType code;
    long ndims, objno;
    int Npiv, Ipiv;
    unsigned long dims[7];
    StatusType status = STATUS__OK;
    char mbuf[100];
    char name[NAMELEN], label[NAMELEN];
    short allocSkyCnt, unallocSkyCnt;
    SdsIdType field;
    int sky;

    fieldrad = FpilGetFieldRadius(*Instrument);

    topid = (SdsIdType) (atol(argv[1]));
    SdsFind(topid, "objects", &oid, &status);
    SdsFind(oid, "ra", &rid, &status);
    SdsInfo(rid, name, &code, &ndims, dims, &status);
    Npiv = dims[0];
    SdsFreeId(rid, &status);
    SdsFreeId(oid, &status);
/*
 *  Determine the angle between each pivot. 
 */
    thint = 2 * PI / Npiv;
/*
 *  Get half that value.
 */
    th = thint / 2;
/*
 *  Determine the three radii we will work at.  We use the following
 *  proportions of the field size, which are chosen to get roughly
 *  the same dimensions as the original code which assumed a fixed
 *  field radius (for 2dF). 
 */
#   define SKYGRID_RADIUS_1 0.87
#   define SKYGRID_RADIUS_2 0.64
#   define SKYGRID_RADIUS_3 0.37

    xrad1 = fieldrad * SKYGRID_RADIUS_1;
    xrad2 = fieldrad * SKYGRID_RADIUS_2;
    xrad3 = fieldrad * SKYGRID_RADIUS_3;

    sprintf(mbuf, "SkyGrid: %d", Npiv);
    ConfMessage(interp, mbuf);


/*
 * Need a unique name for the sky.
 */
    SdsFind(topid, "fieldData", &field, &status);
    ArgGets(field, "unallocSky", &unallocSkyCnt, &status);
    if (status == STATUS__OK) {
        allocSkyCnt = 0;
        ArgGets(field, "allocSky", &allocSkyCnt, &status);
        if (status == SDS__NOITEM)
            ErsAnnul(&status);
    }
    SdsFreeId(field, &status);

/*
 *  Number for first sky's name.
 */
    sky = unallocSkyCnt + allocSkyCnt + 1;


    s = c = 0;
/*
 *  Create the grid.
 */
    for (Ipiv = 0; Ipiv < Npiv; Ipiv++) {


        if ((Ipiv % 5) == 0) {
            tp1 = th + Ipiv * thint;
            s = sin(tp1);
            c = cos(tp1);
            xx = xrad1 * c;
            yy = xrad1 * s;
            sprintf(label, "Grid_Sky_%d", sky);
            ConfDoAddSkyPosition(topid, xx, yy, label, &objno, &status);
            ++sky;
        }
        if ((Ipiv % 8) == 0) {
            xx = xrad2 * c;
            yy = xrad2 * s;
            sprintf(label, "Grid_Sky_%d", sky);
            ConfDoAddSkyPosition(topid, xx, yy, label, &objno, &status);
            ++sky;
        }
        if ((Ipiv % 13) == 0) {
            xx = xrad3 * c;
            yy = xrad3 * s;
            sprintf(label, "Grid_Sky_%d", sky);
            ConfDoAddSkyPosition(topid, xx, yy, label, &objno, &status);
            ++sky;
        }
    }
    if (status == STATUS__OK) {
        interp->result = "OK";
        return TCL_OK;
    } else {
        char buff[200];
        MessGetMsg(status, 0, sizeof(buff), buff);
        Tcl_AppendResult(interp, "\nError Setting Sky Grid\n", (char *) NULL);
        Tcl_AppendResult(interp, buff, (char *) NULL);
        return TCL_ERROR;

    }
}


/*  ------------------------------------------------------------------------- */

/*
 *                    F i x  B r e a k a g e s  { }
 */
 
static int ConfFixBreakages(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])
/*+
 *  Command name:
 *    FixBreakages
 *
 *  Function:
 *    Deallocates any broken fibres.
 *  
 *  Description:
 *    This routine deallocates any allocations that have been made to fibres
 *    flagged as broken.
 *
 *  Call:
 *    FixBreakages PivotId Cfid
 *
 *  Parameters:
 *
 *    (>) PivotId  (Sds Id)  Id of pivot data structure.
 *    (>) Cfid     (Sds Id)  Id of top level of configuration data structure.
 *
 *   Support:
 *     Gavin Dalton, Oxford
 *-
 */
{
    SdsIdType pivid, topid, objid = 0;
    unsigned fibre_no;
    StatusType status = STATUS__OK;
    short s;
    char c;
    int doneOne = 0;
    unsigned NumPivots = FpilGetNumPivots(*Instrument);


    if (argc != 3) {
        interp->result = "Wrong # args";
        return TCL_ERROR;
    }
/*
 *  Fetch arguments, pivot structure and configuration.
 */
    pivid = (SdsIdType) (atol(argv[1]));
    topid = (SdsIdType) (atol(argv[2]));
/* 
 *  Find the allocated objects.
 */
    SdsFind(topid, "objects", &objid, &status);
    ConfSdsCheck("Error finding objects structure", &status);
/*
 *  For each pivot, if we are allowed to change pivots associated with
 *  the spectrograph, get the inUse flag.
 */
    for (fibre_no = 0; fibre_no < NumPivots; fibre_no++) {
        int spec;
        spec = ConfWhichSpec((int) fibre_no, 0);
        if (UseFibreType[spec]) {
            ConfArrayGets(pivid, "inUse", fibre_no, &s, &status);
/*
 *          If the pivot is not in use, then we grab the type of the object
 *          associated with this fibre.
 */
            if (s != 1) {
                ConfArrayGetc(objid, "type", fibre_no, &c, &status);
                if (c != FpilUnallocatedType(*Instrument)) {
                    char command[80];
                    char message[80];
                    doneOne = 1;
/*
 *                  If an object is allocated, deallocate it using the 
 *                  DummyDeallocateFib Tcl Command.
 */
                    sprintf(message, "Deassigning Allocated Broken Fibre %d",
                        fibre_no + 1);
                    ConfMessage(interp, message);
                    sprintf(command, "DummyDeallocateFib %d", fibre_no + 1);
                    if (Tcl_Eval(interp, command) == TCL_ERROR) {
                        Tcl_AddErrorInfo(interp,
                            "\n(Error deallocating broken fibre)");
                        SdsFreeId(objid, &status);
                        return TCL_ERROR;
                    }
                }
            }
        }
    }
    SdsFreeId(objid, &status);
    if (!doneOne) {
        ConfMessage(interp, "No Allocated Broken Fibres were found");

    }
    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*
 *                       R e m o v e  S k y  { }
 */
 
static int ConfRemoveSky(ClientData clientData DUNUSED, Tcl_Interp * interp,
    int argc, char *argv[])
    
/*+
 *  Command name:
 *    RemoveSky
 *
 *  Function:
 *    Deallocates any fibres allocated to sky targets.
 *  
 *  Description:
 *    This routine deallocates any allocations that have been made to sky
 *    targets.
 *
 *  Call:
 *    RemoveSky PivotId Cfid
 *
 *  Parameters:
 *
 *    (>) PivotId  (Sds Id)  Id of pivot data structure.
 *    (>) Cfid     (Sds Id)  Id of top level of configuration data structure.
 *
 *   Support:
 *     Gavin Dalton, Oxford
 *-
 */
{
    SdsIdType pivid, topid, objid = 0;
    unsigned fibre_no;
    StatusType status = STATUS__OK;
    short s;
    char c;
    int doneOne = 0;
    unsigned NumPivots = FpilGetNumPivots(*Instrument);
    int spec;

    if (argc != 3) {
        interp->result = "Wrong # args";
        return TCL_ERROR;
    }
/*
 *  Fetch arguments, pivot structure and configuration.
 */
    pivid = (SdsIdType) (atol(argv[1]));
    topid = (SdsIdType) (atol(argv[2]));
/* 
 *  Find the allocated objects.
 */
    SdsFind(topid, "objects", &objid, &status);
    ConfSdsCheck("Error finding objects structure", &status);
/*
 *  For each pivot on a spectrograph we are allowed to change, 
 *  get the inUse flag.
 */
    for (fibre_no = 0; fibre_no < NumPivots; fibre_no++) {
        spec = ConfWhichSpec((int) fibre_no, 0);
        if (UseFibreType[spec]) {
            ConfArrayGets(pivid, "inUse", fibre_no, &s, &status);
            if (s == 1) {
/*
 *          Get any object allocation to this pivot. If the object
 *          is a Sky object, deallocate it by calling the
 *          Tcl command "DummyDeallocateFib".
 */
                ConfArrayGetc(objid, "type", fibre_no, &c, &status);
                if (FpilIsTargetSky(*Instrument, c)) {
                    char command[80];
                    char message[80];
                    doneOne = 1;
                    sprintf(message,
                        "Deassigning Allocated Sky Fibre %d", fibre_no + 1);
                    ConfMessage(interp, message);
                    sprintf(command, "DummyDeallocateFib %d", fibre_no + 1);

                    if (Tcl_Eval(interp, command) == TCL_ERROR) {
                        Tcl_AddErrorInfo(interp,
                            "\n(Error deallocating sky fibre)");
                        SdsFreeId(objid, &status);
                        return TCL_ERROR;
                    }
                }
            }
        }
    }
    SdsFreeId(objid, &status);

    if (!doneOne) {
        ConfMessage(interp, "No Allocated Sky Fibres were found");
    }

    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*
 *                    A l l o c T o U n a l l o c  { }
 */
 
static int ConfAllocToUnalloc(ClientData clientdata DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])
    
/*+                      
 *  Command name:
 *    AllocToUnalloc
 *
 *  Function:
 *    Put the allocated objects into the unallocated part of a structure.
 *
 *  Description:
 *      This command is used to prepare a currently configured field for
 *      reconfiguration on the other plate. 
 *
 *      The intention is that whilst we may need to reconfigure, we should
 *      only be configuring the objects that were configured on the other
 *      field.
 *
 *      We assume this is a structure in the style normalled used by
 *      this program (index item in "objects", allocated in the other
 *      parts etc).  
 *
 *      We delete all existing unallocated program and guide objects.
 *
 *      We copy each allocated object to the appropiate part of the structure
 *       
 *      We delete the index and allocated item.s
 *
 *      In the "field" part of the structure, we 
 *
 *          set allocSky1 = 0
 *              allocSky2 = 0
 *               allocSky  = 0
 *               allocOby  = 0
 *               allocGui  = 0
 *
 *          set unallocSky to previous value of allocSky
 *          set unallocObj to previous value of allocObj
 *          set unallocGui to previous value of allocGui
 *
 *      The resulting structure is appropiate for reading into the program, but
 *      note that all the unallocGuide and unallocObject details in the 
 *      originial structure are lost.
 *
 *  Call:
 *     AllocToUnalloc id
 *
 *  Parameters:
 *     (>) id (int) SDS id of the structure to modify.
 *
 *  Language:
 *     Tcl
 *
 *  Support: Tony Farrell, AAO
 *-
 *  History:
 *       13-Apr-2000 - TJF - Original version (DtclTrnName)
 *  
 */

{
    SdsIdType topId;
    SdsIdType fieldId = 0;
    SdsIdType objectsId = 0;
    SdsIdType unallocObjId = 0;
    SdsIdType unallocGuiId = 0;
    SdsIdType tid;
    StatusType status = STATUS__OK;
    StatusType ignore = STATUS__OK;
    int result = TCL_OK;

    char name[SDS_C_NAMELEN];
    long ndims;
    unsigned long dims[7];
    SdsCodeType code;

    long allocObj = 0;
    long allocSky = 0;
    long allocGui = 0;

    char *typePnt;
    unsigned long length;

    register unsigned i;
    long nextObj;
    long nextGui;

    if (argc != 2) {
        sprintf(interp->result, "%s:Wrong # args", argv[0]);
        return TCL_ERROR;
    }

    topId = atoi(argv[1]);

/*
 *  Find the various components we want.  Note, ArgFind is like SdsFind except
 *  that it uses ErsRep if it gets an error to report the name of the
 *  object concerned.
 */
    ArgFind(topId, "fieldData", &fieldId, &status);
    ArgFind(topId, "objects", &objectsId, &status);
    ArgFind(topId, "unallocObject", &unallocObjId, &status);
    ArgFind(topId, "unallocGuide", &unallocGuiId, &status);

    if (status != STATUS__OK)
        goto Exit;

/*
 *  Work out how many of each type of allocated object we have.
 */
    ArgGeti(fieldId, "allocObj", &allocObj, &status);
    if (status == SDS__NOITEM)
        ErsAnnul(&status);
    ArgGeti(fieldId, "allocGui", &allocGui, &status);
    if (status == SDS__NOITEM)
        ErsAnnul(&status);
    ArgGeti(fieldId, "allocSky", &allocSky, &status);
    if (status == SDS__NOITEM)
        ErsAnnul(&status);

    if (status != STATUS__OK)
        goto Exit;

/*
 *  If we don't have any allocated objects, this makes no sense.
 */
    if ((allocObj + allocGui + allocSky) == 0) {
        sprintf(interp->result,
            "%s:This makes no sense, there are no allocated objects", argv[0]);
        result = TCL_ERROR;
        goto Exit;
    }

/*
 *  Set the field values appropriately for the transfer of the allocated
 *  details to the unallocated details.
 */
    ArgPuts(fieldId, "unallocObj", allocObj, &status);
    ArgPuts(fieldId, "unallocGui", allocGui, &status);
    ArgPuts(fieldId, "unallocSky", allocSky, &status);

    ArgPuts(fieldId, "allocObj", 0, &status);
    ArgPuts(fieldId, "allocGui", 0, &status);
    ArgPuts(fieldId, "allocSky", 0, &status);
    for (i = 0; i < InstNumFibreTypes; ++i) {
        char buffer[20];
        sprintf(buffer, "allocSky%d", i + 1);
        ArgPuts(fieldId, buffer, 0, &status);
    }

/*
 *  Resize our structures.
 */
    ConfResizeAllItems(unallocObjId, allocObj + allocSky, &status);
    ConfResizeAllItems(unallocGuiId, allocGui, &status);

    SdsFind(objectsId, "x", &tid, &status);
    SdsInfo(tid, name, &code, &ndims, dims, &status);
    SdsFreeId(tid, &status);


    tid = 0;
    SdsFind(objectsId, "type", &tid, &status);
    SdsPointer(tid, (void **) &typePnt, &length, &status);


    nextGui = 0;
    nextObj = 0;

    for (i = 0; (status == STATUS__OK) && (i < dims[0]); ++i) {
        SdsIdType uid;
        unsigned int obj_no;
        if (typePnt[i] == FpilUnallocatedType(*Instrument)) {
/*  
 *          Must be an unallocated pivot, goto the next object.
 */
            continue;
        } else if (FpilIsTargetGuide(*Instrument, typePnt[i])) {
/*
 *          Guide objects in the unallocGuide array.
 */
            uid = unallocGuiId;
            obj_no = nextGui++;
            ConfArrayCopy(objectsId, uid, "dhdx", sizeof(double), i, obj_no,
                &status);
            ConfArrayCopy(objectsId, uid, "dhdy", sizeof(double), i, obj_no,
                &status);
            ConfArrayCopy(objectsId, uid, "dddx", sizeof(double), i, obj_no,
                &status);
            ConfArrayCopy(objectsId, uid, "dddy", sizeof(double), i, obj_no,
                &status);
        } else {
/*
 *          Program objects or sky objects go in the unallocObject array.
 */
            uid = unallocObjId;
            obj_no = nextObj++;
            ConfArrayCopy(objectsId, uid, "spectrograph", sizeof(char),
                i, obj_no, &status);

        }

/*
 *      Details common to both arrays.
 */
        ConfArrayCopy(objectsId, uid, "name", NAMELEN, NAMELEN * i,
            NAMELEN * obj_no, &status);
        ConfArrayCopy(objectsId, uid, "ra", sizeof(double), i, obj_no, &status);
        ConfArrayCopy(objectsId, uid, "dec", sizeof(double), i, obj_no,
            &status);
        ConfArrayCopy(objectsId, uid, "x", sizeof(int), i, obj_no, &status);
        ConfArrayCopy(objectsId, uid, "y", sizeof(int), i, obj_no, &status);
        ConfArrayCopy(objectsId, uid, "type", sizeof(char), i, obj_no, &status);
        ConfArrayCopy(objectsId, uid, "priority", sizeof(short), i,
            obj_no, &status);
        ConfArrayCopy(objectsId, uid, "magnitude", sizeof(double), i,
            obj_no, &status);
        ConfArrayCopy(objectsId, uid, "pId", sizeof(short), i, obj_no, &status);
        ConfArrayCopy(objectsId, uid, "comment", CMTLEN, CMTLEN * i,
            CMTLEN * obj_no, &status);


        if (nextObj > (allocObj + allocSky)) {
            fprintf(stderr,
                "Programming error, nextObj %ld, values  %ld + %ld\n",
                nextObj, allocObj, allocSky);
            exit(-1);
        }
        if (nextGui > allocGui) {
            fprintf(stderr, "Programming error, nextGui %ld, value  %ld\n",
                nextGui, allocGui);
            exit(-1);
        }

    }

    if (tid != 0)
        SdsFreeId(tid, &ignore);


/*
 *  Remove the objects structure and the allocated item of each
 *  of the other structures.
 */
    SdsDelete(objectsId, &status);

    SdsFind(unallocObjId, "allocated", &tid, &status);
    SdsDelete(tid, &status);
    SdsFreeId(tid, &status);

    SdsFind(unallocGuiId, "allocated", &tid, &status);
    SdsDelete(tid, &status);
    SdsFreeId(tid, &status);

/*
 *  Tidy up.
 */
  Exit:
    if (fieldId)
        SdsFreeId(fieldId, &ignore);
    if (objectsId)
        SdsFreeId(objectsId, &ignore);
    if (unallocObjId)
        SdsFreeId(unallocObjId, &ignore);
    if (unallocGuiId)
        SdsFreeId(unallocGuiId, &ignore);

    if (status != STATUS__OK) {
        char buff[200];
        sprintf(buff, "%s:Bad status failure\n", argv[0]);
        Tcl_AppendResult(interp, buff, (char *) NULL);

        MessGetMsg(status, 0, sizeof(buff), buff);
        Tcl_AppendResult(interp, buff, (char *) NULL);

        ErsFlush(&status);

        return TCL_ERROR;

    }
    return result;

}

/*  ------------------------------------------------------------------------- */

/*
 *                 T r a n s l a t e N a m e  { } 
 */
 
static int TranslateName(ClientData clientdata DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])
 
/*+                      
 *  Command name:
 *    TranslateName
 *
 *  Function:
 *    Translate a logical name/environment variable.
 *
 *  Description:
 *    Translate the value of the specified environment variable.
 *
 *  Call:
 *    TranslateName name [options]
 *
 *  Parameters:
 *    (>) name (string) Name of the environment variable to translate.
 *
 *  Options:
 *     [-file (string)]   We intend to access the specified file with
 *                        name being the path to that file.  Returns an
 *                        appropriate file name for use in open calls
 *     [-default (string)] If there is no such name, then use the specified
 *                        string (file will not  be attached) as the result. If 
 *                        default  is not specified and the name does 
 *                        not exist, an error is returned.
 *
 *  Language:
 *     Tcl
 *
 *  Support: Tony Farrell, AAO
 *-
 *  History:
 *     10-May-1994 - TJF - Original version (DtclTrnName)
 *     24-Mar-1995 - TJF - Use DtclCmdError to report errors.
 *     21-Jul-1998 - TJF - Call DitsGetSymbol instead of getenv() to translate
 *                         symbols.
 *     04-Nov-1998 - TJF - Configure specific version.  Go back to getenv
 *                         to remove IMP dependency.  Also remove calls
 *                         to DctlCmdError and VMS and WIN32 code.
 *     15-Mar-2000 - TJF - Incorporate code from KS's OzPoz configure demo
 *                         to support macintosh - separator is a colon
 *                         instead of a slash.
 */

{
    char *defval = "";
    char *file = 0;
    int isfile = 0;
    int hasdefault = 0;
    register int i;
    char Separator[2];

#ifdef macintosh
    strcpy(Separator, ":");
#else
    strcpy(Separator, "/");
#endif

    if (argc < 2) {
        sprintf(interp->result, "%s:wrong # args", argv[0]);
        return TCL_ERROR;
    }

    for (i = 2; i < argc; ++i) {
        if (strcmp(argv[i], "-file") == 0) {
            ++i;
            file = argv[i];
            isfile = 1;
        } else if (strcmp(argv[i], "-default") == 0) {
            ++i;
            defval = argv[i];
            hasdefault = 1;
        } else {
            sprintf(interp->result,
                "%s:Unknown option - \"%s\"\n", argv[0], argv[i]);
            return TCL_ERROR;
        }
    }
/*
 * Get the symbol value.
 */
    if (getenv(argv[1]) == 0) {
/*
 *     No symbol value.  Use default if there is one.
 */
        if (hasdefault) {
            Tcl_SetResult(interp, defval, TCL_VOLATILE);
            return TCL_OK;
        } else {
            sprintf(interp->result, "%s:No translation for \"%s\"",
                argv[0], argv[1]);
            return TCL_ERROR;
        }
    }
    strcpy(interp->result, getenv(argv[1]));
    if (isfile) {
/*
 *     Append file.  Separator is system-dependent.   
 */
        Tcl_AppendResult(interp, Separator, file, NULL);
    }
    return TCL_OK;
}


/*  ------------------------------------------------------------------------- */

/*
 *                        E x e  D i r  { }
 */
 
static int ExeDir(ClientData clientdata DUNUSED, Tcl_Interp * interp,
    int argc DUNUSED, char *argv[]DUNUSED)
    
/*+                
 *  Command name:
 *     ExeDir
 *
 *  Function:
 *     Return the executable directory of this program.
 *
 *  Description:
 *     Returns the executable directory saved when main was invoked.
 *
 *  Call:
 *     ExeDir 
 *
 *  Language:
 *     Tcl
 *
 *  Support: Tony Farrell, AAO
 *-
 *
 *  History:
 *       04-Nov-1998 - TJF - Original version
 *  
 */

{
    Tcl_SetResult(interp, ExecutableDir, TCL_STATIC);
    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*
 *                        E r s F l u s h  { }
 */
 
static int ConfigErsFlush(ClientData clientdata DUNUSED,
    Tcl_Interp * interp DUNUSED, int argc DUNUSED, char *argv[]DUNUSED)

/*+                     
 *  Command name:
 *     ErsFlush
 *
 *  Function:
 *    Invoke ErsFlush routine.
 *
 *  Description:
 *    Invokes the Ers routine ErsFlush. This flushes out any queued error
 *    messages, ensuring that the user sees them.
 *
 *  Call:
 *     ErsFlush
 *
 *  Language:
 *     Tcl
 *
 *  Support: Tony Farrell, AAO
 *-
 *  History:
 *       18-Dec-1998 - TJF - Original version
 */

{
    StatusType status = STATUS__OK;
    ErsFlush(&status);
    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*
 *                          M i m i c  { }
 */

static int ConfMimic(ClientData clientdata DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])

/*+                     
 *  Command name:
 *     Mimic
 *
 *  Function:
 *      Return information about the mimic.
 *
 *  Description:
 *      Various values are used in both the C code and the mimic Tcl
 *      code to determine the size and scalling of the mimic.  This
 *      command returns these values.
 *
 *      Note that FieldRadius represents the area of the configable field,
 *      but the park positions of the fibres will be outside this, but
 *      within CircleRadius.
 *
 *  Call:
 *      Mimic value
 *
 *  Parameters:
 *      (>) value (string) The value to fetch, WindowSize, CircleRadius
 *                           CircleOffset or FieldRadius.
 *
 *  Language:
 *      Tcl
 *
 *  Support: Tony Farrell, AAO
 *-
 *  History:
 *       18-Dec-1998 - TJF - Original version
 *  
 */

{
    if (argc != 2) {
        sprintf(interp->result, "%s:wrong # args", argv[0]);
        return TCL_ERROR;
    }
    if (strcmp(argv[1], "WindowSize") == 0) {
        sprintf(interp->result, "%d", MimicWindowSize);
    } else if (strcmp(argv[1], "CircleRadius") == 0) {
        sprintf(interp->result, "%d", MimicCircleRadius);
    } else if (strcmp(argv[1], "CircleOffset") == 0) {
        sprintf(interp->result, "%d", MimicCircleOffset);
    } else if (strcmp(argv[1], "FieldRadius") == 0) {
        /* Grab the field radius and convert it to pixels */
        double fieldRadius = FpilGetFieldRadius(*Instrument);
        double scale = (double) MimicCircleRadius /
            (double) FpilGetFieldRadiusIncPark(*Instrument);
        sprintf(interp->result, "%d", (int) (fieldRadius * scale));
    } else {
        sprintf(interp->result,
            "%s:Invalid argument, must be one of \"WindowSize\"", argv[0]);
        strcat(interp->result, " \"CircleRadius\" or \"CircleOffset\"\n");
        return TCL_ERROR;
    }
    return TCL_OK;
}


/*  ------------------------------------------------------------------------- */

/*                      P l a t e  X  t o  D i s p  X  { }
 *
 *                      P l a t e  Y  t o  D i s p  Y  { }
 */
 
/*+
 *  Command name:
 *     PlateXtoDispX,PlateXtoDispX
 *
 *  Function:
 *     Converts between an plate position and a display position.
 *
 *  Description:
 *     These routines take a plate position in X or Y (in microns) and return
 *     the corresponding X or Y coordinate on the display (in pixels).
 *
 *  Call:
 *     PlateXtoDispX PlatePosn zoom
 *     PlateYtoDispY PlatePosn zoom
 *
 *  Language:
 *     Tcl
 *      
 *  Parameter:
 *     PlatePosn  (int) Plate X or Y coordinate in microns
 *     Zoom       (double) Current zoom factor.
 *
 *  Returns:
 *     Display pixel position.
 *
 *  Support: Keith Shortridge, AAO
 *-
 *  History:
 *       23-May-2001 - KS - Original version
 */
 
static int PlateXtoDispX(ClientData clientdata DUNUSED, Tcl_Interp * interp,
    int argc, char *argv[])
    
{
    char String[32];
    if (argc != 3) {
        sprintf(interp->result, "%s:wrong # args", argv[0]);
        return TCL_ERROR;
    }
    sprintf (String,"%d",FieldPlateToDispX(0,atoi(argv[1]),atof(argv[2])));
    Tcl_SetResult(interp, String, TCL_VOLATILE);
    return TCL_OK;
}

static int PlateYtoDispY(ClientData clientdata DUNUSED, Tcl_Interp * interp,
    int argc, char *argv[])
{
    char String[32];
    if (argc != 3) {
        sprintf(interp->result, "%s:wrong # args", argv[0]);
        return TCL_ERROR;
    }
    sprintf (String,"%d",FieldPlateToDispY(0,atoi(argv[1]),atof(argv[2])));
    Tcl_SetResult(interp, String, TCL_VOLATILE);
    return TCL_OK;
}


/*  ------------------------------------------------------------------------- */

/*+                     D i s p  X  t o  P l a t e  X  { }
 *
 *                      D i s p  Y  t o  P l a t e  Y  { }
 *
 *  Command name:
 *      DispXtoPlateX,DispYtoPlateY
 *
 *  Function:
 *      Converts between a display position and a plate position.
 *
 *  Description:
 *      These routines take a coordinate in X or Y (in pixels) and return
 *      the corresponding plate X or Y coordinate (in microns). Note that
 *      the pixel value is the pixel position in the scroll region of the
 *      canvas being displayed, not the pixel position in the window (which
 *      only displays a fraction of the canvas scroll region when zoomed).
 *
 *  Call:
 *      DispXtoPlateX DispPosn zoom
 *      DispYtoPlateY DispPosn zoom
 *
 *  Language:
 *      Tcl
 *      
 *  Parameter:
 *      DispPosn   (int) Display X or Y coordinate in pixels
 *      Zoom       (double) Current zoom factor.
 *
 *  Returns:
 *      Plate position in microns.
 *
 *  Support: Keith Shortridge, AAO
 *-
 *  History:
 *       07-Jun-2001 - KS - Original version
 */

static int DispXtoPlateX(ClientData clientdata DUNUSED, Tcl_Interp * interp,
    int argc, char *argv[])
{
    char String[32];
    if (argc != 3) {
        sprintf(interp->result, "%s:wrong # args", argv[0]);
        return TCL_ERROR;
    }
    
    /*  It is because the position is the position in the scroll region
     *  and not in the window that the second argument (the offset) here is
     *  zero. If it were the window position, we would need the offset.
     */
     
    sprintf (String,"%d",DispToFieldPlateX(atoi(argv[1]),0.0,atof(argv[2])));
    Tcl_SetResult(interp, String, TCL_VOLATILE);
    return TCL_OK;
}

static int DispYtoPlateY(ClientData clientdata DUNUSED, Tcl_Interp * interp,
    int argc, char *argv[])
{
    char String[32];
    if (argc != 3) {
        sprintf(interp->result, "%s:wrong # args", argv[0]);
        return TCL_ERROR;
    }
    sprintf (String,"%d",DispToFieldPlateY(atoi(argv[1]),0.0,atof(argv[2])));
    Tcl_SetResult(interp, String, TCL_VOLATILE);
    return TCL_OK;
}


/*  ------------------------------------------------------------------------- */

/*+                     P l a t e  X Y  t o  R a  D e c  { }
 *
 *  Command name:
 *      PlateXYtoRaDec
 *
 *  Function:
 *      Returns the mean Ra,Dec coordinates for a given plate position.
 *
 *  Description:
 *      This routine takes a plate position (the X and Y coordinates in
 *      microns) and returns the mean Ra,Dec coordinates that correspond to
 *      that plate position.
 *
 *  Call:
 *      PlateXYtoRaDec Cfid Xp Yp Ra Dec
 *
 *  Language:
 *      Tcl
 *      
 *  Parameters:
 *      Cfid       (Sds id) Id of the overall configuration structure
 *      Xp         (int) Plate X position in microns
 *      Yp         (int) Plate Y position in microns
 *      Ra         (double Tcl variable) Variable to receive Ra in radians
 *      Dec        (double Tcl variable) Variable to receive Dec in radians
 *
 *  Support: Keith Shortridge, AAO
 *-
 *  History:
 *       07-Jun-2001 - KS - Original version
 *       17-Oct-2002 - KS - Added use of FpilGetPointingWavelength()
 */

static int PlateXYtoRaDec(ClientData clientdata DUNUSED, Tcl_Interp * interp,
    int argc, char *argv[])
{
    SdsIdType id, idf;
    double cra, cdec, mjd;
    double xp,yp;
    double xf,yf;
    double ara,adec;
    double raMean,decMean;
    double gmap[21];                   /* Mean to apparent parameters  */
    char buff[200];
    double Wavelength;
    StatusType status = STATUS__OK;
    
    if (argc != 6) {
        interp->result = "wrong # args";
        return TCL_ERROR;
    }
    
    id = atol(argv[1]);
    
    /*
     *  Get field centre details
     */
     
    SdsFind(id, "fieldData", &idf, &status);
    ArgGetd(idf, "appRa", &cra, &status);
    ArgGetd(idf, "appDec", &cdec, &status);
    ArgGetd(idf, "configMjd", &mjd, &status);
    SdsFreeId(idf, &status);

    /*
     *  Set up x,y conversion parameters. We use the main pointing
     *  wavelength, since there isn't a specific target in question here,
     *  at least, not so far as we know.
     */

    slaMappa(2000.0, mjd, gmap);
    
    Wavelength = FpilGetPointingWavelength(*Instrument);
    FpilModelCvtInit(TelModel, mjd, 0.0, gtemp, gpress, ghumid, 
                       Wavelength, Wavelength, TelModelParams, &status);
        
    xp = atof(argv[2]);
    yp = atof(argv[3]);
    FpilModelPos2Xy(TelModel, xp, yp, &xf, &yf, &status);
    FpilModelXy2Rd(TelModel, cra, cdec, xf, yf, mjd, &ara, &adec,&status);
    slaAmpqk (ara,adec,gmap,&raMean,&decMean);
    sprintf(buff, "%.16g", raMean);
    Tcl_SetVar(interp, argv[4], buff, 0);
    sprintf(buff, "%.16g", decMean);
    Tcl_SetVar(interp, argv[5], buff, 0);
    
    if (status != STATUS__OK) {
        MessGetMsg(status, 0, sizeof(buff), buff);
        Tcl_AppendResult(interp, "\nError converting to RA,Dec\n",
            (char *) NULL);
        Tcl_AppendResult(interp, buff, (char *) NULL);
        return TCL_ERROR;
    }
    return TCL_OK;
}


/*  ------------------------------------------------------------------------- */

/*+                 R a  D e c  T o  P l a t e  X Y { }
 *
 *  Command name:
 *      RaDecToPlateXY
 *
 *  Function:
 *      Returns the plate position corresponding to mean Ra,Dec coordinates.
 *
 *  Description:
 *      This routine takes a pair of mean Ra,Dec coordinates and returns
 *      the corresponding plate position (the X and Y coordinates in
 *      microns).
 *
 *  Call:
 *      RaDecToPlateXY Cfid Ra Dec Xp Yp 
 *
 *  Language:
 *      Tcl
 *      
 *  Parameters:
 *      Cfid       (Sds id) Id of the overall configuration structure
 *      Ra         (double) Ra in radians
 *      Dec        (double) Dec in radians
 *      Xp         (int Tcl variable) Variable to receive Plate X position
 *                 in microns
 *      Yp         (int Tcl variable) Variable to receive Plate Y position
 *                 in microns
 *
 *  Support: Keith Shortridge, AAO
 *-
 *  History:
 *       07-Aug-2001 - KS - Original version
 *       17-Oct-2002 - KS - Added use of FpilGetPointingWavelength()
 */

static int RaDecToPlateXY(ClientData clientdata DUNUSED, Tcl_Interp * interp,
    int argc, char *argv[])
{
    SdsIdType id, idf;
    double cra, cdec, mjd;
    double xp,yp,xf,yf;
    double ara,adec;
    double raMean,decMean;
    double gmap[21];                   /* Mean to apparent parameters  */
    char buff[200];
    double Wavelength;
    StatusType status = STATUS__OK;
    
    if (argc != 6) {
        interp->result = "wrong # args";
        return TCL_ERROR;
    }
    
    id = atol(argv[1]);
    
    /*
     *  Get field centre details
     */
     
    SdsFind(id, "fieldData", &idf, &status);
    ArgGetd(idf, "appRa", &cra, &status);
    ArgGetd(idf, "appDec", &cdec, &status);
    ArgGetd(idf, "configMjd", &mjd, &status);
    SdsFreeId(idf, &status);

    /*
     *  Set up x,y conversion parameters. We use the main pointing
     *  wavelength, since there isn't a specific target in question here,
     *  at least, not so far as we know.

     */

    slaMappa(2000.0, mjd, gmap);
    
    Wavelength = FpilGetPointingWavelength(*Instrument);
    FpilModelCvtInit(TelModel, mjd, 0.0, gtemp, gpress, ghumid, 
                       Wavelength, Wavelength, TelModelParams, &status);
        
    raMean = atof(argv[2]);
    decMean = atof(argv[3]);
    
    slaMapqkz(raMean, decMean, gmap, &ara, &adec);
    FpilModelRd2Xy(TelModel, cra, cdec, ara, adec, mjd, &xf, &yf, &status);
    FpilModelXy2Pos(TelModel, xf, yf, &xp, &yp, &status);
            
    sprintf(buff, "%g", xp);
    Tcl_SetVar(interp, argv[4], buff, 0);
    sprintf(buff, "%g", yp);
    Tcl_SetVar(interp, argv[5], buff, 0);
    
    if (status != STATUS__OK) {
        MessGetMsg(status, 0, sizeof(buff), buff);
        Tcl_AppendResult(interp, "\nError converting from RA,Dec\n",
            (char *) NULL);
        Tcl_AppendResult(interp, buff, (char *) NULL);
        return TCL_ERROR;
    }
    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*+                     M e a n  T o  A p p  R a  D e c  { }
 *
 *  Command name:
 *      MeanToAppRaDec
 *
 *  Function:
 *      Converts mean Ra,Dec coordinates to apparent coordinates.
 *
 *  Description:
 *      This routine takes a pair of mean Ra,Dec coordinates and returns the
 *      corresponding apparent Ra,Dec coordinates, given the Mjd in the
 *      current configuration.
 *
 *  Call:
 *      MeanToAppRaDec Cfid MeanRa MeanDec AppRa AppDec
 *
 *  Language:
 *      Tcl
 *      
 *  Parameters:
 *      Cfid       (Sds id) Id of the overall configuration structure
 *      MeanRa     (double) Mean Ra in radians
 *      MeanDec    (double) Mean Dec in radians
 *      AppRa      (double Tcl variable) Variable to receive app Ra in radians
 *      AppDec     (double Tcl variable) Variable to receive app Dec in radians
 *
 *  Support: Keith Shortridge, AAO
 *-
 *  History:
 *       07-Jun-2001 - KS - Original version
 */

static int MeanToAppRaDec(ClientData clientdata DUNUSED, Tcl_Interp * interp,
    int argc, char *argv[])
{
    SdsIdType id, idf;
    double mjd;
    double raApp,decApp;
    double raMean,decMean;
    char buff[200];
    StatusType status = STATUS__OK;
    
    if (argc != 6) {
        interp->result = "wrong # args";
        return TCL_ERROR;
    }
    id = atol(argv[1]);
    
    /*
     *  Get Mjd from field centre details
     */
     
    SdsFind(id, "fieldData", &idf, &status);
    ArgGetd(idf, "configMjd", &mjd, &status);
    SdsFreeId(idf, &status);
    
    /*  Do the conversion */
    
    raMean = atof(argv[2]);
    decMean = atof(argv[3]);
    slaMap(raMean, decMean, 0 ,0,0,0,2000,  mjd, &raApp, &decApp);
    
    /*  Return the values */
    
    sprintf(buff, "%f", raApp);
    Tcl_SetVar(interp, argv[4], buff, 0);
    sprintf(buff, "%f", decApp);
    Tcl_SetVar(interp, argv[5], buff, 0);
    
    if (status != STATUS__OK) {
        MessGetMsg(status, 0, sizeof(buff), buff);
        Tcl_AppendResult(interp, "\nError converting mean to apparent Ra,Dec\n",
            (char *) NULL);
        Tcl_AppendResult(interp, buff, (char *) NULL);
        return TCL_ERROR;
    }
    return TCL_OK;
    
}

/*  ------------------------------------------------------------------------- */

/*+                  U n a l l o c a t e d  T y p e  { }
 *
 *  Command name:
 *      UnallocatedType
 *
 *  Function:
 *      Return the type code used for an unallocated fibre.
 *
 *  Description:
 *      The code used for an unallocated fibre in the "objects" section
 *      depends on the instrument. This routine returns the code (a single
 *      character) used for the currently selected instrument.
 *
 *  Call:
 *      UnallocatedType 
 *
 *  Language:
 *      Tcl
 *
 *  Support: Keith Shortridge, AAO
 *-
 *  History:
 *      31-Jul-2000 - KS - Original version
 */

static int UnallocatedType(ClientData clientdata DUNUSED, Tcl_Interp * interp,
    int argc DUNUSED, char *argv[]DUNUSED)
{
    char String[2];
    String[0] = FpilUnallocatedType(*Instrument);
    String[1] = '\0';
    Tcl_SetResult(interp, String, TCL_VOLATILE);
    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*+                   T y p e  I s  S k y  { }
 *
 *  Command name:
 *      TypeIsSky
 *
 *  Function:
 *      Tests if a specified target type indicates a sky position.
 *
 *  Description:
 *      It is possible to tell from just the 'type' character used if a given
 *      target represents a sky position, and this routine does so, returning 
 *      true if it is and false otherwise. 
 *
 *  Call:
 *      TypeIsSky targetType 
 *
 *   Parameters:
 * 
 *      (>) targetType  (char) The type code character for the target in
 *                             question.
 *     
 *  Support: Keith Shortridge, AAO
 *-
 *  History:
 *       16-Aug-2000 - KS - Original version
 */

static int TypeIsSky(ClientData clientdata DUNUSED, Tcl_Interp * interp,
    int argc, char *argv[])
{
    /*  Validate the arguments */

    if (argc != 2) {
        Tcl_SetResult (interp,"wrong # args",TCL_STATIC);
        return TCL_ERROR;
    }
    if (FpilIsTargetSky(*Instrument,*(argv[1]))) {
       Tcl_SetResult(interp, "1",TCL_STATIC);
    } else {
       Tcl_SetResult(interp, "0",TCL_STATIC);
    }
    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*+                   T y p e  I s  G u i d e  { }
 *
 *  Command name:
 *      TypeIsTelGuide
 *
 *  Function:
 *      Tests if a specified target type indicates a guide star.
 *
 *  Description:
 *      It is possible to tell from just the 'type' character used if a given
 *      target is a fiducial guide star, and this routine does so, returning 
 *      true if it is and false otherwise. Note that this also returns true
 *      if the target is a telescope guide object, so one may want to use 
 *      the TypeIsTelGuide command in conjunction with it to distinguish 
 *      between the two cases.
 *
 *  Call:
 *      TypeIsGuide targetType
 *
 *   Parameters:
 * 
 *      (>) targetType  (char) The type code character for the target in
 *                             question.
 *     
 *  Support: Keith Shortridge, AAO
 *-
 *  History:
 *      16-Aug-2000 - KS - Original version
 */

static int TypeIsGuide(ClientData clientdata DUNUSED, Tcl_Interp * interp,
    int argc, char *argv[])
{
    /*  Validate the arguments */

    if (argc != 2) {
        Tcl_SetResult (interp,"wrong # args",TCL_STATIC);
        return TCL_ERROR;
    }
    if (FpilIsTargetGuide(*Instrument,*(argv[1]))) {
       Tcl_SetResult(interp, "1",TCL_STATIC);
    } else {
       Tcl_SetResult(interp, "0",TCL_STATIC);
    }
    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*+                   T y p e  I s  T e l  G u i d e
 *
 *  Command name:
      TypeIsTelGuide

 *  Function:
        Tests if a specified target type indicates a telescope guide object.

 *  Description:
        Some instruments - such as the VLT - support the concept of a
        target type that is intended only for use by a telescope guider
        and not by a fibre. It is possible to tell from just the 'type'
        character used if a given target is such a telescope guider target,
        and this routine does so, returning true if it is and false
        otherwise.

 *  Call:
      TypeIsTelGuide  

 *   Parameters:
 
     (>) targetType  (char) The type code character for the target in question.
     
 *  Support: Keith Shortridge, AAO

 *-

 *  History:
       16-Aug-2000 - KS - Original version
  
*/

static int TypeIsTelGuide(ClientData clientdata DUNUSED, Tcl_Interp * interp,
    int argc, char *argv[])
{
    /*  Validate the arguments */

    if (argc != 2) {
        Tcl_SetResult (interp,"wrong # args",TCL_STATIC);
        return TCL_ERROR;
    }
    if (FpilIsTargetTelGuide(*Instrument,*(argv[1]))) {
       Tcl_SetResult(interp, "1",TCL_STATIC);
    } else {
       Tcl_SetResult(interp, "0",TCL_STATIC);
    }
    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*+                     U s e  G r e y
 *
 *  Command name:
      UseGrey

 *  Function:
        Tests if the grey scale display flag is set.

 *  Description:
        This routine returns the value of the global flag 'useGrey',
        which is used to indicate that a grey-scale colour scheme
        should be used.

 *  Call:
        UseGrey  

 *   Parameters: None
      
 *  Support: Keith Shortridge, AAO

 *-

 *  History:
       15-Jun-2001 - KS - Original version
  
*/

static int ConfUseGrey(ClientData clientdata DUNUSED, Tcl_Interp * interp,
    int argc DUNUSED, char *argv[] DUNUSED)
{
    if (useGrey) {
       Tcl_SetResult(interp, "1",TCL_STATIC);
    } else {
       Tcl_SetResult(interp, "0",TCL_STATIC);
    }
    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*+                         S e t  G r e y  { }
 *
 *  Command name:
 *      SetGrey
 *
 *  Function:
 *        Sets the grey scale display flag.
 *
 *  Description:
 *        This routine sets the value of the global flag 'useGrey',
 *        which is used to indicate that a grey-scale colour scheme
 *        should be used.
 *
 *  Call:
 *        SetGrey [flag]  
 *
 *  Parameters:
 * 
 *        (>) Flag  (int) Non-zero if a grey-scale scheme is to be used. If
 *                        omitted, the grey-scale flag is set true.
 *     
 *  Support: Keith Shortridge, AAO
 *-
 *  History:
 *       15-Jun-2001 - KS - Original version
 */

static int ConfSetGrey(ClientData clientdata DUNUSED, Tcl_Interp * interp,
    int argc, char *argv[])
{
    /*  Validate the arguments */

    if (argc == 1) {
        useGrey = 1;
    } else if (argc == 2) {
        useGrey = atoi(argv[1]);
    } else {
        Tcl_SetResult (interp,"wrong # args",TCL_STATIC);
        return TCL_ERROR;
    }
    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*+                        E x p e r t  A l l o w e d
 *
 *  Command name:
      ExpertAllowed

 *  Function:
        Tests if the 'expert mode allowed' flag is set.

 *  Description:
        This routine returns the value of the global flag 'ExpertAllowed',
        which is used to indicate that the program is being run in a way
        that allows expert mode - in practice, this is if the command used
        to run the program included the word 'expert'. (The obvious way
        to do this is to have a link to the usual configure executable that
        has expert in the name - at least under UNIX.)

 *  Call:
        ExpertAllowed  

 *   Parameters: None
      
 *  Support: Keith Shortridge, AAO

 *-

 *  History:
        5-Nov-2001 - KS - Original version
  
*/

static int ConfExpertAllowed(ClientData clientdata DUNUSED,
        Tcl_Interp * interp, int argc DUNUSED, char *argv[] DUNUSED)
{
    if (ExpertAllowed) {
       Tcl_SetResult(interp, "1",TCL_STATIC);
    } else {
       Tcl_SetResult(interp, "0",TCL_STATIC);
    }
    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*
 *                        V L T  G u i d e r  { }
 */
 
static int VLTGuider(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])
    
/*+
 *  Command Name:
 *    VLTGuider
 *
 *  Function:
 *    Determine the two guide probe positions for a specified target star.
 *
 *  Call:
 *    VLTGuider id index posn1 posn2
 *
 *  Description:
 *    This routine is passed the details of a possible VLT guide probe target
 *    star, in terms of the SDS Id for a sub-structure of the configuration
 *    data (in practice this should be the unallocated guide sub-structure)
 *    and an index. It calculates the two possible VLT guide probe positions
 *    and returns their details.
 *
 *  Parameters:
 *    (>)  id    (Sds Id)  Sds identifier of the substructure
 *    (>)  index (int)     index number - starting at one
 *    (<)  posn1 (char)    String giving "XP YP XS YS" for guide
 *                         probe in first position, where XP,YP is
 *                         location of guide pivot and XS,YS is
 *                         the location of the star.
 *    (<)  posn2 (char)    String giving "XP YP XS YS" for guide
 *                         probe in second position.
 *    (<)  pospos (int)    Either 1 or 2, depending on whether the
 *                         first position is classed as +ve or -ve.
 *                         If the first is +ve, this is returned as
 *                         1. If the second is +ve, it is returned
 *                         as 2.
 *
 *   Support:
 *     Keith Shortridge, AAO
 *
 *   Version Date:
 *     9-Aug-1999  Original version.
 *    22-Oct-2002  Code setting POS and NEG simplified and corrected.    
 *-
 */
{
    StatusType status;
    long index_no;
    SdsIdType id;
    SdsIdType topid;
    char type;
    unsigned long actlen;

    if (argc != 6) {
        interp->result = "wrong # args";
        return TCL_ERROR;
    }
    status = SDS__OK;
    topid = (SdsIdType) (atol(argv[1]));
    index_no = atol(argv[2]) - 1;

    SdsFind(topid, "type", &id, &status);
    SdsGet(id, sizeof(char), index_no, &type, &actlen, &status);
    SdsFreeId(id, &status);

    /*  This test for a 'Guide' target isn't strictly correct, as it will
     *  include ordinary guide stars as well as the more specialised VLT
     *  guide probe, which is what we should really be testing for here.
     */

    if (FpilIsTargetGuide(*Instrument, type)) {

        double GuideStarRadius;
        double GuideProbeLength = 560000;
        double Dx, Dy;
        int X1, Y1, X2, Y2;
        int X, Y;               /*  Guide star plate coordinates */
        double Theta, Phi, PhiOne, PhiTwo;
        double Pi = 3.14159;
        char Buffer[80];
        
        SdsFind(topid, "x", &id, &status);
        SdsGet(id, sizeof(int), index_no, &X, &actlen, &status);
        SdsFreeId(id, &status);
        SdsFind(topid, "y", &id, &status);
        SdsGet(id, sizeof(int), index_no, &Y, &actlen, &status);
        SdsFreeId(id, &status);

        /*  This calculation, which is messier than it needs to be, 
         *  works out two angles: Theta is the Angle between the
         *  line going straight up from the center of the field (0,0)
         *  and the line from the center of the field to the
         *  position of the guide star (X,Y), measured anti-clockwise.
         *  Now imagine a triangle. One of the sides is the 
         *  line from (0,0) to (X,Y), that is the line from the center
         *  of the field to the guide star. This line has a length 
         *  we calculate as GuideStarRadius. One of the sides is a line
         *  from the center of the field to the position of the rotator
         *  axis of the guide probe in one of the two possible positions
         *  it can use for this star. This line must be of length
         *  GuideProbeLength. The position of the rotator axis of the
         *  guide probe may be anywhere on the circle of radius 
         *  GuideProbeLength drawn around the center of the field. The
         *  third side of the triangle is the line from the center of
         *  the rotator axis to the guide star. This also has length
         *  GuideProbeLength - as far as I know the radius of the circle
         *  is exactly the length from the rotator axis to the center of
         *  the probe itself, although this isn't necessarily true.
         *  This is therefore an isosceles triangle, with two angles
         *  the same, both having the value we call Phi in this code.
         *  Trivial trig then gives Phi. This would be a lot
         *  easier with a diagram!
         */

        if ((abs(X) < 1) && (abs(Y) < 1)) {
            Theta = 0.0;
            GuideStarRadius = 0.0;
            Phi = 0.0;
        } else {
            Dx = abs(X);
            Dy = abs(Y);
            GuideStarRadius = sqrt((Dx * Dx) + (Dy * Dy));
            Phi = acos(GuideStarRadius / (GuideProbeLength * 2.0));
            Theta = atan(Dy / Dx);
            if ((X < 0) && (Y > 0))
                Theta = Pi - Theta;
            if ((X < 0) && (Y < 0))
                Theta += Pi;
            if ((X > 0) && (Y < 0))
                Theta = (2 * Pi) - Theta;
        }

        /*  You can now draw two lines from the center of the field to
         *  the two possible positions of the guide probe rotator axis.
         *  Both of these will be Phi radians away from the line
         *  joining the field center to the guide star, one on each
         *  side of that line. These angles (using the same convention
         *  as described above for Theta) we call PhiOne and PhiTwo.
         *  Having got those, we can then calculate the X,Y positions
         *  for the two possible rotator axis positions, (X1,Y1) and
         *  (X2,Y2). The code gets messy because of the different 
         *  quadrants, and I think that could be avoided, but it
         *  works.
         */

        PhiOne = Theta + Phi;
        if (PhiOne > (2.0 * Pi))
            PhiOne -= (2.0 * Pi);
        PhiTwo = Theta - Phi;
        if (PhiTwo < 0.0)
            PhiTwo = (2.0 * Pi) + PhiTwo;
        if (PhiOne > (3.0 * Pi / 2.0)) {
            PhiOne = (2.0 * Pi) - PhiOne;
            Y1 = -(int) (GuideProbeLength * sin(PhiOne));
            X1 = (int) (GuideProbeLength * cos(PhiOne));
        } else if (PhiOne > Pi) {
            PhiOne = PhiOne - Pi;
            Y1 = -(int) (GuideProbeLength * sin(PhiOne));
            X1 = -(int) (GuideProbeLength * cos(PhiOne));
        } else if (PhiOne > (Pi / 2.0)) {
            PhiOne = Pi - PhiOne;
            Y1 = (int) (GuideProbeLength * sin(PhiOne));
            X1 = -(int) (GuideProbeLength * cos(PhiOne));
        } else {
            Y1 = (int) (GuideProbeLength * sin(PhiOne));
            X1 = (int) (GuideProbeLength * cos(PhiOne));
        }
        if (PhiTwo > (3.0 * Pi / 2.0)) {
            PhiTwo = (2.0 * Pi) - PhiTwo;
            Y2 = -(int) (GuideProbeLength * sin(PhiTwo));
            X2 = (int) (GuideProbeLength * cos(PhiTwo));
        } else if (PhiTwo > Pi) {
            PhiTwo = PhiTwo - Pi;
            Y2 = -(int) (GuideProbeLength * sin(PhiTwo));
            X2 = -(int) (GuideProbeLength * cos(PhiTwo));
        } else if (PhiTwo > (Pi / 2.0)) {
            PhiTwo = Pi - PhiTwo;
            Y2 = (int) (GuideProbeLength * sin(PhiTwo));
            X2 = -(int) (GuideProbeLength * cos(PhiTwo));
        } else {
            Y2 = (int) (GuideProbeLength * sin(PhiTwo));
            X2 = (int) (GuideProbeLength * cos(PhiTwo));
        }
        sprintf(Buffer, "%d %d %d %d", X1, Y1, X, Y);
        Tcl_SetVar(interp, argv[3], Buffer, 0);
        sprintf(Buffer, "%d %d %d %d", X2, Y2, X, Y);
        Tcl_SetVar(interp, argv[4], Buffer, 0);
        
        /*  We need to calculate which of the two positions is 
         *  classified as +ve and which is -ve and return this
         *  as the final parameter. The convention is as follows:
         *
         *  POS is when the probe arm has to move CLOCKWISE around 
         *  its rotation point when moving from center position to the
         *  actual guide star, when you look at the adapter from outside.
         *
         *  (This is quoted directly from an e-mail from Luca Pasquini.
         *  We tried to clarify what 'outside' means, and got the reply
         *  that this means from the opposite side to that drawn by
         *  configure - so if you look down on the plate and see the guide
         *  probe between you and the plate, you're looking from 'inside'.)
         *
         *  There was an attempt made to code a test based on calculating
         *  the actual angles involved and comparing them, but it took
         *  Andreas Kaufer at ESO to point out that in fact, with Alpha1
         *  always being less than Phi in the above code, it follows that
         *  the point at X1,Y1 is in fact always the POS position and
         *  the point at X2,Y2 is always the NEG position. So nothing
         *  needs to be calculated. (AKA comment 2002-10-18)
         */
        
	     Tcl_SetVar(interp, argv[5], "1", 0);
    }	

    if (status == SDS__OK) {
        return TCL_OK;
    } else {
        char mbuf[100];
        MessGetMsg(status, 0, 100, mbuf);
        sprintf(interp->result, "Error Getting Object Data\n%s", mbuf);
        return TCL_ERROR;
    }
}

/* End of VLTGuider */

/*  ------------------------------------------------------------------------- */

/*                V L T  G u i d e  P r o b e  C o n s t a n t s
 *
 *  The following values define the shape of the VLT Guide probe shaddow.
 *  The shaddow is assumed to consist of a circle centered on the pivot
 *  position of the guide probe, of radius GuideProbeOvalRadius (all
 *  values are in microns), and then a polygon with a number of vertices
 *  given by GuideProbeVertices. The positions of the vertices, relative
 *  to the pivot position, are given by the array GuideProbeVertexValues.
 *  This consists of a set of values, two for each vertex, giving the
 *  x,y coordinates of each vertex. The final vertex should be the same as
 *  the first, so this contains the complete set of boundary lines for
 *  the polygon. So it should have a number of entries
 *  given by 2 * GuideProbeVertices. Note that the code assumes that the
 *  polygon is symmetrical about the X axis. GuideProbeLength should
 *  be the length of the probe in microns - it should be the largest
 *  X-value found in GuideProbeVertexValues.
 *
 *  These vertex coordinates have been taken from the code for vfArmPolygon,
 *  as supplied by Rodrigo Amestica. The earlier versions of this code used
 *  a slightly simplified vertex set, taken from vltguiArmDisplay.tcl as
 *  supplied to AAO by Jason Spyromilio. After discussion with Rodrigo, it
 *  was decided to use the vfArmPolygon values. Luca Pasquini's report on the
 *  FLAMES commissioning included the suggestion 'Suggest enlarging the probe
 *  shadow by 2-3 arcseconds in CONFIGURE to allow deviations from 45 degrees
 *  position angle'. This code does not yet incorporate that additional 
 *  shaddow, as I wanted to experiment with the revised vertex values first.
 */

#define GuideProbeOvalRadius 137000
#define GuideProbeLength 594000
#define GuideProbeVertices 11

static int GuideProbeVertexValues[] =
    { 0, 137000, 212000, 101000, 314000, 64000, 496000, 50000, 594000, 37000,
    594000, -37000, 496000, -50000, 314000, -64000, 212000, -101000,
    0, -137000, 0, 137000
};

/*  ------------------------------------------------------------------------- */

/*
 *                    V L T  D r a w  G u i d e r  { }
 */
 
static int VLTDrawGuider(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])
    
/*+
 *  Command Name:
 *    VLTDrawGuider
 *
 *  Function:
 *    Returns the Tk canvas drawing commands to draw the VLT guide probe.
 *
 *  Call:
 *    VLTDrawGuider posn zoom oval polygon
 *
 *  Description:
 *    Given a position string for the VLT guide probe (as generated by
 *    VLTGuider{}), this generates a series of parameters that can be used
 *    in Tk canvas drawing commands to draw the guide in the specified
 *    position.
 *
 *  Parameters:
 *    (>)  posn (char)    String giving "XP YP XS YS" for guide
 *                        probe in its position, where XP,YP is
 *                        location of guide pivot and XS,YS is
 *                        the location of the star.
 *    (>)  zoom (float)   The current zoom factor.
 *    (<)  pivot (char)   Numeric parameters for an oval canvas
 *                        command that will draw a small circle
 *                        to indicate the position of the guide
 *                        probe pivot.
 *    (<)  oval (char)    Numeric parameters for an oval canvas
 *                        command that will draw the circular
 *                        part of the guide probe.
 *    (<)  polygon (char) Numeric parameters for a polygon canvas
 *                        command that will draw the remaining
 *                        part of the guide probe.
 *
 *   Support:
 *     Keith Shortridge, AAO
 *
 *   Version Date:
 *     9-Aug-1999    
 *-
 */
{
    int XP, YP, XS, YS;
    int DeltaX, DeltaY;
    char Oval[64];
    char Polygon[512];
    char Polygon_crack[2048];
    int VIndex;
    float Zoom;
    double Theta;
    int Vertex;
    double Pi = 3.14159;
    if (argc != 7) {
        interp->result = "wrong # args";
        return TCL_ERROR;
    }
    sscanf(argv[1], "%d %d %d %d", &XP, &YP, &XS, &YS);
    Zoom = atof(argv[2]);
    sprintf(Oval, "%d %d %d %d",
        FieldPlateToDispX(0, XP, Zoom) - 10,
        FieldPlateToDispY(0, YP, Zoom) - 10,
        FieldPlateToDispX(0, XP, Zoom) + 10,
        FieldPlateToDispY(0, YP, Zoom) + 10);
    Tcl_SetVar(interp, argv[3], Oval, 0);
    sprintf(Oval, "%d %d %d %d",
        FieldPlateToDispX(0, XP - GuideProbeOvalRadius, Zoom),
        FieldPlateToDispY(0, YP - GuideProbeOvalRadius, Zoom),
        FieldPlateToDispX(0, XP + GuideProbeOvalRadius, Zoom),
        FieldPlateToDispY(0, YP + GuideProbeOvalRadius, Zoom));
    Tcl_SetVar(interp, argv[4], Oval, 0);
    Polygon[0] = '\0';
    DeltaX = XS - XP;
    DeltaY = YS - YP;
    Theta = atan((double) abs(DeltaY) / (double) abs(DeltaX));
    if ((DeltaX > 0) && (DeltaY < 0)) {
        Theta = Theta;
    } else if ((DeltaX > 0) && (DeltaY > 0)) {
        Theta = (Pi * 2.0) - Theta;
    } else if ((DeltaX < 0) && (DeltaY > 0)) {
        Theta = Pi + Theta;
    } else if ((DeltaX < 0) && (DeltaY < 0)) {
        Theta = Pi - Theta;
    }
    VIndex = 0;
    for (Vertex = 0; Vertex < GuideProbeVertices; Vertex++) {
        int XVertex;
        int YVertex;
        int XVertPos;
        int YVertPos;
        char Buffer[32];
        XVertex = GuideProbeVertexValues[VIndex++];
        YVertex = GuideProbeVertexValues[VIndex++];
        XVertPos = (cos(Theta) * XVertex) + (sin(Theta) * YVertex)
            + XP;
        YVertPos = (cos(Theta) * YVertex) - (sin(Theta) * XVertex)
            + YP;
        sprintf(Buffer, "%d %d ", FieldPlateToDispX(0, XVertPos, Zoom),
            FieldPlateToDispY(0, YVertPos, Zoom));
        strcat(Polygon, Buffer);
    }
    Tcl_SetVar(interp, argv[5], Polygon, 0);

    Polygon_crack[0] = '\0';
    /* First figure out the radius via coordinates... */
    SdsIdType id, idf;
    double cra, cdec, mjd;
    double xp,yp,xf,yf;
    double cxp,cyp,cxf,cyf;
    double ara,adec;
    double phi, phi_start, phi_end, phi_step;
    double dx,dy;
    double Rmax,Rthis,Rstart,Rend,Rstep;
    double zp_angle=100.;
    double num_curve_steps=20.;
    StatusType status = STATUS__OK;
    id=1;
    SdsFind(id, "fieldData", &idf, &status);
    ArgGetd(idf, "appRa", &cra, &status);
    ArgGetd(idf, "appDec", &cdec, &status);
    ArgGetd(idf, "configMjd", &mjd, &status);
    SdsFreeId(idf, &status);
    FpilModelRd2Xy(TelModel, cra, cdec, cra, cdec, mjd, &cxf, &cyf, &status);
    FpilModelXy2Pos(TelModel, cxf, cyf, &cxp, &cyp, &status);

    int crack_no_go_zone_start_vertices=2;
    /*
        * Calculating with RA/Dec means tha FPOSS applies field distortion corrctions
        * which means an object 11.2arcmin from the centre doesn't quite appear on
        * a circle drawn at 11.2, so we fudge the radii to make the shaded area
        * look correct.
    */
    double crack_no_go_zone_start[] = 
    {
        11.4, 121.5,
        14.25, 121.5,
    };
    int crack_no_go_zone_end_vertices=2;
    double crack_no_go_zone_end[] = 
    {
        14.1, 142.,
        11.3, 142.
    };
    /* Straight line at one end... */
    VIndex = 0;
    for (Vertex = 0; Vertex < crack_no_go_zone_start_vertices; Vertex++) {
        double XVertex;
        double YVertex;
        double XVertPos;
        double YVertPos;
        char Buffer[32];
        XVertex = crack_no_go_zone_start[VIndex++];
        YVertex = crack_no_go_zone_start[VIndex++];

        phi = -1.*(YVertex-zp_angle)/265. * 360. * Pi/180.;
        
        ara=cra+(XVertex/60./180.*Pi)*cos(phi);
        adec=cdec+(XVertex/60./180.*Pi)*sin(phi);
        FpilModelRd2Xy(TelModel, cra, cdec, ara, adec, mjd, &xf, &yf, &status);
        FpilModelXy2Pos(TelModel, xf, yf, &xp, &yp, &status);
        
        sprintf(Buffer, "%d %d ",
            FieldPlateToDispX(0, xp, Zoom),
            FieldPlateToDispY(0, yp, Zoom)
        );
        strcat(Polygon_crack, Buffer);
    }
    /* curve between ...*/
    phi_start=-1.*(crack_no_go_zone_start[3]-zp_angle)/265. * 360. * Pi/180.;
    phi_end  =-1.*(crack_no_go_zone_end[1]-zp_angle)/265. * 360. * Pi/180.;
    phi_step = (phi_end - phi_start)/num_curve_steps;
    phi = phi_start;
    Rstart=crack_no_go_zone_start[2];
    Rend=crack_no_go_zone_end[0];
    Rstep=(Rend-Rstart)/num_curve_steps;
    Rthis=Rstart;
    while (phi >= phi_end) {
        char Buffer[32];
        
        ara=cra+(Rthis/60./180.*Pi)*cos(phi);
        adec=cdec+(Rthis/60./180.*Pi)*sin(phi);
        FpilModelRd2Xy(TelModel, cra, cdec, ara, adec, mjd, &xf, &yf, &status);
        FpilModelXy2Pos(TelModel, xf, yf, &xp, &yp, &status);
        
        sprintf(Buffer, "%d %d ",
            FieldPlateToDispX(0, xp, Zoom),
            FieldPlateToDispY(0, yp, Zoom)
        );
        strcat(Polygon_crack, Buffer);
        phi += phi_step;
        Rthis += Rstep;
    }
    /* Straight line at other end... */
    VIndex = 0;
    for (Vertex = 0; Vertex < crack_no_go_zone_end_vertices; Vertex++) {
        double XVertex;
        double YVertex;
        double XVertPos;
        double YVertPos;
        char Buffer[32];
        XVertex = crack_no_go_zone_end[VIndex++];
        YVertex = crack_no_go_zone_end[VIndex++];

        phi = -1.*(YVertex-zp_angle)/265. * 360. * Pi/180.;
        
        ara=cra+(XVertex/60./180.*Pi)*cos(phi);
        adec=cdec+(XVertex/60./180.*Pi)*sin(phi);
        FpilModelRd2Xy(TelModel, cra, cdec, ara, adec, mjd, &xf, &yf, &status);
        FpilModelXy2Pos(TelModel, xf, yf, &xp, &yp, &status);
        
        sprintf(Buffer, "%d %d ",
            FieldPlateToDispX(0, xp, Zoom),
            FieldPlateToDispY(0, yp, Zoom)
        );
        strcat(Polygon_crack, Buffer);
    }
    /* curve between ...*/
    phi_start=-1.*(crack_no_go_zone_end[3]-zp_angle)/265. * 360. * Pi/180.;
    phi_end  =-1.*(crack_no_go_zone_start[1]-zp_angle)/265. * 360. * Pi/180.;
    phi_step = (phi_end - phi_start)/num_curve_steps;
    phi = phi_start;
    Rstart=crack_no_go_zone_end[2];
    Rend=crack_no_go_zone_start[0];
    Rstep=(Rend-Rstart)/num_curve_steps;
    Rthis=Rstart;
    while (phi <= phi_end) {
        char Buffer[32];
        
        ara=cra+(Rthis/60./180.*Pi)*cos(phi);
        adec=cdec+(Rthis/60./180.*Pi)*sin(phi);
        FpilModelRd2Xy(TelModel, cra, cdec, ara, adec, mjd, &xf, &yf, &status);
        FpilModelXy2Pos(TelModel, xf, yf, &xp, &yp, &status);
        
        sprintf(Buffer, "%d %d ",
            FieldPlateToDispX(0, xp, Zoom),
            FieldPlateToDispY(0, yp, Zoom)
        );
        strcat(Polygon_crack, Buffer);
        phi += phi_step;
        Rthis += Rstep;
    }
    Tcl_SetVar(interp, argv[6], Polygon_crack, 0);

    return TCL_OK;

}

/*  End of VLTDrawGuider */


/*  ------------------------------------------------------------------------- */

/*
 *                 V L T  D r a w  G u i d e r  O u t l i n e  { }
 */
 
static int VLTDrawGuiderOutline(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])
    
/*+
 *  Command Name:
 *    VLTDrawGuiderOutline
 *
 *  Function:
 *    Returns Tk canvas drawing commands to draw the VLT guide probe outline.
 *
 *  Call:
 *    VLTDrawGuiderOutline posn zoom pivot lines arc
 *
 *  Description: 
 *    Given a position string for the VLT guide probe (as generated by
 *    VLTGuider{}, this generates a series of parameters that can be used
 *    in Tk canvas drawing commands to draw the guide in the specified
 *    position, in outline, as opposed to as a filled item.
 *
 *  Parameters:
 *    (>)  posn (char)    String giving "XP YP XS YS" for guide
 *                        probe in its position, where XP,YP is
 *                        location of guide pivot and XS,YS is
 *                        the location of the star.
 *    (>)  zoom (float)   The current zoom factor.
 *    (<)  pivot (char)   Numeric parameters for an oval canvas
 *                        command that will draw a small circle
 *                        to indicate the position of the guide
 *                        probe pivot.
 *    (<)  lines (char)   Numeric parameters for a line canvas
 *                        command that will draw the remaining
 *                        part of the guide probe outline.
 *    (<)  arc (char)     Numeric parameters for an arc canvas
 *                        command that will draw the required arc of
 *                        the circular part of the guide probe.
 *
 *   Support:
 *     Keith Shortridge, AAO
 *
 *   Version Date:
 *     21-Aug-2000    
 *-
 */
{
    int XP, YP, XS, YS;
    int DeltaX, DeltaY;
    char Arc[64];
    char Oval[64];
    char Lines[512];
    char Lines_crack[2048];
    int VIndex;
    float Zoom;
    double Theta;
    double ThetaDegrees;
    int Vertex;
    double Pi = 3.14159;
    if (argc != 7) {
        interp->result = "wrong # args";
        return TCL_ERROR;
    }
    sscanf(argv[1], "%d %d %d %d", &XP, &YP, &XS, &YS);
    Zoom = atof(argv[2]);
    sprintf(Oval, "%d %d %d %d",
        FieldPlateToDispX(0, XP, Zoom) - 10,
        FieldPlateToDispY(0, YP, Zoom) - 10,
        FieldPlateToDispX(0, XP, Zoom) + 10,
        FieldPlateToDispY(0, YP, Zoom) + 10);
    Tcl_SetVar(interp, argv[3], Oval, 0);
    Lines[0] = '\0';
    
    /*  Theta is the angle between the positive X-axis and the line joining
     *  the center of the guide pivot to the guide star, measured in a 
     *  clockwise direction.
     */
     
    DeltaX = XS - XP;
    DeltaY = YS - YP;
    Theta = atan((double) abs(DeltaY) / (double) abs(DeltaX));
    if ((DeltaX > 0) && (DeltaY < 0)) {
        Theta = Theta;
    } else if ((DeltaX > 0) && (DeltaY > 0)) {
        Theta = (Pi * 2.0) - Theta;
    } else if ((DeltaX < 0) && (DeltaY > 0)) {
        Theta = Pi + Theta;
    } else if ((DeltaX < 0) && (DeltaY < 0)) {
        Theta = Pi - Theta;
    }
    VIndex = 0;
    
    /*  The set of parameters for the line command is generated in the
     *  same way as the polygon parameters generated by VLTDrawGuider(),
     *  but we omit the final closing line of the polygon, so we end
     *  up with an open line.
     */

    for (Vertex = 0; Vertex < GuideProbeVertices - 1; Vertex++) {
        int XVertex;
        int YVertex;
        int XVertPos;
        int YVertPos;
        char Buffer[32];
        XVertex = GuideProbeVertexValues[VIndex++];
        YVertex = GuideProbeVertexValues[VIndex++];
        XVertPos = (cos(Theta) * XVertex) + (sin(Theta) * YVertex)
            + XP;
        YVertPos = (cos(Theta) * YVertex) - (sin(Theta) * XVertex)
            + YP;
        sprintf(Buffer, "%d %d ", FieldPlateToDispX(0, XVertPos, Zoom),
            FieldPlateToDispY(0, YVertPos, Zoom));
        strcat(Lines, Buffer);
    }
    Tcl_SetVar(interp, argv[4], Lines, 0);
    
    /*  The parameters for the arc need to be based on the oval drawn
     *  by VLTDrawGuider(), but with a start and extent added. The start
     *  angle comes from the Theta calculated above, and the extent is 
     *  always going to be 180 degrees. We use -180 because the arc command
     *  convention is that the extent is measured in an anti-clockwise
     *  direction from the start angle. The start angle is the angle between
     *  the positive X-axis and the line joining the center of the arc to
     *  its start point and so is 270-Theta, in degrees.
     */
     
    ThetaDegrees = Theta * 180.0 / Pi;
    sprintf(Arc, "%d %d %d %d -start %f -extent %d",
        FieldPlateToDispX(0, XP - GuideProbeOvalRadius, Zoom),
        FieldPlateToDispY(0, YP - GuideProbeOvalRadius, Zoom),
        FieldPlateToDispX(0, XP + GuideProbeOvalRadius, Zoom),
        FieldPlateToDispY(0, YP + GuideProbeOvalRadius, Zoom),
        270.0 - ThetaDegrees, -180);
    Tcl_SetVar(interp, argv[5], Arc, 0);

    Lines_crack[0] = '\0';
    /* First figure out the radius via coordinates... */
    SdsIdType id, idf;
    double cra, cdec, mjd;
    double xp,yp,xf,yf;
    double cxp,cyp,cxf,cyf;
    double ara,adec;
    double phi, phi_start, phi_end, phi_step;
    double dx,dy;
    double Rmax,Rthis,Rstart,Rend,Rstep;
    double zp_angle=100.;
    double num_curve_steps=20.;
    StatusType status = STATUS__OK;
    id=1;
    SdsFind(id, "fieldData", &idf, &status);
    ArgGetd(idf, "appRa", &cra, &status);
    ArgGetd(idf, "appDec", &cdec, &status);
    ArgGetd(idf, "configMjd", &mjd, &status);
    SdsFreeId(idf, &status);
    FpilModelRd2Xy(TelModel, cra, cdec, cra, cdec, mjd, &cxf, &cyf, &status);
    FpilModelXy2Pos(TelModel, cxf, cyf, &cxp, &cyp, &status);

    int crack_no_go_zone_start_vertices=2;
    double crack_no_go_zone_start[] = 
    {
        11.4, 121.5,
        14.25, 121.5,
    };
    int crack_no_go_zone_end_vertices=2;
    double crack_no_go_zone_end[] = 
    {
        14.1, 142.,
        11.3, 142.
    };
    /* Straight line at one end... */
    VIndex = 0;
    for (Vertex = 0; Vertex < crack_no_go_zone_start_vertices; Vertex++) {
        double XVertex;
        double YVertex;
        double XVertPos;
        double YVertPos;
        char Buffer[32];
        XVertex = crack_no_go_zone_start[VIndex++];
        YVertex = crack_no_go_zone_start[VIndex++];

        phi = -1.*(YVertex-zp_angle)/265. * 360. * Pi/180.;
        
        ara=cra+(XVertex/60./180.*Pi)*cos(phi);
        adec=cdec+(XVertex/60./180.*Pi)*sin(phi);
        FpilModelRd2Xy(TelModel, cra, cdec, ara, adec, mjd, &xf, &yf, &status);
        FpilModelXy2Pos(TelModel, xf, yf, &xp, &yp, &status);
        /*
        xp=cxp+Rthis*cos(phi);
        yp=cyp+Rthis*sin(phi);
        */
        sprintf(Buffer, "%d %d ",
            FieldPlateToDispX_float(0, xp, Zoom),
            FieldPlateToDispY_float(0, yp, Zoom)
        );
        /*
         sprintf(Buffer, "%d %d ",
            xp,
            yp
        );
        */
       strcat(Lines_crack, Buffer);
    }
    /* curve between ...*/
    phi_start=-1.*(crack_no_go_zone_start[3]-zp_angle)/265. * 360. * Pi/180.;
    phi_end  =-1.*(crack_no_go_zone_end[1]-zp_angle)/265. * 360. * Pi/180.;
    phi_step = (phi_end - phi_start)/num_curve_steps;
    phi = phi_start;
    Rstart=crack_no_go_zone_start[2];
    Rend=crack_no_go_zone_end[0];
    Rstep=(Rend-Rstart)/num_curve_steps;
    Rthis=Rstart;
    while (phi >= phi_end) {
        char Buffer[32];
        
        ara=cra+(Rthis/60./180.*Pi)*cos(phi);
        adec=cdec+(Rthis/60./180.*Pi)*sin(phi);
        FpilModelRd2Xy(TelModel, cra, cdec, ara, adec, mjd, &xf, &yf, &status);
        FpilModelXy2Pos(TelModel, xf, yf, &xp, &yp, &status);
        
        sprintf(Buffer, "%d %d ",
            FieldPlateToDispX_float(0, xp, Zoom),
            FieldPlateToDispY_float(0, yp, Zoom)
        );
        strcat(Lines_crack, Buffer);
        phi += phi_step;
        Rthis += Rstep;
    }
    /* Straight line at other end... */
    VIndex = 0;
    for (Vertex = 0; Vertex < crack_no_go_zone_end_vertices; Vertex++) {
        double XVertex;
        double YVertex;
        double XVertPos;
        double YVertPos;
        char Buffer[32];
        XVertex = crack_no_go_zone_end[VIndex++];
        YVertex = crack_no_go_zone_end[VIndex++];

        phi = -1.*(YVertex-zp_angle)/265. * 360. * Pi/180.;
        
        ara=cra+(XVertex/60./180.*Pi)*cos(phi);
        adec=cdec+(XVertex/60./180.*Pi)*sin(phi);
        FpilModelRd2Xy(TelModel, cra, cdec, ara, adec, mjd, &xf, &yf, &status);
        FpilModelXy2Pos(TelModel, xf, yf, &xp, &yp, &status);
        
        sprintf(Buffer, "%d %d ",
            FieldPlateToDispX_float(0, xp, Zoom),
            FieldPlateToDispY_float(0, yp, Zoom)
        );
        strcat(Lines_crack, Buffer);
    }
    /* curve between ...*/
    phi_start=-1.*(crack_no_go_zone_end[3]-zp_angle)/265. * 360. * Pi/180.;
    phi_end  =-1.*(crack_no_go_zone_start[1]-zp_angle)/265. * 360. * Pi/180.;
    phi_step = (phi_end - phi_start)/num_curve_steps;
    phi = phi_start;
    Rstart=crack_no_go_zone_end[2];
    Rend=crack_no_go_zone_start[0];
    Rstep=(Rend-Rstart)/num_curve_steps;
    Rthis=Rstart;
    while (phi <= phi_end) {
        char Buffer[32];
        
        ara=cra+(Rthis/60./180.*Pi)*cos(phi);
        adec=cdec+(Rthis/60./180.*Pi)*sin(phi);
        FpilModelRd2Xy(TelModel, cra, cdec, ara, adec, mjd, &xf, &yf, &status);
        FpilModelXy2Pos(TelModel, xf, yf, &xp, &yp, &status);
        
        sprintf(Buffer, "%d %d ",
            FieldPlateToDispX_float(0, xp, Zoom),
            FieldPlateToDispY_float(0, yp, Zoom)
        );
        strcat(Lines_crack, Buffer);
        phi += phi_step;
        Rthis += Rstep;
    }
    /*
    char Buffer[32];
    sprintf(Buffer, "%d %d ",
        FieldPlateToDispX(0, cxp, Zoom),
        FieldPlateToDispY(0, cyp, Zoom)
    );
    strcat(Lines_crack, Buffer);

    phi = -1.*(crack_no_go_zone_end[1]-zp_angle)/265. * 360. * Pi/180.;
    ara=cra+(crack_no_go_zone_end[0]/60./180.*Pi)*cos(phi);
    adec=cdec+(crack_no_go_zone_end[0]/60./180.*Pi)*sin(phi);
    FpilModelRd2Xy(TelModel, cra, cdec, ara, adec, mjd, &xf, &yf, &status);
    FpilModelXy2Pos(TelModel, xf, yf, &xp, &yp, &status);
    sprintf(Buffer, "%d %d ",
        FieldPlateToDispX(0, xp, Zoom),
        FieldPlateToDispY(0, yp, Zoom)
    );
    strcat(Lines_crack, Buffer);
    */
    Tcl_SetVar(interp, argv[6], Lines_crack, 0);

    return TCL_OK;

}

/*  End of VLTDrawGuiderOutline */

/*  ------------------------------------------------------------------------  */

/*
 *                    V L T  G u i d e  S h a d d o w  { }
 */
 
static int VLTGuideShaddow(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])
    
/*+
 *  Command Name:
 *    VLTGuideShaddow
 *
 *  Function:
 *    Determines if a target is shaddowed by the VLT guide probe.
 *
 *  Call:
 *    VLTGuideShaddow id index posn
 *
 *  Description: 
 *    This routine is passed the details of a target star, in terms of the 
 *    SDS Id for a sub-structure of the configuration data  and an index.
 *    Given a position string for the VLT guide probe (as generated by
 *    VLTGuider{}), this routine determines whether or not the target will
 *    be sahddowed by the guide probe. If it is, it flags the target star
 *    as disabled.
 *
 *  Parameters:
 *    (>)  id    (Sds Id)  Sds identifier of the substructure
 *    (>)  index (int)     index number - starting at one
 *    (>)  posn  (char)    String giving "XP YP XS YS" for guide
 *                         probe in its position, where XP,YP is
 *                         location of guide pivot and XS,YS is
 *                         the location of the star. If all of
 *                         XP,YP,XS and YS are zero, then no guide
 *                         probe position has been selected.
 *   Returns:
 *     "OK" if the target is not shaddowed by the guide probe, "shaddowed"
 *     if it is.
 *
 *   Side effects:
 *     The type field for the target object in the Sds structure is set to 
 *     indicate that the target is shaddowed or not.
 *
 *   Support:
 *     Keith Shortridge, AAO
 *
 *   Version Date:
 *     17-Aug-1999    
 *-
 */
{

    int XP, YP, XS, YS;
    int X, Y;
    int Shaddowed;
    double Dx, Dy;
    double DRadius;
    double Dtheta;
    StatusType SdsStatus;
    SdsIdType TopId, Id;
    int IndexNo;
    double Pi = 3.14159;
    unsigned long Actlen;
    char Type;

    if (argc != 4) {
        interp->result = "wrong # args";
        return TCL_ERROR;
    }

    SdsStatus = SDS__OK;
    TopId = (SdsIdType) (atol(argv[1]));
    IndexNo = atol(argv[2]) - 1;
    SdsFind(TopId, "x", &Id, &SdsStatus);
    SdsGet(Id, sizeof(int), IndexNo, &X, &Actlen, &SdsStatus);
    SdsFreeId(Id, &SdsStatus);
    SdsFind(TopId, "y", &Id, &SdsStatus);
    SdsGet(Id, sizeof(int), IndexNo, &Y, &Actlen, &SdsStatus);
    SdsFreeId(Id, &SdsStatus);

    Shaddowed = FALSE;
    sscanf(argv[3], "%d %d %d %d", &XP, &YP, &XS, &YS);
    
    /*  Check for the special case where there is no guide pivot selected */
    
    if ((XP != 0) || (YP != 0) || (XS != 0) || (YS != 0)) {

        /*  Are we under the shaddow of the pivot circle? Are we closer
         *  to the pivot point than the circle radius, in other words?
         */
  
        Dx = (double) abs(X - XP);
        Dy = (double) abs(Y - YP);
        DRadius = (double) GuideProbeOvalRadius;
        if (((Dx * Dx) + (Dy * Dy)) <= (DRadius * DRadius)) {
            Shaddowed = TRUE;

        } else {

            /*  In 2022-11, a crack was discovered in the field correction lens.
             *  we can avoid allocating any stars in this affected region by setting them
             *  as shaddowed targets.
             * 
             *  As per https://pdm.eso.org/kronodoc/HQ/ESO-531457/1,
             *  The region is defined as:
             *    R > 11.2 arcmin
             *    122 <= position <= 147
            */

            /* First figure out the radius via coordinates... */
            SdsIdType id, idf;
            double cra, cdec, mjd;
            double xp,yp,xf,yf;
            double cxp,cyp,cxf,cyf;
            double ara,adec;
            double phi;
            double dx,dy;
            double Rmax,Rthis;
            StatusType status = STATUS__OK;
            id=1;
            SdsFind(id, "fieldData", &idf, &status);
            ArgGetd(idf, "appRa", &cra, &status);
            ArgGetd(idf, "appDec", &cdec, &status);
            ArgGetd(idf, "configMjd", &mjd, &status);
            SdsFreeId(idf, &status);
            ara=cra;
            adec=cdec+11.2/60./180.*Pi;
            FpilModelRd2Xy(TelModel, cra, cdec, cra, cdec, mjd, &cxf, &cyf, &status);
            FpilModelXy2Pos(TelModel, cxf, cyf, &cxp, &cyp, &status);
            FpilModelRd2Xy(TelModel, cra, cdec, ara, adec, mjd, &xf, &yf, &status);
            FpilModelXy2Pos(TelModel, xf, yf, &xp, &yp, &status);

            /* Then calculate the angle subtended by the crack... */
            /*
                * The crack runs from buttons 122 to 142
                * there are 265 buttons in total more or less evently arranged over
                * 360 degrees, so the angle subtended is
                *     (142. - 122.)/265 * 360. * PI/180.
                * This is not quite exact, so we 122. --> 121.5 to insure button 122
                * is in the exclusion zone.
            */
            double angle_start     = 121.5/265. * 360. * Pi/180.;
            double angle_subtended = (142.-121.5)/265. * 360. * Pi/180.;

            dx=(X-cxp);
            dy=(Y-cyp);
            Rmax=(xp-cxp)*(xp-cxp)+(yp-cyp)*(yp-cyp);
            Rthis=dx*dx+dy*dy;

            /* then the angle of the object relative to the cetral coordinate... */
            phi = atan((double) abs(dy) / (double) abs(dx));
            if ((dx > 0) && (dy < 0))
                phi = phi;
            if ((dx > 0) && (dy > 0))
                phi = (Pi * 2.0) - phi;
            if ((dx < 0) && (dy > 0))
                phi = Pi + phi;
            if ((dx < 0) && (dy < 0))
                phi = Pi - phi;

            if (
                (Rthis > Rmax)
                &&
                (
                    (angle_start < phi)
                    &&
                    (phi < angle_start + angle_subtended )
                )
            ) {
            /*
                * For testing it is easier to consider the fibres In the zone...
            */
            /*
            if (
                (Rthis < Rmax)
                ||
                (
                    (angle_start > phi)
                    ||
                    (phi > angle_start + angle_subtended )
                )
            ) {
            */
                Shaddowed = TRUE;
            } else {

                /*  Now, we split the polygon describing the rest of the shaddow
                *  into a series of lines and see if the line joining our target
                *  to the point at the center of the shaddow crosses any of
                *  them. If it does, the target is outside the shaddow. (This
                *  isn't a valid general-purpose way of doing this sort of 
                *  'inside' analysis - we're making use of some knowledge about
                *  the symmetry and regularity of the shaddow.)
                */

                double Theta;
                double DeltaX, DeltaY;
                double XCent;
                double XCentPos, YCentPos;
                int VIndex;
                int Vertex;
                int Outside;

                DeltaX = XS - XP;
                DeltaY = YS - YP;
                Theta = atan((double) abs(DeltaY) / (double) abs(DeltaX));
                if ((DeltaX > 0) && (DeltaY < 0))
                    Theta = Theta;
                if ((DeltaX > 0) && (DeltaY > 0))
                    Theta = (Pi * 2.0) - Theta;
                if ((DeltaX < 0) && (DeltaY > 0))
                    Theta = Pi + Theta;
                if ((DeltaX < 0) && (DeltaY < 0))
                    Theta = Pi - Theta;

                XCent = GuideProbeLength / 2;
                XCentPos = (cos(Theta) * XCent) + XP;
                YCentPos = -(sin(Theta) * XCent) + YP;
                VIndex = 0;
                Outside = FALSE;
                for (Vertex = 0; Vertex < (GuideProbeVertices - 1); Vertex++) {
                    int X1, Y1, X2, Y2;
                    double X1Pos, Y1Pos, X2Pos, Y2Pos;
                    X1 = GuideProbeVertexValues[VIndex++];
                    Y1 = GuideProbeVertexValues[VIndex++];
                    X2 = GuideProbeVertexValues[VIndex];
                    Y2 = GuideProbeVertexValues[VIndex + 1];
                    X1Pos = (cos(Theta) * X1) + (sin(Theta) * Y1) + XP;
                    Y1Pos = (cos(Theta) * Y1) - (sin(Theta) * X1) + YP;
                    X2Pos = (cos(Theta) * X2) + (sin(Theta) * Y2) + XP;
                    Y2Pos = (cos(Theta) * Y2) - (sin(Theta) * X2) + YP;
                    if (LinesIntersect((double) X, (double) Y, XCentPos, YCentPos,
                            X1Pos, Y1Pos, X2Pos, Y2Pos)) {
                        Outside = TRUE;
                        break;
                    }
                }
                Shaddowed = !Outside;
            }

        }
    }
    
    /*  We make sure that the target type flag reflects its current
     *  shaddowed state - shaddowed or not, as the case may be - and we
     *  set our return value accordingly.
     */
     
    SdsFind(TopId, "type", &Id, &SdsStatus);
    SdsGet(Id, sizeof(char), IndexNo, &Type, &Actlen, &SdsStatus);
    if (Shaddowed) {
        Tcl_SetResult(interp, "shaddowed", TCL_STATIC);
        Type = FpilDisabledTargetType(*Instrument, Type);
    } else {
        Tcl_SetResult(interp, "OK", TCL_STATIC);
        Type = FpilEnabledTargetType(*Instrument, Type);
    }
    SdsPut(Id, sizeof(char), IndexNo, &Type, &SdsStatus);
    SdsFreeId(Id, &SdsStatus);
    
    return (TCL_OK);
}

/*  ------------------------------------------------------------------------- */

/*
 *                   W a v e l e n g t h  C o m b o s  { }
 */

static int ConfWavelengthCombos(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc DUNUSED, char *argv[]DUNUSED)

/*+
 *  Command Name:
 *    WavelengthCombos
 *
 *  Function:
 *    Returns the set of possible wavelength combinations.
 *
 *  Call:
 *    WavelengthCombos
 *
 *  Description:
 *    Returns a list giving the names of the the possible wavelength 
 *    combinations supported by the instrument given the current fibre
 *    combination. The current fibre combination should have been set
 *    using SetFibreCombo{}.
 *
 *  Parameters: none
 *
 *   Support:
 *     Keith Shortridge, AAO
 *
 *   Version Date:
 *     25-Mar-2002
 *-
 */
{

    int NumCombos;
    char* ComboNames[16];
    register unsigned i;
    Tcl_DString Names;

    if (Instrument == 0) {
        sprintf(interp->result, "%s:Configure not yet initialised", argv[0]);
        return TCL_ERROR;
    }

    FpilWavelengthCombos(*Instrument, sizeof(ComboNames)/sizeof(char*),
                                                   &NumCombos, ComboNames);

    Tcl_DStringInit(&Names);
    for (i = 0; i < NumCombos; ++i) {
        Tcl_DStringAppendElement(&Names, ComboNames[i]);
    }
    Tcl_DStringResult(interp, &Names);
    Tcl_DStringFree(&Names);

    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*
 *                    S e t  O b s  W a v e l e n g t h  { }
 */
 
static int ConfSetObsWavelength(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])

/*+
 *  Command Name:
 *    SetObsWavelength
 *
 *  Function:
 *    Sets the current observing wavelength, for astrometric purposes.
 *
 *  Call:
 *    SetObsWavelength Wavelength
 *
 *  Description:
 *    This routine sets the current observing wavelength. This will be
 *    used by the Fpil routines that return the wavelength to be used
 *    by different target types for different wavelength combinations.
 *    This routine provides a direct Tcl interface to FpilSetObsWavelength().
 *
 *  Parameters:
 *    (>)  Wavelength  (double)  The main observing wavelength.
 *
 *   Support:
 *     Keith Shortridge, AAO
 *
 *   Version Date:
 *     21st Mar 2002.  KS  Original version.  
 *-
 */
{
    double Wavelength;
       
    if (argc != 2) {
        interp->result = "wrong # args";
        return TCL_ERROR;
    }
    Wavelength = atof(argv[1]);
    FpilSetObsWavelength(*Instrument,Wavelength);
    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*
 *                    S e t  F i b r e  C o m b o  { }
 */
 
static int ConfSetFibreCombo(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])

/*+
 *  Command Name:
 *    SetFibreCombo
 *
 *  Function:
 *    Sets the current fibre combination, for astrometric purposes.
 *
 *  Call:
 *    SetFibreCombo FibreComboId 
 *
 *  Description:
 *    This routine sets the current fibre combination. This will be
 *    used by the Fpil routines that return the wavelength to be used
 *    by different target types for different wavelength combinations.
 *    (It may not seem obvious that the fibre combination is relevant
 *    here, but in fact which combination of fibres is in use affects
 *    the number of possible wavelength combinations - more wavelength
 *    combinations are possible if there are fibres in use that are fed
 *    to more than one spectrograph.) This routine provides a direct
 *    Tcl interface to FpilSetFibreCombo().
 *
 *  Parameters:
 *    (>)  FibreComboId  (int)  The fibre combination id.
 *
 *   Support:
 *     Keith Shortridge, AAO
 *
 *   Version Date:
 *     21st Mar 2002.  KS  Original version.  
 *-
 */
{
    int FibreComboId;
       
    if (argc != 2) {
        interp->result = "wrong # args";
        return TCL_ERROR;
    }
    FibreComboId = atoi(argv[1]);
    FpilSetFibreCombo(*Instrument,FibreComboId);
    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*
 *               S e t  W a v e l e n g t h  C o m b o  { }
 */
 
static int ConfSetWavelengthCombo(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])

/*+
 *  Command Name:
 *    SetWavelengthCombo
 *
 *  Function:
 *    Sets the current wavelength combination, for astrometric purposes.
 *
 *  Call:
 *    SetWavelengthCombo WavelengthComboId 
 *
 *  Description:
 *    This routine sets the current fibre combination. This will be
 *    used by the Fpil routines that return the wavelength to be used
 *    by different target types for different wavelength combinations.
 *    This routine provides a direct Tcl interface to FpilSetWavelengthCombo().
 *
 *  Parameters:
 *    (>)  WavelengthComboId  (int)  The wavelength combination id.
 *
 *   Support:
 *     Keith Shortridge, AAO
 *
 *   Version Date:
 *     21st Mar 2002.  KS  Original version.  
 *-
 */
{
    int WavelengthComboId;
       
    if (argc != 2) {
        interp->result = "wrong # args";
        return TCL_ERROR;
    }
    WavelengthComboId = atoi(argv[1]);
    FpilSetWavelengthCombo(*Instrument,WavelengthComboId);
    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*                       A r g u s   R o u t i n e s
 *
 *   ARGUS is part of the FLAMES instrument, but can be treated as a general
 *   facility for the purposes of this code - for any instrument other than
 *   FLAMES, we simply set it as not in use.  It is specified, for our purposes
 *   here, by three quantities: an 'in use' flag; a position angle in degrees;
 *   and a scale which is one of '1:1' or '1:1.67' and which we maintain as
 *   a flag set for 1:1. The only real interraction with this code, other
 *   than the need to maintain the values and make them available to the Tcl
 *   layer, is that when the position angle is changed, we need to invoke
 *   the FpilSetPositionAngle() routine.
 *
 *   The ARGUS values are maintained by the C layer, which keeps them in the
 *   three global variables defined here. They may be set by the Tcl layer,
 *   as the result of an ARGUS settings dialogue, or by the C layer when
 *   a file is read in. So we provide both Tcl and C interfaces to these
 *   three global variables.
 *
 *   Note that the variable here called ArgusAngle is the value that is 
 *   used as the first Telescope model parameter for FLAMES (see JAB's
 *   document on the OzPoz Astrometric Model). This is the what the
 *   astrometric code calls the position angle for the observation. 
 *   This angle, which we call Aa ('a' for astrometric) is defined as
 *   follows:  (Sorry this is so long, but it took a while to get this
 *   straight during commissioning, and it's worth recording the details.)
 *
 *   The positioner is an R-theta system, with theta being zero for the line
 *   from the center of the plate to retractor 1, which defines the +ve X-axis
 *   of the plate, and 90 degrees for the line from the center of the plate
 *   to to (roughly) retractor 68, which is more or less the +ve Y-axis of the
 *   plate. 
 *   
 *   The job of the telescope software is to use the rotator so that a 
 *   specified angle on the plate aligns with North, and to keep it
 *   that way throughout the exposure. Loosely, we think of a point on the
 *   circumfrence of the plate being made to point North. A position 
 *   angle is passed to the telescope control software, which we call Ap ('p'
 *   for PAF, because this is the angle written into the PAF file as
 *   INS.ARGS.ANGLE. If this is zero, a specific point on the plate circumfrence
 *   will be made to point North. Call this point the 'zero position angle
 *   reference point'. In general, this point will be made to point Ap degrees
 *   from North, in the sense NESW. The original assumption when the astrometric
 *   code was written was that this point would correspond to the +ve Y-axis of
 *   the plate. So the angle that we call Aa here (sometimes called ypa in the
 *   astrometric model documentation, on the assumption that it would be the
 *   position angle of the y-axis) was originally defined so that when it was
 *   zero, the +ve Y axis (~fibre 64) would point North, and when it was 90
 *   degrees the +ve Y axis would point East.
 *   
 *   However, because of considerations to do with providing the maximum
 *   possible duration for an observation, the final choice of the zero
 *   position angle reference point was the line at theta = 45 degrees,
 *   roughly corresponding to fibre 34. This choice was implemented in the
 *   astrometric code not by changing the definition of Aa, which might
 *   have been more logical (but was a more complex software change) but
 *   by treating this as a field rotation of 45 degrees. The effect is
 *   that Aa becomes the position angle (the angle from North, in the sense
 *   NESW) of the line joining the field center to the home position of fibre
 *   34. (More strictly, the zero position angle reference point.)
 *   
 *   This is actually quite sensible. The result is that both Aa and Ap are
 *   the same, and correspond to the position angle of the line from the 
 *   center of the field to the zero position angle reference point (roughly
 *   fibre 34). The fact that this point does not correspond to the natural
 *   axes of the positioner, or to the coordinate system used by FPOSS, is
 *   handled by the introduction of a field rotation of 45 degrees when
 *   converting into plate coordinates.
 *   
 *   Now, when the Observer specifies that she wants ARGUS to be at a 
 *   position angle Ao (the value she enters in the GUI), she wants
 *   the long axis of ARGUS to point Ao degrees from North, in the sense NESW.
 *   Or, putting it the other way round, if the long axis of ARGUS points Ao 
 *   degrees from North in the sense NESW, the observer will say that the
 *   Argus position angle is Ao. ARGUS itself is mounted at an essentially
 *   arbitrary angle on the field plate, however, and we have to allow for
 *   
 *   that. 
 *   
 *   Define Am to be the angle from fibre 34 (the zero position reference
 *   point) to the long axis of ARGUS in the sense NESW. Now, given this, 
 *   it's easy. When the system applies a position angle Ap, fibre 34 will
 *   point Ap degrees from North, and the long axis of ARGUS will point a
 *   further Am degrees from North.  So the angle that Argus points
 *   from North in the sense NESW, which in the previous paragraph IS the
 *   ARGUS position angle, Ao, is given by:
 *   
 *      Ao = Ap + Am
 *   
 *   The Tcl code used by FPOSS calculates (this was written some time before
 *   all the details of the field rotation etc were known, just on the
 *   assumption that there would be some sort of offset involved)
 *    
 *      Ao = Aa - ArgusOffset
 *      
 *   From which it follows that the value defined here for ArgusOffset should
 *   be -Am, with Am defined as above. As it happens, Am turns out to be 90
 *   degrees, hence the value included here.
 *      
 */
 
static int ArgusInUse = 0;                 /* If non-zero, Argus is in use */
static double ArgusAngle = 0.0;            /* Argus angle in degrees */
static int ArgusScale1to1 = 1;             /* If true, 1:1, if false, 1:1.67 */
static double ArgusOffset = -90.0;         /* Argus offset in degrees */

/*  ------------------------------------------------------------------------- */

/*
 *                    G e t  A r g u s  D a t a  { }
 */
 
static int ConfGetArgusData(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])

/*+
 *  Command Name:
 *    GetArgusData
 *
 *  Function:
 *    Get all the information about the current ARGUS state.
 *
 *  Call:
 *    GetArgusData InUse Angle Scale Offset
 *
 *  Description:
 *    This routine returns all the information known to the system about the
 *    ARGUS settings.
 *
 *  Parameters:
 *    (<)  InUse  (Tcl int variable)     Set to 1 if in use, 0 if not.
 *    (<)  Angle  (Tcl double variable)  Set to the argus angle, in degrees.
 *    (<)  Scale  (Tcl string variable)  Set to the current scale in use - a 
 *                                       string, either '1:1' or '1:1.67'
 *    (<)  Offset (Tcl double variable)  Set to the argus offset, in degrees.
 *
 *   Support:
 *     Keith Shortridge, AAO
 *
 *   Version Date:
 *     15-Aug-2001  KS  Original version. 
 *     23-Aug-2001  KS  Added Offset.
 *-
 */
{
    char buffer[80];
    int IntAngle;

    if (argc != 5) {
        interp->result = "wrong # args";
        return TCL_ERROR;
    }
    if (ArgusInUse) {
        Tcl_SetVar(interp, argv[1], "1", 0);
    } else {
        Tcl_SetVar(interp, argv[1], "0", 0);
    }
    IntAngle = (int)ArgusAngle;
    if ((float)IntAngle == ArgusAngle) {  
       sprintf(buffer, "%d", IntAngle);
    } else {
       sprintf(buffer, "%f", ArgusAngle);
    }
    Tcl_SetVar(interp, argv[2], buffer, 0);
    if (ArgusScale1to1) {
       Tcl_SetVar(interp, argv[3], "1:1", 0);
    } else {
       Tcl_SetVar(interp, argv[3], "1:1.67", 0);
    }
    IntAngle = (int)ArgusOffset;
    if ((float)IntAngle == ArgusOffset) {  
       sprintf(buffer, "%d", IntAngle);
    } else {
       sprintf(buffer, "%f", ArgusOffset);
    }
    Tcl_SetVar(interp, argv[4], buffer, 0);
    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*
 *                    S e t  A r g u s  D a t a  { }
 */
 
static int ConfSetArgusData(ClientData clientData DUNUSED,
    Tcl_Interp * interp, int argc, char *argv[])

/*+
 *  Command Name:
 *    SetArgusData
 *
 *  Function:
 *    Sets all the information about the current ARGUS state.
 *
 *  Call:
 *    SetArgusData InUse Angle Scale Offset
 *
 *  Description:
 *    This routine sets all the information needed by the system about the
 *    ARGUS settings.
 *
 *  Parameters:
 *    (>)  InUse  (int)     1 if in use, 0 if not.
 *    (>)  Angle  (double)  The argus angle, in degrees.
 *    (>)  Scale  (string)  The current scale in use - a  string, either '1:1'
 *                          or '1:1.67'
 *    (>)  Offset (double)  The argus offset, in degrees.
 *
 *   Support:
 *     Keith Shortridge, AAO
 *
 *   Version Date:
 *     15-Aug-2001  KS  Original version.  
 *-
 */
{
    int InUse;
    double Angle;
    char* Scale;
    double Offset;
    
    if (argc != 5) {
        interp->result = "wrong # args";
        return TCL_ERROR;
    }
    InUse = atoi(argv[1]);
    Angle = atof(argv[2]);
    Scale = argv[3];
    Offset = atof(argv[4]);
    ConfCSetArgusData(InUse,Angle,Scale,Offset);
    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */
 
/*+
 *                    C o n f  C  S e t  A r g u s  D a t a
 *
 *  Routine:
 *    ConfCSetArgusData
 *
 *  Function:
 *    Sets all the information about the current ARGUS state.
 *
 *  Call:
 *    ConfCSetArgusData(InUse,Angle,Scale,Offset);
 *
 *  Description:
 *    This routine provides a C interface to set all the information needed by
 *    the system about the ARGUS settings.  If the angle specified has changed,
 *    it calls FpilSetPositionAngle() to make the new angle effective. If
 *    ARGUS is not in use, then it calls FpilSetPositionAngle() to set up
 *    the default position angle.
 *
 *  Parameters:
 *    (>)  InUse  (short)   1 if in use, 0 if not.
 *    (>)  Angle  (double)  The argus angle, in degrees.
 *    (>)  Scale  (char*)   The current scale in use - a  string, either '1:1'
 *                          or '1:1.67'
 *    (>)  Offset (double)  The argus offset, in degrees.
 *
 *   Support:
 *     Keith Shortridge, AAO
 *
 *   Version Date:
 *     15-Aug-2001  KS  Original version.  
 *     23-Aug-2001  KS  Added Offset.
 *-
 */
 
void ConfCSetArgusData(
    short InUse,
    double Angle,
    char* Scale,
    double Offset)
{
    double Radians;
    short UseDefault;
    
    /*  If the angle has changed, or if we have switched between having
     *  ARGUS in use and not in use, we need to set the position angle for
     *  the system.
     */
     
    ArgusScale1to1 = (!strcmp(Scale,"1:1"));
    if (InUse) {
        if ((ArgusAngle != Angle) || !ArgusInUse) {
            Radians = Angle * 3.14159265358979323846 / 180.0;
            UseDefault = 0;
            FpilSetPositionAngle(*Instrument,UseDefault,Radians,TelModelParams);
        }
    } else {
        if (ArgusInUse) {
            UseDefault = 1;
            FpilSetPositionAngle(*Instrument,UseDefault,0.0,TelModelParams);
         }
    }
    ArgusAngle = Angle;
    ArgusInUse = InUse;
    ArgusOffset = Offset;
}

/*  ------------------------------------------------------------------------- */

/*+
 *                    C o n f  C  G e t  A r g u s  D a t a
 *
 *  Routine:
 *    ConfCGetArgusData
 *
 *  Function:
 *    Gets all the information about the current ARGUS state.
 *
 *  Call:
 *    ConfGetCArgusData(InUse,Angle,Scale,Offset);
 *
 *  Description:
 *    This routine provides a C interface to get all the information needed by
 *    the system about the ARGUS settings.
 *
 *  Parameters:
 *    (<)  InUse  (short*)   1 if in use, 0 if not.
 *    (<)  Angle  (double*)  The argus angle, in degrees.
 *    (<)  Scale  (char*)    The current scale in use - a  string, either '1:1'
 *                           or '1:1.67'  Must be at least 7 characters long.
 *    (<)  Offset (double)   The argus offset, in degrees.
 *
 *   Support:
 *     Keith Shortridge, AAO
 *
 *   Version Date:
 *     15-Aug-2001  KS  Original version.  
 *     23-Aug-2001  KS  Added Offset.
 *-
 */
 
void ConfCGetArgusData(
    short* InUse,
    double* Angle,
    char* Scale,
    double* Offset)
{    
    *InUse = ArgusInUse;
    if (ArgusScale1to1) {
        strcpy(Scale,"1:1");
    } else {
        strcpy(Scale,"1:1.67");
    }
    *Angle = ArgusAngle;
    *Offset = ArgusOffset;
}

/*  ------------------------------------------------------------------------- */

/*
 *                    C o n f i g  V e r  N u m  { }
 */
 
static int ConfigVerNum(ClientData clientdata DUNUSED, Tcl_Interp * interp,
    int argc DUNUSED, char *argv[]DUNUSED)
    
/*+
 *  Command Name:
 *    ConfigVerNum
 *
 *  Function:
 *    Returns the current version number for the configuration program.
 *
 *  Call:
 *    ConfigVerNum
 *
 *  Description:
 *    This routine returns the current version number for the configuration
 *    program. Note that this is obtained from configversion.c.
 *
 *  Parameters:  None.
 *
 *  Returns:
 *    The version number, as a string.
 *-
 */
{
    char *cp;

    if (strlen(ConfigVersion) <= 13)
        interp->result = "unknown";
    else
    	{
	strcpy(interp->result, ConfigVersion + 11);
	cp = interp->result;
	*(cp + strlen(cp) - 2) = '\0';
	}
    return TCL_OK;
}

/*  ------------------------------------------------------------------------- */

/*                           B i t m a p s
 *
 *  This section defines a couple of specialised bitmaps used by the 
 *  Tk interface to the configuration program. They are passed to the
 *  Tcl/Tk system during execution of ConfAddCommands().
 */
 
/*  Definition of the arrow bitmap used for highlighting */

#define arrow_width 16
#define arrow_height 16
static const char arrow_bits[] = {
   0x00, 0xc0, 0x00, 0xe0, 0x00, 0xf0, 0x04, 0x78, 0x0c, 0x3c, 0x0c, 0x1e,
   0x0c, 0x0f, 0x8c, 0x07, 0xcc, 0x03, 0xec, 0x01, 0xfc, 0x00, 0x7c, 0x00,
   0xfc, 0x0f, 0xfc, 0x1f, 0x00, 0x00, 0x00, 0x00};

/*  And of the tick bitmap used in the handholder pane */

#define tick_width 16
#define tick_height 16
static const char tick_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x00, 0x30, 0x00, 0x18, 0x00, 0x0c,
   0x00, 0x06, 0x00, 0x03, 0x80, 0x01, 0xc0, 0x00, 0x62, 0x00, 0x36, 0x00,
   0x1c, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00};

/*  ------------------------------------------------------------------------- */

/*+
 *                    C o n f  A d d  C o m m a n d s
 *
 *  Initialise this application into Tcl, adding all the required commands.
 *  This routine is set up by main() so that it is called during Tcl/Tk
 *  initialisation. It defines all the Tcl/Tk extensions used by the
 *  configuration program - the C-implemented additional Tcl commands, the
 *  bitmaps used, and the X-error handler.
 */
 
static int ConfAddCommands(Tcl_Interp * interp)
 {
    Tk_Window win;

    if (Tcl_Init(interp) == TCL_ERROR) {
        return TCL_ERROR;
    }
    if (!batchMode) {
        if (Tk_Init(interp) == TCL_ERROR) {
            return TCL_ERROR;
        }
        Tcl_StaticPackage(interp, "Tk", Tk_Init, Tk_SafeInit);
    }

#ifdef macintosh
    SetupMainInterp(interp);
    Tcl_SetVar(interp, "tcl_rcFileName", CONF_TCLFILE, TCL_GLOBAL_ONLY);
#endif


    Sds_Init(interp);
    Tcl_CreateCommand(interp, "GetFibre", ConfGetFibre, (ClientData *) NULL,
        (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "GetButton", ConfGetButton, (ClientData *) NULL,
        (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "GetObject", ConfGetObject, (ClientData *) NULL,
        (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "GetObjectData", ConfGetObjectData,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "SetFibre", ConfSetFibre, (ClientData *) NULL,
        (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "SetButtonAngle", ConfSetButtonAngle,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "ConvertAscii", ConfConvertAscii,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "CountAsciiObjects", ConfCountAsciiObjects,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "SetUTMeridian", ConfSetUTMeridian,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "ConvertXy", ConfConvertXy,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "ParamsChanged", ConfParamsChanged,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "Ra2string", ConfRa2string,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "Dec2string", ConfDec2string,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "Mjd2date", ConfMjd2Date,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "Mjd2time", ConfMjd2Time,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "RaDec2Ha", ConfRaDec2Ha,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "RaDec2Zd", ConfRaDec2Zd,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "MakeAllocated", ConfMakeAllocated,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "UpdateFibreStats", ConfUpdateFibreStats,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "GetFibreStats", ConfGetFibreStats,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "SetSkyCounts", ConfSetSkyCounts,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "AddIndex", ConfAddIndex,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "AllocateFibre", ConfAllocateFibre,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "PivotInUse", ConfPivotInUse,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "MarkFibreType", ConfMarkFibreType,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "DeallocateFibre", ConfDeallocateFibre,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "SdsFileName", ConfFileName,
        (ClientData *) ".sds", (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "NewSdsFileName", ConfFileName,
        (ClientData *) "_new.sds", (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "AsciiFileName", ConfFileName,
        (ClientData *) ".fld", (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "ImportFileName", ConfFileName,
        (ClientData *) ".imp", (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "DSSFileName", ConfFileName,
        (ClientData *) ".dss", (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "CompressUnalloc", ConfCompressUnalloc,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "StartUpdate", ConfStartUpdate,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "DoAllocation", ConfDoAllocation,
        (ClientData *) ALLOCATE_NORMAL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "DoAllocationBatch", ConfDoAllocation,
        (ClientData *) ALLOCATE_BATCH, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "DoCheck", ConfDoCheck,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "DoCheck1", ConfDoCheck1,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "ParameterDetails", ConfParameterDetails,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "NumberParameters", ConfNumberParameters,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "SetParameter", ConfSetParameter,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "GetParameter", ConfGetParameter,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "SetUT", ConfSetUT,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "NextDay", ConfNextDay,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "MakeListing", ConfMakeListing,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "SkycatListing", ConfSkycatListing,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "ClassifyFile", ConfClassifyFile,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "WritePAF", ConfWritePAF,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "PAFFileName", ConfPAFFileName,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "ReadPAF", ConfReadPAF,
        (ClientData *) ".ins", (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "MakeSkyvals", ConfMakeSkyvals,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "MakeDSSListing", ConfMakeDSSListing,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "ListFileName", ConfFileName,
        (ClientData *) ".lis", (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "PivotsId", ConfPivotsId,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "Initialise", ConfInitialise,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "Instruments", ConfInstruments,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "Instrument", ConfInstrument,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "NumPivots", ConfNumPivots,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "NumFields", ConfNumFields,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "FibreTypes", ConfFibreTypes,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "TargetDescr", ConfTargetDescr,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "TargetCriteria", ConfTargetCriteria,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "FibreCombos", ConfFibreCombos,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "FibreComboMode", ConfFibreComboMode,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "FibreComboTypes", ConfFibreComboTypes,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "Telescope", ConfTelescope,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "GetWavelength", ConfGetWavelength,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "SetWavelength", ConfSetWavelength,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "GetRefPars", ConfGetRefPars,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "SetRefPars", ConfSetRefPars,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "GetTelPars", ConfGetTelPars,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "SetTelPars", ConfSetTelPars,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "GetPlate", ConfGetPlate,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "SetPlate", ConfSetPlate,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "GetType", ConfGetType,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "ReallyExists", ConfReallyExists,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "AddSkyPosition", ConfAddSkyPosition,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "SetSkyGrid", ConfSetSkyGrid,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "SetImportFile", ConfSetImportFile,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "SetImportOnlyMode", ConfSetImportOnlyMode,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "TranslateName", TranslateName,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "AllocToUnalloc", ConfAllocToUnalloc,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "ExeDir", ExeDir,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "ErsFlush", ConfigErsFlush,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "UnallocatedType", UnallocatedType,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "TypeIsTelGuide", TypeIsTelGuide,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "TypeIsGuide", TypeIsGuide,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "TypeIsSky", TypeIsSky,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "PlateXtoDispX", PlateXtoDispX,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "PlateYtoDispY", PlateYtoDispY,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "DispXtoPlateX", DispXtoPlateX,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "DispYtoPlateY", DispYtoPlateY,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "PlateXYtoRaDec", PlateXYtoRaDec,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "RaDecToPlateXY", RaDecToPlateXY,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "MeanToAppRaDec", MeanToAppRaDec,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);

    Tcl_CreateCommand(interp, "ConfigVerNum", ConfigVerNum,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);

    Tcl_CreateCommand(interp, "UseGrey", ConfUseGrey,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "SetGrey", ConfSetGrey,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "ExpertAllowed", ConfExpertAllowed,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);

    Tcl_CreateCommand(interp, "LeakPrep", TclLeakPrep,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "LeakCheck", TclLeakCheck,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);


    Tcl_CreateCommand(interp, "FixBreakages", ConfFixBreakages,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "RemoveSky", ConfRemoveSky,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);

    Tcl_CreateCommand(interp, "Mimic", ConfMimic,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);

    Tcl_CreateCommand(interp, "CheckTargetCriteria", ConfCheckTargetCriteria,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);

    Tcl_CreateCommand(interp, "InvokeDebugger", ConfInvokeDebugger,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);


    /* VLTGuider commands  */
    
    Tcl_CreateCommand(interp, "VLTGuider", VLTGuider,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "VLTDrawGuider", VLTDrawGuider,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "VLTDrawGuiderOutline", VLTDrawGuiderOutline,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "VLTGuideShaddow", VLTGuideShaddow,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
        
    /* ARGUS commands  */
    
    Tcl_CreateCommand(interp, "SetArgusData", ConfSetArgusData,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "GetArgusData", ConfGetArgusData,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
        
    /*  Commands used by the wavelength combination stuff */
    
    Tcl_CreateCommand(interp, "WavelengthCombos", ConfWavelengthCombos,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "SetObsWavelength", ConfSetObsWavelength,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "SetWavelengthCombo", ConfSetWavelengthCombo,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "SetFibreCombo", ConfSetFibreCombo,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);
        
    /* Checksum commands */
    Tcl_CreateCommand(interp, "PafChkVerify", PafChkVerify,
        (ClientData *) NULL, (Tcl_CmdDeleteProc *) NULL);


    /*
     *  Setup an X error handler, and define any bitmaps that are used.
     *  Note that batchMode is a global set by main().
     */
     
    if (!batchMode) {
        win = Tk_MainWindow(interp);
        Tk_CreateErrorHandler(Tk_Display(win), -1, -1, -1, XError,
                                                       (ClientData) interp);
        Tk_DefineBitmap(interp, Tk_GetUid("arrow"), arrow_bits,
            arrow_width, arrow_height);
        Tk_DefineBitmap(interp, Tk_GetUid("tick"), tick_bits,
            tick_width, tick_height);
    }

    return TCL_OK;

}


/* -------------------------------------------------------------------------- */

/*
 *                   L i n e s   I n t e r s e c t
 *  Name:
 *     LinesIntersect
 *
 *  Function:
 *     Determine if two lines intersect.
 *
 *  Description:
 *     This routine is passed the end coordinates of a pair of lines,
 *     and returns true or false depending on whether or not they
 *     intersect (true if they intersect, false if they don't). Note
 *     that just touching at the ends counts as intersecting. This is
 *     a rather heavy-handed routine - I believe there are far more
 *     elegant algorithms.
 *
 */

#ifndef TRUE
#define TRUE 1
#endif

#ifndef FALSE
#define FALSE 0
#endif

static int LinesIntersect(
    double VecX1,               /* X-coordinate of start of first line */
    double VecY1,               /* Y-coordinate of start of first line */
    double VecX2,               /* X-coordinate of end of first line */
    double VecY2,               /* Y-coordinate of end of first line */
    double X1,                  /* X-coordinate of start of second line */
    double Y1,                  /* Y-coordinate of start of second line */
    double X2,                  /* X-coordinate of end of second line */
    double Y2)
{                               /* Y-coordinate of end of second line */
    /*  Note: the use of 'Vec' to refer to the first line is historical,
     *  since the two lines are really quite interchangeable, but we have
     *  to distinguish between them somehow. (This code was lifted from
     *  an experimental C++ vector class which never got finished properly.)
     */

    double DeltaX;
    double DeltaXVec;
    double MaxX, MaxY, VecMaxX, VecMaxY;
    double MinX, MinY, VecMinX, VecMinY;

    int Intersect;              /* Function return value. */
    int MayIntersect;           /* True if the line extremes overlap */

    Intersect = FALSE;
    MayIntersect = TRUE;

    /*  First, a quick check to see if there is an overlap of the
     *  extreme positions of the lines. If there isn't, they can't
     *  overlap and that's that.
     */

    MaxX = (X1 < X2) ? X2 : X1;
    MaxY = (Y1 < Y2) ? Y2 : Y1;
    VecMaxX = (VecX1 < VecX2) ? VecX2 : VecX1;
    VecMaxY = (VecY1 < VecY2) ? VecY2 : VecY1;
    MinX = (X1 > X2) ? X2 : X1;
    MinY = (Y1 > Y2) ? Y2 : Y1;
    VecMinX = (VecX1 > VecX2) ? VecX2 : VecX1;
    VecMinY = (VecY1 > VecY2) ? VecY2 : VecY1;

    if (MaxX < VecMinX)
        MayIntersect = FALSE;
    if (MinX > VecMaxX)
        MayIntersect = FALSE;
    if (MaxY < VecMinY)
        MayIntersect = FALSE;
    if (MinY > VecMaxY)
        MayIntersect = FALSE;

    if (MayIntersect) {

        /*  The extremes overlap, so the lines might. We have to do
         *  this one the hard way, solving for the individual line
         *  equations and looking for any intersection. Along the
         *  way, we have to allow for the special cases where lines
         *  are vertical, since we can't handle infinite slopes.
         */

        DeltaX = X1 - X2;
        DeltaXVec = VecX1 - VecX2;

        if ((DeltaX == 0.0) || (DeltaXVec == 0.0)) {

            /*  At least one of the vectors is vertical. There will be an
             *  intersection of the projected lines so long as both are not
             *  vertical.
             */

            if ((DeltaX == 0.0) && (DeltaXVec == 0.0)) {

                /*  Both are vertical. This is really a special case.
                 *  Since they have an overlap of extremes, they must
                 *  actually be co-linear, ie must both have the same
                 *  X value. The overlap is therefore a vector, not a
                 *  point. We return a value of true.
                 */

                Intersect = TRUE;

            } else {

                /*  Only one is vertical. The projected lines will intersect
                 *  somewhere, so work out where and see if it is in the
                 *  overlap region of the lines.
                 */

                double X;       /* X-coordinate of intersection point */
                double Y;       /* Y-coordinate of intersection point */

                if (DeltaX == 0.0) {
                    double VecSlope = (VecY2 - VecY1) / (VecX2 - VecX1);
                    double VecIntercept = VecY1 - (VecSlope * VecX1);
                    X = X1;
                    Y = (VecSlope * X) + VecIntercept;
                } else {
                    double Slope = (Y2 - Y1) / (X2 - X1);
                    double Intercept = Y1 - (Slope * X1);
                    X = VecX1;
                    Y = (Slope * X) + Intercept;
                }
                if ((Y >= MinY) && (Y <= MaxY) &&
                    (Y >= VecMinY) && (Y <= VecMaxY)) {
                    Intersect = TRUE;
                }
            }
        } else {

            /*  Neither vector is vertical, so we don't have problems
             *  calculating slopes. This is really the general case.
             */

            double VecSlope = (VecY2 - VecY1) / (VecX2 - VecX1);
            double VecIntercept = VecY1 - (VecSlope * VecX1);
            double Slope = (Y2 - Y1) / (X2 - X1);
            double Intercept = Y1 - (Slope * X1);

            /*  If the slopes are the same, we probably don't have
             *  any intersection, but we should try to check for 
             *  co-linearity.
             */

            if ((Slope == VecSlope)) {
                if ((Intercept == VecIntercept)) {

                    /*  Slopes and intercepts are the same. We know there's
                     *  an overlap, so we return true.
                     */

                    Intersect = TRUE;
                }

            } else {

                /*  Slopes are not the same, so the projected lines intersect.
                 *  Calculate where, and see if it lies in the range of both
                 *  the lines.
                 */

                double X = (Intercept - VecIntercept) / (VecSlope - Slope);
                if ((X >= MinX) && (X <= MaxX) &&
                    (X >= VecMinX) && (X <= VecMaxX)) {
                    Intersect = TRUE;
                }
            }
        }
    }

    return (Intersect);

}

/*  ------------------------------------------------------------------------- */

/*+
 *                       R C S  V e r s i o n
 *
 *  Routine:
 *    RCSVersion
 *
 *  Function:
 *    Returns the current RCS version string, if available.
 *
 *  Call:
 *    RCSVersion(Version,Length);
 *
 *  Description:
 *    This routine returns a string in the form "n.n" giving the current
 *    RCS version number for the file in which it is included.
 *
 *  Parameters:
 *    (<)  Version  (char*)  The RCS version. Allow for at least nnn.nnn, so
 *                           make this at least 8 characters long. Will be
 *                           truncated if necessary, but will always be
 *                           properly nul-terminated.
 *    (>)  Length (double)   The number of characters available at Version.
 *
 *   Support:
 *     Keith Shortridge, AAO
 *
 *   Version Date:
 *     03-Jan-2002  KS  Original version.  
 *-
 */
 
static void RCSVersion(
   char* Version,
   int Length)
{
   /*  This string is the key to this routine. This string contains an RCS
    *  keyword, and so is modified each time the version is checked in and
    *  out. So this code doesn't actually know what is going to be in this
    *  string, except that it ought to have the form $Revision: 290518 $". If
    *  RCS is not being used, then this string will at least remain unchanged,
    *  and the routine should at least return a valid (if uninformative) string.
    */
    
   char* VersionString = "$Revision: 290519 $";
   
   /*  Local variables */
   
   int VersionLength;
   int OutIndex = 0;
   int StartIndex;
   int Index;
   
   /*  Work through the version string, looking for the ':' that indicates
    *  the end of the keyword string
    */
   
   StartIndex = 0;
   VersionLength = strlen(VersionString);
   for (Index = 0; Index < VersionLength; Index++) {
      if (VersionString[Index] == ':') {
         StartIndex = Index + 1;
         break;
      }
   }
   
   /*  Once we have the ':', copy from there to the terminating '$', ignoring
    *  any blank characters. The break out on a nul character shouldn't
    *  ever happen, but is there just in case.
    */
    
   if (StartIndex > 0) {
      for (Index = StartIndex; Index < VersionLength; Index++) {
         char TheChar = VersionString[Index];
         if (TheChar == '$') break;
         if (TheChar != ' ') {
            Version[OutIndex++] = TheChar;
            if (OutIndex >= Length) break;
         }
      }
   }
   
   /*  Try to terminate the output string nicely. In any case, make sure
    *  it's terminated.
    */
    
   if (OutIndex < Length) Version[OutIndex] = '\0';
   Version[Length - 1] = '\0';
            
}

/*  ------------------------------------------------------------------------- */

/*
 *                           X E r r o r 
 *
 * Routine name:
 *   XError
 *  
 * Description:
 *   Handle X errors.  This routine is called by Tk when X errors occur.  
 *   We attempt to continue, after outputing messages.  The effect of 
 *   this is unclear  but is probably more errors!   It is hoped that 
 *   experience will show which errors we can actually continue from.
 *   
 *   Note that we cannot do any output to the X display within this handler,
 *   all messages must go to stderr.
 *  
 * History:
 *   30-Oct-1995 - TJF - Original version (for tdfct)
 *   04-Apr-1997 - TJF - tdFPos version.  Exit if too many errors.
 *   27-Oct-1997 - TJF - Configure version.
 *   03-Nov-1998 - TJF - Revamp for non-DRAMA configure.
 *   15-Mar-2000 - TJF - Support macintosh. 
 */

static int XError(ClientData clientData, XErrorEvent * errorEventPtr)
{
#   if !defined(WIN32) && !defined(macintosh)
    char err_msg[80];
#   endif
    Tcl_Interp *interp = (Tcl_Interp *) clientData;


    Tcl_Channel errChannel;
    errChannel = Tcl_GetStdChannel(TCL_STDERR);
    if (errChannel) {
        char buffer[100];

/*
 *      Get the details of this error message.
 */
#       if !defined(WIN32) && !defined(macintosh)
        XGetErrorText(Tk_Display(Tk_MainWindow(interp)),
            errorEventPtr->error_code, err_msg, sizeof(err_msg));
#       endif


        XErrorCount++;          /* Increment error count */
/*
 *      Output the details
 */
        Tcl_Write(errChannel,
            "+++++++++++++++++++++++++++++++++++++++++++++++++++++++\n", -1);

        sprintf(buffer,
            "Configure:X error handler hit\07 (hit number %d)\n", XErrorCount);
        Tcl_Write(errChannel, buffer, -1);

#       if !defined(WIN32) && !defined(macintosh)
        sprintf(buffer, "Error detected:\n %s\n", err_msg);
        Tcl_Write(errChannel, buffer, -1);

        sprintf(buffer, "  Protocol request: %d\n",
            errorEventPtr->request_code);
        Tcl_Write(errChannel, buffer, -1);

        sprintf(buffer, "  Resource ID:      0x%lx\n",
            errorEventPtr->resourceid);
        Tcl_Write(errChannel, buffer, -1);

        sprintf(buffer, "  Minor Op code:    %d\n", errorEventPtr->minor_code);
        Tcl_Write(errChannel, buffer, -1);

        sprintf(buffer, "  Serial Number:    %ld\n", errorEventPtr->serial);
        Tcl_Write(errChannel, buffer, -1);
#       endif

/*
 *      If we are in a Tcl procedure, output the location.  This may not
 *      achieve much due to asychronous behaviour of X windows.
 */

        Tcl_Eval(interp, "if { [info level] > 0 } { \n\
                  puts stderr \"Error location is \" \n\
                       for {set i [info level]} {$i > 0} {incr i -1} { \n\
                                puts stderr \"Level $i: [info level $i]\" \n\
                       } \n\
                  } else { puts stderr \"Error occured at global level\" } \n");

/*
 *      On second error, enable X syncrhnoize mode to help catch errors.
 */

#       if !defined(WIN32) && !defined(macintosh)
        if (XErrorCount == 2) {
            Tcl_Write(errChannel, "Setting X protocol to Synchronous\n", -1);
            XSynchronize(Tk_Display(Tk_MainWindow(interp)), 1);

        }
#       endif


        if (XErrorCount < MAX_XERRORS) {
            Tcl_Write(errChannel, "Attempting to continue\n", -1);
            Tcl_Write(errChannel,
                "+++++++++++++++++++++++++++++++++++++++++++++++++++++++\n",
                -1);
            return 0;
        } else {
            Tcl_Write(errChannel,
                "CONFIGURE:too many errors, am exiting\n", -1);
            Tcl_Eval(interp, "exit 1");

        }
    }
    return 1;
}


/*  ------------------------------------------------------------------------- */

/*                         C o n f  M e s s a g e
 *
 *  This routine is used internally and passed to ConfigInitialise() for
 *  outputing informational messages to the user during allocation.
 *
 *  We invoke the Tcl command MsgOut to output the message.
 */
 
static void ConfMessage(void *argument, const char *string)
{
    Tcl_Interp *interp = (Tcl_Interp *) argument;
    Tcl_DString command;

    Tcl_DStringInit(&command);
    Tcl_DStringAppend(&command, "MsgOut", -1);
    Tcl_DStringAppendElement(&command, string);

    if (Tcl_GlobalEval(interp, command.string) != TCL_OK) {
        char s[200];
        sprintf(s,
            "\n(Configure InfoMessage routine for string \"%s\")", string);
        Tcl_AddErrorInfo(interp, s);
        Tcl_BackgroundError(interp);
    }

}

/*  ------------------------------------------------------------------------- */

/*                      C o n f  R p t  C o n s t  F i l e
 *
 *  This callback routine is invoked when the instrument support layer
 *  opens its constants files.  Tcl user interface will not yet be running
 *  so we report it using printf().
 */
 
static void ConfRptConstFile(void *argument DUNUSED,
    const char *string, StatusType * status)
{
    if (*status != STATUS__OK)
        return;
    fprintf(stdout, "Opened constants file \"%s\"\n", string);
}

/*  ------------------------------------------------------------------------- */

/*                 C o n f  R p t  T e l  M o d e l  F i l e
 *
 *  This routine is invoked when the instrument support layer opens
 *  its telescope model files.  
 *
 *  If the Tcl user interface is running at this stage, the argument
 *  is a pointer to the TCL interpreter and you should use ConfMessage
 *  to report the message.
 *
 *  Otherwise, report it using printf().
 */
 
static void ConfRptTelModelFile(void *argument, const char *string,
    StatusType * status)
{
    if (*status != STATUS__OK)
        return;
    if (argument) {
        char buffer[300];
        snprintf( buffer, sizeof(buffer), 
            "Opened telescope model file \"%s\"", string);
        ConfMessage(argument, buffer);
    } else
        fprintf(stdout, "Opened telescope model file \"%s\"\n", string);
}

/*  ------------------------------------------------------------------------- */

/*                           E r s  H a n d l e r
 *
 *  This routine is invoked to handle the output of Ers messages.  It is
 *  invoked by the Ers C code, being passed one or more messages that are
 *  to be output. When ErsStart() is invoked, this routine is passed to it
 *  as the reporting routine to be used.
 *  
 */
 
static void ErsHandler(void *outArg, unsigned int count,
    const ErsMessageType messages[], StatusType * status)
 {
    register unsigned int i;
    Tcl_Interp *interp = (Tcl_Interp *) outArg;

    char buffer[2000];
    int n;
    int result;
    Tcl_DString command;
    int highlight = 0, bell = 0, alarm = 0;
    buffer[0] = 0;
    if (*status != STATUS__OK)
        return;
/* 
 *      Construct the error message.
 */
    for (i = 0; i < count; i++) {
        n = 2000 - strlen(buffer) - 2;
        strncat(buffer, messages[i].message, n);
        buffer[n] = '\0';
        if (i + 1 < count)
            strcat(buffer, "\n");
        if (messages[i].flags & ERS_M_ALARM)
            alarm = 1;
        if (messages[i].flags & ERS_M_HIGHLIGHT)
            highlight = 1;
        if (messages[i].flags & ERS_M_BELL)
            bell = 1;
    }
/*
 *  Convert to a C language string.
 */
    Tcl_DStringInit(&command);
    Tcl_DStringAppend(&command, "ErsOut", -1);
    Tcl_DStringAppendElement(&command, buffer);

/*
 *  Add Ers flags if required.
 */
    if (alarm)
        Tcl_DStringAppendElement(&command, "-alarm");
    if (highlight)
        Tcl_DStringAppendElement(&command, "-highlight");
    if (bell)
        Tcl_DStringAppendElement(&command, "-bell");

/*
 *  Output the message.
 */
    result = Tcl_GlobalEval(interp, command.string);
/*
 *  IF we have a failure whist reporting an error, then we can't
 *  use the normal background error reporter routine, as it uses
 *  ErsRep and we can't use ErsRep whlist in the middle of flushing
 *  Ers messages.  We must report it ourselves.  This code is a
 *  simplification of the code in DtclBackgroundError().
 */
    if (result != TCL_OK) {
        char *targv[2];
        char *scommand;
        char *error;
        char *errorInfo;
        const char *tmp;

/*
 *      Add our own error context.
 */
        Tcl_AddErrorInfo(interp, "\n(Configure Ers Handler)");

/*
 *      Copy the error message.
 */
        error = malloc((unsigned) (strlen(interp->result) + 1));
        strcpy(error, interp->result);
/*
 *      Copy the contents of errorInfo.  (Note, we are ensured there is
 *      one since we just added to it)
 */
        tmp = Tcl_GetVar(interp, "errorInfo", TCL_GLOBAL_ONLY);
        errorInfo = malloc((unsigned) (strlen(tmp) + 1));
        strcpy(errorInfo, tmp);
/*
 *      Create the command to execute and do so.
 */
        targv[0] = "tkerror";
        targv[1] = error;
        scommand = Tcl_Merge(2, targv);
        result = Tcl_GlobalEval(interp, scommand);
/*
 *      Failure in tkerror.  This is the only place where we know
 *      if the command actually existed.  If it did not exist, we just
 *      report the original error to stderr.  
 */
        if (result != TCL_OK) {
            if ((strcmp(interp->result,
                        "invalid command name: \"tkerror\"") == 0) ||
                (strcmp(interp->result,
                        "invalid command name \"tkerror\"") == 0)) 
               fprintf(stderr, "%s\n", errorInfo);
            else {

/*
 *              This indicates an error in the actual call to
 *              tkerror.  tkerror does exist.  We want to report
 *              both errors.
 */
                fprintf(stderr, "tkerror failed to handle background error.\n");
                fprintf(stderr, "    Original error: %s\n\n", errorInfo);
                fprintf(stderr, "    Error in tkerror: %s\n", tmp);
            }
        }
        Tcl_ResetResult(interp);
        free(scommand);
        free(error);
        free(errorInfo);

    }
    Tcl_DStringFree(&command);
}

/* -------------------------------------------------------------------------- */

/*                 C h e c k s u m  R o u t i n e s
 *
 *  The following set of routines have been lifted directly from the
 *  file checksum.c obtained from ftp://iraf.noao.edu/misc/checksum.
 *  They implement the 1's complement checksum scheme described for FITS 
 *  by Seaman and Pence and which was adopted by ESO as the PAF file
 *  checksum.
 *
 *  I have made the following changes to the distributed code in order for
 *  it to compile cleanly under gcc -Wall -ansi:
 *  1) checksum() is now declared as a static routine returning void. 
 *  2) char_encode() is now declared as a static routine returning void. 
 */

/* Explicitly exclude those ASCII characters that fall between the
 * upper and lower case alphanumerics (<=>?@[\]^_`) from the encoding.
 * Which is to say that only the digits 0-9, letters A-Z, and letters
 * a-r should appear in the ASCII coding for the unsigned integers.
 */
#define	NX	13
unsigned exclude[NX] = { 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 0x40,
			 0x5b, 0x5c, 0x5d, 0x5e, 0x5f, 0x60 };

int offset = 0x30;		/* ASCII 0 (zero) character */


/* CHECKSUM -- Increment the checksum of a character array.  The
 * calling routine must zero the checksum initially.  Shorts are
 * assumed to be 16 bits, ints 32 bits.
 */

/* Internet checksum algorithm, 16/32 bit unsigned integer version:
 */
static void checksum (buf, length, sum16, sum32)
char		*buf;
int		length;
unsigned short	*sum16;
unsigned int	*sum32;
{
	unsigned short	*sbuf;
	int	 	len, remain, i;
	unsigned int	hi, lo, hicarry, locarry, tmp16;

	sbuf = (unsigned short *) buf;

	len = 2*(length / 4);	/* make sure len is even */
	remain = length % 4;	/* add remaining bytes below */

	/* Extract the hi and lo words - the 1's complement checksum
	 * is associative and commutative, so it can be accumulated in
	 * any order subject to integer and short integer alignment.
	 * By separating the odd and even short words explicitly, both
	 * the 32 bit and 16 bit checksums are calculated (although the
	 * latter follows directly from the former in any case) and more
	 * importantly, the carry bits can be accumulated efficiently
	 * (subject to short integer overflow - the buffer length should
	 * be restricted to less than 2**17 = 131072).
	 */
	hi = (*sum32 >> 16);
	lo = (*sum32 << 16) >> 16;

	for (i=0; i < len; i+=2) {
	    hi += sbuf[i];
	    lo += sbuf[i+1];
	}

	/* any remaining bytes are zero filled on the right
	 */
	if (remain) {
	    if (remain >= 1)
		hi += buf[2*len] * 0x100;
	    if (remain >= 2)
		hi += buf[2*len+1];
	    if (remain == 3)
		lo += buf[2*len+2] * 0x100;
	}

	/* fold the carried bits back into the hi and lo words
	 */
	hicarry = hi >> 16;
	locarry = lo >> 16;

	while (hicarry || locarry) {
	    hi = (hi & 0xFFFF) + locarry;
	    lo = (lo & 0xFFFF) + hicarry;
	    hicarry = hi >> 16;
	    locarry = lo >> 16;
	}

	/* simply add the odd and even checksums (with carry) to get the
	 * 16 bit checksum, mask the two to reconstruct the 32 bit sum
	 */
	tmp16 = hi + lo;
	while (tmp16 >> 16)
	    tmp16 = (tmp16 & 0xFFFF) + (tmp16 >> 16);

	*sum16 = tmp16;
	*sum32 = (hi << 16) + lo;
}


/* CHAR_ENCODE -- Encode an unsigned integer into a printable ASCII
 * string.  The input bytes are each represented by four output bytes
 * whose sum is equal to the input integer, offset by 0x30 per byte.
 * The output is restricted to alphanumerics.
 *
 * This is intended to be used to embed the complement of a file checksum
 * within an (originally 0'ed) ASCII field in the file.  The resulting
 * file checksum will then be the 1's complement -0 value (all 1's).
 * This is an additive identity value among other nifty properties.  The
 * embedded ASCII field must be 16 or 32 bit aligned, or the characters
 * can be permuted to compensate.
 *
 * To invert the encoding, simply subtract the offset from each byte
 * and pass the resulting string to checksum.
 */

static void char_encode (value, ascii, nbytes, permute)
unsigned int	value;
char		*ascii;	/* at least 17 characters long */
int		nbytes;
int		permute;
{
	int	byte, quotient, remainder, ch[4], check, i, j, k;
	char	asc[32];

	for (i=0; i < nbytes; i++) {
	    byte = (value << 8*(i+4-nbytes)) >> 24;

	    /* Divide each byte into 4 that are constrained to be printable
	     * ASCII characters.  The four bytes will have the same initial
	     * value (except for the remainder from the division), but will be
	     * shifted higher and lower by pairs to avoid special characters.
	     */
	    quotient = byte / 4 + offset;
	    remainder = byte % 4;

	    for (j=0; j < 4; j++)
		ch[j] = quotient;

	    /* could divide this between the bytes, but the 3 character
	     * slack happens to fit within the ascii alphanumeric range
	     */
	    ch[0] += remainder;

	    /* Any run of adjoining ASCII characters to exclude must be
	     * shorter (including the remainder) than the runs of regular
	     * characters on either side.
	     */
	    check = 1;
	    while (check)
		for (check=0, k=0; k < NX; k++)
		    for (j=0; j < 4; j+=2)
			if (ch[j]==exclude[k] || ch[j+1]==exclude[k]) {
			    ch[j]++;
			    ch[j+1]--;
			    check++;
			}

	    /* ascii[j*nbytes+(i+permute)%nbytes] = ch[j]; */
	    for (j=0; j < 4; j++)
		asc[j*nbytes+i] = ch[j];
	}

	for (i=0; i < 4*nbytes; i++)
	    ascii[i] = asc[(i+4*nbytes-permute)%(4*nbytes)];

	ascii[4*nbytes] = 0;
}

/* -------------------------------------------------------------------------- */

/*           I m p  Z  E x e  D i r  -  P r e l i m i n a r i e s
 *
 *  To provide the configuration program with the means to determine which
 *  directory it is being run from, the utility routine ImpZExeDir() has been
 *  extracted from the code for the IMP message system used by DRAMA. This
 *  was the most conveniently packaged version of such a routine.
 *  
 *  The ImpZExeDir function and its dependencies have been copied
 *  verbatim from the impz_unix.c file. This short section sets the various
 *  constants used by IMP that are normally defined in IMP include files
 *  and which are used by the ImpZExeDir() code.
 */
 
#define IMPZ_PROTOTYPES_OK 1
#define IMP_FALSE 0
#define IMP_TRUE 1
#if defined(unix) || defined(__unix__) || defined(__unix) || \
                                                      defined(__macosx__)

/* -------------------------------------------------------------------------- */


/*                         I m p  Z  E x e  D i r
 *
 *  Routine:
 *     ImpZExeDir
 *
 *  Function:
 *     Gives the name of the directory from which the current program was run.
 *
 *  Description:
 *     In systems where individual tasks are started by loading executable
 *     programs from disk, it is often useful for a program to be able to
 *     locate the directory from which it was loaded. This enables it to
 *     find related files, which are often held in the same directory, 
 *     in a straightforward way that doesn not require the use of logical
 *     names (VMS), environment variables (UNIX), or some other such scheme.
 *
 *     The interface to this routine requires the zeroth argument passed to
 *     the program when it was started up. This is needed by the UNIX 
 *     implementation, although not by the VMS version. It returns the full
 *     pathname or file specification of the directory containing the
 *     program currently being run. This will be formatted such that it
 *     can be prepended to a filename to get a valid full file specification.
 *     (So, for example, under UNIX it will end with a '/', under VMS it will
 *     end with a ']'.) If the system is such that the concept does not make
 *     sense (embedded systems like VxWorks), or where it simply cannot be
 *     implemented), a 'Known' flag argument will be returned set false.
 *    
 *  Implementation details for UNIX:
 *     Most of the work is done by the routine findx(), originally written
 *     by Greg Limes (limes@eng.sun.com) which is appended to this file. It
 *     works by looking through the directories specified by the PATH
 *     environment variable looking for the program name passed as the
 *     zeroth argument (although there's a number of additional wrinkles).
 */

#ifdef IMPZ_PROTOTYPES_OK
static int findx(char *cmd, char *cwd, char *dir, char **pgm, char **run,
    char *path);
static int resolve(             /* get link resolution name */
    char *indir, char *cmd, char *dir, char **run);

static void ImpZExeDir(
    char *Arg0,                 /* Zeroth argument to program */
    char *Directory,            /* Name of 'exe' directory */
    int LDir,                   /* Number of chars in Directory */
    int *Known)                 /* True if Directory is valid */
#else

static int findx();
static int resolve();

static void ImpZExeDir(Arg0, Directory, LDir, Known)
char *Arg0;                     /* Zeroth argument to program */
char *Directory;                /* Name of 'exe' directory */
int LDir;                       /* Number of chars in Directory */
int *Known;                     /* True if Directory is valid */
#endif

/*  The following is my (KS) wrap-up of findx(), producing the UNIX version of
 *  ImpZExeDir(). This is a  routine which a) sets up a call to findx()
 *  to get the name of the 'exe' directory, b) manipulates the result
 *  slightly for aesthetic reasons and to make sure the result ends with a
 *  '/' character, and c) returns it in the nul-terminated character string
 *  Directory. The manipulation of the string returned by findx() is because
 *  the findx() algorithm tends to produce strings such as '/home/aatssb/ks/.'
 *  or '/home/aatssb/ks/work/../scrap/.' which are valid but messy if the
 *  result is ever to be output. (This is a small modification of the Figaro
 *  Fortran-callable utility GEN_EXEDIR, which was canibalised to produce
 *  this code, which is probably over fussy for this IMP application.)
 */

{
    /*  Local variables  */

    int Back;                    /* Number of '/' chars to work back through */
    char Char;                   /* Character in directory name */
    char CurrentDir[MAXPATHLEN] = ""; 
                                 /* Work space used for current directory */
    int Dots;                    /* Dot count between '/' characters */
    char ExeDir[MAXPATHLEN] = "";/* Receives the exe directory from findx() */
    int Ichar;                   /* Index into ExeDir */
    int Ioutchar;                /* Index into returned directory string */
    char LastChar;               /* Last character written into Directory */

    /*  Functions used  */


    /*  Set up the call to findx() to get the directory name  */

    *Known = IMP_FALSE;
    Ioutchar = 0;
    if (findx(Arg0, CurrentDir, ExeDir, (char **) 0,
            (char **) 0, getenv("PATH")) == 0) {

        /*  The call to findx() worked, so we now have a directory string. We
         *  combine the copying of it into the Directory string and the tidying
         *  up. We copy it character by character, keeping a count of the number
         *  of dots after each '/', clearing the count on an ordinary character.
         *  If we come to a new '/' or the end of the string with a positive 
         *  dot count, we skip back by a number of '/' chars equal to the 
         *  number of dots we found.  This has the effect of turning a '/./'
         *  sequence into a '/' and a 'xx/../yy' sequence into 'yy'.
         */

        Char = '\0';
        Ichar = 0;
        Dots = 0;
        do {
            Back = 0;
            LastChar = Char;
            Char = ExeDir[Ichar++];
            Directory[Ioutchar] = Char;
            if (Char == '\0') {
                Back = Dots;
                Ioutchar--;
            } else if (Char == '/') {
                Back = Dots;
                Dots = 0;
            } else if (Char == '.') {
                Dots++;
            } else {
                Dots = 0;
            }
            if (Back)
                LastChar = '/';
            while (Back) {
                if (Directory[--Ioutchar] == '/')
                    Back--;
                if (Ioutchar <= 0)
                    break;
            }
            if (Ioutchar < LDir)
                Ioutchar++;
        } while (Char);
        if (LastChar != '/')
            Directory[Ioutchar++] = '/';
        if (Ioutchar < LDir) {
            Directory[Ioutchar] = '\0';
            *Known = IMP_TRUE;
        }
    }

}




/* -------------------------------------------------------------------------- */

/*                               F i n d x
 *
 *  The following is the body of the code of the routine findx() as written
 *  by Greg Lines. I wanted to include it to keep this file self-contained,
 *  but I didn't want to dive into it making minor changes and so propagate
 *  a 'slightly' different version if I could help it. I have added a
 *  'static' to the declaration of findx() itself in order to hide the name
 *  and keep it a local routine. Other than that this is the code as supplied.
 *  The original release includes a man page which documents the routine fully,
 *  but I have not included that. Note that the main() routine declared in the
 *  code is not normally compiled.
 *
 *  Having said that in the original version of this routine, I have now made
 *  a more significant modification to the findx() code. I have introduced the
 *  variables dp and dstr, making it copy the individual path elements into
 *  dstr rather than work on them in place - the original code modified the
 *  PATH string it was passed, and so could not be called more than once. I
 *  have also removed the now unused variable 's'.  Subsequently, I have also
 *  eliminated the BSD function rindex() to allow this to run under SOLARIS,
 *  by using the following Rindex() function and using it instead. I have
 *  also changed the name of the symlink variable to symblink to prevent lint
 *  complaining about it hiding the name of a system function. Moreover, I
 *  have removed the TAB characters from the original code, since the TAB
 *  settings being used didn't match those in the rest of this file and so
 *  made the code layout very odd indeed.
 */

#ifdef IMPZ_PROTOTYPES_OK

char *Rindex(char *s, char c)
#else

char *Rindex(s, c)
char *s, c;

#endif

{
    char *ptr = s;
    char *cptr = (char *) 0;
    while (*ptr) {
        if (*ptr == c)
            cptr = ptr;
        ptr++;
    }
    return (cptr);
}

/*  ------------------------------------------------------------------------- */

#include <errno.h>
#include <sys/param.h>

#ifndef   X_OK
#define   X_OK   1              /* for "access" */
#endif

#ifndef   MAXPATHLEN
#define   MAXPATHLEN   1024
#endif

#ifndef   ENAMETOOLONG
#define   ENAMETOOLONG   EINVAL
#endif

#ifdef   FINDX_TESTMAIN
/*
 * An example of how to use this routine.
 */
extern char *getenv();          /* read value from environment */
char *pn = (char *) 0;          /* program name */
char *rn = (char *) 0;          /* run name */
char rd[MAXPATHLEN];            /* run directory */
char wd[MAXPATHLEN] = ".";      /* working directory */

int main(argc, argv)
int argc;
char **argv;
{
    findx(*argv, wd, rd, &pn, &rn, getenv("PATH"));
    printf("%s: %s running in %s from %s\n", pn, rn, wd, rd);
    return 0;
}
#endif

static int findx(char *cmd, char *cwd, char *dir, char **pgm,
    char **run, char *path)
{
    int rv = 0;
    char dstr[MAXPATHLEN], *dp, *f;
    if (!cmd || !*cmd || !cwd || !dir) {
        errno = EINVAL;         /* stupid arguments! */
        return -1;
    }
    if (!path || !*path)        /* missing or null path */
        path = ".";             /* assume sanity */

    if (*cwd != '/')
        if (!(getcwd(cwd, MAXPATHLEN))) {
            return -1;          /* cant get working directory */
        }

    f = Rindex(cmd, '/');
    if (pgm)                    /* user wants program name */
        *pgm = f ? f + 1 : cmd;

    if (dir) {                  /* user wants program directory */
        rv = -1;
        if (*cmd == '/')        /* absname given */
            rv = resolve("", cmd + 1, dir, run);
        else if (f)             /* relname given */
            rv = resolve(cwd, cmd, dir, run);
        else if ((f = path)) {  /* from searchpath */
            rv = -1;
            errno = ENOENT;     /* errno gets this if path empty */
            while (*f && (rv < 0)) {
                dp = dstr;
                while (*f && (*f != ':'))
                    *dp++ = *f++;
                if (*f) {
                    f++;
                }
                *dp = '\0';
                if (dstr[0] == '/')
                    rv = resolve(dstr, cmd, dir, run);
                else {
                    char abuf[MAXPATHLEN];
                    sprintf(abuf, "%s/%s", cwd, dstr);
                    rv = resolve(abuf, cmd, dir, run);
                }
            }
        }
    }
    return rv;
}

/*
 * resolve - check for specified file in specified directory. sets up dir,
 * following symlinks. returns zero for success, or -1 for error (with
 * errno set properly)
 */
static int resolve(char *indir, /* search directory */
    char *cmd,                  /* search for name */
    char *dir,                  /* directory buffer */
    char **run)
{                               /* resultion name ptr ptr */
    char *p;
#ifdef   ELOOP
    int sll;
    char symblink[MAXPATHLEN + 1];
#endif

    errno = ENAMETOOLONG;
    if ((int) (strlen(indir) + strlen(cmd) + 2) > MAXPATHLEN) {
        return -1;
    }

    sprintf(dir, "%s/%s", indir, cmd);
    if (access(dir, X_OK) < 0) {
        return -1;              /* not an executable program */
    }
#ifdef   ELOOP
    while ((sll = readlink(dir, symblink, MAXPATHLEN)) >= 0) {
        symblink[sll] = 0;
        if (*symblink == '/')
            strcpy(dir, symblink);
        else
            sprintf(Rindex(dir, '/'), "/%s", symblink);
    }
    if (errno != EINVAL) {
        return -1;
    }
#endif

    p = Rindex(dir, '/');
    *p++ = '\0';
    if (run)                    /* user wants resolution name */
        *run = p;
    return 0;
}
#elif defined (WIN32)
/*
 *  Implementation details for WIN32:
 *      We can get the full path name of Arg0 using GetFullPathName().  That
 *      routine also returns a pointer to the name component, so we can
 *      work out what the directory component is and return it.
 */

void ImpZExeDir(char *Arg0,     /* Zeroth argument to program */
    char *Directory,            /* Name of 'exe' directory */
    int LDir,                   /* Number of chars in Directory */
    int *Known)
 {                              /* True if Directory is valid */
    char FullName[MAX_PATH + 1];/* Space for full file name */
    char *NamePtr;              /* Pointer to name component */
/*
 *  Get the full path name of the executable.
 */
    if (GetFullPathName(Arg0, sizeof(FullName), FullName, &NamePtr) == 0) {
        *Known = FALSE;
        goto Exit;
    }
/*
 *  NamePtr points to the the first character of the name component of the
 *  file name.  We null terminate the string here so that we only have
 *  the directory.  We then copy it to the result string and ensure that
 *  is null terminated.
 */
    *NamePtr = '\0';
    strncpy(Directory, FullName, LDir);
    Directory[LDir - 1] = '\0';

    *Known = TRUE;

  Exit:;
}
#endif

/*
         1         2         3         4         5         6         7         8
12345678901234567890123456789012345678901234567890123456789012345678901234567890
*/
