/*******************************************************************************
* E.S.O. - VLT project
*
* "@(#) $Id: config.c 186402 2009-06-26 16:05:56Z mpruemm $"
*
* who       when      what
* --------  --------  ----------------------------------------------
* mpruemm   2009-06-23  Use strcmp() when comparing strings!
* rschmutz  03/02/03  PPRS8708: by default: use fposs clear values.
* rschmutz  23/11/02  this header created - previous history see below.
*/

/*+
 *                                C O N F I G
 *
 *  Function:
 *     Routines used to handle the 2dF fibre allocations.
 *
 *  Description:
 *     This file contains a set of routines that between them provide the
 *     2dF configuration algorithm. These can be used to build a separate
 *     DRAMA task to handle the configuration, or may be built into some
 *     other task.
 *
 *  Version date: 29th January 2003.
 *
 *  Sccs Id:     config.c, Release 1.18, 01/29/03
 *
 *  Support: K. Shortridge, AAO.
 *-
 *  History:
 *      9th Sep 1994.  Original version. KS/AAO.
 *     11th Sep 1994.  Combined different method matrices into a structure,
 *                     making the code simpler to modify. KS/AAO.
 *     27th Sep 1994.  Method-specific code moved to separate file. KS/AAO.
 *     14th Oct 1994.  Now allows control over clearance and angle limit
 *                     parameters. KS/AAO.
 *      2nd Dec 1994.  Not all errors found by ConfigFieldCheck were resulting
 *                     in bad status being returned. Fixed. KS/AAO.
 *      3rd May 1995.  Now allows for empty or missing 'unallocGuide",
 *                     "unallocObject" or "objects" structures. KS/AAO.
 *     25th Jul 1995.  Corrected bug in ConfigAllocate which prevented
 *                     Field 1 being allocated properly. Also fixed a
 *                     couple of gcc warnings. KS/AAO.
 *     30th Sep 1996.  Ensured AllocInfoPtr[i].State gets set to zero
 *                     if the fibre is broken on each new target file.
 *                     Previously it was possible for it to get unzeroed
 *                     if a file contained an allocation to a broken fibre.
 *                     This would allow a subsequent reallocation to the
 *                     broken fibre.
 *
 *      5th Nov 1996.  GBD: Changed ConfigSetTargets so that the allocation
 *                     is not checked at the end of this routine as this 
 *                     prevented ConfigMethodInit from ever being called
 *                     for an existing, but flawed configuration. -Method
 *                     now has an option for automatically tidying up error 
 *                     conditions which are due to a change of circumstances.
 *     14th Nov 1996.  GBD: Changed PIV_MAX in tdFbuttonDims.h (my local
 *                     version) to 0.25 radians to agree with positioner
 *                     limits. (The positioner code does not appear to use
 *                     the limits specified in this file, but a seperate
 *                     SDS structure.) This file now points to a version 
 *                     of tdFbuttonDims.h in the current directory.
 *     20th Feb 1998.  Revert to using system tdFbuttonDims.h which has
 *                     been modified as appropiate.
 *     22nd Jan 1999.  Add a large number of changes made by Gavin Dalton.
 *                     These include Add ObjName to CONFIG_TargetInfo 
 *                     structure.  Added function ConfigSetImportFile() and
 *                     ConfigSetImportOnlyMode().  Various other changes.
 *                     Previous SCCS version was 1.13.
 *                     Add  SCCS Id.  and fix up GCC compiler warnings TJF/AAO.
 *    01st Feb 1999.   ConfigAccessItem was incorrectly using != instead
 *                     of strcmp to complare strings.  TJF/AAO.
 *    12th Feb 1999.   Rename Config.h to tdFconfig.h to avoid conflicts
 *                     with DRAMA's config.h in case insentivie enviroments. 
 *                     TJF/AAO.
 *    22nd Jun 1999.   Most of the time we call ConfigAccessItem() we don't
 *                     want the Sds ID of the item and in some cases we
 *                     are leaking this SDS id.  So make it optional and
 *                     modify calls to ConfigAccessItem() appropiately.  Make
 *                     relevant poninter's const to catch cases where we need
 *                     SdsFlush(). 
 *
 *                     This change was mode to ConfigInitTargets() and
 *                     ConfigInitPivots().
 *
 *                     In addition to changes in relation to ConfigAccessItem()
 *                     ConfigInitPivots() has been modified to only free
 *                     the ID if the data was read from a file, and then
 *                     to call SdsReadFree().
 *
 *                     ConfigReadTargetFile() modified to call SdsReadFree().
 *                     
 *    14th Oct 1999.   Fixed a bug in ConfigAccessItem() where a dims[0] == 0
 *                     was not accepted.  This is quite ok and just indicates
 *                     there are no objects in the array.  Note there is no
 *                     need to check for < 0 as was previously done as the 
 *                     value is unsigned. TJF/AAO.
 *    07th Mar 2000.   Major changes to support new style where the
 *                     algorithim routines are called via function pointer. 
 *                     Add ConfigRegisterMethod() and ConfigSelectMethod. 
 *                     ConfigSetTargets() now sets the priority of
 *                     out of field targets to CONFIG_PRIO_INVALID such
 *                     that the method specific does does not have to 
 *                     do this.  
 *                     NoteDeallocation() now handles objects which
 *                     were preallocated, removing the need for
 *                     config_method.c to play with TargetInfo. 
 *                     Add ConfigNoteDeallocationBroken(). 
 *                     Add Config collision detection lay which will
 *                     call FPIL collision detection.  
 *                     Add ConfigNumMethods and ConfigGetMethodName. 
 *                     Remove dependence on DRAMA includes files except
 *                     for SDS and ERS.  Replace calls to 
 *                     MsgOut by use of InfoRoutine.  TJF/AAO.
 *   08th Mar 2000.    Start conversion to use of FPIL to specify instrument
 *                     details rather then assuming 2dF.
 *                     Add ConfigErrRep routine used by algorithim layer
 *                     to report errors such that it is not dependent on
 *                     the ERS library.  But this layer still uses ERS.
 *                     Remove ConfigVerify(), ConfigAllocateFibre() and
 *                     ConfigDeallocateFibre() as these where just dummies
 *                     and if they have not been needed by now we probably
 *                     won't need them.
 *                     Use FPIL collision detection.
 *                     Set parameter stuff now supports setting FPIL
 *                     equivalent value.
 *                     Set parameter initialise values and ranges from
 *                     FPIL values within ConfigInitailise(). TJF/AAO.
 *   08th Mar 2000.    No longer include 2dF include files, use FPIL
 *                     for all values previously retrived from these. TJF/AAO.
 *   16th Mar 2000.    Set up the "Spect" field correctly using information
 *                     from FPIL about which pivots goto which spectrograph.
 *                     TJF/AAO.
 *   20th Mar 2000.    Add and support ConfigSetSpecPivotsFlag() and
 *                     ConfigSetPivotFlag() which allow allocation of
 *                     pivots to be disabled.  Add the AllowAllocChange
 *                     item to PivotInfo.  TJF/AAO.
 *   28th Mar 2000.    ConfigNoteAllocation and ConfigNoteDeallocation
 *                     now support allocation reports. TJF/AAO.
 *   31th Mar 2000.    Support tentative allocation and deallocation to
 *                     avoid calling user interface for tentative changes. 
 *                     TJF/AAO
 *   06th Apr 2000.    Use FpilCollision1() in ConfigCheckAllocation().
 *                     Add support for MethodInfo->Collision() function using
 *                     ConfigCollision.  Macro CHECK_FPIL can
 *                     be enabled to check old way versions new approach.
 *                     When we are confident, can remove all code marked
 *                     by CHECK_FPIL.
 *                     Check for collisions against parked fibres as
 *                     in 6dF this is possible.  (Later, may make this 
 *                     optional, needs support from FPIL.) TJF/AAO.
 *  11th Apr 2000.     When clearning a status of CANCELLED, flush errors
 *                     rather then just clearing status, to make sure any
 *                     reports go out.  TJF/AAO.
 *  12th Apr 2000.     ConfigNoteDeallocation() was not treating all
 *                     pre-allocated fibres needing the fudging of
 *                     the structures instead of just the ones with
 *                     a negative index.  TJF/AAO.
 *  19th Apr 2000.     Support ParkMayCollide stuff.  Drop ConfigFieldCheck1()
 *                     routine (Gavin's debugging version of ConfigFieldCheck).
 *                     Support plate argument to FpilColInvPos().
 *   1st Aug 2000.     Introduced use of FpilUnallocateType() instead of
 *                     using 'U' for unallocated pivots in the "objects" 
 *                     section.  Note that 'U' is still used in the State
 *                     fields, which are not instrument-dependent. KS/AAO.
 *   9th Aug 2000.     The 'Spect' field in a CONFIG_PivotInfo structure is
 *                     now the more general 'FibreType'. Made changes to 
 *                     support this change and to use Fpil routines to handle
 *                     the values it contains. KS/AAO.
 *   5th Sep 2000.     Add new ConfigSetFibrePos routine.
 *                     Was not correctly setting the maximum fibre angle,
 *                     failing to convert is from radius to degrees.  Fixed.
 *                     TJF/AAO.
 *  20th Sep 2000.     Made use of #warning conditional. KS.
 *  28th Nov 2000.     Modified parameter names to include "(expert)" to 
 *                     indicate they should only be made available via the
 *                     user interface in 'expert' mode. KS.
 *   1st Feb 2001.     Added ConfAdjPark(). KS.
 *   2nd Feb 2001.     Added ConfigFibreReallyExists() and added code to the
 *                     routines ConfigCheckAllocation() and ConfigHAFieldCheck()
 *                     so that non-existent fibres aren't included
 *                     in the collision code. KS.
 *  21st Feb 2001.     Now allows for the case where different fibres have 
 *                     different maximum extensions. KS.
 *   7th Jun 2001.     Made non-radial pivot angle an expert option. KS.
 *  13th Nov 2001.     Added ConfigTargetIsGuide() & ConfigTargetIsGuide().
 *                     Field check code now uses the values that we expect to
 *                     be used by the positioner rather than the current values
 *                     used to create the configuration. KS.
 *  16th Nov 2001.     Added ConfigFibreSharesSky(), ConfigFibreTypesMatch(),
 *                     and ConfigFibreText(). KS.
 *  19th Nov 2001.     Added ConfigSetSky(). KS.
 *  27th Nov 2001.     Default values for the various button/fibre clearances
 *                     and fibre angles are now reduced to provide headroom
 *                     in the allocation, using CONFIG_PERCENT. KS.
 *   4th Dec 2001.     Corrected code for fibre length check. Was doing the
 *                     comparision against maximum length incorrectly. KS.
 *   5th Aug 2002.     Now allows for the possibility that a button can collide
 *                     with the fibre (not just the button) of an adjacent
 *                     parked pivot. (Now true for OzPoz guide fibres.) KS.
 *  26th Aug 2002.     The code up to now has used the actual theta value 
 *                     taken from the button park positions as if this were the
 *                     theta of a line from the field center to the pivot
 *                     position. This is not the case if the button is parked
 *                     away from that ideal line. The code has been corrected,
 *                     but there may be implications for the different
 *                     positioners - we need to be sure they use the correct
 *                     value as well. KS.
 *   5th Sep 2002.     Don't allow allocation of a fibre behind its park
 *                     position - picks up problems with OzPoz FACB's which
 *                     are currently parked on the plate. TJF fix originally 
 *                     applied at Paranal 21stAug 2002, now incorporated into
 *                     main code by KS.
 *  17th Sep 2002.     ConfigErrRep() now flushes out error reports. KS.
 *  18th Sep 2002.     Previous change reversed. Produces too many single
 *                     line error message windows. KS.
 *   5th Nov 2002.     Now allows for fibres of different types having 
 *                     different lengths, and constrains the maximum length
 *                     slightly using CONFIG_LENGTH_ADJUST. KS.
 *  12th Nov 2002.     Removed clearance discrepancy diagnostics from
 *                     ConfigHKFieldCheck(). KS.
 *  29th Jan 2003.     Added checks for collisions with adjacent park positions
 *                     to ConfigHKFieldCheck(). KS.
 *
 *  Bugs and limitations:
 *     o  There is no code that attempts to use twist angles to increase the
 *        number of allocated fibres, even for instruments that allow this.
 *+
 */


/*  Include files used by CONFIG  */

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/*  DRAMA include files  */

#include "drama.h"  /* Added by KS during standalone testing */
#include "sds.h"
#include "arg.h"    /* Added by KS during standalone testing */
#include "Ers.h"

#ifdef DEBUG
#undef DEBUG
#endif
#define DEBUG (void) printf

/*  Include files specific to CONFIG itself */

#include "tdFconfig.h"
#include "config_err.h"


/*#define CHECK_FPIL*/
/*  The following messages are handy until the FPIL stuff is deemed to be
 *  OK, but is non-standard. To date, the only system not to support it
 *  is HP-UX (of course!), so we omit the messages on that system.
 */

#ifndef _HPUX_SOURCE
#ifdef CHECK_FPIL
#warning "Checking Fpil wrap up routines against original sequence"
#else
/* #warning "Not checking FPIL wrap up routines" */
#endif
#endif

/* -------------------------------------------------------------------------- */

/*  Parameters used by the algorithm-non-specific code.
 *
 *  These are parameters for the allocation that are handled by the 
 *  non-algorithm-specific routines in this file. Parameter calls are
 *  handled initially by the code here and then requests relating to
 *  non-matching parameters are sent on to the algorithm-specific code.
 *  At present, only the various limit values are handled as parameters,
 *  so the use of the structures here is probably overkill, but it has
 *  the advantage of being essentially the same as the scheme used by
 *  the standard algorithm-specific code, and as such is very easy to
 *  extend.
 */

#define CONFIG_STATIC_NPARMS 4

/*  A array of structures like this goes into the method-specific global
 *  variables. For each parameter, there is a double precision value (used
 *  only for floating point parameters) and an integer value. For logical 
 *  parameters the integer value will be 0 (false) or 1 (true). For integer
 *  parameters the integer value is that of the parameter. For character
 *  parameters with an unlimited number of possible values, when the parameter
 *  is set to a new value an array is allocated in memory (using malloc())
 *  for a static copy of that new value string, and the address of this
 *  allocated string is held in common. For character parameters with a
 *  limited set of possible values, all possible values are held in a static
 *  array and the address of the value currently set is held in common. In
 *  this case, the integer value is also used to hold the index number of
 *  the current value being used.
 */

typedef struct CONFIG_STATIC_ParamValue {
   int IntegerValue;                         /* Integer value - see above */
   char *StringAddress;                      /* Address of string */
   double DoubleValue;                       /* Floating point value */
} CONFIG_STATIC_ParamValue;

/*  The four parameters used here are the various limiting values - the
 *  fibre clearance, the button clearance, the maximum non-radial pivot angle
 *  and the maximum button twist angle.  Two of these are integer values
 *  and two are floating point values. Note that we only allow the calling
 *  code to make these more stringent, not less so.
 */

#define CONFIG_FIBRE_CLEAR_PARM 0
#define CONFIG_BUTTON_CLEAR_PARM 1
#define CONFIG_MAX_PIV_ANGLE_PARM 2
#define CONFIG_MAX_BUT_ANGLE_PARM 3

/*  
 *  The following two values contain the FPOSS clear values
 */
static long fpossButClear;
static long fpossFibClear;

/*  The following arrays contain the parameter details as used by
 *   ConfigParameterDetails() and ConfigSetParameter().
 */

static char *CONFIG_STATIC_ParmNames[CONFIG_STATIC_NPARMS] =
               { "Fibre clearance (microns) (expert)",
                 "Button clearance (microns) (expert)",
                 "Max non-radial pivot angle (degrees) (expert)",
                 "Max button twist angle (degrees) (expert)"};

static char CONFIG_STATIC_ParmTypes[CONFIG_STATIC_NPARMS] =
               { 'I', 'I', 'F', 'F' };

static double CONFIG_STATIC_ParmLimits[CONFIG_STATIC_NPARMS][2]; 
                                          /* Initialised by ConfigInitialise */

static int CONFIG_STATIC_ParmNValues[CONFIG_STATIC_NPARMS] =
               { 0, 0, 0, 0 };

static char **CONFIG_STATIC_ParmValues[CONFIG_STATIC_NPARMS] =
               { NULL, NULL, NULL, NULL};

/*
 * The above parameters may or may not correspond to values managed
 * by the Fpil library and used in collision detection.  The parameter
 * values and the Fpil library values must always match, so whenever
 * we set a parameter, we must call the appropriate routine from 
 * FPIL.
 *
 * These types are used for Fpil routine pointers, one for routines which
 * set integer values (FpilSetFibClear() for example) and
 * one for routines which set real values (FpilSetMaxFibAng() for example).
 *
 * Note: On a particularly fussy compiler (CodeWarrior) I had to add the const
 * to the int routine declaration but not to the real routine declaration -
 * I suspect this should be checked with TJF. (Added by KS, 21/8/2000)
 */
 
typedef void (*CONFIG_FpilSetIntRoutine)  (const FpilType, unsigned long);
typedef void (*CONFIG_FpilSetRealRoutine) (FpilType, double);

/*
 * Now for each of the parameters, if it is an int parameter with a
 * corresponding Fpil routine, specify the routine. 
 */
static CONFIG_FpilSetIntRoutine 
    CONFIG_STATIC_SetFpilInt[CONFIG_STATIC_NPARMS] = 
        { FpilSetFibClear,
          FpilSetButClear,
          0, 
          0 };

/*
 * Now for each of the parameters, if it is an real parameter with a
 * corresponding Fpil routine, specify the routine.   But for the
 * pivot and button angles, the values is set in degrees but 
 * FPIL wants radians, so we must use conversion routines
 */
static void ConfigSetFpilMaxPivAng(FpilType, double);
static void ConfigSetFpilMaxFibAng(FpilType, double);

static CONFIG_FpilSetRealRoutine 
    CONFIG_STATIC_SetFpilReal[CONFIG_STATIC_NPARMS] = 
        { 0, 
          0,
          ConfigSetFpilMaxPivAng,
          ConfigSetFpilMaxFibAng };

/*  To provide headroom for our allocations, we make the default values for
 *  such things as button and fibre clearances and fibre and pivot angles
 *  not those that will be applied by the positioner (which we do use
 *  when we check the validity of an allocation), but rather make them more
 *  contstaining by a percentage specified as CONFIG_PERCENT. This should be
 *  chosen by experiment and experience to give configurations that are
 *  going to be valid over a range of wavelengths and hour angles. This
 *  value of 15% seems to provide FLAMES configurations good over +/-5 hours
 *  and the full wavelength range.
 */
 
#define CONFIG_PERCENT 15

/*  For similar reasons, we subtract off a small amount from the maximum
 *  fibre length allowed. We choose a fairly arbitrary one millimetre,
 *  given in microns.
 */
 
#define CONFIG_LENGTH_ADJUST 1000

/* ------------------------------------------------------------------------- */

/*  Global items used by CONFIG.
 */

typedef struct CONFIG_GlobalType {
   CONFIG_AllocInfo *AllocInfoPtr[FPIL_MAXFIELDS];
                                     /* Allocations for the two fields */
   int NumberPivots[FPIL_MAXFIELDS]; /* Number of Pivots for the 2 fields */
   int NumberTargets;                /* Number of targets in configuration */
   int Field;                        /* The field being configured */
   int NUnGuide;                     /* Number of unallocated guide */
   int NUnTarget;                    /* Number of unallocated object */
   CONFIG_PivotInfo *PivotInfoPtr[FPIL_MAXFIELDS];
                                     /* Base address of pivot information
                                      * structure arrays for the 2 fields. */
   CONFIG_TargetInfo *TargetInfoPtr; /* Base address of target information
                                      * structure array. */
   CONFIG_STATIC_ParamValue ParameterValues[CONFIG_STATIC_NPARMS];
                                     /* Current parameter values */
   CONFIG_MethodDetails MethodDetails[CONFIG_NUM_METHODS];
   int CurrentMethod;                /* Index to current method */
   unsigned NumMethods;              /* Number of methods currently known */
   CONFIG_MethodInfo    MethodInfo;  /* Method Call information */
   int PositionerFibClear;           /* Fibre clearance used by positioner */
   int PositionerButClear;           /* Button clearance used by positioner */
   double PositionerMaxPivAng;       /* Max pivot angle used by positioner */
   double PositionerMaxFibAng;       /* Max fibre angle used by positioner */
   int FibreTypesForSky[20];         /* Fibre types for which a specified number
                                        are to be allocated to sky targets. */
   int FibreSkyCounts[20];           /* Sky counts for FibreTypesForSky */
   int FibreSkyEntries;              /* Number of Fibre type/sky entries */
} CONFIG_GlobalType;

static CONFIG_GlobalType CONFIG_Global; /* Holds the actual global variables */

static int ConfigImportOnlyMode=0;

/* ------------------------------------------------------------------------- */

/*
 *  These are the internal routines which provide support to the
 *  allocation method routines
 */
static long ConfigMaxExtension(
    const CONFIG_MethodInfo * MethodInfo);
static void ConfigGetVirtPiv (
    const CONFIG_MethodInfo * MethodInfo,
    double OriginX,
    double OriginY,
    double Theta,
    double *VirtPivX,
    double *VirtPivY);

static double ConfigThetaFib (
    const CONFIG_MethodInfo * MethodInfo,
    const CONFIG_TargetInfo *TargetInfo,
    const CONFIG_PivotInfo *PivotInfo);

static double ConfigThetaPiv (
    int XPark,
    int YPark);

static CONFIG_Boolean ConfigCheckAlpha(
    const CONFIG_MethodInfo * MethodInfo,
    const CONFIG_TargetInfo *TargetInfo,
    const CONFIG_PivotInfo *PivotInfo,
    double *Alpha);

static CONFIG_Boolean ConfigCheckAllocation (
    const CONFIG_MethodInfo * MethodInfo,
    int Pivot,
    int Target,
    double Theta,
    int *CollidingPivot);

static void ConfigHKFieldCheck (
    const CONFIG_MethodInfo * MethodInfo,
    CONFIG_ReportFunction ReportFunction,
    void *FunctionArg, 
    int PositionerTolerances,
    StatusType  *status);

static void ConfigNoteAllocation (
    const CONFIG_MethodInfo * MethodInfo,
    int Pivot,
    int Target,
    double Theta,
    int Tentative,
    StatusType *Status);

static void ConfigTentativeFix (
    const CONFIG_MethodInfo * MethodInfo,
    int Pivot,
    int Confirm,
    StatusType *Status);

static void ConfigNoteDeallocation (
    const CONFIG_MethodInfo * MethodInfo,
    int Pivot,
    StatusType *Status);

static void ConfigNoteDeallocationBroken (
    const CONFIG_MethodInfo * MethodInfo,
    int Pivot,
    StatusType *Status);

static long ConfigButtonClear (
    const CONFIG_MethodInfo * MethodInfo);

static long ConfigFibreClear (
    const CONFIG_MethodInfo * MethodInfo);

static double ConfigMaxButAngle (
    const CONFIG_MethodInfo * MethodInfo);

static double ConfigMaxPivAngle (
    const CONFIG_MethodInfo * MethodInfo);

static long ConfigSafeClearance (
    const CONFIG_MethodInfo * MethodInfo);
  
static int  ConfigColFibFib (
   const CONFIG_MethodInfo *MethodInfo,/* (>) Supplies house keeping details */
   double  pivAX,                /* (>) Fibre A X ordinate */
   double  pivAY,                /* (>) Fibre A Y ordinate */
   double  fvpAX,                /* (>) Fibre A virtual pivot X ordinate   */
   double  fvpAY,                /* (>) Fibre A virtual pivot Y ordinate */
   double  pivBX,                /* (>) Fibre B X ordinate */
   double  pivBY,                /* (>) Fibre B Y ordinate */
   double  fvpBX,                /* (>) Fibre B virtual pivot X ordinate   */
   double  fvpBY);               /* (>) Fibre B virtual pivot Y ordinate */

static int  ConfigColButFib (
   const CONFIG_MethodInfo *MethodInfo,/* (>) Supplies house keeping details */
   double butX,                  /* (>) Button X ordinate */
   double butY,                  /* (>) Button Y ordinate */
   double theta,                 /* (>) Button theta      */
   double fvpX,                  /* (>) Fibre virtual pivot X ordinate */
   double fvpY,                  /* (>) Fibre virtual pivot Y ordinate */
   double pivX,                  /* (>) Fibre pivot X ordinate */
   double pivY);                 /* (>) Fibre pivot Y ordinate*/

static int  ConfigColButBut (
   const CONFIG_MethodInfo *MethodInfo,/* (>) Supplies house keeping details */
   double butXa,                 /* (>) Button A X ordinate */
   double butYa,                 /* (>) Button A Y ordinate */
   double thetaa,                /* (>) Button A theta      */
   double butXb,                 /* (>) Button B X ordinate */
   double butYb,                 /* (>) Button B Y ordinate */
   double thetab);               /* (>) Button B theta      */

static int  ConfigColInvPos (
   const CONFIG_MethodInfo *MethodInfo,/* (>) Supplies house keeping details */
   int       fibreType,		 /* (>) Fibre type        */
   long int  butx,               /* (>) Button X ordinate */
   long int  buty,               /* (>) Button Y ordinate */
   double    theta);             /* (>) Button theta      */


static int  ConfigCollision (
    const CONFIG_MethodInfo *MethodInfo,/* (>) house keeping details*/
    long int butXa,                     /* (>) Button A fibre X position */
    long int butYa,                     /* (>) Button A fibre Y position */
    double thetaa,                      /* (>) Button A theta angle      */
    long int pivXa,                     /* (>) Button A pivot X position */
    long int pivYa,                     /* (>) Button A pivot Y position */
    long int butXb,                     /* (>) Button B fibre X position */
    long int butYb,                     /* (>) Button B fibre Y position */
    double thetab,                      /* (>) Button B theta angle      */
    long int pivXb,                     /* (>) Button B pivot X position */
    long int pivYb);                    /* (>) Button B pivot Y position */

static int  ConfigAdjPark (
    const CONFIG_MethodInfo *MethodInfo,/* (>) house keeping details*/
    int IPiv,                           /* (>) The pivot number */
    long int targetX,                   /* (>) The target X position */
    long int targetY,                   /* (>) The target Y position */
    double theta);                      /* (>) Pivot theta angle */
    
static int  ConfigFibreReallyExists (
    const CONFIG_MethodInfo *MethodInfo,/* (>) house keeping details*/
    int IPiv);                          /* (>) The pivot number */
    
static int  ConfigParkMayCollide (
    const CONFIG_MethodInfo *MethodInfo);/* (>) house keeping details*/

static void ConfigSetFibrePos (
    const CONFIG_MethodInfo *MethodInfo,/* (>) house keeping details*/
    int Pivot,                          /* (>) Pivot to change      */
    int X,                              /* (>) New X position       */
    int Y,                              /* (>) New Y position       */
    double Theta,                       /* (>) New Theta position   */
    int Target,                         /* (>) New target index     */
    CONFIG_Boolean Allocated);          /* (>) Is it to be allocated */
    
static int  ConfigFibreIsGuide (
    const CONFIG_MethodInfo *MethodInfo, /* (>) house keeping details*/
    int FibreType);                      /* (>) Fibre type code */
    
static int  ConfigFibreIsBroken (
    const CONFIG_MethodInfo *MethodInfo, /* (>) house keeping details*/
    int FibreType);                      /* (>) Fibre type code */
    
static int  ConfigTargetIsGuide (
    const CONFIG_MethodInfo *MethodInfo, /* (>) house keeping details*/
    char TargetType);                    /* (>) Target type code */
    
static int  ConfigTargetIsSky (
    const CONFIG_MethodInfo *MethodInfo, /* (>) house keeping details*/
    char TargetType);                    /* (>) Target type code */
    
static char* ConfigFibreText (
    const CONFIG_MethodInfo *MethodInfo, /* (>) house keeping details*/
    int FibreType);                      /* (>) Fibre type code */
    
static int ConfigFibreTypeMatch (
    const CONFIG_MethodInfo *MethodInfo, /* (>) house keeping details*/
    int FibreType,                       /* (>) First fibre type code */
    int SecondType);                     /* (>) Second fibre type code */
    
static int ConfigFibreSharesSky (
    const CONFIG_MethodInfo *MethodInfo, /* (>) house keeping details*/
    int FibreType);                      /* (>) Fibre type code */
    
static int  ConfigFibreTargetCompatible (
    const CONFIG_MethodInfo *MethodInfo, /* (>) house keeping details*/
    char TargetType,                     /* (>) target type code */
    char SpectType,                      /* (>) target spectrograph code */
    int FibreType);                      /* (>) Fibre type code */
    
static void ConfigErrRep(
    void *FunctionArg,
    const char *String);

/* -------------------------------------------------------------------------- */

/*  These are the internal routines used by the 'housekeeping and external
 *  interface' part of the allocation routines. These are not called from
 *  outside the package, nor are they expected to be used by the method-
 *  specific routines.
 */

static void ConfigAccessItem (    /*     Accesses an array in a structure */
   SdsIdType StructID,            /* (>) SDS ID of enclosing structure */
   char *Name,                    /* (>) Name of array in structure */
   int ElementSize,               /* (>) Expected element size in bytes */
   char *StructName,              /* (>) Description of structure -for errors */
   int *Elements,                 /* (<) Receives # elements in array */
   void **VPointer,               /* (<) Receives pointer to array data */
   SdsIdType *ItemID,             /* (<) Receives SDS ID of array item */
   StatusType *Status);           /* (!) Inherited status */

static void ConfigInitPivots (
   FpilConstantsType InstConstDetails,/* (>)  Instrument constants details */
   StatusType *Status);           /* (!) Inherited status variable */

static void ConfigInitTargets (
   SdsIdType TopID,               /* (>) ID of top item in the SDS structure */
   char *Source,                  /* (>) Description of structure origin - only
                                   *     needed for error messages. */
   int Field,                     /* (>) The field to be used for the targets */
   CONFIG_ReportFunction 
             ReportFunction,      /* (>) Feedback routine to call */
   void *FunctionArg,             /* (>) Caller-supplied argument */
   StatusType *Status);           /* (!) Inherited status variable */

static void ConfigReadTargetFile( /*    Reads target information from a file */
   char *FileName,                /* (>) The name of the SDS structure file */
   int Field,                     /* (>) The field to be used for the targets */
   CONFIG_ReportFunction 
             ReportFunction,      /* (>) Feedback routine to call */
   void *FunctionArg,             /* (>) Caller-supplied argument */
   StatusType *Status);           /* (!) Inherited status variable */

static void ConfigReleaseMem (void);     /*     Releases any dynamic memory */


/* Function to set a flag to suppress errors for the importonly mode*/
void ConfigSetImportOnlyMode(int value)
{
  ConfigImportOnlyMode = value;
}


/* ------------------------------------------------------------------------- */

/*                    C o n f i g  I n i t  P i v o t s
 *
 *  Function:
 *     Initialises the global pivot information structure.
 *
 *  Description:
 *     This routine reads the global pivot information from an SDS structure
 *     that should contain the latest version of this information. This SDS
 *     structure can be passed as an already open SDS ID, or as the name of
 *     a file containing the structure. If the SDS ID passed to it is zero,
 *     the structure is read from the specified file.
 *
 */

void ConfigInitPivots (
   FpilConstantsType InstConstDetails, /* (>)  Instrument constants details */
   StatusType *Status)              /* (!) Inherited status variable */
{

   /*  Local variables  */

   CONFIG_AllocInfo *AllocInfoPtr;  /* Points to pivot allocation structure */
   unsigned Field;                  /* Index through both fields */
   SdsIdType FieldID;               /* SDS ID for field structure in file */
   int I;                           /* General loop index */
   const int *IPtr;                 /* Pointer to integer item data */
   int IType;                       /* Index through pivot types */
   StatusType LocalStatus;          /* Inherited status we don't care about */
   int NPiv;                        /* Number of pivots in field */
   int NumberTypes;                 /* Number of types in sky count array */
   CONFIG_PivotInfo *PivotInfoPtr;  /* Points to pivot info structure array */
   const short *SPtr;               /* Pointer to short item data */
   char StructName[256];            /* Structure description */
   SdsIdType TestID;                /* Test SDS ID to see if field exists */
   const unsigned int *UIPtr;       /* Pointer to unsigned integer item data */
   void *VPointer;                  /* Pointer to structure item data */
   unsigned NumFields;              /* Number of fields in instrument */

   if (*Status != STATUS__OK) return;

   /*  Initial values */

   FieldID = (SdsIdType) 0;
   LocalStatus = STATUS__OK;

   /*
    *  Get the number of fields.  It is assumed this is less then
    *  FPIL_MAXFIELDS (fpil.h).
    */
   NumFields = FpilGetNumFields(CONFIG_Global.MethodInfo._Instrument);

   /*  The following loop is repeated  once for each field.
    */

   for (Field = 0; Field <  NumFields; Field++) {

      /*  Starting with the top level ID to the structure, work down into the
       *  structure and find the number of pivots for the field. Given
       *  that information, allocate the necessary memory for the internal
       *  arrays that will be used to hold that information. We arbitrarily
       *  use the xPiv array to get the size.
       */

      FpilConFieldId(InstConstDetails, Field, &FieldID, Status);
      if (*Status != STATUS__OK) {
          ErsRep (0,&LocalStatus,
              "Error seeking field %d structure in field constants structure",
                  Field);
         goto Exit;
      }

      /*  Now get the array size from the xPiv array and take that as the
       *  number of pivots and allocate memory for the pivot position
       *  structure and the 'allocation' structure.
       */

      NPiv = 0;
      ConfigAccessItem (FieldID,"xPiv",sizeof(int),StructName,&NPiv,
                                                  &VPointer,0,Status);
      if (*Status != STATUS__OK) goto Exit;

      CONFIG_Global.NumberPivots[Field] = NPiv;
      PivotInfoPtr =
           (CONFIG_PivotInfo *) malloc (NPiv * sizeof(CONFIG_PivotInfo));
      if (PivotInfoPtr == NULL) {
         ErsRep (0,Status,
         "Unable to allocate pivot structure for %d field %d pivots (%d bytes)",
                             Field,NPiv,NPiv * (int) sizeof(CONFIG_PivotInfo));
         *Status = CONFIG__NO_MEMORY;
         goto Exit;
      }
      CONFIG_Global.PivotInfoPtr[Field] = PivotInfoPtr;
      AllocInfoPtr =
           (CONFIG_AllocInfo *) malloc (NPiv * sizeof(CONFIG_AllocInfo));
      if (AllocInfoPtr == NULL) {
         ErsRep (0,Status,
         "Unable to allocate info structure for %d field %d pivots (%d bytes)",
                              Field,NPiv,NPiv * (int) sizeof(CONFIG_AllocInfo));
         *Status = CONFIG__NO_MEMORY;
         goto Exit;
      }
      CONFIG_Global.AllocInfoPtr[Field] = AllocInfoPtr;

      /*  We should clear the allocation information structure, since at
       *  this point nothing is allocated.
       */

      for (I = 0; I < NPiv; I++) {
         AllocInfoPtr[I].TargetIndex = 0;
         AllocInfoPtr[I].X = 0;
         AllocInfoPtr[I].Y = 0;
         AllocInfoPtr[I].Theta = 0.0;
         AllocInfoPtr[I].State = 'U';
         AllocInfoPtr[I].Tentative = 0;
      }

      /*  Now copy the data from the xPiv array into the pivot structure */

      IPtr = (int *) VPointer;
      for (I = 0; I < NPiv; I++) {
         PivotInfoPtr[I].X = IPtr[I];
      }

      /*  Repeat this for the yPiv array */

      ConfigAccessItem (FieldID,"yPiv",sizeof(int),StructName,&NPiv,
                                                  &VPointer,0,Status);
      if (*Status != STATUS__OK) goto Exit;
      IPtr = (int *) VPointer;
      for (I = 0; I < NPiv; I++) {
         PivotInfoPtr[I].Y = IPtr[I];
      }

      /*  And againt for the xPark array */

      ConfigAccessItem (FieldID,"xPark",sizeof(int),StructName,&NPiv,
                                                  &VPointer,0,Status);
      if (*Status != STATUS__OK) goto Exit;
      IPtr = (int *) VPointer;
      for (I = 0; I < NPiv; I++) {
         PivotInfoPtr[I].ParkX = IPtr[I];
      }

      /*  And againt for the yPark array */

      ConfigAccessItem (FieldID,"yPark",sizeof(int),StructName,&NPiv,
                                                  &VPointer,0,Status);
      if (*Status != STATUS__OK) goto Exit;
      IPtr = (int *) VPointer;
      for (I = 0; I < NPiv; I++) {
         PivotInfoPtr[I].ParkY = IPtr[I];
      }

      /*  And again for the tPark array (the Theta values). Actually, we
       *  don't use the tPark array in the sds structure, since we want the 
       *  theta values for the line joining the field center to the 
       *  pivot point, and this is only the same as the tPark value
       *  if the button is parked exactly on the line from field center to
       *  pivot point. This isn't usually the case. The parked theta for
       *  each pivot is given by Dptr[I} in the main loop, where
       *  Dptr = (double *) VPointer; and that is what the code used to use.
       */  

      ConfigAccessItem (FieldID,"tPark",sizeof(double),StructName,&NPiv,
                                                  &VPointer,0,Status);
      if (*Status != STATUS__OK) goto Exit;
      for (I = 0; I < NPiv; I++) {
         PivotInfoPtr[I].Theta = ConfigThetaPiv(PivotInfoPtr[I].X,
                                                    PivotInfoPtr[I].Y);
         
      }
      
      /*  Now we want to set up the 'FibreType' field for the pivot. Note that
       *  in the original 2dF-only version of this program this field was
       *  called 'Spect' and was used to indicate which of the two 
       *  spectrographs was in use. This has now been generalised, and what
       *  it actually contains is a fibre type code as returned by 
       *  FpilConfWhichSpec() (note the historical name for this routine!),
       *  modified if necessary to indicate that the fibre is broken.
       */

      for (I = 0; I < NPiv; I++) {
         unsigned FibreType;
         const char *FibreTypeDescr;
         PivotInfoPtr[I].AllowAllocChange = CONFIG_TRUE;
         FpilConWhichSpec(CONFIG_Global.MethodInfo._Instrument,
                              InstConstDetails,Field, I+1,
                              &FibreType, &FibreTypeDescr,Status);
         PivotInfoPtr[I].FibreType = FibreType;
      }

      /*  The FibreLength field in the pivot information structure is special.
       *  It is taken from the maxExt field in the SDS structre. However, this
       *  structure may not be present for all instruments. If it is not, we
       *  set each fibre length to the maximum length for the instrument.
       *  We need to do this after getting the fibre type, as this may
       *  influence the maximum length. Note also that we don't use the full
       *  fibre length as indicated by this routine - we subtract off a small
       *  allowance to give some clearance to allow for changes due to
       *  atmospheric pressure effects etc.
       */
      
      LocalStatus = 0;
      SdsFind (FieldID,"maxExt",&TestID,&LocalStatus);
      if (LocalStatus == 0) {
         SdsFreeId (TestID,&LocalStatus);
         ConfigAccessItem (FieldID,"maxExt",sizeof(unsigned int),
                                     StructName,&NPiv,&VPointer,0,Status);
         if (*Status != STATUS__OK) goto Exit;
         UIPtr = (unsigned int *) VPointer;
         for (I = 0; I < NPiv; I++) {
            PivotInfoPtr[I].FibreLength = UIPtr[I] - CONFIG_LENGTH_ADJUST;
         }
      } else {
         for (I = 0; I < NPiv; I++) {
            PivotInfoPtr[I].FibreLength = 
                  FpilGetFibreLength (CONFIG_Global.MethodInfo._Instrument,
                         PivotInfoPtr[I].FibreType) - CONFIG_LENGTH_ADJUST;
         }
      }

      /*  The 'inUse' flags are used to disable any pivot, if necessary */

      ConfigAccessItem (FieldID,"inUse",sizeof(short),StructName,&NPiv,
                                                  &VPointer,0,Status);
      if (*Status != STATUS__OK) goto Exit;
      SPtr = (short *) VPointer;
      for (I = 0; I < NPiv; I++) {
         if (!SPtr[I]) {
            PivotInfoPtr[I].FibreType = 
                FpilBrokenFibreType(CONFIG_Global.MethodInfo._Instrument,
                                                    PivotInfoPtr[I].FibreType);
            AllocInfoPtr[I].State = 0;
         }
      }
      
      /*  Finally, free the SDS ID used for this field  */

      SdsFreeId (FieldID,Status);
      FieldID = 0;
      *Status = STATUS__OK;
   }
   
   /*  Reset all the sky target counts for each fibre type. Setting them to
    *  -1 means that no attempt will be made to distinguish between sky and
    *  non-sky types for each type. Setting the types to -1 means that these
    *  are not valid type codes anyway.
    */
    
   NumberTypes = sizeof(CONFIG_Global.FibreTypesForSky)/sizeof(int);
   for (IType = 0; IType < NumberTypes; IType++) {
      CONFIG_Global.FibreTypesForSky[IType] = -1;
      CONFIG_Global.FibreSkyCounts[IType] = -1;
   }
   CONFIG_Global.FibreSkyEntries = 0;
   
   /*  On the way out, release any SDS ID's that we still have active.
    *  If there has been an error, release any memory allocated for the
    *  pivot information structures.
    */

Exit:;

   if (FieldID != (SdsIdType) 0) {
      LocalStatus = STATUS__OK;
      SdsFreeId (FieldID,&LocalStatus);
   }
   if (*Status != STATUS__OK) {
      for (Field = 0; Field <  FPIL_MAXFIELDS; Field++) {
         if (CONFIG_Global.PivotInfoPtr[Field] != NULL) {
            free (CONFIG_Global.PivotInfoPtr[Field]);
            CONFIG_Global.PivotInfoPtr[Field] = NULL;
         }
         if (CONFIG_Global.AllocInfoPtr[Field] != NULL) {
            free (CONFIG_Global.AllocInfoPtr[Field]);
            CONFIG_Global.AllocInfoPtr[Field] = NULL;
         }

      }
   }

}

/* -------------------------------------------------------------------------- */

/*              C o n f i g  R e a d  T a r g e t  F i l e
 *
 *  Function:
 *     Reads the global target information structure from a file.
 *
 *  Description:
 *     This routine reads the global target information from an SDS file.
 *
 */

void ConfigReadTargetFile (
   char *FileName,                /* (>) The name of the SDS structure file */
   int Field,                     /* (>) The field to be used for the targets */
   CONFIG_ReportFunction 
             ReportFunction,      /* (>) Feedback routine to call */
   void *FunctionArg,             /* (>) Caller-supplied argument */
   StatusType *Status)            /* (!) Inherited status variable */
{
   /*  Local variables */

   StatusType LocalStatus;        /* Inherited status we choose to ignore */
   char Source[256];              /* Structure description */
   SdsIdType TopID;               /* SDS ID for top of structure in file */

   if (*Status != STATUS__OK) return;

   /*  Initial values */

   TopID = (SdsIdType) 0;

   /*  Most of the work is done by the routine ConfigInitTargets(), which
    *  works from an SDS ID for the top item in the target information
    *  structure. In this case, where the structure is in a file, we have
    *  to read it in first, then pass the ID we get to ConfigInitTargets()
    */

   LocalStatus = STATUS__OK;
   SdsRead (FileName,&TopID,Status);
   if (*Status != STATUS__OK) {
      ErsRep (0,&LocalStatus,"Error reading target allocation file '%s'", 
                                                                  FileName);
      goto Exit;
   }
   strcpy (Source,"target information file ");
   strncat (Source,FileName,sizeof(Source) - strlen(Source) - 1);

   ConfigInitTargets (TopID,Source,Field,ReportFunction,FunctionArg,Status);

   if (*Status == CONFIG__CANCELLED)
      ErsFlush(Status); 

Exit:;

   /*  On the way out, if the SDS ID for the top of the structure was
    *  allocated, release it.
    */

   if (TopID != (SdsIdType) 0) {
      LocalStatus = STATUS__OK;
      SdsReadFree(TopID, &LocalStatus);
      SdsFreeId (TopID,&LocalStatus);
   }

}

/* -------------------------------------------------------------------------- */

/*+
 *                    C o n f i g  I n i t  T a r g e t s
 *
 *  Function:
 *     Initialises the global target information structure.
 *
 *  Description:
 *     This routine reads the global target information from an SDS structure.
 *     This may have been read from a file, or may have been sent in a
 *     message. The structure should match that defined in the document
 *     '2dF Field Configuration File Structure Definition', although only
 *     a small subset of the fields defined in that are actually used
 *     by this program. This routine is passed the SDS ID for the top item in
 *     the structure, and it then searches the structure for what it needs.
 *-
 *  Global variable use (in CONFIG_Global):
 *
 *     TargetInfoPtr  If non-null when the routine is called, the memory
 *                    it points to is released. Memory for a new array of
 *                    Target information structures is allocated, and the
 *                    address of this array is set in TargetInfoPtr. The
 *                    elements of this array are initialised by this routine.
 *                    The contents of this array should not be changed by
 *                    other routines.
 *     NumberTargets  Is set to the number of targets.
 *     Field          Is set to the value of the Field parameter passed to
 *                    this routine.
 *     AllocInfoPtr   The allocation information for the field in question
 *                    (whose address is held in AllocInfoPtr[Field] in the
 *                    global variables) is reset so that it reflects only the
 *                    allocations, if any, specified in the SDS structure.
 *     PivotInfoPtr   The pivot information for the  field in question
 *                    (whose address is held in PivotInfoPtr[Field] in the
 *                    global variables) is examined so that allocations can
 *                    be checked against broken fibres and for incompatible
 *                    fibre assignments (guide fibres to program objects etc.)
 *     NumberPivots   The number of pivots for the field in question is 
 *                    obtained from the global variable NumberPivots[Field].
 *+
 */

void ConfigInitTargets (
   SdsIdType TopID,               /* (>) ID of top item in the SDS structure */
   char *Source,                  /* (>) Description of structure origin - only
                                   *     needed for error messages. */
   int Field,                     /* (>) The field to be used for the targets*/
   CONFIG_ReportFunction 
   ReportFunction DUNUSED/* Why */,/* (>) Feedback routine to call */
   void *FunctionArg DUNUSED,     /* (>) Caller-supplied argument */
   StatusType *Status)            /* (!) Inherited status variable */
{

   /*  Local variables  */

   CONFIG_Boolean Allocated;        /* True if target or guide is allocated */
   char *AllocatedPtr;              /* Points to allocated array data */
   int AllocatedTargets;            /* Number of already allocated targets */
   CONFIG_AllocInfo *AllocInfoPtr;  /* Points to allocation info structures */
   int I;                           /* General loop index */
   const short *IndexPtr;           /* Points to data in index array */
   SdsIdType ItemID;                /* SDS ID for any general item in file */
   StatusType LocalStatus;          /* Inherited status we don't care about */
   int NPiv;                        /* Number of pivots in objects structure */
   int NTargets[2];                 /* Number of described guides & objects */
   SdsIdType ObjectID;              /* SDS ID for objects structure in file */
   char ObjectsName[128];           /* Description of objects structure */
   CONFIG_PivotInfo *PivotInfoPtr;  /* Points to pivot info structure array */
   const short *PriorityPtr;        /* Points to data in priority array */
   const char *SpectPtr;            /* Points to data in spectrograph array */
   SdsIdType StructID;              /* SDS ID for unalloc structures in file */
   char StructName[128];            /* Describes the unallocated structures */
   char *StructNamePtr[2];          /* Names of the unallocated structures */
   int TargetIndex;                 /* Index into target info array */
   CONFIG_TargetInfo *TargetInfoPtr;/* Points to target info structure array */
   int TargetType;                  /* 0 for guides, 1 for objects */
   const double *ThetaPtr;          /* Points to theta array data */
   int TotalTargets;                /* Total number of targets in file */
   const char *TypePtr;             /* Points to a 'type' array data */
   const int *XPtr;                 /* Points to data in X array */
   const int *YPtr;                 /* Points to data in Y array */
   const char *NPtr;                /* Points to data in name array */
   char UnallocatedType;            /* Type code used for unallocated objects */
   int UnallocTargets[2];           /* Targets shown as unallocated */
   int UnlistedTargets;             /* Targets pre-allocated & not in lists */
   void *VPointer;                  /* Pointer to structure item data */
   FpilType Instrument;             /* Instrument description pointer */

   if (*Status != STATUS__OK) return;

   /*  Get easy access to the FPIL instrument description. */
    
   Instrument = CONFIG_Global.MethodInfo._Instrument;

   /*  Initial values */

   ItemID = ObjectID = StructID = (SdsIdType) 0;
   IndexPtr = NULL;
   TypePtr = NULL;
   TargetIndex = 0;
   UnallocatedType = FpilUnallocatedType(CONFIG_Global.MethodInfo._Instrument);
                 
   /*  Get any global values that will be used */

   NPiv = CONFIG_Global.NumberPivots[Field];
   AllocInfoPtr = CONFIG_Global.AllocInfoPtr[Field];
   PivotInfoPtr = CONFIG_Global.PivotInfoPtr[Field];

   /*  Free any memory already allocated for target information */

   if (CONFIG_Global.TargetInfoPtr != NULL) {
      free (CONFIG_Global.TargetInfoPtr);
      CONFIG_Global.TargetInfoPtr = NULL;
   }

   /*  This is all a little tricky, since the information about the
    *  possible targets is scattered through three structures. The
    *  'objects' structure can contain information about already allocated
    *  objects. The 'unallocObjects' structure contains information about
    *  unallocated target objects, and the the 'unallocGuide' structure
    *  contains information about the unallocated guide stars. Since a
    *  'deallocate' command may release an allocated object back into the
    *  target list, we make the target list big enough to hold all the
    *  allocated and unallocated objects. This means we have to find all
    *  these. 
    *
    *  The whole business is complicated by the fact that the configuration
    *  interface task makes use of the 'allocated' arrays in the two
    *  'unallocObjects' and 'unallocGuide' structures, together with the
    *  'index' array in the 'objects' structure to allow objects to be
    *  allocated and deallocated interactively without having to constantly
    *  resize all the arrays. This means that we if the structure has come
    *  from a file, we should expect to see none of these arrays (the two
    *  'allocated' and the one 'index'), and if it comes from the interface,
    *  we should expect to see all three. This matters because in the case
    *  where the arrays are present, targets shown as allocated in the 
    *  'objects' structure will also appear in the 'unallocXXX' structures,
    *  and so a simple look at the size of the arrays doesn't give us an
    *  immediate target count.
    *
    *  We start with a preliminary look through the allocated
    *  objects - the 'type' field indicates which are allocated. If there
    *  is an 'index' array, these will also be in the 'unalloc' structures.
    *  So we have to be careful with the 'index' array,  since it may not be
    *  present. We allow for the case where there is no 'objects' structure,
    *  or where it is empty, and treat this as a case where there are no
    *  objects already allocated.
    */

   AllocatedTargets = 0;
   UnlistedTargets = 0;

   SdsFind (TopID,"objects",&ObjectID,Status);
   SdsIndex (ObjectID,1,&ItemID,Status);
   if (*Status != STATUS__OK) {
      *Status = STATUS__OK;
      if (ObjectID != (SdsIdType) 0) SdsFreeId (ObjectID,Status);
      if (ItemID   != (SdsIdType) 0) SdsFreeId (ItemID,Status);
      ObjectID = (SdsIdType) 0;
   } else {

      /*  "objects" exists  */
      
      SdsFreeId(ItemID, Status);
      ItemID = 0;
      strcpy (ObjectsName,"objects in ");
      strncat (ObjectsName,Source,sizeof(ObjectsName) 
                                          - strlen(ObjectsName) - 1);
      ConfigAccessItem (ObjectID,"type",sizeof(char),ObjectsName,&NPiv,
                                                  &VPointer,0,Status);
      if (*Status != STATUS__OK) goto Exit;
      TypePtr = (char *) VPointer;

      LocalStatus = STATUS__OK;
      
      /*  Try to find an "index" item. */
      
      SdsFind (ObjectID,"index",&ItemID,&LocalStatus);
      if (LocalStatus == STATUS__OK) {
         SdsFreeId (ItemID,&LocalStatus);
         ItemID = 0;
         ConfigAccessItem (ObjectID,"index",sizeof(short),ObjectsName,&NPiv,
                                                   &VPointer,0,Status);
         if (*Status != STATUS__OK) goto Exit;
         IndexPtr = (short *) VPointer;
      } else {
         IndexPtr = NULL;
      }

      for (I = 0; I < NPiv; I++) {
          /* *** Code originally also tested here for (TypePtr[I] != 0). I've
           * removed this CHECK */
          if (TypePtr[I] != UnallocatedType) {
             AllocatedTargets++;
             if (IndexPtr == NULL) {
                 printf("ConfigInitTarget:Errant code hit\n");
                 UnlistedTargets++ /*AllocatedTargets++*/;
             } else {
                 if (IndexPtr[I] < 0) 
                     UnlistedTargets++;
             }
          }
      }
   }
   
   /*  So if we have an "objects" item, we now have a count of the number
    *  of allocated items (AllocatedTargets) and the number of those targets
    *  which should not have entries in the unallocated area (UnlistedTargets).
    *
    *  Starting with the top level ID to the structure, work down into the
    *  two structures that contain details of the unallocated objects and
    *  guide stars.  The main thing we need to know is the total number of
    *  targets. We get this by looking at the 'x' arrays in both the
    *  'unallocGuide" and "unallocObject" structures. If either of these
    *  structures don't exist, we assume there are no unallocated targets
    *  of that type. If the structure exists, we also check for the case
    *  where it is empty. Only if it exists and has components do we look
    *  for the 'x' array.
    */

   StructNamePtr[0] = "unallocGuide";
   StructNamePtr[1] = "unallocObject";

   for (TargetType = 0; TargetType < 2; TargetType++) {

      NTargets[TargetType] = 0;
      SdsFind (TopID,StructNamePtr[TargetType],&StructID,Status);
      if (*Status == STATUS__OK) {
         StatusType ignore = STATUS__OK;
         SdsIndex (StructID,1,&ItemID,Status);
         if (*Status == STATUS__OK) {
            SdsFreeId (ItemID,Status);
            sprintf (StructName,"'%s' in ",StructNamePtr[TargetType]);
            strncat (StructName,Source,sizeof(StructName) - 
                                                  strlen(StructName) - 1);

            ConfigAccessItem (StructID,"x",sizeof(int),StructName,
                          &NTargets[TargetType],&VPointer,0,Status);
         }
         SdsFreeId(StructID, &ignore);
         StructID = 0;
      }
      else
          *Status = STATUS__OK;
   }

   /*  Now that we know the number of unallocated targets, as well as
    *  the number of allocated objects, we can allocate the memory needed
    *  to hold the target information.
    */

   TotalTargets = NTargets[0] + NTargets[1] + UnlistedTargets;

   TargetInfoPtr = (CONFIG_TargetInfo *)
                       malloc (TotalTargets * sizeof(CONFIG_TargetInfo));
   if (TargetInfoPtr == NULL) {
      ErsRep (0,Status,
         "Unable to allocate target structure for %d targets (%d bytes)",
                   TotalTargets,TotalTargets * (int) sizeof(CONFIG_TargetInfo));
      *Status = CONFIG__NO_MEMORY;
      goto Exit;
   }
   CONFIG_Global.TargetInfoPtr = TargetInfoPtr;
   CONFIG_Global.NumberTargets = TotalTargets;
   CONFIG_Global.Field = Field;

   /*  We  clear out the newly allocated target information structure, and
    *  also clear out the allocation information structure for the field in
    *  question. (Note that we don't change the state of a broken pivot!)
    */

   for (I = 0; I < TotalTargets; I++) {
      TargetInfoPtr[I].X = 0;
      TargetInfoPtr[I].Y = 0;
      TargetInfoPtr[I].Type = 0;
      TargetInfoPtr[I].PreAllocated = CONFIG_FALSE;
      TargetInfoPtr[I].Priority = 0;
      TargetInfoPtr[I].Spect = 0;
      TargetInfoPtr[I].ObjName = (char *)NULL;
   }
   for (I = 0; I < NPiv; I++) {
      AllocInfoPtr[I].X = 0;
      AllocInfoPtr[I].Y = 0;
      AllocInfoPtr[I].TargetIndex = 0;
      AllocInfoPtr[I].Theta = 0.0;
      AllocInfoPtr[I].Tentative = 0;
      if (FpilIsFibreBroken(Instrument,PivotInfoPtr[I].FibreType)) {
           AllocInfoPtr[I].State = 0;
      } else {
           AllocInfoPtr[I].State = 'U';
      }
   }

   /*  Now that we have the target information arrays allocated, we have
    *  to fill them up.  We start with the targets already allocated,
    *  described in the 'Objects' structure - we already have ObjectID
    *  pointing to this. The only ones we're interested in are those
    *  where the type field shows as allocated. TypePtr already points to
    *  the type array, and NPiv already holds the expected size of each
    *  array. We start by getting pointers to the arrays we're interested
    *  in. IndexPtr already points to the index array, or is null if there
    *  was no such array. If ObjectID is zero, there was no "objects"
    *  structure (or it was empty) and we treat all objects as unallocated.
    */

   if (ObjectID != (SdsIdType) 0) {
      ConfigAccessItem (ObjectID,"name",80,ObjectsName,&NPiv,&VPointer,
                                                              0,Status);
      if (*Status != STATUS__OK) goto Exit;
      NPtr = (char *) VPointer;

      ConfigAccessItem (ObjectID,"x",sizeof(int),ObjectsName,&NPiv,
                                                  &VPointer,0,Status);
      if (*Status != STATUS__OK) goto Exit;
      XPtr = (int *) VPointer;

      ConfigAccessItem (ObjectID,"y",sizeof(int),ObjectsName,&NPiv,
                                                  &VPointer,0,Status);
      if (*Status != STATUS__OK) goto Exit;
      YPtr = (int *) VPointer;

      ConfigAccessItem (ObjectID,"theta",sizeof(double),ObjectsName,&NPiv,
                                                  &VPointer,0,Status);
      if (*Status != STATUS__OK) goto Exit;
      ThetaPtr = (double *) VPointer;

      ConfigAccessItem (ObjectID,"spectrograph",sizeof(char),ObjectsName,&NPiv,
                                                  &VPointer,0,Status);
      if (*Status != STATUS__OK) goto Exit;
      SpectPtr = (char *) VPointer;

      ConfigAccessItem (ObjectID,"priority",sizeof(short),ObjectsName,&NPiv,
                                                  &VPointer,0,Status);
      if (*Status != STATUS__OK) goto Exit;
      PriorityPtr = (short *) VPointer;

      /*  Now, we go through through all the pivots and for those that are
       *  allocated, we put information about the target into the global
       *  target information structure, and information about the allocation
       *  into the global allocation structure. If these do not come from
       *  either list of unallocated objects, they are given an original 
       *  index of -1.
       */

      for (I = 0; I < NPiv; I++) {
         if ((TypePtr[I] != UnallocatedType) && (TypePtr[I] != 0)) {
            TargetInfoPtr[TargetIndex].X = XPtr[I];
            TargetInfoPtr[TargetIndex].Y = YPtr[I];
            if (IndexPtr == NULL) {
               TargetInfoPtr[TargetIndex].OriginalIndex = -1;
            } else {
               TargetInfoPtr[TargetIndex].OriginalIndex = IndexPtr[I];
            }
            TargetInfoPtr[TargetIndex].PreAllocated = CONFIG_TRUE;
            TargetInfoPtr[TargetIndex].Type = TypePtr[I];
            TargetInfoPtr[TargetIndex].Priority = PriorityPtr[I];
            TargetInfoPtr[TargetIndex].Spect = SpectPtr[I];
            TargetInfoPtr[TargetIndex].ObjName = (char *)malloc(80);
            if (TargetInfoPtr[TargetIndex].ObjName == NULL) {
               ErsRep (0,Status,
                   "Unable to allocate ObjName structure of 80 bytes");
               *Status = CONFIG__NO_MEMORY;
               goto Exit;
            }
            strncpy(TargetInfoPtr[TargetIndex].ObjName,NPtr+(I*80),80);
            AllocInfoPtr[I].TargetIndex = TargetIndex;
            AllocInfoPtr[I].X = XPtr[I];
            AllocInfoPtr[I].Y = YPtr[I];
            AllocInfoPtr[I].Theta = ThetaPtr[I];
            AllocInfoPtr[I].State = TypePtr[I] ? 'A' : 0;
            TargetIndex++;

            /*  Make sure that the allocation is valid and warn if there's a 
             *  problem. It isn't clear if bad status should be returned under
             *  these circumstances.
             *  Note that we use I as the index into all of TypePtr, SpectPtr
             *  and PivotInfoPtr. TypePtr and SpectPtr point to the Type and
             *  Spect arrays in the "objects" structure, while PivotInfoPtr
             *  points to the overall array of structures describing each
             *  pivot. We can use the same index because there is a one to one
             *  correspondence between items in the "objects" structure and
             *  the pivots.
             *  Another note: GBD took out the error reporting of broken fibre
             *  allocations, commenting that these really shouldn't generate
             *  errors at this stage of the process. I (KS) am not convinced of
             *  that, but we should note that the use of the Fpil routine
             *  FpilTargetPivotCompatible() at this point now effectively
             *  re-introduces that error reporting.
             */

            if (!FpilTargetPivotCompatible(Instrument,
                         TypePtr[I],SpectPtr[I],PivotInfoPtr[I].FibreType)) {
               ErsRep (0,&LocalStatus,
                      "Fibre %d allocated to an incompatible target",I+1);
               ErsRep (0,&LocalStatus,
                      "Fibre type was %d, target type %d, target spect %d",
                      PivotInfoPtr[I].FibreType,TypePtr[I],SpectPtr[I]);
               ErsRep (0,&LocalStatus,
                 "Allocation specified in 'objects' structure in '%s'",Source);
            }
         }
      }
   }

   /*  Now we go through the unallocated objects, held in the structures
    *  'unallocGuide' and 'unallocObject'. We do this, as before, as a loop
    *  through these two structures. We already have set up StructNamePtr[]
    *  and NTargets[].
    */

   for (TargetType = 0; TargetType < 2; TargetType++) {
       
      UnallocTargets[TargetType] = 0;
      SdsFind (TopID,StructNamePtr[TargetType],&StructID,Status);
      if ((*Status == STATUS__OK) && (NTargets[TargetType] > 0)) {
         StatusType ignore = STATUS__OK;
         sprintf (StructName,"'%s' in ",StructNamePtr[TargetType]);
         strncat (StructName,Source,sizeof(StructName) 
                                                 - strlen(StructName) - 1);
         ConfigAccessItem (StructID,"name",80,StructName,
                                &NTargets[TargetType],&VPointer,0,Status);
         if (*Status != STATUS__OK) goto Exit;
         NPtr = (char *) VPointer;

         ConfigAccessItem (StructID,"x",sizeof(int),StructName,
                            &NTargets[TargetType],&VPointer,0,Status);
         if (*Status != STATUS__OK) goto Exit;
         XPtr = (int *) VPointer;

         ConfigAccessItem (StructID,"y",sizeof(int),StructName,
                            &NTargets[TargetType],&VPointer,0,Status);
         if (*Status != STATUS__OK) goto Exit;
         YPtr = (int *) VPointer;

         ConfigAccessItem (StructID,"type",sizeof(char),StructName,
                            &NTargets[TargetType],&VPointer,0,Status);
         if (*Status != STATUS__OK) goto Exit;
         TypePtr = (char *) VPointer;

         ConfigAccessItem (StructID,"priority",sizeof(short),StructName,
                            &NTargets[TargetType],&VPointer,0,Status);
         if (*Status != STATUS__OK) goto Exit;
         PriorityPtr = (short *) VPointer;

         /*  The 'allocated' array is different, since it does not appear in
          *  all instances of these structures - ones saved to file as final
          *  or intermediate configurations don't have it - so we need to check
          *  on its presence first since we don't want ConfigAccessItem to
          *  complain about its being missing.
          */

         LocalStatus = STATUS__OK;
         SdsFind (StructID,"allocated",&ItemID,&LocalStatus);
         if (LocalStatus == STATUS__OK) {
            SdsFreeId (ItemID,&LocalStatus);
            ConfigAccessItem (StructID,"allocated",sizeof(char),StructName,
                            &NTargets[TargetType],&VPointer,0,Status);
            if (*Status != STATUS__OK) goto Exit;
            AllocatedPtr = (char *) VPointer;
         } else {
            AllocatedPtr = NULL;
         }

         /*  The 'spectrograph' array doesn't appear in the unallocated Guide
          *  structure.
          */

         if (TargetType == 0) {
            SpectPtr = NULL;
         } else {
            ConfigAccessItem (StructID,"spectrograph",sizeof(char),StructName,
                            &NTargets[TargetType],&VPointer,0,Status);
            if (*Status != STATUS__OK) goto Exit;
            SpectPtr = (char *) VPointer;
         }

         /*  Now, update the target information structures from what we have
          *  here. If there was an 'allocated' array, use that, if not assume
          *  all targets are unallocated. If the target is present but flagged
          *  as allocated, this means it was pre-allocated and will already
          *  be in the target list, having been put there when the objects
          *  structure was searched. If we find that we are getting more targets
          *  than we expected this indicates an error in the SDS structure, but
          *  we continue to run through to pick up some useful diagnostics such
          *  as the total number of unallocated targets.
          */

         for (I = 0; I < NTargets[TargetType]; I++) {
            if (AllocatedPtr == NULL) {
               Allocated = CONFIG_FALSE;
            } else {
               Allocated = AllocatedPtr[I];
            }
            if (!Allocated) {
               UnallocTargets[TargetType]++;
               if (TargetIndex < TotalTargets) {
                  TargetInfoPtr[TargetIndex].X = XPtr[I];
                  TargetInfoPtr[TargetIndex].Y = YPtr[I];
                  TargetInfoPtr[TargetIndex].OriginalIndex = I;
                  TargetInfoPtr[TargetIndex].Type = TypePtr[I];
                  TargetInfoPtr[TargetIndex].Priority = PriorityPtr[I];
                  if (SpectPtr == NULL) {
                     TargetInfoPtr[TargetIndex].Spect = 0;
                  } else {
                     TargetInfoPtr[TargetIndex].Spect = SpectPtr[I];
                  }
                  TargetInfoPtr[TargetIndex].ObjName = (char *)malloc(80);
                  if (TargetInfoPtr[TargetIndex].ObjName == NULL) {
                     ErsRep (0,Status,
                         "Unable to allocate ObjName structure of 80 bytes");
                     *Status = CONFIG__NO_MEMORY;
                     goto Exit;
                  }
                  strncpy(TargetInfoPtr[TargetIndex].ObjName,NPtr+(80*I),80);
               }
               TargetIndex++;
            }
         }
         SdsFreeId(StructID, &ignore);
         StructID = 0;
      }
      *Status = STATUS__OK;
   }
   if (TargetIndex != TotalTargets) {
      ErsRep (0,Status,"Internal inconsistency in SDS structure.");
      ErsRep (0,Status,"%s %d %s %d %s",
              "Objects sub-structure shows",AllocatedTargets,
              "allocated targets, of which",UnlistedTargets,
              "are not indexed into the target lists.");
      ErsRep (0,Status,
            "So %d objects should be shown as unallocated in the target lists.",
                                            AllocatedTargets - UnlistedTargets);
      ErsRep (0,Status,"%s %d %s %d %s",
              "But the target lists show ",UnallocTargets[0],
              "unallocated guide objects and ",UnallocTargets[1],
              "unallocated target objects.");
      *Status = CONFIG__INVAL_CONFIG;
      goto Exit;
   }

   /*  If there were any targets already allocated, we ought to check that these
    *  allocations are valid.  We do this by doing a field check at this point.
    */

/*   if (AllocatedTargets > 0) {
      ConfigFieldCheck (ReportFunction,FunctionArg,Status);
   } ******GBD taken this call out*****/



   /*
    *  Catch any objects outside the field and set the priority to
    *  CONFIG_PRIO_INVALID to ensure they are not configured.
    *
    *  Also count the number of guide and program objects so that
    *  we can deallocate already allocated fibres.
    */
   CONFIG_Global.NUnGuide = 0;
   CONFIG_Global.NUnTarget = 0;
   for (I = 0; I < CONFIG_Global.NumberTargets; I++){
       if (!FpilOnField(Instrument,
                         CONFIG_Global.TargetInfoPtr[I].X,
                         CONFIG_Global.TargetInfoPtr[I].Y)) {
           CONFIG_Global.TargetInfoPtr[I].Priority = CONFIG_PRIO_INVALID;
       }
       if (CONFIG_Global.TargetInfoPtr[I].OriginalIndex != -1){
           if (CONFIG_Global.TargetInfoPtr[I].Type == 'F'){
               CONFIG_Global.NUnGuide++;
           } else {
               CONFIG_Global.NUnTarget++;
           }
       }
   }

   /*  On the way out, free any allocated SDS IDs, and if there was an
    *  error, release the memory allocated for the target arrays.
    */
Exit:;

   if (ItemID != (SdsIdType) 0) {
      LocalStatus = STATUS__OK;
      SdsFreeId (ItemID,&LocalStatus);
   }
   if (ObjectID != (SdsIdType) 0) {
      LocalStatus = STATUS__OK;
      SdsFreeId (ObjectID,&LocalStatus);
   }
   if (StructID != (SdsIdType) 0) {
      LocalStatus = STATUS__OK;
      SdsFreeId (StructID,&LocalStatus);
   }
   if (*Status != STATUS__OK) {
      if (CONFIG_Global.TargetInfoPtr != NULL) {
         if (TargetIndex != 0){
            for (I=0; I < TargetIndex; I++)
                free (CONFIG_Global.TargetInfoPtr[I].ObjName);
         }
         free (CONFIG_Global.TargetInfoPtr);
         CONFIG_Global.TargetInfoPtr = NULL;
      }
   }

}

/* -------------------------------------------------------------------------- */

/*                    C o n f i g  A c c e s s  I t e m
 *
 *  Function:
 *     Accesses an array in an SDS structure.
 *
 *  Description:
 *     This is a utility routine used to access an array in an SDS structure,
 *     reporting the size of the array, checking its type (strictly the size
 *     of its elements) and returning a pointer to the data in it. It is
 *     passed the name of the array and the SDS ID of the structure that
 *     holds it. It also needs a description of the structure to use for
 *     error reporting. If there is an error the inherited status variable
 *     will reflect this, and the ItemID returned will be zero, which is
 *     never a valid SDS ID. If the number of elements is passed as a non
 *     zero value, this is taken to be the expected number of elements, and
 *     the actual number of elements is checked against this.
 */

void ConfigAccessItem (
   SdsIdType StructID,           /* (>) SDS ID of enclosing structure */
   char *Name,                   /* (>) Name of array in structure */
   int ElementSize,              /* (>) Expected element size in bytes */
   char *StructName,             /* (>) Description of structure - for errors */
   int *Elements,                /* (!) Passes & receives # elements in array */
   void **VPointer,              /* (<) Receives pointer to array data */
   SdsIdType *ItemID,            /* (<) Receives SDS ID of array item 
                                        If a NULL pointer is passed, the
                                        value is not returned.  IF you pass
                                        a pointer here you are responsible
                                        for calling SdsFreeId() on the ID.
                                        Note that if you modify the data 
                                        pointed to by VPointer, you should
                                        call SdsFlush() on this id to update
                                        the date.*/
   StatusType *Status)           /* (!) Inherited status */
{
   /*  Local variables  */

   SdsCodeType Code;                /* Type code for structure item - ignored */
   unsigned long Dims[10];          /* Dimensions of structure object */
   char ItemName[20];               /* Name of item in structure - ignored */
   unsigned long Length;            /* Size of structure item in bytes */
   StatusType LocalStatus;          /* Inherited status we don't care about */
   long NDims;                      /* Number of dimensions for object */
   SdsIdType ID;                    /* Local version of item id         */

   if (*Status != STATUS__OK) return;

   ID = 0;
   LocalStatus = STATUS__OK;
   SdsFind (StructID,Name,&ID,Status);
   if (*Status != STATUS__OK) {
      ErsRep (0,&LocalStatus,
         "Error looking for '%s' array in structure %s",Name,StructName);
      goto Exit;
   }
   SdsInfo (ID,ItemName,&Code,&NDims,Dims,Status);
   if (*Status != STATUS__OK) {
      ErsRep (0,&LocalStatus,
         "Error getting size of '%s' array in structure %s",Name,StructName);
      goto Exit;
   }
   if (((NDims != 1) && (strcmp(Name,"name") != 0)) || 
        (NDims > 2))  {
      ErsRep (0,Status,
        "Unexpected size for '%s' array in structure %s: ndim = %ld, dim = %ld",
                                                Name,StructName,NDims,Dims[0]);
      *Status = CONFIG__INVAL_DIMS;
      goto Exit;
   }
   if (*Elements == 0) {
      *Elements = Dims[0];
   } else {
      if ((*Elements != (long)Dims[0]) && 
          (strcmp(Name, "name") == 0 && (*Elements != (long)Dims[1]))){
         ErsRep (0,Status,
         "Unexpected size for '%s' array in structure %s: expected %d, got %ld",
                                            Name,StructName,*Elements,Dims[0]);
         *Status = CONFIG__INVAL_SIZE;
         goto Exit;
      }
   }
   SdsPointer (ID,VPointer,&Length,Status);
   if (*Status != STATUS__OK) {
      ErsRep (0,&LocalStatus,
             "Error accessing '%s' array in structure %s",Name,StructName);
      goto Exit;
   }
   if ((long)Length != (*Elements * ElementSize)) {
      ErsRep (0,Status,
          "'%s' array in structure %s has %ld bytes: expected %d",
                            Name,StructName,Length,(*Elements * ElementSize));
      *Status = CONFIG__INVAL_SIZE;
      goto Exit;
   }

/*
 * If user wants the SDS ID, return it and return immediately to avoid
 * freeing the SDS ID below.
 */
   if (ItemID) 
   {
       *ItemID = ID;
       return;
   }

Exit:;
   if (ID)
   {
/*
 *     Free Sds ID if required.  If ItemID is true at this point, we have
 *     an error and should ensure the value pointed to is 0.
 */
       StatusType ignore = STATUS__OK;
       SdsFreeId(ID, &ignore);
       if (ItemID) 
           *ItemID = 0;
   }
}

/* -------------------------------------------------------------------------- */

/*                    C o n f i g  R e l e a s e  M e m
 *
 *  Function:
 *     Releases any memory allocated by the program.
 *
 *  Description:
 *     This routine is called at the end of the program in order to give 
 *     it a chance to release any dynamic memory that may have been allocated.
 *     Any such memory should have its address recorded in the global data
 *     structures.
 */

void ConfigReleaseMem (void) 
{
   /*  Local variables  */

   int Field;                        /* Loop index through the two fields */

   /*  Check all the dynamically allocated memory addresses, and free
    *  any that are still allocated.
    */

   if (CONFIG_Global.TargetInfoPtr != NULL) {
      free (CONFIG_Global.TargetInfoPtr);
      CONFIG_Global.TargetInfoPtr = NULL;
   }
   for (Field = 0; Field <  FPIL_MAXFIELDS; Field++) {
      if (CONFIG_Global.AllocInfoPtr[Field] != NULL) {
         free (CONFIG_Global.AllocInfoPtr[Field]);
         CONFIG_Global.AllocInfoPtr[Field] = NULL;
      }
      if (CONFIG_Global.PivotInfoPtr[Field] != NULL) {
         free (CONFIG_Global.PivotInfoPtr[Field]);
         CONFIG_Global.PivotInfoPtr[Field] = NULL;
      }
   }

}

/* -------------------------------------------------------------------------- */

/*                    C o n f i g  M a x  E x t e n s i o n
 *
 *  Function:
 *     Returns the maximum extension of a fibre.
 *
 *  Description:
 *     This is a utility routine intended for use by the method specific code.
 *     It returns the maximum extension of a fibre, in microns. Note that
 *     not all fibres will have the same maximum length - this routine
 *     returns the length of the longest fibre. The FibreLength field of the
 *     Pivot information structure should be used to get the length of any
 *     individual fibre.
 */

static long ConfigMaxExtension(const CONFIG_MethodInfo * MethodInfo)
{
   /*  The value is instrument dependent and so is handled by an Fpil routine.
    */
   
   return(FpilGetMaxExtension(MethodInfo->_Instrument));
}

/* -------------------------------------------------------------------------- */

/*                    C o n f i g  G e t  V i r t  P i v
 *
 *  Function:
 *     Locates the virtual pivot point for a button.
 *
 *  Description:
 *     This is a utility routine intended for use by the method specific code.
 *     Given the position of a button - ie the x,y position of its origin, which
 *     is the position of the fibre end, and its theta value (the angle measured
 *     anticlockwise from the vertical through that origin to the fibre) returns
 *     the coordinates of the virtual pivot point - the point at the end where
 *     the fibre is assumed to emerge.
 */

static void ConfigGetVirtPiv (
   const CONFIG_MethodInfo * MethodInfo DUNUSED,
   double OriginX,               /* (>) The x-coordinate of the button origin */
   double OriginY,               /* (>) The y-coordinate of the button origin */
   double Theta,                 /* (>) The theta angle for the button */
   double *VirtPivX,             /* (<) The x-coordinate of the virtual pivot */
   double *VirtPivY)             /* (<) The y-coordinate of the virtual pivot */
{

   /*  Local variables  */

   FpilGetVirtPiv(MethodInfo->_Instrument, OriginX, OriginY,
                         Theta, VirtPivX, VirtPivY);
}


/* -------------------------------------------------------------------------- */

/*                    C o n f i g  C h e c k  A l p h a
 *
 *  Function:
 *     Checks that a possible allocation against the maximum pivot angle.
 *
 *  Description:
 *     This is a utility routine intended for use by the method specific code.
 *     It is passed the details of a pivot and a possible target. It 
 *     calculates the non-radial pivot angle (alpha) - the angle the fibre
 *     will have to bend at at the pivot, and checks this against the maximum
 *     pivot angle allowed. (This code has been taken from James Wilcox's
 *     configuration validation code and modified slightly to fit into the
 *     structure of this program.)
 */

static CONFIG_Boolean ConfigCheckAlpha (
   const CONFIG_MethodInfo * MethodInfo,
   const CONFIG_TargetInfo *TargetInfo, /* (>) Target info structure */
   const CONFIG_PivotInfo *PivotInfo,   /* (>) Pivot info structure */
   double *Alpha)                 /* (<) Returns calculated alpha value */
{

   /*  Local variables  */

   double thetaFib;                /* Fibre theta value */
   double thetaPivFib;             /* The non-radial pivot angle for fibre */

   /*
    *  Calculate angle of fibre about pivot point (theta=0 @ 12o'clock,
    *  increasing anti-clockwise).
    */

   thetaFib = ConfigThetaFib (MethodInfo, TargetInfo,PivotInfo);
   /*
    *  Calulate fibre/pivot angle.
    */

   thetaPivFib = thetaFib - PivotInfo->Theta - PI;

   /*
    *  Get absolute value of angle, between 0 and 2PI.
    */

   thetaPivFib = thetaPivFib<0 ? -thetaPivFib: thetaPivFib;
   while (thetaPivFib > PI) {
      thetaPivFib -= (2*PI);
      thetaPivFib = thetaPivFib<0 ? -thetaPivFib: thetaPivFib;
   }

   *Alpha = thetaPivFib;
   /*
    *  Validate angle.
    */
   if ((thetaPivFib > ConfigMaxPivAngle(MethodInfo)))
       return CONFIG_FALSE;
   else
       return CONFIG_TRUE;

}

/* -------------------------------------------------------------------------- */

/*                    C o n f i g  T h e t a  F i b
 *
 *  Function:
 *     Calculates the theta angle for a given fibre.
 *
 *  Description:
 *     This is a utility routine intended for use by the method specific code.
 *     It is passed the coordinates of a pivot and a target and returns the
 *     theta angle for the fibre line joining the two, in a coordinate system
 *     where theta is 0 at 12 o'clock, increasing anti-clockwise, which is the
 *     convention used throughout this code. (This code has been taken 
 *     from James Wilcox's configuration validation code and modified 
 *     slightly to fit into the structure of this program.) Note that what
 *     this calculates is the theta angle for the Fibre, measured at the
 *     pivot - it is the angle from the vertical line through the pivot,
 *     moving anti-clockwise, to the fibre. We call this value 'ThetaFib'.
 *     The equivalent value for the button, ie the anticlockwise angle from
 *     the vertical through the button to the fibre, 'ThetaButFib', is given by
 *     ThetaButFib = ThetaFib + PI if ThetaFib is less than PI, and is
 *     ThetaButFib = ThetaFib - PI if ThetaFib is greater than PI.
 */

static double ConfigThetaFib (
   const CONFIG_MethodInfo * MethodInfo DUNUSED,
   const CONFIG_TargetInfo *TargetInfo, /* (>) Target info structure */
   const CONFIG_PivotInfo *PivotInfo)   /* (>) Pivot info structure */
{

   /*  Local variables  */

   long dx;                        /* X distance target -> pivot */
   long dy;                        /* Y distance target -> pivot */
   double thetaFib;                /* Fibre theta value */
   long tmpdx;                     /* Absolute value of dx */
   long tmpdy;                     /* Absolute value of dy */

   /*
    *  Calculate angle of fibre about pivot point (theta=0 @ 12o'clock,
    *  increasing anti-clockwise).
    */

   dx = TargetInfo->X - PivotInfo->X;
   dy = TargetInfo->Y - PivotInfo->Y;
   tmpdx = (dx < 0)? -dx: dx;
   tmpdy = (dy < 0)? -dy: dy;
   if (dx>0 && dy==0)
      thetaFib = 3*PI/2;
   else if (dx>0 && dy>0)
      thetaFib = 3*PI/2 + atan((double)tmpdy/(double)tmpdx);
   else if (dx==0 && dy>0)
      thetaFib = 0;
   else if (dx<0 && dy>0)
      thetaFib = PI/2 - atan((double)tmpdy/(double)tmpdx);
   else if (dx<0 && dy==0)
      thetaFib = PI/2;
   else if (dx<0 && dy<0)
      thetaFib = PI/2 + atan((double)tmpdy/(double)tmpdx);
   else if (dx==0 && dy<0)
      thetaFib = PI;
   else /* (dx>0 && dy<0) */
      thetaFib = 3*PI/2 - atan((double)tmpdy/(double)tmpdx);

   return (thetaFib);
}


/* -------------------------------------------------------------------------- */

/*                    C o n f i g  T h e t a  P i v
 *
 *  Function:
 *     Calculates the theta angle for a given pivot.
 *
 *  Description:
 *     This is a utility routine intended for use by the method specific code.
 *     It is passed the park coordinates of a pivot and returns the
 *     theta angle for the fibre line joining the center of the field
 *     plate to the pivot point, in a coordinate system where theta is 0 at
 *     12 o'clock, increasing anti-clockwise, which is the convention used
 *     throughout this code. This would be the theta for the button when it
 *     is parked on that line joining its pivot point to the field center,
 *     which isn't always the same in practice as the theta for the button 
 *     when it is actually parked, since it may be parked off that ideal line.
 *     In calculations of fibre bend angle, it is this ideal theta value for
 *     the button that is needed.
 */

static double ConfigThetaPiv (
   int XPark,                           /* (>) Pivot X park position */
   int YPark)                           /* (>) Pivot Y park position */
{

   /*  Local variables  */

   long dx;                        /* X distance target -> pivot */
   long dy;                        /* Y distance target -> pivot */
   double thetaFib;                /* Fibre theta value */
   long tmpdx;                     /* Absolute value of dx */
   long tmpdy;                     /* Absolute value of dy */

   /*
    *  Calculate angle of fibre about pivot point (theta=0 @ 12o'clock,
    *  increasing anti-clockwise).
    */

   dx = XPark;
   dy = YPark;
   tmpdx = (dx < 0)? -dx: dx;
   tmpdy = (dy < 0)? -dy: dy;
   if (dx>0 && dy==0)
      thetaFib = 3*PI/2;
   else if (dx>0 && dy>0)
      thetaFib = 3*PI/2 + atan((double)tmpdy/(double)tmpdx);
   else if (dx==0 && dy>0)
      thetaFib = 0;
   else if (dx<0 && dy>0)
      thetaFib = PI/2 - atan((double)tmpdy/(double)tmpdx);
   else if (dx<0 && dy==0)
      thetaFib = PI/2;
   else if (dx<0 && dy<0)
      thetaFib = PI/2 + atan((double)tmpdy/(double)tmpdx);
   else if (dx==0 && dy<0)
      thetaFib = PI;
   else /* (dx>0 && dy<0) */
      thetaFib = 3*PI/2 - atan((double)tmpdy/(double)tmpdx);

   return (thetaFib);
}

/* -------------------------------------------------------------------------- */

/*              C o n f i g  C h e c k  G e o m e t r y
 *
 *  Function:
 *     Checks whether or not a given allocation is geometrically possible.
 *
 *  Description:
 *     This routine checks whether or not a specified allocation of fibre
 *     to target, with a given button twist angle (theta) is invalid because
 *     it is not geometrically possible in terms of fibre-length, non-radial
 *     pivot angle of the fibre, or the specified value of the twist angle.
 *     It does not test whether the allocation is possible in terms of 
 *     collisions with other, already allocated, fibres. An allocation is
 *     valid if both this routine and ConfigCheckAllocation() pass it as OK,
 *     but it is convenient to separate the tests. This routine checks for
 *     those conditions that are immutable, while ConfigCheckAllocation()
 *     checks for conditions that will change as the allocations are revised.
 *     Since it is assumed that these check routines may be called, in a 
 *     tentative way, by the method-specific code, to rule out possible
 *     allocations these do not output error messages. (This simplifies 
 *     their use - using Ers calls to control error reporting would be a
 *     slightly more correct alternative, but is harder to use in non-DRAMA
 *     programs.)
 */

CONFIG_Boolean ConfigCheckGeometry (
                                  /* (<) Returns true if allocation is OK */
   int Field DUNUSED,             /* (>) The field in question */
   int Pivot DUNUSED,             /* (>) The number of the pivot in question */
   int Target DUNUSED,            /* (>) The number of the target in question*/
   double Theta DUNUSED)          /* (>) The proposed twist angle, theta */

/*  Once upon a time, this routine performed the tests contained within the
 *  #ifdef CRAP section. At some time, this code was disabled (CRAP is simply
 *  never intended to be defined!) and the routine now returns true in all
 *  cases. We need to sort out why and remove this routine permanently.
 *  Note that the disabled code depends on the assumption that all fibres
 *  have the same length, since it uses ConfigMaxExtension() instead of the
 *  FibreLength field of the pivot information structure.
 */
{
   return (CONFIG_TRUE);
}
#ifdef CRAP
   MaxExtensionSquared = (double) ConfigMaxExtension();
   MaxExtensionSquared = MaxExtensionSquared * MaxExtensionSquared;

      PivotX = PivotInfo[IPiv].X;
      PivotY = PivotInfo[IPiv].Y;
      PivotType = PivotInfo[IPiv].Spect;
      TargetX = TargetInfo[ITarget].X;
      TargetY = TargetInfo[ITarget].Y;
      DeltaX = (double) PivotX - TargetX;
      DeltaY = (double) PivotY - TargetY;
      DistanceSquared = (DeltaX * DeltaX) + (DeltaY * DeltaY);
      if (DistanceSquared < MaxExtensionSquared) {
               if (ConfigCheckAlpha(&(TargetInfo[ITarget]),
                                                &(PivotInfo[IPiv]),&Alpha)) {
#endif

/* -------------------------------------------------------------------------- */

/*              C o n f i g  C h e c k  A l l o c a t i o n
 *
 *  Function:
 *     Checks whether or not a given allocation is possible.
 *
 *  Description:
 *     This routine checks whether or not a specified allocation of fibre
 *     to target, with a given button twist angle (theta) is invalid because
 *     of collisions with other, already allocated, fibres. It assumes that
 *     the allocation is valid in terms of fibre-length, non-radial pivot
 *     angle of the fibre, and the specified value of the twist angle. See
 *     the notes to ConfigCheckGeometry() for a discussion of how that routine
 *     relates to this, and why these routines do no error reporting.
 *
 *     NOTE, CURRENTLY, MOST COMMENTS REFER TO THE CASE WHERE THE
 *     CHECK_FPIL MACRO IS DEFINED.  THEY MUST BE MODIFIED FOR
 *     USE OF FpilCollision1().    
 *
 */

CONFIG_Boolean ConfigCheckAllocation (
   const CONFIG_MethodInfo * MethodInfo DUNUSED,
                                  /* (<) Returns true if allocation is OK */
   int Pivot,                     /* (>) The number of the pivot in question */
   int Target,                    /* (>) The number of the target in question */
   double Theta,                  /* (>) The proposed twist angle, theta */
   int *CollidingPivot)           /* (<) Index of pivot generating collision */
{

   /*  A note on variable names.  A new allocation of a fibre to a pivot is 
    *  being considered.  If it is made there will be a new, unalterable,
    *  position for a button and there will be a fibre running from the newly 
    *  allocated pivot point to the virtual pivot point of that fibre. All the
    *  variables describing this new allocation contain 'New'. All the variables
    *  describing the existing allocations that we must check against do not
    *  have 'New' but are otherwise the same names. In many cases, these 
    *  qualtities are coordinates in microns, and as such are held as integers.
    *  However, we sometimes need these values in floating point as well, and
    *  in these cases we prefix the name with a 'D'.
    */ 
 
   /*  Local variables  */

   CONFIG_AllocInfo *AllocInfo;/* Allocation info array of structures */
   CONFIG_Boolean Collision;   /* True if there is a collision */
#ifdef CHECK_FPIL
   long ButtonX1;              /* X-start of safe rectangle around old target */
   long ButtonX2;              /* X-end of safe rectangle around old target */
   long ButtonY1;              /* Y-start of safe rectangle around old target */
   long ButtonY2;              /* Y-end of safe rectangle around old target */
   CONFIG_Boolean CheckCollision;   /* True if there is a collision */
   double DFibrePivotX;        /* Virtual pivot position in X for new fibre */
   double DFibrePivotY;        /* Virtual pivot position in Y for new fibre */
   double DNewFibrePivotX;     /* Virtual pivot position in X for fibre */
   double DNewFibrePivotY;     /* Virtual pivot position in Y for fibre */
   double DNewPivotX;          /* Floating point value of NewPivotX */
   double DNewPivotY;          /* Floating point value of NewPivotY */
   double DNewTargetX;         /* Floating point value of NewTargetX */
   double DNewTargetY;         /* Floating point value of NewTargetY */
   double DPivotX;             /* Floating point value of PivotX */
   double DPivotY;             /* Floating point value of PivotY */
   double DTargetX;            /* Floating point value of TargetX */
   double DTargetY;            /* Floating point value of TargetY */
   long FibreX1;               /* X-start of safe rectangle around old fibre */
   long FibreX2;               /* X-end of safe rectangle around old fibre */
   long FibreY1;               /* Y-start of safe rectangle around old fibre */
   long FibreY2;               /* Y-end of safe rectangle around old fibre */
   long NewButtonX1;           /* X-start of safe rectangle around new target */
   long NewButtonX2;           /* X-end of safe rectangle around new target */
   long NewButtonY1;           /* Y-start of safe rectangle around new target */
   long NewButtonY2;           /* Y-end of safe rectangle around new target */
   long NewFibreX1;            /* X-start of safe rectangle around fibre posn */
   long NewFibreX2;            /* X-end of safe rectangle around fibre posn */
   long NewFibreY1;            /* Y-start of safe rectangle around fibre posn */
   long NewFibreY2;            /* Y-end of safe rectangle around fibre posn */
   CONFIG_Boolean Safe;        /* Indicates rects have no overlap regions */
   long SafeClearance;         /* Safe clearance (microns) about button */
#endif
   int IPiv;                   /* Loop index through pivots */
   int ITarget;                /* Loop index through targets */
   int NPiv;                   /* Number of pivots */
   long NewTargetX;            /* X-coordinate of newly allocated target */
   long NewTargetY;            /* Y-coordinate of newly allocated target */
   long NewPivotX;             /* X-coordinate of newly allocated pivot */
   long NewPivotY;             /* Y-coordinate of newly allocated pivot */
   int NTargets;               /* Number of possible targets */
   CONFIG_PivotInfo *PivotInfo;/* Pivot info array of structures */
   long PivotX;                /* X-coordinate of previously allocated pivot */
   long PivotY;                /* Y-coordinate of previously allocated pivot */
   CONFIG_TargetInfo *TargetInfo; /* Target info array of structures */
   long TargetX;               /* X-coordinate of previously allocated target */
   long TargetY;               /* Y-coordinate of previously allocated target */
   double TargetTheta;         /* Theta of previously allocated target */
   int Field = MethodInfo->Field;/* The field in question */
   FpilType Instrument;        /* Instrument description pointer */
   int ParkMayCollide;         /* Indicates if park positions may collide */

   /*  Note that this code could be simplified enormously if we were game
    *  just to make one call to FpilColButBut() and two calls to
    *  FpilColButFib() for each of the possible allocations. However,
    *  these are very precise, careful, slow routines, and we want to avoid
    *  using them if possible, simply because we expect this routine to be
    *  called a great many times. Hence the business with 'safe' rectangles,
    *  and the huge number of variables needed to describe them.
    */

   /*
    *  Get easy access to the FPIL instrument description.
    */
   Instrument = CONFIG_Global.MethodInfo._Instrument;

   /*  Useful constants. We need to know the maximum offset of a button
    *  from the target point, so we know, crudely, how much clearance we
    *  need to be really safe when looking at overlapping rectangles later
    *  in the code. Since there is a utility routine that calculates this
    *  for use by the algorithm-specific code, we use that here as well.
    */
#  ifdef CHECK_FPIL
   SafeClearance = ConfigSafeClearance(MethodInfo);
#  endif

   /*  Pick up local pointers to the various information structures whose
    *  addresses are held in the method-specific global structure. Also
    *  pick up the number of possible targets and pivots.
    */

   NTargets = CONFIG_Global.NumberTargets;
   NPiv = CONFIG_Global.NumberPivots[Field];
   PivotInfo = CONFIG_Global.PivotInfoPtr[Field];
   TargetInfo = CONFIG_Global.TargetInfoPtr;
   AllocInfo = CONFIG_Global.AllocInfoPtr[Field];

   /*  Calculate the rectangles surrounding the newly proposed fibre and
    *  its button. The NewButtonxxx values give the rectangle that safely
    *  clears the newly allocated button position.  The NewFibrexxx values
    *  give the rectangle that safely clears the whole new allocated fibre.
    */

   NewTargetX = TargetInfo[Target].X;
   NewTargetY = TargetInfo[Target].Y;
   NewPivotX = PivotInfo[Pivot].X;
   NewPivotY = PivotInfo[Pivot].Y;
#ifdef CHECK_FPIL
   DNewTargetX = (double) NewTargetX;
   DNewTargetY = (double) NewTargetY;
   NewButtonX1 = NewTargetX - SafeClearance;
   NewButtonX2 = NewTargetX + SafeClearance;
   NewButtonY1 = NewTargetY - SafeClearance;
   NewButtonY2 = NewTargetY + SafeClearance;
   DNewPivotX = (double) NewPivotX;
   DNewPivotY = (double) NewPivotY;
   if (NewPivotX > NewTargetX) {
      NewFibreX1 = NewTargetX - SafeClearance;
      NewFibreX2 = NewPivotX + SafeClearance;
   } else {
      NewFibreX1 = NewPivotX - SafeClearance;
      NewFibreX2 = NewTargetX + SafeClearance;
   }
   if (NewPivotY > NewTargetY) {
      NewFibreY1 = NewTargetY - SafeClearance;
      NewFibreY2 = NewPivotY + SafeClearance;
   } else {
      NewFibreY1 = NewPivotY - SafeClearance;
      NewFibreY2 = NewTargetY + SafeClearance;
   }
#endif

   /*  We we also need to know the position of the virtual pivot point
    *  of the newly allocated button, since that is where the fibre is
    *  regarded as joining the button.
    */
#ifdef CHECK_FPIL
   ConfigGetVirtPiv (MethodInfo,DNewTargetX,DNewTargetY,Theta,
                     &DNewFibrePivotX, &DNewFibrePivotY);
#endif

   /*  We need to pass through every pivot. In every case
    *  we have to consider whether this proposed new
    *  allocation will collide with the existing fibre.  
    *
    */

   Collision = CONFIG_FALSE;
#ifdef CHECK_FPIL
   CheckCollision  = CONFIG_FALSE;
#endif

   /*
    * We need to determine if we have to check for collisions against
    * parked fibres.
    */
   ParkMayCollide = FpilParkMayCollide(Instrument);

   for (IPiv = 0; IPiv < NPiv; IPiv++) {
      if (IPiv != Pivot) {
         PivotX = PivotInfo[IPiv].X;
         PivotY = PivotInfo[IPiv].Y;
#        ifdef CHECK_FPIL
         DPivotX = (double) PivotX;
         DPivotY = (double) PivotY;
#        endif

         if (AllocInfo[IPiv].State == 'A') {
             /* Allocated fibre, get details from target position */
             ITarget     = AllocInfo[IPiv].TargetIndex;
             TargetX     = TargetInfo[ITarget].X;
             TargetY     = TargetInfo[ITarget].Y;
             TargetTheta =  AllocInfo[IPiv].Theta;
         } else if (!ParkMayCollide) {
             /* If park won't collide, can skip this one */
             continue;
         } else if (!FpilFibreReallyExists(Instrument,
                             PivotInfo[IPiv].FibreType)) {
             /*  If the parked fibre doesn't really exist, we also skip it */
             continue;
         } else {
             /* Unallocated, get details from park position */
             TargetX     = PivotInfo[IPiv].ParkX;
             TargetY     = PivotInfo[IPiv].ParkY;
             TargetTheta =  PivotInfo[IPiv].Theta;
         }
             
         /*  We need to check
          *  1) whether there is a collision between the proposed new button
          *  position and the existing button position, 2) whether there is
          *  a collision between the proposed new button and the 
          *  existing fibre path, 3) whether there is a collision
          *  between the proposed new fibre path and the existing
          *  button position. The standard routines for doing the formal
          *  checks for these are quite time-consuming, so we try to do 
          *  some quick and dirty checks first to see if we need to call
          *  them.  We crudely define 'rectangles of interest' around
          *  the two button positions and the two fibres - these are just
          *  the smallest rectangles in the standard coordinate system 
          *  that fully enclose (and allow for clearance) each button and
          *  each fibre. Only if these overlap do we need to call the
          *  more detailed check routines.
          */

#        ifdef CHECK_FPIL
         DTargetX = (double) TargetX;
         DTargetY = (double) TargetY;

         ButtonX1 = TargetX - SafeClearance;
         ButtonX2 = TargetX + SafeClearance;
         ButtonY1 = TargetY - SafeClearance;
         ButtonY2 = TargetY + SafeClearance;
            
         /*  So, do we need to call the button-button collision routine?  */

         Safe = CONFIG_FALSE;
         if (ButtonX2 < NewButtonX1) {
            Safe = CONFIG_TRUE;
         } else if (NewButtonX2 < ButtonX1) {
            Safe = CONFIG_TRUE;
         } else if (ButtonY2 < NewButtonY1) {
            Safe = CONFIG_TRUE;
         } else if (NewButtonY2 < ButtonY1) {
            Safe = CONFIG_TRUE;
         }

         if (!Safe) {
            if (FpilColButBut (Instrument, 
                               DTargetX, DTargetY, TargetTheta, 
                               DNewTargetX, DNewTargetY, Theta)) {
               CheckCollision = CONFIG_TRUE;
               *CollidingPivot = IPiv;
            }
         }

         /*  Now, what about the possibility of the proposed new fibre
          *  colliding with the already allocated button? The already
          *  allocated button is at (TargetX,TargetY), and its 'safe'
          *  rectangle is delimited by (ButtonX1,ButtonX2,ButtonY1,ButtonY2).
          *  The fibre leading to the proposed new button goes
          *  from its pivot (NewPivotX,NewPivotY) to the virtual pivot
          *  point of its button (NewFibrePivotX,NewFibrePivotY). Its 'safe'
          *  rectangle is given by (NewFibreX1,NewFibreX2,NewFibreY1,
          *  NewFibreY2).
          */

         if (!CheckCollision) {
            Safe = CONFIG_FALSE;
            if (ButtonX2 < NewFibreX1) {
               Safe = CONFIG_TRUE;
            } else if (NewFibreX2 < ButtonX1) {
               Safe = CONFIG_TRUE;
            } else if (ButtonY2 < NewFibreY1) {
               Safe = CONFIG_TRUE;
            } else if (NewFibreY2 < ButtonY1) {
               Safe = CONFIG_TRUE;
            }

            if (!Safe) {
               if (FpilColButFib (Instrument,
                                  DTargetX, DTargetY, TargetTheta,
                                  DNewFibrePivotX, DNewFibrePivotY,
                                  DNewPivotX,      DNewPivotY )) {
                  CheckCollision = CONFIG_TRUE;
                  *CollidingPivot = IPiv;
               }
            }
         }

         /*  And, finally, what about the possibility of the already
          *  allocated fibre colliding with the proposed new button?
          *  The proposed new button is at (NewTargetX,NewTargetY), and
          *  its 'safe' rectangle is given by (NewButtonX1,NewButtonX2,
          *  NewButtonY1,NewButtonY2).  The fibre leading to the already
          *  allocated button goes from its pivot (PivotX,PivotY) to the
          *  virtual pivot point of its button, which we still have to
          *  calculate as (FibrePivotX,FibrePivotY). The 'safe' rectangle
          *  for the fibre is (FibreX1,FibreX2,FibreY1,FibreY2), which we
          *  also still have to calculate.
          */

         if (!CheckCollision) {

            if (PivotX > TargetX) {
               FibreX1 = TargetX - SafeClearance;
               FibreX2 = PivotX + SafeClearance;
            } else {
               FibreX1 = PivotX - SafeClearance;
               FibreX2 = TargetX + SafeClearance;
            }
            if (PivotY > TargetY) {
               FibreY1 = TargetY - SafeClearance;
               FibreY2 = PivotY + SafeClearance;
            } else {
               FibreY1 = PivotY - SafeClearance;
               FibreY2 = TargetY + SafeClearance;
            }

            Safe = CONFIG_FALSE;
            if (NewButtonX2 < FibreX1) {
               Safe = CONFIG_TRUE;
            } else if (FibreX2 < NewButtonX1) {
               Safe = CONFIG_TRUE;
            } else if (NewButtonY2 < FibreY1) {
               Safe = CONFIG_TRUE;
            } else if (FibreY2 < NewButtonY1) {
               Safe = CONFIG_TRUE;
            }

            if (!Safe) {
               ConfigGetVirtPiv (MethodInfo, DTargetX,DTargetY,
                     TargetTheta,&DFibrePivotX,&DFibrePivotY);

               if (FpilColButFib (Instrument,
                                  DNewTargetX, DNewTargetY, Theta, 
                                  DFibrePivotX,DFibrePivotY,
                                  DPivotX,DPivotY)) {
                  CheckCollision = CONFIG_TRUE;
                  *CollidingPivot = IPiv;
               }
            }
         }
#        endif /* CHECK_FPIL */

         Collision = FpilCollision1 ( Instrument,
                                      NewTargetX, NewTargetY, Theta,
                                      NewPivotX, NewPivotY,
                                      TargetX, TargetY, TargetTheta, 
                                      PivotX, PivotY) ;
                                      
#        ifdef CHECK_FPIL
         if ((CheckCollision)&&(!Collision))
         {
             fprintf(stderr,
                    "ConfigCheckAllocation:CHECK_FPIL failure, %d %d %d %d\n",
                     (int)CheckCollision, (int)Collision, Pivot, NPiv);
             Collision = 1;
         }
#        endif

         if (Collision) {
             *CollidingPivot = IPiv;
             break;
         }
      }
   }

   return (!Collision);

}

/* -------------------------------------------------------------------------- */

/*                    C o n f i g  S e t  S k y
 *
 *  Function:
 *     Sets the number of fibres of a given type to be allocated to sky.
 *
 *  Description:
 *     This routine allows the higher levels to exercise precise control
 *     over the number of fibres of each type that ought to be allocated
 *     to sky targets. This is a requirement that all allocation algorithms
 *     need to support, so it is more appropriate to handle it in this way
 *     than through the method-specific parameters. The settings laid down
 *     by a call to this routine should be taken into account by the next
 *     call to ConfigAllocate(). The settings made by this routine 
 *     are only relevant for fibre types that can be allocated to both
 *     sky and non-sky targets. Callign this routine with a negative FibreType
 *     -resets the sky allocation settings for all fibre types to a mode 
 *     where no distinction is made between sky and non-sky targets.
 *
 *     Note: This routine is almost identical to the ConfigMethodSetSky()
 *     code, and having the same thing duplicated at both levels in the code
 *     is rather inelegant. It might have been better to include this 
 *     information in the pivots structure, but that isn't clear.
 */

extern void ConfigSetSky (
   int FibreType,                    /* (>) Fibre type code */
   int NumberSkyAllocations,         /* (>) Number of fibres for sky targets */
   StatusType *Status)               /* (!) Inherited status */
{
   /*  Local variables */
   
   CONFIG_Boolean Found;             /* True if entry found for this type */
   int IType;                        /* Index through pivot types */
   int MaxTypes;                     /* Size of sky count array */
   int NumberTypes;                  /* Number of types in sky count array */
   
   if (*Status != STATUS__OK) return;
   
   /*  If the type is -ve, then we just clear out all the entries in the
    *  global array. (This isn't duplicated in the method-specific code,
    *  which has a slightly different usage sequence and the initialisation
    *  code does this job.)
    */
   
   if (FibreType < 0) {
      NumberTypes = sizeof(CONFIG_Global.FibreTypesForSky)/sizeof(int);
      for (IType = 0; IType < NumberTypes; IType++) {
         CONFIG_Global.FibreTypesForSky[IType] = -1;
         CONFIG_Global.FibreSkyCounts[IType] = -1;
      }
      CONFIG_Global.FibreSkyEntries = 0;
   } else {
    
      /*  See if we already have an entry for this type in the global array
       *  used to return these settings.
       */

      Found = CONFIG_FALSE;
      MaxTypes = sizeof(CONFIG_Global.FibreTypesForSky)/sizeof(int);
      NumberTypes = CONFIG_Global.FibreSkyEntries;
      for (IType = 0; IType < NumberTypes; IType++) {
         if (CONFIG_Global.FibreTypesForSky[IType] == FibreType) {
            CONFIG_Global.FibreSkyCounts[IType] = NumberSkyAllocations;
            Found = CONFIG_TRUE;
            break;
         }
      }

      /*  If not, we create a new entry */

      if (NumberTypes >= MaxTypes) {
         *Status = CONFIG__PARM_OUT_RANGE;
      } else {
         CONFIG_Global.FibreTypesForSky[NumberTypes] = FibreType;
         CONFIG_Global.FibreSkyCounts[NumberTypes] = NumberSkyAllocations;
         CONFIG_Global.FibreSkyEntries++;
      }
   }
}   


/* -------------------------------------------------------------------------- */

/*+        
 *                  C o n f i g  HK F i e l d  C h e c k
 *
 *  Function:
 *     Performs a full formal validity check on the current allocations.
 *
 *  Description:
 *     This routine is supposed to be functionally equivalent to the
 *     routine tdFdeltaFieldCheck() that is used by the 2dF positioner
 *     code to check a field for validity before trying to move to it.
 *     It has been produced by a fairly heavy edit of that routine in
 *     order to form a routine suitable for use by CONFIG, but the actual
 *     details of the tests should be unchanged, and the intention is that
 *     the algorithms used should be absolutely identical. This means that
 *     if a set of allocations passes this routine (which it should do if
 *     the configuration program has done its job properly) then they will
 *     produce a field that the positioner task is prepared to configure.
 *     It also means that any changes to the positioner code ought to be
 *     reflected in this code as soon as possible. It's a pity that the
 *     structure of the two tasks doesn't make it easy to have an identical
 *     subroutine used by each.
 *
 *     The checks that are performed are -
 *
 *      - no button/button collisions
 *      - no button/fibre collisions
 *      - maximum fibre extension is not exceeded
 *      - maximum button/fibre bend angle is not exceeded
 *      - maximum pivot/fibre bend angle is not exceeded
 *      - position is a within field plate limits
 *
 *     The PositionerTolerances argument is used to control whether the
 *     routine uses the current values of the various constraint parameters
 *     or the values that the positioner will use. In most cases, the
 *     values used by the positioner itself are what we really need to test
 *     against, but there are cases where we want to check a field against
 *     more constraining parameter values.
 *
 */

static void  ConfigHKFieldCheck (
   const CONFIG_MethodInfo * MethodInfo,
   CONFIG_ReportFunction 
             ReportFunction,     /* (>) Feedback routine to call */
   void *FunctionArg,            /* (>) Caller-supplied argument */
   int PositionerTolerances,     /* (>) If true, use positioner tolerances */
   StatusType  *status)          /* (!) Inherited status variable */
{
    double        thetaButFib,         /* Angle between button and fibre    */
                  thetaPivFib,         /* Angle between pivot and fibre     */
                  thetaFib,            /* Angle of fibre about piv point    */
                  MaxPivAngle,         /* Maximum allowed pivot angle       */
                  MaxButAngle,         /* Maximum allowed button angle      */
                  CurrentMaxPivAngle,  /* Current max allowed pivot angle   */
                  CurrentMaxButAngle,  /* Current max allowed button angle  */
                  deltaX,deltaY,       /* Used to calculate fibre length    */
                  fvpx,fvpy;           /* Coordinates of button virtual piv */
    long int      dx, dy,              /* Used to check button/fibre...     */
                  tmpdx, tmpdy,        /* ...and pivot fibre bend angles    */
                  CurrentButClear,     /* Button clearance currently used   */
                  CurrentFibClear;     /* Fibre clearance currently used    */
    int           i, j,                /* Counters                          */
                  iTarget,             /* Target allocated to fibre         */
                  Field,               /* Field being checked               */
                  fieldOK,             /* Field validity flag               */
                  numErrors,           /* Total number of error detected    */
                  NTargets,            /* Total number of known targets     */
                  NPiv;                /* Number of pivots                  */
    CONFIG_Boolean flag;               /* Used during collision detection   */
    CONFIG_AllocInfo *AllocInfo;       /* Address of allocation info array  */
    CONFIG_PivotInfo *PivotInfo;       /* Address of pivot info array       */
    CONFIG_TargetInfo *TargetInfo;     /* Address of target info array      */
    double PercentDone;                /* Percentage of task completed      */
    double FibreLength;                /* Length of fibre                   */
    int ReportPivs;                    /* Report after processing this many */
    FpilType Instrument;               /* Instrument description pointer */
    int ParkMayCollide;                /* Indicates if parked fibre may  
                                          collide with other fibres      */
    double AllocLength;                /* Length of allocated fibre */
    double ParkedLength;               /* Length of parked fibre */
     

    if (*status != STATUS__OK) return;

    /*
     *  Get easy access to the FPIL instrument description.
     */
    Instrument = CONFIG_Global.MethodInfo._Instrument;

    /*  Initial values */


    Field = CONFIG_Global.Field;
    fieldOK = CONFIG_TRUE;
    numErrors = 0;

    /*  Get the current values for various limiting angles for the fibres. 
     *
     *  Usually we want this check to match that performed by the positioner,
     *  which we assume will be using the default constraints supplied by
     *  the Fpil layer, and recorded in the CONFIG_Global.PositionerXxxx
     *  global variables. The configuration that we are testing may well have
     *  been created with more constraining values, in an attempt to get a
     *  configuration with straighter fibres, or one that will survive a
     *  greater range of hour angles and wavelengths. We need to restore the
     *  current values at the end of the check. If the PositionerTolerances
     *  argument is set false, however, we use the current values for the
     *  constraints.
     *
     *  Note that the current values of the constraining parameters are 
     *  available in two separate places. (These values are the ones that 
     *  have been used to generate the configuration we are testing.) They
     *  can be found from the Config routines such as ConfigMaxPivAngle(),
     *  which give the values of the corresponding allocation parameter,
     *  and from Fpil routines such as FpilGetMaxPivAngle(). These ought to
     *  be completely in step, because the parameter handling code is set
     *  up to automatically call the appropriate FpilSetXxxx routine when
     *  it reads the value of a parameter. We check this for the integer
     *  values here, but not for the floating point ones, since the Config
     *  routines store the values in degrees while the Fpil layer uses
     *  radians, and we don't want error messages generated by rounding error.
     */
     
    CurrentMaxPivAngle = ConfigMaxPivAngle(MethodInfo);
    CurrentMaxButAngle = ConfigMaxButAngle(MethodInfo);
    if (PositionerTolerances) {
       MaxPivAngle = CONFIG_Global.PositionerMaxPivAng;
       MaxButAngle = CONFIG_Global.PositionerMaxFibAng;
    } else {
       MaxPivAngle = CurrentMaxPivAngle;
       MaxButAngle = CurrentMaxButAngle;
    }
    FpilSetMaxPivAng(CONFIG_Global.MethodInfo._Instrument,MaxPivAngle);
    FpilSetMaxFibAng(CONFIG_Global.MethodInfo._Instrument,MaxButAngle);
    CurrentButClear = FpilGetButClear(CONFIG_Global.MethodInfo._Instrument);
    if (PositionerTolerances) {
       FpilSetButClear(CONFIG_Global.MethodInfo._Instrument,
                                          CONFIG_Global.PositionerButClear);
    } else {
       FpilSetButClear(CONFIG_Global.MethodInfo._Instrument, fpossButClear);
    }
    CurrentFibClear = FpilGetFibClear(CONFIG_Global.MethodInfo._Instrument);
    if (PositionerTolerances) {
       FpilSetFibClear(CONFIG_Global.MethodInfo._Instrument,
                                          CONFIG_Global.PositionerFibClear);
    } else {
       FpilSetFibClear(CONFIG_Global.MethodInfo._Instrument, fpossFibClear);
    }

    /*  Set local pointers to the various global structures maintained by
     *  the CONFIG program.  Most of this program is driven from the
     *  allocation, pivot and target information arrays.
     */

    AllocInfo = CONFIG_Global.AllocInfoPtr[Field];
    PivotInfo = CONFIG_Global.PivotInfoPtr[Field];
    TargetInfo = CONFIG_Global.TargetInfoPtr;
    NPiv = CONFIG_Global.NumberPivots[Field];
    NTargets = CONFIG_Global.NumberTargets;

    /*  For the purposes of reporting progress, we pick a suitable number
     *  of pivots. Here we report every 10% of pivots, since this should
     *  be a pretty rapid operation.
     */

    ReportPivs = (int) ((float) NPiv / 10.0);
    if (ReportPivs < 0) ReportPivs = 1;

    /*
    * We need to determine if we have to check for collisions against
    * parked fibres.
    */
   ParkMayCollide = FpilParkMayCollide(Instrument);

    /*
     *  Check for duplicate allocations, invalid allocations, clashes
     *  between fibre and target type.
     */
    for (i=0; i<NPiv; i++) {
       if (ReportFunction != NULL) {
          if ((i % ReportPivs) == 0) {
             PercentDone = ((float) i / (float) NPiv) * 100.0;
             if ((*ReportFunction)(FunctionArg,'P',
                     "Checking for duplicate and inconsistent allocations",
                                         0,0,0,0,0,0.0,PercentDone) != 0) {
                
                MethodInfo->InfoRoutine(MethodInfo->InfoArgument,
                                        "Cancel request received");
                *status = CONFIG__CANCELLED;
                goto Exit;
             }
           }
        } 
        if (AllocInfo[i].State == 'A') {
           if (FpilIsFibreBroken(MethodInfo->_Instrument,
                                              PivotInfo[i].FibreType)) {
               ErsRep (0,status,"Broken fibre %d is allocated\n",i+1);
               numErrors++;
               fieldOK = CONFIG_FALSE;
            }
            iTarget = AllocInfo[i].TargetIndex;
            if ((iTarget < 0) || (iTarget >= NTargets)) {
               ErsRep (0,status,
                  "Invalid target index (%d) for allocated fibre (%d)",
                                                          iTarget,i+1);
               numErrors++;
               fieldOK = CONFIG_FALSE;
            } else {
               if (!FpilTargetPivotCompatible(MethodInfo->_Instrument,
                       TargetInfo[iTarget].Type, TargetInfo[iTarget].Spect,
                                                   PivotInfo[i].FibreType)) {
                  ErsRep (0,status,
                      "Fibre %d allocated to an incompatible target",i+1);
                  ErsRep (0,status,
                      "Fibre type was %d, target type %d, target spect %d",
                      PivotInfo[i].FibreType,TargetInfo[iTarget].Type, 
                      TargetInfo[iTarget].Spect);
                  numErrors++;
                  fieldOK = CONFIG_FALSE;
               }
            }
            for (j=i+1; j < NPiv; j++) {
               if (AllocInfo[j].State == 'A') {
                  if (iTarget == AllocInfo[j].TargetIndex) {
                     ErsRep (0,status,
                         "Fibre %d and fibre %d allocated to same target",
                                                                   i+1,j+1);
                     numErrors++;
                     fieldOK = CONFIG_FALSE;
                  }
               }
            }
        }
    }
               
    /*
     *  Check for button/button collisions.
     */

    for (i=0; i<NPiv; i++) {

       int XI, YI;
       double ThetaI;

       if (ReportFunction != NULL) {
          if ((i % ReportPivs) == 0) {
             PercentDone = ((float) i / (float) NPiv) * 100.0;
             if ((*ReportFunction)(FunctionArg,'P',
                     "Checking for button/button collisions",
                                         0,0,0,0,0,0.0,PercentDone) != 0) {
                MethodInfo->InfoRoutine(MethodInfo->InfoArgument,
                                        "Cancel request received");
                *status = CONFIG__CANCELLED;
                goto Exit;
             }
           }
        } 

        if (AllocInfo[i].State == 'A') {
            /* Allocate fibre, check against allocated position */
            XI     = AllocInfo[i].X;
            YI     = AllocInfo[i].Y;
            ThetaI = AllocInfo[i].Theta;
        } else if (!ParkMayCollide) {
             /* If park won't collide, can skip this one */
             continue;
        } else if (!FpilFibreReallyExists(Instrument,
                             PivotInfo[i].FibreType)) {
             /*  If the parked fibre doesn't really exist, we also skip it */
             continue;
        } else {
            /*  An unallocated fibre - check against park position */
            XI     = PivotInfo[i].ParkX;
            YI     = PivotInfo[i].ParkY;
            ThetaI = PivotInfo[i].Theta;
        }

        /*
         *  Check against each of the other buttons.
         */
        for (j=0; j<NPiv; j++) {

            int XJ, YJ;
            double ThetaJ;

            /*
             *  Do not check a button against itself or a button that it has 
             *  already been checked against.
             */
            if (i >= j) continue;

            if (AllocInfo[j].State == 'A') {
                /* Allocate fibre, check against allocated position */
                XJ     = AllocInfo[j].X;
                YJ     = AllocInfo[j].Y;
                ThetaJ = AllocInfo[j].Theta;
            } else if (!ParkMayCollide) {
                /* If park won't collide, can skip this one */
                continue;
            } else if (!FpilFibreReallyExists(Instrument,
                             PivotInfo[j].FibreType)) {
             /*  If the parked fibre doesn't really exist, we also skip it */
             continue;
            } else {
                /*  An unallocated fibre - check against park position */
                XJ     = PivotInfo[j].ParkX;
                YJ     = PivotInfo[j].ParkY;
                ThetaJ = PivotInfo[j].Theta;
            }
            /*
             *  Check button i against button j for collision.
             */

            flag = FpilColButBut(Instrument, 
                                 XI, YI, ThetaI,
                                 XJ, YJ, ThetaJ);
                                 
            if (flag) {
                ErsRep(0,status,
                 "Button/button collision detected in target field (but=%d,%d)",
                       i+1,j+1);
                numErrors++;
                fieldOK = CONFIG_FALSE;
            }
        }
    }

    /*
     *  Check for button/fibre collisions.
     */
    for (i=0; i<NPiv; i++) {


       int XI, YI;
       double ThetaI;


       if (ReportFunction != NULL) {
          if ((i % ReportPivs) == 0) {
             PercentDone = ((float) i / (float) NPiv) * 100.0;
             if ((*ReportFunction)(FunctionArg,'P',
                     "Checking for button/fibre collisions",
                                         0,0,0,0,0,0.0,PercentDone) != 0) {
                MethodInfo->InfoRoutine(MethodInfo->InfoArgument,
                                        "Cancel request received");
                *status = CONFIG__CANCELLED;
                goto Exit;
             }
           }
        } 

        if (AllocInfo[i].State == 'A') {
            /* Allocate fibre, check against allocated position */
            XI     = AllocInfo[i].X;
            YI     = AllocInfo[i].Y;
            ThetaI = AllocInfo[i].Theta;
        } else if (!ParkMayCollide) {
             /* If park won't collide, can skip this one */
             continue;
        } else if (!FpilFibreReallyExists(Instrument,
                             PivotInfo[i].FibreType)) {
             /*  If the parked fibre doesn't really exist, we also skip it */
             continue;
        } else {
            /*  An unallocated fibre - check against park position */
            XI     = PivotInfo[i].ParkX;
            YI     = PivotInfo[i].ParkY;
            ThetaI = PivotInfo[i].Theta;
        }

        /*
         *  Check against all other buttons.
         */
        for (j=0; j<NPiv; j++) {

            int XJ, YJ;
            double ThetaJ;

            /*
             *  Do not check a button/fibre against itself or a button/fibre
             *  that it has already been checked against.
             */
            if (i >= j) continue;


            if (AllocInfo[j].State == 'A') {
                /* Allocate fibre, check against allocated position */
                XJ     = AllocInfo[j].X;
                YJ     = AllocInfo[j].Y;
                ThetaJ = AllocInfo[j].Theta;
            } else if (!ParkMayCollide) {
                /* If park won't collide, can skip this one */
                continue;
            } else if (!FpilFibreReallyExists(Instrument,
                             PivotInfo[j].FibreType)) {
                /*  If the parked fibre doesn't really exist, we also skip it */
                continue;
            } else {
                /*  An unallocated fibre - check against park position */
                XJ     = PivotInfo[j].ParkX;
                YJ     = PivotInfo[j].ParkY;
                ThetaJ = PivotInfo[j].Theta;
            }

            /*
             *  Check button i against fibre j for collision.
             */
            ConfigGetVirtPiv(MethodInfo,XJ, YJ, ThetaJ,&fvpx,&fvpy);
            flag = FpilColButFib(Instrument, 
                                 XI, YI, ThetaI, 
                                 fvpx, fvpy,
                                 PivotInfo[j].X, PivotInfo[j].Y);
            if (flag) {
                ErsRep(0,status,
                   "Button/fibre collision in target field (but=%d,fib=%d)",
                                                                      i+1,j+1);
                numErrors++;
                fieldOK = CONFIG_FALSE;
            }

            /*
             *  Check button j against fibre i for collision.
             */
            ConfigGetVirtPiv(MethodInfo, XI, YI, ThetaI, &fvpx,&fvpy);
            flag = FpilColButFib(Instrument, 
                                 XJ, YJ, ThetaJ,
                                 fvpx,fvpy,
                                 PivotInfo[i].X, PivotInfo[i].Y);
            if (flag) {
                ErsRep(0,status,
                   "Button/fibre collision in target field (but=%d,fib=%d)",
                   j+1,i+1);
                numErrors++;
                fieldOK = CONFIG_FALSE;
            }
        }
    }

    /*
     *  Check for collisions with adjacent parked positions. (For some
     *  instruments, we can't allow a fibre to collide with the park
     *  position of another fibre, even if that other fibre isn't
     *  parked in the configuration. This is because the positioner
     *  delta algorithm usually requires that it be free to park any
     *  fibre if necessary as it changes from one configuration to another.)
     *
     *  Note on the code in this loop: this is a modified version of the code
     *  in ConfigAdjPark(). Ideally, this loop would just be a series of
     *  calls to ConfigAdjPark(), but unfortunately the structure of the
     *  program is such that ConfigAdjPark() assumes that MethodInfo->
     *  PivotInfo can be used to get the pivot information, and this isn't
     *  the case at this point. Ideally, ConfigAdjPark() would be restructured
     *  so it could be used here, but this loop was introduced late in the
     *  day to fix a problem and this was the safest aproach to take.
     */
     
    if (FpilParkMayCollide(Instrument)) { 
    
        for (i=0; i<NPiv; i++) {

            int Collision = CONFIG_FALSE;/*  True for a collision */
            int HigherPivot;             /*  Index of adjacent pivot - above */
            int LowerPivot;              /*  Index of adjacent pivot - below */
            long ParkButtonX;            /*  X position of parked button */
            long ParkButtonY;            /*  Y position of parked button */
            long ParkPivotX;             /*  X position of parked pivot */
            long ParkPivotY;             /*  Y position of parked pivot */
            double ParkTheta;            /*  Theta angle for parked button */
            double ParkVirtPivX;         /*  X position of parked virtual pivot */
            double ParkVirtPivY;         /*  Y position of parked virtual pivot */
            long PivotX;                 /*  X position of pivot for the fibre */
            long PivotY;                 /*  Y position of pivot for the fibre */
            long TargetX;                /*  X position of target */
            long TargetY;                /*  Y position of target */
            double Theta;                /*  Theta angle for parked button */
            double VirtPivX;             /*  X position of botton virtual pivot */
            double VirtPivY;             /*  Y position of botton virtual pivot */
    
            if (AllocInfo[i].State != 'A') continue;
            
            /*  Get the virtual pivot location for the button at the target 
             *  position, and the position of the pivot point for the fibre.
             */
    
            TargetX = AllocInfo[i].X;
            TargetY = AllocInfo[i].Y;
            Theta = AllocInfo[i].Theta;
            FpilGetVirtPiv(Instrument, TargetX, TargetY, Theta, &VirtPivX, 
                                                                  &VirtPivY);
            PivotX = PivotInfo[i].X;
            PivotY = PivotInfo[i].Y;
    
            /*  Check for a collision with the pivot on the lower numbered side */
    
            LowerPivot = i - 1;
            if (LowerPivot < 0) LowerPivot = NPiv - 1;
            if (FpilFibreReallyExists(Instrument,
                                   PivotInfo[LowerPivot].FibreType)) {
                ParkButtonX = PivotInfo[LowerPivot].ParkX;
                ParkButtonY = PivotInfo[LowerPivot].ParkY;
                ParkTheta = PivotInfo[LowerPivot].Theta;
                if (FpilColButBut(Instrument,ParkButtonX,ParkButtonY,ParkTheta,
                                             TargetX,TargetY,Theta)) {
                    Collision = CONFIG_TRUE;
                } else {
                    if (FpilColButFib(Instrument,ParkButtonX,ParkButtonY,ParkTheta,
                               VirtPivX,VirtPivY,PivotX,PivotY)) {
                        Collision = CONFIG_TRUE;
                    } else {
                        ParkPivotX = PivotInfo[LowerPivot].X;
                        ParkPivotY = PivotInfo[LowerPivot].Y;
                        FpilGetVirtPiv(Instrument, ParkButtonX, ParkButtonY,
                               ParkTheta, &ParkVirtPivX, &ParkVirtPivY);
                        if (FpilColButFib(Instrument,TargetX,TargetY,Theta,
                           ParkVirtPivX,ParkVirtPivY,ParkPivotX,ParkPivotY)) {
                            Collision = CONFIG_TRUE;
                        }
                    }
                }
            }
            if (!Collision) {
                HigherPivot = i + 1;
                if (HigherPivot >= NPiv) HigherPivot = 0;
                if (FpilFibreReallyExists(Instrument,
                    PivotInfo[HigherPivot].FibreType)) {
                    ParkButtonX = PivotInfo[HigherPivot].ParkX;
                    ParkButtonY = PivotInfo[HigherPivot].ParkY;
                    ParkTheta = PivotInfo[HigherPivot].Theta;
                    if (FpilColButBut(Instrument,ParkButtonX,ParkButtonY,ParkTheta,
                                                  TargetX,TargetY,Theta)) {
                        Collision = CONFIG_TRUE;
                    } else {
                        if (FpilColButFib(Instrument,ParkButtonX,ParkButtonY,
                               ParkTheta,VirtPivX,VirtPivY,PivotX,PivotY)) {
                            Collision = CONFIG_TRUE;
                        } else {
                            ParkPivotX = PivotInfo[HigherPivot].X;
                            ParkPivotY = PivotInfo[HigherPivot].Y;
                            FpilGetVirtPiv(Instrument, ParkButtonX, ParkButtonY,
                               ParkTheta, &ParkVirtPivX, &ParkVirtPivY);
                            if (FpilColButFib(Instrument,TargetX,TargetY,Theta,
                                ParkVirtPivX,ParkVirtPivY,ParkPivotX,ParkPivotY)) {
                                Collision = CONFIG_TRUE;
                            }
                        }
                    }
                }
            }
            if (Collision) {
                ErsRep(0,status,
                   "Button collides with adjacent park position (but=%d)",i+1);
                numErrors++;
                fieldOK = CONFIG_FALSE;
            }
        }
    }
           
    /*
     *  Check for invalid position.
     */
    for (i=0; i<NPiv; i++) {

       if (ReportFunction != NULL) {
          if ((i % ReportPivs) == 0) {
             PercentDone = ((float) i / (float) NPiv) * 100.0;
             if ((*ReportFunction)(FunctionArg,'P',
                     "Checking for button/screw collisions",
                                         0,0,0,0,0,0.0,PercentDone) != 0) {
                MethodInfo->InfoRoutine(MethodInfo->InfoArgument,
                                        "Cancel request received");
                *status = CONFIG__CANCELLED;
                goto Exit;
             }
           }
        } 

        /*
         *  No need to check an unallocated fibre.
         */
        if (AllocInfo[i].State != 'A') continue;
        flag = FpilColInvPos(Instrument,
                             Field,
                	     PivotInfo[i].FibreType,
                             (double)AllocInfo[i].X,
                             (double)AllocInfo[i].Y,
                             AllocInfo[i].Theta);
        if (flag) {
           ErsRep(0,status,
               "Button in Invalid position in target field (but=%d)",i+1);
           numErrors++;
           fieldOK = CONFIG_FALSE;
        }

    }



    /* End of invalid position check */
    /*
     *  Check that fibre extension, fibre/button angle and pivot/fibre
     *  angle are within limits. Also check for target within plate limits.
     *  (These were separate loops in JW's original code. Combining them
     *  simplifies the use of the reporting function.)
     */
    for (i=0; i<NPiv; i++) {

       if (ReportFunction != NULL) {
          if ((i % ReportPivs) == 0) {
             PercentDone = ((float) i / (float) NPiv) * 100.0;
             if ((*ReportFunction)(FunctionArg,'P',
                     "Checking fibre extensions, fibre and button angles",
                                         0,0,0,0,0,0.0,PercentDone) != 0) {
                MethodInfo->InfoRoutine(MethodInfo->InfoArgument,
                                        "Cancel request received");
                *status = CONFIG__CANCELLED;
                goto Exit;
             }
           }
        } 

        /*
         *  No need to check an unallocated fibre.
         */
        if (AllocInfo[i].State != 'A') continue;

        /*
         *  Check fibre extension.
         */
        deltaX = (double) AllocInfo[i].X - PivotInfo[i].X;
        deltaY = (double) AllocInfo[i].Y - PivotInfo[i].Y;
        AllocLength = sqrt((deltaX * deltaX) + (deltaY * deltaY));
        deltaX = (double) PivotInfo[i].ParkX - PivotInfo[i].X;
        deltaY = (double) PivotInfo[i].ParkY - PivotInfo[i].Y;
        ParkedLength = sqrt((deltaX * deltaX) + (deltaY * deltaY));
        FibreLength = (double) PivotInfo[i].FibreLength;
        if (PositionerTolerances) FibreLength += CONFIG_LENGTH_ADJUST;
        if (FibreLength < AllocLength) {
            ErsRep(0,status,
              "Maximum fibre length exceeded in target field (piv=%d)",i+1);
            numErrors++;
            fieldOK = CONFIG_FALSE;
        }
        if (AllocLength < ParkedLength) {
            ErsRep(0,status,
              "Attempt to allocate fibre behind park position (piv=%d)",i+1);
            numErrors++;
            fieldOK = CONFIG_FALSE;
        }

        /*
         *  Calculate angle of fibre about pivot point (theta=0 @ 12o'clock,
         *  increasing anti-clockwise).
         */
        ConfigGetVirtPiv (MethodInfo,
                          (double) AllocInfo[i].X,
                          (double) AllocInfo[i].Y,
                          AllocInfo[i].Theta,&fvpx,&fvpy);
        dx = fvpx - PivotInfo[i].X;
        dy = fvpy - PivotInfo[i].Y;
        tmpdx = (dx < 0)? -dx: dx;
        tmpdy = (dy < 0)? -dy: dy;
        if (dx>0 && dy==0)
            thetaFib = 3*PI/2;
        else if (dx>0 && dy>0)
            thetaFib = 3*PI/2 + atan((double)tmpdy/(double)tmpdx);
        else if (dx==0 && dy>0)
            thetaFib = 0;
        else if (dx<0 && dy>0)
            thetaFib = PI/2 - atan((double)tmpdy/(double)tmpdx);
        else if (dx<0 && dy==0)
            thetaFib = PI/2;
        else if (dx<0 && dy<0)
            thetaFib = PI/2 + atan((double)tmpdy/(double)tmpdx);
        else if (dx==0 && dy<0)
            thetaFib = PI;
        else /* (dx>0 && dy<0) */
            thetaFib = 3*PI/2 - atan((double)tmpdy/(double)tmpdx);

        /*
         *  Calulate fibre/button fibre/pivot angles.
         */
        thetaButFib = thetaFib - AllocInfo[i].Theta - PI;
        thetaPivFib = thetaFib - PivotInfo[i].Theta - PI;

        /*
         *  Get absolute value of angles, between 0 and 2PI.
         */
        thetaButFib = thetaButFib<0 ? -thetaButFib: thetaButFib;
        while (thetaButFib > PI) {
            thetaButFib -= (2*PI);
            thetaButFib = thetaButFib<0 ? -thetaButFib: thetaButFib;
        }
        thetaPivFib = thetaPivFib<0 ? -thetaPivFib: thetaPivFib;
        while (thetaPivFib > PI) {
            thetaPivFib -= (2*PI);
            thetaPivFib = thetaPivFib<0 ? -thetaPivFib: thetaPivFib;
        }

        /*
         *  Compare bend angles against maximum values.  Note that
         *  not all instruments have a variable button/fibre angle.
         */
        if (FpilGetFibAngVar(Instrument))
        {
            if (thetaButFib > MaxButAngle) {
                ErsRep(0,status,
   "Out of range %s angle (button) detected in target field (piv=%d,ang=%.3f)",
                       "button/fibre",i+1,thetaButFib*180/PI);
                numErrors++;
                fieldOK = CONFIG_FALSE;
            }
        }
        if (thetaPivFib > MaxPivAngle) {
            ErsRep(0,status,
   "Out or range %s angle (pivot) detected in target field (piv=%d,ang=%.3f)",
                   "pivot/fibre",i+1,thetaPivFib*180/PI);
            numErrors++;
            fieldOK = CONFIG_FALSE;
        }
 
        /*
         *  Currently just check if within field radius.
         */
        if (!FpilOnField(Instrument,
                         AllocInfo[i].X,
                         AllocInfo[i].Y)) {
            ErsRep(0,status,
               "Button outside usable field plate area and not parked (piv=%d)",
               i+1);
            numErrors++;
            fieldOK = CONFIG_FALSE;
        }
    }

 Exit:;

    /*  Revert to current pivot and button angle, and clearance values */
    
    FpilSetMaxPivAng(CONFIG_Global.MethodInfo._Instrument,CurrentMaxPivAngle);
    FpilSetMaxFibAng(CONFIG_Global.MethodInfo._Instrument,CurrentMaxButAngle);
    FpilSetFibClear(CONFIG_Global.MethodInfo._Instrument,CurrentFibClear);
    FpilSetButClear(CONFIG_Global.MethodInfo._Instrument,CurrentButClear);

    /*
     *  Summarise field check findings.
     */
    if (!fieldOK) {
        *status = CONFIG__INVAL_CONFIG;
        ErsRep(0,status,
          "Target field configuration is INVALID - %d error(s) detected",
               numErrors);
    }

}

/*
 * External interface to field check.
 */

void ConfigFieldCheck (          /*     Performs a full field validation */
   CONFIG_ReportFunction ReportFunction,
                                 /* (>) Feedback routine to call */
   void *FunctionArg,            /* (>) Caller-supplied argument */
   int PositionerTolerances,     /* (>) If true, use positioner tolerances */
   StatusType  *status)          /* (!) Inherited status variable */
{
    ConfigHKFieldCheck(
        &CONFIG_Global.MethodInfo,
        ReportFunction,
        FunctionArg,
        PositionerTolerances,
        status);
}

/* -------------------------------------------------------------------------- */

/*                    C o n f i g  I n i t i a l i s e
 *
 *  Function:
 *     Initialises the allocation algorithms.
 *
 *  Description:
 *     This routine initialises the allocation algorithms, reading the 
 *     global pivot information from an SDS structure that should contain 
 *     the latest version of this information. This SDS structure can be
 *     passed as an already open SDS ID, or as the name of a file containing
 *     the structure. If the SDS ID passed to it is zero, the structure is 
 *     read from the specified file. Any necessary internal initialisations
 *     are performed. ConfigClose() should be called to release any
 *     resources obtained by the allocation routines. After ConfigClose()
 *     has been called, ConfigInitialise() can be called again in necessary
 *     to re-initialise the routines, perhaps for a different pivot setup.
 *
 */

void ConfigInitialise (
   FpilType          Instrument, /* (>) Fibre Instrument Description */
   FpilConstantsType InstConstDetails, /* (>)  Instrument constants details */
   ConfigInfoRoutine InfoRoutine,/* (>) Info reporting routine */
   void *FunctionArg,             /* (>) Caller-supplied argument */
   StatusType *Status)              /* (!) Inherited status variable */
{
   unsigned int index;
   CONFIG_MethodDetails DefaultMethodDetails;
   long FibreClear;
   long ButtonClear;
   long ConfigClear;
   double DefaultAngle;
   double MaxPivAngle;
   double MaxFibAngle;

   if (*Status != STATUS__OK) return;

   /*  Initialise the general global variables  */

   for (index = 0; index < FPIL_MAXFIELDS ; ++index)
   {
       CONFIG_Global.AllocInfoPtr[index] = NULL;
       CONFIG_Global.NumberPivots[index] = FpilGetNumPivots(Instrument);
       CONFIG_Global.PivotInfoPtr[index] = NULL;
   }
   CONFIG_Global.NumberTargets = 0;
   CONFIG_Global.TargetInfoPtr = NULL;

   /*  Set the various controllable limit parameters to the values given
    *  currently by FPIL and set appropiate range limits based on the
    *  current FPIL values. Note that we assume that the values used by
    *  the positioner itself will be the default values that we are passed
    *  at this time. To allow enough headroom in our allocations, the initial
    *  values used are constrained by a percentage given by CONFIG_PERCENT.
    */

   /*
    *  Button Clearance.  The range should be from the FPIL value up.
    */
   ButtonClear = FpilGetButClear(Instrument);
   ConfigClear = (int)(((double)ButtonClear * (1.0 + (CONFIG_PERCENT / 100.0))) + 0.5);
   CONFIG_Global.ParameterValues[CONFIG_BUTTON_CLEAR_PARM].IntegerValue = ConfigClear;
   fpossButClear = ConfigClear;

   CONFIG_STATIC_ParmLimits[CONFIG_BUTTON_CLEAR_PARM][0] = ButtonClear;
   CONFIG_STATIC_ParmLimits[CONFIG_BUTTON_CLEAR_PARM][1] = ButtonClear*100;
   CONFIG_Global.PositionerButClear = ButtonClear;


   /*
    *  Fibre Clearance.  The range should be from the FPIL value up.
    */
   FibreClear = FpilGetFibClear(Instrument);
   ConfigClear = (int)(((double)FibreClear * (1.0 + (CONFIG_PERCENT / 100.0))) + 0.5);
   CONFIG_Global.ParameterValues[CONFIG_FIBRE_CLEAR_PARM].IntegerValue = ConfigClear;
   fpossFibClear = ConfigClear;

   CONFIG_STATIC_ParmLimits[CONFIG_FIBRE_CLEAR_PARM][0] = FibreClear;
   CONFIG_STATIC_ParmLimits[CONFIG_FIBRE_CLEAR_PARM][1] = FibreClear*100;
   CONFIG_Global.PositionerFibClear = FibreClear;


   /*
    *  Maximum pivot angle.  Range should be from zero to the FPIL value.
    *
    *  Remember that FPIL returns radians whilst the parameter is in
    *  degrees. The rather messy calculation of default angle gives a value
    *  rounded down to the nearest half a degree, which looks neater than
    *  just the value from a straight calculation of the percentage.
    */
   MaxPivAngle = FpilGetMaxPivAng(Instrument) * 360.0 / (2.0 * PI);
   DefaultAngle = ((double)((int)(MaxPivAngle * 
                       (1.0 - (CONFIG_PERCENT / 100.0)) * 2.0))) * 0.5;
   CONFIG_Global.ParameterValues[CONFIG_MAX_PIV_ANGLE_PARM].DoubleValue = 
       DefaultAngle;
   CONFIG_STATIC_ParmLimits[CONFIG_MAX_PIV_ANGLE_PARM][0] = 0;
   CONFIG_STATIC_ParmLimits[CONFIG_MAX_PIV_ANGLE_PARM][1] = MaxPivAngle;
   CONFIG_Global.PositionerMaxPivAng = FpilGetMaxPivAng(Instrument);


   /*
    *  Maximum button/fibre angle.  This angle is not always setable, but
    *  in those cases, the value returned should be zero and our range
    *  will be from zero to zero.
    *
    *  Otherwise, the range should be from zero to the FPIL value.
    *
    *  Remember that FPIL returns radians whilst the parameter is in
    *  degrees.
    */
   MaxFibAngle = FpilGetMaxFibAng(Instrument) * 360.0 / (2.0 * PI);
   DefaultAngle = ((double)((int)(MaxFibAngle * 
                       (1.0 - (CONFIG_PERCENT / 100.0)) * 2.0))) * 0.5;
   CONFIG_Global.ParameterValues[CONFIG_MAX_BUT_ANGLE_PARM].DoubleValue = 
       DefaultAngle;
   CONFIG_STATIC_ParmLimits[CONFIG_MAX_BUT_ANGLE_PARM][0] = 0;
   CONFIG_STATIC_ParmLimits[CONFIG_MAX_BUT_ANGLE_PARM][1] = MaxFibAngle;
   CONFIG_Global.PositionerMaxFibAng = FpilGetMaxFibAng(Instrument);

   /* Fill out the structure describing all the house keeping routines
    * which may be called by the allocation method layer */
   CONFIG_Global.MethodInfo.MaxExtension     = ConfigMaxExtension;
   CONFIG_Global.MethodInfo.GetVirtPiv       = ConfigGetVirtPiv;
   CONFIG_Global.MethodInfo.ThetaFib         = ConfigThetaFib;
   CONFIG_Global.MethodInfo.CheckAlpha       = ConfigCheckAlpha;
   CONFIG_Global.MethodInfo.CheckAllocation  = ConfigCheckAllocation;
   CONFIG_Global.MethodInfo.FieldCheck       = ConfigHKFieldCheck;
   CONFIG_Global.MethodInfo.NoteAllocation   = ConfigNoteAllocation;
   CONFIG_Global.MethodInfo.TentativeFix     = ConfigTentativeFix;
   CONFIG_Global.MethodInfo.NoteDeallocation = ConfigNoteDeallocation ;
   CONFIG_Global.MethodInfo.NoteDeallocationBroken = 
                                               ConfigNoteDeallocationBroken ;
   CONFIG_Global.MethodInfo.ButtonClear      = ConfigButtonClear;
   CONFIG_Global.MethodInfo.FibreClear       = ConfigFibreClear;

   CONFIG_Global.MethodInfo.MaxButAngle      = ConfigMaxButAngle;
   CONFIG_Global.MethodInfo.MaxPivAngle      = ConfigMaxPivAngle;
   CONFIG_Global.MethodInfo.SafeClearance    = ConfigSafeClearance;

   CONFIG_Global.MethodInfo.InfoRoutine      = InfoRoutine;
   CONFIG_Global.MethodInfo.InfoArgument     = FunctionArg;

   CONFIG_Global.MethodInfo.ErrRepRoutine    = ConfigErrRep;

   CONFIG_Global.MethodInfo.ColFibFib        = ConfigColFibFib;
   CONFIG_Global.MethodInfo.ColButFib        = ConfigColButFib;
   CONFIG_Global.MethodInfo.ColButBut        = ConfigColButBut;
   CONFIG_Global.MethodInfo.ColInvPos        = ConfigColInvPos;
   CONFIG_Global.MethodInfo.Collision        = ConfigCollision;
   CONFIG_Global.MethodInfo.AdjPark          = ConfigAdjPark;
   CONFIG_Global.MethodInfo.FibreReallyExists= ConfigFibreReallyExists;
   CONFIG_Global.MethodInfo.ParkMayCollide   = ConfigParkMayCollide;
   CONFIG_Global.MethodInfo.SetFibrePos      = ConfigSetFibrePos;
   CONFIG_Global.MethodInfo.FibreIsGuide     = ConfigFibreIsGuide;
   CONFIG_Global.MethodInfo.FibreIsBroken    = ConfigFibreIsBroken;
   CONFIG_Global.MethodInfo.TargetIsGuide    = ConfigTargetIsGuide;
   CONFIG_Global.MethodInfo.TargetIsSky      = ConfigTargetIsSky;
   CONFIG_Global.MethodInfo.FibreText        = ConfigFibreText;
   CONFIG_Global.MethodInfo.FibreTypeMatch   = ConfigFibreTypeMatch;
   CONFIG_Global.MethodInfo.FibreSharesSky   = ConfigFibreSharesSky;
   CONFIG_Global.MethodInfo.FibreTargetCompatible = ConfigFibreTargetCompatible;

   /* In order to ensure the allocation algorithim is not dependent
    * on include fpil.h, CONFIG_Global.MethodInfo._Instrument is
    * defined as a void * instead of a FpilType.  I know this is safe
    * becase FpilType is really a void star but I need to check this
    * bit of unsafe programming at run time just to be sure it is not
    * changed.  TJF.
    */
   if (sizeof(CONFIG_Global.MethodInfo._Instrument) != sizeof(Instrument))
   {
       fprintf(stderr,"%s:%d Assertion failed\n", __FILE__,__LINE__);
       fprintf(stderr,"sizeof(void *) is not the same as sizeof(FpilType)\n");
       fprintf(stderr,"Talk to tjf@aaoepp.aao.gov.au\n");
       exit(-1);
   }
   CONFIG_Global.MethodInfo._Instrument       = Instrument;

   CONFIG_Global.CurrentMethod = -1;  /* minus 1 indicates no current */
   CONFIG_Global.NumMethods    = 0;   /* No methods available         */
   CONFIG_Global.MethodInfo.NumberPivots  = FpilGetNumPivots(Instrument);

   /* Grab details of the default method */
   ConfigMethodDetails(&DefaultMethodDetails);
   /* Register the default method and enable it */
   ConfigRegisterMethod( &DefaultMethodDetails,  1, &index, Status);
 
   /*  Read in the pivot information from the supplied SDS structure
    *  or file.
    */

   ConfigInitPivots (InstConstDetails,Status);
}

/* -------------------------------------------------------------------------- */

/*                    C o n f i g  S e t  T a r g e t s
 *
 *  Function:
 *     Sets up a new target field.
 *
 *  Description:
 *     This routine sets up a new target field for the allocation 
 *     algorithms, reading the target information from an SDS structure.
 *     This SDS structure can be passed as an already open SDS ID, or 
 *     as the name of a file containing the structure. If the SDS ID passed
 *     is zero, the structure is read from the specified file. If the
 *     target field contains pre-allocated fibres, the allocation will be
 *     validated by this routine. Validation can be time-consuming, and
 *     a report function can be used to report progress and accept cancel
 *     commands if necessary.
 */

void ConfigSetTargets (
   char *FileName,              /* (>) File containing structure */
   SdsIdType TopID,             /* (>) SDS ID of top of structure */
   int Field,                   /* (>) Field in question */
   CONFIG_ReportFunction 
             ReportFunction,    /* (>) Feedback routine to call */
   void *FunctionArg,           /* (>) Caller-supplied argument */
   StatusType *Status)          /* (!) Inherited status */
{
   if (*Status != STATUS__OK) return;

   /*  See if we have been passed an already open SDS structure or a
    *  file name. Call the appropriate routine - ConfigReadTargetFile()
    *  merely reads the SDS structure and then calls ConfigInitTargets().
    */

   if (TopID == (SdsIdType) 0) {
      ConfigReadTargetFile (FileName,Field,ReportFunction,FunctionArg,Status);
   } else {
      ConfigInitTargets (TopID,"SDS structure",Field,ReportFunction,
                                                          FunctionArg,Status);
   }

   /*  We trap any cancelled status - using status makes it easy for a 
    *  cancel request to 'fall through' layers of code, but it's easier for
    *  an interface if it isn't treated as an error.
    */

   if (*Status == CONFIG__CANCELLED)  ErsFlush(Status);
}

/* -------------------------------------------------------------------------- */

/*                    C o n f i g  A l l o c a t e
 *
 *  Function:
 *     Performs a complete allocation of fibres to targets.
 *
 *  Description:
 *     Once the allocation routines have been initialised using
 *     ConfigInitialise() so that the pivot details are known, and
 *     ConfigSetTargets() has been called to describe the target objects
 *     this routine can be called to attempt to allocate any unallocated
 *     fibres to targets. Between calling ConfigSetTargets() and calling
 *     this routine, it is possible to call ConfigAllocateFibre() and
 *     ConfigDeallocateFibre() to modify the set of pre-allocated targets.
 *     Allocation can be time-consuming, and a report function should be 
 *     supplied so that progress can be reported and allocation cancelled
 *     if necessary.
 */

void ConfigAllocate (
   CONFIG_ReportFunction 
             ReportFunction,    /* (>) Feedback routine to call */
   void *FunctionArg,           /* (>) Caller-supplied argument */
   CONFIG_AllocRptFunction AllocReport,
                                  /* (>) Allocation report function */
   void *AllocRptArg,             /* (>) Caller-supplied argument */
   StatusType *Status)          /* (!) Inherited status */
{
   /* Local variables */

   int Field;                   /* Field being allocated */
   int IType;                   /* Index through pivot types */
   int FibreType;               /* Fibre type for which sky counts are given */
   int SkyCount;                /* Sky count for a given fibre type */

   if (*Status != STATUS__OK) return;
   Field = CONFIG_Global.Field;

   /* Fill out method field details */
   
   CONFIG_Global.MethodInfo.Field         = Field;
   CONFIG_Global.MethodInfo.NumberPivots  = CONFIG_Global.NumberPivots[Field];
   CONFIG_Global.MethodInfo.NumberTargets = CONFIG_Global.NumberTargets;
   CONFIG_Global.MethodInfo.TargetInfo    = CONFIG_Global.TargetInfoPtr;
   CONFIG_Global.MethodInfo.PivotInfo     = CONFIG_Global.PivotInfoPtr[Field];
   CONFIG_Global.MethodInfo.AllocInfo     = CONFIG_Global.AllocInfoPtr[Field];
   CONFIG_Global.MethodInfo._AllocRpt     = AllocReport;
   CONFIG_Global.MethodInfo._AllocRptArg  = AllocRptArg;


   /* Initialise allocation algorithim and then allocate */
   
   (*CONFIG_Global.MethodDetails[CONFIG_Global.CurrentMethod].Init) 
       (&CONFIG_Global.MethodInfo,
        ReportFunction, FunctionArg,
        Status);
        
   /*  If required, set the sky target counts by fibre type */
   
   for (IType = 0; IType < CONFIG_Global.FibreSkyEntries; IType++) {
      FibreType = CONFIG_Global.FibreTypesForSky[IType];
      if (FibreType >= 0) {
         SkyCount = CONFIG_Global.FibreSkyCounts[IType];
         (*CONFIG_Global.MethodDetails[CONFIG_Global.CurrentMethod].SetSky)
             (&CONFIG_Global.MethodInfo,FibreType,SkyCount,Status);
      }
   }
   
   /*  Now do the actual allocation */
   
   (*CONFIG_Global.MethodDetails[CONFIG_Global.CurrentMethod].Allocate) 
       (&CONFIG_Global.MethodInfo,
        ReportFunction,FunctionArg,Status);
   
   /* Why was is this here ??? */
   /*ConfigMethodReleaseMem();*/
   /*ConfigReleaseMem();*/
   /* Trap any cancelled status and clear it */

   if (*Status == CONFIG__CANCELLED)
      ErsFlush(Status);
   else if (*Status == STATUS__OK) {
       /*
        * Ensure we have not been left with tentative allocations.
        */
       register int Pivot;
       CONFIG_AllocInfo *AllocInfo = CONFIG_Global.AllocInfoPtr[Field];
       for (Pivot = 0; Pivot < CONFIG_Global.NumberPivots[Field] ; ++Pivot) {
           if ((AllocInfo[Pivot].State == 'A')&&
               (AllocInfo[Pivot].Tentative)) {
               if (*Status == STATUS__OK) {
                   *Status = CONFIG__METHOD_ERR;
                   ErsRep(0,Status,
                   "Programming error in the allocation method implemetation");
               }
               ErsRep(0,Status,
                   "Allocation completed with pivot %d Tentatively allocated",
                      Pivot+1);
           }
           
       }
   }
}

/* -------------------------------------------------------------------------- */

/*                    C o n f i g  C l o s e
 *
 *  Function:
 *     Closes down the allocation algorithms.
 *
 *  Description:
 *     This routine closes down the allocation algorithms, releasing any
 *     resources obtained by the allocation routines. After ConfigClose()
 *     has been called, ConfigInitialise() can be called again in necessary
 *     to re-initialise the routines, perhaps for a different pivot setup.
 *
 */

void ConfigClose(
   StatusType *Status)              /* (!) Inherited status variable */
{
   unsigned i;
   if (*Status != STATUS__OK) {
     DEBUG("Bad status on entering ConfigClose...\n");
     return;
   }

   /*  Release resources used by the common, housekeeping and interface
    *  routines, and then by the method-specific routines.
    */

   for (i = 0 ; i < CONFIG_Global.NumMethods ; ++i)
   {
       CONFIG_Global.MethodInfo.clientData = 
           CONFIG_Global.MethodDetails[i].clientData;
       (*CONFIG_Global.MethodDetails[i].ReleaseMem) 
            (&CONFIG_Global.MethodInfo);
   }
   ConfigReleaseMem ();
}


/* -------------------------------------------------------------------------- */

/*               C o n f i g  N o t e  A l l o c a t i o n
 *
 *  Function:
 *     Records an allocation in the allocation information array.
 *
 *  Description:
 *     This is a utility routine provided for use by the method-specific
 *     routines in the allocation package. It is used to update the
 *     allocation information structure array when a new allocation is
 *     made. It really just updates that array, making things just a little
 *     cleaner - it means that the method-specific routines never themselves
 *     modify the allocation information structure array.
 */

static void ConfigNoteAllocation (
   const CONFIG_MethodInfo * MethodInfo,
   int Pivot,                    /* (>) Pivot being allocated */
   int Target,                   /* (>) Target allocated to pivot */
   double Theta,                 /* (>) Theta (twist) angle for fibre */
   int Tentative,                /* (>) Is allocation tentative */
   StatusType *Status)           /* (!) Inherited status variable */
{
   /*  Local variables  */

   CONFIG_AllocInfo *AllocInfo;  /* Allocation info array of structures */
   int Field;                    /* Field being configured */
   int NPiv;                     /* Number of pivots */
   int NTargets;                 /* Number of targets */
   CONFIG_TargetInfo *TargetInfo;/* Target info array of structures */
   CONFIG_PivotInfo  *PivotInfo; /* Address of pivot info array */
   int IsGuide;                  /* Is this a guide fibre */
   int OrigTarget;               /* Original target index */

   if (*Status != STATUS__OK) return;

   /*  Pick up local pointers to the various information structures whose
    *  addresses are held in the global structure. Also pick up the number
    *  of possible targets and pivots.
    */

   Field = CONFIG_Global.Field;
   NTargets = CONFIG_Global.NumberTargets;
   NPiv = CONFIG_Global.NumberPivots[Field];
   AllocInfo = CONFIG_Global.AllocInfoPtr[Field];
   TargetInfo = CONFIG_Global.TargetInfoPtr;
   PivotInfo = CONFIG_Global.PivotInfoPtr[Field];
   
   /*  Check that this allocation does not conflict with the information
    *  already held in the global structures. This is an internal check
    *  and probably unnecessary, but not trusting your input is a good
    *  programming rule.
    */

   if ((Pivot >= NPiv) || (Pivot < 0)) {
      ErsRep (0,Status,"Attempt to allocate non-existent pivot, # %d",
                                                                    Pivot + 1);
      *Status = CONFIG__INVAL_PIVOT;
      goto Exit;
   }

   if ((Target >= NTargets) || (Target < 0)) {
      ErsRep (0,Status,"Attempt to allocate non-existent target, # %d",
                                                                    Target + 1);
      *Status = CONFIG__INVAL_TARGET;
      goto Exit;
   }

   if (AllocInfo[Pivot].State != 'U') {
      if (AllocInfo[Pivot].State == 0) {
         if (ConfigImportOnlyMode == 0){
            ErsRep (0,Status,"Attempt to allocate broken fibre, pivot # %d",
                                                                   Pivot + 1);
            *Status = CONFIG__USE_BROKEN;
         }
      } else {
         ErsRep (0,Status,"Attempt to allocate allocated pivot # %d",
                                                                    Pivot + 1);
         *Status = CONFIG__IN_USE;
      }
      goto Exit;
   }

   if (PivotInfo[Pivot].AllowAllocChange != CONFIG_TRUE) {
       ErsRep(0,Status,
           "Attempt to allocate pivot which should not be changed, pivot # %d",
                                                                   Pivot + 1);
       *Status = CONFIG__NOCHANGE;
       goto Exit;
   }
       
   /*  Set the relevant fields in the allocation info structure array */

   AllocInfo[Pivot].TargetIndex = Target;
   AllocInfo[Pivot].X = TargetInfo[Target].X;
   AllocInfo[Pivot].Y = TargetInfo[Target].Y;
   AllocInfo[Pivot].Theta = Theta;
   AllocInfo[Pivot].State = 'A';
   AllocInfo[Pivot].Tentative = Tentative ? CONFIG_TRUE : CONFIG_FALSE;
   OrigTarget = TargetInfo[Target].OriginalIndex;

   if (FpilIsFibreGuide(MethodInfo->_Instrument,PivotInfo[Pivot].FibreType)) {
       IsGuide = CONFIG_TRUE;
   } else {
       IsGuide = CONFIG_FALSE;
   }
/*
   printf(
   "Noting%s allocation of pivot %d to %s target %d (%d), (%ld, %ld, %f)\n",
       (Tentative ? " Tentative" : ""),
       Pivot+1, 
       (IsGuide ? "Guide" : "Object"),
       Target,
       OrigTarget,
       TargetInfo[Target].X,
       TargetInfo[Target].Y,
       Theta);
*/

/*
 * If the allocation is not tentative, we report the allocation
 */
   if (!Tentative) {
       (MethodInfo->_AllocRpt)(MethodInfo->_AllocRptArg,
                           CONFIG_TRUE,
                           Pivot,
                           IsGuide,
                           OrigTarget,
                           TargetInfo[Target].X,
                           TargetInfo[Target].Y,
                           Theta,
                           Status);
   }
Exit:;

}

/* -------------------------------------------------------------------------- */

/*               C o n f i g  T e n t a t i v e  F i x
 *
 *  Function:
 *     Allows a tentative allocation to be confirmed or cancelled.
 *
 *  Description:
 *     This is a utility routine provided for use by the method-specific
 *     routines in the allocation package. It is used after a call to
 *     ConfigNoteAllocation() which specified the Tentative flag as true.
 *     It is called when the method-specific code has decided whether or
 *     not to make this a definite allocation. 
 */

static void ConfigTentativeFix (
    const CONFIG_MethodInfo * MethodInfo,
    int Pivot,                   /* Pivot being allocated */
    int Confirm,                 /* True if allocation is confirmed, false if
                                  * if it to be cancelled. */
    StatusType *Status)          /* Inherited status */
{
   CONFIG_AllocInfo *AllocInfo;  /* Allocation info array of structures */
   int NPiv;                     /* Number of pivots */
   int Field;                    /* Field being configured */

   if (*Status != STATUS__OK) return;

   Field = CONFIG_Global.Field;
   NPiv = CONFIG_Global.NumberPivots[Field];
   AllocInfo = CONFIG_Global.AllocInfoPtr[Field];

   if ((Pivot >= NPiv) || (Pivot < 0)) {
      ErsRep (0,Status,"Attempt to fix a non-existent pivot, # %d",Pivot + 1);
      *Status = CONFIG__INVAL_PIVOT;
      goto Exit;
   }

   if (AllocInfo[Pivot].Tentative) {
       if (Confirm) {
           int Target;                   /* Target object index */
           double Theta;                 /* Allocation theta */
           CONFIG_TargetInfo *TargetInfo;/* Target info array of structures */
           CONFIG_PivotInfo  *PivotInfo; /* Address of pivot info array */
           int IsGuide;                  /* Is this a guide fibre */
           int OrigTarget;               /* Original target index */

           /*  Pick up local pointers to the various information structures
            *  whose addresses are held in the global structure. 
            */

           TargetInfo = CONFIG_Global.TargetInfoPtr;
           PivotInfo = CONFIG_Global.PivotInfoPtr[Field];

           /* Fetch details */
           
           Target = AllocInfo[Pivot].TargetIndex;
           OrigTarget = TargetInfo[Target].OriginalIndex;
           Theta = AllocInfo[Pivot].Theta;
           if (FpilIsFibreGuide(MethodInfo->_Instrument,
                                             PivotInfo[Pivot].FibreType)) {
               IsGuide = CONFIG_TRUE;
           } else {
               IsGuide = CONFIG_FALSE;
           }

/*
           printf(
   "Confirming allocation of pivot %d to %s target %d (%d), (%ld, %ld, %f)\n",
                 Pivot+1, 
                 (IsGuide ? "Guide" : "Object"),
                 Target,
                 OrigTarget,
                 TargetInfo[Target].X,
                 TargetInfo[Target].Y,
                 Theta);
*/


           /* Confirm and report the allocation */
           AllocInfo[Pivot].Tentative = CONFIG_FALSE;
           (MethodInfo->_AllocRpt)(MethodInfo->_AllocRptArg,
                           CONFIG_TRUE,
                           Pivot,
                           IsGuide,
                           OrigTarget,
                           TargetInfo[Target].X,
                           TargetInfo[Target].Y,
                           Theta,
                           Status);           

       } else {
/*
 *        Deallocate it.  
 */
/*
           printf("Rejecting tentative allocation of pivot %d",
                  Pivot+1);
*/
           AllocInfo[Pivot].Tentative = CONFIG_FALSE;
           AllocInfo[Pivot].State = 'U';
/*         Not really needed, but config_method.c
 *         was doing it so may want it for backward complatiblity 
 */
           AllocInfo[Pivot].TargetIndex = 0;  
       }
   } else {
/*
       printf("Attempt to fix non-tentative allocation of pivot %d\n",Pivot+1);
*/
       *Status = CONFIG__FIXNOTTENT;
   }

 Exit:;
    
}

/* -------------------------------------------------------------------------- */

/*               C o n f i g  N o t e  D e a l l o c a t i o n
 *
 *  Function:
 *     Records a deallocation in the allocation information array.
 *
 *  Description:
 *     This is a utility routine provided for use by the method-specific
 *     routines in the allocation package. It is used to update the
 *     allocation information structure array when a fibre is deallocated.
 *     It really just updates that array, making things just a little
 *     cleaner - it means that the method-specific routines never themselves
 *     modify the allocation information structure array.
 */

static void ConfigNoteDeallocation (
   const CONFIG_MethodInfo * MethodInfo,
   int Pivot,                    /* (>) Pivot being allocated */
   StatusType *Status)           /* (!) Inherited status variable */
{
   /*  Local variables  */

   CONFIG_AllocInfo *AllocInfo;  /* Allocation info array of structures */
   CONFIG_TargetInfo *TargetInfo;/* Target info array of structures     */
   CONFIG_PivotInfo  *PivotInfo; /* Address of pivot info array */
   int Field;                    /* Field being configured */
   int NPiv;                     /* Number of pivots */
   int TargetIndex;              /* Index of target in TargetInfo */
   int IsGuide;                  /* Is the pivot a guide pivot */
   
   if (*Status != STATUS__OK) return;

   /*  Pick up local pointers to the various information structures whose
    *  addresses are held in the global structure. Also pick up the number
    *  of possible targets and pivots.
    */

   Field = CONFIG_Global.Field;
   NPiv = CONFIG_Global.NumberPivots[Field];
   AllocInfo = CONFIG_Global.AllocInfoPtr[Field];
   TargetInfo = CONFIG_Global.TargetInfoPtr;
   PivotInfo = CONFIG_Global.PivotInfoPtr[Field];

   if ((Pivot >= NPiv) || (Pivot < 0)) {
      ErsRep (0,Status,"Attempt to dellocate non-existent pivot, # %d",
                                                                Pivot + 1);
      *Status = CONFIG__INVAL_PIVOT;
      goto Exit;
   }
   if (AllocInfo[Pivot].State != 'A') {
      ErsRep (0,Status,"Attempt to deallocate unallocated pivot, # %d",
                                                                Pivot + 1);
      *Status = CONFIG__INVAL_PIVOT;
      goto Exit;
   }

   if (PivotInfo[Pivot].AllowAllocChange != CONFIG_TRUE) {
       ErsRep(0,Status,
         "Attempt to deallocate pivot which should not be changed, pivot # %d",
                                                               Pivot + 1);
       *Status = CONFIG__NOCHANGE;
       goto Exit;
   }
      
   if (AllocInfo[Pivot].Tentative) {
       printf("Attempt to deallocate tentatively allocation pivot %d\n",
              Pivot + 1);
   }

   AllocInfo[Pivot].State = 'U';
   TargetIndex = AllocInfo[Pivot].TargetIndex;
   AllocInfo[Pivot].TargetIndex = 0;  /* Not really needed, but config_method.c
                                         was doing it so may want it for
                                         backward complatiblity */

   if (FpilIsFibreGuide(MethodInfo->_Instrument,PivotInfo[Pivot].FibreType)) {
       IsGuide = CONFIG_TRUE;
   } else {
       IsGuide = CONFIG_FALSE;
   }
   
/*
printf("Noting Deallocation of %s pivot %d\n",
       (IsGuide ? "Guide" : "Object"),
       Pivot+1);
*/

/*
 * If this pivot was not tentatively allocated, we report the deallocation
 */
   if (!AllocInfo[Pivot].Tentative) {
       (MethodInfo->_AllocRpt)(MethodInfo->_AllocRptArg, CONFIG_FALSE,
                                           Pivot,IsGuide,0,0,0,0,Status);
   }

   /*
    * If this was a preallocated object, then it does not have
    * an appropiate OriginalIndex entry.  Generate one based on
    * the number of unallocated objects of the appropiate type
    * and then increment that number.  This is required for the
    * "configure" program which puts such deallocated fibres back in
    * the appropiate array during execution of the Report Routine.
    *
    * (Unclear if there are other implications here ?)
    */
   if ((TargetInfo[TargetIndex].PreAllocated)&&
       (TargetInfo[TargetIndex].OriginalIndex == -1)) {
      if (TargetInfo[TargetIndex].Type == 'F'){
         TargetInfo[TargetIndex].OriginalIndex = CONFIG_Global.NUnGuide++;
      } else {
         TargetInfo[TargetIndex].OriginalIndex = CONFIG_Global.NUnTarget++;
      }
      TargetInfo[TargetIndex].PreAllocated = CONFIG_FALSE;
   } 

Exit:;

}

/*
 * A version of the above routine the algorithim layer can call if it
 * finds when reading an existing configuration that an allocation
 * has been made of a broken fibre.  
 */
static void ConfigNoteDeallocationBroken (
   const CONFIG_MethodInfo * MethodInfo,
   int Pivot,                    /* (>) Pivot being allocated */
   StatusType *Status)           /* (!) Inherited status variable */
{
    ConfigNoteDeallocation(MethodInfo, Pivot, Status);
    if (*Status == STATUS__OK)
        CONFIG_Global.AllocInfoPtr[CONFIG_Global.Field][Pivot].State = 0;
    
}
/* -------------------------------------------------------------------------- */

/*               C o n f i g  B u t t o n  C l e a r
 *
 *  Function:
 *     Returns the current limit used for the clearance around a button.
 *
 *  Description:
 *     This is a utility routine used within the 'housekeeping' layer
 *     of routines and provided for use by the algorithm-specific code.
 *     It returns the current value being used for the button clearance.
 *     This is originally read from the include files as BUTTON_CLEAR,
 *     but can be modified by the external routines to impose tighter
 *     constraints on the configuration - to enable it to remain valid 
 *     over a greater range of telescope angles, for example. It returns the
 *     value currently held in the global variables, but these are not
 *     accessible from outside this file, so a callable routine is needed.
 *     The value returned is in microns.
 */

static long ConfigButtonClear (
   const CONFIG_MethodInfo * MethodInfo DUNUSED)
{
   return(CONFIG_Global.ParameterValues[CONFIG_BUTTON_CLEAR_PARM].IntegerValue);
}

/* -------------------------------------------------------------------------- */

/*               C o n f i g  F i b r e  C l e a r
 *
 *  Function:
 *     Returns the current limit used for the clearance around a fibre.
 *
 *  Description:
 *     This is a utility routine used within the 'housekeeping' layer
 *     of routines and provided for use by the algorithm-specific code.
 *     It returns the current value being used for the fibre clearance.
 *     This is originally read from the include files as FIBRE_CLEAR, and
 *     the general comments about BUTTON_CLEAR in ConfigButtonClear() apply
 *     to this value as well. The value returned is in microns.
 */

static long ConfigFibreClear (
   const CONFIG_MethodInfo * MethodInfo DUNUSED)
{
   return(CONFIG_Global.ParameterValues[CONFIG_FIBRE_CLEAR_PARM].IntegerValue);
}

/* -------------------------------------------------------------------------- */

/*               C o n f i g  M a x  P i v  A n g l e
 *
 *  Function:
 *     Returns the current limit used for the non-radial fibre angle.
 *
 *  Description:
 *     This is a utility routine used within the 'housekeeping' layer
 *     of routines and provided for use by the algorithm-specific code.
 *     It returns the current value being used for the non-radial fibre
 *     angle (the angle between the fibre and the line running from the 
 *     pivot to the centre of the field). This is originally read from the 
 *     include files as MAX_PIV_ANGLE, and the general comments about 
 *     BUTTON_CLEAR in ConfigButtonClear() apply to this value as well.
 *     The value returned is in radians.
 */

static double ConfigMaxPivAngle (
   const CONFIG_MethodInfo * MethodInfo DUNUSED)
{
   /*  Remember that the parameter is specified in degrees  */

   return ( 
      CONFIG_Global.ParameterValues[CONFIG_MAX_PIV_ANGLE_PARM].DoubleValue 
                                                         * 2.0 * PI / 360.0);
}

/* -------------------------------------------------------------------------- */

/*               C o n f i g  M a x  B u t  A n g l e
 *
 *  Function:
 *     Returns the current limit used for the fibre twist angle.
 *
 *  Description:
 *     This is a utility routine used within the 'housekeeping' layer
 *     of routines and provided for use by the algorithm-specific code.
 *     It returns the current value being used for the fibre twist
 *     angle (the angle between the fibre and the line running along the
 *     middle of the button). This is originally read from the 
 *     include files as MAX_BUT_ANGLE, and the general comments about 
 *     BUTTON_CLEAR in ConfigButtonClear() apply to this value as well.
 *     The value returned is in radians.
 */

static double ConfigMaxButAngle (
   const CONFIG_MethodInfo * MethodInfo DUNUSED)
{
   /*  Remember that the parameter is specified in degrees  */

   return ( 
      CONFIG_Global.ParameterValues[CONFIG_MAX_BUT_ANGLE_PARM].DoubleValue 
                                                         * 2.0 * PI / 360.0);
}

/* -------------------------------------------------------------------------- */

/*               C o n f i g  S a f e  C l e a r a n c e
 *
 *  Function:
 *     Returns a safe value for the clearance required around a button.
 *
 *  Description:
 *     This is a utility routine used within the 'housekeeping' layer
 *     of routines and provided for use by the algorithm-specific code.
 *     It calculates a 'safe' value that can be used as the distance that
 *     must be left clear around a button. This is used to calculate 
 *     'safe rectangles' around a button. Anything that is more than
 *     this 'safe' distance from the button target position is known to
 *     be clear of the button. Something closer may also be clear of 
 *     the button, but this then becomes a tricky calculation involving
 *     the angle of the button and its precise geometric shape. Using this
 *     safe distance concept can limit the use of the more complex - and
 *     time-consuming - tests to those cases where the test object really is
 *     awkwardly close to the button. The value returned is in microns.
 */

static long ConfigSafeClearance (
   const CONFIG_MethodInfo * MethodInfo)
{
   return (FpilGetMaxButSize(MethodInfo->_Instrument));
}

/* ------------------------------------------------------------------------- */

/*             C o n f i g C o l F i b F i b
 *
 *  Function:
 *     Called by method layer to perform fibre-fibre collision detection.
 *
 *  Description:
 */

static int  ConfigColFibFib (
   const CONFIG_MethodInfo *MethodInfo,/* (>) Supplies house keeping details */
   double  pivAX,                /* (>) Fibre A X ordinate */
   double  pivAY,                /* (>) Fibre A Y ordinate */
   double  fvpAX,                /* (>) Fibre A virtual pivot X ordinate   */
   double  fvpAY,                /* (>) Fibre A virtual pivot Y ordinate */
   double  pivBX,                /* (>) Fibre B X ordinate */
   double  pivBY,                /* (>) Fibre B Y ordinate */
   double  fvpBX,                /* (>) Fibre B virtual pivot X ordinate   */
   double  fvpBY)                /* (>) Fibre B virtual pivot Y ordinate */
{
    FpilType Instrument = MethodInfo->_Instrument;
    return FpilColFibFib(Instrument, 
                         pivAX, pivAY, fvpAX, fvpAY,
                         pivBX, pivBY, fvpBX, fvpBY);
}
/* ------------------------------------------------------------------------- */


/*             C o n f i g  C o l B u t F i b
 *
 *  Function:
 *     Called by method layer to perform button-fibre collision detection.
 *
 *  Description:
 */

static int  ConfigColButFib (
   const CONFIG_MethodInfo *MethodInfo,/* (>) Supplies house keeping details */
   double butX,                  /* (>) Button X ordinate */
   double butY,                  /* (>) Button Y ordinate */
   double theta,                 /* (>) Button theta      */
   double fvpX,                  /* (>) Fibre virtual pivot X ordinate */
   double fvpY,                  /* (>) Fibre virtual pivot Y ordinate */
   double pivX,                  /* (>) Fibre pivot X ordinate */
   double pivY)                  /* (>) Fibre pivot Y ordinate*/
{
    FpilType Instrument = MethodInfo->_Instrument;
    return FpilColButFib(Instrument, 
                         butX, butY, theta,
                         fvpX, fvpY, pivX, pivY);
}

/* ------------------------------------------------------------------------- */

/*             C o n f i g C o l B u t B u t
 *
 *  Function:
 *     Called by method layer to perform button-button collision detection.
 *
 *  Description:
 */

static int  ConfigColButBut (
   const CONFIG_MethodInfo *MethodInfo,/* (>) Supplies house keeping details */
   double butXa,                 /* (>) Button A X ordinate */
   double butYa,                 /* (>) Button A Y ordinate */
   double thetaa,                /* (>) Button A theta      */
   double butXb,                 /* (>) Button B X ordinate */
   double butYb,                 /* (>) Button B Y ordinate */
   double thetab)                /* (>) Button B theta      */
{
    FpilType Instrument = MethodInfo->_Instrument;
    return FpilColButBut(Instrument, 
                         butXa, butYa, thetaa,
                         butXb, butYb, thetab);
}

/* ------------------------------------------------------------------------- */


/*             C o n f i g C o l I n v P o s
 *
 *  Function:
 *     Called by method layer to perform invalid position collision detection.
 *
 */

static int  ConfigColInvPos (
   const CONFIG_MethodInfo *MethodInfo,/* (>) Supplies house keeping details */
   int       fibreType,		 /* (>) Fibre type        */
   long int  butx,               /* (>) Button X ordinate */
   long int  buty,               /* (>) Button Y ordinate */
   double    theta)              /* (>) Button theta      */

{
    FpilType Instrument = MethodInfo->_Instrument;
    return FpilColInvPos(Instrument, 
                         CONFIG_Global.Field,
			 fibreType, butx, buty, theta);
}

/* -------------------------------------------------------------------------- */

/*             C o n f i g C o l l i s i o n
 *
 *  Function:
 *     Called by method layer to perform collision detection between
 *     two buttons.  Does a button-button check, button to fibre and
 *     fibre to fibre check.
 *
 */
static int  ConfigCollision (
    const CONFIG_MethodInfo *MethodInfo,/* (>) house keeping details*/
    long int butXa,                     /* (>) Button A fibre X position */
    long int butYa,                     /* (>) Button A fibre Y position */
    double thetaa,                      /* (>) Button A theta angle      */
    long int pivXa,                     /* (>) Button A pivot X position */
    long int pivYa,                     /* (>) Button A pivot Y position */
    long int butXb,                     /* (>) Button B fibre X position */
    long int butYb,                     /* (>) Button B fibre Y position */
    double thetab,                      /* (>) Button B theta angle      */
    long int pivXb,                     /* (>) Button B pivot X position */
    long int pivYb)                     /* (>) Button B pivot Y position */
{
    FpilType Instrument = MethodInfo->_Instrument;
    return FpilCollision1(Instrument,
                         butXa, butYa, thetaa, pivXa, pivYa,
                         butXb, butYb, thetab, pivXb, pivYb);
}

/* -------------------------------------------------------------------------- */

/*             C o n f i g  A d j  P a r k
 *
 *  Function:
 *     Determines if a given allocation of a pivot to a target will
 *     result in an overlap between the fibre and the parked positions
 *     of the adjacent fibres. If parked positions cannot collide with
 *     fibres, then this routine always returns false.
 *
 */
 
static int  ConfigAdjPark (
    const CONFIG_MethodInfo *MethodInfo,/* (>) house keeping details*/
    int IPiv,                           /* (>) The pivot number */
    long int targetX,                   /* (>) The target X position */
    long int targetY,                   /* (>) The target Y position */
    double theta)                       /* (>) Pivot theta angle */
{
    int Collision = CONFIG_FALSE;     /*  Eventual returned value */
    int HigherPivot;                  /*  Index of adjacent pivot - above */
    int LowerPivot;                   /*  Index of adjacent pivot - below */
    long ParkButtonX;                 /*  X position of parked button */
    long ParkButtonY;                 /*  Y position of parked button */
    long ParkPivotX;                  /*  X position of parked pivot */
    long ParkPivotY;                  /*  Y position of parked pivot */
    double ParkTheta;                 /*  Theta angle for parked button */
    double ParkVirtPivX;              /*  X position of parked virtual pivot */
    double ParkVirtPivY;              /*  Y position of parked virtual pivot */
    long PivotX;                      /*  X position of pivot for the fibre */
    long PivotY;                      /*  Y position of pivot for the fibre */
    double VirtPivX;                  /*  X position of botton virtual pivot */
    double VirtPivY;                  /*  Y position of botton virtual pivot */
    
    FpilType Instrument = MethodInfo->_Instrument;
    
    /*  If parked pivots cannot cause a collision, we don't have to
     *  do any more detailed tests.
     */
    
    if (FpilParkMayCollide(Instrument)) { 
    
        /*  Get the virtual pivot location for the button at the target 
         *  position, and the position of the pivot point for the fibre.
         */
    
        FpilGetVirtPiv(Instrument, targetX, targetY, theta, &VirtPivX, 
                                                                  &VirtPivY);
        PivotX = MethodInfo->PivotInfo[IPiv].X;
        PivotY = MethodInfo->PivotInfo[IPiv].Y;
    
        /*  Check for a collision with the pivot on the lower numbered side */
    
        LowerPivot = IPiv - 1;
        if (LowerPivot < 0) LowerPivot = MethodInfo->NumberPivots - 1;
        if (FpilFibreReallyExists(Instrument,
                        MethodInfo->PivotInfo[LowerPivot].FibreType)) {
            ParkButtonX = MethodInfo->PivotInfo[LowerPivot].ParkX;
            ParkButtonY = MethodInfo->PivotInfo[LowerPivot].ParkY;
            ParkTheta = MethodInfo->PivotInfo[LowerPivot].Theta;
            if (FpilColButBut(Instrument,ParkButtonX,ParkButtonY,ParkTheta,
                                             targetX,targetY,theta)) {
                Collision = CONFIG_TRUE;
            } else {
                if (FpilColButFib(Instrument,ParkButtonX,ParkButtonY,ParkTheta,
                               VirtPivX,VirtPivY,PivotX,PivotY)) {
                    Collision = CONFIG_TRUE;
                } else {
                    ParkPivotX = MethodInfo->PivotInfo[LowerPivot].X;
                    ParkPivotY = MethodInfo->PivotInfo[LowerPivot].Y;
                    FpilGetVirtPiv(Instrument, ParkButtonX, ParkButtonY,
                               ParkTheta, &ParkVirtPivX, &ParkVirtPivY);
                    if (FpilColButFib(Instrument,targetX,targetY,theta,
                           ParkVirtPivX,ParkVirtPivY,ParkPivotX,ParkPivotY)) {
                        Collision = CONFIG_TRUE;
                    }
                }
            }
        }
        if (!Collision) {
            HigherPivot = IPiv + 1;
            if (HigherPivot >= MethodInfo->NumberPivots) HigherPivot = 0;
            if (FpilFibreReallyExists(Instrument,
                    MethodInfo->PivotInfo[HigherPivot].FibreType)) {
                ParkButtonX = MethodInfo->PivotInfo[HigherPivot].ParkX;
                ParkButtonY = MethodInfo->PivotInfo[HigherPivot].ParkY;
                ParkTheta = MethodInfo->PivotInfo[HigherPivot].Theta;
                if (FpilColButBut(Instrument,ParkButtonX,ParkButtonY,ParkTheta,
                                                  targetX,targetY,theta)) {
                    Collision = CONFIG_TRUE;
                } else {
                    if (FpilColButFib(Instrument,ParkButtonX,ParkButtonY,
                               ParkTheta,VirtPivX,VirtPivY,PivotX,PivotY)) {
                        Collision = CONFIG_TRUE;
                    } else {
                        ParkPivotX = MethodInfo->PivotInfo[HigherPivot].X;
                        ParkPivotY = MethodInfo->PivotInfo[HigherPivot].Y;
                        FpilGetVirtPiv(Instrument, ParkButtonX, ParkButtonY,
                               ParkTheta, &ParkVirtPivX, &ParkVirtPivY);
                        if (FpilColButFib(Instrument,targetX,targetY,theta,
                            ParkVirtPivX,ParkVirtPivY,ParkPivotX,ParkPivotY)) {
                            Collision = CONFIG_TRUE;
                        }
                    }
                }
            }
        }
    }
    return (Collision);
}
        

/* -------------------------------------------------------------------------- */

/*             C o n f i g  F i b r e  R e a l l y  E x i s t s
 *
 *  Function:
 *     Determines if a given fibre really exists, based on the pivot
 *     number it has.
 */
 
static int  ConfigFibreReallyExists (
    const CONFIG_MethodInfo *MethodInfo,/* (>) house keeping details*/
    int IPiv)                           /* (>) Pivot index */
{
    
    FpilType Instrument = MethodInfo->_Instrument;
    
    return (FpilFibreReallyExists(Instrument,
                           MethodInfo->PivotInfo[IPiv].FibreType));
}
        
/* ------------------------------------------------------------------------- */

/*               C o n f i g P a r k M a y C o l l i d e
 *
 *  Function:
 *     Called by method layer to determine if park positions may
 *     collide with other fibres.
 *
 */
static int  ConfigParkMayCollide (
    const CONFIG_MethodInfo *MethodInfo)/* (>) house keeping details*/
{
    FpilType Instrument = MethodInfo->_Instrument;
    return FpilParkMayCollide(Instrument);
}

/* ------------------------------------------------------------------------- */

/*               C o n f i g  F i b r e  I s  G u i d e
 *
 *  Function:
 *     Called by method layer to determine if a given fibre type code 
 *     indicates that the fibre in question is a guide fibre. (It returns
 *     true if the fibre is a guide fibre type. Note that a broken guide
 *     fibre is still treated as a guide fibre.)
 *
 */
 
static int  ConfigFibreIsGuide (
    const CONFIG_MethodInfo *MethodInfo, /* (>) house keeping details*/
    int FibreType)                       /* (>) Fibre type code */
{
    FpilType Instrument = MethodInfo->_Instrument;
    return FpilIsFibreGuide(Instrument,FibreType);
}

/* ------------------------------------------------------------------------- */

/*               C o n f i g  F i b r e  I s  B r o k e n
 *
 *  Function:
 *     Called by method layer to determine if a given fibre type code 
 *     indicates that the fibre in question is broken. (It returns
 *     true if the fibre is broken, false otherwise.)
 *
 */
 
static int  ConfigFibreIsBroken (
    const CONFIG_MethodInfo *MethodInfo, /* (>) house keeping details*/
    int FibreType)                       /* (>) Fibre type code */
{
    FpilType Instrument = MethodInfo->_Instrument;
    return FpilIsFibreBroken(Instrument,FibreType);
}

/* ------------------------------------------------------------------------- */

/*               C o n f i g  T a r g e t  I s  G u i d e
 *
 *  Function:
 *     Called by method layer to determine if a given target type code 
 *     indicates that the target in question is a guide target. (It returns
 *     true if the target is a guide target type.)
 *
 */
 
static int  ConfigTargetIsGuide (
    const CONFIG_MethodInfo *MethodInfo, /* (>) house keeping details*/
    char TargetType)                     /* (>) Target type code */
{
    FpilType Instrument = MethodInfo->_Instrument;
    return FpilIsTargetGuide(Instrument,TargetType);
}

/* ------------------------------------------------------------------------- */

/*               C o n f i g  T a r g e t  I s  S k y
 *
 *  Function:
 *     Called by method layer to determine if a given target type code 
 *     indicates that the target in question is a sky target. (It returns
 *     true if the target is a sky target type.)
 *
 */
 
static int  ConfigTargetIsSky (
    const CONFIG_MethodInfo *MethodInfo, /* (>) house keeping details*/
    char TargetType)                     /* (>) Target type code */
{
    FpilType Instrument = MethodInfo->_Instrument;
    return FpilIsTargetSky(Instrument,TargetType);
}

/* ------------------------------------------------------------------------- */

/*               C o n f i g  F i b r e  T e x t
 *
 *  Function:
 *     Called by method layer to obtain the text associated with a fibre
 *     type code. It returns a pointer to a string that can be used to
 *     produce a message about a target of this type. 
 */
 
static char* ConfigFibreText (
    const CONFIG_MethodInfo *MethodInfo, /* (>) house keeping details*/
    int FibreType)                       /* (>) Fibre type code */
{
    char* FibreText;
    FpilType Instrument = MethodInfo->_Instrument;
    FibreText = FpilGetPivotText (Instrument,FibreType);
    
    return FibreText;
}

/* ------------------------------------------------------------------------- */

/*               C o n f i g  F i b r e  T y p e  M a t c h
 *
 *  Function:
 *     Called by method layer to see if two fibre type codes should be
 *     taken as matching for configuration purposes (for example, there
 *     are 3 UVES fibre types in FLAMES, but they should be treated as
 *     identical when configuring). 
 */
 
static int ConfigFibreTypeMatch (
    const CONFIG_MethodInfo *MethodInfo, /* (>) house keeping details*/
    int FibreType,                       /* (>) First fibre type code */
    int SecondType)                      /* (>) Second fibre type code */
{
    FpilType Instrument = MethodInfo->_Instrument;
    return FpilPivotTypeMatch (Instrument,FibreType,SecondType);  
}

/* ------------------------------------------------------------------------- */

/*               C o n f i g  F i b r e  S h a r e s  S k y
 *
 *  Function:
 *     Called by method layer to see if a fibre type code refers to a fibre
 *     type that can be allocated to both sky and non-sky targets. 
 */
 
static int ConfigFibreSharesSky (
    const CONFIG_MethodInfo *MethodInfo, /* (>) house keeping details*/
    int FibreType)                      /* (>) Second fibre type code */
{
    FpilType Instrument = MethodInfo->_Instrument;
    return FpilPivotSharesSky (Instrument,FibreType);  
}

/* ------------------------------------------------------------------------- */

/*          C o n f i g  F i b r e  T a r g e t  C o m p a t i b l e
 *
 *  Function:
 *     Called by method layer to determine if a given fibre can be allocated
 *     to a given target. It is passed the type and spect codes for the
 *     target (see the Fpil documentation for more details on these type
 *     codes, which are intended only for use within Fpil) and returns true
 *     if the allocation is valid, false if it is not.
 *
 */
 
static int  ConfigFibreTargetCompatible (
    const CONFIG_MethodInfo *MethodInfo, /* (>) house keeping details*/
    char TargetType,                     /* (>) target type code */
    char SpectType,                      /* (>) target spectrograph code */
    int FibreType)                       /* (>) Fibre type code */
{
    FpilType Instrument = MethodInfo->_Instrument;
    return FpilTargetPivotCompatible(Instrument,TargetType,SpectType,FibreType);
}

/* ------------------------------------------------------------------------- */

/*                      C o n f i g E r r R e p
 *
 *  Function:
 *     Called by the method layer to report errors
 *
 */

static void ConfigErrRep(
    void *FunctionArg DUNUSED,
    const char *String)
{
    StatusType status = STATUS__OK;
    ErsRep(0,&status,String);
}

/* ------------------------------------------------------------------------- */

/*               C o n f i g S e t F p i l M a x P i v A n g
 *
 * Interface to FpilSetMaxPivAng that converts its value from
 * degrees to radians.
 */
static void ConfigSetFpilMaxPivAng(FpilType Instrument, double value)
{
    FpilSetMaxPivAng(Instrument, (value * 360.0 / (2.0 * PI)));
}

/* ------------------------------------------------------------------------- */

/*              C o n f i g S e t F p i l M a x B u t A n g
 *
 * Interface to FpilSetMaxButAng that converts its value from
 * degrees to radians.
 */
static void ConfigSetFpilMaxFibAng(FpilType Instrument, double value)
{
    FpilSetMaxFibAng(Instrument, (value * 360.0 / (2.0 * PI)));
}

/* ------------------------------------------------------------------------- */

/*             C o n f i g  N u m b e r  P a r a m e t e r s
 *
 *  Function:
 *     Returns the number of configuration parameters.
 *
 *  Description:
 *     This routine returns the number of parameters that can be used
 *     to control the details of the allocation algorithm. This has to
 *     include those parameters supported by the non-algorithm-specific
 *     layer as well as those supported by the algorithm-specific layer.
 */

int ConfigNumberParameters (void) 
{
   return (CONFIG_STATIC_NPARMS + 
       (*CONFIG_Global.MethodDetails[CONFIG_Global.CurrentMethod].NumParams) 
                                                (&CONFIG_Global.MethodInfo));
}

/* -------------------------------------------------------------------------- */

/*         C o n f i g  P a r a m e t e r  D e t a i l s
 *
 *  Function:
 *     Returns details about the configuration parameters.
 *
 *  Description:
 *     This routine returns details about the various parameters
 *     that can be used to control the details of the allocation algorithm.
 *     This includes their names, types, and possible values.
 */

void ConfigParameterDetails (
   int ParameterNumber,
   char **Name,
   char *Type,
   double Limits[],
   int *NValues,
   char ***Values,
   StatusType *Status)
{
   /*  Local variables  */

   int Index;                         /* Index into parameter arrays */

   if (*Status != STATUS__OK) return;

   /*  We have to handle parameters supported by these routines as well
    *  as those supported by the algorithm-specific routines.
    */

   if (ParameterNumber < 1) {
      ErsRep (0,Status,"'%d' is not a valid parameter number",ParameterNumber); 
      *Status = CONFIG__INVAL_PARM_NO;
   } else if (ParameterNumber > CONFIG_STATIC_NPARMS) {

      (*CONFIG_Global.MethodDetails[CONFIG_Global.CurrentMethod].ParamDetails) 
            (&CONFIG_Global.MethodInfo,
             ParameterNumber - CONFIG_STATIC_NPARMS,
             Name, Type, Limits, NValues, Values, Status);
   } else {
      Index = ParameterNumber - 1;
      *Name = CONFIG_STATIC_ParmNames[Index];
      *Type = CONFIG_STATIC_ParmTypes[Index];
      Limits[0] = CONFIG_STATIC_ParmLimits[Index][0];
      Limits[1] = CONFIG_STATIC_ParmLimits[Index][1];
      *NValues = CONFIG_STATIC_ParmNValues[Index];
      *Values = CONFIG_STATIC_ParmValues[Index];
   }

}

/* -------------------------------------------------------------------------- */

/*              C o n f i g  S e t  P a r a m e t e r
 *
 *  Function:
 *     Sets the value of one of the configuration parameters.
 *
 *  Description:
 *     This routine sets the value of one of the various  
 *     parameters that can be used to control the details of the 
 *     allocation algorithm. The parameter is specified by name, and
 *     the value is specified as a character string.
 */

void ConfigSetParameter (
   char *Name,                        /* (>) Name of parameter */
   char *Value,                       /* (>) Value of parameter */
   StatusType *Status)                /* (!) Inherited status value */
{
   /*  Local variables  */

   int Bytes;                         /* Number of bytes for character param */
   double FloatValue;                 /* Value of floating point param */
   int Index;                         /* Index into parameter arrays */
   long IntegerValue;                 /* Value of integer parameter */
   double MaxLimit;                   /* Maximum value for parameter */
   double MinLimit;                   /* Minimum value for parameter */
   CONFIG_Boolean NameMatched;        /* True if Name is a valid name */
   int NValues;                       /* Number of values for char param */
   double Tolerance;                  /* Allows for floating point rounding */
   CONFIG_Boolean ParamOK;            /* True if char param is valid string */
   char **ParmValues;                 /* Address of possible char values */
   char *StringAddress;               /* Address of parameter value string */
   int StringIndex;                   /* Index through possible parm values */
   char Type;                         /* Parameter type 'I','F','C','L' */
   FpilType Instrument;               /* Instrument description pointer */

   if (*Status != STATUS__OK) return;
   /*
    *  Get easy access to the FPIL instrument description.
    */
   Instrument = CONFIG_Global.MethodInfo._Instrument;

   /*  Work through the names of the various parameters looking for
    *  a match with that specified. Note that this code is almost exactly
    *  the same as that used by the standard algorithm-specific code.
    */

   NameMatched = CONFIG_FALSE;
   for (Index = 0; Index < CONFIG_STATIC_NPARMS; Index++) {
      if (!strcmp(Name,CONFIG_STATIC_ParmNames[Index])) {

         /*  Name matches, so pick up the type, limits (for numrical values)
          *  and number of possible values (for character parameters).
          */

         NameMatched = CONFIG_TRUE;
         Type = CONFIG_STATIC_ParmTypes[Index];
         MinLimit = CONFIG_STATIC_ParmLimits[Index][0];
         MaxLimit = CONFIG_STATIC_ParmLimits[Index][1];
         NValues = CONFIG_STATIC_ParmNValues[Index];

         /*  Now handle the parameter according to type  */

         if (Type == 'L') {

            /*  Logical parameters must be either 'T' or 'F' */

            if (!strcmp(Value,"T")) {
               CONFIG_Global.ParameterValues[Index].IntegerValue = 
                                                                   CONFIG_TRUE;
            } else if (!strcmp(Value,"F")) {
               CONFIG_Global.ParameterValues[Index].IntegerValue = 
                                                                  CONFIG_FALSE;
            } else {
               ErsRep (0,Status,
                   "'%s' is a logical parameter and cannot be set to '%s'",
                                                                   Name,Value);
               *Status = CONFIG__INVAL_PARM_VALUE;
            }

         } else if (Type == 'I') {

            /*  Integer parameters must be valid integer values within
             *  the specified range.
             */

            if (sscanf(Value,"%ld",&IntegerValue) == 1) {
               if ((IntegerValue < (int) MinLimit) ||
                    (IntegerValue > (int) MaxLimit)) {
                  ErsRep (0,Status,
                    "Parameter '%s' must be between %d and %d. %ld is invalid.",
                            Name,(int) MinLimit, (int) MaxLimit, IntegerValue);
                  *Status = CONFIG__PARM_OUT_RANGE;
               } else {
                  CONFIG_Global.ParameterValues[Index].IntegerValue = 
                                                               IntegerValue;
                  /* If this parameter has a corresonding FPIL item, a routine
                   * will be specified to set the value within FPIL, so
                   * invoke it
                   */
                  if (CONFIG_STATIC_SetFpilInt[Index]) {
                      (*CONFIG_STATIC_SetFpilInt[Index])(Instrument, 
                                                         IntegerValue);
                  }
               }
            } else {
               ErsRep (0,Status,
                  "'%s' is not a valid value for integer parameter '%s'",
                                                                 Value,Name);
               *Status = CONFIG__INVAL_PARM_VALUE;
            }

         } else if (Type == 'F') {

            /*  Floating point parameters must be valid real values within
             *  the specified range. We allow a little tolerance for rounding
             *  errors.
             */

            if (sscanf(Value,"%lf",&FloatValue) == 1) {
               Tolerance = MaxLimit * 0.005;
               if ((FloatValue < (MinLimit - Tolerance)) 
                                    || (FloatValue > (MaxLimit + Tolerance))) {
                  ErsRep (0,Status,
                    "Parameter '%s' must be between %f and %f. %f is invalid.",
                                            Name,MinLimit,MaxLimit,FloatValue);
                  *Status = CONFIG__PARM_OUT_RANGE;
               } else {
                  CONFIG_Global.ParameterValues[Index].DoubleValue = 
                                                                 FloatValue;

                  /* If this parameter has a corresonding FPIL item, a routine
                   * will be specified to set the value within FPIL, so
                   * invoke it
                   */
                  if (CONFIG_STATIC_SetFpilInt[Index]) {
                      (*CONFIG_STATIC_SetFpilReal[Index])(Instrument, 
                                                         FloatValue);
                  }

               }
            } else {
               ErsRep (0,Status,
                  "'%s' is not a valid value for parameter '%s'",Value,Name);
               *Status = CONFIG__INVAL_PARM_VALUE;
            }

         } else if (Type == 'C') {

            /*  Character values can have either any value, or one of a 
             *  specified set. 
             */

            if (NValues <= 0) {

               /*  This paramter can have any value, so it must be valid.
                *  We allocate memory to hold a copy of the new value (this
                *  allows the value to be any length at all), first freeing
                *  any previously allocated.
                */

               if (CONFIG_Global.ParameterValues[Index].StringAddress != NULL) {
                  free (CONFIG_Global.ParameterValues[Index].StringAddress);
               }
               Bytes = strlen(Value);
               if (Bytes == 0) {
                  CONFIG_Global.ParameterValues[Index].StringAddress = NULL;
               } else {
                  StringAddress = (char *) malloc(Bytes + 1);
                  strcpy (StringAddress,Value);
                  CONFIG_Global.ParameterValues[Index].StringAddress
                                                             = StringAddress;
               }
            } else {

               /*  This character string must have one of a set of specified
                *  values. Check that this is one, and set the index in
                *  the global variables.
                */

               ParmValues = CONFIG_STATIC_ParmValues[Index];
               ParamOK = CONFIG_FALSE;
               for (StringIndex = 0; StringIndex < NValues; StringIndex++) {
                  if (!strcmp(Value,ParmValues[StringIndex])) {
                     CONFIG_Global.ParameterValues[Index].StringAddress
                                                    = ParmValues[StringIndex];
                     CONFIG_Global.ParameterValues[Index].IntegerValue
                                                                = StringIndex;
                     ParamOK = CONFIG_TRUE;
                     break;
                  }
               }
               if (!ParamOK) {
                  ErsRep (0,Status,
                     "'%s' is not one of the possible values for '%s'",
                                                              Value,Name);
                  *Status = CONFIG__INVAL_PARM_VALUE;
               }
            }
         }

         /*  Finished processing a matched name, so break out of the loop
          *  through the various parameter names.
          */

         break;
      }
   }

   /*  Now, if we didn't match the name here, we need to see if it is one
    *  of the algorithm-specific parameters.
    */

   if (!NameMatched) {
       (*CONFIG_Global.MethodDetails[CONFIG_Global.CurrentMethod].SetParam) 
            (&CONFIG_Global.MethodInfo,
             Name, Value, Status);
   }
   
}
                  
/* -------------------------------------------------------------------------- */

/*              C o n f i g  G e t  P a r a m e t e r
 *
 *  Function:
 *     Gets the value of one of the configuration parameters.
 *
 *  Description:
 *     This routine gets the value of one of the various  
 *     parameters that can be used to control the details of the 
 *     allocation algorithm. The parameter is specified by name, and
 *     the value is returned as a character string.
 */

void ConfigGetParameter (
   char *Name,                        /* (>) Name of parameter */
   char *Value,                       /* (<) Value of parameter */
   int VSize,                         /* (>) Maximum number of characters */
   StatusType *Status)                /* (!) Inherited status value */
{
   /*  Local variables  */

   int Index;                         /* Index into parameter arrays */
   CONFIG_Boolean NameMatched;        /* True if Name is a valid name */
   char Type;                         /* Parameter type 'I','F','C','L' */
   char *ValuePtr;                    /* Points to string to be returned */
   char ValueString[64];              /* Used to format numbers and logicals */

   if (*Status != STATUS__OK) return;

   /*  Work through the names of the various parameters looking for
    *  a match with that specified. This code is almost identical to that
    *  used by the standard algorithm-specific code.
    */

   NameMatched = CONFIG_FALSE;
   ValuePtr = NULL;
   for (Index = 0; Index < CONFIG_STATIC_NPARMS; Index++) {
      if (!strcmp(Name,CONFIG_STATIC_ParmNames[Index])) {

         /*  Name matches, so pick up its type.
          */

         NameMatched = CONFIG_TRUE;
         Type = CONFIG_STATIC_ParmTypes[Index];

         /*  Now handle the parameter according to type  */

         if (Type == 'L') {

            /*  Logical parameters are returned as either 'T' or 'F' */

            if (CONFIG_Global.ParameterValues[Index].IntegerValue) {
               ValuePtr = "T";
            } else {
               ValuePtr = "F";
            }

         } else if (Type == 'I') {

            /*  Integer parameters are formatted into a value string.
             */

            sprintf(ValueString,"%d",
                CONFIG_Global.ParameterValues[Index].IntegerValue);
            ValuePtr = ValueString;

         } else if (Type == 'F') {

            /*  Floating point parameters are also formatted into a value
             *  string.
             */

            sprintf(ValueString,"%f",
                CONFIG_Global.ParameterValues[Index].DoubleValue);
            ValuePtr = ValueString;

         } else if (Type == 'C') {

            /*  Character values can have either any value, or one of a 
             *  specified set. In either case, the address of the current
             *  value is held in the global variables.
             */

            ValuePtr = 
                 CONFIG_Global.ParameterValues[Index].StringAddress;
         }

         /*  Finished processing a matched name, so break out of the loop
          *  through the various parameter names.
          */

         break;
      }
   }

   /*  After working through the various accepted parameter names, we either
    *  we have a pointer (ValuePtr) to a string giving the current value,
    *  or we didn't find the name specified, in which case we see if it is one
    *  of the algorithm-specific parameters.
    */

   if (NameMatched) {
      if (ValuePtr == NULL) {
         Value[0] = '\0';
      } else {
         strncpy (Value,ValuePtr,VSize);
         Value[VSize - 1] = '\0';
      }
   } else {
       (*CONFIG_Global.MethodDetails[CONFIG_Global.CurrentMethod].GetParam) 
            (&CONFIG_Global.MethodInfo,
             Name, Value, VSize, Status);
   }
}
 
/* -------------------------------------------------------------------------- */
void ConfigSetImportFile(
  char name[80])
{
    (*CONFIG_Global.MethodDetails[CONFIG_Global.CurrentMethod].SetImportFile) 
            (&CONFIG_Global.MethodInfo,
             name);
}
/* -------------------------------------------------------------------------- */

/*              C o n f i g  R e g i s t e r M e t h o d
 *  Function:
 *    Register a new configuration method.
 *
 *  Description:
 *     Registers a new configuration and optionally enables it.
 */

void ConfigRegisterMethod(
    const CONFIG_MethodDetails * const Method,/* (>) Details of new method */
    int Enable,                  /* (>) Is method to be enabled */
    unsigned * const Index,      /* (<) Method Index          */
    StatusType * const status)   /* (!) Inherited status variable */
{
    if (*status != STATUS__OK) return;
    if (CONFIG_Global.NumMethods >= CONFIG_NUM_METHODS)
        *status = CONFIG__MAXMTHS;
    else if (Method->Version != CONFIG_VERSION)
        *status = CONFIG__MTHVER;
    else
    {
/*
 *      Save the current method and get the index for the new method.
 */
        unsigned currentMethod = CONFIG_Global.CurrentMethod;
        *Index = CONFIG_Global.NumMethods++;
/*
 *      Save the method details.
 */
        CONFIG_Global.MethodDetails[*Index] = *Method;
/*
 *      Select the new method as the current method and then invoke the
 *      global init routine.
 */
        ConfigSelectMethod(*Index, status);
        (*CONFIG_Global.MethodDetails[*Index].GlobalInit) 
            (&CONFIG_Global.MethodInfo);
/*
 *      If the new method is not to be enabled by default, reselect
 *      the old method.
 */
        if (!Enable)
            ConfigSelectMethod(currentMethod, status);
    }
    
}

/* -------------------------------------------------------------------------- */

/*              C o n f i g  S e l e c t M e t h o d
 *  Function:
 *     Select a method from one of the set of registered methods.
 *
 *  Description:
 *     Selects the method.
 */

void ConfigSelectMethod(
    unsigned  Index,             /* (>) Method Index          */
    StatusType * const status)   /* (!) Inherited status variable */
{
    if (*status != STATUS__OK) return;
    if (Index >= CONFIG_Global.NumMethods)
        *status = CONFIG__INVMTH;
    else
    {
/*
 *      Save the index and copy the client data to the MethodInfo structure
 *      so that it is available when desired.
 */
        CONFIG_Global.CurrentMethod = Index;
        CONFIG_Global.MethodInfo.clientData = 
            CONFIG_Global.MethodDetails[Index].clientData;
        
    }
    
}

/* -------------------------------------------------------------------------- */

/*             C o n f i g  N u m M e t h o d s
 *  Function:
 *     Returns the number of methods available.
 *
 *  Description:
 *     Just returns the value of NumMethods.
 *
 */
extern void ConfigNumMethods(
    unsigned *Number,            /* (>) Number of methods available */
    StatusType *status)         /* (!) Inherited status variable */
{
    if (*status != STATUS__OK) return;
    *Number = CONFIG_Global.NumMethods;
}

/* -------------------------------------------------------------------------- */

/*             C o n f i g  G e t M e t h o d  N a m e
 *  Function:
 *     Returns the name of a method
 *
 *  Description:
 *
 */
extern void ConfigGetMethodName(
    unsigned Index,              /* (>) Index of method to select */
    unsigned Namelen,            /* (>) Length of name            */
    char *Name,                  /* (<) Name is written here      */
    StatusType *status)          /* (!) Inherited status variable */
{
    if (*status != STATUS__OK) return;
    if (Index >= CONFIG_Global.NumMethods)
        *status = CONFIG__INVMTH;
    else
    {
        strncpy(Name,CONFIG_Global.MethodDetails[Index].Name,Namelen);
        Name[Namelen-1] = '\0';
    }
}

/* -------------------------------------------------------------------------- */

/*             C o n f i g  S e t P i v o t F l a g
 *  Function:
 *     Sets the flag which indicates if a pivot is allowed to have its
 *     allocation change.
 *
 *  Description:
 *
 */
extern void ConfigSetPivotFlag(
    int Pivot,                   /* (>) Pivot to change the flag */
    int flag,                    /* (>) New value for the flag   */
    StatusType *Status)          /* (!) Inherited status variable */
{
    CONFIG_PivotInfo  *PivotInfo; /* Address of pivot info array */
    int Field;                    /* Field being configured */
    int NPiv;                     /* Number of pivots */

    if (*Status != STATUS__OK) return;

    /*
     * Access info.
     */
    Field = CONFIG_Global.Field;
    PivotInfo = CONFIG_Global.PivotInfoPtr[Field];
    NPiv = CONFIG_Global.NumberPivots[Field];

    /*
     * Validate pivot number.
     */
    if ((Pivot > NPiv) || (Pivot <= 0)) {
        ErsRep (0,Status,
                "Attempt to set pivot flag for non-existent pivot, # %d",
                Pivot );
        *Status = CONFIG__INVAL_PIVOT;
        return;
   }

    /*
     * Set the flag.
     */
    PivotInfo[Pivot-1].AllowAllocChange = flag ? CONFIG_TRUE : CONFIG_FALSE;
}

/* -------------------------------------------------------------------------- */

/*             C o n f i g  S e t  S p e c t  P i v o t s  F l a g
 *
 *  Description:
 *     Sets the flags that indicate the pivots assocated with a
 *     particular fibre type are allowed to have their 
 *     allocation changed.  This overrides any affect caused
 *     by a previous call to ConfigSetPivotFlag().
 *
 *     A fibre type code number of -1 indicates all fibre types.
 */
 
extern void ConfigSetSpecPivotsFlag(
    int FibreType,               /* (>) Fibre type in question */
    int Flag,                    /* (>) New value for the flag   */
    StatusType *Status)          /* (!) Inherited status variable */
{
    CONFIG_PivotInfo  *PivotInfo; /* Address of pivot info array */
    int Field;                    /* Field being configured */
    FpilType Instrument;          /* Instrument description pointer */
    unsigned int NumFibreTypes;   /* Will receive number of fibre types */
    const char * const * FibreTypeNames;
                                  /* Will receive names of fibre types */
    int NPiv;                     /* Number of pivots */
    int FlagVal;                  /* Flag value to be used - true or false */
    register int i;               /* Index through pivots */


    if (*Status != STATUS__OK) return;
    
    /*
     *  Get easy access to the FPIL instrument description.
     */
    Instrument = CONFIG_Global.MethodInfo._Instrument;

    /*
     *  Access details
     */
     
    Field = CONFIG_Global.Field;
    PivotInfo = CONFIG_Global.PivotInfoPtr[Field];
    NPiv = CONFIG_Global.NumberPivots[Field];

    /*
     *  Get the value to set the Flag to. We could just use flag, but this
     *  makes sure we only use the usual true or false values.
     */
     
    FlagVal = Flag ? CONFIG_TRUE : CONFIG_FALSE;

    if (FibreType == -1) {
    
        /*
         * We want to set the flag for all fibre types - ie all pivots.  
         */
         
        for (i = 0; i < NPiv ; ++i) {
            PivotInfo[i].AllowAllocChange = FlagVal;
        }
        
    } else {

        /*
         *  One fibre type only to be changed.
         */
         
        FpilConSpecInfo(Instrument, &NumFibreTypes, &FibreTypeNames, Status);

        /*
         *  Validate fibre type number.
         */
         
        if ((FibreType >= (int)NumFibreTypes) || (FibreType < 0)) {
            ErsRep (0,Status,
             "Attempt to set pivot flags for invalid fibre type, code =  # %d",
                    FibreType );
            *Status = CONFIG__INVAL_SPEC;
            return;
        }
        
        /*
         * Set the flags.
         */
         
        for (i = 0; i < NPiv ; ++i) {
            if (PivotInfo[i].FibreType == FibreType)
                PivotInfo[i].AllowAllocChange = FlagVal;
        }
    }
}

/* -------------------------------------------------------------------------- */

/*      C o n f i g S e t  F i b r e P o s
 *
 * Does a temporarly change of a fibre position.  This is used to
 * allow collision checks etc. to be run with a temp fibre position.
 * 
 * AllocInfo[Pivot].X is set to X.
 * AllocInfo[Pivot].Y is set to Y.
 * AllocInfo[Pivot].Theta is set to Theta.
 * AllocInfo[Pivot].State is set to 'A'  or 'U' depending on  Allocated.
 * AllocInfo[Pivot].Target is set to Target if allocated, 0 otherwise.
 *
 * You MUST call this routine again to restore the values.
 */
static void ConfigSetFibrePos (
    const CONFIG_MethodInfo *MethodInfo DUNUSED,/* (>) house keeping details*/
    int Pivot,                          /* (>) Pivot to change      */
    int X,                              /* (>) New X position       */
    int Y,                              /* (>) New Y position       */
    double Theta,                       /* (>) New Theta position   */
    int Target,                         /* (>) New target index     */
    CONFIG_Boolean Allocated)           /* (>) Is it to be allocated */
{
    CONFIG_AllocInfo *AllocInfo;  /* Allocation info array of structures */
    int Field;                    /* Field being configured */
    Field = CONFIG_Global.Field;
    AllocInfo = CONFIG_Global.AllocInfoPtr[Field];

    AllocInfo[Pivot].X = X;
    AllocInfo[Pivot].Y = Y;
    AllocInfo[Pivot].Theta = Theta;
    if (Allocated) {
        AllocInfo[Pivot].TargetIndex = Target;
        AllocInfo[Pivot].State = 'A';
    } else {
        AllocInfo[Pivot].TargetIndex = 0;
        AllocInfo[Pivot].State = 'U';
    }
}
