/**
 * @file src/sequence_diagram/generators/mermaid/sequence_diagram_generator.h
 *
 * Copyright (c) 2021-2025 Bartek Kryza <bkryza@gmail.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#pragma once

#include "common/generators/mermaid/generator.h"
#include "config/config.h"
#include "sequence_diagram/model/diagram.h"
#include "sequence_diagram/visitor/translation_unit_visitor.h"
#include "util/util.h"

#include <glob/glob.hpp>

#include <filesystem>
#include <fstream>
#include <iostream>
#include <sstream>

namespace clanguml {
namespace sequence_diagram {
namespace generators {
namespace mermaid {

using clanguml::common::eid_t;

using diagram_config = clanguml::config::sequence_diagram;
using diagram_model = clanguml::sequence_diagram::model::diagram;

template <typename C, typename D>
using common_generator = clanguml::common::generators::mermaid::generator<C, D>;

/**
 * @brief Sequence diagram MermaidJS generator
 */
class generator : public common_generator<diagram_config, diagram_model> {
public:
    generator(diagram_config &config, diagram_model &model);

    using common_generator<diagram_config, diagram_model>::generate;

    /**
     * @brief Main generator method.
     *
     * This method is called first and coordinates the entire diagram
     * generation.
     *
     * @param ostr Output stream.
     */
    void generate_diagram(std::ostream &ostr) const override;

    /**
     * @brief Generate the diagram type
     *
     * @param ostr Output stream
     */
    void generate_diagram_type(std::ostream &ostr) const override;

    /**
     * @brief Generate sequence diagram message.
     *
     * @param m Message model
     * @param ostr Output stream
     */
    void generate_call(const clanguml::sequence_diagram::model::message &m,
        std::ostream &ostr) const;

    /**
     * @brief Generate sequence diagram return message
     *
     * @param m Message model
     * @param ostr Output stream
     */
    void generate_return(const clanguml::sequence_diagram::model::message &m,
        std::ostream &ostr) const;

    /**
     * @brief Generate sequence diagram participant
     *
     * @param ostr Output stream
     * @param id Participant id
     * @param force If true, generate the participant even if its not in
     *              the set of active participants
     * @return Id of the generated participant
     */
    void generate_participant(
        std::ostream &ostr, eid_t id, bool force = false) const;

    /**
     * @brief Generate sequence diagram participant by name
     *
     * This is convenience wrapper over `generate_participant()` by id.
     *
     * @param ostr Output stream
     * @param name Full participant name
     */
    void generate_participant(
        std::ostream &ostr, const std::string &name) const;

    /**
     * @brief Generate sequence diagram activity.
     *
     * @param activity_id Activity id
     * @param ostr Output stream
     * @param visited List of already visited participants, this is necessary
     *                for breaking infinite recursion on recursive calls
     */
    void generate_activity(eid_t activity_id, std::ostream &ostr,
        std::vector<eid_t> &visited) const;

private:
    /**
     * @brief Check if specified participant has already been generated.
     *
     * @param id Participant id.
     * @return True, if participant has already been generated.
     */
    bool is_participant_generated(eid_t id) const;

    /**
     * @brief Generate MermaidJS alias for participant
     *
     * @param participant Sequence diagram participant model
     * @return Particpant alias
     */
    std::string generate_alias(const model::participant &participant) const;

    /**
     * @brief Generate message call note
     *
     * @param ostr Output stream
     * @param m Message
     */
    void generate_message_comment(
        std::ostream &ostr, const model::message &m) const;

    std::string render_message_name(const std::string &m) const;

    /**
     * @brief Convert config to model message render mode.
     *
     * @return Method render mode.
     */
    model::function::message_render_mode
    select_method_arguments_render_mode() const;

    void generate_from_to_sequences(
        std::ostream &ostr, bool star_participant_generated) const;

    void generate_to_sequences(std::ostream &ostr) const;

    void generate_from_sequences(std::ostream &ostr) const;

    std::vector<eid_t> find_from_activities() const;

    std::vector<model::message_chain_t> find_to_message_chains() const;

    mutable std::set<eid_t> generated_participants_;
    mutable std::set<unsigned int> generated_comment_ids_;
    mutable std::vector<model::message> already_generated_in_static_context_;
    mutable std::set<eid_t> generated_activities_;
};

} // namespace mermaid
} // namespace generators
} // namespace sequence_diagram
} // namespace clanguml
