/***************************************************************************************************************************************************************
 *
 * cfonts
 *
 * Sexy fonts for the console. (CLI output)
 *
 * @license     https://github.com/dominikwilkowski/cfonts/blob/released/LICENSE  GNU GPL-3.0-or-later
 * @author      Dominik Wilkowski  hi@dominik-wilkowski.com
 * @repository  https://github.com/dominikwilkowski/cfonts
 *
 * Cli
 *   Run cli commands
 *
 **************************************************************************************************************************************************************/

'use strict';

const { DisplayVersion } = require('./DisplayVersion.js');
const { DisplayHelp } = require('./DisplayHelp.js');
const { CLIOPTIONS } = require('./constants.js');
const { Debugging } = require('./Debugging.js');
const { ParseArgs } = require('./ParseArgs.js');
const { Render } = require('./Render.js');
const { Color } = require('./Color.js');
const { Log } = require('./Log.js');
const { Say } = require('./Say.js');

/**
 * Run cli commands
 *
 * @param  {object} inputOptions - All possible options registered for this app
 * @param  {array}  inputArgs    - The arguments given to us in our cli, default: process.argv
 */
const Cli = (inputOptions = CLIOPTIONS, inputArgs = process.argv) => {
	const args = ParseArgs(inputOptions, inputArgs);

	Debugging.report(
		`OPTIONS:\n` +
			`  CFonts.say("${args.text}", {\n` +
			`    font: "${args.font}",\n` +
			`    align: "${args.align}",\n` +
			`    colors: ${args.colors ? JSON.stringify(args.colors.split(',')) : []},\n` +
			`    background: "${args.background}",\n` +
			`    letterSpacing: ${args['letter-spacing']},\n` +
			`    lineHeight: ${args['line-height']},\n` +
			`    spaceless: ${args.spaceless},\n` +
			`    maxLength: ${args['max-length']},\n` +
			`    gradient: ${args.gradient},\n` +
			`    independentGradient: ${args['independent-gradient']},\n` +
			`    transitionGradient: ${args['transition-gradient']},\n` +
			`    rawMode: ${args['raw-mode']},\n` +
			`    env: ${args.env},\n` +
			`  }, ${args.debug}, ${args['debug-level']} );`,
		3,
		args.debug,
		args['debug-level']
	);

	if (args.help) {
		DisplayHelp();
		return;
	}

	if (args.version) {
		DisplayVersion();
		return;
	}

	if (!args.text) {
		const { open: green_open, close: green_close } = Color('green');
		Log.error(
			`Please provide text to convert with ${green_open}cfonts "Text"${green_close}\n` +
				`Run ${green_open}cfonts --help${green_close} for more infos`
		);
		return;
	}

	Say(
		args.text,
		{
			font: args.font,
			align: args.align,
			colors: args.colors ? args.colors.split(',') : [],
			background: args.background,
			letterSpacing: args['letter-spacing'],
			lineHeight: args['line-height'],
			spaceless: args.spaceless,
			maxLength: args['max-length'],
			gradient: args.gradient,
			independentGradient: args['independent-gradient'],
			transitionGradient: args['transition-gradient'],
			rawMode: args['raw-mode'],
			env: args.env,
		},
		args.debug,
		args['debug-level']
	);
};

module.exports = exports = {
	render: Render,
	say: Say,
	Cli,
};
