% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/roll_values.R
\name{roll_values}
\alias{roll_values}
\alias{wrap_to_range}
\alias{roll_values_vec}
\alias{wrap_to_range_vec}
\title{Shift values and wrap to range}
\usage{
roll_values(
  data,
  cols = NULL,
  add = 0,
  .min = NULL,
  .max = NULL,
  between = 0,
  na.rm = FALSE,
  suffix = "_rolled",
  keep_original = TRUE,
  range_col_name = ".range",
  overwrite = FALSE
)

wrap_to_range(
  data,
  cols = NULL,
  .min = NULL,
  .max = NULL,
  between = 0,
  na.rm = FALSE,
  suffix = "_wrapped",
  keep_original = TRUE,
  range_col_name = ".range",
  overwrite = FALSE
)

roll_values_vec(
  data,
  add = 0,
  .min = NULL,
  .max = NULL,
  between = 0,
  na.rm = FALSE
)

wrap_to_range_vec(data, .min = NULL, .max = NULL, between = 0, na.rm = FALSE)
}
\arguments{
\item{data}{\code{vector} or \code{data.frame} to roll/wrap values of. When a \code{data.frame} is
grouped, the rolling/wrapping is applied group-wise.}

\item{cols}{Names of columns to roll/wrap in \code{`data`}. Must be specified
when \code{`data`} is a \code{data.frame}.}

\item{add}{Amount to add to each element. (\code{numeric scalar})

When \code{0}, the wrapping is applied without any rolling.}

\item{.min}{Minimum value allowed. If \code{NULL}, the
minimum value in the \code{vector}/\code{column} is used.}

\item{.max}{Maximum value allowed. If \code{NULL}, the
maximum value in the \code{vector}/\code{column} is used.}

\item{between}{The wrapping distance between \code{`.max`} and \code{`.min`}.

When \code{0}, they are considered the same. I.e. \eqn{`.max == .min`}.

When \code{1}, \code{`x`} can be greater than \code{`.max`} by up to \code{1}, why
\code{`.min`} and \code{`.max`} are two separate values with \code{1} in-between them. I.e.
\eqn{`.max + 1 == .min`}.}

\item{na.rm}{Whether to remove missing values (\code{NA}s)
when finding the \code{`.min`} and \code{`.max`} values.}

\item{suffix}{Suffix to add to the names of the generated columns.

Use an empty string (i.e. \code{""}) to overwrite the original columns.}

\item{keep_original}{Whether to keep the original columns. (Logical)

Some columns may have been overwritten, in which case only the newest versions are returned.}

\item{range_col_name}{Name of new column with the min-max range. If \code{NULL}, no column is added.

\strong{N.B.} Ignored when \code{`data`} is a \code{vector}.}

\item{overwrite}{Whether to allow overwriting of existing columns. (Logical)}
}
\value{
\code{`data`} with new columns with values in the specified min-max range(s)
and columns with the applied ranges.

The \code{*_vec()} versions return a \code{vector}.
}
\description{
\Sexpr[results=rd, stage=render]{lifecycle::badge("experimental")}

Adds a specified value to each element in the vector and wraps the values around
the min-max range with:

\eqn{(x - .min)}\code{ \% }\eqn{(.max - .min + between) + .min}

Useful when adding to the degrees of a circle, where the values should remain in the
\code{0-360} range. A value larger than \code{360} will start over from \code{0}, e.g. \eqn{365 -> 5},
while a value smaller than \code{0} would become e.g. \eqn{-5 -> 355}.
Here, \code{0} and \code{360} are considered the same angle.
If we were instead adding days to the weekdays \code{1-7},
where \code{1} and \code{7} are separate days,
we can set \code{`between = 1`} to have one day in-between them.

\code{wrap_to_range()} is a wrapper with \code{`add = 0`}.

The \code{*_vec()} versions take and return a vector.

Should not be confused with \code{\link[rearrr:roll_elements]{roll_elements()}},
which changes the \emph{positions} of the elements.
}
\examples{
# Attach packages
library(rearrr)

# Add 90 to all degrees
# Note that 0 and 360 is considered the same angle
# why there is no distance between the two
roll_values(c(0:360), add = 90)

# Get as vector
roll_values_vec(c(0:360), add = 90)

# Change limits to 0-180
# so e.g. 270 becomes 90
roll_values(c(0:360), .min = 0, .max = 180)

# Change values first, then wrap to range
x <- c(1:7)
x <- x^2
wrap_to_range(x, .min = 1, .max = 7)

# With 1 in-between .min and .max
wrap_to_range(x, .min = 1, .max = 7, between = 1)

# Get as vector
wrap_to_range_vec(x, .min = 1, .max = 7, between = 1)

#
# Roll data.frame
#

# Set seed
set.seed(1)

# Create a data frame
df <- data.frame(
  "w" = 1:7,
  "d" = c(0, 45, 90, 135, 180, 270, 360)
)

# Roll weekdays by 1 day
roll_values(
  df,
  cols = "w",
  add = 1,
  .min = 1,
  .max = 7,
  between = 1
)

# Roll degrees by -90 degrees
roll_values(
  df,
  cols = "d",
  add = -90,
  .min = 0,
  .max = 360,
  between = 0
)

# Roll both weekdays and degrees by 1
# We don't specify .min and .max, so they
# are based on the values in the columns
# Note: This is not that meaningful but shows
# the option
roll_values(
  df,
  cols = c("w", "d"),
  add = 1
)

# Wrap weekdays to 2-5 range
wrap_to_range(
  df,
  cols = "w",
  .min = 2,
  .max = 5,
  between = 1
)
}
\seealso{
Other roll functions: 
\code{\link{roll_elements}()}

Other mutate functions: 
\code{\link{apply_transformation_matrix}()},
\code{\link{cluster_groups}()},
\code{\link{dim_values}()},
\code{\link{expand_distances_each}()},
\code{\link{expand_distances}()},
\code{\link{flip_values}()},
\code{\link{rotate_2d}()},
\code{\link{rotate_3d}()},
\code{\link{shear_2d}()},
\code{\link{shear_3d}()},
\code{\link{swirl_2d}()},
\code{\link{swirl_3d}()}
}
\author{
Ludvig Renbo Olsen, \email{r-pkgs@ludvigolsen.dk}
}
\concept{mutate functions}
\concept{roll functions}
