% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/formula-gp.R
\name{gp}
\alias{gp}
\title{Set up Gaussian process terms in \pkg{brms}}
\usage{
gp(
  ...,
  by = NA,
  k = NA,
  cov = "exp_quad",
  iso = TRUE,
  gr = TRUE,
  cmc = TRUE,
  scale = TRUE,
  c = 5/4
)
}
\arguments{
\item{...}{One or more predictors for the GP.}

\item{by}{A numeric or factor variable of the same length as
each predictor. In the numeric vector case, the elements multiply
the values returned by the GP. In the factor variable
case, a separate GP is fitted for each factor level.}

\item{k}{Optional number of basis functions for computing Hilbert-space
approximate GPs. If \code{NA} (the default), exact GPs are computed.}

\item{cov}{Name of the covariance kernel. Currently supported are
\code{"exp_quad"} (exponentiated-quadratic kernel; default),
\code{"matern32"} (Matern 3/2 kernel), \code{"matern52"} (Matern 5/2 kernel),
and \code{"exponential"} (exponential kernel; alias: \code{"matern12"}).}

\item{iso}{A flag to indicate whether an isotropic (\code{TRUE}; the
default) or a non-isotropic GP should be used.
In the former case, the same amount of smoothing is applied to all
predictors. In the latter case, predictors may have different smoothing.
Ignored if only a single predictor is supplied.}

\item{gr}{Logical; Indicates if auto-grouping should be used (defaults
to \code{TRUE}). If enabled, observations sharing the same
predictor values will be represented by the same latent variable
in the GP. This will improve sampling efficiency
drastically if the number of unique predictor combinations is small
relative to the number of observations.}

\item{cmc}{Logical; Only relevant if \code{by} is a factor. If \code{TRUE}
(the default), cell-mean coding is used for the \code{by}-factor, that is
one GP per level is estimated. If \code{FALSE}, contrast GPs are estimated
according to the contrasts set for the \code{by}-factor.}

\item{scale}{Logical; If \code{TRUE} (the default), predictors are
scaled so that the maximum Euclidean distance between two points
is 1. This often improves sampling speed and convergence.
Scaling also affects the estimated length-scale parameters
in that they resemble those of scaled predictors (not of the original
predictors) if \code{scale} is \code{TRUE}.}

\item{c}{Numeric value only used in approximate GPs. Defines the
multiplicative constant of the predictors' range over which
predictions should be computed. A good default could be \code{c = 5/4}
but we are still working on providing better recommendations.}
}
\value{
An object of class \code{'gp_term'}, which is a list
  of arguments to be interpreted by the formula
  parsing functions of \pkg{brms}.
}
\description{
Set up a Gaussian process (GP) term in \pkg{brms}. The function does not
evaluate its arguments -- it exists purely to help set up a model with
GP terms.
}
\details{
A GP is a stochastic process, which
 describes the relation between one or more predictors
 \eqn{x = (x_1, ..., x_d)} and a response \eqn{f(x)}, where
 \eqn{d} is the number of predictors. A GP is the
 generalization of the multivariate normal distribution
 to an infinite number of dimensions. Thus, it can be
 interpreted as a prior over functions. The values of \eqn{f( )}
 at any finite set of locations are jointly multivariate
 normal, with a covariance matrix defined by the covariance
 kernel \eqn{k_p(x_i, x_j)}, where \eqn{p} is the vector of parameters
 of the GP:
 \deqn{(f(x_1), \ldots f(x_n) \sim MVN(0, (k_p(x_i, x_j))_{i,j=1}^n) .}
 The smoothness and general behavior of the function \eqn{f}
 depends only on the choice of covariance kernel.
 For a more detailed introduction to Gaussian processes,
 see \url{https://en.wikipedia.org/wiki/Gaussian_process}.

 For mathematical details on the supported kernels, please see the Stan manual:
 \url{https://mc-stan.org/docs/functions-reference/matrix_operations.html}
 under "Gaussian Process Covariance Functions".
}
\examples{
\dontrun{
# simulate data using the mgcv package
dat <- mgcv::gamSim(1, n = 30, scale = 2)

# fit a simple GP model
fit1 <- brm(y ~ gp(x2), dat, chains = 2)
summary(fit1)
me1 <- conditional_effects(fit1, ndraws = 200, spaghetti = TRUE)
plot(me1, ask = FALSE, points = TRUE)

# fit a more complicated GP model and use an approximate GP for x2
fit2 <- brm(y ~ gp(x0) + x1 + gp(x2, k = 10) + x3, dat, chains = 2)
summary(fit2)
me2 <- conditional_effects(fit2, ndraws = 200, spaghetti = TRUE)
plot(me2, ask = FALSE, points = TRUE)

# fit a multivariate GP model with Matern 3/2 kernel
fit3 <- brm(y ~ gp(x1, x2, cov = "matern32"), dat, chains = 2)
summary(fit3)
me3 <- conditional_effects(fit3, ndraws = 200, spaghetti = TRUE)
plot(me3, ask = FALSE, points = TRUE)

# compare model fit
loo(fit1, fit2, fit3)

# simulate data with a factor covariate
dat2 <- mgcv::gamSim(4, n = 90, scale = 2)

# fit separate gaussian processes for different levels of 'fac'
fit4 <- brm(y ~ gp(x2, by = fac), dat2, chains = 2)
summary(fit4)
plot(conditional_effects(fit4), points = TRUE)
}

}
\seealso{
\code{\link{brmsformula}}
}
