/*#############################################################################
#                                                                             #
# Pakfire - The IPFire package management system                              #
# Copyright (C) 2023 Pakfire development team                                 #
#                                                                             #
# This program is free software: you can redistribute it and/or modify        #
# it under the terms of the GNU General Public License as published by        #
# the Free Software Foundation, either version 3 of the License, or           #
# (at your option) any later version.                                         #
#                                                                             #
# This program is distributed in the hope that it will be useful,             #
# but WITHOUT ANY WARRANTY; without even the implied warranty of              #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               #
# GNU General Public License for more details.                                #
#                                                                             #
# You should have received a copy of the GNU General Public License           #
# along with this program.  If not, see <http://www.gnu.org/licenses/>.       #
#                                                                             #
#############################################################################*/

#include <argp.h>

#include <pakfire/packagelist.h>
#include <pakfire/pakfire.h>

#include "command.h"
#include "dump.h"
#include "info.h"
#include "pakfire.h"
#include "transaction.h"

static const char* args_doc = "info [OPTIONS...] PACKAGES...";

static const char* doc = "Shows package information";

#define MAX_PACKAGES 128

struct cli_local_args {
	int dump_flags;

	// Packages
	char* packages[MAX_PACKAGES];
	unsigned int num_packages;
};

enum {
	OPT_DEVEL    = 1,
	OPT_FILELIST = 2,
};

static struct argp_option options[] = {
	{ "devel",    OPT_DEVEL,    NULL, 0, "Show development information", 0 },
	{ "filelist", OPT_FILELIST, NULL, 0, "Show the filelist",            0 },
	{ NULL },
};

static error_t parse(int key, char* arg, struct argp_state* state, void* data) {
	struct cli_local_args* args = data;

	switch (key) {
		case OPT_DEVEL:
			args->dump_flags |= PAKFIRE_PKG_DUMP_LONG;
			break;

		case OPT_FILELIST:
			args->dump_flags |= PAKFIRE_PKG_DUMP_FILELIST;
			break;

		case ARGP_KEY_ARG:
			if (args->num_packages >= MAX_PACKAGES)
				return -ENOBUFS;

			args->packages[args->num_packages++] = arg;
			break;

		default:
			return ARGP_ERR_UNKNOWN;
	}

	return 0;
}

int cli_info(void* data, int argc, char* argv[]) {
	struct cli_global_args* global_args = data;
	struct cli_local_args local_args = {};
	struct pakfire* pakfire = NULL;
	struct pakfire_packagelist* list = NULL;
	int r;

	// Parse the command line
	r = cli_parse(options, NULL, args_doc, doc, parse, 0, argc, argv, &local_args);
	if (r)
		goto ERROR;

	// Setup Pakfire
	r = cli_setup_pakfire(&pakfire, global_args);
	if (r)
		goto ERROR;

	// Allocate a new packagelist
	r = pakfire_packagelist_create(&list, global_args->ctx);
	if (r)
		goto ERROR;

	// Perform search
	for (unsigned int i = 0; i < local_args.num_packages; i++) {
		r = pakfire_search(pakfire, local_args.packages[i], PAKFIRE_SEARCH_NAME_ONLY, list);
		if (r)
			goto ERROR;
	}

	// Dump the packagelist
	r = cli_dump_packagelist(list, local_args.dump_flags);

ERROR:
	if (list)
		pakfire_packagelist_unref(list);
	if (pakfire)
		pakfire_unref(pakfire);

	return r;
}

