/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import { TalerMerchantApi } from "@gnu-taler/taler-util";
import { useTranslationContext } from "@gnu-taler/web-util/browser";
import { h, VNode } from "preact";
import { FormProvider } from "../../../../components/form/FormProvider.js";
import { InputSelector } from "../../../../components/form/InputSelector.js";
import { CardTableIncoming, CardTableVerified } from "./Table.js";

const TALER_SCREEN_ID = 71;

export interface Props {
  transfers: TalerMerchantApi.TransferDetails[];
  incomings: TalerMerchantApi.IncomingTransferDetails[];
  onLoadMoreBefore?: () => void;
  onLoadMoreAfter?: () => void;
  onShowVerified: () => void;
  onShowUnverified: () => void;
  isVerifiedTransfers?: boolean;
  isNonVerifiedTransfers?: boolean;
  accounts: string[];
  onChangePayTo: (p?: string) => void;
  payTo?: string;
  onSelectedToConfirm: (wid: TalerMerchantApi.IncomingTransferDetails) => void;
}

export function ListPage({
  payTo,
  onChangePayTo,
  transfers,
  incomings,
  onSelectedToConfirm,
  accounts,
  onLoadMoreBefore,
  onLoadMoreAfter,
  isNonVerifiedTransfers,
  isVerifiedTransfers,
  onShowUnverified,
  onShowVerified,
}: Props): VNode {
  const form = { payto_uri: payTo };

  const { i18n } = useTranslationContext();
  return (
    <section class="section is-main-section">
      <div class="columns">
        <div class="column" />
        <div class="column is-10">
          <FormProvider
            object={form}
            valueHandler={(updater) => onChangePayTo(updater(form).payto_uri)}
          >
            <InputSelector
              name="payto_uri"
              label={i18n.str`Bank account`}
              values={accounts}
              fromStr={(d) => {
                const idx = accounts.indexOf(d);
                if (idx === -1) return undefined;
                return d;
              }}
              placeholder={i18n.str`All accounts`}
              tooltip={i18n.str`Filter by account address`}
            />
          </FormProvider>
        </div>
        <div class="column" />
      </div>
      <div class="tabs">
        <ul>
          <li class={isNonVerifiedTransfers ? "is-active" : ""}>
            <div
              class="has-tooltip-right"
              data-tooltip={i18n.str`Only display transfers that have already been transferred to your bank account by the payment service provider.`}
            >
              <a onClick={onShowUnverified}>
                <i18n.Translate>Incoming</i18n.Translate>
              </a>
            </div>
          </li>
          <li class={isVerifiedTransfers ? "is-active" : ""}>
            <div
              class="has-tooltip-right"
              data-tooltip={i18n.str`Only show wire transfers confirmed by the merchant`}
            >
              <a onClick={onShowVerified}>
                <i18n.Translate>Verified</i18n.Translate>
              </a>
            </div>
          </li>
        </ul>
      </div>

      {isNonVerifiedTransfers ? (
        <CardTableIncoming
          transfers={incomings.map((o) => ({
            ...o,
            id: String(o.wtid),
          }))}
          onLoadMoreBefore={onLoadMoreBefore}
          onLoadMoreAfter={onLoadMoreAfter}
          onSelectedToConfirm={onSelectedToConfirm}
        />
      ) : (
        <CardTableVerified
          transfers={transfers.map((o) => ({
            ...o,
            id: String(o.wtid),
          }))}
          onLoadMoreBefore={onLoadMoreBefore}
          onLoadMoreAfter={onLoadMoreAfter}
        />
      )}
    </section>
  );
}
