/*
   This file is part of TALER
   Copyright (C) 2025 Taler Systems SA

   TALER is free software; you can redistribute it and/or modify it under the
   terms of the GNU General Public License as published by the Free Software
   Foundation; either version 3, or (at your option) any later version.

   TALER is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
   A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along with
   TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
/**
 * @file backenddb/pg_select_report.c
 * @brief Implementation of the select_report function for Postgres
 * @author Christian Grothoff
 */
#include "platform.h"
#include <taler/taler_error_codes.h>
#include <taler/taler_dbevents.h>
#include <taler/taler_pq_lib.h>
#include "pg_select_report.h"
#include "pg_helper.h"

enum GNUNET_DB_QueryStatus
TMH_PG_select_report (void *cls,
                      const char *instance_id,
                      uint64_t report_id,
                      char **report_program_section,
                      char **report_description,
                      char **mime_type,
                      char **data_source,
                      char **target_address,
                      struct GNUNET_TIME_Relative *frequency,
                      struct GNUNET_TIME_Relative *frequency_shift,
                      struct GNUNET_TIME_Absolute *next_transmission,
                      enum TALER_ErrorCode *last_error_code,
                      char **last_error_detail)
{
  struct PostgresClosure *pg = cls;
  struct GNUNET_PQ_QueryParam params[] = {
    GNUNET_PQ_query_param_string (instance_id),
    GNUNET_PQ_query_param_uint64 (&report_id),
    GNUNET_PQ_query_param_end
  };
  uint32_t code;
  struct GNUNET_PQ_ResultSpec rs[] = {
    GNUNET_PQ_result_spec_string ("report_program_section",
                                  report_program_section),
    GNUNET_PQ_result_spec_string ("report_description",
                                  report_description),
    GNUNET_PQ_result_spec_string ("mime_type",
                                  mime_type),
    GNUNET_PQ_result_spec_string ("data_source",
                                  data_source),
    GNUNET_PQ_result_spec_string ("target_address",
                                  target_address),
    GNUNET_PQ_result_spec_relative_time ("frequency",
                                         frequency),
    GNUNET_PQ_result_spec_relative_time ("frequency_shift",
                                         frequency_shift),
    GNUNET_PQ_result_spec_absolute_time ("next_transmission",
                                         next_transmission),
    GNUNET_PQ_result_spec_allow_null (
      GNUNET_PQ_result_spec_uint32 ("last_error_code",
                                    &code),
      NULL),
    GNUNET_PQ_result_spec_allow_null (
      GNUNET_PQ_result_spec_string ("last_error_detail",
                                    last_error_detail),
      NULL),
    GNUNET_PQ_result_spec_end
  };
  enum GNUNET_DB_QueryStatus qs;

  *last_error_detail = NULL;
  code = TALER_EC_NONE;
  check_connection (pg);
  PREPARE (pg,
           "select_report",
           "SELECT"
           "  mr.report_program_section"
           " ,mr.report_description"
           " ,mr.mime_type"
           " ,mr.data_source"
           " ,mr.target_address"
           " ,mr.frequency"
           " ,mr.frequency_shift"
           " ,mr.next_transmission"
           " ,mr.last_error_code"
           " ,mr.last_error_detail"
           " FROM merchant_reports mr"
           " JOIN merchant_instances mi"
           "   USING (merchant_serial)"
           " WHERE merchant_id=$1"
           "   AND report_serial=$2;");
  qs = GNUNET_PQ_eval_prepared_singleton_select (
    pg->conn,
    "select_report",
    params,
    rs);
  *last_error_code = (enum TALER_ErrorCode) code;
  return qs;
}
