/*
 * This file is part of LibEuFin.
 * Copyright (C) 2024-2025 Taler Systems S.A.

 * LibEuFin is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation; either version 3, or
 * (at your option) any later version.

 * LibEuFin is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General
 * Public License for more details.

 * You should have received a copy of the GNU Affero General Public
 * License along with LibEuFin; see the file COPYING.  If not, see
 * <http://www.gnu.org/licenses/>
 */

package tech.libeufin.nexus.cli

import com.github.ajalt.clikt.core.CliktCommand
import com.github.ajalt.clikt.core.Context
import com.github.ajalt.clikt.core.subcommands
import com.github.ajalt.clikt.parameters.arguments.*
import com.github.ajalt.clikt.parameters.groups.provideDelegate
import com.github.ajalt.clikt.parameters.options.convert
import com.github.ajalt.clikt.parameters.options.default
import com.github.ajalt.clikt.parameters.options.flag
import com.github.ajalt.clikt.parameters.options.option
import com.github.ajalt.clikt.parameters.types.*
import com.github.ajalt.mordant.terminal.*
import tech.libeufin.common.*
import tech.libeufin.nexus.*
import tech.libeufin.nexus.db.*
import tech.libeufin.ebics.randEbicsId
import tech.libeufin.nexus.iso20022.*
import java.util.zip.*
import java.time.Instant
import java.io.*

class ExportCmt: TalerCmd("export") {
    override fun help(context: Context) = "Export pending batches as pain001 messages"

    private val out by argument().outputStream()

    override fun run() = cliCmd(logger) {
        nexusConfig(config).withDb { db, cfg ->
            // Create and get pending batches
            db.initiated.batch(Instant.now(), randEbicsId(), cfg.submit.requireAck)
            val batches = db.initiated.submittable()

            var nbTx: Int = 0
            ZipOutputStream(BufferedOutputStream(out)).use { zip ->
                val ebicsCfg = cfg.ebics
                val metadata = buildString {
                    append("Exported ${batches.size} pain.001 files:")
                    for (batch in batches) {
                        nbTx = batch.payments.size
                        val entry = ZipEntry("${batch.creationDate.toDateTimeFilePath()}-${batch.messageId}.xml")
                        zip.putNextEntry(entry)
                        val msg = batchToPain001Msg(ebicsCfg.account, batch)
                        
                        val xml = createPain001(
                            msg = msg,
                            dialect = ebicsCfg.dialect,
                            instant = false
                        )
                        zip.write(xml)
                        append("\nbatch ${batch.messageId}:")
                        for (tx in batch.payments) {
                            append("\n- tx ${tx.endToEndId} ${tx.amount} ${tx.creditor.iban} '${tx.creditor.receiverName}'")
                        }
                        append("\n")
                    }
                }

                zip.putNextEntry(ZipEntry("README.txt"))
                zip.write(metadata.toByteArray())
                logger.info(metadata)
            }
        }
    } 
}

class ImportCmt: TalerCmd("import") {
    override fun help(context: Context) = "Import EBICS camt files"

    private val sources by argument().inputStream().multiple(required = true)

    override fun run() = cliCmd(logger) {
        nexusConfig(config).withDb { db, cfg ->
            for (source in sources) {
                var nbTx: Int = 0
                source.use { xml ->
                    nbTx += registerTxs(db, cfg, xml)
                }
                logger.info("Imported $nbTx transactions from $source")
            }
        }
    } 
}

class StatusCmd: TalerCmd("status") {
    override fun help(context: Context) = "Change batches or transactions status"

    enum class Kind {
        batch,
        tx
    }

    private val kind by argument().enum<Kind>()
    private val id by argument()
    private val status by argument().enum<StatusUpdate>()
    private val msg by argument().optional()

    override fun run() = cliCmd(logger) {
        nexusConfig(config).withDb { db, cfg ->
            when (kind) {
                Kind.batch -> if (db.initiated.batchStatusUpdate(id, status, msg)) {
                    logger.info("Updated batch '${id}' to ${status}")
                } else {
                    throw Exception("Unknown batch '$id'")
                }
                Kind.tx -> if (db.initiated.txStatusUpdate(id, null, status, msg)) {
                    logger.info("Updated tx '${id}' to ${status}")
                } else {
                    throw Exception("Unknown tx '$id'")
                }
            }
        }
    } 
}

class AckCmd: TalerCmd("ack") {
    override fun help(context: Context) = "Manually acknowledge the outgoing transaction for submission"

    private val ids by argument().long().multiple()

    override fun run() = cliCmd(logger) {
        nexusConfig(config).withDb { db, cfg ->
            for (id in ids) {
                if (db.initiated.ack(id)) {
                    logger.info("Mark $id as acknowledge for submission")
                } else {
                    logger.warn("Unknown transaction $id")
                }
            }
        }
    } 
}

class ManualCmd : TalerCmd("manual") {
    init {
        subcommands(ExportCmt(), ImportCmt(), StatusCmd(), AckCmd())
    }

    override fun help(context: Context) = "Manual management commands"

    override fun run() = Unit
}