/*
 *  Copyright 2008-2013 NVIDIA Corporation
 *  Modifications Copyright© 2019-2025 Advanced Micro Devices, Inc. All rights reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

#include <thrust/detail/config.h>

// need to suppress deprecation warnings inside a lot of thrust headers
THRUST_SUPPRESS_DEPRECATED_PUSH

// Disabled on MSVC && NVCC < 11.1 for GH issue #1098.
#if THRUST_HOST_COMPILER == THRUST_HOST_COMPILER_MSVC && defined(__CUDACC__)
#  if (__CUDACC_VER_MAJOR__ < 11) || (__CUDACC_VER_MAJOR__ == 11 && __CUDACC_VER_MINOR__ < 1)
#    define THRUST_BUG_1098_ACTIVE
#  endif // NVCC version check
#endif // MSVC + NVCC check

#if THRUST_CPP_DIALECT >= 2017 && !defined(THRUST_BUG_1098_ACTIVE)

#  include <thrust/async/sort.h>
#  include <thrust/device_vector.h>
#  include <thrust/host_vector.h>

#  include "test_param_fixtures.hpp"
#  include "test_utils.hpp"

TESTS_DEFINE(AsyncSortTests, NumericalTestsParams);

enum wait_policy
{
  wait_for_futures,
  do_not_wait_for_futures
};

template <typename T>
struct custom_greater
{
  THRUST_HOST_DEVICE bool operator()(T rhs, T lhs) const
  {
    return lhs > rhs;
  }
};

#  define DEFINE_SORT_INVOKER(name, ...)                                                                      \
    template <typename T>                                                                                     \
    struct name                                                                                               \
    {                                                                                                         \
      template <typename ForwardIt, typename Sentinel>                                                        \
      THRUST_HOST static void sync(ForwardIt&& first, Sentinel&& last)                                        \
      {                                                                                                       \
        ::thrust::sort(THRUST_FWD(first), THRUST_FWD(last));                                                  \
      }                                                                                                       \
                                                                                                              \
      template <typename ForwardIt, typename Sentinel>                                                        \
      THRUST_HOST static auto async(ForwardIt&& first, Sentinel&& last) THRUST_RETURNS(::thrust::async::sort( \
        __VA_ARGS__ THRUST_PP_COMMA_IF(THRUST_PP_ARITY(__VA_ARGS__)) THRUST_FWD(first), THRUST_FWD(last)))    \
    };                                                                                                        \
    /**/

DEFINE_SORT_INVOKER(sort_invoker);
DEFINE_SORT_INVOKER(sort_invoker_device, thrust::device);

#  define DEFINE_SORT_OP_INVOKER(name, op, ...)                                                                     \
    template <typename T>                                                                                           \
    struct name                                                                                                     \
    {                                                                                                               \
      template <typename ForwardIt, typename Sentinel>                                                              \
      THRUST_HOST static void sync(ForwardIt&& first, Sentinel&& last)                                              \
      {                                                                                                             \
        ::thrust::sort(THRUST_FWD(first), THRUST_FWD(last), op<T>{});                                               \
      }                                                                                                             \
                                                                                                                    \
      template <typename ForwardIt, typename Sentinel>                                                              \
      THRUST_HOST static auto async(ForwardIt&& first, Sentinel&& last) THRUST_RETURNS(::thrust::async::sort(       \
        __VA_ARGS__ THRUST_PP_COMMA_IF(THRUST_PP_ARITY(__VA_ARGS__)) THRUST_FWD(first), THRUST_FWD(last), op<T>{})) \
    };                                                                                                              \
    /**/

DEFINE_SORT_OP_INVOKER(sort_invoker_less, thrust::less);
DEFINE_SORT_OP_INVOKER(sort_invoker_less_device, thrust::less, thrust::device);

DEFINE_SORT_OP_INVOKER(sort_invoker_greater, thrust::greater);
DEFINE_SORT_OP_INVOKER(sort_invoker_greater_device, thrust::greater, thrust::device);

DEFINE_SORT_OP_INVOKER(sort_invoker_custom_greater, custom_greater);
DEFINE_SORT_OP_INVOKER(sort_invoker_custom_greater_device, custom_greater, thrust::device);

#  undef DEFINE_SORT_INVOKER
#  undef DEFINE_SORT_OP_INVOKER

TYPED_TEST(AsyncSortTests, AsyncSortInstance)
{
  SCOPED_TRACE(testing::Message() << "with device_id= " << test::set_device_from_ctest());
  using T = typename TestFixture::input_type;
  for (auto size : get_sizes())
  {
    SCOPED_TRACE(testing::Message() << "with size = " << size);
    for (auto seed : get_seeds())
    {
      SCOPED_TRACE(testing::Message() << "with seed= " << seed);

      thrust::host_vector<T> h0_data =
        get_random_data<T>(size, get_default_limits<T>::min(), get_default_limits<T>::max(), seed);
      thrust::device_vector<T> d0_data(h0_data);

      ASSERT_EQ(h0_data, d0_data);

      thrust::sort(h0_data.begin(), h0_data.end());

      auto f0 = thrust::async::sort(d0_data.begin(), d0_data.end());

      f0.wait();

      ASSERT_EQ(h0_data, d0_data);
    }
  }
}

TYPED_TEST(AsyncSortTests, AsyncSortWithPolicyInstance)
{
  SCOPED_TRACE(testing::Message() << "with device_id= " << test::set_device_from_ctest());
  using T = typename TestFixture::input_type;
  for (auto size : get_sizes())
  {
    SCOPED_TRACE(testing::Message() << "with size = " << size);
    for (auto seed : get_seeds())
    {
      SCOPED_TRACE(testing::Message() << "with seed= " << seed);

      thrust::host_vector<T> h0_data =
        get_random_data<T>(size, get_default_limits<T>::min(), get_default_limits<T>::max(), seed);
      thrust::device_vector<T> d0_data(h0_data);

      ASSERT_EQ(h0_data, d0_data);

      thrust::sort(h0_data.begin(), h0_data.end());

      auto f0 = thrust::async::sort(thrust::device, d0_data.begin(), d0_data.end());

      f0.wait();

      ASSERT_EQ(h0_data, d0_data);
    }
  }
}

///////////////////////////////////////////////////////////////////////////////

template <typename T, template <typename> class SortInvoker, wait_policy WaitPolicy>
THRUST_HOST void test_async_sort()
{
  for (auto size : get_sizes())
  {
    SCOPED_TRACE(testing::Message() << "with size = " << size);
    for (auto seed : get_seeds())
    {
      SCOPED_TRACE(testing::Message() << "with seed= " << seed);

      thrust::host_vector<T> h0_data(
        get_random_data<T>(size, get_default_limits<T>::min(), get_default_limits<T>::max(), seed));
      thrust::device_vector<T> d0_data(h0_data);

      ASSERT_EQ(h0_data, d0_data);

      SortInvoker<T>::sync(h0_data.begin(), h0_data.end());

      auto f0 = SortInvoker<T>::async(d0_data.begin(), d0_data.end());

      THRUST_IF_CONSTEXPR (wait_for_futures == WaitPolicy)
      {
        f0.wait();

        ASSERT_EQ(h0_data, d0_data);
      }
    }
  }
}

TYPED_TEST(AsyncSortTests, test_async_sort)
{
  SCOPED_TRACE(testing::Message() << "with device_id= " << test::set_device_from_ctest());
  using T = typename TestFixture::input_type;
  test_async_sort<T, sort_invoker, wait_for_futures>();
}

TYPED_TEST(AsyncSortTests, test_async_sort_no_wait)
{
  SCOPED_TRACE(testing::Message() << "with device_id= " << test::set_device_from_ctest());
  using T = typename TestFixture::input_type;
  test_async_sort<T, sort_invoker, do_not_wait_for_futures>();
}

TYPED_TEST(AsyncSortTests, test_async_sort_policy)
{
  SCOPED_TRACE(testing::Message() << "with device_id= " << test::set_device_from_ctest());
  using T = typename TestFixture::input_type;
  test_async_sort<T, sort_invoker_device, wait_for_futures>();
}

TYPED_TEST(AsyncSortTests, test_async_sort_policy_no_wait)
{
  SCOPED_TRACE(testing::Message() << "with device_id= " << test::set_device_from_ctest());
  using T = typename TestFixture::input_type;
  test_async_sort<T, sort_invoker_device, do_not_wait_for_futures>();
}

TYPED_TEST(AsyncSortTests, test_async_sort_less)
{
  SCOPED_TRACE(testing::Message() << "with device_id= " << test::set_device_from_ctest());
  using T = typename TestFixture::input_type;
  test_async_sort<T, sort_invoker_less, wait_for_futures>();
}

TYPED_TEST(AsyncSortTests, test_async_sort_less_no_wait)
{
  SCOPED_TRACE(testing::Message() << "with device_id= " << test::set_device_from_ctest());
  using T = typename TestFixture::input_type;
  test_async_sort<T, sort_invoker_less, do_not_wait_for_futures>();
}

TYPED_TEST(AsyncSortTests, test_async_sort_policy_less)
{
  SCOPED_TRACE(testing::Message() << "with device_id= " << test::set_device_from_ctest());
  using T = typename TestFixture::input_type;
  test_async_sort<T, sort_invoker_less_device, wait_for_futures>();
}

TYPED_TEST(AsyncSortTests, test_async_sort_policy_less_no_wait)
{
  SCOPED_TRACE(testing::Message() << "with device_id= " << test::set_device_from_ctest());
  using T = typename TestFixture::input_type;
  test_async_sort<T, sort_invoker_less_device, do_not_wait_for_futures>();
}

TYPED_TEST(AsyncSortTests, test_async_sort_greater)
{
  SCOPED_TRACE(testing::Message() << "with device_id= " << test::set_device_from_ctest());
  using T = typename TestFixture::input_type;
  test_async_sort<T, sort_invoker_greater, wait_for_futures>();
}

TYPED_TEST(AsyncSortTests, test_async_sort_greater_no_wait)
{
  SCOPED_TRACE(testing::Message() << "with device_id= " << test::set_device_from_ctest());
  using T = typename TestFixture::input_type;
  test_async_sort<T, sort_invoker_greater, do_not_wait_for_futures>();
}

TYPED_TEST(AsyncSortTests, test_async_sort_policy_greater)
{
  SCOPED_TRACE(testing::Message() << "with device_id= " << test::set_device_from_ctest());
  using T = typename TestFixture::input_type;
  test_async_sort<T, sort_invoker_greater_device, wait_for_futures>();
}

TYPED_TEST(AsyncSortTests, test_async_sort_policy_greater_no_wait)
{
  SCOPED_TRACE(testing::Message() << "with device_id= " << test::set_device_from_ctest());
  using T = typename TestFixture::input_type;
  test_async_sort<T, sort_invoker_greater_device, do_not_wait_for_futures>();
}

TYPED_TEST(AsyncSortTests, test_async_sort_custom_greater)
{
  SCOPED_TRACE(testing::Message() << "with device_id= " << test::set_device_from_ctest());
  using T = typename TestFixture::input_type;
  test_async_sort<T, sort_invoker_custom_greater, wait_for_futures>();
}

TYPED_TEST(AsyncSortTests, test_async_sort_custom_greater_no_wait)
{
  SCOPED_TRACE(testing::Message() << "with device_id= " << test::set_device_from_ctest());
  using T = typename TestFixture::input_type;
  test_async_sort<T, sort_invoker_custom_greater, do_not_wait_for_futures>();
}

TYPED_TEST(AsyncSortTests, test_async_sort_policy_custom_greater)
{
  SCOPED_TRACE(testing::Message() << "with device_id= " << test::set_device_from_ctest());
  using T = typename TestFixture::input_type;
  test_async_sort<T, sort_invoker_custom_greater_device, wait_for_futures>();
}

TYPED_TEST(AsyncSortTests, test_async_sort_policy_custom_greater_no_wait)
{
  SCOPED_TRACE(testing::Message() << "with device_id= " << test::set_device_from_ctest());
  using T = typename TestFixture::input_type;
  test_async_sort<T, sort_invoker_custom_greater_device, do_not_wait_for_futures>();
}

///////////////////////////////////////////////////////////////////////////////

// TODO: Async copy then sort.

// TODO: Test future return type.

#endif

// we need to leak the suppression on clang/MSVC to suppresses warnings from the cudafe1.stub.c file
#if THRUST_HOST_COMPILER != THRUST_HOST_COMPILER_CLANG && THRUST_HOST_COMPILER != THRUST_HOST_COMPILER_MSVC
THRUST_SUPPRESS_DEPRECATED_POP
#endif // THRUST_HOST_COMPILER != THRUST_HOST_COMPILER_CLANG && THRUST_HOST_COMPILER != THRUST_HOST_COMPILER_MSVC
