// Package keyvault implements the Azure ARM Keyvault service API version 2015-06-01.
//
// Performs cryptographic key operations and vault operations against the Key Vault service.
package keyvault

// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License. See License.txt in the project root for license information.
//
// Code generated by Microsoft (R) AutoRest Code Generator.
// Changes may cause incorrect behavior and will be lost if the code is regenerated.

import (
	"context"
	"github.com/Azure/go-autorest/autorest"
	"github.com/Azure/go-autorest/autorest/azure"
	"github.com/Azure/go-autorest/autorest/validation"
	"github.com/Azure/go-autorest/tracing"
	"net/http"
)

// BaseClient is the base client for Keyvault.
type BaseClient struct {
	autorest.Client
}

// New creates an instance of the BaseClient client.
func New() BaseClient {
	return NewWithoutDefaults()
}

// NewWithoutDefaults creates an instance of the BaseClient client.
func NewWithoutDefaults() BaseClient {
	return BaseClient{
		Client: autorest.NewClientWithUserAgent(UserAgent()),
	}
}

// BackupKey requests that a backup of the specified key be downloaded to the client.
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
// keyName - the name of the key
func (client BaseClient) BackupKey(ctx context.Context, vaultBaseURL string, keyName string) (result BackupKeyResult, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.BackupKey")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.BackupKeyPreparer(ctx, vaultBaseURL, keyName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "BackupKey", nil, "Failure preparing request")
		return
	}

	resp, err := client.BackupKeySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "BackupKey", resp, "Failure sending request")
		return
	}

	result, err = client.BackupKeyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "BackupKey", resp, "Failure responding to request")
		return
	}

	return
}

// BackupKeyPreparer prepares the BackupKey request.
func (client BaseClient) BackupKeyPreparer(ctx context.Context, vaultBaseURL string, keyName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name": autorest.Encode("path", keyName),
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/keys/{key-name}/backup", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// BackupKeySender sends the BackupKey request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) BackupKeySender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// BackupKeyResponder handles the response to the BackupKey request. The method always
// closes the http.Response Body.
func (client BaseClient) BackupKeyResponder(resp *http.Response) (result BackupKeyResult, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// CreateCertificate creates a new certificate version. If this is the first version, the certificate resource is
// created.
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
// certificateName - the name of the certificate
// parameters - the parameters to create certificate.
func (client BaseClient) CreateCertificate(ctx context.Context, vaultBaseURL string, certificateName string, parameters CertificateCreateParameters) (result CertificateOperation, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.CreateCertificate")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: certificateName,
			Constraints: []validation.Constraint{{Target: "certificateName", Name: validation.Pattern, Rule: `^[0-9a-zA-Z-]+$`, Chain: nil}}},
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.CertificatePolicy", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "parameters.CertificatePolicy.X509CertificateProperties", Name: validation.Null, Rule: false,
					Chain: []validation.Constraint{{Target: "parameters.CertificatePolicy.X509CertificateProperties.ValidityInMonths", Name: validation.Null, Rule: false,
						Chain: []validation.Constraint{{Target: "parameters.CertificatePolicy.X509CertificateProperties.ValidityInMonths", Name: validation.InclusiveMinimum, Rule: int64(0), Chain: nil}}},
					}},
				}}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "CreateCertificate", err.Error())
	}

	req, err := client.CreateCertificatePreparer(ctx, vaultBaseURL, certificateName, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "CreateCertificate", nil, "Failure preparing request")
		return
	}

	resp, err := client.CreateCertificateSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "CreateCertificate", resp, "Failure sending request")
		return
	}

	result, err = client.CreateCertificateResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "CreateCertificate", resp, "Failure responding to request")
		return
	}

	return
}

// CreateCertificatePreparer prepares the CreateCertificate request.
func (client BaseClient) CreateCertificatePreparer(ctx context.Context, vaultBaseURL string, certificateName string, parameters CertificateCreateParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"certificate-name": autorest.Encode("path", certificateName),
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/{certificate-name}/create", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// CreateCertificateSender sends the CreateCertificate request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) CreateCertificateSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// CreateCertificateResponder handles the response to the CreateCertificate request. The method always
// closes the http.Response Body.
func (client BaseClient) CreateCertificateResponder(resp *http.Response) (result CertificateOperation, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusAccepted),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// CreateKey creates a new, named, key in the specified vault.
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
// keyName - the name of the key
// parameters - the parameters to create key.
func (client BaseClient) CreateKey(ctx context.Context, vaultBaseURL string, keyName string, parameters KeyCreateParameters) (result KeyBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.CreateKey")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: keyName,
			Constraints: []validation.Constraint{{Target: "keyName", Name: validation.Pattern, Rule: `^[0-9a-zA-Z-]+$`, Chain: nil}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "CreateKey", err.Error())
	}

	req, err := client.CreateKeyPreparer(ctx, vaultBaseURL, keyName, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "CreateKey", nil, "Failure preparing request")
		return
	}

	resp, err := client.CreateKeySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "CreateKey", resp, "Failure sending request")
		return
	}

	result, err = client.CreateKeyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "CreateKey", resp, "Failure responding to request")
		return
	}

	return
}

// CreateKeyPreparer prepares the CreateKey request.
func (client BaseClient) CreateKeyPreparer(ctx context.Context, vaultBaseURL string, keyName string, parameters KeyCreateParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name": autorest.Encode("path", keyName),
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/keys/{key-name}/create", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// CreateKeySender sends the CreateKey request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) CreateKeySender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// CreateKeyResponder handles the response to the CreateKey request. The method always
// closes the http.Response Body.
func (client BaseClient) CreateKeyResponder(resp *http.Response) (result KeyBundle, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// Decrypt decrypts a single block of encrypted data
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
// keyName - the name of the key
// keyVersion - the version of the key
// parameters - the parameters for key operations.
func (client BaseClient) Decrypt(ctx context.Context, vaultBaseURL string, keyName string, keyVersion string, parameters KeyOperationsParameters) (result KeyOperationResult, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.Decrypt")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.Value", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "Decrypt", err.Error())
	}

	req, err := client.DecryptPreparer(ctx, vaultBaseURL, keyName, keyVersion, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "Decrypt", nil, "Failure preparing request")
		return
	}

	resp, err := client.DecryptSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "Decrypt", resp, "Failure sending request")
		return
	}

	result, err = client.DecryptResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "Decrypt", resp, "Failure responding to request")
		return
	}

	return
}

// DecryptPreparer prepares the Decrypt request.
func (client BaseClient) DecryptPreparer(ctx context.Context, vaultBaseURL string, keyName string, keyVersion string, parameters KeyOperationsParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name":    autorest.Encode("path", keyName),
		"key-version": autorest.Encode("path", keyVersion),
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/keys/{key-name}/{key-version}/decrypt", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// DecryptSender sends the Decrypt request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) DecryptSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// DecryptResponder handles the response to the Decrypt request. The method always
// closes the http.Response Body.
func (client BaseClient) DecryptResponder(resp *http.Response) (result KeyOperationResult, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// DeleteCertificate deletes a certificate from the specified vault.
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
// certificateName - the name of the certificate in the given vault
func (client BaseClient) DeleteCertificate(ctx context.Context, vaultBaseURL string, certificateName string) (result CertificateBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.DeleteCertificate")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.DeleteCertificatePreparer(ctx, vaultBaseURL, certificateName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteCertificate", nil, "Failure preparing request")
		return
	}

	resp, err := client.DeleteCertificateSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteCertificate", resp, "Failure sending request")
		return
	}

	result, err = client.DeleteCertificateResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteCertificate", resp, "Failure responding to request")
		return
	}

	return
}

// DeleteCertificatePreparer prepares the DeleteCertificate request.
func (client BaseClient) DeleteCertificatePreparer(ctx context.Context, vaultBaseURL string, certificateName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"certificate-name": autorest.Encode("path", certificateName),
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsDelete(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/{certificate-name}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// DeleteCertificateSender sends the DeleteCertificate request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) DeleteCertificateSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// DeleteCertificateResponder handles the response to the DeleteCertificate request. The method always
// closes the http.Response Body.
func (client BaseClient) DeleteCertificateResponder(resp *http.Response) (result CertificateBundle, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// DeleteCertificateContacts deletes the certificate contacts for the specified vault.
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
func (client BaseClient) DeleteCertificateContacts(ctx context.Context, vaultBaseURL string) (result Contacts, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.DeleteCertificateContacts")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.DeleteCertificateContactsPreparer(ctx, vaultBaseURL)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteCertificateContacts", nil, "Failure preparing request")
		return
	}

	resp, err := client.DeleteCertificateContactsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteCertificateContacts", resp, "Failure sending request")
		return
	}

	result, err = client.DeleteCertificateContactsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteCertificateContacts", resp, "Failure responding to request")
		return
	}

	return
}

// DeleteCertificateContactsPreparer prepares the DeleteCertificateContacts request.
func (client BaseClient) DeleteCertificateContactsPreparer(ctx context.Context, vaultBaseURL string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsDelete(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPath("/certificates/contacts"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// DeleteCertificateContactsSender sends the DeleteCertificateContacts request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) DeleteCertificateContactsSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// DeleteCertificateContactsResponder handles the response to the DeleteCertificateContacts request. The method always
// closes the http.Response Body.
func (client BaseClient) DeleteCertificateContactsResponder(resp *http.Response) (result Contacts, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// DeleteCertificateIssuer deletes the specified certificate issuer.
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
// issuerName - the name of the issuer.
func (client BaseClient) DeleteCertificateIssuer(ctx context.Context, vaultBaseURL string, issuerName string) (result IssuerBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.DeleteCertificateIssuer")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.DeleteCertificateIssuerPreparer(ctx, vaultBaseURL, issuerName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteCertificateIssuer", nil, "Failure preparing request")
		return
	}

	resp, err := client.DeleteCertificateIssuerSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteCertificateIssuer", resp, "Failure sending request")
		return
	}

	result, err = client.DeleteCertificateIssuerResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteCertificateIssuer", resp, "Failure responding to request")
		return
	}

	return
}

// DeleteCertificateIssuerPreparer prepares the DeleteCertificateIssuer request.
func (client BaseClient) DeleteCertificateIssuerPreparer(ctx context.Context, vaultBaseURL string, issuerName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"issuer-name": autorest.Encode("path", issuerName),
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsDelete(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/issuers/{issuer-name}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// DeleteCertificateIssuerSender sends the DeleteCertificateIssuer request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) DeleteCertificateIssuerSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// DeleteCertificateIssuerResponder handles the response to the DeleteCertificateIssuer request. The method always
// closes the http.Response Body.
func (client BaseClient) DeleteCertificateIssuerResponder(resp *http.Response) (result IssuerBundle, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// DeleteCertificateOperation deletes the certificate operation.
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
// certificateName - the name of the certificate
func (client BaseClient) DeleteCertificateOperation(ctx context.Context, vaultBaseURL string, certificateName string) (result CertificateOperation, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.DeleteCertificateOperation")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.DeleteCertificateOperationPreparer(ctx, vaultBaseURL, certificateName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteCertificateOperation", nil, "Failure preparing request")
		return
	}

	resp, err := client.DeleteCertificateOperationSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteCertificateOperation", resp, "Failure sending request")
		return
	}

	result, err = client.DeleteCertificateOperationResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteCertificateOperation", resp, "Failure responding to request")
		return
	}

	return
}

// DeleteCertificateOperationPreparer prepares the DeleteCertificateOperation request.
func (client BaseClient) DeleteCertificateOperationPreparer(ctx context.Context, vaultBaseURL string, certificateName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"certificate-name": autorest.Encode("path", certificateName),
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsDelete(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/{certificate-name}/pending", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// DeleteCertificateOperationSender sends the DeleteCertificateOperation request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) DeleteCertificateOperationSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// DeleteCertificateOperationResponder handles the response to the DeleteCertificateOperation request. The method always
// closes the http.Response Body.
func (client BaseClient) DeleteCertificateOperationResponder(resp *http.Response) (result CertificateOperation, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// DeleteKey deletes the specified key
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
// keyName - the name of the key
func (client BaseClient) DeleteKey(ctx context.Context, vaultBaseURL string, keyName string) (result KeyBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.DeleteKey")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.DeleteKeyPreparer(ctx, vaultBaseURL, keyName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteKey", nil, "Failure preparing request")
		return
	}

	resp, err := client.DeleteKeySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteKey", resp, "Failure sending request")
		return
	}

	result, err = client.DeleteKeyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteKey", resp, "Failure responding to request")
		return
	}

	return
}

// DeleteKeyPreparer prepares the DeleteKey request.
func (client BaseClient) DeleteKeyPreparer(ctx context.Context, vaultBaseURL string, keyName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name": autorest.Encode("path", keyName),
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsDelete(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/keys/{key-name}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// DeleteKeySender sends the DeleteKey request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) DeleteKeySender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// DeleteKeyResponder handles the response to the DeleteKey request. The method always
// closes the http.Response Body.
func (client BaseClient) DeleteKeyResponder(resp *http.Response) (result KeyBundle, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// DeleteSecret deletes a secret from the specified vault.
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
// secretName - the name of the secret in the given vault
func (client BaseClient) DeleteSecret(ctx context.Context, vaultBaseURL string, secretName string) (result SecretBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.DeleteSecret")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.DeleteSecretPreparer(ctx, vaultBaseURL, secretName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteSecret", nil, "Failure preparing request")
		return
	}

	resp, err := client.DeleteSecretSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteSecret", resp, "Failure sending request")
		return
	}

	result, err = client.DeleteSecretResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "DeleteSecret", resp, "Failure responding to request")
		return
	}

	return
}

// DeleteSecretPreparer prepares the DeleteSecret request.
func (client BaseClient) DeleteSecretPreparer(ctx context.Context, vaultBaseURL string, secretName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"secret-name": autorest.Encode("path", secretName),
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsDelete(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/secrets/{secret-name}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// DeleteSecretSender sends the DeleteSecret request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) DeleteSecretSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// DeleteSecretResponder handles the response to the DeleteSecret request. The method always
// closes the http.Response Body.
func (client BaseClient) DeleteSecretResponder(resp *http.Response) (result SecretBundle, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// Encrypt encrypts an arbitrary sequence of bytes using an encryption key that is stored in Azure Key Vault.
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
// keyName - the name of the key
// keyVersion - the version of the key
// parameters - the parameters for key operations.
func (client BaseClient) Encrypt(ctx context.Context, vaultBaseURL string, keyName string, keyVersion string, parameters KeyOperationsParameters) (result KeyOperationResult, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.Encrypt")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.Value", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "Encrypt", err.Error())
	}

	req, err := client.EncryptPreparer(ctx, vaultBaseURL, keyName, keyVersion, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "Encrypt", nil, "Failure preparing request")
		return
	}

	resp, err := client.EncryptSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "Encrypt", resp, "Failure sending request")
		return
	}

	result, err = client.EncryptResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "Encrypt", resp, "Failure responding to request")
		return
	}

	return
}

// EncryptPreparer prepares the Encrypt request.
func (client BaseClient) EncryptPreparer(ctx context.Context, vaultBaseURL string, keyName string, keyVersion string, parameters KeyOperationsParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name":    autorest.Encode("path", keyName),
		"key-version": autorest.Encode("path", keyVersion),
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/keys/{key-name}/{key-version}/encrypt", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// EncryptSender sends the Encrypt request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) EncryptSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// EncryptResponder handles the response to the Encrypt request. The method always
// closes the http.Response Body.
func (client BaseClient) EncryptResponder(resp *http.Response) (result KeyOperationResult, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetCertificate gets a Certificate.
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
// certificateName - the name of the certificate in the given vault
// certificateVersion - the version of the certificate
func (client BaseClient) GetCertificate(ctx context.Context, vaultBaseURL string, certificateName string, certificateVersion string) (result CertificateBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetCertificate")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.GetCertificatePreparer(ctx, vaultBaseURL, certificateName, certificateVersion)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificate", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetCertificateSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificate", resp, "Failure sending request")
		return
	}

	result, err = client.GetCertificateResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificate", resp, "Failure responding to request")
		return
	}

	return
}

// GetCertificatePreparer prepares the GetCertificate request.
func (client BaseClient) GetCertificatePreparer(ctx context.Context, vaultBaseURL string, certificateName string, certificateVersion string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"certificate-name":    autorest.Encode("path", certificateName),
		"certificate-version": autorest.Encode("path", certificateVersion),
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/{certificate-name}/{certificate-version}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetCertificateSender sends the GetCertificate request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetCertificateSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetCertificateResponder handles the response to the GetCertificate request. The method always
// closes the http.Response Body.
func (client BaseClient) GetCertificateResponder(resp *http.Response) (result CertificateBundle, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetCertificateContacts gets the certificate contacts for the specified vault.
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
func (client BaseClient) GetCertificateContacts(ctx context.Context, vaultBaseURL string) (result Contacts, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetCertificateContacts")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.GetCertificateContactsPreparer(ctx, vaultBaseURL)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificateContacts", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetCertificateContactsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificateContacts", resp, "Failure sending request")
		return
	}

	result, err = client.GetCertificateContactsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificateContacts", resp, "Failure responding to request")
		return
	}

	return
}

// GetCertificateContactsPreparer prepares the GetCertificateContacts request.
func (client BaseClient) GetCertificateContactsPreparer(ctx context.Context, vaultBaseURL string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPath("/certificates/contacts"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetCertificateContactsSender sends the GetCertificateContacts request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetCertificateContactsSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetCertificateContactsResponder handles the response to the GetCertificateContacts request. The method always
// closes the http.Response Body.
func (client BaseClient) GetCertificateContactsResponder(resp *http.Response) (result Contacts, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetCertificateIssuer gets the specified certificate issuer.
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
// issuerName - the name of the issuer.
func (client BaseClient) GetCertificateIssuer(ctx context.Context, vaultBaseURL string, issuerName string) (result IssuerBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetCertificateIssuer")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.GetCertificateIssuerPreparer(ctx, vaultBaseURL, issuerName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificateIssuer", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetCertificateIssuerSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificateIssuer", resp, "Failure sending request")
		return
	}

	result, err = client.GetCertificateIssuerResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificateIssuer", resp, "Failure responding to request")
		return
	}

	return
}

// GetCertificateIssuerPreparer prepares the GetCertificateIssuer request.
func (client BaseClient) GetCertificateIssuerPreparer(ctx context.Context, vaultBaseURL string, issuerName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"issuer-name": autorest.Encode("path", issuerName),
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/issuers/{issuer-name}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetCertificateIssuerSender sends the GetCertificateIssuer request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetCertificateIssuerSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetCertificateIssuerResponder handles the response to the GetCertificateIssuer request. The method always
// closes the http.Response Body.
func (client BaseClient) GetCertificateIssuerResponder(resp *http.Response) (result IssuerBundle, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetCertificateIssuers list certificate issuers for the specified vault.
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
// maxresults - maximum number of results to return in a page. If not specified the service will return up to
// 25 results.
func (client BaseClient) GetCertificateIssuers(ctx context.Context, vaultBaseURL string, maxresults *int32) (result CertificateIssuerListResultPage, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetCertificateIssuers")
		defer func() {
			sc := -1
			if result.cilr.Response.Response != nil {
				sc = result.cilr.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: maxresults,
			Constraints: []validation.Constraint{{Target: "maxresults", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "maxresults", Name: validation.InclusiveMaximum, Rule: int64(25), Chain: nil},
					{Target: "maxresults", Name: validation.InclusiveMinimum, Rule: int64(1), Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "GetCertificateIssuers", err.Error())
	}

	result.fn = client.getCertificateIssuersNextResults
	req, err := client.GetCertificateIssuersPreparer(ctx, vaultBaseURL, maxresults)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificateIssuers", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetCertificateIssuersSender(req)
	if err != nil {
		result.cilr.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificateIssuers", resp, "Failure sending request")
		return
	}

	result.cilr, err = client.GetCertificateIssuersResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificateIssuers", resp, "Failure responding to request")
		return
	}
	if result.cilr.hasNextLink() && result.cilr.IsEmpty() {
		err = result.NextWithContext(ctx)
		return
	}

	return
}

// GetCertificateIssuersPreparer prepares the GetCertificateIssuers request.
func (client BaseClient) GetCertificateIssuersPreparer(ctx context.Context, vaultBaseURL string, maxresults *int32) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if maxresults != nil {
		queryParameters["maxresults"] = autorest.Encode("query", *maxresults)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPath("/certificates/issuers"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetCertificateIssuersSender sends the GetCertificateIssuers request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetCertificateIssuersSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetCertificateIssuersResponder handles the response to the GetCertificateIssuers request. The method always
// closes the http.Response Body.
func (client BaseClient) GetCertificateIssuersResponder(resp *http.Response) (result CertificateIssuerListResult, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// getCertificateIssuersNextResults retrieves the next set of results, if any.
func (client BaseClient) getCertificateIssuersNextResults(ctx context.Context, lastResults CertificateIssuerListResult) (result CertificateIssuerListResult, err error) {
	req, err := lastResults.certificateIssuerListResultPreparer(ctx)
	if err != nil {
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getCertificateIssuersNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.GetCertificateIssuersSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getCertificateIssuersNextResults", resp, "Failure sending next results request")
	}
	result, err = client.GetCertificateIssuersResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "getCertificateIssuersNextResults", resp, "Failure responding to next results request")
	}
	return
}

// GetCertificateIssuersComplete enumerates all values, automatically crossing page boundaries as required.
func (client BaseClient) GetCertificateIssuersComplete(ctx context.Context, vaultBaseURL string, maxresults *int32) (result CertificateIssuerListResultIterator, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetCertificateIssuers")
		defer func() {
			sc := -1
			if result.Response().Response.Response != nil {
				sc = result.page.Response().Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	result.page, err = client.GetCertificateIssuers(ctx, vaultBaseURL, maxresults)
	return
}

// GetCertificateOperation gets the certificate operation response.
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
// certificateName - the name of the certificate
func (client BaseClient) GetCertificateOperation(ctx context.Context, vaultBaseURL string, certificateName string) (result CertificateOperation, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetCertificateOperation")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.GetCertificateOperationPreparer(ctx, vaultBaseURL, certificateName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificateOperation", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetCertificateOperationSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificateOperation", resp, "Failure sending request")
		return
	}

	result, err = client.GetCertificateOperationResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificateOperation", resp, "Failure responding to request")
		return
	}

	return
}

// GetCertificateOperationPreparer prepares the GetCertificateOperation request.
func (client BaseClient) GetCertificateOperationPreparer(ctx context.Context, vaultBaseURL string, certificateName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"certificate-name": autorest.Encode("path", certificateName),
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/{certificate-name}/pending", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetCertificateOperationSender sends the GetCertificateOperation request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetCertificateOperationSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetCertificateOperationResponder handles the response to the GetCertificateOperation request. The method always
// closes the http.Response Body.
func (client BaseClient) GetCertificateOperationResponder(resp *http.Response) (result CertificateOperation, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetCertificatePolicy gets the policy for a certificate.
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
// certificateName - the name of the certificate in the given vault.
func (client BaseClient) GetCertificatePolicy(ctx context.Context, vaultBaseURL string, certificateName string) (result CertificatePolicy, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetCertificatePolicy")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.GetCertificatePolicyPreparer(ctx, vaultBaseURL, certificateName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificatePolicy", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetCertificatePolicySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificatePolicy", resp, "Failure sending request")
		return
	}

	result, err = client.GetCertificatePolicyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificatePolicy", resp, "Failure responding to request")
		return
	}

	return
}

// GetCertificatePolicyPreparer prepares the GetCertificatePolicy request.
func (client BaseClient) GetCertificatePolicyPreparer(ctx context.Context, vaultBaseURL string, certificateName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"certificate-name": autorest.Encode("path", certificateName),
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/{certificate-name}/policy", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetCertificatePolicySender sends the GetCertificatePolicy request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetCertificatePolicySender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetCertificatePolicyResponder handles the response to the GetCertificatePolicy request. The method always
// closes the http.Response Body.
func (client BaseClient) GetCertificatePolicyResponder(resp *http.Response) (result CertificatePolicy, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetCertificates list certificates in the specified vault
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
// maxresults - maximum number of results to return in a page. If not specified the service will return up to
// 25 results.
func (client BaseClient) GetCertificates(ctx context.Context, vaultBaseURL string, maxresults *int32) (result CertificateListResultPage, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetCertificates")
		defer func() {
			sc := -1
			if result.clr.Response.Response != nil {
				sc = result.clr.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: maxresults,
			Constraints: []validation.Constraint{{Target: "maxresults", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "maxresults", Name: validation.InclusiveMaximum, Rule: int64(25), Chain: nil},
					{Target: "maxresults", Name: validation.InclusiveMinimum, Rule: int64(1), Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "GetCertificates", err.Error())
	}

	result.fn = client.getCertificatesNextResults
	req, err := client.GetCertificatesPreparer(ctx, vaultBaseURL, maxresults)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificates", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetCertificatesSender(req)
	if err != nil {
		result.clr.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificates", resp, "Failure sending request")
		return
	}

	result.clr, err = client.GetCertificatesResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificates", resp, "Failure responding to request")
		return
	}
	if result.clr.hasNextLink() && result.clr.IsEmpty() {
		err = result.NextWithContext(ctx)
		return
	}

	return
}

// GetCertificatesPreparer prepares the GetCertificates request.
func (client BaseClient) GetCertificatesPreparer(ctx context.Context, vaultBaseURL string, maxresults *int32) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if maxresults != nil {
		queryParameters["maxresults"] = autorest.Encode("query", *maxresults)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPath("/certificates"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetCertificatesSender sends the GetCertificates request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetCertificatesSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetCertificatesResponder handles the response to the GetCertificates request. The method always
// closes the http.Response Body.
func (client BaseClient) GetCertificatesResponder(resp *http.Response) (result CertificateListResult, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// getCertificatesNextResults retrieves the next set of results, if any.
func (client BaseClient) getCertificatesNextResults(ctx context.Context, lastResults CertificateListResult) (result CertificateListResult, err error) {
	req, err := lastResults.certificateListResultPreparer(ctx)
	if err != nil {
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getCertificatesNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.GetCertificatesSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getCertificatesNextResults", resp, "Failure sending next results request")
	}
	result, err = client.GetCertificatesResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "getCertificatesNextResults", resp, "Failure responding to next results request")
	}
	return
}

// GetCertificatesComplete enumerates all values, automatically crossing page boundaries as required.
func (client BaseClient) GetCertificatesComplete(ctx context.Context, vaultBaseURL string, maxresults *int32) (result CertificateListResultIterator, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetCertificates")
		defer func() {
			sc := -1
			if result.Response().Response.Response != nil {
				sc = result.page.Response().Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	result.page, err = client.GetCertificates(ctx, vaultBaseURL, maxresults)
	return
}

// GetCertificateVersions list the versions of a certificate.
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
// certificateName - the name of the certificate
// maxresults - maximum number of results to return in a page. If not specified the service will return up to
// 25 results.
func (client BaseClient) GetCertificateVersions(ctx context.Context, vaultBaseURL string, certificateName string, maxresults *int32) (result CertificateListResultPage, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetCertificateVersions")
		defer func() {
			sc := -1
			if result.clr.Response.Response != nil {
				sc = result.clr.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: maxresults,
			Constraints: []validation.Constraint{{Target: "maxresults", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "maxresults", Name: validation.InclusiveMaximum, Rule: int64(25), Chain: nil},
					{Target: "maxresults", Name: validation.InclusiveMinimum, Rule: int64(1), Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "GetCertificateVersions", err.Error())
	}

	result.fn = client.getCertificateVersionsNextResults
	req, err := client.GetCertificateVersionsPreparer(ctx, vaultBaseURL, certificateName, maxresults)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificateVersions", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetCertificateVersionsSender(req)
	if err != nil {
		result.clr.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificateVersions", resp, "Failure sending request")
		return
	}

	result.clr, err = client.GetCertificateVersionsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetCertificateVersions", resp, "Failure responding to request")
		return
	}
	if result.clr.hasNextLink() && result.clr.IsEmpty() {
		err = result.NextWithContext(ctx)
		return
	}

	return
}

// GetCertificateVersionsPreparer prepares the GetCertificateVersions request.
func (client BaseClient) GetCertificateVersionsPreparer(ctx context.Context, vaultBaseURL string, certificateName string, maxresults *int32) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"certificate-name": autorest.Encode("path", certificateName),
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if maxresults != nil {
		queryParameters["maxresults"] = autorest.Encode("query", *maxresults)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/{certificate-name}/versions", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetCertificateVersionsSender sends the GetCertificateVersions request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetCertificateVersionsSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetCertificateVersionsResponder handles the response to the GetCertificateVersions request. The method always
// closes the http.Response Body.
func (client BaseClient) GetCertificateVersionsResponder(resp *http.Response) (result CertificateListResult, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// getCertificateVersionsNextResults retrieves the next set of results, if any.
func (client BaseClient) getCertificateVersionsNextResults(ctx context.Context, lastResults CertificateListResult) (result CertificateListResult, err error) {
	req, err := lastResults.certificateListResultPreparer(ctx)
	if err != nil {
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getCertificateVersionsNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.GetCertificateVersionsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getCertificateVersionsNextResults", resp, "Failure sending next results request")
	}
	result, err = client.GetCertificateVersionsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "getCertificateVersionsNextResults", resp, "Failure responding to next results request")
	}
	return
}

// GetCertificateVersionsComplete enumerates all values, automatically crossing page boundaries as required.
func (client BaseClient) GetCertificateVersionsComplete(ctx context.Context, vaultBaseURL string, certificateName string, maxresults *int32) (result CertificateListResultIterator, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetCertificateVersions")
		defer func() {
			sc := -1
			if result.Response().Response.Response != nil {
				sc = result.page.Response().Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	result.page, err = client.GetCertificateVersions(ctx, vaultBaseURL, certificateName, maxresults)
	return
}

// GetKey retrieves the public portion of a key plus its attributes
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
// keyName - the name of the key
// keyVersion - the version of the key
func (client BaseClient) GetKey(ctx context.Context, vaultBaseURL string, keyName string, keyVersion string) (result KeyBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetKey")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.GetKeyPreparer(ctx, vaultBaseURL, keyName, keyVersion)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetKey", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetKeySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetKey", resp, "Failure sending request")
		return
	}

	result, err = client.GetKeyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetKey", resp, "Failure responding to request")
		return
	}

	return
}

// GetKeyPreparer prepares the GetKey request.
func (client BaseClient) GetKeyPreparer(ctx context.Context, vaultBaseURL string, keyName string, keyVersion string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name":    autorest.Encode("path", keyName),
		"key-version": autorest.Encode("path", keyVersion),
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/keys/{key-name}/{key-version}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetKeySender sends the GetKey request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetKeySender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetKeyResponder handles the response to the GetKey request. The method always
// closes the http.Response Body.
func (client BaseClient) GetKeyResponder(resp *http.Response) (result KeyBundle, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetKeys list keys in the specified vault
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
// maxresults - maximum number of results to return in a page. If not specified the service will return up to
// 25 results.
func (client BaseClient) GetKeys(ctx context.Context, vaultBaseURL string, maxresults *int32) (result KeyListResultPage, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetKeys")
		defer func() {
			sc := -1
			if result.klr.Response.Response != nil {
				sc = result.klr.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: maxresults,
			Constraints: []validation.Constraint{{Target: "maxresults", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "maxresults", Name: validation.InclusiveMaximum, Rule: int64(25), Chain: nil},
					{Target: "maxresults", Name: validation.InclusiveMinimum, Rule: int64(1), Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "GetKeys", err.Error())
	}

	result.fn = client.getKeysNextResults
	req, err := client.GetKeysPreparer(ctx, vaultBaseURL, maxresults)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetKeys", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetKeysSender(req)
	if err != nil {
		result.klr.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetKeys", resp, "Failure sending request")
		return
	}

	result.klr, err = client.GetKeysResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetKeys", resp, "Failure responding to request")
		return
	}
	if result.klr.hasNextLink() && result.klr.IsEmpty() {
		err = result.NextWithContext(ctx)
		return
	}

	return
}

// GetKeysPreparer prepares the GetKeys request.
func (client BaseClient) GetKeysPreparer(ctx context.Context, vaultBaseURL string, maxresults *int32) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if maxresults != nil {
		queryParameters["maxresults"] = autorest.Encode("query", *maxresults)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPath("/keys"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetKeysSender sends the GetKeys request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetKeysSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetKeysResponder handles the response to the GetKeys request. The method always
// closes the http.Response Body.
func (client BaseClient) GetKeysResponder(resp *http.Response) (result KeyListResult, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// getKeysNextResults retrieves the next set of results, if any.
func (client BaseClient) getKeysNextResults(ctx context.Context, lastResults KeyListResult) (result KeyListResult, err error) {
	req, err := lastResults.keyListResultPreparer(ctx)
	if err != nil {
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getKeysNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.GetKeysSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getKeysNextResults", resp, "Failure sending next results request")
	}
	result, err = client.GetKeysResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "getKeysNextResults", resp, "Failure responding to next results request")
	}
	return
}

// GetKeysComplete enumerates all values, automatically crossing page boundaries as required.
func (client BaseClient) GetKeysComplete(ctx context.Context, vaultBaseURL string, maxresults *int32) (result KeyListResultIterator, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetKeys")
		defer func() {
			sc := -1
			if result.Response().Response.Response != nil {
				sc = result.page.Response().Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	result.page, err = client.GetKeys(ctx, vaultBaseURL, maxresults)
	return
}

// GetKeyVersions list the versions of the specified key
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
// keyName - the name of the key
// maxresults - maximum number of results to return in a page. If not specified the service will return up to
// 25 results.
func (client BaseClient) GetKeyVersions(ctx context.Context, vaultBaseURL string, keyName string, maxresults *int32) (result KeyListResultPage, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetKeyVersions")
		defer func() {
			sc := -1
			if result.klr.Response.Response != nil {
				sc = result.klr.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: maxresults,
			Constraints: []validation.Constraint{{Target: "maxresults", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "maxresults", Name: validation.InclusiveMaximum, Rule: int64(25), Chain: nil},
					{Target: "maxresults", Name: validation.InclusiveMinimum, Rule: int64(1), Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "GetKeyVersions", err.Error())
	}

	result.fn = client.getKeyVersionsNextResults
	req, err := client.GetKeyVersionsPreparer(ctx, vaultBaseURL, keyName, maxresults)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetKeyVersions", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetKeyVersionsSender(req)
	if err != nil {
		result.klr.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetKeyVersions", resp, "Failure sending request")
		return
	}

	result.klr, err = client.GetKeyVersionsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetKeyVersions", resp, "Failure responding to request")
		return
	}
	if result.klr.hasNextLink() && result.klr.IsEmpty() {
		err = result.NextWithContext(ctx)
		return
	}

	return
}

// GetKeyVersionsPreparer prepares the GetKeyVersions request.
func (client BaseClient) GetKeyVersionsPreparer(ctx context.Context, vaultBaseURL string, keyName string, maxresults *int32) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name": autorest.Encode("path", keyName),
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if maxresults != nil {
		queryParameters["maxresults"] = autorest.Encode("query", *maxresults)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/keys/{key-name}/versions", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetKeyVersionsSender sends the GetKeyVersions request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetKeyVersionsSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetKeyVersionsResponder handles the response to the GetKeyVersions request. The method always
// closes the http.Response Body.
func (client BaseClient) GetKeyVersionsResponder(resp *http.Response) (result KeyListResult, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// getKeyVersionsNextResults retrieves the next set of results, if any.
func (client BaseClient) getKeyVersionsNextResults(ctx context.Context, lastResults KeyListResult) (result KeyListResult, err error) {
	req, err := lastResults.keyListResultPreparer(ctx)
	if err != nil {
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getKeyVersionsNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.GetKeyVersionsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getKeyVersionsNextResults", resp, "Failure sending next results request")
	}
	result, err = client.GetKeyVersionsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "getKeyVersionsNextResults", resp, "Failure responding to next results request")
	}
	return
}

// GetKeyVersionsComplete enumerates all values, automatically crossing page boundaries as required.
func (client BaseClient) GetKeyVersionsComplete(ctx context.Context, vaultBaseURL string, keyName string, maxresults *int32) (result KeyListResultIterator, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetKeyVersions")
		defer func() {
			sc := -1
			if result.Response().Response.Response != nil {
				sc = result.page.Response().Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	result.page, err = client.GetKeyVersions(ctx, vaultBaseURL, keyName, maxresults)
	return
}

// GetSecret gets a secret.
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
// secretName - the name of the secret in the given vault
// secretVersion - the version of the secret
func (client BaseClient) GetSecret(ctx context.Context, vaultBaseURL string, secretName string, secretVersion string) (result SecretBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetSecret")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.GetSecretPreparer(ctx, vaultBaseURL, secretName, secretVersion)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetSecret", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetSecretSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetSecret", resp, "Failure sending request")
		return
	}

	result, err = client.GetSecretResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetSecret", resp, "Failure responding to request")
		return
	}

	return
}

// GetSecretPreparer prepares the GetSecret request.
func (client BaseClient) GetSecretPreparer(ctx context.Context, vaultBaseURL string, secretName string, secretVersion string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"secret-name":    autorest.Encode("path", secretName),
		"secret-version": autorest.Encode("path", secretVersion),
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/secrets/{secret-name}/{secret-version}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetSecretSender sends the GetSecret request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetSecretSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetSecretResponder handles the response to the GetSecret request. The method always
// closes the http.Response Body.
func (client BaseClient) GetSecretResponder(resp *http.Response) (result SecretBundle, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetSecrets list secrets in the specified vault
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
// maxresults - maximum number of results to return in a page. If not specified the service will return up to
// 25 results.
func (client BaseClient) GetSecrets(ctx context.Context, vaultBaseURL string, maxresults *int32) (result SecretListResultPage, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetSecrets")
		defer func() {
			sc := -1
			if result.slr.Response.Response != nil {
				sc = result.slr.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: maxresults,
			Constraints: []validation.Constraint{{Target: "maxresults", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "maxresults", Name: validation.InclusiveMaximum, Rule: int64(25), Chain: nil},
					{Target: "maxresults", Name: validation.InclusiveMinimum, Rule: int64(1), Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "GetSecrets", err.Error())
	}

	result.fn = client.getSecretsNextResults
	req, err := client.GetSecretsPreparer(ctx, vaultBaseURL, maxresults)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetSecrets", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetSecretsSender(req)
	if err != nil {
		result.slr.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetSecrets", resp, "Failure sending request")
		return
	}

	result.slr, err = client.GetSecretsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetSecrets", resp, "Failure responding to request")
		return
	}
	if result.slr.hasNextLink() && result.slr.IsEmpty() {
		err = result.NextWithContext(ctx)
		return
	}

	return
}

// GetSecretsPreparer prepares the GetSecrets request.
func (client BaseClient) GetSecretsPreparer(ctx context.Context, vaultBaseURL string, maxresults *int32) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if maxresults != nil {
		queryParameters["maxresults"] = autorest.Encode("query", *maxresults)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPath("/secrets"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetSecretsSender sends the GetSecrets request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetSecretsSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetSecretsResponder handles the response to the GetSecrets request. The method always
// closes the http.Response Body.
func (client BaseClient) GetSecretsResponder(resp *http.Response) (result SecretListResult, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// getSecretsNextResults retrieves the next set of results, if any.
func (client BaseClient) getSecretsNextResults(ctx context.Context, lastResults SecretListResult) (result SecretListResult, err error) {
	req, err := lastResults.secretListResultPreparer(ctx)
	if err != nil {
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getSecretsNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.GetSecretsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getSecretsNextResults", resp, "Failure sending next results request")
	}
	result, err = client.GetSecretsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "getSecretsNextResults", resp, "Failure responding to next results request")
	}
	return
}

// GetSecretsComplete enumerates all values, automatically crossing page boundaries as required.
func (client BaseClient) GetSecretsComplete(ctx context.Context, vaultBaseURL string, maxresults *int32) (result SecretListResultIterator, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetSecrets")
		defer func() {
			sc := -1
			if result.Response().Response.Response != nil {
				sc = result.page.Response().Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	result.page, err = client.GetSecrets(ctx, vaultBaseURL, maxresults)
	return
}

// GetSecretVersions list the versions of the specified secret
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
// secretName - the name of the secret in the given vault
// maxresults - maximum number of results to return in a page. If not specified the service will return up to
// 25 results.
func (client BaseClient) GetSecretVersions(ctx context.Context, vaultBaseURL string, secretName string, maxresults *int32) (result SecretListResultPage, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetSecretVersions")
		defer func() {
			sc := -1
			if result.slr.Response.Response != nil {
				sc = result.slr.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: maxresults,
			Constraints: []validation.Constraint{{Target: "maxresults", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "maxresults", Name: validation.InclusiveMaximum, Rule: int64(25), Chain: nil},
					{Target: "maxresults", Name: validation.InclusiveMinimum, Rule: int64(1), Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "GetSecretVersions", err.Error())
	}

	result.fn = client.getSecretVersionsNextResults
	req, err := client.GetSecretVersionsPreparer(ctx, vaultBaseURL, secretName, maxresults)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetSecretVersions", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetSecretVersionsSender(req)
	if err != nil {
		result.slr.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetSecretVersions", resp, "Failure sending request")
		return
	}

	result.slr, err = client.GetSecretVersionsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "GetSecretVersions", resp, "Failure responding to request")
		return
	}
	if result.slr.hasNextLink() && result.slr.IsEmpty() {
		err = result.NextWithContext(ctx)
		return
	}

	return
}

// GetSecretVersionsPreparer prepares the GetSecretVersions request.
func (client BaseClient) GetSecretVersionsPreparer(ctx context.Context, vaultBaseURL string, secretName string, maxresults *int32) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"secret-name": autorest.Encode("path", secretName),
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if maxresults != nil {
		queryParameters["maxresults"] = autorest.Encode("query", *maxresults)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/secrets/{secret-name}/versions", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetSecretVersionsSender sends the GetSecretVersions request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) GetSecretVersionsSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetSecretVersionsResponder handles the response to the GetSecretVersions request. The method always
// closes the http.Response Body.
func (client BaseClient) GetSecretVersionsResponder(resp *http.Response) (result SecretListResult, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// getSecretVersionsNextResults retrieves the next set of results, if any.
func (client BaseClient) getSecretVersionsNextResults(ctx context.Context, lastResults SecretListResult) (result SecretListResult, err error) {
	req, err := lastResults.secretListResultPreparer(ctx)
	if err != nil {
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getSecretVersionsNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.GetSecretVersionsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "keyvault.BaseClient", "getSecretVersionsNextResults", resp, "Failure sending next results request")
	}
	result, err = client.GetSecretVersionsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "getSecretVersionsNextResults", resp, "Failure responding to next results request")
	}
	return
}

// GetSecretVersionsComplete enumerates all values, automatically crossing page boundaries as required.
func (client BaseClient) GetSecretVersionsComplete(ctx context.Context, vaultBaseURL string, secretName string, maxresults *int32) (result SecretListResultIterator, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.GetSecretVersions")
		defer func() {
			sc := -1
			if result.Response().Response.Response != nil {
				sc = result.page.Response().Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	result.page, err = client.GetSecretVersions(ctx, vaultBaseURL, secretName, maxresults)
	return
}

// ImportCertificate imports a certificate into the specified vault
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
// certificateName - the name of the certificate
// parameters - the parameters to import certificate.
func (client BaseClient) ImportCertificate(ctx context.Context, vaultBaseURL string, certificateName string, parameters CertificateImportParameters) (result CertificateBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.ImportCertificate")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: certificateName,
			Constraints: []validation.Constraint{{Target: "certificateName", Name: validation.Pattern, Rule: `^[0-9a-zA-Z-]+$`, Chain: nil}}},
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.Base64EncodedCertificate", Name: validation.Null, Rule: true, Chain: nil},
				{Target: "parameters.CertificatePolicy", Name: validation.Null, Rule: false,
					Chain: []validation.Constraint{{Target: "parameters.CertificatePolicy.X509CertificateProperties", Name: validation.Null, Rule: false,
						Chain: []validation.Constraint{{Target: "parameters.CertificatePolicy.X509CertificateProperties.ValidityInMonths", Name: validation.Null, Rule: false,
							Chain: []validation.Constraint{{Target: "parameters.CertificatePolicy.X509CertificateProperties.ValidityInMonths", Name: validation.InclusiveMinimum, Rule: int64(0), Chain: nil}}},
						}},
					}}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "ImportCertificate", err.Error())
	}

	req, err := client.ImportCertificatePreparer(ctx, vaultBaseURL, certificateName, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "ImportCertificate", nil, "Failure preparing request")
		return
	}

	resp, err := client.ImportCertificateSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "ImportCertificate", resp, "Failure sending request")
		return
	}

	result, err = client.ImportCertificateResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "ImportCertificate", resp, "Failure responding to request")
		return
	}

	return
}

// ImportCertificatePreparer prepares the ImportCertificate request.
func (client BaseClient) ImportCertificatePreparer(ctx context.Context, vaultBaseURL string, certificateName string, parameters CertificateImportParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"certificate-name": autorest.Encode("path", certificateName),
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/{certificate-name}/import", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ImportCertificateSender sends the ImportCertificate request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) ImportCertificateSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ImportCertificateResponder handles the response to the ImportCertificate request. The method always
// closes the http.Response Body.
func (client BaseClient) ImportCertificateResponder(resp *http.Response) (result CertificateBundle, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// ImportKey imports a key into the specified vault
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
// keyName - the name of the key
// parameters - the parameters to import key.
func (client BaseClient) ImportKey(ctx context.Context, vaultBaseURL string, keyName string, parameters KeyImportParameters) (result KeyBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.ImportKey")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: keyName,
			Constraints: []validation.Constraint{{Target: "keyName", Name: validation.Pattern, Rule: `^[0-9a-zA-Z-]+$`, Chain: nil}}},
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.Key", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "ImportKey", err.Error())
	}

	req, err := client.ImportKeyPreparer(ctx, vaultBaseURL, keyName, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "ImportKey", nil, "Failure preparing request")
		return
	}

	resp, err := client.ImportKeySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "ImportKey", resp, "Failure sending request")
		return
	}

	result, err = client.ImportKeyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "ImportKey", resp, "Failure responding to request")
		return
	}

	return
}

// ImportKeyPreparer prepares the ImportKey request.
func (client BaseClient) ImportKeyPreparer(ctx context.Context, vaultBaseURL string, keyName string, parameters KeyImportParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name": autorest.Encode("path", keyName),
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPut(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/keys/{key-name}", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ImportKeySender sends the ImportKey request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) ImportKeySender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ImportKeyResponder handles the response to the ImportKey request. The method always
// closes the http.Response Body.
func (client BaseClient) ImportKeyResponder(resp *http.Response) (result KeyBundle, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// MergeCertificate merges a certificate or a certificate chain with a key pair existing on the server.
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
// certificateName - the name of the certificate
// parameters - the parameters to merge certificate.
func (client BaseClient) MergeCertificate(ctx context.Context, vaultBaseURL string, certificateName string, parameters CertificateMergeParameters) (result CertificateBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.MergeCertificate")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.X509Certificates", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "MergeCertificate", err.Error())
	}

	req, err := client.MergeCertificatePreparer(ctx, vaultBaseURL, certificateName, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "MergeCertificate", nil, "Failure preparing request")
		return
	}

	resp, err := client.MergeCertificateSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "MergeCertificate", resp, "Failure sending request")
		return
	}

	result, err = client.MergeCertificateResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "MergeCertificate", resp, "Failure responding to request")
		return
	}

	return
}

// MergeCertificatePreparer prepares the MergeCertificate request.
func (client BaseClient) MergeCertificatePreparer(ctx context.Context, vaultBaseURL string, certificateName string, parameters CertificateMergeParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"certificate-name": autorest.Encode("path", certificateName),
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/{certificate-name}/pending/merge", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// MergeCertificateSender sends the MergeCertificate request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) MergeCertificateSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// MergeCertificateResponder handles the response to the MergeCertificate request. The method always
// closes the http.Response Body.
func (client BaseClient) MergeCertificateResponder(resp *http.Response) (result CertificateBundle, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusCreated),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// RestoreKey restores the backup key in to a vault
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
// parameters - the parameters to restore key
func (client BaseClient) RestoreKey(ctx context.Context, vaultBaseURL string, parameters KeyRestoreParameters) (result KeyBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.RestoreKey")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.KeyBundleBackup", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "RestoreKey", err.Error())
	}

	req, err := client.RestoreKeyPreparer(ctx, vaultBaseURL, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "RestoreKey", nil, "Failure preparing request")
		return
	}

	resp, err := client.RestoreKeySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "RestoreKey", resp, "Failure sending request")
		return
	}

	result, err = client.RestoreKeyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "RestoreKey", resp, "Failure responding to request")
		return
	}

	return
}

// RestoreKeyPreparer prepares the RestoreKey request.
func (client BaseClient) RestoreKeyPreparer(ctx context.Context, vaultBaseURL string, parameters KeyRestoreParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPath("/keys/restore"),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// RestoreKeySender sends the RestoreKey request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) RestoreKeySender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// RestoreKeyResponder handles the response to the RestoreKey request. The method always
// closes the http.Response Body.
func (client BaseClient) RestoreKeyResponder(resp *http.Response) (result KeyBundle, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// SetCertificateContacts sets the certificate contacts for the specified vault.
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
// contacts - the contacts for the vault certificates.
func (client BaseClient) SetCertificateContacts(ctx context.Context, vaultBaseURL string, contacts Contacts) (result Contacts, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.SetCertificateContacts")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.SetCertificateContactsPreparer(ctx, vaultBaseURL, contacts)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "SetCertificateContacts", nil, "Failure preparing request")
		return
	}

	resp, err := client.SetCertificateContactsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "SetCertificateContacts", resp, "Failure sending request")
		return
	}

	result, err = client.SetCertificateContactsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "SetCertificateContacts", resp, "Failure responding to request")
		return
	}

	return
}

// SetCertificateContactsPreparer prepares the SetCertificateContacts request.
func (client BaseClient) SetCertificateContactsPreparer(ctx context.Context, vaultBaseURL string, contacts Contacts) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	contacts.ID = nil
	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPut(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPath("/certificates/contacts"),
		autorest.WithJSON(contacts),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// SetCertificateContactsSender sends the SetCertificateContacts request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) SetCertificateContactsSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// SetCertificateContactsResponder handles the response to the SetCertificateContacts request. The method always
// closes the http.Response Body.
func (client BaseClient) SetCertificateContactsResponder(resp *http.Response) (result Contacts, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// SetCertificateIssuer sets the specified certificate issuer.
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
// issuerName - the name of the issuer.
// parameter - certificate issuer set parameter.
func (client BaseClient) SetCertificateIssuer(ctx context.Context, vaultBaseURL string, issuerName string, parameter CertificateIssuerSetParameters) (result IssuerBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.SetCertificateIssuer")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: parameter,
			Constraints: []validation.Constraint{{Target: "parameter.Provider", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "SetCertificateIssuer", err.Error())
	}

	req, err := client.SetCertificateIssuerPreparer(ctx, vaultBaseURL, issuerName, parameter)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "SetCertificateIssuer", nil, "Failure preparing request")
		return
	}

	resp, err := client.SetCertificateIssuerSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "SetCertificateIssuer", resp, "Failure sending request")
		return
	}

	result, err = client.SetCertificateIssuerResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "SetCertificateIssuer", resp, "Failure responding to request")
		return
	}

	return
}

// SetCertificateIssuerPreparer prepares the SetCertificateIssuer request.
func (client BaseClient) SetCertificateIssuerPreparer(ctx context.Context, vaultBaseURL string, issuerName string, parameter CertificateIssuerSetParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"issuer-name": autorest.Encode("path", issuerName),
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPut(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/issuers/{issuer-name}", pathParameters),
		autorest.WithJSON(parameter),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// SetCertificateIssuerSender sends the SetCertificateIssuer request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) SetCertificateIssuerSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// SetCertificateIssuerResponder handles the response to the SetCertificateIssuer request. The method always
// closes the http.Response Body.
func (client BaseClient) SetCertificateIssuerResponder(resp *http.Response) (result IssuerBundle, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// SetSecret sets a secret in the specified vault.
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
// secretName - the name of the secret in the given vault
// parameters - the parameters for secret set
func (client BaseClient) SetSecret(ctx context.Context, vaultBaseURL string, secretName string, parameters SecretSetParameters) (result SecretBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.SetSecret")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: secretName,
			Constraints: []validation.Constraint{{Target: "secretName", Name: validation.Pattern, Rule: `^[0-9a-zA-Z-]+$`, Chain: nil}}},
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.Value", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "SetSecret", err.Error())
	}

	req, err := client.SetSecretPreparer(ctx, vaultBaseURL, secretName, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "SetSecret", nil, "Failure preparing request")
		return
	}

	resp, err := client.SetSecretSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "SetSecret", resp, "Failure sending request")
		return
	}

	result, err = client.SetSecretResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "SetSecret", resp, "Failure responding to request")
		return
	}

	return
}

// SetSecretPreparer prepares the SetSecret request.
func (client BaseClient) SetSecretPreparer(ctx context.Context, vaultBaseURL string, secretName string, parameters SecretSetParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"secret-name": autorest.Encode("path", secretName),
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPut(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/secrets/{secret-name}", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// SetSecretSender sends the SetSecret request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) SetSecretSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// SetSecretResponder handles the response to the SetSecret request. The method always
// closes the http.Response Body.
func (client BaseClient) SetSecretResponder(resp *http.Response) (result SecretBundle, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// Sign creates a signature from a digest using the specified key in the vault
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
// keyName - the name of the key
// keyVersion - the version of the key
// parameters - the parameters for key operations.
func (client BaseClient) Sign(ctx context.Context, vaultBaseURL string, keyName string, keyVersion string, parameters KeySignParameters) (result KeyOperationResult, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.Sign")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.Value", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "Sign", err.Error())
	}

	req, err := client.SignPreparer(ctx, vaultBaseURL, keyName, keyVersion, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "Sign", nil, "Failure preparing request")
		return
	}

	resp, err := client.SignSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "Sign", resp, "Failure sending request")
		return
	}

	result, err = client.SignResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "Sign", resp, "Failure responding to request")
		return
	}

	return
}

// SignPreparer prepares the Sign request.
func (client BaseClient) SignPreparer(ctx context.Context, vaultBaseURL string, keyName string, keyVersion string, parameters KeySignParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name":    autorest.Encode("path", keyName),
		"key-version": autorest.Encode("path", keyVersion),
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/keys/{key-name}/{key-version}/sign", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// SignSender sends the Sign request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) SignSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// SignResponder handles the response to the Sign request. The method always
// closes the http.Response Body.
func (client BaseClient) SignResponder(resp *http.Response) (result KeyOperationResult, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// UnwrapKey unwraps a symmetric key using the specified key in the vault that has initially been used for wrapping the
// key.
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
// keyName - the name of the key
// keyVersion - the version of the key
// parameters - the parameters for key operations.
func (client BaseClient) UnwrapKey(ctx context.Context, vaultBaseURL string, keyName string, keyVersion string, parameters KeyOperationsParameters) (result KeyOperationResult, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.UnwrapKey")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.Value", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "UnwrapKey", err.Error())
	}

	req, err := client.UnwrapKeyPreparer(ctx, vaultBaseURL, keyName, keyVersion, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UnwrapKey", nil, "Failure preparing request")
		return
	}

	resp, err := client.UnwrapKeySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UnwrapKey", resp, "Failure sending request")
		return
	}

	result, err = client.UnwrapKeyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UnwrapKey", resp, "Failure responding to request")
		return
	}

	return
}

// UnwrapKeyPreparer prepares the UnwrapKey request.
func (client BaseClient) UnwrapKeyPreparer(ctx context.Context, vaultBaseURL string, keyName string, keyVersion string, parameters KeyOperationsParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name":    autorest.Encode("path", keyName),
		"key-version": autorest.Encode("path", keyVersion),
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/keys/{key-name}/{key-version}/unwrapkey", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// UnwrapKeySender sends the UnwrapKey request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) UnwrapKeySender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// UnwrapKeyResponder handles the response to the UnwrapKey request. The method always
// closes the http.Response Body.
func (client BaseClient) UnwrapKeyResponder(resp *http.Response) (result KeyOperationResult, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// UpdateCertificate updates the attributes associated with the specified certificate
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
// certificateName - the name of the certificate in the given vault
// certificateVersion - the version of the certificate
func (client BaseClient) UpdateCertificate(ctx context.Context, vaultBaseURL string, certificateName string, certificateVersion string, parameters CertificateUpdateParameters) (result CertificateBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.UpdateCertificate")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.UpdateCertificatePreparer(ctx, vaultBaseURL, certificateName, certificateVersion, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateCertificate", nil, "Failure preparing request")
		return
	}

	resp, err := client.UpdateCertificateSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateCertificate", resp, "Failure sending request")
		return
	}

	result, err = client.UpdateCertificateResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateCertificate", resp, "Failure responding to request")
		return
	}

	return
}

// UpdateCertificatePreparer prepares the UpdateCertificate request.
func (client BaseClient) UpdateCertificatePreparer(ctx context.Context, vaultBaseURL string, certificateName string, certificateVersion string, parameters CertificateUpdateParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"certificate-name":    autorest.Encode("path", certificateName),
		"certificate-version": autorest.Encode("path", certificateVersion),
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPatch(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/{certificate-name}/{certificate-version}", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// UpdateCertificateSender sends the UpdateCertificate request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) UpdateCertificateSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// UpdateCertificateResponder handles the response to the UpdateCertificate request. The method always
// closes the http.Response Body.
func (client BaseClient) UpdateCertificateResponder(resp *http.Response) (result CertificateBundle, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// UpdateCertificateIssuer updates the specified certificate issuer.
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
// issuerName - the name of the issuer.
// parameter - certificate issuer update parameter.
func (client BaseClient) UpdateCertificateIssuer(ctx context.Context, vaultBaseURL string, issuerName string, parameter CertificateIssuerUpdateParameters) (result IssuerBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.UpdateCertificateIssuer")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.UpdateCertificateIssuerPreparer(ctx, vaultBaseURL, issuerName, parameter)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateCertificateIssuer", nil, "Failure preparing request")
		return
	}

	resp, err := client.UpdateCertificateIssuerSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateCertificateIssuer", resp, "Failure sending request")
		return
	}

	result, err = client.UpdateCertificateIssuerResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateCertificateIssuer", resp, "Failure responding to request")
		return
	}

	return
}

// UpdateCertificateIssuerPreparer prepares the UpdateCertificateIssuer request.
func (client BaseClient) UpdateCertificateIssuerPreparer(ctx context.Context, vaultBaseURL string, issuerName string, parameter CertificateIssuerUpdateParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"issuer-name": autorest.Encode("path", issuerName),
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPatch(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/issuers/{issuer-name}", pathParameters),
		autorest.WithJSON(parameter),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// UpdateCertificateIssuerSender sends the UpdateCertificateIssuer request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) UpdateCertificateIssuerSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// UpdateCertificateIssuerResponder handles the response to the UpdateCertificateIssuer request. The method always
// closes the http.Response Body.
func (client BaseClient) UpdateCertificateIssuerResponder(resp *http.Response) (result IssuerBundle, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// UpdateCertificateOperation updates a certificate operation.
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
// certificateName - the name of the certificate
// certificateOperation - the certificate operation response.
func (client BaseClient) UpdateCertificateOperation(ctx context.Context, vaultBaseURL string, certificateName string, certificateOperation CertificateOperationUpdateParameter) (result CertificateOperation, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.UpdateCertificateOperation")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.UpdateCertificateOperationPreparer(ctx, vaultBaseURL, certificateName, certificateOperation)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateCertificateOperation", nil, "Failure preparing request")
		return
	}

	resp, err := client.UpdateCertificateOperationSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateCertificateOperation", resp, "Failure sending request")
		return
	}

	result, err = client.UpdateCertificateOperationResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateCertificateOperation", resp, "Failure responding to request")
		return
	}

	return
}

// UpdateCertificateOperationPreparer prepares the UpdateCertificateOperation request.
func (client BaseClient) UpdateCertificateOperationPreparer(ctx context.Context, vaultBaseURL string, certificateName string, certificateOperation CertificateOperationUpdateParameter) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"certificate-name": autorest.Encode("path", certificateName),
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPatch(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/{certificate-name}/pending", pathParameters),
		autorest.WithJSON(certificateOperation),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// UpdateCertificateOperationSender sends the UpdateCertificateOperation request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) UpdateCertificateOperationSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// UpdateCertificateOperationResponder handles the response to the UpdateCertificateOperation request. The method always
// closes the http.Response Body.
func (client BaseClient) UpdateCertificateOperationResponder(resp *http.Response) (result CertificateOperation, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// UpdateCertificatePolicy updates the policy for a certificate. Set appropriate members in the certificatePolicy that
// must be updated. Leave others as null.
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
// certificateName - the name of the certificate in the given vault.
// certificatePolicy - the policy for the certificate.
func (client BaseClient) UpdateCertificatePolicy(ctx context.Context, vaultBaseURL string, certificateName string, certificatePolicy CertificatePolicy) (result CertificatePolicy, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.UpdateCertificatePolicy")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.UpdateCertificatePolicyPreparer(ctx, vaultBaseURL, certificateName, certificatePolicy)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateCertificatePolicy", nil, "Failure preparing request")
		return
	}

	resp, err := client.UpdateCertificatePolicySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateCertificatePolicy", resp, "Failure sending request")
		return
	}

	result, err = client.UpdateCertificatePolicyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateCertificatePolicy", resp, "Failure responding to request")
		return
	}

	return
}

// UpdateCertificatePolicyPreparer prepares the UpdateCertificatePolicy request.
func (client BaseClient) UpdateCertificatePolicyPreparer(ctx context.Context, vaultBaseURL string, certificateName string, certificatePolicy CertificatePolicy) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"certificate-name": autorest.Encode("path", certificateName),
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	certificatePolicy.ID = nil
	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPatch(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/{certificate-name}/policy", pathParameters),
		autorest.WithJSON(certificatePolicy),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// UpdateCertificatePolicySender sends the UpdateCertificatePolicy request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) UpdateCertificatePolicySender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// UpdateCertificatePolicyResponder handles the response to the UpdateCertificatePolicy request. The method always
// closes the http.Response Body.
func (client BaseClient) UpdateCertificatePolicyResponder(resp *http.Response) (result CertificatePolicy, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// UpdateKey updates the Key Attributes associated with the specified key
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
// keyName - the name of the key
// keyVersion - the version of the key
// parameters - the parameters to update key.
func (client BaseClient) UpdateKey(ctx context.Context, vaultBaseURL string, keyName string, keyVersion string, parameters KeyUpdateParameters) (result KeyBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.UpdateKey")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.UpdateKeyPreparer(ctx, vaultBaseURL, keyName, keyVersion, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateKey", nil, "Failure preparing request")
		return
	}

	resp, err := client.UpdateKeySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateKey", resp, "Failure sending request")
		return
	}

	result, err = client.UpdateKeyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateKey", resp, "Failure responding to request")
		return
	}

	return
}

// UpdateKeyPreparer prepares the UpdateKey request.
func (client BaseClient) UpdateKeyPreparer(ctx context.Context, vaultBaseURL string, keyName string, keyVersion string, parameters KeyUpdateParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name":    autorest.Encode("path", keyName),
		"key-version": autorest.Encode("path", keyVersion),
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPatch(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/keys/{key-name}/{key-version}", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// UpdateKeySender sends the UpdateKey request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) UpdateKeySender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// UpdateKeyResponder handles the response to the UpdateKey request. The method always
// closes the http.Response Body.
func (client BaseClient) UpdateKeyResponder(resp *http.Response) (result KeyBundle, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// UpdateSecret updates the attributes associated with the specified secret
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
// secretName - the name of the secret in the given vault
// secretVersion - the version of the secret
func (client BaseClient) UpdateSecret(ctx context.Context, vaultBaseURL string, secretName string, secretVersion string, parameters SecretUpdateParameters) (result SecretBundle, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.UpdateSecret")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	req, err := client.UpdateSecretPreparer(ctx, vaultBaseURL, secretName, secretVersion, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateSecret", nil, "Failure preparing request")
		return
	}

	resp, err := client.UpdateSecretSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateSecret", resp, "Failure sending request")
		return
	}

	result, err = client.UpdateSecretResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "UpdateSecret", resp, "Failure responding to request")
		return
	}

	return
}

// UpdateSecretPreparer prepares the UpdateSecret request.
func (client BaseClient) UpdateSecretPreparer(ctx context.Context, vaultBaseURL string, secretName string, secretVersion string, parameters SecretUpdateParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"secret-name":    autorest.Encode("path", secretName),
		"secret-version": autorest.Encode("path", secretVersion),
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPatch(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/secrets/{secret-name}/{secret-version}", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// UpdateSecretSender sends the UpdateSecret request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) UpdateSecretSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// UpdateSecretResponder handles the response to the UpdateSecret request. The method always
// closes the http.Response Body.
func (client BaseClient) UpdateSecretResponder(resp *http.Response) (result SecretBundle, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// Verify verifies a signature using the specified key
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
// keyName - the name of the key
// keyVersion - the version of the key
// parameters - the parameters for key operations.
func (client BaseClient) Verify(ctx context.Context, vaultBaseURL string, keyName string, keyVersion string, parameters KeyVerifyParameters) (result KeyVerifyResult, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.Verify")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.Digest", Name: validation.Null, Rule: true, Chain: nil},
				{Target: "parameters.Signature", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "Verify", err.Error())
	}

	req, err := client.VerifyPreparer(ctx, vaultBaseURL, keyName, keyVersion, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "Verify", nil, "Failure preparing request")
		return
	}

	resp, err := client.VerifySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "Verify", resp, "Failure sending request")
		return
	}

	result, err = client.VerifyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "Verify", resp, "Failure responding to request")
		return
	}

	return
}

// VerifyPreparer prepares the Verify request.
func (client BaseClient) VerifyPreparer(ctx context.Context, vaultBaseURL string, keyName string, keyVersion string, parameters KeyVerifyParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name":    autorest.Encode("path", keyName),
		"key-version": autorest.Encode("path", keyVersion),
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/keys/{key-name}/{key-version}/verify", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// VerifySender sends the Verify request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) VerifySender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// VerifyResponder handles the response to the Verify request. The method always
// closes the http.Response Body.
func (client BaseClient) VerifyResponder(resp *http.Response) (result KeyVerifyResult, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// WrapKey wraps a symmetric key using the specified key
// Parameters:
// vaultBaseURL - the vault name, e.g. https://myvault.vault.azure.net
// keyName - the name of the key
// keyVersion - the version of the key
// parameters - the parameters for key operations.
func (client BaseClient) WrapKey(ctx context.Context, vaultBaseURL string, keyName string, keyVersion string, parameters KeyOperationsParameters) (result KeyOperationResult, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/BaseClient.WrapKey")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.Value", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("keyvault.BaseClient", "WrapKey", err.Error())
	}

	req, err := client.WrapKeyPreparer(ctx, vaultBaseURL, keyName, keyVersion, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "WrapKey", nil, "Failure preparing request")
		return
	}

	resp, err := client.WrapKeySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "WrapKey", resp, "Failure sending request")
		return
	}

	result, err = client.WrapKeyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.BaseClient", "WrapKey", resp, "Failure responding to request")
		return
	}

	return
}

// WrapKeyPreparer prepares the WrapKey request.
func (client BaseClient) WrapKeyPreparer(ctx context.Context, vaultBaseURL string, keyName string, keyVersion string, parameters KeyOperationsParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name":    autorest.Encode("path", keyName),
		"key-version": autorest.Encode("path", keyVersion),
	}

	const APIVersion = "2015-06-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/keys/{key-name}/{key-version}/wrapkey", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// WrapKeySender sends the WrapKey request. The method will close the
// http.Response Body if it receives an error.
func (client BaseClient) WrapKeySender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// WrapKeyResponder handles the response to the WrapKey request. The method always
// closes the http.Response Body.
func (client BaseClient) WrapKeyResponder(resp *http.Response) (result KeyOperationResult, err error) {
	err = autorest.Respond(
		resp,
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}
