# Licensed to the Software Freedom Conservancy (SFC) under one
# or more contributor license agreements.  See the NOTICE file
# distributed with this work for additional information
# regarding copyright ownership.  The SFC licenses this file
# to you under the Apache License, Version 2.0 (the
# "License"); you may not use this file except in compliance
# with the License.  You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an
# "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
# KIND, either express or implied.  See the License for the
# specific language governing permissions and limitations
# under the License.

import os
import subprocess
import sys
from unittest.mock import patch

import pytest

from selenium.common.exceptions import SessionNotCreatedException
from selenium.webdriver.edge.service import Service


@pytest.mark.no_driver_after_test
def test_reuses_edgedriver_log(clean_driver, clean_options, driver_executable) -> None:
    log_file = "msedgedriver.log"

    service1 = Service(
        log_output=log_file,
        executable_path=driver_executable,
    )

    service2 = Service(
        log_output=log_file,
        service_args=["--append-log"],
        executable_path=driver_executable,
    )

    driver = None
    try:
        driver = clean_driver(options=clean_options, service=service1)
        with open(log_file) as fp:
            lines = len(fp.readlines())
    finally:
        if driver:
            driver.quit()
    try:
        driver = clean_driver(options=clean_options, service=service2)
        with open(log_file) as fp:
            assert len(fp.readlines()) >= 2 * lines
    finally:
        if driver:
            driver.quit()
        os.remove(log_file)


@pytest.mark.no_driver_after_test
def test_log_output_as_filename(clean_driver, clean_options, driver_executable) -> None:
    log_file = "msedgedriver.log"
    service = Service(log_output=log_file, executable_path=driver_executable)
    try:
        assert "--log-path=msedgedriver.log" in service.service_args
        driver = clean_driver(options=clean_options, service=service)
        with open(log_file) as fp:
            out = fp.read()
        assert "Starting" in out
        assert "started successfully" in out
    finally:
        driver.quit()
        os.remove(log_file)


@pytest.mark.no_driver_after_test
def test_log_output_as_file(clean_driver, clean_options, driver_executable) -> None:
    log_name = "msedgedriver.log"
    log_file = open(log_name, "w", encoding="utf-8")
    service = Service(log_output=log_file, executable_path=driver_executable)
    try:
        driver = clean_driver(options=clean_options, service=service)
        with open(log_name) as fp:
            out = fp.read()
        assert "Starting" in out
        assert "started successfully" in out
    finally:
        driver.quit()
        log_file.close()
        os.remove(log_name)


@pytest.mark.no_driver_after_test
def test_log_output_as_stdout(clean_driver, clean_options, capfd, driver_executable) -> None:
    service = Service(log_output=subprocess.STDOUT, executable_path=driver_executable)
    driver = clean_driver(options=clean_options, service=service)
    out, err = capfd.readouterr()
    assert "Starting" in out
    assert "started successfully" in out
    driver.quit()


@pytest.mark.no_driver_after_test
def test_log_output_null_default(driver, capfd) -> None:
    out, err = capfd.readouterr()
    assert "Starting" not in out
    assert "started successfully" not in out
    driver.quit()


@pytest.mark.xfail(
    sys.platform == "win32", reason="edgedriver doesn't return an error on windows if you use an invalid profile path"
)
@pytest.mark.no_driver_after_test
def test_driver_is_stopped_if_browser_cant_start(clean_driver, clean_options, clean_service, driver_executable) -> None:
    clean_options.add_argument("--user-data-dir=/no/such/location")
    service = Service(executable_path=driver_executable)
    with pytest.raises(SessionNotCreatedException):
        clean_driver(options=clean_options, service=service)
    assert not service.is_connectable()
    assert service.process.poll() is not None


@pytest.fixture
def service():
    return Service()


@pytest.mark.usefixtures("service")
class TestEdgeDriverService:
    service_path = "/path/to/msedgedriver"

    @pytest.fixture(autouse=True)
    def setup_and_teardown(self):
        os.environ["SE_EDGEDRIVER"] = self.service_path
        yield
        os.environ.pop("SE_EDGEDRIVER", None)

    def test_uses_path_from_env_variable(self, service):
        assert "msedgedriver" in service.path

    def test_updates_path_after_setting_env_variable(self, service):
        service.executable_path = self.service_path  # Simulating the update
        with patch.dict("os.environ", {"SE_EDGEDRIVER": "/foo/bar"}):
            assert "msedgedriver" in service.executable_path
